/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_SystemPowerManager.h
  * @brief        This is header file contains functions to handle error memory and shutdown functionality.
  * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

// -----------------------------------------------------------------------------
// includes
// -----------------------------------------------------------------------------
#include <map>
#include <list>
#include <string>

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#include "spm_ISystemPowerManager.h"

class ISpmSupervisionManager;
class ISpmOsalProxy;
class ISpmWorkerServer;
class ISpmLocalAppManager;
class ISpmCcaServiceServer;
class ISpmSubStateClient;
class ISpmOsLinux;
class ISpmApplicationDatabase;
class ISpmPowerSupplyStatistics;
class ISpmSystemLoadStatistics;

// -----------------------------------------------------------------------------
// class definition
// -----------------------------------------------------------------------------

/*!
  *  \class spm_tclSystemPowerManager
  *  \brief This class handles error memory and shutdown functionality. It writes project specific shutdown information and
          power on count to error memory.

        It uses class spm_tclSystemStateManagerConfig to get new system state and class spm_tclLocalApplicationManagerConfig to
        get culprit ID , requested state and uninitialised application state.


          The spm_tclFactory object creates an instance of this class. it used by spm_tclLocalApplicationManager to kill all application ready.


  */
class spm_tclSystemPowerManager : public ISpmSystemPowerManager
{

private:
tBool                   _bTerminateSpm;                             // !< indicates that no SPM termination is called before

tU32                    _u32ConsecutiveResetCounter;                // !< reset counter for consecutive resets without normal shutdown -> stored in lmm

tBool                   _bSupervisionResetEnabled;
tBool                   _bResetEnabled;

tBool                   _bTriggerRestartTypeECU;
tBool                   _bTriggerRestartTypeAPP;



ISpmOsalProxy          *_poclOsalProxy;
ISpmWorkerServer       *_poclWorkerServer;
ISpmLocalAppManager    *_poclLocalAppManager;
ISpmSupervisionManager *_poclSupervisionManager;
ISpmCcaServiceServer   *_poclCcaServiceHandler;
ISpmSubStateClient     *_poclSubStateHandler;
ISpmOsLinux            *_poclSpmOsLinux;
ISpmPowerSupplyStatistics        *_poclPowerSupplyStatistic;
ISpmSystemLoadStatistics         *_poclCpuUsageStatistic;
ISpmSystemLoadStatistics         *_poclRamUsageStatistic;
ISpmSystemLoadStatistics         *_poclStartupTimeStatistic;
tU32                    _u32TimeAtReset;
tU32                    _u32CvmAtReset;
tU32                    _u16UbatAtReset;
tU32                    _u32RestartType;
tBool                   _bRestartActive;
tU32                    _u32InternalResetEvent;
tU8                     _u8TestThreadCounter;
OSAL_tThreadID          _hTestThreadId;                                 // !< thread ID from Test thread
tBool                   _bTerminateTestThread;
tBool                   _bShutdownThreadStarted;

private:
virtual tVoid vTerminateSpm( );

virtual tVoid vStartMyDummyEndlessThread( tU32 u32Prio );

static tVoid vSpmEndlessThread( tVoid *pvArg );

virtual tVoid vStartAsyncPrepareShutdown( );

static tVoid vAsyncThread( tVoid *pvArg );

public:
spm_tclSystemPowerManager( const ISpmFactory& factory );
virtual ~spm_tclSystemPowerManager( );

virtual tVoid vKillAllApplicationsReady( );

virtual tVoid vSpmTraceThreadInfo( );

virtual tVoid main( );

virtual tVoid vShutdownSystem( tU32 u32ShutdownOption );

virtual tVoid vPrepareShutdown( );

// emtrace helper functions
virtual tVoid vShowErrMem( ) const;

virtual tVoid vWriteErrmem( tU16       u16Entry,
                            const tU8 *pu8Buf = NULL,
                            tU16       u16Len = SPM_MAX_ERRMEM_BUF,
                            tBool      bWriteDirectly = TRUE );

virtual tVoid vReadFile( const std::string& strFilename,
                         tBool              bWriteErrmem );

virtual tBool bWriteFile( const std::string& strFilename,
                          const std::string& strContent );

virtual tBool bRemoveFile( const std::string& strFilename );

virtual tVoid vCheckErrmemForVersion( );

virtual tVoid vUpdateInternalData( );

virtual tVoid vWritePowerOnCounterToErrMem( );

virtual tVoid vWriteProjectSpecificShutdownInfoToErrMem( tU32 u32ShutdownOption = 0 );

#ifdef LCM_UNIT_TESTS
   tU32 u32GetInternalResetEvent( ) const { return( _u32InternalResetEvent ); }
#endif

protected:
// ISpmBase interface functions
virtual tVoid vHandleTraceMessage( const tUChar *puchData );

virtual tVoid vGetReferences( );

virtual tVoid vStartCommunication( );

virtual const tChar*getName( ) const { return( "spm_tclSystemPowerManager" ); }

// implementation of interface ISpmWorkerClient
virtual tVoid vHandleMessage( tU32 u32Message,
                              tU32 u32Parm = 0 );

virtual tBool bHandleSynchrounousCall( tU32   u32Message,
                                       tVoid *args );

virtual tVoid vSetRestartType( tU32 u32RestartType );

virtual tU32 u32GetRestartType( ){ return( _u32RestartType ); }


};

