/*!
  * \file spm_SystemPowerManager.cpp
  *  \brief
  *    Errormemory and shutdown functionality
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * 12.08.11  | TMS Fischer       |
  * 03.04.12  | CM-AI/VW32 kollai | Adaptation for NISSAN LCN2 KAI
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"
#include "dp_generic_if.h"

// SPM  configuration
#include "spm_Config.h"
#include "spm_GlobDefs.h"

// my class header
#include "spm_SystemPowerManager.h"

// spm class definitions

// interfaces class definitions
#include "spm_ISuperVisionManager.h"
#include "spm_IOsalProxy.h"
#include "spm_ISystemStateManager.h"
#include "spm_ICcaServiceServer.h"
#include "spm_ISubStateClient.h"
#include "spm_ILocalAppManager.h"
#include "spm_IVersionInfo.h"
#include "spm_IOsLinux.h"
#include "spm_IStatistics.h"
#include "spm_IStartupSupervisor.h"
#include "spm_ICvmClient.h"
#include "spm_IApplicationDatabase.h"
#include "spm_ISupervisionEnableSupervisor.h"
#include "spm_IProcessSupervision.h"
#include "spm_IFactory.h"
#include "spm_IWakeupHandler.h"

// spm helper
#include "spm_PowerOnCount.h"
#include "spm_IPowerSupplyStatistics.h"
#include "spm_ISystemLoadStatistics.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_SystemPowerManager.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

// -----------------------------------------------------------------------------
// defines
// -----------------------------------------------------------------------------
// #define SPM_TRACE_FILE_ID   SPM_FILE_SYSTEMPOWERMANAGER

#define SPM_U32_CCA_HEARTBEAT_SHUTDOWN_TIME     15

#define SPM_U32_SHUTDOWN_THREAD_PRIO                                 ((tU32)100)
#define SPM_U32_SHUTDOWN_THREAD_STACKSIZE                           ((tU32)2048)

#include <limits.h>
#include <unistd.h>

// -----------------------------------------------------------------------------
// implementation
// -----------------------------------------------------------------------------

spm_tclSystemPowerManager::spm_tclSystemPowerManager( const ISpmFactory& factory )
   : ISpmSystemPowerManager( factory )
   , _bTerminateSpm( FALSE )
   , _u32ConsecutiveResetCounter( 0 )
   , _bSupervisionResetEnabled( TRUE )
   , _bResetEnabled( TRUE )
   , _bTriggerRestartTypeECU( FALSE )
   , _bTriggerRestartTypeAPP( FALSE )
   , _poclOsalProxy( NULL )
   , _poclWorkerServer( NULL )
   , _poclLocalAppManager( NULL )
   , _poclSupervisionManager( NULL )
   , _poclCcaServiceHandler( NULL )
   , _poclSubStateHandler( NULL )
   , _poclSpmOsLinux( NULL )
   , _poclPowerSupplyStatistic( NULL )
   , _poclCpuUsageStatistic( NULL )
   , _poclRamUsageStatistic( NULL )
   , _poclStartupTimeStatistic( NULL )
   , _u32RestartType( ( tU32 ) spm_fi_tcl_SPM_e32_RESTART_TYPE::FI_EN_SPM_U32_RESTART_ECU )
   , _bRestartActive( FALSE )
   , _u32InternalResetEvent( SPM_U32_SHUTDOWN_UNKOWN )
   , _u8TestThreadCounter( 0 )
   , _hTestThreadId( OSAL_ERROR )
   , _bTerminateTestThread(FALSE)
   , _bShutdownThreadStarted(FALSE){
/*!
  * \fn
  *  \brief
  *    Main FC SPM Constructor.
  *    - Initialize the SPM component.
  *    - Creates registry for the platform (base.reg) software. Registry entries are read
  *      from the flash (binary) or as file from the FFS.
  *    - Intanziate all SPM subclasses and starts communcation.
  *
  *  \param
  *    u16SpmID:  parameter used to identify errormemory entries
  *
  *  \return
  *    tVoid
  ******
  */
   dp_tclSpmDpInternDataLastOsalElapsedTime oTime;

   _u32TimeAtReset = oTime.tGetData( );

   dp_tclSpmDpInternDataLastVoltageLevel    oCvmState;
   _u32CvmAtReset  = oCvmState.tGetData( );

   dp_tclSpmDpInternDataLastUbatSense       oUbat;
   _u16UbatAtReset = oUbat.tGetData( );

   dp_tclSpmDpConfigResetDisable            oRstDisabled;

   if ( oRstDisabled.tGetData( ) != 0 ){
      _bResetEnabled = FALSE;
   }
}

spm_tclSystemPowerManager::~spm_tclSystemPowerManager( ){
/*!
  * \fn
  *  \brief
  *    Destructor. ...
  *
  *  \return
  *    tVoid
  ******
  */
   ETG_TRACE_USR4( ( "FC SPM destructed" ) );

   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   _poclWorkerServer->vRemoveClient( this );

   _poclOsalProxy          = NULL;
   _poclWorkerServer       = NULL;
   _poclSupervisionManager = NULL;
   _poclLocalAppManager    = NULL;
   _poclCcaServiceHandler  = NULL;
   _poclSubStateHandler    = NULL;
   _poclSpmOsLinux         = NULL;
   _poclPowerSupplyStatistic  = NULL;
   _poclCpuUsageStatistic     = NULL;
   _poclRamUsageStatistic     = NULL;
   _poclStartupTimeStatistic  = NULL;
}

tVoid spm_tclSystemPowerManager::vGetReferences( ){
   // get all needed references now -> all SPM objects are now available
   SPM_GET_IF_REFERENCE_USE_VAR( _poclOsalProxy,          ISpmOsalProxy );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer,       ISpmWorkerServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclLocalAppManager,    ISpmLocalAppManager );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSupervisionManager, ISpmSupervisionManager );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclCcaServiceHandler,  ISpmCcaServiceServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSubStateHandler,    ISpmSubStateClient );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSpmOsLinux,         ISpmOsLinux );
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poclPowerSupplyStatistic, spm_tclPowerSupplyStatistics, ISpmPowerSupplyStatistics );
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poclCpuUsageStatistic,    spm_tclCpuUsageStatistics,    ISpmSystemLoadStatistics );
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poclRamUsageStatistic,    spm_tclRamUsageStatistics,    ISpmSystemLoadStatistics );
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poclStartupTimeStatistic, spm_tclStartupTimeStatistics, ISpmSystemLoadStatistics );
} // vGetReferences

tVoid spm_tclSystemPowerManager::vStartCommunication( ){
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   _poclWorkerServer->vAddClient( this );
}

tVoid spm_tclSystemPowerManager::vStartAsyncPrepareShutdown( ){

   OSAL_trThreadAttribute rAttr;

   std::string            strThreadName = "SpmAsyncPrepare";

   rAttr.szName       = &strThreadName[0];
   rAttr.s32StackSize = 2048;
   rAttr.u32Priority  = 100;
   rAttr.pfEntry      = (OSAL_tpfThreadEntry)vAsyncThread;
   rAttr.pvArg        = ( tPVoid ) this;

   OSAL_tThreadID hThreadId = OSAL_ThreadSpawn( &rAttr );

   if ( hThreadId == OSAL_ERROR ){
      ETG_TRACE_ERRMEM( ( "spm_tclSystemPowerManager::vStartAsyncPrepareShutdown(): Failed to spawn thread!" ) );
   }
} // vStartAsyncPrepareShutdown

tVoid spm_tclSystemPowerManager::vAsyncThread( tVoid *pvArg ){

   spm_tclSystemPowerManager *poSpm = (spm_tclSystemPowerManager*)pvArg;

   SPM_NULL_POINTER_CHECK( poSpm );

   poSpm->vPrepareShutdown( );

}

tVoid spm_tclSystemPowerManager::vPrepareShutdown( ){

   // Update the Statistics Data to Datapool before shutdown
   ETG_TRACE_FATAL( ( "spm_tclSystemPowerManager::vPrepareShutdown(): write the statistics data to datapool now" ) );
   _poclPowerSupplyStatistic->vSetNewData( SPM_U32_STATISTIC_VOLTAGE_RANGE_HISTOGRAM );
   _poclCpuUsageStatistic->vSetNewData( SPM_U32_STATISTIC_CPU_USAGE );
   _poclRamUsageStatistic->vSetNewData( SPM_U32_STATISTIC_RAM_USAGE );
   _poclStartupTimeStatistic->vSetNewData( SPM_U32_STATISTIC_STARTUP_TIME );

   dp_tclSpmDpInternDataShutdownConfirmed oShutdownConfirmedSet;

   oShutdownConfirmedSet.vSetData( TRUE );

   tU32                                   u32SysTime = OSAL_ClockGetElapsedTime( ) / 1000;
   dp_tclSpmDpPowOnLastSystemRunTime      oSysOnTime;
   oSysOnTime << u32SysTime;

   dp_tclSpmDpPowOnCurrentSWVersionLastSystemRunTime oCurrentSWVersionSysOnTime;
   oCurrentSWVersionSysOnTime << u32SysTime;

   {
      SPM_GET_CLASS_REFERENCE_NEW_VAR( poResetStat, spm_tclResetStatistics, ISpmStatistics );
      poResetStat->vUpdateResetCounter( );
   }

   // store datapool now
   dp_tclSrvIf dp;
   tS32        dpRetValue = dp.s32StoreNow( );
   if ( DP_S32_NO_ERR != dpRetValue ){
      ETG_TRACE_ERRMEM( ( "spm_tclSystemPowerManager::vPrepareShutdown(): ERROR --> Datapool StoreNow returned with %d!", dpRetValue ) );
   }

   if ( _poclCcaServiceHandler ){
      spm_corefi_tclMsgRunlevelStateStatus tRunLevelState;

      tRunLevelState.eRunlevel.enType = spm_fi_tcl_SPM_e32_RUNLEVEL::FI_EN_SPM_U32_RUNLEVEL_SYNC_STARTED;
      _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_RUNLEVELSTATE, &tRunLevelState );
   }

   if ( _poclSpmOsLinux ){
      _poclSpmOsLinux->bSyncFilesystem( 100000 );
   }
   ETG_TRACE_FATAL( ( "spm_tclSystemPowerManager::vPrepareShutdown(): --> FS sync ready!" ) );

   if ( _poclSubStateHandler ){
      _poclSubStateHandler->vSetSubStateType( SPM_U32_SUBSTATE_SYNC_SHUTDOWN_ACK, TRUE );
   }

   if ( _bRestartActive && _poclWorkerServer ){
      _poclWorkerServer->bPostMessageToWorker( SPM_U32_WORKER_SPM_SHUTDOWN, _u32InternalResetEvent );
   }
} // vPrepareShutdown

tVoid spm_tclSystemPowerManager::main( ){
/*!
  * \fn
  *  \brief
  *    Asynchronoulsy running shutdown thread which is started when method spm_tclSystemPowerManager::vShutdownSystem()
  *    is called with SPM_U32_SHUTDOWN_NORMAL. These shutdown actions are parallelized to the main LCM execution
  *    flow (receive and respond to messages) because these shutdown actions might need a significant amount of time
  *    until they are executed. Without this parallelization the LCM might run into other message reception or supervision
  *    timeouts and this should be avoided.
  *
  *    This shutdown thread shall NEVER return because it is expected that we are powered off before.
  *
  ******
  */

   ETG_TRACE_FATAL( ( "spm_tclSystemPowerManager::main(): --> Final shutdown thread entered ..." ) );

   _poclSupervisionManager->vSetCCAHeartBeatInterval( SPM_U32_CCA_HEARTBEAT_SHUTDOWN_TIME );

   vWritePowerOnCounterToErrMem( );

   _poclSpmOsLinux->bShutdown( 5000 );
    ETG_TRACE_FATAL( ( "spm_tclSystemPowerManager::main(): --> RW partitions unmounted!" ) );

   if ( !_poclOsalProxy->bPrepareForShutdown( ) ){
      ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   }
   ETG_TRACE_FATAL( ( "spm_tclSystemPowerManager::main(): --> Prepare shutdown called!" ) );

   if ((!_bTriggerRestartTypeECU) && (!_bTriggerRestartTypeAPP)){
       if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_SWITCH_OFF ) ){
           ETG_TRACE_ERRMEM( ( "ERROR using DEV_SPM  SPM_C_S32_SYSTEMINIT_SWITCH_OFF " ) );
       }
   } else {
      tU32 u32PowerManagerShutdownOption;
      tU32 u32OsalProxyShutdownOption;

      if ( _bTriggerRestartTypeECU ){
         ETG_TRACE_ERRMEM( ( "vShutdownSystem, system shutdown -> ECU reset" ) );
         u32PowerManagerShutdownOption = SPM_U32_SHUTDOWN_ECU_RESET;
         u32OsalProxyShutdownOption = SPM_C_S32_SYSTEMINIT_RESET_ECU_API ;
	  }
      if ( _bTriggerRestartTypeAPP ){
         ETG_TRACE_ERRMEM( ( "vShutdownSystem, system shutdown -> APP reset" ) );
         u32PowerManagerShutdownOption = SPM_U32_SHUTDOWN_APP_RESET;
         u32OsalProxyShutdownOption = SPM_C_S32_SYSTEMINIT_RESET_APP_API;
	  }

      ETG_TRACE_FATAL( ( "spm_tclSystemPowerManager::main(): !!! System will Shutdown and ShutdownOption: %u !!!", ETG_ENUM( SPM_RESET_REASON, u32PowerManagerShutdownOption ) ) );

      std::string strStringBuffer = "LCM requested reset (u32ShutdownOption=" + std::to_string(u32PowerManagerShutdownOption) + "). Calling ShutdownSystem() NOW.";
      vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strStringBuffer.c_str( ), (tU16)strStringBuffer.length( ), FALSE );

      if ( !_poclOsalProxy->bShutdownSystem( u32OsalProxyShutdownOption ) ){
         TRACE_SPM_FAILURE;
      }
   }

   //This point should never be reached.
   ETG_TRACE_ERRMEM( ( "spm_tclSystemPowerManager::main(): --> Final shutdown thread left although this should never happen." ) );
   SPM_RESET_VIA_ASSERT;

   vSetTerminate();
}

tVoid spm_tclSystemPowerManager::vShutdownSystem( tU32 u32ShutdownOption ){
/*!
  * \fn
  *  \brief
  *    shutdown the system and store the shutdown reason in the error memory
  *
  *  \param
  *    u32ShutdownOption:  defines the shutdown reason. Possible defines see "spm_GlobDefs.h"
  *
  *  \note
  *     All shutdown optins are defined in "spm_GlobDefs.h"
  *  \see
  *     spm_GlobDefs.h
  ******
  */

   tU32 u32GetCulpritAppID;
   tU32 u32RequestedAppState;
   tU32 u32UninitializedState;

   tU8  u8Dummy = 0;

   SPM_NULL_POINTER_CHECK( _poclOsalProxy );
   SPM_NULL_POINTER_CHECK( _poclSupervisionManager );
   SPM_NULL_POINTER_CHECK( _poclSpmOsLinux );

   ETG_TRACE_FATAL( ( "spm_tclSystemPowerManager::vShutdownSystem(): !!! System Shutdown & ShutdownOption: %u !!!", ETG_ENUM( SPM_RESET_REASON, u32ShutdownOption ) ) );

   SPM_GET_IF_REFERENCE_NEW_VAR( poclCcaServiceHandler, ISpmCcaServiceServer );
   spm_corefi_tclMsgRunlevelStateStatus tRunLevelState;
   tRunLevelState.eRunlevel.enType = spm_fi_tcl_SPM_e32_RUNLEVEL::FI_EN_SPM_U32_RUNLEVEL_SHUTDOWN_NOW;
   poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_RUNLEVELSTATE, &tRunLevelState );

   // stop retriggering the processor watchdog
   _poclSupervisionManager->vEnableRetriggerHwWdt( FALSE );
   vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (const tU8*)&u8Dummy, (tU16)sizeof( u8Dummy ), FALSE );

   SPM_GET_IF_REFERENCE_NEW_VAR( poclSupervisionEnableSupervisor, ISpmSupervisionEnableSupervisor );
   if ( TRUE == poclSupervisionEnableSupervisor->bIsSupervisionStopped( ) ){
      _bResetEnabled = FALSE;
      ETG_TRACE_ERRMEM( ( "spm_tclSystemPowerManager::vShutdownSystem(): --> Reset disabled by Supervision settings (WD_OFF/disable_reset.txt/TTFis-SpmResetSupervisionDisable)" ) );
   }

   if (   (u32ShutdownOption == SPM_U32_SHUTDOWN_ECU_RESET)
       || (u32ShutdownOption == SPM_U32_SHUTDOWN_USER_RESET)
      ) {
       _bResetEnabled = TRUE;
       ETG_TRACE_ERRMEM( ( "spm_tclSystemPowerManager::vShutdownSystem(): --> Reset enabled (DIAG request)!" ) );
   }

   if ( SPM_U32_SHUTDOWN_NORMAL == u32ShutdownOption ){


#ifdef DEV_WUP_C_U8_FAST_SHUTDOWN_ACTIVE
      if (_poclSubStateHandler->bIsTriggerSet(SPM_U32_FAST_SHUTDOWN)
               || _bTriggerRestartTypeECU
               || _bTriggerRestartTypeAPP

      ) {
          _poclOsalProxy->bSetFastShutdown(DEV_WUP_C_U8_FAST_SHUTDOWN_ACTIVE);
      }
#endif

      if (_bShutdownThreadStarted == FALSE) {
         vStartThread( "SpmShutdown", SPM_U32_SHUTDOWN_THREAD_PRIO, SPM_U32_SHUTDOWN_THREAD_STACKSIZE );
         _bShutdownThreadStarted = TRUE;
      } else {
         ETG_TRACE_FATAL( ( "spm_tclSystemPowerManager::vShutdownSystem(): Shutdown thread already running, reject to start a second time" ) );
      } 
   } else {

      if ( !_bResetEnabled ){
         ETG_TRACE_FATAL( ( "spm_tclSystemPowerManager::vShutdownSystem(): !!! Reset disabled (ShutdownOption: %u) !!!", ETG_ENUM( SPM_RESET_REASON, u32ShutdownOption ) ) );
         if ( _poclSupervisionManager != NULL ){
            // start retriggering the processor watchdog
            _poclSupervisionManager->vEnableRetriggerHwWdt( TRUE );
         }
         return;
      }

      dp_tclSpmDpInternDataResetSetBySpm oResetData;
      oResetData << (tU8)u32ShutdownOption;

      // store DP now
      dp_tclSrvIf                        dp;
      dp.s32StoreNow( );

      std::string strStringBuffer = "LCM requested reset (u32ShutdownOption=" + std::to_string(u32ShutdownOption)  + "). Calling ShutdownSystem() NOW.";
      vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strStringBuffer.c_str( ), (tU16)strStringBuffer.length( ), FALSE );

      switch ( u32ShutdownOption ){
         case SPM_U32_SHUTDOWN_RESET:
         {
               ETG_TRACE_ERRMEM( ( "vShutdownSystem, system failure, reset shutdown" ) );
            if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESET_SYSTEM ) ){
               TRACE_SPM_FAILURE;
            }
            break;
         }

         case SPM_U32_SHUTDOWN_OVERTEMPERATURE:
         {
               ETG_TRACE_ERRMEM( ( "vShutdownSystem, system failure, reset shutdown overtemperature" ) );

            if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESET_SYSTEM ) ){
               TRACE_SPM_FAILURE;
            }
            break;
         }

         case SPM_U32_SHUTDOWN_WATCHDOG_HEARTBEAT_SUPERVISION:
         {
               ETG_TRACE_ERRMEM( ( "vShutdownSystem, system failure, reset shutdown heartbeat supervision" ) );

            if ( _bSupervisionResetEnabled ){
               if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESET_SYSTEM ) ){
                  TRACE_SPM_FAILURE;
               }
            } else {
               if ( _poclSupervisionManager != NULL ){
                  // start retriggering the processor watchdog
                  _poclSupervisionManager->vEnableRetriggerHwWdt( TRUE );
               }
            }
            break;
         }

         case SPM_U32_SHUTDOWN_WATCHDOG_MID_SUPERVISION:
         {
               ETG_TRACE_ERRMEM( ( "vShutdownSystem, system failure, reset shutdown mid watchdog supervision (reset enabled: %d)", ETG_ENUM( SPM_ONOFF_STATE, _bSupervisionResetEnabled ) ) );

            if ( _bSupervisionResetEnabled ){
               ETG_TRACE_ERR( ( "!!!!!!!!!!!!!!!! and now call OSAL device!!!!!!!!!!!!!!" ) );
               if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESET_SYSTEM ) ){
                  TRACE_SPM_FAILURE;
               }
            } else {
               if ( _poclSupervisionManager != NULL ){
                  // stop retriggering the processor watchdog
                  _poclSupervisionManager->vEnableRetriggerHwWdt( TRUE );
               }
            }

            break;
         }

         case SPM_U32_SHUTDOWN_WATCHDOG_HIGH_SUPERVISION:
         {
            ETG_TRACE_ERRMEM( ( "vShutdownSystem, system failure, reset shutdown high watchdog supervision (reset enabled: %d)", ETG_ENUM( SPM_ONOFF_STATE, _bSupervisionResetEnabled ) ) );

            if ( _bSupervisionResetEnabled ){
               ETG_TRACE_ERR( ( "!!!!!!!!!!!!!!!! and now call OSAL device!!!!!!!!!!!!!!" ) );
               if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESET_SYSTEM ) ){
                  TRACE_SPM_FAILURE;
               }
            } else {
               if ( _poclSupervisionManager != NULL ){
                  // stop retriggering the processor watchdog
                  _poclSupervisionManager->vEnableRetriggerHwWdt( TRUE );
               }
            }
            break;
         }

         case SPM_U32_SHUTDOWN_AIL_CCA_QUEUE_FULL:
         {
               ETG_TRACE_ERRMEM( ( "vShutdownSystem, system failure, reset shutdown CCA QUEUE FULL" ) );

            if ( _bSupervisionResetEnabled ){
               if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESET_SYSTEM ) ){
                  TRACE_SPM_FAILURE;
               }
            } else {
               if ( _poclSupervisionManager != NULL ){
                  // stop retriggering the processor watchdog
                  _poclSupervisionManager->vEnableRetriggerHwWdt( TRUE );
               }
            }
            break;
         }


         case SPM_U32_SHUTDOWN_NOTIFY_PROBLEM:
         {
               ETG_TRACE_ERRMEM( ( "vShutdownSystem, system failure, reset shutdown notify problem" ) );

            if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESET_SYSTEM ) ){
               TRACE_SPM_FAILURE;
            }
            break;
         }

         case SPM_U32_SHUTDOWN_APP_END_WANTED:
         {
               ETG_TRACE_ERRMEM( ( "vShutdownSystem, system shutdown -> application end wanted" ) );

            if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESET_SYSTEM ) ){
               TRACE_SPM_FAILURE;
            }
            break;
         }

         case SPM_U32_SHUTDOWN_AFTER_DOWNLOAD:
         {
               ETG_TRACE_ERRMEM( ( "vShutdownSystem, system shutdown -> download ready" ) );

            if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESET_WARMSTART ) ){
               TRACE_SPM_FAILURE;
            }
            break;
         }

         case SPM_U32_SHUTDOWN_AFTER_DOWNLOAD_ERROR:
         {
               ETG_TRACE_ERRMEM( ( "vShutdownSystem, system shutdown -> download error" ) );

            if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESET_SYSTEM ) ){
               TRACE_SPM_FAILURE;
            }
            break;
         }

         case SPM_U32_SHUTDOWN_AFTER_DOWNLOAD_TERMINATED:
         {
               ETG_TRACE_ERRMEM( ( "vShutdownSystem, system shutdown -> download terminated" ) );
            if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESET_SYSTEM ) ){
               TRACE_SPM_FAILURE;
            }
            break;
         }

         case SPM_U32_SHUTDOWN_AFTER_LONG_PRESS:
         {
               ETG_TRACE_ERRMEM( ( "vShutdownSystem, system shutdown -> after long press" ) );
            if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESET_SYSTEM ) ){
               TRACE_SPM_FAILURE;
            }
            break;
         }

         case SPM_U32_SHUTDOWN_USER_RESET:
         {
               ETG_TRACE_ERRMEM( ( "vShutdownSystem, system shutdown -> UserReset" ) );
            if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESET_SYSTEM ) ){
               TRACE_SPM_FAILURE;
            }
            break;
         }

         case SPM_U32_SHUTDOWN_AFTER_NO_STATE_REQUEST:
         {
            if ( _poclLocalAppManager == NULL ){
               ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!" ) );
               u32RequestedAppState  = 0;
               u32UninitializedState = 0;
               u32GetCulpritAppID    = 0xffff;
            } else {
               u32GetCulpritAppID    = _poclLocalAppManager->u32GetCulpritAppID( );
               u32RequestedAppState  = _poclLocalAppManager->u32GetRequestedAppState( u32GetCulpritAppID );
               u32UninitializedState = _poclLocalAppManager->u32GetUninitializedAppState( u32GetCulpritAppID );
            }

            strStringBuffer = "Additional info for SHUTDOWN_AFTER_NO_STATE_REQUEST: CulpritApp-Id=" + std::to_string(u32GetCulpritAppID) + ", u32RequestedAppState=" + std::to_string(u32RequestedAppState) + ", u32UninitializedState=" + std::to_string(u32UninitializedState);
            vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strStringBuffer.c_str( ), (tU16)strStringBuffer.length( ), FALSE );

            if ( _bSupervisionResetEnabled ){
               TRACE_SPM_ERROR_MY_THREAD( this );     // Dump SPM Callstacks also to see if we a are hanging on a semaphore
            }

            ETG_TRACE_ERRMEM( ( "vShutdownSystem, system shutdown -> after no state request" ) );

            strStringBuffer = "SPM_U32_SHUTDOWN_AFTER_NO_STATE_REQUEST-> Calling ShutdownSystem() NOW.";
            vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strStringBuffer.c_str( ), (tU16)strStringBuffer.length( ), SPM_WRITE_DIRECTLY );

            if ( _bSupervisionResetEnabled ){
               if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESET_SYSTEM ) ){
                  TRACE_SPM_FAILURE;
               }
            } else {
               if ( _poclSupervisionManager != NULL ){
                  // start retriggering the processor watchdog
                  _poclSupervisionManager->vEnableRetriggerHwWdt( TRUE );
               }
            }
            break;
         }

         case SPM_U32_SHUTDOWN_ECU_RESET:
         {

            ETG_TRACE_ERRMEM( ( "vShutdownSystem, system shutdown -> ECU reset" ) );
            _poclSpmOsLinux->bSyncFilesystem( 10000 );
            //As it is a "ECU_Restart" we need a reset of system.
            if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESET_ECU_API ) ){
               TRACE_SPM_FAILURE;
            }
            break;
         }

         case SPM_U32_SHUTDOWN_APP_RESET:
         {

               ETG_TRACE_ERRMEM( ( "vShutdownSystem, system shutdown -> APP reset" ) );
            //As it is a "ECU_Restart" we need a reset of system.
            if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESET_APP_API ) ){
               TRACE_SPM_FAILURE;
            }
            break;
         }

         case SPM_U32_SHUTDOWN_SCC_RESET:
         {

               ETG_TRACE_ERRMEM( ( "vShutdownSystem, system shutdown -> SCC reset" ) );
            //As it is a "ECU_Restart" we need a reset of system.
            if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESET_SCC_API ) ){
               TRACE_SPM_FAILURE;
            }
            break;
         }

         case SPM_U32_SHUTDOWN_ALTSW_TESTMANAGER:
         {

               ETG_TRACE_ERRMEM( ( "vShutdownSystem, system shutdown -> ecu reset" ) );
            if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESTART_ALTSW_TESTMANAGER ) ){
               TRACE_SPM_FAILURE;
            }
            break;
         }

         case SPM_U32_SHUTDOWN_LATE_WAKEKUP:
         {

               ETG_TRACE_ERRMEM( ( "vShutdownSystem, system shutdown -> late wakeup" ) );
            if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESET_LATE_WAKEUP ) ){
               TRACE_SPM_FAILURE;
            }
            break;
         }

         case SPM_U32_SHUTDOWN_PROCESS_SUPERVISION:
         {
               ETG_TRACE_ERRMEM( ( "vShutdownSystem, system failure, process supervision detects missing process (reset enabled: %d)", ETG_ENUM( SPM_ONOFF_STATE, _bSupervisionResetEnabled ) ) );

            if ( _bSupervisionResetEnabled ){
               ETG_TRACE_ERR( ( "!!!!!!!!!!!!!!!! and now call OSAL device!!!!!!!!!!!!!!" ) );
               if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESET_SYSTEM ) ){
                  TRACE_SPM_FAILURE;
               }
            } else {
               if ( _poclSupervisionManager != NULL ){
                  // stop retriggering the processor watchdog
                  _poclSupervisionManager->vEnableRetriggerHwWdt( TRUE );
               }
            }
            break;
         }

         default:
         {
            ETG_TRACE_ERRMEM( ( "vShutdownSystem, system shutdown -> other" ) );
            if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_RESET_SYSTEM ) ){
               TRACE_SPM_FAILURE;
            }
         }
      } // switch
   }
} // vShutdownSystem

tVoid spm_tclSystemPowerManager::vTerminateSpm( ){
/*!
  * \fn
  *  \brief
  *    Terminates the SPM. All vTerminate methods will be called to stop internal threads.
  *    SPM marks shutdown sequence as successful in the FFS.
  *    Event SPM_EV_SHUTDOWN_NAME is send to the thread which has created the SPM instance.
  *
  ******
  */
// check if terminate is already triggered
   if ( !_bTerminateSpm ){

      _bTerminateSpm = TRUE;

      ETG_TRACE_FATAL( ( "vTerminateSpm, all applications closed" ) );

      // at least -> set shutdown as confirmed
      dp_tclSpmDpInternDataShutdownConfirmed oShutdownConfirmedSet;
      oShutdownConfirmedSet.vSetData( TRUE );

      if ( NULL != _poclCcaServiceHandler ){
         if ( _poclCcaServiceHandler->bRestartTrigger( ) ){
            // ECU restart detected
            vShutdownSystem( SPM_U32_SHUTDOWN_ECU_RESET );
         }
      }
      // \todo: if needed here anymore
      // terminate myself
      // vTerminate();
   }
} // vTerminateSpm

tVoid spm_tclSystemPowerManager::vKillAllApplicationsReady( ){
/*!
  * \fn
  *  \brief
  *    Indicates that all applications are killed (terminated and destructed).
  *    And now terminate the SPM itself.
  *
  *  \param none
  ******
  */
   dp_tclSpmDpConfigShutdownLevel oShutdownLevel;
   tU32                           u32ShutdownLevel = oShutdownLevel.tGetData( );

   SPM_NULL_POINTER_CHECK( _poclOsalProxy );

   if ( u32ShutdownLevel == SPM_SHUTDOWN_LEVEL_PWR_DOWN_BEFORE_PF_SHUTDOWN ){
      ETG_TRACE_USR1( ( "Workaround: No PF/SPM terminate, switch off power supply NOW." ) );
      // at least -> set shutdown as confirmed
      dp_tclSpmDpInternDataShutdownConfirmed oShutdownConfirmedSet;
      oShutdownConfirmedSet.vSetData( TRUE );

      /* Actived the Abort-of-shutdown check in dev_wup, so incoming wakeup reasons
         during tengine/linux shutdown are memorized in pram and a reset is then
         automaticaly perfomend by dev_wup instead of shutting down
        */
      if ( !_poclOsalProxy->bPrepareForShutdown( ) ){
         ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!" ) );
      }
      if ( !_poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_SWITCH_OFF_IMMEDIATELY ) ){
         ETG_TRACE_USR1( ( "ERROR using DEV_SPM  SPM_C_S32_SYSTEMINIT_SWITCH_OFF" ) );
      }
   }
   vTerminateSpm( );
} // vKillAllApplicationsReady

tVoid spm_tclSystemPowerManager::vWriteProjectSpecificShutdownInfoToErrMem( tU32 u32ShutdownOption ){
   // #define SPM_MAX_ERRMEM_BUF    33
   (tVoid)u32ShutdownOption;

   SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemStateManager, ISpmSystemStateManager );
   tU32 u32SysState = poclSystemStateManager->u32GetSystemStateBeforeReset( );

   tU8  u8Dummy     = 0;
      vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_DOUBLE_SEPERATOR ),      (const tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );
   {
      tChar cBuf = SPM_U8_ADDITIONAL_ERRMEM_DATA;
      vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_SPM_CONFIG_ADDITIONAL_DATA ), (tU8*)&cBuf,          sizeof( cBuf ),    SPM_WRITE_DIRECTLY );
   }
      vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ),      (const tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );

      ETG_TRACE_USR1( ( "spm_tclSystemPowerManagerConfig::vWriteProjectSpecificShutdownInfoToErrMem()" ) );

   {
      std::string strBuf;

      strBuf.insert( 0, 1, (tChar)SPM_U8_ADDITIONAL_ERRMEM_DATA_STATE );
      SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 1, u32SysState )

      vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_SPM_CONFIG_ADDITIONAL_DATA ), (const tU8*)strBuf.c_str( ), (tU16)strBuf.length( ), SPM_WRITE_DIRECTLY );
   }
   {
      std::string strBuf;

      strBuf.insert( 0, 1, (tChar)SPM_U8_ADDITIONAL_ERRMEM_DATA_UBAT );
      SPM_M_INSERT_STRING_AT_INDEX_T16( strBuf, 1, _u16UbatAtReset )

      vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_SPM_CONFIG_ADDITIONAL_DATA ), (const tU8*)strBuf.c_str( ), (tU16)strBuf.length( ), SPM_WRITE_DIRECTLY );
   }
   {
      std::string strBuf;

      strBuf.insert( 0, 1, (tChar)SPM_U8_ADDITIONAL_ERRMEM_DATA_CVM );
      SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 1, _u32CvmAtReset )

      vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_SPM_CONFIG_ADDITIONAL_DATA ), (const tU8*)strBuf.c_str( ), (tU16)strBuf.length( ), SPM_WRITE_DIRECTLY );
   }
   {
      std::string strBuf;

      strBuf.insert( 0, 1, (tChar)SPM_U8_ADDITIONAL_ERRMEM_DATA_TIME );
      SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 1, _u32TimeAtReset )

      vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_SPM_CONFIG_ADDITIONAL_DATA ), (const tU8*)strBuf.c_str( ), (tU16)strBuf.length( ), SPM_WRITE_DIRECTLY );
   }
   {
      std::string strBuf;

      SPM_GET_IF_REFERENCE_NEW_VAR( _poclWakeupHandler,  ISpmWakeupHandler );
      strBuf.insert( 0, 1, (tChar)SPM_U8_ADDITIONAL_ERRMEM_RESET_COUNTER );
      strBuf.insert( 1, 1, _poclWakeupHandler->u8GetResetCounter() );

      vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_SPM_CONFIG_ADDITIONAL_DATA ), (const tU8*)strBuf.c_str( ), (tU16)strBuf.length( ), SPM_WRITE_DIRECTLY );
   }

   vWritePowerOnCounterToErrMem( );

   vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_DOUBLE_SEPERATOR ), (const tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );
} // vWriteProjectSpecificShutdownInfoToErrMem

tVoid spm_tclSystemPowerManager::vSpmTraceThreadInfo( ){

   std::string strBuffer;
   tU8         u8Dummy = 0;

   SPM_GET_IF_REFERENCE_NEW_VAR( poclSupervisionEnableSupervisor, ISpmSupervisionEnableSupervisor );

   if ( TRUE == poclSupervisionEnableSupervisor->bIsSupervisionStopped( ) ){
      ETG_TRACE_ERRMEM( ( "spm_tclSystemPowerManager::vSpmTraceThreadInfo: Triggering Callstack disabled due to Supervision settings (WD_OFF/disable_reset.txt/TTFis-SpmResetSupervisionDisable)" ) );
      return;
   }

   vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_DOUBLE_SEPERATOR ), (const tU8*)&u8Dummy,           (tU16)sizeof( u8Dummy ), FALSE );
   strBuffer = "SPM: Printout OSAL timer task info (Timer-xxx)!";
   vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ),                (const tU8*)strBuffer.c_str( ), (tU16)strBuffer.length( ) );
   #if OSAL_CONF == OSAL_LINUX
      _poclOsalProxy->bDumpThreadInfo( "Timer" );
   #else
      _poclOsalProxy->bWdtTimeOutWarning( SPM_S32_EMTRACE_THREAD, "OSTMR" );
      _poclOsalProxy->bWdtTimeOutWarning( SPM_S32_EMTRACE_THREAD, "TimHdr" );
   #endif

   vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (const tU8*)&u8Dummy,           (tU16)sizeof( u8Dummy ), FALSE );

   strBuffer = "SPM: Printout SPM task info!!!!";
   vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ),                (const tU8*)strBuffer.c_str( ), (tU16)strBuffer.length( ) );
   vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (const tU8*)&u8Dummy,           (tU16)sizeof( u8Dummy ), FALSE );

   SPM_GET_IF_REFERENCE_NEW_VAR( poProcessSupervision, ISpmProcessSupervision );
   poProcessSupervision->vDumpOOMInfo( );

   std::string strThreadName = "Spm";
   #if OSAL_CONF == OSAL_LINUX
      _poclOsalProxy->bDumpThreadInfo( strThreadName.c_str( ) );
   #else
      _poclOsalProxy->bWdtTimeOutWarning( SPM_S32_EMTRACE_THREAD, strThreadName.c_str( ) );
   #endif

   #if OSAL_CONF == OSAL_LINUX
      std::string strExePath( PATH_MAX, 0 );
      ssize_t     count = readlink( "/proc/self/exe", &strExePath[0], PATH_MAX );
      if ( count == 0 ){
         strExePath = "procbase_out.out";
      } else {
         strExePath.resize( count );
      }
      strBuffer = "SPM: procbase callstack generation started";
      vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strBuffer.c_str( ), (tU16)strBuffer.length( ) );

      _poclOsalProxy->bDumpProcessInfo( strExePath, "OSAL" );
   #endif

      vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (const tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );

      vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_DOUBLE_SEPERATOR ), (const tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );
} // vSpmTraceThreadInfo

tBool spm_tclSystemPowerManager::bHandleSynchrounousCall( tU32   u32Message,
                                                          tVoid *args ){
   SPM_NULL_POINTER_CHECK_VAL( _poclCcaServiceHandler );
   switch ( u32Message ){
      case SPM_U32_WORKER_SPM_SHUTDOWN:
         ETG_TRACE_USR1( ( "bHandleSynchrounousCall(SPM_U32_WORKER_SPM_SHUTDOWN): %d", * (tU32*)args ) );
         vShutdownSystem( * (tU32*)args );
         return( TRUE );

      case SPM_U32_WORKER_SPM_PREPARE_DOWNLOAD_READY:
         ETG_TRACE_USR4( ( "bHandleSynchrounousCall(SPM_U32_WORKER_SPM_PREPARE_DOWNLOAD_READY)" ) );
         {
            #ifndef LCM_UNIT_TESTS
               spm_corefi_tclMsgPrepareDownloadMethodResult methodResult;
               methodResult.Result = 1;
               _poclCcaServiceHandler->vUpdateMethodResult( SPM_COREFI_C_U16_PREPAREDOWNLOAD, &methodResult );
            #endif
         }
         return( TRUE );

      default:
         // nothing to do
         break;
   } // switch
   return( FALSE );
}    // bHandleSynchrounousCall

tVoid spm_tclSystemPowerManager::vShowErrMem( ) const {
   /*
      -------------------------------------------------------------------------
      FUNCTION:    vShowErrMem

      DESCRIPTION:

      PARAMETER:   -

      RETURNVALUE: tVoid

      HISTORY:
      Date      | Author            | Modification
      -         | CM-CM/ESU1 Kollai | initial version
      -------------------------------------------------------------------------
     */

   #define SPM_LOC_BUFFERSIZE 250
   tLcmUString        ustrBuf;
   trErrmemEntry      tErrmemElem;
   trErrmemInfo       tErrmemInfo;
   tS32               s32Ret;
   tU16               u16Index;

   tErrmemInfo.u32Size = 0; // uselesss init parameter, just for LINT

   tU32               u32StartTimeEmRead = OSAL_ClockGetElapsedTime( );

   OSAL_tIODescriptor fd                 = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ERRMEM, OSAL_EN_READWRITE );

   if ( fd != OSAL_ERROR ){
      s32Ret = OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_ERRMEM_CHECK, ( intptr_t )&tErrmemInfo );
      if ( s32Ret == OSAL_OK ){
         ustrBuf.insert( 0, 1, 0 );
         SPM_M_INSERT_STRING_AT_INDEX_T32( ustrBuf, 1,  tErrmemInfo.u32Size );
         SPM_M_INSERT_STRING_AT_INDEX_T32( ustrBuf, 5,  tErrmemInfo.u32UsedSize );
         SPM_M_INSERT_STRING_AT_INDEX_T32( ustrBuf, 9,  tErrmemInfo.u32NumberOfEntries );
         SPM_M_INSERT_STRING_AT_INDEX_T32( ustrBuf, 13, tErrmemInfo.u32CountOfEntries );
         SPM_M_INSERT_STRING_AT_INDEX_T32( ustrBuf, 17, tErrmemInfo.u32NumberOfFatals );
         SPM_M_INSERT_STRING_AT_INDEX_T32( ustrBuf, 21, tErrmemInfo.u32CountOfFatals );
         SPM_M_INSERT_STRING_AT_INDEX_T32( ustrBuf, 25, tErrmemInfo.u32NumberOfCompressions );
         // LLD_vTrace (OSAL_C_TR_CLASS_SYS_ERROR,TR_LEVEL_FATAL,ustrBuf,29);

         if ( tErrmemInfo.u32NumberOfEntries > 0 ){
            /* Trace all Entries except the last one, this is traced later */
            for ( u16Index = 0; u16Index < ( tErrmemInfo.u32NumberOfEntries - 1 ); u16Index++ ){
               tErrmemElem.u16Entry = u16Index;
               s32Ret               = OSAL_s32IORead( fd, ( tPS8 ) & tErrmemElem, sizeof( tErrmemElem ) );
               tU16 u16CopyLength = tErrmemElem.u16EntryLength;
               if ( SPM_LOC_BUFFERSIZE < tErrmemElem.u16EntryLength ){
                  u16CopyLength = SPM_LOC_BUFFERSIZE;
               }
               ustrBuf.assign( tErrmemElem.au8EntryData, u16CopyLength );

               if ( 0 < tErrmemElem.u16EntryLength ){
                  ETG_TRACE_USR4( ( "vShowErrMem, errMem entry(%u): Date: %02d.%02d.%04d Time: %02d:%02d:%02d, Error: %s",
                                    ETG_ENUM( ERRMEM_TYPE, tErrmemElem.eEntryType ),
                                    tErrmemElem.rEntryTime.s32Day,
                                    tErrmemElem.rEntryTime.s32Month,
                                    ( tErrmemElem.rEntryTime.s32Year + 1900 ),
                                    tErrmemElem.rEntryTime.s32Hour,
                                    tErrmemElem.rEntryTime.s32Minute,
                                    tErrmemElem.rEntryTime.s32Second,
                                    &ustrBuf[5]
                                    ) );
               }
            } /* trace for all entries except last */

            /* Trace last Entry */
            tErrmemElem.u16Entry = (tU16)(((tU16)tErrmemInfo.u32NumberOfEntries) - 1);
            s32Ret               = OSAL_s32IORead( fd, ( tPS8 ) & tErrmemElem, sizeof( tErrmemElem ) );
            tU16 u16CopyLength = tErrmemElem.u16EntryLength;
            if ( SPM_LOC_BUFFERSIZE < tErrmemElem.u16EntryLength ){
               u16CopyLength = SPM_LOC_BUFFERSIZE;
            }

            ustrBuf.assign( tErrmemElem.au8EntryData, u16CopyLength );

            ETG_TRACE_USR4( ( "vShowErrMem, errMem entry(%u): Date: %02d.%02d.%04d Time: %02d:%02d:%02d, Error: %s",
                              ETG_ENUM( ERRMEM_TYPE, tErrmemElem.eEntryType ),
                              tErrmemElem.rEntryTime.s32Day,
                              tErrmemElem.rEntryTime.s32Month,
                              ( tErrmemElem.rEntryTime.s32Year + 1900 ),
                              tErrmemElem.rEntryTime.s32Hour,
                              tErrmemElem.rEntryTime.s32Minute,
                              tErrmemElem.rEntryTime.s32Second,
                              &ustrBuf[5]
                              ) );
         }
      }
      if ( OSAL_s32IOClose( fd ) != OSAL_OK ){
         ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!" ) );
      }
   }
   ETG_TRACE_FATAL( ( "SPM : EMTRACE check duration: %d", OSAL_ClockGetElapsedTime( ) - u32StartTimeEmRead ) );
} // vShowErrMem

tVoid spm_tclSystemPowerManager::vCheckErrmemForVersion( ){
   SPM_GET_CLASS_REFERENCE_NEW_VAR( poVerInfo, spm_tclVersionInfo, ISpmVersionInfo );
   poVerInfo->vUpdateErrmemVersion( );
} // vCheckErrmemForVersion

/*******************************************************************************
 * \brief: update some LCM internal data in PRAM for use after reset
 *         - current battery voltage
 *         - current system run time
 *         - last received system voltage level
 *
 * \param  None
 *
 * \return: None.
 *
 *******************************************************************************/
tVoid spm_tclSystemPowerManager::vUpdateInternalData( ){
   SPM_GET_CLASS_REFERENCE_NEW_VAR( poCvmClient, spm_tclCriticalVoltageManagerIntern, ISpmCvmClient );
   SPM_NULL_POINTER_CHECK( _poclOsalProxy );

   tU32                                     u32CvmState = poCvmClient->u32GetCurVoltageState( );
   tU16                                     u16Ubat     = 0;

   _poclOsalProxy->bUbatSense( &u16Ubat );
   _poclPowerSupplyStatistic->vUpdateVoltageHistogram( u16Ubat );

   dp_tclSpmDpInternDataLastOsalElapsedTime oTime;
   oTime << OSAL_ClockGetElapsedTime( );

   dp_tclSpmDpInternDataLastVoltageLevel    oCvmState;
   oCvmState << u32CvmState;

   dp_tclSpmDpInternDataLastUbatSense       oUbat;
   oUbat << u16Ubat;

} // vCheckErrmemForVersion

/*******************************************************************************
 * \brief: Writes the given buffer into errormemory
 *
 *
 * \param  u16Entry  - type of entry see spm_GobDefs.h e.g. SPM_U8_ERRMEM_TYPE_RESET
 *         pu8Buf    - pointer to buffer containing the message
 *         u16Len    - length of the message
 *         bWriteDirectly - ignored
 *
 * \return: None.
 *
 *******************************************************************************/
tVoid spm_tclSystemPowerManager::vWriteErrmem( tU16       u16Entry,
                                               const tU8 *pu8Buf,
                                               tU16       u16Len,
                                               tBool      bWriteDirectly ){
   (tVoid)bWriteDirectly;

   OSAL_tIODescriptor fd_errmem    = OSAL_ERROR;
   trErrmemEntry      rErrmemEntry = { 0,0,eErrmemEntryUndefined,{ 0,0,0,0,0,0,0,0,0 },0,0,0};

   fd_errmem = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ERRMEM, OSAL_EN_WRITEONLY );

   if ( fd_errmem != OSAL_ERROR ){
      rErrmemEntry.u16Entry   = u16Entry;
      (tVoid)OSAL_s32ClockGetTime( &rErrmemEntry.rEntryTime );
      rErrmemEntry.eEntryType = eErrmemEntryNormal;

      #define ENTRY_START_POS ((tU16)5)
      // map to dragon em_trace structure
      if ( u16Len > ( ERRMEM_MAX_ENTRY_LENGTH - ENTRY_START_POS ) ){
         u16Len = ERRMEM_MAX_ENTRY_LENGTH - ENTRY_START_POS;
         ETG_TRACE_ERR( ( "spm_tclStartupInvestigation::vWriteErrmem() string longer then the osal buffer of %d!", ERRMEM_MAX_ENTRY_LENGTH ) );
         ETG_TRACE_ERR( ( "ErrMem string: %s", pu8Buf ) );
      }

      rErrmemEntry.u16EntryLength  = u16Len;
      rErrmemEntry.u16EntryLength  = (tU16)(rErrmemEntry.u16EntryLength + ENTRY_START_POS);

      rErrmemEntry.au8EntryData[0] = (tU8)( ( 0x00ff & SPM_TRACE_CLASS_SPM ) >> 0 );
      rErrmemEntry.au8EntryData[1] = (tU8)( ( 0xff00 & SPM_TRACE_CLASS_SPM ) >> 8 );
      rErrmemEntry.au8EntryData[2] = (tU8)( ( 0xff00 & I_SPM_EMTRACE_ENTRY ) >> 8 ); // trace ID
      rErrmemEntry.au8EntryData[3] = (tU8)( ( 0x00ff & I_SPM_EMTRACE_ENTRY ) >> 0 ); // trace ID
      rErrmemEntry.au8EntryData[4] = (tU8)( 0xff & u16Entry );

      (tVoid)OSAL_pvMemoryCopy( &rErrmemEntry.au8EntryData[5], pu8Buf, u16Len );

      (tVoid)OSAL_s32IOWrite( fd_errmem, ( tPCS8 ) & rErrmemEntry, sizeof( rErrmemEntry ) );

      (tVoid)OSAL_s32IOClose( fd_errmem );
   }
} // vWriteErrmem

tVoid spm_tclSystemPowerManager::vReadFile( const std::string& strFilename,
                                            tBool              bWriteErrmem ){
   #define LINE_LENGTH  250
//#define OSAL_M_INSERT_T8(buf, Byte) (buf)[0] = (tU8) ((Byte) & 0xFF);

   tS32               s32Val;
   OSAL_tIODescriptor fd;
   tS32               size, i;
   tS32               offset      = 0;
   std::string        cTraceBuf( LINE_LENGTH + 5, 0 );
   tChar             *pBuffer     = NULL;
   tBool              bIgnoreSize = FALSE;

   ETG_TRACE_USR4( ( "spm_tclSystemPowerManager::vReadFile(): Write file '%s'.", strFilename.c_str( ) ) );

   fd = OSAL_IOOpen( strFilename.c_str( ), OSAL_EN_READONLY );
   if ( fd != OSAL_ERROR ){
      size = OSALUTIL_s32FGetSize( fd );
      if ( size == OSAL_ERROR ){
         size        = 100000;
         bIgnoreSize = TRUE;
      }
      if ( ( pBuffer = (tChar*)OSAL_pvMemoryAllocate( (tU32)size ) ) != 0 ){
         memset( pBuffer, 0, (tU32)size );

         if ( ( s32Val = OSAL_s32IORead( fd, (tPS8)pBuffer, (tU32)size ) ) != size ){
            if ( bIgnoreSize != TRUE ){
               //TraceString((const char*)"Read File Error");
               ETG_TRACE_FATAL( ( "spm_tclSystemPowerManager::vReadFile(): Read File Error." ) );
               s32Val = - 1;
            }
         }

         if ( s32Val >= 0 ){
            //OSAL_M_INSERT_T8(&cTraceBuf[0],OSAL_STRING_OUT);
            ETG_TRACE_USR4( ( "spm_tclSystemPowerManager::vReadFile(): And now start to read file line by line." ) );

            while ( size ){
               /* search next line if available*/
               cTraceBuf.assign( LINE_LENGTH, 0 );
               for ( i = 0; i < LINE_LENGTH; i++ ){
                  if ( ( * ( pBuffer + offset + i ) == /*EOF*/ 0xff ) || ( * ( pBuffer + offset + i ) == 0x0a /*EOL*/ ) ){
                     i++;
                     break;
                  }
                  if ( offset + i > s32Val ){
                     break;
                  }
               }
               cTraceBuf.assign( pBuffer + offset, (tU32)i - 1 );
               ETG_TRACE_FATAL( ( "spm_tclSystemPowerManager::vReadFile(): '%s'", cTraceBuf.c_str( ) ) );
               if ( bWriteErrmem ){
                  vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)cTraceBuf.c_str( ), (tU16)cTraceBuf.length( ) );
               }
               offset += ( i );
               if ( offset >= s32Val ){
                  break;
               }
            } // end while
         } else {
            ETG_TRACE_FATAL( ( "spm_tclSystemPowerManager::vReadFile(): Read File Error , Try it again." ) );
         }
         OSAL_vMemoryFree( pBuffer );
      } else {
         ETG_TRACE_FATAL( ( "spm_tclSystemPowerManager::vReadFile():Allocate Memory Error" ) );
         //TraceString((const char*)"Allocate Memory Error");
      }
      OSAL_s32IOClose( fd );
   } else {
      ETG_TRACE_FATAL( ( "spm_tclSystemPowerManager::vReadFile():Error opening file." ) );
   }
} // vReadFile

tBool spm_tclSystemPowerManager::bRemoveFile( const std::string& strFilename ){
   tBool bRet = FALSE;

   if ( OSAL_OK == OSAL_s32IORemove( strFilename.c_str( ) ) ){
      bRet = TRUE;
   }
   return( bRet );
}

tBool spm_tclSystemPowerManager::bWriteFile( const std::string& strFilename,
                                             const std::string& strContent ){
   tBool              bRet = FALSE;

   OSAL_tIODescriptor fd   = OSAL_IOOpen( strFilename.c_str( ), OSAL_EN_READWRITE );

   if ( fd == OSAL_ERROR ){
      fd = OSAL_IOCreate( strFilename.c_str( ), OSAL_EN_READWRITE );
      if ( fd != OSAL_ERROR ){
      } else {
         ETG_TRACE_USR1( ( "spm_tclSystemPowerManager::vWriteFile(): Cound not create file '%s'.", strFilename.c_str( ) ) );
      }
   }

   if ( fd != OSAL_ERROR ){
      tS32 s32Result = OSAL_s32IOWrite( fd, (tPCS8)strContent.c_str( ), (tU32)strContent.length( ) );
      if ( (std::size_t)s32Result != strContent.length( ) ){
         ETG_TRACE_USR1( ( "spm_tclSystemPowerManager::vWriteFile(): Failed to write file '%s'.", strFilename.c_str( ) ) );
      } else {
         bRet = TRUE;
      }

      if ( OSAL_s32IOClose( fd ) != OSAL_OK ){
      }
   }

   return( bRet );
} // bWriteFile

tVoid spm_tclSystemPowerManager::vHandleMessage( tU32 u32Message,
                                                 tU32 u32Parm ){
   (tVoid)u32Parm;
   switch ( u32Message ){
      case SPM_U32_WORKER_SPM_CHECK_EMTRACE_VERS:
         vUpdateInternalData( );
         if ( OSAL_ClockGetElapsedTime( ) >= SPM_START_CHECK_EMTRACE_VERS ){
            vCheckErrmemForVersion( );
         }
         break;

      case SPM_U32_WORKER_SPM_PREPARE_SHUTDOWN:
         ETG_TRACE_USR4( ( "spm_tclSystemPowerManager::vHandleMessage(SPM_U32_WORKER_SPM_PREPARE_SHUTDOWN)" ) );
         {
            SPM_GET_IF_REFERENCE_NEW_VAR( poclCcaServiceHandler, ISpmCcaServiceServer );
            spm_corefi_tclMsgRunlevelStateStatus tRunLevelState;
            tRunLevelState.eRunlevel.enType = spm_fi_tcl_SPM_e32_RUNLEVEL::FI_EN_SPM_U32_RUNLEVEL_SHUTDOWN_STARTED;
            poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_RUNLEVELSTATE, &tRunLevelState );
         }
         vStartAsyncPrepareShutdown( );
         break;

      default:
         // nothing to do
         break;
   } // switch
}    // vHandleMessage

tVoid spm_tclSystemPowerManager::vStartMyDummyEndlessThread( tU32 u32Prio ){
   OSAL_trThreadAttribute rAttr;

   std::string            strThreadName = "SpmLoop";

   _u8TestThreadCounter++;

   ETG_TRACE_USR4( ( "Start SPM Thread %d (endless loop). Priority = %d", _u8TestThreadCounter, u32Prio ) );

   // start with lowest priority to give navi the change to delete all threads befor starting download
   rAttr.szName       = &strThreadName[0];
   rAttr.s32StackSize = 10000;
   rAttr.u32Priority  = u32Prio;
   rAttr.pfEntry      = (OSAL_tpfThreadEntry)vSpmEndlessThread;
   rAttr.pvArg        = ( tPVoid ) this;

   _bTerminateTestThread = FALSE;

   _hTestThreadId         = OSAL_ThreadSpawn( &rAttr );

   if ( _hTestThreadId == OSAL_ERROR ){
      ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   }
} // vStartMyDummyEndlessThread

/**
  *  \brief write the time the system was on to ErrorMemory
  *
  *  \return none
  *
  *  \details Gets the time from spm_tclPowerOnCount and writes it to ErrorMemory to see how long the device
  *  has been running when it reset.
  */
tVoid spm_tclSystemPowerManager::vWritePowerOnCounterToErrMem( ){
   std::string   strBuffer;

   SPM_GET_CLASS_REFERENCE_NEW_VAR( poPwrOn, spm_tclPowerOnCount, ISpmStatistics );

   strBuffer = "PowerOnCounter: " + poPwrOn->szAsString( ) + ", SessionCounter: " + std::to_string(OSAL_ClockGetElapsedTime( )) + "ms";

   vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strBuffer.c_str( ), (tU16)strBuffer.length( ), SPM_WRITE_DIRECTLY );
   ETG_TRACE_FATAL( ( "SPM: !!!!!! POWERONCOUNT: '%s' !!!!!!", strBuffer.c_str( ) ) );

} // vWritePowerOnCounterToErrMem

/**
  *  \brief trigger restart the system depending on the passed parameter
  *
  *  \param [in] u32RestartType enumeration containing the type of restart of the system
  *  \return none
  *
  *  \details depending on the restart type the system is reset immediately or shutdown to be restarted
  */
tVoid spm_tclSystemPowerManager::vSetRestartType( tU32 u32RestartType ){
   spm_corefi_tclMsgRunlevelStateStatus tRunLevelState;

   SPM_GET_IF_REFERENCE_NEW_VAR( poclStartupSupervisor, ISpmStartupSupervisor );
         SPM_NULL_POINTER_CHECK( _poclSubStateHandler );
         SPM_NULL_POINTER_CHECK( _poclCcaServiceHandler );
         SPM_NULL_POINTER_CHECK( _poclWorkerServer );

   _u32RestartType                 = u32RestartType;

   _bRestartActive                 = TRUE;

   tRunLevelState.eRunlevel.enType = spm_fi_tcl_SPM_e32_RUNLEVEL::FI_EN_SPM_U32_RUNLEVEL_RESTART_REQUESTED;
   _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_RUNLEVELSTATE, &tRunLevelState );

   dp_tclSpmDpInternDataShutdownConfirmed oShutdownConfirmedSet;
   switch ( u32RestartType ){
      case spm_fi_tcl_SPM_e32_RESTART_TYPE::FI_EN_SPM_U32_RESTART_ECU:
      case spm_fi_tcl_SPM_e32_RESTART_TYPE::FI_EN_SPM_U32_RESTART_APP:
      {
	// trigger restart type ECU or APP to enable correspoding restart

	if ( u32RestartType == spm_fi_tcl_SPM_e32_RESTART_TYPE::FI_EN_SPM_U32_RESTART_ECU ){
	   _bTriggerRestartTypeECU = TRUE;
	} else {
	   _bTriggerRestartTypeAPP = TRUE;
	}

	_poclCcaServiceHandler->SetRestartTrigger( TRUE );

	/* ONLY FOR ECU RESETS (triggerd by DIAGNOSIS), we disable the emergency off counter here.*/
	poclStartupSupervisor->vClearEmergencyCounter( );

	// trigger emergency off
	_poclSubStateHandler->vSetSubStateType( SPM_U32_EMERGENCY_OFF, TRUE );
	_poclSubStateHandler->vSetSubStateType( SPM_U32_FAST_SHUTDOWN, TRUE );

	break;
      }

      case spm_fi_tcl_SPM_e32_RESTART_TYPE::FI_EN_SPM_U32_RESTART_WDT:
      {
         // stop triggering HW watchdog
         SPM_NULL_POINTER_CHECK( _poclSupervisionManager );
         ETG_TRACE_FATAL( ( "Disabling WDT retriggering." ) );
         _poclSupervisionManager->vEnableRetriggerHwWdt( FALSE );
         break;
      }

      case spm_fi_tcl_SPM_e32_RESTART_TYPE::FI_EN_SPM_U32_RESTART_ECU_FAST:
      case spm_fi_tcl_SPM_e32_RESTART_TYPE::FI_EN_SPM_U32_RESTART_APP_FAST:
      {
         // mark reset as normal shutdown
         // shutdown process (fc spm) is performed
         oShutdownConfirmedSet.vSetData( TRUE );

         /* ONLY FOR ECU RESETS (triggerd by DIAGNOSIS), we disable the emergency off counter here.*/
         poclStartupSupervisor->vClearEmergencyCounter( );

         // store now DP now
         dp_tclSrvIf dp;
         dp.s32StoreNow( );

         tRunLevelState.eRunlevel.enType = spm_fi_tcl_SPM_e32_RUNLEVEL::FI_EN_SPM_U32_RUNLEVEL_RESTART_NOW;
         _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_RUNLEVELSTATE, &tRunLevelState );
         // ECU restart detected
         if ( u32RestartType == spm_fi_tcl_SPM_e32_RESTART_TYPE::FI_EN_SPM_U32_RESTART_ECU_FAST ){
            _poclWorkerServer->bPostMessageToWorker( SPM_U32_WORKER_SPM_SHUTDOWN, SPM_U32_SHUTDOWN_ECU_RESET );
         } else {
            _poclWorkerServer->bPostMessageToWorker( SPM_U32_WORKER_SPM_SHUTDOWN, SPM_U32_SHUTDOWN_APP_RESET );
         }
         break;
      }

      case spm_fi_tcl_SPM_e32_RESTART_TYPE::FI_EN_SPM_U32_RESTART_ECU_WITH_SYNC:
      case spm_fi_tcl_SPM_e32_RESTART_TYPE::FI_EN_SPM_U32_RESTART_APP_WITH_SYNC:
      {

         // mark reset as normal shutdown
         // shutdown process (fc spm) is performed
         oShutdownConfirmedSet.vSetData( TRUE );

         // store now DP now
         dp_tclSrvIf dp;
         dp.s32StoreNow( );

         /* ONLY FOR ECU RESETS (triggerd by DIAGNOSIS), we disable the emergency off counter here.*/
         poclStartupSupervisor->vClearEmergencyCounter( );

         // ECU restart detected
         if ( u32RestartType == spm_fi_tcl_SPM_e32_RESTART_TYPE::FI_EN_SPM_U32_RESTART_ECU_WITH_SYNC ){
            _u32InternalResetEvent = SPM_U32_SHUTDOWN_ECU_RESET;
         } else {
            _u32InternalResetEvent = SPM_U32_SHUTDOWN_APP_RESET;
         }

         //restart with sync --> call async thread to sync
         vStartAsyncPrepareShutdown( );
         break;
      }

      case spm_fi_tcl_SPM_e32_RESTART_TYPE::FI_EN_SPM_U32_RESTART_ECU_ALTSW_TESTMANAGER:
      {
         // mark reset as normal shutdown
         // shutdown process (fc spm) is performed
         oShutdownConfirmedSet.vSetData( TRUE );

         // store now DP now
         dp_tclSrvIf dp;
         dp.s32StoreNow( );

         /* ONLY FOR ECU RESETS (triggerd by DIAGNOSIS), we disable the emergency off counter here.*/
         poclStartupSupervisor->vClearEmergencyCounter( );

         // ECU restart detected
         _u32InternalResetEvent = SPM_U32_SHUTDOWN_ALTSW_TESTMANAGER;

         //restart with sync --> call async thread to sync
         vStartAsyncPrepareShutdown( );
         break;
      }

      case spm_fi_tcl_SPM_e32_RESTART_TYPE::FI_EN_SPM_U32_RESTART_SCC:
      {
         _poclWorkerServer->bPostMessageToWorker( SPM_U32_WORKER_SPM_SHUTDOWN, SPM_U32_SHUTDOWN_SCC_RESET );
         break;
      }

      case spm_fi_tcl_SPM_e32_RESTART_TYPE::FI_EN_SPM_U32_RESTART_USER_RESET:
      {
         _poclWorkerServer->bPostMessageToWorker( SPM_U32_WORKER_SPM_SHUTDOWN, SPM_U32_SHUTDOWN_USER_RESET );
         break;
      }

      case spm_fi_tcl_SPM_e32_RESTART_TYPE::FI_EN_SPM_U32_RESTART_DOWNLOAD_ERROR:
      {
         // mark reset as normal shutdown
         // shutdown process (fc spm) is performed
         oShutdownConfirmedSet.vSetData( TRUE );

         tRunLevelState.eRunlevel.enType = spm_fi_tcl_SPM_e32_RUNLEVEL::FI_EN_SPM_U32_RUNLEVEL_RESTART_NOW;
         _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_RUNLEVELSTATE, &tRunLevelState );

         // force coldstart to pass startupcode to load new software release
         _poclWorkerServer->bPostMessageToWorker( SPM_U32_WORKER_SPM_SHUTDOWN, SPM_U32_SHUTDOWN_AFTER_DOWNLOAD_ERROR );
         break;
      }

      case spm_fi_tcl_SPM_e32_RESTART_TYPE::FI_EN_SPM_U32_RESTART_DOWNLOAD_FINISHED:
      {
         // mark reset as normal shutdown
         // shutdown process (fc spm) is performed
         oShutdownConfirmedSet.vSetData( TRUE );

         // force coldstart
         _poclWorkerServer->bPostMessageToWorker( SPM_U32_WORKER_SPM_SHUTDOWN, SPM_U32_SHUTDOWN_AFTER_DOWNLOAD );
         break;
      }

      case spm_fi_tcl_SPM_e32_RESTART_TYPE::FI_EN_SPM_U32_RESTART_DOWNLOAD_ABORTED:
      {
         // mark reset as normal shutdown
         // shutdown process (fc spm) is performed
         oShutdownConfirmedSet.vSetData( TRUE );

         tRunLevelState.eRunlevel.enType = spm_fi_tcl_SPM_e32_RUNLEVEL::FI_EN_SPM_U32_RUNLEVEL_RESTART_NOW;
         _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_RUNLEVELSTATE, &tRunLevelState );

         // force coldstart
         _poclWorkerServer->bPostMessageToWorker( SPM_U32_WORKER_SPM_SHUTDOWN, SPM_U32_SHUTDOWN_AFTER_DOWNLOAD_TERMINATED );
         break;
      }

      default:
      {
         ETG_TRACE_ERRMEM( ( "spm_tclSystemPowerManager::vSetRestartType: Unkown value for u32RestartType (%d)", (tUInt)u32RestartType ) );
      }
   } // switch
}    // vSetRestartType

/**
  *  \brief just run a while loop
  *
  *  \param [in] pvArg those are currently ignored
  *  \return none
  *
  *  \details This while loop is used to generate system load in a test thread
  */
tVoid spm_tclSystemPowerManager::vSpmEndlessThread( tVoid *pvArg ){
/*!
  * \fn
  *  \brief
  *    Worker Thread e.g.  start Navi/Platform/SDS. Delete and destruct Navi. Used for DOWNLOAD scenario.
  *
  *  \param
  *     tVoid*  -> pointer to this-pointer of the SPM object.
  ******
  */
   spm_tclSystemPowerManager *poSpm = (spm_tclSystemPowerManager*)pvArg;

   while ( !poSpm->_bTerminateTestThread ){
   }
}

// spm_tclSystemPowerManager_user: user specific code end

// EOF

