/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_OsalWupSupervisionErrorWarning.cpp
  * @brief        This is file contains functions to handle supervision error warning event from dev_wup.
  *               In case of supervision error system might go for a reset.
  *               Base on warning code retrieved from dev_wup corresponding debug info will be dumping to error memory
  * @copyright    (C) 2017 Robert Bosch Engineering and Business Solutions Vietnam Company
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

// SPM  configuration
#include "spm_Config.h"
#include "spm_GlobDefs.h"

// my class header
#include "spm_OsalWupSupervisionErrorWarning.h"

// interfaces class definitions
#include "spm_IFactory.h"
#include "spm_ISystemStateManager.h"
#include "spm_ISystemStateStatistics.h"
#include "spm_IApplicationErrorHandler.h"
#include "spm_ISystemPowerManager.h"
#include "spm_IProcessSupervision.h"
#include "spm_IStartupInvestigationServer.h"
#include "spm_IOsLinux.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_OsalWupSupervisionErrorWarning.cpp.trc.h"
#endif

#include "spm_trace.h"

#define SPM_WUP_NOT_C_U32_EVENT_MASK_STOP_THREAD   0x01000000

#define SPM_WUP_NOT_C_U32_EVENT_MASK_ALL \
   ( DEV_WUP_C_U32_EVENT_MASK_CPU_SUPERVISION_ERROR_WARNING_CHANGED_NOTIFY   \
     | SPM_WUP_NOT_C_U32_EVENT_MASK_STOP_THREAD )


spm_tclOsalWupSupervisionErrorWarning::spm_tclOsalWupSupervisionErrorWarning( const ISpmFactory& factory ) : ISpmWupSupervisionErrorWarning( factory )
   , _hDeviceWupDescr( ( tS32 )OSAL_C_INVALID_HANDLE )
   , _hEventHandle( OSAL_C_INVALID_HANDLE )
   , _hThreadIdWupEvent( OSAL_ERROR ){

}

spm_tclOsalWupSupervisionErrorWarning::~spm_tclOsalWupSupervisionErrorWarning( ){

   if ( ( _hThreadIdWupEvent != OSAL_ERROR ) && ( _hEventHandle != OSAL_C_INVALID_HANDLE ) ){
      // thread is running --> stop now
         ETG_TRACE_USR1( ( "spm_tclOsalWupSupervisionErrorWarning::~spm_tclOsalWupSupervisionErrorWarning(): Send event to stop thread" ) );
      if ( OSAL_s32EventPost( _hEventHandle, SPM_WUP_NOT_C_U32_EVENT_MASK_STOP_THREAD, OSAL_EN_EVENTMASK_OR ) == OSAL_OK ){
         ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
      }
   }
   
}

tVoid spm_tclOsalWupSupervisionErrorWarning::vGetReferences( ){
   // get all needed references now -> all SPM objects are now available
  
}

tVoid spm_tclOsalWupSupervisionErrorWarning::vStartCommunication( ){

   OSAL_trThreadAttribute rAttr;

   std::string            szThreadName = "SpmWupSupervErr\0";

   // open device
   _hDeviceWupDescr = OSAL_IOOpen( OSAL_C_STRING_DEVICE_WUP, OSAL_EN_READWRITE );
   if ( OSAL_ERROR == _hDeviceWupDescr ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: spm_OsalWupSupervisionErrorWarning ! Error detected in OSAL_IOOpen of DEV_WUP !: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
      _hDeviceWupDescr = (tS32)OSAL_C_INVALID_HANDLE;
   } else {

      rAttr.szName       = &szThreadName[0];
      rAttr.s32StackSize = 10000;
      rAttr.u32Priority  = 100;
      rAttr.pfEntry      = (OSAL_tpfThreadEntry)vWupWorkerThread;
      rAttr.pvArg        = ( tPVoid ) this;

      _hThreadIdWupEvent = OSAL_ThreadSpawn( &rAttr );

      if ( _hThreadIdWupEvent == OSAL_ERROR ){
          tU32 u32ErrorReason = OSAL_u32ErrorCode( );
          ETG_TRACE_ERRMEM( ( "SPM: spm_OsalWupSupervisionErrorWarning ! Error detected in OSAL_IOOpen of DEV_WUP !: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
      }
   }
} // vStartCommunication


/*!
  * \fn
  *  \brief
  *    Worker Thread to handle Supervision Error Warning event
  *    This Worker thread register for a notification and will be in "Pended" status until getting event from dev_wup.
  *  \param
  *     tVoid*  -> pointer to this-pointer of the SPM object.
  *  \version
  *    1.0   - Initial 
  ******
  */
tVoid spm_tclOsalWupSupervisionErrorWarning::vWupWorkerThread( tVoid *pvArg )
{
   tBool                                    bStopThread      = FALSE;
   OSAL_tEventMask                          rEventMaskResult = 0;

   spm_tclOsalWupSupervisionErrorWarning               *poThis           = (spm_tclOsalWupSupervisionErrorWarning*)pvArg;

   DEV_WUP_trClientRegistration             rClientRegistration ={0,0};

   tU8  u8SupervisionErrCode             = DEV_WUP_C_U8_CPU_SUPERVISION_ERROR_WARNING_NONE;

   if ( OSAL_s32IOControl( poThis->_hDeviceWupDescr, OSAL_C_S32_IOCTRL_WUP_REGISTER_CLIENT, ( intptr_t )&rClientRegistration ) == OSAL_OK )
   {
      if ( OSAL_s32EventOpen( rClientRegistration.szNotificationEventName, &poThis->_hEventHandle ) == OSAL_OK )
       {

         if ( OSAL_s32IOControl( poThis->_hDeviceWupDescr, OSAL_C_S32_IOCTRL_WUP_REGISTER_CPU_SUPERVISION_ERROR_WARNING_CHANGED_NOTIFICATION, ( intptr_t )rClientRegistration.u32ClientId ) == OSAL_OK )
        {

                        ETG_TRACE_USR1( ( "spm_tclOsalWupSupervisionErrorWarning::vWupWorkerThread(): Thread to handle supervision error warning event" ) );

           
            while ( FALSE == bStopThread )
            {
                // Thread Pended until getting event from dev_wup
               if ( OSAL_s32EventWait( poThis->_hEventHandle, SPM_WUP_NOT_C_U32_EVENT_MASK_ALL, OSAL_EN_EVENTMASK_OR, OSAL_C_TIMEOUT_FOREVER, &rEventMaskResult ) == OSAL_OK )
               {

                  if ( OSAL_s32EventPost( poThis->_hEventHandle, ~rEventMaskResult, OSAL_EN_EVENTMASK_AND ) == OSAL_OK )
                  {  // Clear evaluated event bits
                     tBool bEventFound = FALSE;
                     if ( rEventMaskResult & DEV_WUP_C_U32_EVENT_MASK_CPU_SUPERVISION_ERROR_WARNING_CHANGED_NOTIFY )
                     {
                        ETG_TRACE_USR1( ( "spm_tclOsalWupSupervisionErrorWarning::vWupWorkerThread(): DEV_WUP_C_U32_EVENT_MASK_CPU_SUPERVISION_ERROR_WARNING_CHANGED_NOTIFY " ) );
                       
                        u8SupervisionErrCode = poThis->u8GetSupervisionErrCode();
                        if (u8SupervisionErrCode != DEV_WUP_C_U8_CPU_SUPERVISION_ERROR_WARNING_NONE )
                        {
                            poThis->vDumpInfo( u8SupervisionErrCode );
                        }
                        else 
                        {        
                         ETG_TRACE_FATAL( ( "spm_tclOsalWupSupervisionErrorWarning::vWupWorkerThread :DEV_WUP_C_U8_CPU_SUPERVISION_ERROR_WARNING_NONE no warning detected ! " ) );
                        }
                        bEventFound = TRUE;
                     }
                     if ( rEventMaskResult & SPM_WUP_NOT_C_U32_EVENT_MASK_STOP_THREAD )
                     {
                        bStopThread = TRUE;
                        bEventFound = TRUE;
                     }
                     
                     if ( bEventFound == FALSE )
                     {
                        ETG_TRACE_FATAL( ( " spm_tclOsalWupSupervisionErrorWarning::vWupWorkerThread  Event Not Matched, Event mask is %08x.",rEventMaskResult ) );
                     }
                  } else 
                  {
                       tU32 u32ErrorReason = OSAL_u32ErrorCode( );
                       ETG_TRACE_ERRMEM( ( "SPM: spm_OsalWupSupervisionErrorWarning::vWupWorkerThread , Error detected! OSAL_s32EventPost: Error 0x%08X (%s)",
                                          (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
                  }
               } else 
               {
                       tU32 u32ErrorReason = OSAL_u32ErrorCode( );
                       ETG_TRACE_ERRMEM( ( "SPM: spm_OsalWupSupervisionErrorWarning::vWupWorkerThread , Error detected! OSAL_s32EventWait: Error 0x%08X (%s)",
                                          (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
               }
            } // End thread
            //  Now unregister client for SUPERVISION_ERROR_WARNING
            if ( OSAL_s32IOControl( poThis->_hDeviceWupDescr, OSAL_C_S32_IOCTRL_WUP_UNREGISTER_CPU_SUPERVISION_ERROR_WARNING_CHANGED_NOTIFICATION, 
                                                              (intptr_t)rClientRegistration.u32ClientId ) == OSAL_ERROR )
            {
                       tU32 u32ErrorReason = OSAL_u32ErrorCode( );
                       ETG_TRACE_ERRMEM( ( "SPM: spm_OsalWupSupervisionErrorWarning::vWupWorkerThread , Cannot unregister ! OSAL_s32IOControl: Error 0x%08X (%s)",
                                          (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
            }
			
        } 
        
       } else 
        {
            tU32 u32ErrorReason = OSAL_u32ErrorCode( );
            ETG_TRACE_ERRMEM( ( "SPM: spm_OsalWupSupervisionErrorWarning::vWupWorkerThread , Error Detected ! OSAL_s32EventOpen : Error 0x%08X (%s)",
                                              (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );                  
        }
     // Now unregister General registration of a client
    if ( OSAL_s32IOControl( poThis->_hDeviceWupDescr, OSAL_C_S32_IOCTRL_WUP_UNREGISTER_CLIENT, (intptr_t)rClientRegistration.u32ClientId ) == OSAL_ERROR )
        {
            tU32 u32ErrorReason = OSAL_u32ErrorCode( );
            ETG_TRACE_ERRMEM( ( "SPM: spm_OsalWupSupervisionErrorWarning::vWupWorkerThread , Cannot Unregister client OSAL_s32IOControl : Error 0x%08X (%s)",
                                            (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
        }
           
   } else   // If register general dev_wup client failed
   {
           tU32 u32ErrorReason = OSAL_u32ErrorCode( );
           ETG_TRACE_ERRMEM( ( "SPM: spm_OsalWupSupervisionErrorWarning::vWupWorkerThread ,register general dev_wup client failed ! OSAL_s32IOControl : Error 0x%08X (%s)",
                                          (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );       
   }
} // vWupWorkerThread

/*!
  * \fn
  *  \brief
  *    This function read supervision warning error code from dev_wup, current support are :
  *         SHUTDOWN_TIMEOUT                 0x00
  *         THERMAL_SHUTDOWN_TIMEOUT         0x01
  *         COMMUNICATION_WATCHDOG_TIMEOUT   0x02
  *         STARTUP_FINISHED_TIMEOUT         0x03
  *         USER_RESET                       0x04  
  *     Function return DEV_WUP_C_U8_CPU_SUPERVISION_ERROR_WARNING_NONE as default in case of error.
  *     https://inside-docupedia.bosch.com/confluence/pages/viewpage.action?pageId=418320282
  *  \param
  *    
  *  \return  
  *    tU8    - Error Code retrieved from dev_wup.
  *  \version
  *    1.0    - Initial 
  ******
  */
tU8 spm_tclOsalWupSupervisionErrorWarning::u8GetSupervisionErrCode( ){ 

 tU8 u8SupervisionErrCode = DEV_WUP_C_U8_CPU_SUPERVISION_ERROR_WARNING_NONE;
 
   if ( OSAL_ERROR != _hDeviceWupDescr ){

      if ( OSAL_s32IOControl( _hDeviceWupDescr, OSAL_C_S32_IOCTRL_WUP_GET_CPU_SUPERVISION_ERROR_WARNING, ( intptr_t )&u8SupervisionErrCode ) == OSAL_ERROR ){
         ETG_TRACE_FATAL( ( "spm_tclOsalWupSupervisionErrorWarning::u8GetSupervisionErrCode: OSAL_C_S32_IOCTRL_WUP_GET_CPU_SUPERVISION_ERROR_WARNING IO Ctrl Failed" ) );
      } else {
          ETG_TRACE_USR3( ( "spm_tclOsalWupSupervisionErrorWarning::u8GetSupervisionErrCode(): read Supervision Error Warning from dev_wup: %u", ETG_ENUM( DEV_WUP_u8CpuSupervisionErrorWarning, u8SupervisionErrCode ) ) );
      }
   }
         
 return u8SupervisionErrCode;
}

/*!
  * \fn
  *  \brief
  *    This function copy debug info needed to error memory and datapool
  *    The information would be helpful for debug, analysis purpose in case of reset, shutdown caused by supervision timeout.
  *  \param [in] u8SupervisionErrCode  -> Supervision Error warning Code
  *  \version
  *    1.0    - Initial 
  ******
  */
tVoid spm_tclOsalWupSupervisionErrorWarning::vDumpInfo( tU8 u8SupervisionErrCode ){ 
 
 SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemPowerManager, ISpmSystemPowerManager );
 SPM_GET_IF_REFERENCE_NEW_VAR( poclStartupInvest, ISpmStartupInvestigationServer);
 SPM_GET_CLASS_REFERENCE_NEW_VAR( poStateHistory,   spm_tclSystemStateStatistics, ISpmSystemStateStatistics );
 SPM_GET_IF_REFERENCE_NEW_VAR( poclAppErrorHdl, ISpmApplicationErrorHandler );
 SPM_GET_IF_REFERENCE_NEW_VAR( poProcessSupervision, ISpmProcessSupervision );
 SPM_GET_IF_REFERENCE_NEW_VAR( poOsLinuxProxy, ISpmOsLinux );
 
 
tU32 u32AddedStackId = 0;

 switch (u8SupervisionErrCode) {
     case DEV_WUP_C_U8_CPU_SUPERVISION_ERROR_WARNING_SHUTDOWN_TIMEOUT                :
     {       
        u32AddedStackId = poStateHistory->u32AddErrorHistory();
        ETG_TRACE_ERRMEM( ( "Detected SUPERVISION_ERROR_WARNING : SHUTDOWN_TIMEOUT, ErrorHistory stack %d added, TTFis cmd <SpmErrorStateHistory stack_Id> to read", u32AddedStackId) );
        break;
     }
     case DEV_WUP_C_U8_CPU_SUPERVISION_ERROR_WARNING_THERMAL_SHUTDOWN_TIMEOUT        :
     {                 
        u32AddedStackId = poStateHistory->u32AddErrorHistory();
         ETG_TRACE_ERRMEM( ( "Detected SUPERVISION_ERROR_WARNING : THERMAL_SHUTDOWN_TIMEOUT, ErrorHistory stack %d added, TTFis cmd <SpmErrorStateHistory stack_Id> to read", u32AddedStackId) );
        break;
         
     }
     case DEV_WUP_C_U8_CPU_SUPERVISION_ERROR_WARNING_COMMUNICATION_WATCHDOG_TIMEOUT  :
     {       
        ETG_TRACE_ERRMEM( ( "Detected SUPERVISION_ERROR_WARNING : COMMUNICATION_WATCHDOG_TIMEOUT, now START to write systemctl status inc_wdt.service > /dev/errmem " ) );	
        std::string strCmdLine = "systemctl status inc_wdt.service > /dev/errmem";
        poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strCmdLine.c_str( ), (tU16)strCmdLine.length( ) );
        poOsLinuxProxy->vRunCommandLine( strCmdLine );
        ETG_TRACE_ERRMEM( ( "Detected SUPERVISION_ERROR_WARNING : COMMUNICATION_WATCHDOG_TIMEOUT, write END please check status of inc_wdt.service in errmem" ) );
        break;
     }
     case DEV_WUP_C_U8_CPU_SUPERVISION_ERROR_WARNING_STARTUP_FINISHED_TIMEOUT        :
     {  
        ETG_TRACE_ERRMEM( ( "Detected SUPERVISION_ERROR_WARNING : STARTUP_FINISHED_TIMEOUT, now START dumping info") );
        poclAppErrorHdl->vTriggerUnitFailedListReceived();          // Write list of failed service to error memory and further info
        u32AddedStackId = poStateHistory->u32AddErrorHistory();     // Add state history
        ETG_TRACE_ERRMEM( ( "SUPERVISION_ERROR_WARNING : STARTUP_FINISHED_TIMEOUT, ErrorHistory stack %d added, TTFis cmd <SpmErrorStateHistory stack_Id> to read", u32AddedStackId) );
        poclStartupInvest->vDump();           				        // Dump Startup timing info to error memory
        ETG_TRACE_ERRMEM( ( "Detected SUPERVISION_ERROR_WARNING : STARTUP_FINISHED_TIMEOUT,  DONE dumping info" ) );
        break;
     }
     case DEV_WUP_C_U8_CPU_SUPERVISION_ERROR_WARNING_USER_RESET                      :
     {    
        ETG_TRACE_ERRMEM( ( "Detected SUPERVISION_ERROR_WARNING : WARNING_USER_RESET-------START dumping info") );
        
        u32AddedStackId = poStateHistory->u32AddErrorHistory();
        ETG_TRACE_ERRMEM( ( "---------WARNING_USER_RESET , ErrorHistory stack %d added ,use TTFis cmd <SpmErrorStateHistory stack_Id> to read ", u32AddedStackId ) );
        ETG_TRACE_ERRMEM( ( "--------------START Startup Investigation Info---------") );
        poclStartupInvest->vDump();
        ETG_TRACE_ERRMEM( ( "--------------END Startup Investigation Info-----------") );
        ETG_TRACE_ERRMEM( ( "--------------START OOM Out Of Memory Info---------") );
        poProcessSupervision->vDumpOOMInfo();
        ETG_TRACE_ERRMEM( ( "--------------END OOM Out Of Memory Info---------") );
        // Dumps IRQ load info 
        ETG_TRACE_ERRMEM( ( "--------------START IRQ load Info---------") );
        std::string strCmdLine_softirqs = "cat /proc/softirqs > /dev/errmem";
        poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strCmdLine_softirqs.c_str( ), (tU16)strCmdLine_softirqs.length( ) );
        poOsLinuxProxy->vRunCommandLine( strCmdLine_softirqs );
        ETG_TRACE_ERRMEM( ( "cat /proc/interrupts > /dev/errmem") );
        std::string strCmdLine_interrupts = "cat /proc/interrupts > /dev/errmem";
        poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strCmdLine_interrupts.c_str( ), (tU16)strCmdLine_interrupts.length( ) );
        poOsLinuxProxy->vRunCommandLine( strCmdLine_interrupts );
        ETG_TRACE_ERRMEM( ( "--------------END IRQ load Info---------") );
        ETG_TRACE_ERRMEM( ( "Detected SUPERVISION_ERROR_WARNING : WARNING_USER_RESET------- END dumping info") );
     }
     default :
     {
      
         break;
     }

 }

}



