/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_OsalProxy.h
  * @brief        This is header file contains functions to handle OSAL and notify LCM for system events..
  * @copyright    (C) 2014 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#include <set>
#include <string>
#include <list>

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "spm_IOsalProxy.h"

class ISpmWorkerServer;

#define SPM_SRAM_TYPE_INVALID 0xffffffff

/* Incoming wakeup reasons during shutdown*/
#define SPM_SHUTDOWN_ABORT_REASON_UNDEF 0
#define SPM_SHUTDOWN_ABORT_REASON_MOST 1
#define SPM_SHUTDOWN_ABORT_REASON_CAN 2

#define SPM_PROXY_CVM_EVENTS                    0x00000001

// Exit Mask for thread waiting for CVM events from dev_volt device.
// Mask has been taken from DEV_VOLT_C_U32_EVENT_MASK_FREE_FOR_CLIENT_25 from osioctrl.h
#define SPM_DEV_VOLT_HDLR_EXIT_MASK             0x01000000
// Mask for Voltage Events from dev_volt
#define SPM_CVM_DEV_VOLT_EVENTS_MASK \
   ( DEV_VOLT_C_U32_BIT_MASK_INDICATE_LOW_VOLTAGE           \
     | DEV_VOLT_C_U32_BIT_MASK_INDICATE_CRITICAL_LOW_VOLTAGE    \
     | DEV_VOLT_C_U32_BIT_MASK_INDICATE_HIGH_VOLTAGE            \
     | DEV_VOLT_C_U32_BIT_MASK_INDICATE_CRITICAL_HIGH_VOLTAGE )

// Mask for thread waiting for CVM events from dev_volt device
#define SPM_DEV_VOLT_HDLR_MASK \
   ( SPM_CVM_DEV_VOLT_EVENTS_MASK   \
     | SPM_DEV_VOLT_HDLR_EXIT_MASK )

#define SPM_CVM_C_U32_EVENT_MASK_ALL \
   ( DEV_VOLT_C_U32_EVENT_MASK_SYSTEM_VOLTAGE_CHANGED_NOTIFY    \
     | DEV_VOLT_C_U32_EVENT_MASK_PERMANENT_HIGH_VOLTAGE           \
     | DEV_VOLT_C_U32_EVENT_MASK_PERMANENT_CRITICAL_HIGH_VOLTAGE )

// definition of SYSTEM INFO
#define SPM_PROXY_DEVSPM_DV_TEST_SHUTDOWN       0x00000100

#define SPM_SHUTDOWN_OVERRIDE_INACTIVE           0x01
#define SPM_SHUTDOWN_OVERRIDE_DISABLE_POWER_OFF  0x01
#define SPM_SHUTDOWN_OVERRIDE_DISABLE_RESETS     0x02
#define SPM_SHUTDOWN_OVERRIDE_DISABLE_LINUX_HALT 0x04

// -----------------------------------------------------------------------------
// class definition
// -----------------------------------------------------------------------------

/*!
  *  \class spm_tclOsalProxy
  *  \brief This class acts as clienthandler for sytem power management. It handles critical voltage management and proxy system info events.
   It resets reset reason and gets reset reasons. It prepares spm for emergency and normal shutdown. It also suspends start and restart.

   it uses spm_tclSystemPowerManagerConfig for writing U16_M_ERRMEM_SPM_ERROR in error memory.

   The spm_tclFactory object creates an instance of this class. Class spm_tclCriticalVoltageManagerIntern uses to install OSAl
   critical voltage management callback.
  */

class ISpmOsLinux;
class ISpmProcessSupervision;
class ISpmSupervisionEnableSupervisor;

class spm_tclOsalProxy : public ISpmOsalProxy
{
protected:
// Private variables
OSAL_tIODescriptor _hDeviceWup;
OSAL_tIODescriptor _hDeviceVolt;
OSAL_tIODescriptor _hDevicePramLcm;
OSAL_tIODescriptor _hDevicePramResetCounter;

ISpmWorkerServer  *_poclWorkerServer;

tBool              _bTerminate;

tBool              _bShutdownActive;

private:
tU32               _u32OrgWakeupType;
tU32               _u32WakeupType;

public:
// constructor
spm_tclOsalProxy( const ISpmFactory& factory,
                  tU32               u32AppId );
virtual ~spm_tclOsalProxy( );

// ISpmBase interface functioms
virtual tVoid vHandleTraceMessage( const tUChar */*puchData*/ ){} //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used
virtual tVoid vTerminate( );


// common interface to OSAL
virtual tBool bGetWupVersion( tU32 *pu32Version );

virtual tU8  u8GetResetReason( tVoid );
virtual tU8  u8GetResetReasonScc( tVoid );


virtual tBool bGetUtcTime( OSAL_trTimeDate *tTimeDate );

virtual tU32 u32GetAdrResetCount( tVoid );

virtual tBool bStartupProcessed( tU8 u8StartupState = SPM_U8_OSALPROXY_STARTUPSTATE_UP ); // SPM internal startup finished

virtual tBool bGetStartType( tU8 *pU8Starttype );

virtual tBool bResetProcessor( tU8  u8Processor,
                               tU8  u8ResetMode,
                               tU16 u16ResetReason );

virtual tBool bPrepareForShutdown( );

virtual tBool bGetWakeupReason( tU32 *pu32WakeUpReason );

virtual tBool bSetWakeupConfig( tU32 u32WakeupConfig );

virtual tBool bTriggerWatchdog( );

virtual tBool bStartWatchdog( tU32 u32WatchdogInterval );

virtual tBool bSetGpio( tU32  u32Gpio,
                        tBool bState );

virtual tBool bGetGpio( tU32   u32Gpio,
                        tBool *pbState );

virtual tBool bInstallGpioCallback( tU32                 u32Gpio,
                                    OSAL_tpfGPIOCallback pvCallback,
                                    tVoid               *pvArg = NULL,
                                    tU16                 u16TriggerLevel = OSAL_GPIO_EDGE_BOTH );


virtual tBool bUbatSense( tU16 *u16UbatSense );

virtual tBool bCurrentSense( tU16 *u16UbatSense );

virtual tBool bWdtGetThreadInfoById( tU32  u32ThreadId,
                                     tBool fErrmemWrite = TRUE,
                                     tBool fTraceWrite = TRUE );

virtual tU32  u32GetSRamData( tU32 u32Type );

virtual tBool bSetSRamData( tU32 u32Type,
                            tU32 u32Data );

virtual tBool bSetResetCounter( tU8 u8Counter );

virtual tU8   u8GetResetCounter( void );

virtual tU8   u8GetApplicationMode( );

virtual tU8   u8GetOperationalState( );

virtual tU8   u8GetDmVerityState( void );
virtual tBool bSetDmVerityResult( tU8 u8DmVerityResult );
virtual tBool bSetFastShutdown( tU8 u8FastShutdownType );

virtual tBool bShutdownSystem( tU32 u32ShutdownOption );

virtual tBool bIsShutdownActive( ) const { return( _bShutdownActive ); }

virtual tBool bExtendPowerOffTimeout( tU32  u32TimeOut,
                                      tU32 *pu32ResultTimeOut = NULL );

virtual tBool bConfigureSccResetHandling( tBool bDisable,
                                          tU8   u8ExceptionBitmask );

private:
tBool                           bGetDevWupHandle( std::string strFunction );

ISpmOsLinux                     *_poOsLinuxHdl;
ISpmProcessSupervision          *_poclProcessSupervision;
ISpmSupervisionEnableSupervisor *_poclSupervisionEnableSupervisor;

std::string                      _strProcessName;
std::list < std::string >        _oListThreadName;

protected:
// implementation of interface ISpmBase
virtual tVoid vStartCommunication( );

virtual tVoid vGetReferences( );

virtual const tChar*getName( ) const { return( "spm_tclOsalProxy" ); }

// implementation of interface ISpmWorkerClient
virtual tVoid vHandleMessage( tU32 u32Message,
                              tU32 u32Parm = 0 );

virtual tBool bHandleSynchrounousCall( tU32   /*u32Message*/,
                                       tVoid */*args*/ ){ return( FALSE ); } //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

virtual tBool bWdtTimeOutWarning( tS32         s32EmTraceType,
                                  const tChar *cThreadName = "" );

virtual tBool bDumpThreadInfoReady( tS32  s32Prio = 0,
                                    tBool bAlways = SPM_BL_OSALPROXY_IF_ENABLED );

virtual tBool bDumpThreadInfo( const std::string& cThreadName,
                               tBool              bAlways );

virtual tBool bDumpProcessInfo( const std::string& cProcessName,
                                const std::string& strStartType,
                                tBool              bDumpCallstacks = TRUE,
                                tBool              bAlways = SPM_BL_OSALPROXY_IF_ENABLED );

virtual tBool bDumpProcessInfo( tU32  u32ProcdId,
                                tBool bDumpCallstacks = TRUE,
                                tBool bAlways = SPM_BL_OSALPROXY_IF_ENABLED );

virtual tBool bDumpProcessList( tBool bErrMem,
                                tBool bTrace,
                                tBool bAlways );

virtual tBool bDumpSemInfo( const tChar *cSemName );

virtual tBool bDumpTimeTrace( tBool bProcesses = FALSE,
                              tBool bAlways = SPM_BL_OSALPROXY_IF_ENABLED );

tVoid vCheckForFake( );

};

