/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_OsalProxy.cpp
  * @brief        This class acts as clienthandler for sytem power management
  * @copyright    (C) 2011 - 2017 Robert Bosch GmbH
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#if OSAL_CONF == OSAL_LINUX
#ifndef LCM_UNIT_TESTS
#include <sys/types.h>
#include <unistd.h>

#ifndef VARIANT_S_FTR_DISABLE_USE_SYSTEMD
#include <systemd/sd-daemon.h> // you also need to add -lsystemd-daemon to the linker settings
#else
         void sd_notify( unsigned long i,
                         const char   *strInfo ){
            return;
         }

         void sd_notifyf( unsigned long i,
                          const char   *strInfo,
                          unsigned long j,
                          unsigned long k ){
            return;
         }

#endif

#endif // ifndef LCM_UNIT_TESTS
#endif // if OSAL_CONF == OSAL_LINUX

   #define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_STDVISITORS
   #define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_FUNCTIONIDS
#include "spm_fi_if.h"

   #define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

#include "spm_Config.h"
#include "spm_GlobDefs.h"
#include "spm_OsalProxy.h"
#include "spm_ICvmClient.h"
#include "spm_SystemPowerManager.h"
#include "spm_SubStateHandler.h"
#include "spm_SuperVisionManager.h" // for wdt-retrigger workarround (linux-halt)

#include "spm_IWakeupHandler.h"
#include "spm_IOsLinux.h"
#include "spm_IProcessSupervision.h"
#include "spm_ISupervisionEnableSupervisor.h"

#include "spm_IFactory.h"

#include "spm_WorkerServerConfig.h"
#include "spm_Registry_pathes.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
    #include "trcGenProj/Header/spm_OsalProxy.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

#ifdef SPM_FEATURE_ENABLE_ADR3
#include "adr3ctrl_if.h"
#endif


/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
// #define SPM_TRACE_FILE_ID   SPM_FILE_OSALPROXY_G2

#ifndef OSAL_VOLT_CVM_OPERATING_VOLTAGE
      #define OSAL_VOLT_CVM_OPERATING_VOLTAGE 7
#endif

#ifndef DEV_WUP_C_U16_RESET_REASON_DIAGNOSIS
      #define DEV_WUP_C_U16_RESET_REASON_DIAGNOSIS   DEV_WUP_C_U16_RESET_REASON_UNSPECIFIED
#endif

/* check define consistency */
#if SPM_SHUTDOWN_REQ_MAX_TRY_ASSERT < SPM_SHUTDOWN_REQ_MAX_TRY_RESET
   #error check values of defines SPM_SHUTDOWN_REQ_MAX_TRY_RESET and SPM_SHUTDOWN_REQ_MAX_TRY_ASSERT
#endif

#ifndef DEV_WUP_C_U8_APPLICATION_MODE_NORMAL
      #define DEV_WUP_C_U8_APPLICATION_MODE_NORMAL                                0x00
#endif

/** The duration in milliseconds a potential reset line shall be pulled.
 Allowed values are 0 and 1 - 1000. If 0 is set, then the server decides about the duration the reset line is pulled. */
#define SPM_DEVWUP_RESET_PROCESSOR_DURATION   0

/** Define the key that is used to change the INC connection to fake devices */
#define CONNECT_TO_FAKE_KEY                           "CONNECT_TO_FAKE"
/** Define the value of the key that is used to change the INC connection to fake devices */
#define CONNECT_TO_FAKE_VALUE                         0xDEADBEEF

spm_tclOsalProxy::spm_tclOsalProxy( const ISpmFactory& factory,
                                       tU32               u32AppId ) : ISpmOsalProxy( factory )
      , _poclWorkerServer( NULL )
      , _bTerminate( FALSE )
      , _bShutdownActive( FALSE )
      , _u32OrgWakeupType( spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_INVALID )
      , _u32WakeupType( spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_INVALID )
      , _poOsLinuxHdl( NULL )
      , _poclProcessSupervision( NULL )
      , _poclSupervisionEnableSupervisor( NULL ){
/*!
  * \fn
  *  \brief
  *    Constructor. Implements interface functions to the OSAL.
  *
  *  \version
  *    1.0   - Initial
  ******
  */
      (tVoid)u32AppId;

      _hDevicePramLcm          = (OSAL_tIODescriptor)OSAL_ERROR;
      _hDevicePramResetCounter = (OSAL_tIODescriptor)OSAL_ERROR;
      _hDeviceWup              = (OSAL_tIODescriptor)OSAL_ERROR;
      _hDeviceVolt             = (OSAL_tIODescriptor)OSAL_ERROR;

      vCheckForFake( );
   }

/*!
  * \fn
  *  \brief
  *    Destructor.
  *
  *  \version
  *    1.0   - Initial
  ******
  */
   spm_tclOsalProxy::~spm_tclOsalProxy( ){
      // Close the DEV VOLT device
      if ( OSAL_ERROR != _hDeviceVolt ){
         if ( OSAL_s32IOClose( _hDeviceVolt ) != OSAL_OK ){
            ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected in OSAL_s32IOClose for DEV_VOLT !!!!!!" ) );
         }
      }
      if ( OSAL_ERROR != _hDeviceWup ){
         if ( OSAL_s32IOClose( _hDeviceWup ) != OSAL_OK ){
            ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected in OSAL_s32IOClose for DEV_WUP !!!!!!" ) );
         }
      }
      if ( OSAL_ERROR != _hDevicePramLcm ){
         if ( OSAL_OK != OSAL_s32IOClose( _hDevicePramLcm ) ){
            ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected in OSAL_s32IOClose for DEV_PRAM for LCM !!!!!!" ) );
         }
      }
      if ( OSAL_ERROR != _hDevicePramResetCounter ){
         if ( OSAL_OK != OSAL_s32IOClose( _hDevicePramResetCounter ) ){
            ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected in OSAL_s32IOClose for DEV_PRAM for ResetCounter !!!!!!" ) );
         }
      }
      _poclWorkerServer                = NULL;
      _poOsLinuxHdl                    = NULL;
      _poclProcessSupervision          = NULL;
      _poclSupervisionEnableSupervisor = NULL;
   }

/*!
  * \fn
  *  \brief
  *    Start the normal communication with the drivers
  *
  *  \version
  *    1.0   - Initial
  ******
  */
   tVoid spm_tclOsalProxy::vStartCommunication( ){
   }

   tVoid spm_tclOsalProxy::vGetReferences( ){
      SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer,                ISpmWorkerServer );
      SPM_GET_IF_REFERENCE_USE_VAR( _poOsLinuxHdl,                    ISpmOsLinux );
      SPM_GET_IF_REFERENCE_USE_VAR( _poclProcessSupervision,          ISpmProcessSupervision );
      SPM_GET_IF_REFERENCE_USE_VAR( _poclSupervisionEnableSupervisor, ISpmSupervisionEnableSupervisor );
   }

/*!
  * \fn
  *  \brief
  *    Stop the osal proxy input thread and prepare for shutdown
  *
  *  \version
  *    1.0   - Initial
  ******
  */
   tVoid spm_tclOsalProxy::vTerminate( ){
      _bTerminate = TRUE;
   }

/*!
  * \fn
  *  \brief
  *    If not yet done, get the dev_wup handle
  *
  *  \version
  *    1.0   - Initial
  ******
  */
   tBool spm_tclOsalProxy::bGetDevWupHandle( std::string strFunction ){
      tBool bRet = TRUE;
      if ( OSAL_ERROR == _hDeviceWup ){
         std::string strAction = strFunction + "(): Call /dev/wup (" + std::to_string(OSAL_ClockGetElapsedTime( )) + ") ";
         ETG_TRACE_FATAL( ( "%s", strAction.c_str( ) ) );
         _hDeviceWup = OSAL_IOOpen( "/dev/wup", OSAL_EN_READWRITE );
         if ( OSAL_ERROR == _hDeviceWup ){
            std::string strOpenError = strFunction + "() SPM: !!!!!! Error detected in OSAL_IOOpen of '/dev/wup' !!!!!!";
            ETG_TRACE_ERRMEM( ( "%s", strOpenError.c_str( ) ) );
            bRet = FALSE;
         }
         std::string   strError = strFunction + "(): Return from /dev/wup (" + std::to_string(OSAL_ClockGetElapsedTime( )) + ") ";
         ETG_TRACE_FATAL( ( "%s", strError.c_str( ) ) );
      }
      return( bRet );
   } // bGetDevWupHandle

   tU8 spm_tclOsalProxy::u8GetResetReason( tVoid ){
/*!
  * \fn
  *  \brief
  *    .
  *  \version
  *    1.0   - Initial
  ******
  */
      tU8 u8ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_WARMSTART;

      #ifdef OSAL_C_S32_IOCTRL_WUP_GET_LATEST_RESET_REASON
         if ( bGetDevWupHandle( "spm_tclOsalProxy::u8GetResetReason" ) ){
            DEV_WUP_trLatestResetReason rLatestResetReason;
            rLatestResetReason.u16ResetReasonAP = DEV_WUP_C_U16_RESET_REASON_UNKNOWN; // uselesss init parameter, just for LINT
            rLatestResetReason.u8ResetReasonSCC = (tU8)DEV_WUP_C_U8_RESET_REASON_POR; // uselesss init parameter, just for LINT

            if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_GET_LATEST_RESET_REASON, ( intptr_t )&rLatestResetReason ) == OSAL_OK ){
               ETG_TRACE_USR3( ( "spm_tclOsalProxy::u8GetResetReason(): received ResetReasonAP from dev_wup: %u", ETG_ENUM( DEV_WUP_u16ResetReasonAP, rLatestResetReason.u16ResetReasonAP ) ) );
               switch ( rLatestResetReason.u16ResetReasonAP ){
                  case DEV_WUP_C_U16_RESET_REASON_POWER_ON:
                     u8ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_POWER_ON_RESET;
                     break;

                  case DEV_WUP_C_U16_RESET_REASON_POWER_LOSS:
                     u8ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_POWER_LOSS;
                     break;

                  case DEV_WUP_C_U16_RESET_REASON_SELF:
                     u8ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_SOFTWARE;
                     break;

                  case DEV_WUP_C_U16_RESET_REASON_POWER_OFF:
                     u8ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_POWER_OFF;
                     break;

                  case DEV_WUP_C_U16_RESET_REASON_SCC:
                     u8ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_SCC;
                     break;

                  case DEV_WUP_C_U16_RESET_REASON_UNKNOWN:
                  default:
                     u8ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_UNKNOWN;
                     break;
               } // switch

               ETG_TRACE_USR1( ( "spm_tclOsalProxy::u8GetResetReason(): Reset reason detected: %u", ETG_ENUM( CFC_FI_LCMRESETREASON, u8ResetReason ) ) );
            }
         }
      #else // ifdef OSAL_C_S32_IOCTRL_WUP_GET_LATEST_RESET_REASON
         ETG_TRACE_FATAL( ( "IO-Control OSAL_C_S32_IOCTRL_WUP_GET_LATEST_RESET_REASON is used but not defined" ) );
      #endif // ifdef OSAL_C_S32_IOCTRL_WUP_GET_LATEST_RESET_REASON
      return( u8ResetReason );
   } // u8GetResetReason

   tU8 spm_tclOsalProxy::u8GetResetReasonScc( tVoid ){
/*!
  * \fn
  *  \brief
  *    .
  *  \version
  *    1.0   - Initial
  ******
  */
      tU8 u8ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_WARMSTART;

      #ifdef OSAL_C_S32_IOCTRL_WUP_GET_LATEST_RESET_REASON
         if ( bGetDevWupHandle( "spm_tclOsalProxy::u8GetResetReasonScc" ) ){
            DEV_WUP_trLatestResetReason rLatestResetReason;
            rLatestResetReason.u16ResetReasonAP = DEV_WUP_C_U16_RESET_REASON_UNKNOWN; // uselesss init parameter, just for LINT
            rLatestResetReason.u8ResetReasonSCC = (tU8)DEV_WUP_C_U8_RESET_REASON_POR; // uselesss init parameter, just for LINT

            if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_GET_LATEST_RESET_REASON, ( intptr_t )&rLatestResetReason ) == OSAL_OK ){
               ETG_TRACE_USR3( ( "spm_tclOsalProxy::u8GetResetReasonScc(): received ResetReasonSCC from dev_wup: %u", ETG_ENUM( DEV_WUP_u8ResetReasonSCC, rLatestResetReason.u8ResetReasonSCC ) ) );
               switch ( rLatestResetReason.u8ResetReasonSCC ){
                  case DEV_WUP_C_U8_RESET_REASON_HW_WATCHDOG:
                     u8ResetReason       = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_HW_WATCHDOG;
                     break;

                  case DEV_WUP_C_U8_RESET_REASON_POR:
                     u8ResetReason       = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_POWER_ON;
                     break;

                  case DEV_WUP_C_U8_RESET_REASON_COLDSTART:
                     u8ResetReason       = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_COLDSTART;
                     break;

                  case DEV_WUP_C_U8_RESET_REASON_APPMODE_CHANGE:
                     u8ResetReason       = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_APPMODE_CHANGE;
                     break;

                  case DEV_WUP_C_U8_RESET_REASON_DURING_LPW:
                     u8ResetReason       = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_DURING_LPW;
                     break;

                  case DEV_WUP_C_U8_RESET_REASON_PLL_OSZ:
                     u8ResetReason       = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_DURING_PLL_OSZ;
                     break;

                  case DEV_WUP_C_U8_RESET_REASON_SW:
                     u8ResetReason       = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_DURING_SW;
                     break;

                     #ifdef DEV_WUP_C_U8_RESET_REASON_USER_RESET
                        case DEV_WUP_C_U8_RESET_REASON_USER_RESET:
                           u8ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_USER;
                           break;
                     #endif

                  case DEV_WUP_C_U8_RESET_REASON_WARMSTART:
                  default:
                     u8ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_WARMSTART;
                     break;
               } // switch

               ETG_TRACE_USR1( ( "spm_tclOsalProxy::u8GetResetReason(): Reset reason SCC detected: %u", ETG_ENUM( CFC_FI_LCMRESETREASON, u8ResetReason ) ) );
            }
         }
      #else // ifdef OSAL_C_S32_IOCTRL_WUP_GET_LATEST_RESET_REASON
         ETG_TRACE_FATAL( ( "IO-Control OSAL_C_S32_IOCTRL_WUP_GET_LATEST_RESET_REASON is used but not defined" ) );
      #endif // ifdef OSAL_C_S32_IOCTRL_WUP_GET_LATEST_RESET_REASON
      return( u8ResetReason );
   } // u8GetResetReason

   tBool spm_tclOsalProxy::bPrepareForShutdown( ){
/*!
  * \fn
  *  \brief
  *    .
  *  \version
  *    1.0   - Initial
  ******
  */
      tBool bReturn = TRUE;

      ETG_TRACE_USR4( ( "!!!!!!!! *** PREPARE FOR SHUTDOWN CALLED *** !!!!!!!!!!" ) );

      if ( OSAL_ERROR != _hDeviceWup ){
      }
      return( bReturn );
   } // bPrepareForShutdown

   tBool spm_tclOsalProxy::bStartupProcessed( tU8 u8StartupState ){
/*!
  * \fn
  *  \brief
  *    .
  *  \version
  *    1.0   - Initial
  ******
  */
      tBool bReturn = TRUE;

      if ( bGetDevWupHandle( "spm_tclOsalProxy::bStartupProcessed" ) ){
         if ( SPM_U8_OSALPROXY_STARTUPSTATE_UP == u8StartupState ){
            ETG_TRACE_USR1( ( "spm_tclOsalProxy::bStartupProcessed(): Indicate device is up and running!" ) );
            if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_INDICATE_STARTUP_FINISHED, (intptr_t)NULL ) != OSAL_OK ){
               ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bStartupProcessed(): Failed to set 'OSAL_C_S32_IOCTRL_WUP_INDICATE_STARTUP_FINISHED'" ) );
            }
         }
      }
      return( bReturn );

   } // bStartupProcessed

   tBool spm_tclOsalProxy::bGetWakeupReason( tU32 *pu32WakeUpReason ){
/*!
  * \fn
  *  \brief get the wakeup reason reported by SCC - via dev_wup.
  *    .
  *  \version
  *    1.0   - Initial
  ******
  */
      tBool bReturn = FALSE;

      if ( spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_INVALID == _u32WakeupType ){

         if ( bGetDevWupHandle( "spm_tclOsalProxy::bGetWakeupReason" ) ){
            #ifndef VARIANT_S_FTR_ENABLE_SIMULATE_WAKEUP_REASON
               tU32 u32WakeupReason = DEV_WUP_C_U8_WAKEUP_REASON_UNKNOWN;
               if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_GET_WAKEUP_REASON, ( intptr_t )&u32WakeupReason ) == OSAL_OK ){
                  /* \todo KSM2HI, 13.04.2015

                     Interface of /dev/wup was changed and the define DEV_WUP_C_U8_WAKEUP_REASON_RESET is removed.

                     There is no wakeup reason of type RESET. If there is a logic which needs the information if the system was starting up due to a reset,
                     the /dev/wup IO control of OSAL_C_S32_IOCTRL_WUP_GET_STARTTYPE with the following possible values should be used instead :

                      - DEV_WUP_C_U8_STARTTYPE_COLDSTART
                      - DEV_WUP_C_U8_STARTTYPE_RESTART_SCC
                      - DEV_WUP_C_U8_STARTTYPE_WARMSTART
                      - DEV_WUP_C_U8_STARTTYPE_RESTART_CPU
                    */
                  ETG_TRACE_USR4( ( "spm_tclOsalProxy::bGetWakeupReason: WakeupReason from dev_wup = '%u'", ETG_ENUM( DEV_WUP_u8WakeupReason, u32WakeupReason ) ) );

                  /*
                     if (u32WakeupReason == DEV_WUP_C_U8_WAKEUP_REASON_RESET){
                    * pu32WakeUpReason = ( tU32 ) spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_RESTART;
                     } else */
                  switch ( u32WakeupReason ){
                     case DEV_WUP_C_U8_WAKEUP_REASON_POWER_ON_RESET:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_POWER_ON;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_ON_TIPPER:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_ON_TIPPER;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_CAN:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_CAN;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_MOST:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_MOST;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_IGN_PIN:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_IGNITION;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_TEL_MUTE:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_PHONE_MUTE;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_DEBUG_WAKEUP:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_DEBUG_WAKEUP;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_RTC_WAKEUP:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_RTC;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_CD_EJECT_DETECTED:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_EJECT;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_CD_INSERT_DETECTED:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_INSERT;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_S_CONTACT_WAKEUP:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_ACCESSORY;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_CELLNETWORK_WAKEUP:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_CELLNETWORK;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_CAN2_WAKEUP:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_CAN2;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_CAN3_WAKEUP:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_CAN3;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_CAN4_WAKEUP:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_CAN4;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_ODOMETER_WAKEUP:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_ODOMETER;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_LIN_WAKEUP:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_LIN;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_EXTERNAL_PIN_WAKEUP:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_EXTERNAL_GPIO;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_UART:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_UART;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_UART2:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_UART2;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_USB:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_USB;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_BLUETOOTH:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_BLUETOOTH;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_ACCELERATOR_SENSOR:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_ACCELLERATOR_SENSOR;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_EXTERNAL_GPIO2:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_EXTERNAL_GPIO2;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_EXTERNAL_GPIO3:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_EXTERNAL_GPIO3;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_EXTERNAL_GPIO4:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_EXTERNAL_GPIO4;
                        break;
                     }

                     case DEV_WUP_C_U8_WAKEUP_REASON_ILLUMINATION:
                     {
                        * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_ILLUMINATION;
                        break;
                     }

                     // case DEV_WUP_C_U8_WAKEUP_REASON_RESET) {
                     // *pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_DOOR_OPEN;
                     // break;
                     // }
                     default:
                     {
                        * pu32WakeUpReason = ( tU32 ) spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_RESTART;
                     }
                  } // switch
                  _u32WakeupType = * pu32WakeUpReason;
                  ETG_TRACE_USR1( ( "spm_tclOsalProxy::bGetWakeupReason(): Wakeup reason detected: %u", ETG_ENUM( SPM_WAKEUP_REASON, * pu32WakeUpReason ) ) );

                  bReturn        = TRUE;
               }
            #else // ifndef VARIANT_S_FTR_ENABLE_SIMULATE_WAKEUP_REASON
               OSAL_tIODescriptor fd = OSAL_ERROR;
               if ( ( fd = OSAL_IOOpen( "/dev/ffs/spm_wup_restart", OSAL_EN_READONLY ) ) != OSAL_ERROR ){
                  * pu32WakeUpReason = ( tU32 ) spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_RESTART;
               } else if ( ( fd = OSAL_IOOpen( "/dev/ffs/spm_wup_pwron", OSAL_EN_READONLY ) ) != OSAL_ERROR ){
                  * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_POWER_ON;
               } else if ( ( fd = OSAL_IOOpen( "/dev/ffs/spm_wup_eject", OSAL_EN_READONLY ) ) != OSAL_ERROR ){
                  * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_EJECT;
               } else if ( ( fd = OSAL_IOOpen( "/dev/ffs/spm_wup_ignition", OSAL_EN_READONLY ) ) != OSAL_ERROR ){
                  * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_IGNITION;
               } else if ( ( fd = OSAL_IOOpen( "/dev/ffs/spm_wup_cd", OSAL_EN_READONLY ) ) != OSAL_ERROR ){
                  * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_INSERT;
               } else if ( ( fd = OSAL_IOOpen( "/dev/ffs/spm_wup_tipper", OSAL_EN_READONLY ) ) != OSAL_ERROR ){
                  * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_ON_TIPPER;
               } else if ( ( fd = OSAL_IOOpen( "/dev/ffs/spm_wup_door", OSAL_EN_READONLY ) ) != OSAL_ERROR ){
                  * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_DOOR_OPEN;
               } else if ( ( fd = OSAL_IOOpen( "/dev/ffs/spm_wup_can", OSAL_EN_READONLY ) ) != OSAL_ERROR ){
                  * pu32WakeUpReason = spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_CAN;
               } else {
                  * pu32WakeUpReason = ( tU32 ) spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_RESTART;
               }
               if ( fd != OSAL_ERROR ){
                  OSAL_s32IOClose( fd );
               }
               _u32WakeupType = * pu32WakeUpReason;

               ETG_TRACE_USR1( ( "spm_tclOsalProxy::bGetWakeupReason(): Wakeup reason detected: %u", ETG_ENUM( SPM_WAKEUP_REASON, * pu32WakeUpReason ) ) );

               bReturn        = TRUE;
            #endif // ifndef VARIANT_S_FTR_ENABLE_SIMULATE_WAKEUP_REASON

         }
      } else {
         * pu32WakeUpReason = _u32WakeupType;
         bReturn            = TRUE;
      }

      return( bReturn );
   } // bGetWakeupReason

   tBool spm_tclOsalProxy::bStartWatchdog( tU32 u32WatchdogInterval ){
/*!
  * \fn
  *  \brief
  *    .
  *
  *  \param
  *     tU32 u32WatchdogInterval: Set to "0" to Disable Watchdog
  *
  *  \version
  *    1.0   - Initial
  ******
  */
      (tVoid)u32WatchdogInterval;

      tBool bReturn = FALSE;

      #if OSAL_CONF == OSAL_LINUX
         #ifndef LCM_UNIT_TESTS

            sd_notifyf( 0, "READY=1\n"                       // Tells the init system that daemon startup is finished.
                                                             // This is only used by systemd if the service definition file has Type=notify set.
                           "STATUS=Processing requests...\n" // Passes a single-line status string back to the init system that describes the daemon state.
                                                             // This is free-form and can be used for various purposes
                           "MAINPID=%lu\nPARENTID=%lu",
                        (unsigned long)getpid( ),
                        (unsigned long)getppid( ) );
            ETG_TRACE_USR1( ( "spm_tclOsalProxy::bStartWatchdog(): send READY=1 ........." ) );
         #endif
      #endif

      return( bReturn );
   } // bStartWatchdog

   tBool spm_tclOsalProxy::bTriggerWatchdog( ){
/*!
  * \fn
  *  \brief
  *    .
  *  \version
  *    1.0   - Initial
  ******
  */
      tBool bReturn = FALSE;

      if ( bIsShutdownActive( ) ){
         ETG_TRACE_USR4( ( "spm_tclOsalProxy::bTriggerWatchdog(): ShutdownSystem active, skipping startup-counter access." ) );
      } else {
         #if OSAL_CONF == OSAL_LINUX
            #ifndef LCM_UNIT_TESTS
               sd_notify( 0, "WATCHDOG=1" );
               ETG_TRACE_USR4( ( "spm_tclOsalProxy::bTriggerWatchdog(): send WATCHDOG=1 ........." ) );
            #endif
         #endif
      }

      return( bReturn );
   } // bTriggerWatchdog

   tBool spm_tclOsalProxy::bUbatSense( tU16 *u16UbatSense ){
/*!
  * \fn
  *  \brief
  *    This method reads the board voltage.
  *
  *  \param [in] u16UbatSense: Battery ADC Value
  *
  *  \version
  *    1.0   - Initial
  ******
  */
      tBool bReturn = FALSE;

      * u16UbatSense = 0;
      if ( OSAL_ERROR == _hDeviceVolt ){
         _hDeviceVolt = OSAL_IOOpen( OSAL_C_STRING_DEVICE_VOLT, OSAL_EN_READWRITE );
         if ( OSAL_ERROR == _hDeviceVolt ){
            ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected in OSAL_IOOpen of DEV_VOLT !!!!!!" ) );
         }
      }

      if ( OSAL_ERROR != _hDeviceVolt ){
         // after the device has been opend use IOControl read board voltage.
         #ifdef OSAL_C_S32_IOCTRL_VOLT_GET_BOARD_VOLTAGE
            tU16 u16AdcValue = 0; // uselesss init parameter, just for LINT

            // Get the Board Voltage
            if ( OSAL_s32IOControl( _hDeviceVolt, OSAL_C_S32_IOCTRL_VOLT_GET_BOARD_VOLTAGE, ( intptr_t )&u16AdcValue ) == OSAL_OK ){
               * u16UbatSense = (tU16)u16AdcValue;
               ETG_TRACE_USR4( ( "spm_tclOsalProxy::bUbatSense Battery ADC Value %d", * u16UbatSense ) );
               bReturn        = TRUE;
            }
         #else
            (tVoid)u16UbatSense;
            ETG_TRACE_FATAL( ( "IO-Control OSAL_C_S32_IOCTRL_VOLT_GET_BOARD_VOLTAGE is used but not defined" ) );
         #endif // ifdef OSAL_C_S32_IOCTRL_VOLT_GET_BOARD_VOLTAGE


      }

      return( bReturn );
   } // bUbatSense

   tBool spm_tclOsalProxy::bCurrentSense( tU16 *u16CurrentSense ){
/*!
  * \fn
  *  \brief
  *    .
  *  \version
  *    1.0   - Initial
  ******
  */
      tBool bReturn = FALSE;

      * u16CurrentSense = 0;

      if ( OSAL_ERROR == _hDeviceVolt ){
         _hDeviceVolt = OSAL_IOOpen( OSAL_C_STRING_DEVICE_VOLT, OSAL_EN_READWRITE );
         if ( OSAL_ERROR == _hDeviceVolt ){
            ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected in OSAL_IOOpen of DEV_VOLT !!!!!!" ) );
         }
      }

      if ( OSAL_ERROR != _hDeviceVolt ){
         // after the device has been opend use IOControl to read current consumption.
         #ifdef OSAL_C_S32_IOCTRL_VOLT_GET_BOARD_CURRENT
            tU32 u32AdcValue = 0; // uselesss init parameter, just for LINT

            if ( OSAL_s32IOControl( _hDeviceVolt, OSAL_C_S32_IOCTRL_VOLT_GET_BOARD_CURRENT, ( intptr_t )&u32AdcValue ) == OSAL_OK ){
               * u16CurrentSense = (tU16)u32AdcValue;
               ETG_TRACE_USR4( ( "spm_tclOsalProxy::bCurrentSense : Current Value %d", * u16CurrentSense ) );
               bReturn           = TRUE;
            }
         #else
            (tVoid)u16CurrentSense;
            ETG_TRACE_FATAL( ( "IO-Control OSAL_C_S32_IOCTRL_VOLT_GET_BOARD_CURRENT is used but not defined" ) );
         #endif // ifdef OSAL_C_S32_IOCTRL_VOLT_GET_BOARD_CURRENT
      }

      return( bReturn );
   } // bCurrentSense

   tBool spm_tclOsalProxy::bSetWakeupConfig( tU32 u32WakeupConfig ){
/*!
  * \fn
  *  \brief function to set certain wakeup config
  *  \param [in] u32WakeupConfig: wakeup config value to be enabled.
  *  \return tBool: True if successfuly set.
  *
  *  \details This method is used to configure certain wake-up reasons to be considered for the initial wake-up of the system.
  ******
  */
      tBool bReturn = TRUE;

      ETG_TRACE_USR4( ( "spm_tclOsalProxy::bSetWakeupConfig() wakeup config value %d",
                        u32WakeupConfig ) );

      if ( bGetDevWupHandle( "spm_tclOsalProxy::bSetWakeupConfig" ) ){
         #ifdef OSAL_C_S32_IOCTRL_WUP_CONFIGURE_WAKEUP_REASONS
            if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_CONFIGURE_WAKEUP_REASONS, (intptr_t)u32WakeupConfig ) != OSAL_OK ){
               ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bSetWakeupConfig() IO-Control OSAL_C_S32_IOCTRL_WUP_CONFIGURE_WAKEUP_REASONS Failed !!!!" ) );
               bReturn = FALSE;
            }
         #else
            (tVoid)u32WakeupConfig;
            ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bSetWakeupConfig() IO-Control OSAL_C_S32_IOCTRL_WUP_CONFIGURE_WAKEUP_REASONS is used but not defined" ) );
         #endif
      } else {
         ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bSetWakeupConfig() no handle to dev_wup available" ) );
         bReturn = FALSE;
      }
      return( bReturn );
   } // bSetWakeupConfig

   tBool spm_tclOsalProxy::bGetWupVersion( tU32 *pu32Version ){
      /*!
        * \fn
        *  \brief gets the version of the IF of the /dev/wup device
        *  \return tBool: True if successfuly set.
        *
        *  \details currently not supported
        ******
        */
      tBool bReturn = TRUE;

      if ( bGetDevWupHandle( "spm_tclOsalProxy::bGetWupVersion" ) ){
         #ifdef OSAL_C_S32_IOCTRL_VERSION
            if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_VERSION, (intptr_t)pu32Version ) != OSAL_OK ){
               bReturn = FALSE;
            }
         #else
            (tVoid)pu32Version;
            ETG_TRACE_FATAL( ( "IO-Control OSAL_C_S32_IOCTRL_VERSION is used but not defined" ) );
         #endif
      }
      return( bReturn );

   } // bGetWupVersion

   tBool spm_tclOsalProxy::bGetUtcTime( OSAL_trTimeDate *pTimeDate ){

      tBool bReturn   = FALSE;
      tS32  s32Retval = OSAL_s32ClockGetTime( pTimeDate );

      if ( s32Retval != OSAL_OK ){
         OSAL_tIODescriptor hRTC = OSAL_IOOpen( OSAL_C_STRING_DEVICE_RTC, OSAL_EN_READWRITE );
         if ( OSAL_ERROR != hRTC ){
            OSAL_trRtcRead rRtcRead;
            rRtcRead.RegisterForRtcTime.updateRate        = 0;
            rRtcRead.RegisterForRtcTime.registerForNotify = OSAL_RTC_NOTIFY_ON_GPS_TIME;
            if ( OSAL_s32IORead( hRTC, ( tPS8 ) & rRtcRead, sizeof( rRtcRead ) ) != OSAL_ERROR ){
               * pTimeDate = rRtcRead.RtcTimes.gpsUtcTime;
               bReturn     = TRUE;
            } else {
               bReturn = FALSE;
            }
            OSAL_s32IOClose( hRTC );
         } else {
            bReturn = FALSE;
         }
      } else {
         // OSAL_s32ClockGetTime succeeded
         bReturn = TRUE;
      }
      if ( !bReturn ){
               ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bGetUtcTime(): Get OSAL systemtime." ) );
      }
      return( bReturn );
   } // bGetUtcTime

   tU32 spm_tclOsalProxy::u32GetAdrResetCount( tVoid ){
      tU32                  u32ResetCount   = 0;

      #ifdef SPM_FEATURE_ENABLE_ADR3
         OSAL_tIODescriptor hAdr3CtrlDevice = OSAL_IOOpen( "/dev/adr3ctrl", OSAL_EN_READWRITE );
         if ( OSAL_ERROR != hAdr3CtrlDevice ){
            if ( OSAL_s32IOControl( hAdr3CtrlDevice, OSAL_C_S32_IOCTRL_ADR3CTRL_READ_CLEAR_RESET_COUNT, ( intptr_t )&u32ResetCount ) == OSAL_ERROR ){
               ETG_TRACE_ERRMEM( ( "SPM: Get ADR reset count failed." ) );
            }
               OSAL_s32IOClose( hAdr3CtrlDevice );
         }
      #endif
      return( u32ResetCount );
   }

   tBool spm_tclOsalProxy::bGetStartType( tU8 *pU8Starttype ){
/*!
  * \fn
  *  \brief
  *    This method gets the start type from dev_wup
  *
  *  \param [in] pU8Starttype: will be updated as per the start reason.
  *  \param [out] tBool: True if valid start reason acquired from dev_wup.
  *
  *  \version
  *    1.0   - Initial
  ******
  */

      tBool bReturn = TRUE;

      * pU8Starttype = SPM_U8_STARTTYPE_UNKNOWN;

      if ( bGetDevWupHandle( "spm_tclOsalProxy::bGetStartType" ) ){
         tU8 u8StartupType = DEV_WUP_C_U8_STARTTYPE_COLDSTART; // uselesss init parameter, just for LINT
         if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_GET_STARTTYPE, ( intptr_t )&u8StartupType ) != OSAL_ERROR ){
            ETG_TRACE_USR4( ( "spm_tclOsalProxy::Starttype detected to be:%u", ETG_ENUM( SPM_DEV_WUP_STARTTYPE, u8StartupType ) ) );

            switch ( u8StartupType ){
               case DEV_WUP_C_U8_STARTTYPE_COLDSTART:
                  * pU8Starttype = SPM_U8_STARTTYPE_COLDSTART;
                  break;

               case DEV_WUP_C_U8_STARTTYPE_RESTART_VCC:
                  * pU8Starttype = SPM_U8_STARTTYPE_RESTART_SCC;
                  break;

               case DEV_WUP_C_U8_STARTTYPE_WARMSTART:
                  * pU8Starttype = SPM_U8_STARTTYPE_WARMSTART;
                  break;

               case DEV_WUP_C_U8_STARTTYPE_RESTART_CPU:
                  * pU8Starttype = SPM_U8_STARTTYPE_RESTART_AP;
                  break;

               default:
               {
                  bReturn = FALSE;
                  ETG_TRACE_ERRMEM( ( "Invalid start Type from dev_wup namely %d", u8StartupType ) );
               }
            } // switch

         } else {
            bReturn = FALSE;
            ETG_TRACE_FATAL( ( "IOCTRL to dev_wup failed" ) );
         }
      } else {
         bReturn = FALSE;
         ETG_TRACE_FATAL( ( "SPM: !!!!!! Error detected,no handle to dev_wup available !!!!!!" ) );
      }

      return( bReturn );
   } // bGetStartType

   tBool spm_tclOsalProxy::bResetProcessor( tU8  u8Processor,
                                            tU8  u8ResetMode,
                                            tU16 u16ResetReason ){
/*!
  * \fn
  *  \brief
  *    This method provides the functionality to reset a processor
  *
  *  \param [in] u8Processor: Processor to reset. Right now, the processors(iMX, v850(SCC) and ENTIRE_SYSTEM can be reset using SPMS_SPM_C_PROC_RESET_REQUEST.
  *                           For other processors, the reset(using SPMS_SPM_C_PROC_RESET_REQUEST) is disabled at v850.
  *  \param [in] u8ResetMode: Reset Mode.
  *  \param [in] u16ResetReason: Reset Reason.
  *  \param [out] tBool:False if reset can not be performed.
  *
  *  \version
  *    1.0   - Initial
  ******
  */
      tBool bReturn = FALSE;

      if ( bGetDevWupHandle( "spm_tclOsalProxy::bResetProcessor" ) ){
         DEV_WUP_trResetProcessorInfo tResetProcessorInfo;
         tResetProcessorInfo.u8Processor        = u8Processor;
         tResetProcessorInfo.u8ResetMode        = u8ResetMode;
         tResetProcessorInfo.u16ResetReason     = u16ResetReason;
         tResetProcessorInfo.u16ResetDurationMs = SPM_DEVWUP_RESET_PROCESSOR_DURATION;
         if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_RESET_PROCESSOR, ( intptr_t )&tResetProcessorInfo ) != OSAL_ERROR ){
            ETG_TRACE_USR1( ( "spm_tclOsalProxy::bResetProcessor(): --> OSAL_C_S32_IOCTRL_WUP_RESET_PROCESSOR\nProcessor=%d\nResetMode=%d\nResetReason=%d",
                              ETG_ENUM( DEV_WUP_u8Processor,      u8Processor ),
                              ETG_ENUM( DEV_WUP_u8ResetMode,      u8ResetMode ),
                              ETG_ENUM( DEV_WUP_u16ResetReasonAP, u16ResetReason )
                              ) );
            bReturn = TRUE;
         } else {
            ETG_TRACE_FATAL( ( "spm_tclOsalProxy::bResetProcessor(): --> FAILED to Reset Processor\nProcessor=%d\nResetMode=%d\nResetReason=%d",
                               ETG_ENUM( DEV_WUP_u8Processor,      u8Processor ),
                               ETG_ENUM( DEV_WUP_u8ResetMode,      u8ResetMode ),
                               ETG_ENUM( DEV_WUP_u16ResetReasonAP, u16ResetReason )
                               ) );
         }
      } else {
         ETG_TRACE_FATAL( ( "SPM: !!!!!! Error detected,no handle to dev_wup available !!!!!!" ) );
      }

      return( bReturn );
   } // bResetProcessor

   tBool spm_tclOsalProxy::bShutdownSystem( tU32 u32ShutdownOption ){
/*!
  * \fn
  *  \brief shutdown the system. This function should never return
  *
  *  \param Shutdown option (reset type)
  *  \note Some changes are made to make it unit testable. (e.g. endlessloop in the end has been removed)
  *  \version
  *    1.0   - Initial
  ******
  */
      tBool bReturn         = TRUE;
      tBool bWaitForReset	= FALSE;

      tU32  u32DevSpmOption = DEV_WUP_SHUTDOWN_NORMAL;

      _bShutdownActive = TRUE;

      #ifdef OSAL_C_S32_IOCTRL_WUP_SHUTDOWN
         if(!bGetDevWupHandle( "spm_tclOsalProxy::bShutdownSystem" )) {
            return FALSE;
         }
         tBool bDoShutdown = FALSE;
         switch ( u32ShutdownOption ){

            case SPM_C_S32_SYSTEMINIT_SWITCH_OFF:
            case SPM_C_S32_SYSTEMINIT_SWITCH_OFF_IMMEDIATELY:
               ETG_TRACE_USR4( ( "spm_tclOsalProxy::bShutdownSystem(SWITCH_OFF)" ) );
               bDoShutdown     = TRUE;

               /* Workaround: Switch off via EMERGENCY OFF (via WUP) instead of proper platform shutdown
                  (which is not working currently)
                 */
               u32DevSpmOption = DEV_WUP_SHUTDOWN_NORMAL; // DEV_SPM_C_S32_SYSTEM_OFF;
               break;

            case SPM_C_S32_SYSTEMINIT_RESTART_ALTSW_TESTMANAGER:
            {
               DEV_WUP_trResetProcessorInfo rResetProcessorInfo =
               {
                  DEV_WUP_C_U8_APPLICATION_PROCESSOR,
                  DEV_WUP_C_U8_RESET_MODE_UNLOGGED,
                  DEV_WUP_C_U16_RESET_REASON_SWITCH_BOOT_MODE_USB,
                  SPM_DEVWUP_RESET_PROCESSOR_DURATION
               };

               if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_RESET_PROCESSOR, ( intptr_t )&rResetProcessorInfo ) == OSAL_OK ){
                  ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bShutdownSystem(): --> INTENTIONAL RESET - this is just for documentation and no error" ) );
                  ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bShutdownSystem(): --> SPM_C_S32_SYSTEMINIT_RESTART_ALTSW_TESTMANAGER" ) );
               } else {
                  ETG_TRACE_FATAL( ( "spm_tclOsalProxy::bShutdownSystem(): --> FAILED to Reboot with option SPM_C_S32_SYSTEMINIT_RESTART_ALTSW_TESTMANAGER" ) );
               }
               bWaitForReset = TRUE;
            }
            break;

            case SPM_C_S32_SYSTEMINIT_RESET_LATE_WAKEUP:
            {
               DEV_WUP_trResetProcessorInfo rResetProcessorInfo =
               {
                  DEV_WUP_C_U8_APPLICATION_PROCESSOR,
                  DEV_WUP_C_U8_RESET_MODE_UNLOGGED,
                  DEV_WUP_C_U16_RESET_REASON_LATE_WAKEUP,
                  SPM_DEVWUP_RESET_PROCESSOR_DURATION
               };

               if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_RESET_PROCESSOR, ( intptr_t )&rResetProcessorInfo ) == OSAL_OK ){
                  ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bShutdownSystem(): --> INTENTIONAL RESET - this is just for documentation and no error" ) );
                  ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bShutdownSystem(): --> SPM_C_S32_SYSTEMINIT_RESET_LATE_WAKEUP" ) );
               } else {
                  ETG_TRACE_FATAL( ( "spm_tclOsalProxy::bShutdownSystem(): --> FAILED to Reboot with option SPM_C_S32_SYSTEMINIT_RESET_LATE_WAKEUP" ) );
               }
               bWaitForReset = TRUE;
            }
            break;

            case SPM_C_S32_SYSTEMINIT_RESET_ECU_API:
            {
               DEV_WUP_trResetProcessorInfo rResetProcessorInfo =
               {
                  DEV_WUP_C_U8_ENTIRE_SYSTEM,
                  DEV_WUP_C_U8_RESET_MODE_UNLOGGED,
                  DEV_WUP_C_U16_RESET_REASON_DIAGNOSIS,
                  SPM_DEVWUP_RESET_PROCESSOR_DURATION
               };

               if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_RESET_PROCESSOR, ( intptr_t )&rResetProcessorInfo ) == OSAL_OK ){
                  ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bShutdownSystem(): --> INTENTIONAL RESET - this is just for documentation and no error" ) );
                  ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bShutdownSystem(): --> SPM_C_S32_SYSTEMINIT_RESET_ECU_API (DEV_WUP_C_U16_RESET_REASON_DIAGNOSIS)" ) );
               } else {
                  ETG_TRACE_FATAL( ( "spm_tclOsalProxy::bShutdownSystem(): --> FAILED to Reboot ECU with option SPM_C_S32_SYSTEMINIT_RESET_ECU_API (DEV_WUP_C_U16_RESET_REASON_DIAGNOSIS)" ) );
               }
               bWaitForReset = TRUE;
            }
            break;

            case SPM_C_S32_SYSTEMINIT_RESET_APP_API:
            {
               DEV_WUP_trResetProcessorInfo rResetProcessorInfo =
               {
                  DEV_WUP_C_U8_APPLICATION_PROCESSOR,
                  DEV_WUP_C_U8_RESET_MODE_UNLOGGED,
                  DEV_WUP_C_U16_RESET_REASON_DIAGNOSIS,
                  SPM_DEVWUP_RESET_PROCESSOR_DURATION
               };

               if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_RESET_PROCESSOR, ( intptr_t )&rResetProcessorInfo ) == OSAL_OK ){
                  ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bShutdownSystem(): --> INTENTIONAL RESET - this is just for documentation and no error" ) );
                  ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bShutdownSystem(): --> SPM_C_S32_SYSTEMINIT_RESET_APP_API (DEV_WUP_C_U16_RESET_REASON_DIAGNOSIS)" ) );
               } else {
                  ETG_TRACE_FATAL( ( "spm_tclOsalProxy::bShutdownSystem(): --> FAILED to Reboot APP with option SPM_C_S32_SYSTEMINIT_RESET_APP_API (DEV_WUP_C_U16_RESET_REASON_DIAGNOSIS)" ) );
               }
               bWaitForReset = TRUE;
            }
            break;

            case SPM_C_S32_SYSTEMINIT_RESET_SCC_API:
            {
               DEV_WUP_trResetProcessorInfo rResetProcessorInfo =
               {
                  DEV_WUP_C_U8_SYSTEM_COMMUNICATION_CONTROLLER,
                  DEV_WUP_C_U8_RESET_MODE_UNLOGGED,
                  DEV_WUP_C_U16_RESET_REASON_DIAGNOSIS,
                  SPM_DEVWUP_RESET_PROCESSOR_DURATION
               };

               if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_RESET_PROCESSOR, ( intptr_t )&rResetProcessorInfo ) == OSAL_OK ){
                  ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bShutdownSystem(): --> INTENTIONAL RESET - this is just for documentation and no error" ) );
                  ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bShutdownSystem(): --> SPM_C_S32_SYSTEMINIT_RESET_SCC_API (DEV_WUP_C_U16_RESET_REASON_DIAGNOSIS)" ) );
               } else {
                  ETG_TRACE_FATAL( ( "spm_tclOsalProxy::bShutdownSystem(): --> FAILED to Reboot SCC with option SPM_C_S32_SYSTEMINIT_RESET_SCC_API (DEV_WUP_C_U16_RESET_REASON_DIAGNOSIS)" ) );
               }
               bWaitForReset = TRUE;
            }
            break;

            case SPM_C_S32_SYSTEMINIT_COLDSTART:
            case SPM_C_S32_SYSTEMINIT_WARMSTART:
            case SPM_C_S32_SYSTEMINIT_RESET_SYSTEM:
            case SPM_C_S32_SYSTEMINIT_RESET_WARMSTART:
            default:
            {
               ETG_TRACE_USR4( ( "spm_tclOsalProxy::bShutdownSystem()" ) );
               DEV_WUP_trResetProcessorInfo rResetProcessorInfo =
               {
                  DEV_WUP_C_U8_APPLICATION_PROCESSOR,
                  DEV_WUP_C_U8_RESET_MODE_UNLOGGED,
                  DEV_WUP_C_U16_RESET_REASON_UNSPECIFIED,
                  SPM_DEVWUP_RESET_PROCESSOR_DURATION
               };

               if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_RESET_PROCESSOR, ( intptr_t )&rResetProcessorInfo ) == OSAL_OK ){
                  ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bShutdownSystem(): --> INTENTIONAL RESET - this is just for documentation and no error" ) );
                  ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bShutdownSystem(): --> OSAL_C_S32_IOCTRL_WUP_RESET_PROCESSOR" ) );
               } else {
                  ETG_TRACE_FATAL( ( "spm_tclOsalProxy::bShutdownSystem(): --> FAILED to Reboot" ) );
               }
               bWaitForReset = TRUE;
            }
		    break;
         } // switch

         if (TRUE == bWaitForReset)
		 {
        	 // reset might take a while, but if the reset does not happen,
        	 // do it via FATAL_ASSERT
             OSAL_s32ThreadWait( 5000 );
             SPM_RESET_VIA_ASSERT;
		 }

         if ( ( OSAL_ERROR != _hDeviceWup ) && bDoShutdown ){
            ETG_TRACE_FATAL( ( "Calling OSAL_C_S32_IOCTRL_WUP_SHUTDOWN ..." ) );
            tS32 s32ErrorCode = OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_SHUTDOWN, (intptr_t)u32DevSpmOption );

            if ( s32ErrorCode != OSAL_OK ){
               ETG_TRACE_ERRMEM( ( "Call of OSAL_C_S32_IOCTRL_WUP_SHUTDOWN failed with error = 0x%08x", OSAL_u32ErrorCode() ) );
            }

            // Give shutdown a chance and use a quite long delay here to be sure that the SCC performed all actions prior to the reset or power off of the APP.
            // If this delay is too short, the SCC might not have already reset or powered off the APP and then we unintenionally perform a restart.
            OSAL_s32ThreadWait( 2000 );

            // We don't expect to reach this code line because we should have already been switched off here. => Force a self reset.
            ETG_TRACE_ERRMEM( ( "Target failed to shut down => Trigger self reset" ) );

            DEV_WUP_trResetProcessorInfo rResetProcessorInfo =
            {
               DEV_WUP_C_U8_APPLICATION_PROCESSOR,
               DEV_WUP_C_U8_RESET_MODE_UNLOGGED,
               DEV_WUP_C_U16_RESET_REASON_UNSPECIFIED,
               SPM_DEVWUP_RESET_PROCESSOR_DURATION
            };

            (tVoid) OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_RESET_PROCESSOR, ( intptr_t )&rResetProcessorInfo );

            OSAL_s32ThreadWait( 100 ); // Give reset io-control a chance ...
            ETG_TRACE_ERRMEM( ( "Self reset via RESET IO-CONTROL failed => Trigger reset via ASSERT" ) );

            SPM_RESET_VIA_ASSERT;
         } else {
            ETG_TRACE_ERRMEM( ( "(OSAL_ERROR != _hDeviceWup(%d)) && (bDoShutdown(%d))", (tInt)_hDeviceWup, bDoShutdown ) );
         }
      #else // ifdef OSAL_C_S32_IOCTRL_WUP_SHUTDOWN
         (tVoid)u32ShutdownOption;
         bReturn       = FALSE;
         ETG_TRACE_USR4( ( "WARNING: Can not switch off, OSAL_C_S32_IOCTRL_WUP_SHUTDOWN define not supported" ) );
      #endif // ifdef OSAL_C_S32_IOCTRL_WUP_SHUTDOWN

      _bShutdownActive = FALSE;
      return( bReturn );
   } // bShutdownSystem

// *******************************************************************************************************************
// *******************************************************************************************************************
// ****************  PRAM HELPER FUNCTIONS                                                      **********************
// *******************************************************************************************************************
// *******************************************************************************************************************

// *******************************************************************************************************************
// ****************  PRAM HELPER FUNCTIONS for LCM owned data                                  **********************
// *******************************************************************************************************************
   tBool spm_tclOsalProxy::bSetSRamData( tU32 u32Type,
                                         tU32 u32Data ){
/*!
  * \fn
  *  \brief
  *    .
  *  \version
  *    1.0   - Initial
  ******
  */
      tBool bReturn = FALSE;

      if ( OSAL_ERROR == _hDevicePramLcm ){
         _hDevicePramLcm = OSAL_IOOpen( "/dev/pram/fc_spm", OSAL_EN_READWRITE );
         if ( OSAL_ERROR == _hDevicePramLcm ){
            ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected in OSAL_IOOpen of DEV_PRAM for LCM !!!!!!" ) );
         }
      }

      if ( OSAL_ERROR != _hDevicePramLcm ){
         // after the device has been opend use IOControl to hand over callback pointer.
         #ifdef OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK
            tS32 ret = OSAL_ERROR;
            ret = OSAL_s32IOControl( _hDevicePramLcm, OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK, ( intptr_t )u32Type );
            if ( OSAL_ERROR == ret ){
                  ETG_TRACE_USR4( ( "PRAM_SEEK returns error, pram must be uninialized." ) );
            } else {
               ret = OSAL_s32IOWrite( _hDevicePramLcm, ( tPS8 ) & u32Data, sizeof( u32Data ) );
               if ( ( OSAL_ERROR != ret ) && ( sizeof( u32Data ) == ret ) ){
                  // ETG_TRACE_USR4(("PRAM_WRITE OK!"));
                  bReturn = TRUE;
               } else {
                  ETG_TRACE_USR4( ( "SRAM_WRITE ERROR! Error=0x%08x", ret ) );
               }
            }
         #else // ifdef OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK
            (tVoid)u32Type;
            (tVoid)u32Data;
            ETG_TRACE_FATAL( ( "IO-Control OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK is used but not defined" ) );
         #endif // ifdef OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK
      }

      return( bReturn );
   } // bSetSRamData

   tU32 spm_tclOsalProxy::u32GetSRamData( tU32 u32Type ){
/*!
  * \fn
  *  \brief
  *    .
  *  \version
  *    1.0   - Initial
  ******
  */

      tU32 u32SRamData = 0xffffffff; // default value if error

      if ( OSAL_ERROR == _hDevicePramLcm ){
         _hDevicePramLcm = OSAL_IOOpen( "/dev/pram/fc_spm", OSAL_EN_READWRITE );
         if ( OSAL_ERROR == _hDevicePramLcm ){
               ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected in OSAL_IOOpen of DEV_PRAM for LCM !!!!!!" ) );
         }
      }

      if ( OSAL_ERROR != _hDevicePramLcm ){
         // after the device has been opend use IOControl to hand over callback pointer.
         #ifdef OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK
            tS32 ret = OSAL_ERROR;
            ret = OSAL_s32IOControl( _hDevicePramLcm, OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK, ( intptr_t )u32Type );
            if ( OSAL_ERROR == ret ){
               ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!" ) );
            } else {
               ret = OSAL_s32IORead( _hDevicePramLcm, ( tPS8 ) & u32SRamData, sizeof( u32SRamData ) );
               if ( ( OSAL_ERROR == ret ) || ( sizeof( u32SRamData ) != ret ) ){
                  u32SRamData = 0xffffffff; // reset to default value if error
               }
            }
         #else // ifdef OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK
            ETG_TRACE_FATAL( ( "IO-Control OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK is used but not defined" ) );
            (tVoid)u32Type;
         #endif // ifdef OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK
      }

      return( u32SRamData );
   } // u32GetSRamData

// *******************************************************************************************************************
// ****************  PRAM HELPER FUNCTIONS for resetcounter                                   **********************
// *******************************************************************************************************************

/*!
   is the index in the structure of the PRAM area where the real reset counter is stored
  */
   #define SPM_RESET_COUNTER_PRAM_INDEX         1

/*!
   is the index in the structure of the PRAM area where the "inverse" reset counter is stored
  */
   #define SPM_RESET_COUNTER_PRAM_CHECK_INDEX   2

   tBool spm_tclOsalProxy::bSetResetCounter( tU8 u8Counter ){
/*!
  * \fn
  *  \brief
  *    Set the reset counter value from special PRAM section
  *
  *  \version
  *    1.0   - Initial
  ******
  */
      tBool bReturn = FALSE;

      if ( OSAL_ERROR == _hDevicePramResetCounter ){
         _hDevicePramResetCounter = OSAL_IOOpen( "/dev/pram/reset_counter", OSAL_EN_READWRITE );
         if ( OSAL_ERROR == _hDevicePramResetCounter ){
            ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected in OSAL_IOOpen of DEV_PRAM for ResetCounter !!!!!!" ) );
         }
      }
      if ( OSAL_ERROR != _hDevicePramResetCounter ){
         // after the device has been opend use IOControl to hand over callback pointer.
         #ifdef OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK
            tS32 ret = OSAL_ERROR;
            // write the real counter
            ret = OSAL_s32IOControl( _hDevicePramResetCounter, OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK, ( intptr_t )SPM_RESET_COUNTER_PRAM_INDEX );
            if ( OSAL_ERROR == ret ){
                  ETG_TRACE_USR4( ( "PRAM_SEEK returns error, pram must be uninialized." ) );
            } else {
               ret = OSAL_s32IOWrite( _hDevicePramResetCounter, ( tPS8 ) & u8Counter, sizeof( u8Counter ) );
               if ( ( OSAL_ERROR != ret ) && ( sizeof( u8Counter ) == ret ) ){
                  // ETG_TRACE_USR4(("PRAM_WRITE OK!"));
                  bReturn = TRUE;
               } else {
                  ETG_TRACE_USR4( ( "PRAM_WRITE ERROR! Error=0x%08x", ret ) );
               }
            }
            // write the check counter
            ret = OSAL_s32IOControl( _hDevicePramResetCounter, OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK, ( intptr_t )SPM_RESET_COUNTER_PRAM_CHECK_INDEX );
            if ( OSAL_ERROR == ret ){
                  ETG_TRACE_USR4( ( "PRAM_SEEK returns error, pram must be uninialized." ) );
            } else {
               tU8 u8Check = (tU8)(((tU8)0xFF) - u8Counter);
               ret = OSAL_s32IOWrite( _hDevicePramResetCounter, ( tPS8 ) & u8Check, sizeof( u8Check ) );
               if ( ( OSAL_ERROR != ret ) && ( sizeof( u8Check ) == ret ) ){
                  // ETG_TRACE_USR4(("PRAM_WRITE OK!"));
                  bReturn = TRUE;
               } else {
                  ETG_TRACE_USR4( ( "PRAM_WRITE ERROR! Error=0x%08x", ret ) );
               }
            }
         #else // ifdef OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK
            (tVoid)u8Counter;
            ETG_TRACE_FATAL( ( "IO-Control OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK is used but not defined" ) );
         #endif // ifdef OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK
      }

      return( bReturn );
   } // bSetResetCounter

   tU8 spm_tclOsalProxy::u8GetResetCounter( void ){
/*!
  * \fn
  *  \brief
  *   Read the reset counter value from special PRAM section, it checks if the checksum is valid to handle
  *   projects that do not have a valid booloader.
  *
  *  \version
  *    1.0   - Initial
  ******
  */

      tU8 u8ResetCounter      = 0x00; // default value if error
      tU8 u8ResetCounterCheck = 0xFF; // checksum (0xFF-u8ResetCounter)

      if ( OSAL_ERROR == _hDevicePramResetCounter ){
         _hDevicePramResetCounter = OSAL_IOOpen( "/dev/pram/reset_counter", OSAL_EN_READWRITE );
         if ( OSAL_ERROR == _hDevicePramResetCounter ){
                        ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected in OSAL_IOOpen of DEV_PRAM for ResetCounter !!!!!!" ) );
         }
      }
      if ( OSAL_ERROR != _hDevicePramResetCounter ){
         // after the device has been opend use IOControl to hand over callback pointer.
         #ifdef OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK
            tS32 ret = OSAL_ERROR;
            ret = OSAL_s32IOControl( _hDevicePramResetCounter, OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK, ( intptr_t )SPM_RESET_COUNTER_PRAM_INDEX );
            if ( OSAL_ERROR == ret ){
                        ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected reading PRAM for u8GetResetCounter!!!!!!" ) );
            } else {
               ret = OSAL_s32IORead( _hDevicePramResetCounter, ( tPS8 ) & u8ResetCounter, sizeof( u8ResetCounter ) );
               if ( ( OSAL_ERROR == ret )
                    || ( sizeof( u8ResetCounter ) != ret )
                    ){
                  u8ResetCounter = 0x00; // reset to default value if error
                        ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected reading PRAM for u8GetResetCounter!!!!!!\nErrorCode/ReadLenght = %u", ret ) );
               }
               // read checksum
               ret = OSAL_s32IOControl( _hDevicePramResetCounter, OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK, ( intptr_t )SPM_RESET_COUNTER_PRAM_CHECK_INDEX );
               if ( OSAL_ERROR == ret ){
                        ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected reading PRAM for u8GetResetCounterCheck!!!!!!" ) );
               } else {
                  ret = OSAL_s32IORead( _hDevicePramResetCounter, ( tPS8 ) & u8ResetCounterCheck, sizeof( u8ResetCounterCheck ) );
                  if ( ( OSAL_ERROR == ret )
                       || ( sizeof( u8ResetCounterCheck ) != ret )
                       ){
                     u8ResetCounter = 0x00; // reset to default value if error
                        ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected reading PRAM for u8GetResetCounterCheck!!!!!!\nErrorCode/ReadLenght = %u", ret ) );
                  } else {
                     // check the checksum
                     if ( u8ResetCounterCheck != ( 0xFF - u8ResetCounter ) ){
                        u8ResetCounter = 0x00;
                        ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected ResetCounterCheck did not match!!!!!!" ) );
                     }
                  }
               }
            }
         #else // ifdef OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK
            ETG_TRACE_FATAL( ( "IO-Control OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK is used but not defined" ) );
         #endif // ifdef OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK
      }

      return( u8ResetCounter );
   } // u8GetResetCounter

   tU8 spm_tclOsalProxy::u8GetApplicationMode( void ){
      /*!
        * \fn
        *  \brief
        *   Read the current application mode of the application processor. Currently supported are
        *     - NORMAL
        *     - DOWNLOAD
        *
        *  \version
        *    1.0   - Initial
        ******
        */

      tU8 u8ApplicationMode = DEV_WUP_C_U8_APPLICATION_MODE_NORMAL;

      #ifdef OSAL_C_S32_IOCTRL_WUP_GET_APPLICATION_MODE
         if ( bGetDevWupHandle( "spm_tclOsalProxy::bGetApplicationMode" ) ){
            if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_GET_APPLICATION_MODE, ( intptr_t )&u8ApplicationMode ) == OSAL_OK ){
               ETG_TRACE_USR3( ( "spm_tclOsalProxy::bGetApplicationMode(): read ApplicationMode from dev_wup: %u", ETG_ENUM( DEV_WUP_u8ApplicationMode, u8ApplicationMode ) ) );
            } else {
               ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error Reading of application mode failed!!!!!!" ) );
            }
         }
      #endif
      return( u8ApplicationMode );
   }

   tU8 spm_tclOsalProxy::u8GetOperationalState( void ){
      /*!
        * \fn
        *  \brief
        *   Read the current Operation Mode which received from SCC via dev_wup
        *   Currently supported are :
        *     - STATE_DIAG_0     ---------- Only start Diagnosis
        *     - STATE_DIAG_1     ---------- Start Diagnosis and stable apps
        *     - OP_STATE_NORMAL  ---------- Start default application set

        *
        *  \version
        *    1.0   - Initial - Initial Operation Mode Implementation
        ******
        */

      tU8 u8OperationalState = DEV_WUP_C_U8_OPERATIONAL_STATE_NORMAL;

      #ifdef OSAL_C_S32_IOCTRL_WUP_GET_OPERATIONAL_STATE
         if ( bGetDevWupHandle( "spm_tclOsalProxy::u8OperationalState" ) ){
            if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_GET_OPERATIONAL_STATE, ( intptr_t )&u8OperationalState ) == OSAL_OK ){
               ETG_TRACE_USR3( ( "spm_tclOsalProxy::u8OperationalState(): read OperationalState from dev_wup: %u", ETG_ENUM( DEV_WUP_u8OperationalState, u8OperationalState ) ) );
            } else {
               ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error Reading of Operation Mode failed!!!!!!" ) );
            }
         }
      #endif
      return( u8OperationalState );
   }

   tU8 spm_tclOsalProxy::u8GetDmVerityState( void ){
      /*!
        * \fn
        *  \brief
        *   Read the current DM Verity result state which received from SCC via dev_wup
        *   Currently supported are :
        *     - DM_VERITY_STATE_UNKNOWN     ---------- Default if message from SCC was not received yet
        *     - DM_VERITY_STATE_FAILED      ---------- DM-VERITY check failed
        *     - DM_VERITY_STATE_PASSED      ---------- DM-VERITY check passed
        *
        *  \version
        *    1.0   - Initial - Initial DM Verity Implementation
        ******
        */

      #ifdef OSAL_C_S32_IOCTRL_WUP_GET_DM_VERITY_STATE
         tU8 u8DmVerityState = DEV_WUP_C_U8_DM_VERITY_STATE_UNKNOWN;

         if ( bGetDevWupHandle( "spm_tclOsalProxy::u8GetDmVerityState" ) ){
            if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_GET_DM_VERITY_STATE, ( intptr_t )&u8DmVerityState ) == OSAL_OK ){
               ETG_TRACE_USR3( ( "spm_tclOsalProxy::u8GetDmVerityState(): read DmVerityState from dev_wup: %u", ETG_ENUM( DEV_WUP_u8DmVerityState, u8DmVerityState ) ) );
            } else {
               ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error Reading of DmVerityState failed!!!!!!" ) );
            }
         }
         return( u8DmVerityState );
      #else
         return( 0 );
      #endif
   }

   tBool spm_tclOsalProxy::bSetDmVerityResult( tU8 u8DmVerityResult ){
       /*!
         * \fn
         *  \brief
         *   Set the result of DM Verity check in Operational mode "DM_VERITY" to SCC via dev_wup
         *   Currently supported are :
         *     - DM_VERITY_CHECK_RESULT_UNKNOWN     ---------- Default if message from APP has not been received.
         *     - DM_VERITY_CHECK_RESULT_FAILED      ---------- DM-VERITY check failed
         *     - DM_VERITY_CHECK_RESULT_PASSED      ---------- DM-VERITY check passed
         *
         *  \version
         *    1.0   - Initial - Initial DM Verity Implementation
         ******
         */

      tBool bReturn = TRUE;

      ETG_TRACE_USR4( ( "spm_tclOsalProxy::bSetDmVerityResult() DM Verity check result value %d", u8DmVerityResult ) );

      #ifdef OSAL_C_S32_IOCTRL_WUP_SET_DM_VERITY_CHECK_RESULT
      if ( bGetDevWupHandle( "spm_tclOsalProxy::bSetDmVerityResult" ) ){
        if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_SET_DM_VERITY_CHECK_RESULT, (intptr_t)u8DmVerityResult ) != OSAL_OK ){
           ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bSetDmVerityResult() IO-Control OSAL_C_S32_IOCTRL_WUP_SET_DM_VERITY_CHECK_RESULT Failed !!!!" ) );
           bReturn = FALSE;
        }
      }
      #endif
      return( bReturn );
   } // bSetDmVerityResult

   tBool spm_tclOsalProxy::bSetFastShutdown( tU8 u8FastShutdownType ){
       /*!
         * \fn
         *  \brief
         *   Set the FAST SHUTDOWN option to SCC via dev_wup
         *   Currently supported are :
         *     - DEV_WUP_C_U8_FAST_SHUTDOWN_INACTIVE     ----------
         *     - DEV_WUP_C_U8_FAST_SHUTDOWN_ACTIVE       ----------
         *     - DEV_WUP_C_U8_FAST_SHUTDOWN_UNKNOWN      ----------
         *
         *  \version
         *    1.0   - Initial - Initial DM Verity Implementation
         ******
         */

      tBool bReturn = TRUE;

      ETG_TRACE_USR4( ( "spm_tclOsalProxy::bSetFastShutdown() Set FastShutdown option %d", u8FastShutdownType ) );

      #ifdef OSAL_C_S32_IOCTRL_WUP_SET_FAST_SHUTDOWN
      if ( bGetDevWupHandle( "spm_tclOsalProxy::bSetFastShutdown" ) ){
        if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_SET_FAST_SHUTDOWN, (intptr_t)u8FastShutdownType ) != OSAL_OK ){
           ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bSetFastShutdown() IO-Control OSAL_C_S32_IOCTRL_WUP_SET_FAST_SHUTDOWN Failed !!!!" ) );
           bReturn = FALSE;
        }
      }
      #endif
      return( bReturn );
   } // bSetDmVerityResult

// *******************************************************************************************************************
// *******************************************************************************************************************
// ****************  GPIO HELPER FUNCTIONS                                                      **********************
// *******************************************************************************************************************
// *******************************************************************************************************************

   tBool spm_tclOsalProxy::bGetGpio( tU32   u32Gpio,
                                     tBool *pbState ){
/*!
  * \fn
  *  \brief
  *   This method helps to read the gpio pin state as either input/output
  *  \param [in] u32Gpio: pin id.
  *  \param [in] pbState: state of pin updated and returned.
  *  \param [out] tBool: True if read successful.
  *
  *  \version
  *    1.0   - Initial
  ******
  */
      tBool              bReturn = FALSE;

      OSAL_trGPIOData    tGpioData;
      OSAL_tIODescriptor tDevGpioHandle;

      // Update the dev id
      tGpioData.tId  = u32Gpio;

      // Open gpio device in read/write mode
      tDevGpioHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GPIO, OSAL_EN_READWRITE );

      if ( OSAL_ERROR != tDevGpioHandle ){

         // ask for hw mute state
         if ( OSAL_ERROR == OSAL_s32IOControl( tDevGpioHandle,
                                               OSAL_C_32_IOCTRL_GPIO_IS_STATE_ACTIVE,
                                               (intptr_t)( &tGpioData ) ) ){

               ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected reading GPIO %d!!!!!!", (tUInt)u32Gpio ) );
         } else {
            // check state of mute pin
            if ( tGpioData.unData.bState ){
               * pbState = TRUE;
            } else {
               * pbState = FALSE;
            }
            bReturn = TRUE;
         }

         if ( OSAL_s32IOClose( tDevGpioHandle ) != OSAL_OK ){
               ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected closing DevGpio!!!!!!" ) );
         }
      }

      return( bReturn );
   } // bGetGpio

   tBool spm_tclOsalProxy::bSetGpio( tU32  u32Gpio,
                                     tBool bState ){
/*!
  * \fn
  *  \brief
  *   This method configures the gpio pin as either input/output
  *  \param [in] u32Gpio: pin id.
  *  \param [in] bState: out/in.
  *  \param [out] tBool: True if configuration successful.
  *
  *  \version
  *    1.0   - Initial
  ******
  */
      tBool              bReturn = FALSE;

      OSAL_trGPIOData    tGpioData;
      OSAL_tIODescriptor tDevGpioHandle;

      // Update the dev id
      tGpioData.tId           = u32Gpio;

      // Open gpio device in read/write mode
      tDevGpioHandle          = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GPIO, OSAL_EN_READWRITE );

      if ( OSAL_ERROR != tDevGpioHandle ){

         // set to OUTPUT first
         if ( OSAL_ERROR == OSAL_s32IOControl( tDevGpioHandle,
                                               OSAL_C_32_IOCTRL_GPIO_SET_OUTPUT,
                                               (intptr_t)tGpioData.tId ) ){
               ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected setting direction of GPIO %d!!!!!!", (tUInt)u32Gpio ) );
            // else
         }
      
         {
            tS32 s32GpioIoctl = ( TRUE == bState ) ? OSAL_C_32_IOCTRL_GPIO_SET_ACTIVE_STATE : OSAL_C_32_IOCTRL_GPIO_SET_INACTIVE_STATE;
            // set state
            if ( OSAL_ERROR == OSAL_s32IOControl( tDevGpioHandle,
                                                  s32GpioIoctl,
                                                  (intptr_t)( &tGpioData ) ) ){

                  ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected setting state of GPIO %d!!!!!!", (tUInt)u32Gpio ) );
            } else {
               bReturn = TRUE;
            }
         }

         if ( OSAL_s32IOClose( tDevGpioHandle ) != OSAL_OK ){
               ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected closing DevGpio!!!!!!" ) );
         }
      }
      return( bReturn );
   } // bSetGpio

   tBool spm_tclOsalProxy::bInstallGpioCallback( tU32                 u32Gpio,
                                                 OSAL_tpfGPIOCallback pvCallback,
                                                 tVoid               *pvArg,
                                                 tU16                 u16TriggerLevel ){
/*!
  * \fn
  *  \brief
  *   This method facilitates installation of a callback function for a designated  gpio pin
  *  \param [in] u32Gpio: pin id.
  *  \param [in] pvCallback: callback function.
  *  \param [out] u16TriggerLevel: Trigger edge low/high.
  *
  *  \version
  *    1.0   - Initial
  ******
  */
      tBool              bReturn = FALSE;

      OSAL_trGPIOData    tGpioData;
      OSAL_tIODescriptor tDevGpioHandle;

      // Update the dev id
      tGpioData.tId  = u32Gpio;

      // Open gpio device in read/write mode
      tDevGpioHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GPIO, OSAL_EN_READWRITE );

      if ( OSAL_ERROR != tDevGpioHandle ){

         // param to pass to the callback function
         OSAL_trGPIOCallbackData tGpioCallbackData;
         tGpioCallbackData.rData.unData.pfCallback = pvCallback;
         tGpioCallbackData.pvArg                   = pvArg;
         tGpioCallbackData.rData.tId               = u32Gpio;

         // param for setting the trigger edge
         tGpioData.unData.u16Edge                  = u16TriggerLevel;

         // register callback
         if ( OSAL_ERROR == OSAL_s32IOControl( tDevGpioHandle, OSAL_C_32_IOCTRL_GPIO_SET_CALLBACK, (intptr_t)( &tGpioCallbackData ) ) ){
               ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected setting callback for GPIO!!!!!!" ) );
            // Set trigger edge BOTH
         } else if ( OSAL_ERROR == OSAL_s32IOControl( tDevGpioHandle, OSAL_C_32_IOCTRL_GPIO_SET_TRIGGER, (intptr_t)( &tGpioData ) ) ){
               ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected setting trigger for GPIO!!!!!!" ) );
         } else {
            // param for setting enabling interrut
            tGpioData.unData.bState = TRUE;
            // enable interrupts
            if ( OSAL_s32IOControl( tDevGpioHandle, OSAL_C_32_IOCTRL_GPIO_ENABLE_INT, (intptr_t)( &tGpioData ) ) == OSAL_ERROR ){
               ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected enabling interrupt for GPIO!!!!!!" ) );
               tGpioCallbackData.rData.unData.pfCallback = NULL;
               OSAL_s32IOControl( tDevGpioHandle, OSAL_C_32_IOCTRL_GPIO_SET_CALLBACK, ( intptr_t )&tGpioCallbackData );
            }
         }


         if ( OSAL_s32IOClose( tDevGpioHandle ) != OSAL_OK ){
               ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected closing DevGpio!!!!!!" ) );
         }
      }
      return( bReturn );
   } // bInstallGpioCallback

   tBool spm_tclOsalProxy::bExtendPowerOffTimeout( tU32  u32TimeOut,
                                                   tU32 *pu32ResultTimeOut ){
/*!
  * \fn
  *  \brief
  *    This method is used to extend the poweroff timeout supervision
  *  \param [in] u32TimeOut: New timeout in seconds.
  *  \param [out] tBool: True if message could be send successful, but this not
  *                      means that the receveiver has also accepted the new timeout.
  *  \param [out] tPU32  New/current active/accepted timeout, is only set if message
  *                      could be send successfully.
  *
  *  \version
  *    1.0   - Initial
  ******
  */
      tBool bReturn = TRUE;

      if ( bGetDevWupHandle( "spm_tclOsalProxy::bExtendPowerOffTimeout" ) ){
         #ifdef OSAL_C_S32_IOCTRL_WUP_EXTEND_POWER_OFF_TIMEOUT
            tU16 u16MyTimeoutS = (tU16)u32TimeOut;
            tU16 u16TimeoutS   = u16MyTimeoutS;

            if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_EXTEND_POWER_OFF_TIMEOUT, ( intptr_t )&u16TimeoutS ) != OSAL_OK ){
               ETG_TRACE_ERRMEM( ( "SPM: OSAL_C_S32_IOCTRL_WUP_EXTEND_POWER_OFF_TIMEOUT !!!!!! Error detected !!!!!!" ) );
               bReturn = FALSE;
            } else {
               /* If the caller is intersted also in the returned/currently "really" running timer value,
                  store it at the give pointer. Default value for this ptr is NULL.*/
               if ( pu32ResultTimeOut ){
                  * pu32ResultTimeOut = u16TimeoutS;
               }
               if ( u16TimeoutS < u16MyTimeoutS ){
                  /* The returned timeout is smaller then the requested one. This happens if:
                     a) We send the request during the supervised wakeupsrc is currently active, then
                        the value "0" is returned as current active timeout.

                     b) The request ist send twice during inactive wakeupsrc. Only one set of the extensiontime
                        is allowed/processed. If a second extension is requested, the second request is ignored
                       and only the current value of the wakeup cnt is returned.
                    */
                  ETG_TRACE_USR4( ( "SPM: EPOT-ERR: u16TimeoutS(%d) < u16MyTimeoutS(%d) !", u16TimeoutS, u16MyTimeoutS ) );
                  // OK : Your code ...
               } else if ( u16TimeoutS >= u16MyTimeoutS ){
                  ETG_TRACE_USR4( ( "SPM: EPOT-OK: u16TimeoutS(%d) >= u16MyTimeoutS(%d) !", u16TimeoutS, u16MyTimeoutS ) );
                  // OK : Your code ...
               } else { // u16TimeoutS == 0
                  ETG_TRACE_USR4( ( "SPM: EPOT-DISABLED: u16TimeoutS =%d !", u16TimeoutS ) );
                  // OK : Your code ...
               }
            }
         #endif // ifdef OSAL_C_S32_IOCTRL_WUP_EXTEND_POWER_OFF_TIMEOUT
      } else {
         ETG_TRACE_FATAL( ( "OSAL_C_S32_IOCTRL_WUP_EXTEND_POWER_OFF_TIMEOUT: no handle to dev_wup available" ) );
         bReturn = FALSE;
      }
      return( bReturn );
   } // bExtendPowerOffTimeout

   tBool spm_tclOsalProxy::bConfigureSccResetHandling( tBool bDisable,
                                                       tU8   u8ExceptionBitmask ){
      /*!
        * \fn
        *  \brief
        *    This method is used to enable and disable the supervision of the SCC
        *  \param [in] bDisable: enable or disable the complete supervision
        *                    TRUE: supervision is disabled
        *                    FALSE: supervision is enabled
        *  \param [in] u8BitMask: specify which specific supervision option is enabled/disabled
        *                    bDisable=TRUE:  set bits specify which supervision is still enabled
        *                    bDisable=FALSE: set bits specify which supervision is still disabled
        *
        *  \version
        *    1.0   - Initial
        ******
        */
      tBool bReturn = TRUE;

      if ( bGetDevWupHandle( "spm_tclOsalProxy::bConfigureSccResetHandling" ) ){
         #ifdef OSAL_C_S32_IOCTRL_WUP_CONTROL_RESET_MASTER_SUPERVISION
            DEV_WUP_trResetMasterSupervision tResetMasterSupervision;
            tResetMasterSupervision.bSwitchOff         = bDisable;
            tResetMasterSupervision.u8ExceptionBitmask = u8ExceptionBitmask;

            if ( OSAL_s32IOControl( _hDeviceWup, OSAL_C_S32_IOCTRL_WUP_CONTROL_RESET_MASTER_SUPERVISION, ( intptr_t )&tResetMasterSupervision ) != OSAL_OK ){
               ETG_TRACE_ERRMEM( ( "SPM: OSAL_C_S32_IOCTRL_WUP_CONTROL_RESET_MASTER_SUPERVISION !!!!!! Error detected !!!!!!" ) );
               bReturn = FALSE;
            }
         #endif // ifdef OSAL_C_S32_IOCTRL_WUP_CONTROL_RESET_MASTER_SUPERVISION
      } else {
         ETG_TRACE_FATAL( ( "OSAL_C_S32_IOCTRL_WUP_CONTROL_RESET_MASTER_SUPERVISION: no handle to dev_wup available" ) );
         bReturn = FALSE;
      }
      return( bReturn );
   } // bConfigureSccResetHandling

// *******************************************************************************************************************
// *******************************************************************************************************************
// ****************  Process/Thread information handling                                        **********************
// *******************************************************************************************************************
// *******************************************************************************************************************

   tBool spm_tclOsalProxy::bWdtTimeOutWarning( tS32         s32EmTraceType,
                                               const tChar *cThreadName ){
/*!
  * \fn
  *  \brief
  *    .
  *  \version
  *    1.0   - Initial
  ******
  */
      (tVoid)cThreadName;
      (tVoid)s32EmTraceType;

      return( FALSE );
   } // bWdtTimeOutWarning

   tBool spm_tclOsalProxy::bDumpThreadInfoReady( tS32  s32Prio,
                                                 tBool bAlways ){
      (tVoid)s32Prio;
      SPM_NULL_POINTER_CHECK_VAL( _poclSupervisionEnableSupervisor );
      if ( ( TRUE == _poclSupervisionEnableSupervisor->bIsSupervisionStopped( ) )
        && ( bAlways != SPM_BL_OSALPROXY_ALWAYS ) ){
         ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bDumpThreadInfoReady: Triggering ThreadInfo disabled due to Supervision settings (WD_OFF/disable_reset.txt/TTFis-SpmResetSupervisionDisable)" ) );
         return( FALSE );
      }
      SPM_NULL_POINTER_CHECK_VAL( _poOsLinuxHdl );
      return( _poOsLinuxHdl->bGetReadyThreads( ) );
   }

   tBool spm_tclOsalProxy::bDumpThreadInfo( const std::string& cThreadName,
                                            tBool              bAlways ){
         SPM_NULL_POINTER_CHECK_VAL( _poclSupervisionEnableSupervisor );
      if ( ( TRUE == _poclSupervisionEnableSupervisor->bIsSupervisionStopped( ) )
           && ( bAlways != SPM_BL_OSALPROXY_ALWAYS ) ){
         ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bDumpThreadInfo: Triggering ThreadInfo disabled due to Supervision settings (WD_OFF/disable_reset.txt/TTFis-SpmResetSupervisionDisable)" ) );
         return( FALSE );
      }
         SPM_NULL_POINTER_CHECK_VAL( _poOsLinuxHdl );
      _oListThreadName.push_back( cThreadName );
      return( _poOsLinuxHdl->bGetThreadInfo( cThreadName ) );
   }

   tBool spm_tclOsalProxy::bDumpProcessInfo( const std::string& cProcessName,
                                             const std::string& strStartType,
                                             tBool              bDumpCallstacks,
                                             tBool              bAlways ){
         SPM_NULL_POINTER_CHECK_VAL( _poclSupervisionEnableSupervisor );
      if ( ( TRUE == _poclSupervisionEnableSupervisor->bIsSupervisionStopped( ) )
           && ( bAlways != SPM_BL_OSALPROXY_ALWAYS ) ){
         ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bDumpProcessInfo: Triggering ProcessInfo disabled due to Supervision settings (WD_OFF/disable_reset.txt/TTFis-SpmResetSupervisionDisable)" ) );
         return( FALSE );
      }
         SPM_NULL_POINTER_CHECK_VAL( _poOsLinuxHdl );
      _strProcessName = cProcessName;
      return( _poOsLinuxHdl->bGetProcessInfo( cProcessName, bDumpCallstacks, strStartType ) );
   }

   tBool spm_tclOsalProxy::bDumpProcessInfo( tU32  u32ProcdId,
                                             tBool bDumpCallstacks,
                                             tBool bAlways ){
         SPM_NULL_POINTER_CHECK_VAL( _poclSupervisionEnableSupervisor );
      if ( ( TRUE == _poclSupervisionEnableSupervisor->bIsSupervisionStopped( ) )
           && ( bAlways != SPM_BL_OSALPROXY_ALWAYS ) ){
         ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bDumpProcessInfo: Triggering ProcessInfo disabled due to Supervision settings (WD_OFF/disable_reset.txt/TTFis-SpmResetSupervisionDisable)" ) );
         return( FALSE );
      }
         SPM_NULL_POINTER_CHECK_VAL( _poOsLinuxHdl );
      return( _poOsLinuxHdl->bGetProcessInfo( u32ProcdId, bDumpCallstacks ) );
   }

   tBool spm_tclOsalProxy::bDumpProcessList( tBool bErrMem,
                                             tBool bTrace,
                                             tBool bAlways ){
      (tVoid)bErrMem;
      (tVoid)bTrace;
         SPM_NULL_POINTER_CHECK_VAL( _poclSupervisionEnableSupervisor );
      if ( ( TRUE == _poclSupervisionEnableSupervisor->bIsSupervisionStopped( ) )
           && ( bAlways != SPM_BL_OSALPROXY_ALWAYS ) ){
         ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bDumpProcessList: Triggering ProcessList disabled due to Supervision settings (WD_OFF/disable_reset.txt/TTFis-SpmResetSupervisionDisable)" ) );
         return( FALSE );
      }
      SPM_NULL_POINTER_CHECK_VAL( _poclWorkerServer );
      ETG_TRACE_USR1( ( "SPM: !!!!!! SPM_U32_WORKER_PRX_PROC_LIST !!!!!!" ) );
      _poclWorkerServer->bPostMessage( "ISpmOsalProxy", SPM_U32_WORKER_PRX_PROC_LIST );
      return( TRUE );
   }

   tBool spm_tclOsalProxy::bDumpSemInfo( const tChar *cSemName ){
      (tVoid)cSemName;
      return( FALSE );
   }

   tBool spm_tclOsalProxy::bDumpTimeTrace( tBool bProcesses,
                                           tBool bAlways ){
      SPM_NULL_POINTER_CHECK_VAL( _poclSupervisionEnableSupervisor );
      if ( ( TRUE == _poclSupervisionEnableSupervisor->bIsSupervisionStopped( ) )
           && ( bAlways != SPM_BL_OSALPROXY_ALWAYS ) ){
         ETG_TRACE_ERRMEM( ( "spm_tclOsalProxy::bDumpTimeTrace: Triggering TimeTracker disabled due to Supervision settings (WD_OFF/disable_reset.txt/TTFis-SpmResetSupervisionDisable)" ) );
         return( FALSE );
      }
      SPM_NULL_POINTER_CHECK_VAL( _poOsLinuxHdl );
      return( _poOsLinuxHdl->bTriggerTimeTracker( bProcesses ) );
   }

   tBool spm_tclOsalProxy::bWdtGetThreadInfoById( tU32  u32ThreadId,
                                                  tBool fErrmemWrite,
                                                  tBool fTraceWrite ){
      (tVoid)u32ThreadId;
      (tVoid)fErrmemWrite;
      (tVoid)fTraceWrite;

      return( FALSE );
   } // bWdtGetThreadInfoById

   tVoid spm_tclOsalProxy::vHandleMessage( tU32 u32Message,
                                           tU32 u32Parm ){
      (tVoid)u32Parm;
      SPM_NULL_POINTER_CHECK( _poOsLinuxHdl );

      switch ( u32Message ){
         case SPM_U32_WORKER_PRX_TIME_TRACER:
         {
            (tBool)bWdtTimeOutWarning( SPM_S32_EMTRACE_TT_START );
            (tBool)_poOsLinuxHdl->bTriggerTimeTracker( );
            (tBool)bWdtTimeOutWarning( SPM_S32_EMTRACE_TT_DUMP_WITH_CALLSTACK );
            (tBool)bWdtTimeOutWarning( SPM_S32_EMTRACE_TT_STOP );
         }
         break;

         case SPM_U32_WORKER_PRX_THREAD_INFO_RDY:
         {
            _poOsLinuxHdl->bGetReadyThreads( );
         }
         break;

         case SPM_U32_WORKER_PRX_THREAD_INFO:
         {
            if ( !_oListThreadName.empty( ) ){
               std::string strThreadName = _oListThreadName.front( );
               _oListThreadName.pop_front( );
               _poOsLinuxHdl->bGetThreadInfo( strThreadName );
            }
         }
         break;

         case SPM_U32_WORKER_PRX_PROC_INFO:
         {
            ETG_TRACE_USR1( ( "SPM: !!!!!! Received SPM_U32_WORKER_PRX_PROC_INFO !!!!!!" ) );
            _poOsLinuxHdl->bGetProcessInfo( _strProcessName );
         }
         break;

         case SPM_U32_WORKER_PRX_PROC_LIST:
         {
            ETG_TRACE_USR1( ( "SPM: !!!!!! Received SPM_U32_WORKER_PRX_PROC_INFO !!!!!!" ) );
            SPM_NULL_POINTER_CHECK( _poclProcessSupervision );
            _poclProcessSupervision->vDumpStartedProcessList( TRUE, TRUE );
         }
         break;

         default:
            // nothing to do
            break;
      } // switch
   }    // vHandleMessage

/*!
  * \fn
  *  \brief
  *    to check if datapool (dp_tclSpmDpConfigFakeDeviceConfig) for fake is TRUE
  *    then generate runtime registry with KEY:CONNECT_TO_FAKE and VALUE:CONNECT_TO_FAKE_VALUE
  *    else trace msg-connecting to SCC.
  *
  *  \version
  *    1.0   - Initial
  ******
  */
   tVoid spm_tclOsalProxy::vCheckForFake( ){
      dp_tclSpmDpConfigFakeDeviceConfig bFakeDeviceConnect;
      tBool                             bConnect2Fake = bFakeDeviceConnect.tGetData( );

      ETG_TRACE_USR1( ( "vCheckForFake():: bConnect2Fake= %u", bConnect2Fake ) );
      if ( bConnect2Fake ){
         OSAL_tIODescriptor         _hRegFakeDevice  = (OSAL_tIODescriptor)OSAL_ERROR;
         OSAL_tIODescriptor         _hRegFakeKey     = 0;
         OSAL_trIOCtrlRegistryValue sFakeValue;
         tU32                       u32RegistryValue = CONNECT_TO_FAKE_VALUE;
         _hRegFakeDevice = OSAL_IOOpen( OSAL_C_STRING_DEVICE_REGISTRY, OSAL_EN_READWRITE );
         if ( OSAL_ERROR == _hRegFakeDevice ){
                  ETG_TRACE_ERRMEM( ( "vCheckForFake():: ERROR opening registry" ) );
         } else {
            _hRegFakeKey = OSAL_IOCreate( SPM_REG_VERSIONS_OSAL_DEVICES_DEV_WUP_FAKE_KEY_FILE, OSAL_EN_READWRITE );
            if ( OSAL_ERROR == _hRegFakeKey ){
                  ETG_TRACE_ERRMEM( ( "vCheckForFake():: ERROR while creating fake registry" ) );
            } else {
                  OSAL_pvMemoryCopy( sFakeValue.s8Name,  (const void*)CONNECT_TO_FAKE_KEY, sizeof( CONNECT_TO_FAKE_KEY ) );
                  OSAL_pvMemoryCopy( sFakeValue.s8Value, (const void*)&u32RegistryValue,   sizeof( u32RegistryValue ) );
               sFakeValue.s32Type = OSAL_C_S32_VALUE_S32;
               if ( OSAL_ERROR == OSAL_s32IOControl( _hRegFakeKey, OSAL_C_S32_IOCTRL_REGWRITEVALUE, ( intptr_t )&sFakeValue ) ){
                  ETG_TRACE_ERRMEM( ( "vCheckForFake()::ERROR while writing into the fake registry" ) );
               }
               if ( OSAL_s32IOClose( _hRegFakeKey ) == OSAL_ERROR ){
                  ETG_TRACE_ERRMEM(( "vCheckForFake():: ERROR while closing fake key registry with return error %s", OSAL_coszErrorText( OSAL_u32ErrorCode( ) ) ));
               }
            }
         }
         if ( OSAL_s32IOClose( _hRegFakeDevice ) == OSAL_ERROR ){
                  ETG_TRACE_ERRMEM( ( "vCheckForFake()::ERROR while closing the registry" ) );
         }
      } else {
                  ETG_TRACE_USR1( ( "vCheckForFake():: DataPool not set, Connecting to SCC" ) );
      }
   } // vCheckForFake

// EOF

