/*!
  * \file spm_OsalCvmUser.cpp
  *  \brief
  *    Base class to handle user voltage level handling
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2013 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author             | Modification
  * 22.11.13  | CM-AI/CB32 Kollai  | initial version
  ******
  */
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

// SPM  configuration
#include "spm_Config.h"
#include "spm_GlobDefs.h"

// my class header
#include "spm_OsalCvm.h"

// interfaces class definitions
#include "spm_ICvmClient.h"
#include "spm_ISpmCvmUserClient.h"

#include "spm_IFactory.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_OsalCvm.cpp.trc.h"
#endif

#include "spm_trace.h"

#define SPM_CVM_C_U32_EVENT_MASK_STOP_THREAD   0x01000000

#define SPM_CVM_C_U32_EVENT_MASK_ALL \
   ( DEV_VOLT_C_U32_EVENT_MASK_SYSTEM_VOLTAGE_CHANGED_NOTIFY   \
     | DEV_VOLT_C_U32_EVENT_MASK_USER_VOLTAGE_CHANGED_NOTIFY   \
     | DEV_VOLT_C_U32_EVENT_MASK_PERMANENT_HIGH_VOLTAGE   \
     | DEV_VOLT_C_U32_EVENT_MASK_PERMANENT_CRITICAL_HIGH_VOLTAGE   \
     | SPM_CVM_C_U32_EVENT_MASK_STOP_THREAD )

spm_tclOsalCvm::spm_tclOsalCvm( const ISpmFactory& factory ) : ISpmOsalCvm( factory )
   , _hDeviceVoltDescr( ( tS32 )OSAL_C_INVALID_HANDLE )
   , _hEventHandle( OSAL_C_INVALID_HANDLE )
   , _hThreadIdCvmEvent( OSAL_ERROR )
   , _u32ClientId( 0 )
   , _u32PreviousSystemVoltage( DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_OPERATING )
   , _poCvmClient( NULL )
   , _poclWorkerServer( NULL ){

}

spm_tclOsalCvm::~spm_tclOsalCvm( ){

   if ( _hDeviceVoltDescr != (tS32)OSAL_C_INVALID_HANDLE ){
      if ( OSAL_s32IOControl( _hDeviceVoltDescr, OSAL_C_S32_IOCTRL_VOLT_UNREGISTER_CLIENT, (intptr_t)_u32ClientId ) == OSAL_ERROR ){
         ETG_TRACE_ERR( ( "spm_tclOsalCvm::~spm_tclOsalCvm(): Cannot unregister OSAL_C_S32_IOCTRL_VOLT_UNREGISTER_CLIENT." ) );
      }
      if ( ( _hThreadIdCvmEvent != OSAL_ERROR ) && ( _hEventHandle != OSAL_C_INVALID_HANDLE ) ){
         // thread is running --> stop now
         ETG_TRACE_USR1( ( "spm_tclOsalCvm::~spm_tclOsalCvm(): Send event to stop thread" ) );
         if ( OSAL_s32EventPost( _hEventHandle, SPM_CVM_C_U32_EVENT_MASK_STOP_THREAD, OSAL_EN_EVENTMASK_OR ) == OSAL_OK ){
            ETG_TRACE_ERR( ( "spm_tclOsalCvm::~spm_tclOsalCvm(): Cannot post osal event." ) );
         }
      }
   }
   _poCvmClient      = NULL;
   _poclWorkerServer = NULL;
}

tVoid spm_tclOsalCvm::vGetReferences( ){
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poCvmClient, spm_tclCriticalVoltageManagerIntern, ISpmCvmClient );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer, ISpmWorkerServer );
}

tVoid spm_tclOsalCvm::vStartCommunication( ){
}

tVoid spm_tclOsalCvm::vStartLateCommunication( ){

   DEV_VOLT_trClientRegistration rClientRegistration;

   // initialization is useless but required by LINT
   rClientRegistration.u32ClientId                = 0; // magic number used on purpose as no valid value exists
   rClientRegistration.szNotificationEventName[0] = 0;

   // open device
   _hDeviceVoltDescr                              = OSAL_IOOpen( OSAL_C_STRING_DEVICE_VOLT, OSAL_EN_READWRITE );
   if ( OSAL_ERROR == _hDeviceVoltDescr ){
      _hDeviceVoltDescr = (tS32)OSAL_C_INVALID_HANDLE;
            ETG_TRACE_ERRMEM( ( "spm_tclOsalCvm::spm_tclOsalCvm(): Failed to open dev volt." ) );
   } else {
      if ( OSAL_s32IOControl( _hDeviceVoltDescr, OSAL_C_S32_IOCTRL_VOLT_REGISTER_CLIENT, ( intptr_t )&rClientRegistration ) == OSAL_OK ){
         _u32ClientId = rClientRegistration.u32ClientId;
         if ( OSAL_s32EventOpen( rClientRegistration.szNotificationEventName, &_hEventHandle ) != OSAL_OK ){
            ETG_TRACE_ERRMEM( ( "spm_tclOsalCvm::spm_tclOsalCvm(): Failed to open handle." ) );
         }
      }
   }

   if ( _hDeviceVoltDescr != (tS32)OSAL_C_INVALID_HANDLE ){

      OSAL_trThreadAttribute rAttr;
      std::string            szThreadName = "SpmOsalVolt";

      rAttr.szName       = &szThreadName[0];
      rAttr.s32StackSize = 10000;
      rAttr.u32Priority  = 100;
      rAttr.pfEntry      = (OSAL_tpfThreadEntry)vCvmWorkerThread;
      rAttr.pvArg        = ( tPVoid ) this;

      _hThreadIdCvmEvent = OSAL_ThreadSpawn( &rAttr );

      if ( _hThreadIdCvmEvent == OSAL_ERROR ){
            ETG_TRACE_ERRMEM( ( "spm_tclOsalCvm::vStartCommunication(): Failed to spawn thread!" ) );
      }
   }
} // vStartCommunication

tBool spm_tclOsalCvm::bAddUserVoltClient( ISpmCvmUserClient *client,
                                          tU16               u16VoltageLevel,
                                          tU16               u16Hysteresis ){

   tBool bRet = FALSE;

   if ( _hDeviceVoltDescr != (tS32)OSAL_C_INVALID_HANDLE ){

      DEV_VOLT_trUserVoltageRegistration rUserVoltageRegistration;

      tU8                                u8Direction = DEV_VOLT_C_U8_USER_VOLTAGE_CROSS_DIRECTION_UPWARD;
      if ( u16VoltageLevel < 12000 ){
         u8Direction = DEV_VOLT_C_U8_USER_VOLTAGE_CROSS_DIRECTION_DOWNWARD;
      }

      // dev/volt takes care that minimum hysteresis value is used. If value is too small it will be
      // increased automatically, so no check required here.
      rUserVoltageRegistration.u32ClientId           = _u32ClientId;
      rUserVoltageRegistration.u16UserVoltageLevelMv = u16VoltageLevel;
      rUserVoltageRegistration.u16HysteresisMv       = u16Hysteresis;
      rUserVoltageRegistration.u8LevelCrossDirection = u8Direction;

      ETG_TRACE_USR1( ( "spm_tclOsalCvm::bAddUserVoltClient(): Register user voltage notification (for %dmV)", u16VoltageLevel ) );

      if ( OSAL_s32IOControl( _hDeviceVoltDescr,
                              OSAL_C_S32_IOCTRL_VOLT_REGISTER_USER_VOLTAGE_NOTIFICATION,
                              ( intptr_t )&rUserVoltageRegistration ) == OSAL_OK ){

         TClientInfo tClt;
         tClt.u16VoltageLevel = u16VoltageLevel;
         tClt.bBelow          = FALSE;
         tClt.pRef            = client;

         if ( rUserVoltageRegistration.u8UserVoltageState == DEV_VOLT_C_U8_USER_VOLTAGE_LEVEL_EXCEEDED ){
            client->vVoltageLevelOverrun( );
            tClt.bBelow = FALSE;
         } else {
            client->vVoltageLevelUnderrun( );
            tClt.bBelow = TRUE;
         }

         _oClientList.push_back( tClt );
         bRet = TRUE;
      } else {
         ETG_TRACE_ERRMEM( ( "spm_tclOsalCvm::bAddUserVoltClient(): Failed to register user notification (for %dmV)", u16VoltageLevel ) );
      }
   } else {
         ETG_TRACE_ERRMEM( ( "spm_tclOsalCvm::bAddUserVoltClient(): No valid dev_volt handle" ) );
   }
   return( bRet );
} // bAddUserVoltClient

tBool spm_tclOsalCvm::bRemoveUserVoltClient( ISpmCvmUserClient *client ){
   (tVoid)client;
   return( TRUE );
}

tVoid spm_tclOsalCvm::vCvmWorkerThread( tVoid *pvArg ){
/*!
  * \fn
  *  \brief
  *    Worker Thread to handle notfication on chaanged voltage state from device
  *
  *  \param
  *     tVoid*  -> pointer to this-pointer of the SPM object.
  *  \version
  *    1.0   - Initial
  ******
  */
   tBool                                bStopThread            = FALSE;
   OSAL_tEventMask                      rEventMaskResult       = 0;
   tU16                                 u16MyNewBoardVoltageMv = 0;

   spm_tclOsalCvm                      *poThis                 = (spm_tclOsalCvm*)pvArg;

   SPM_NULL_POINTER_CHECK( poThis );

   DEV_VOLT_trSystemVoltageRegistration rSystemVoltageRegistration;

   ETG_TRACE_USR1( ( "spm_tclOsalCvm::vCvmWorkerThread(): Thread to handle user/project specific voltage change started." ) );

   rSystemVoltageRegistration.u32ClientId              = poThis->_u32ClientId;
   rSystemVoltageRegistration.u32VoltageIndicationMask = ( DEV_VOLT_C_U32_BIT_MASK_INDICATE_LOW_VOLTAGE
                                                           | DEV_VOLT_C_U32_BIT_MASK_INDICATE_CRITICAL_LOW_VOLTAGE
                                                           | DEV_VOLT_C_U32_BIT_MASK_INDICATE_HIGH_VOLTAGE
                                                           | DEV_VOLT_C_U32_BIT_MASK_INDICATE_CRITICAL_HIGH_VOLTAGE );

   // Register for System Voltage Notification
   if ( OSAL_s32IOControl( poThis->_hDeviceVoltDescr,
                           OSAL_C_S32_IOCTRL_VOLT_REGISTER_SYSTEM_VOLTAGE_NOTIFICATION,
                           ( intptr_t )&rSystemVoltageRegistration ) != OSAL_OK ){
      ETG_TRACE_FATAL( ( "spm_tclOsalCvm::vCvmWorkerThread(): Failed to register system notification." ) );
   }
   // check the state once for previous changes
   SPM_NULL_POINTER_CHECK( poThis->_poCvmClient );
   switch ( rSystemVoltageRegistration.u32CurrentSystemVoltageState ){
      case DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_CRITICAL_HIGH:
      {
         poThis->_poCvmClient->vHandleCvmEvent( OSALCVM_CRITICAL_HIGH_VOLTAGE_START );
         poThis->_u32PreviousSystemVoltage = DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_CRITICAL_HIGH;
         break;
      }

      case DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_HIGH:
      {
         poThis->_poCvmClient->vHandleCvmEvent( OSALCVM_HIGH_VOLTAGE_START );
         poThis->_u32PreviousSystemVoltage = DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_HIGH;
         break;
      }

      case DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_OPERATING:
      {
         /* nothing to be done */
         break;
      }

      case DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_LOW:
      {
         poThis->_poCvmClient->vHandleCvmEvent( OSALCVM_LOW_VOLTAGE_START );
         poThis->_u32PreviousSystemVoltage = DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_LOW;
         break;
      }

      case DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_CRITICAL_LOW:
      {
         poThis->_poCvmClient->vHandleCvmEvent( OSALCVM_CRITICAL_LOW_VOLTAGE_START );
         poThis->_u32PreviousSystemVoltage = DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_CRITICAL_LOW;
         break;
      }

      default:
      {
         ETG_TRACE_FATAL( ( "spm_tclOsalCvm::vCvmWorkerThread(): Unknwon Voltage State.%i", rSystemVoltageRegistration.u32CurrentSystemVoltageState ) );
      }
   } // switch
   while ( FALSE == bStopThread ){
      rEventMaskResult = 0;
      if ( OSAL_s32EventWait( poThis->_hEventHandle,
                              SPM_CVM_C_U32_EVENT_MASK_ALL,
                              OSAL_EN_EVENTMASK_OR,
                              OSAL_C_TIMEOUT_FOREVER,
                              &rEventMaskResult ) == OSAL_OK ){
         if ( OSAL_s32EventPost( poThis->_hEventHandle, ~rEventMaskResult, OSAL_EN_EVENTMASK_AND ) == OSAL_OK ){  // Clear evaluated event bits
            ETG_TRACE_USR1( ( "spm_tclOsalCvm::vCvmWorkerThread(): Voltage notification received: %08x.", ETG_ENUM( SPM_CVM_DEV_VOLT_EVENT_MASK, rEventMaskResult ) ) );

            if ( rEventMaskResult & DEV_VOLT_C_U32_EVENT_MASK_USER_VOLTAGE_CHANGED_NOTIFY ){

               DEV_VOLT_trUserVoltage rUserVoltage;

               rUserVoltage.u32ClientId = poThis->_u32ClientId;

               if ( OSAL_s32IOControl( poThis->_hDeviceVoltDescr, OSAL_C_S32_IOCTRL_VOLT_GET_USER_VOLTAGE_STATE, ( intptr_t )&rUserVoltage ) == OSAL_OK ){

                  u16MyNewBoardVoltageMv = rUserVoltage.u16LatestCrossedUserVoltageLevelMv;
                  if ( rUserVoltage.u8LatestUserVoltageState == DEV_VOLT_C_U8_USER_VOLTAGE_LEVEL_EXCEEDED ){
                     u16MyNewBoardVoltageMv++;
                     // iterate through client list
                  }
                  std::list < TClientInfo >::iterator iter;
                  for ( iter = poThis->_oClientList.begin( ); iter != poThis->_oClientList.end( ); iter++ ){
                     ETG_TRACE_USR1( ( "spm_tclOsalCvm::vCvmWorkerThread(): Voltage level changed --> %dmV (trigger level: %dmV) .", u16MyNewBoardVoltageMv, iter->u16VoltageLevel ) );
                     if ( ( u16MyNewBoardVoltageMv > iter->u16VoltageLevel ) && ( iter->bBelow ) ){
                        iter->bBelow = FALSE;
                        iter->pRef->vVoltageLevelOverrun( );
                     } else if ( ( u16MyNewBoardVoltageMv <= iter->u16VoltageLevel ) && ( !iter->bBelow ) ){
                        iter->bBelow = TRUE;
                        iter->pRef->vVoltageLevelUnderrun( );
                     }
                  }
               } else {
                  ETG_TRACE_ERRMEM( ( "spm_tclOsalCvm::vCvmWorkerThread(): Failed to read board voltage!" ) );
               }
            }
            if ( rEventMaskResult & DEV_VOLT_C_U32_EVENT_MASK_SYSTEM_VOLTAGE_CHANGED_NOTIFY ){
                  ETG_TRACE_USR1( ( "spm_tclOsalCvm::vCvmWorkerThread(): Voltage level changed --> new system event --> DEV_VOLT_C_U32_EVENT_MASK_SYSTEM_VOLTAGE_CHANGED_NOTIFY." ) );
               poThis->vProcessSystemStateNotification( );
            }
            if ( rEventMaskResult & DEV_VOLT_C_U32_EVENT_MASK_PERMANENT_HIGH_VOLTAGE ){
                  ETG_TRACE_USR1( ( "spm_tclOsalCvm::vCvmWorkerThread(): new system event --> DEV_VOLT_C_U32_EVENT_MASK_PERMANENT_HIGH_VOLTAGE." ) );

                  SPM_NULL_POINTER_CHECK( poThis->_poclWorkerServer );
               poThis->_poclWorkerServer->bPostMessage( "spm_tclCriticalVoltageManagerIntern", SPM_U32_WORKER_CVM_TIMEOUT );
               #ifdef SPM_FEATURE_ENABLE_HIGH_VOLT_PERMANENT_TRIGGER
                  SPM_NULL_POINTER_CHECK( poThis->_poCvmClient );
                  ETG_TRACE_USR1( ( "spm_tclOsalCvm::vCvmWorkerThread(): send event OSALCVM_HIGH_VOLTAGE_START --> spm_tclCriticalVoltageManager with PERMANENT_HIGH_VOLTAGE" ) );
                  poThis->_poCvmClient->vHandleCvmEvent( OSALCVM_HIGH_VOLTAGE_START );
               #endif

            }
            if ( rEventMaskResult & DEV_VOLT_C_U32_EVENT_MASK_PERMANENT_CRITICAL_HIGH_VOLTAGE ){
               ETG_TRACE_USR1( ( "spm_tclOsalCvm::vCvmWorkerThread(): new system event --> DEV_VOLT_C_U32_EVENT_MASK_PERMANENT_CRITICAL_HIGH_VOLTAGE." ) );
               SPM_NULL_POINTER_CHECK( poThis->_poclWorkerServer );
               poThis->_poclWorkerServer->bPostMessage( "spm_tclCriticalVoltageManagerIntern", SPM_U32_WORKER_CVM_CRITICAL_TIMEOUT );
               #ifdef SPM_FEATURE_ENABLE_CRITICAL_HIGH_VOLT_PERMANENT_TRIGGER
                  SPM_NULL_POINTER_CHECK( poThis->_poCvmClient );
                  ETG_TRACE_USR1( ( "spm_tclOsalCvm::vCvmWorkerThread(): send event OSALCVM_CRITICAL_HIGH_VOLTAGE_START --> spm_tclCriticalVoltageManager with PERMANENT_CRITICAL_HIGH_VOLTAGE" ) );
                  poThis->_poCvmClient->vHandleCvmEvent( OSALCVM_CRITICAL_HIGH_VOLTAGE_START );
               #endif
            }
            if ( rEventMaskResult & SPM_CVM_C_U32_EVENT_MASK_STOP_THREAD ){
               bStopThread = TRUE;
            }
         } else {
            ETG_TRACE_ERRMEM( ( "spm_tclOsalCvm::vCvmWorkerThread(): Failed to post osal event!" ) );
         }
      } else {
            ETG_TRACE_ERRMEM( ( "spm_tclOsalCvm::vCvmWorkerThread(): Failed to wait for event!" ) );
            OSAL_s32ThreadWait( 1000 );
      }
   }
} // vCvmWorkerThread

tVoid spm_tclOsalCvm::vProcessSystemStateNotification( ){
/*!
  * \fn
  *  \brief
  *    This method sets the inital CvmState after registration for voltage notification.
  *
  *  \param[in] u32CvmState: Current system voltage state.
  *  \version
  *    1.0   - Initial
  ******
  */
   DEV_VOLT_trSystemVoltageHistory rSystemVoltageHistory;

   rSystemVoltageHistory.u32ClientId = _u32ClientId;

   // Get the system voltage changed history.
   if ( OSAL_OK != OSAL_s32IOControl( _hDeviceVoltDescr, OSAL_C_S32_IOCTRL_VOLT_GET_SYSTEM_VOLTAGE_HISTORY, ( intptr_t )&rSystemVoltageHistory ) ){
      ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected : OSAL IOCTRL Failed to GET SYSTEM VOLTAGE HISTORY !!!!!!" ) );
   } else {
      SPM_NULL_POINTER_CHECK( _poCvmClient );
      tU32 u32CurrentSystemVoltage = rSystemVoltageHistory.rSystemVoltage.u32CurrentSystemVoltageState;

                        ETG_TRACE_USR4( ( "spm_tclOsalCvm::vProcessSystemStateNotification(): Low Voltage Counter: %d, Critical Low Voltage Counter: %d, High Voltage Counter: %d, Critical High Voltage Counter:%d",
                        rSystemVoltageHistory.rSystemVoltage.u32LowVoltageCounter,
                        rSystemVoltageHistory.rSystemVoltage.u32CriticalLowVoltageCounter,
                        rSystemVoltageHistory.rSystemVoltage.u32HighVoltageCounter,
                        rSystemVoltageHistory.rSystemVoltage.u32CriticalHighVoltageCounter ) );
                        ETG_TRACE_USR4( ( "spm_tclOsalCvm::vProcessSystemStateNotification(): Previous CVM state:%d, Current CVM state:%d",
                        ETG_ENUM( SPM_CVMSTATE_DEF, _u32PreviousSystemVoltage ),
                        ETG_ENUM( SPM_CVMSTATE_DEF, u32CurrentSystemVoltage )
                        ) );


      if ( _u32PreviousSystemVoltage >= DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_OPERATING ){
         if (
              ( _u32PreviousSystemVoltage > DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_OPERATING)
           && ( rSystemVoltageHistory.rSystemVoltage.u32LowVoltageCounter > 0 )
         ){
             _poCvmClient->vHandleCvmEvent( OSALCVM_HIGH_VOLTAGE_END );
         }
         if ( rSystemVoltageHistory.rSystemVoltage.u32LowVoltageCounter > 0 ){
            _poCvmClient->vHandleCvmEvent( OSALCVM_LOW_VOLTAGE_START );
         }
         if ( rSystemVoltageHistory.rSystemVoltage.u32CriticalLowVoltageCounter > 0 ){
            _poCvmClient->vHandleCvmEvent( OSALCVM_CRITICAL_LOW_VOLTAGE_START );
            if ( u32CurrentSystemVoltage > DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_CRITICAL_LOW ){
               _poCvmClient->vHandleCvmEvent( OSALCVM_CRITICAL_LOW_VOLTAGE_END );
            }
         }
         if ( rSystemVoltageHistory.rSystemVoltage.u32LowVoltageCounter > 0 ){
            if ( u32CurrentSystemVoltage > DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_LOW ){
               _poCvmClient->vHandleCvmEvent( OSALCVM_LOW_VOLTAGE_END );
            }
         }
      }

      if ( _u32PreviousSystemVoltage == DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_LOW ){
         if ( rSystemVoltageHistory.rSystemVoltage.u32CriticalLowVoltageCounter > 0 ){
            _poCvmClient->vHandleCvmEvent( OSALCVM_CRITICAL_LOW_VOLTAGE_START );
            if ( u32CurrentSystemVoltage > DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_CRITICAL_LOW ){
               _poCvmClient->vHandleCvmEvent( OSALCVM_CRITICAL_LOW_VOLTAGE_END );
            }
         }
         if ( u32CurrentSystemVoltage > DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_LOW ){
            _poCvmClient->vHandleCvmEvent( OSALCVM_LOW_VOLTAGE_END );
         }
      }

      if ( _u32PreviousSystemVoltage == DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_CRITICAL_LOW ){
         if ( u32CurrentSystemVoltage > DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_CRITICAL_LOW ){
            _poCvmClient->vHandleCvmEvent( OSALCVM_CRITICAL_LOW_VOLTAGE_END );
         }
         if ( u32CurrentSystemVoltage > DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_LOW ){
            _poCvmClient->vHandleCvmEvent( OSALCVM_LOW_VOLTAGE_END );
         }
      }

      if ( u32CurrentSystemVoltage == DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_OPERATING ){
         if ( _u32PreviousSystemVoltage == DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_CRITICAL_HIGH ){
            _poCvmClient->vHandleCvmEvent( OSALCVM_CRITICAL_HIGH_VOLTAGE_END );
            _poCvmClient->vHandleCvmEvent( OSALCVM_HIGH_VOLTAGE_END );
         }
         if ( _u32PreviousSystemVoltage == DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_HIGH ){
            _poCvmClient->vHandleCvmEvent( OSALCVM_HIGH_VOLTAGE_END );
         }
      }
      if ( u32CurrentSystemVoltage == DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_HIGH ){
         #ifndef SPM_FEATURE_ENABLE_HIGH_VOLT_PERMANENT_TRIGGER
            if ( _u32PreviousSystemVoltage == DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_OPERATING ){
               _poCvmClient->vHandleCvmEvent( OSALCVM_HIGH_VOLTAGE_START );
            } else {
               _poCvmClient->vHandleCvmEvent( OSALCVM_CRITICAL_HIGH_VOLTAGE_END );
            }
         #endif
      }

      if ( u32CurrentSystemVoltage == DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_CRITICAL_HIGH ){
         #ifndef SPM_FEATURE_ENABLE_CRITICAL_HIGH_VOLT_PERMANENT_TRIGGER
            if ( _u32PreviousSystemVoltage == DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_OPERATING ){
               _poCvmClient->vHandleCvmEvent( OSALCVM_HIGH_VOLTAGE_START );
            }
            _poCvmClient->vHandleCvmEvent( OSALCVM_CRITICAL_HIGH_VOLTAGE_START );
         #endif
      }

      _u32PreviousSystemVoltage = rSystemVoltageHistory.rSystemVoltage.u32CurrentSystemVoltageState;

   }

} // vProcessSystemStateNotification

// EOF

