/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_baseClass.h
  * @brief        This is header file contains functions for basic thread usage (thread starts, terminates, resumes and suspends).
  * @copyright    (C) 2013 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif


#include <string>
// //////////////////////////////////////////////////////////////////////////////
// interface for a client class to work with spm_tclActive (inside a thread)

/*!
  *  \class spm_tclActiveClient
  *  \brief This class provides interfaces for a client class to work with spm_tclActive

  */
class spm_tclActiveClient
{
public:
virtual ~spm_tclActiveClient( ){}
virtual tVoid vMain( ) = 0; // interface for main entry function

};


// //////////////////////////////////////////////////////////////////////////////
// class encapsulate the main routine of a thread and calls either a client or
// a subclassed main function.

/*!
  *  \class spm_tclActive
  *  \brief This class encapsulate the main routine of a thread and calls either a client or a subclassed main function.
          It take following actions on thread starts, terminates, resumes and suspends .
  */
class spm_tclActive
{
public:
spm_tclActive( spm_tclActiveClient *pClient = NULL ){
   _bTerminate = FALSE;
   _hThreadId  = OSAL_ERROR;
   _u32TraceId = ( tU32 ) - 1;
   _u32Prio    = 0;
   _clt        = pClient;
}

virtual ~spm_tclActive( ){
   spm_tclActive::vSetTerminate( ); // terminate the thread if object gets destroyed
}

virtual tVoid main( ){  // has to be overloaded by concrete class
   if ( _clt != NULL ){
      _clt->vMain( );   // call main routine of client
   }
   vSetTerminate( );
}

virtual tVoid vStartThread( const std::string& strName,
                            tU32               u32Prio,
                            tU32               u32Stack ){
   _u32Prio           = u32Prio;
   _hThreadId         = OSAL_ERROR;
   _strName           = strName;

   OSAL_trThreadAttribute rAttr;

   rAttr.szName       = ( tString ) & strName[0]; //lint !e1773 Attempt to cast away const (or volatile)
   rAttr.s32StackSize = u32Stack;
   rAttr.u32Priority  = u32Prio;
   rAttr.pfEntry      = (OSAL_tpfThreadEntry)vThreadMain;
   rAttr.pvArg        = ( tPVoid )this;
   if ( ( _hThreadId = OSAL_ThreadSpawn( &rAttr ) ) == OSAL_ERROR ){
      _bTerminate = TRUE;
   }
   _u32TraceId        = (tU32)_hThreadId; // memorize for later trace output
} // vStartThread

virtual tVoid vSetTerminate( ){ _bTerminate = TRUE; }
virtual tBool bIsTerminated( ) const { return( _bTerminate ); }
virtual OSAL_tThreadID hGetThreadId( ) const { return( _hThreadId ); }

virtual tVoid vTraceActive( ) const {
}

tBool bIsThreadCreated( ) const { return( _hThreadId != OSAL_ERROR ); }
tVoid vResume( ){
   if ( bIsThreadCreated( ) ){
      OSAL_s32ThreadResume( _hThreadId );
   }
}

tVoid vSuspend( ){
   if ( bIsThreadCreated( ) ){
      OSAL_s32ThreadSuspend( _hThreadId );
   }
}

protected:
virtual tVoid vOnStart( ){}
virtual tVoid vOnTerminate( ){}
tVoid vSetPriority( tU32 u32Prio ){
   if ( _hThreadId != OSAL_ERROR ){
      OSAL_s32ThreadPriority( _hThreadId, u32Prio );
   }
}

tVoid vRestoreStartPriority( ){ vSetPriority( _u32Prio ); }

private:
static tVoid vThreadMain( tVoid *pArg ){
   spm_tclActive *pActive = (spm_tclActive*)pArg;

   pActive->vOnStart( );

   while ( !pActive->_bTerminate ){
      pActive->main( );
   }
   pActive->vOnTerminate( );         // call a function just before we stop the thread
   pActive->_hThreadId = OSAL_ERROR; // make handle to thread invalid
   OSAL_vThreadExit( );              // terminate the thread
}

private:
tBool                _bTerminate; // terminate the main loop

OSAL_tThreadID       _hThreadId;  // thread handle
tU32                 _u32TraceId; // handle for tracing
tU32                 _u32Prio;    // default thread priority
spm_tclActiveClient *_clt;        // possible client of this class
std::string          _strName;    // Name of thread
};

