/*!
  * \file spm_Registry.cpp
  *  \brief
  *    To Handle registry related functionalities.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 11.01.11  | TMS Fischer       | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#ifdef VARIANT_S_FTR_ENABLE_USE_REG_ENCRYPTION
   #define ENCRYPTION_S_IMPORT_READER
#include "encryption_if.h"
#endif

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_Registry.h"

// spm class definitions w/o interface

// interfaces class definitions
#include "spm_IStartupInvestigationServer.h"
#include "spm_FsHelper.h"

#include "spm_factory.h"
#include "spm_Registry_pathes.h"

// included
// ahl_if.h include to get definition of macro AMT_C_U16_APPID_INVALID
#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_REG
#include "trcGenProj/Header/spm_Registry.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"


/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
// #define SPM_TRACE_FILE_ID   SPM_FILE_REGISTRY

spm_tclRegistry::spm_tclRegistry( const ISpmFactory& factory ) : ISpmRegistry( factory ){
}

tBool spm_tclRegistry::bImportData( const std::string& strFileName,
                                    tBool              bFileWithPath,
                                    const std::string& strKeyToCheck ){
   tBool bSuccess = FALSE;

   if ( !strFileName.empty( ) ){
      // check if registry is already loaded (by RFD file)
      bSuccess = bImportDataByFile( strFileName, bFileWithPath, strKeyToCheck );
      if ( !bSuccess ){
         ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
      }
   }
   return( bSuccess );
}

tBool spm_tclRegistry::bGetAllApplicationsOfSwBlock( tU16 u16SpmId,
                                                     const std::string& strSwBlockName,
                                                     std::map < tU16, std::vector < tU16 > >& mapAppIds ){
/*!
  * \fn
  *  \brief
  *    gets all applications in the software block
  *
  *  \param[in]  u16SpmId       application id of spm
  *  \param[in]  strSwBlockName name of software block
  *  \param[out] mapAppIds      list of application in software block
  *
  *  \return TRUE if strSwBlockName exist otherwise FALSE
  *
  ******
  */
   tU16                   u16AppID;
   std::string            strName( 255, 0 );

   // iterate processes
   reg_tclProcessIterator oProcIter;

   while ( !oProcIter.bIsDone( ) ){
      oProc = oProcIter.oItem( );
      oProc.bGetName( &strName[0], (tU32)strName.length( ) );
      if ( strSwBlockName == strName.c_str( ) ){
         if ( oProc.bIsValid( ) ){
            reg_tclApplicationIterator oAppIter( oProc );
            while ( !oAppIter.bIsDone( ) ){
               oApp = oAppIter.oItem( );
               if ( oApp.bIsValid( ) ){
                  oApp.bGetID( &u16AppID );
                  if ( ( u16AppID != u16SpmId ) && ( u16AppID != AMT_C_U16_APPID_INVALID ) ){
                     mapAppIds[u16AppID] = std::vector < tU16 >( ); // empty service list

                     // is there a service for this app? -> add to service map
                     reg_tclServiceIterator oSvcIter( oApp );
                     while ( !oSvcIter.bIsDone( ) ){
                        oSvc = oSvcIter.oItem( );
                        if ( oSvc.bIsValid( ) ){
                           tU32 u32SvcID;
                           if ( oSvc.bGetID( &u32SvcID ) ){
                              mapAppIds[u16AppID].push_back( (tU16)u32SvcID );
                              /*if*/
                           }
                        } /*if*/
                        oSvcIter.vNext( );
                     }    /*while*/
                  }
               }
               oAppIter.vNext( );
            }
         }
         return( TRUE );
      }
      oProcIter.vNext( );
   }
   return( FALSE );
} // bGetAllApplicationsOfSwBlock

tBool spm_tclRegistry::bFindSwBlockNameOfApp( tU32         u32AppId,
                                              std::string& strSwBlockName ){
/*!
  * \fn
  *  \brief
  *    find the software block for the input application id
  *
  *  \param[in]   u32AppId       Input application id
  *  \param[out]  strSwBlockName Filled with name of software block for the input application
  *  \return   TRUE if u32AppId exist otherwise FALSE
  *
  ******
  */
   tU16                   u16RegAppID;
   std::string            strName( 255, 0 );

   // iterate processes
   reg_tclProcessIterator oProcIter;

   while ( !oProcIter.bIsDone( ) ){
      oProc = oProcIter.oItem( );
      oProc.bGetName( &strName[0], (tU32)strName.length( ) );
      if ( oProc.bIsValid( ) ){
         reg_tclApplicationIterator oAppIter( oProc );
         while ( !oAppIter.bIsDone( ) ){
            oApp = oAppIter.oItem( );
            if ( oApp.bIsValid( ) ){
               oApp.bGetID( &u16RegAppID );
               if ( ( u16RegAppID == u32AppId ) && ( u16RegAppID != AMT_C_U16_APPID_INVALID ) ){
                  strSwBlockName = strName.c_str( );
                  return( TRUE );
               }
            }
            oAppIter.vNext( );
         }
      }
      oProcIter.vNext( );
   }
   return( FALSE );
} // bFindSwBlockNameOfApp

tBool spm_tclRegistry::bIsSoftwareBlockInRegistry( const std::string& strSwBlockName ){
/*!
  * \fn
  *  \brief
  *    This function returns TRUE, if the softwareblock exists in /dev/registry.
  *
  *  \param[in] strSwBlockName software block name.
  *
  *  \return    Whether software block exists or not.
  ******
  */
   std::string            strName( 255, 0 );

   reg_tclProcessIterator oProcIter;

   while ( !oProcIter.bIsDone( ) ){
      oProc = oProcIter.oItem( );

      oProc.bGetName( &strName[0], (tU32)strName.length( ) );
      if ( strSwBlockName == strName.c_str( ) ){
         return( TRUE );
      }
      oProcIter.vNext( );
   }

   return( FALSE );
} // bIsSoftwareBlockInRegistry

tBool spm_tclRegistry::bGetStringFromRegistry( const std::string& strPath,
                                               const std::string& strKey,
                                               std::string      & strResult ){
/*!
  * \fn
  *  \brief
  *    check whether key corresponds with the given registry path and gets result in string
  *
  *  \param[in]  strPath    relative path of registry file
  *  \param[in]  strKey     name of key
  *  \param[out] strResult  result from bQueryString
  *
  *  \return			 whether is key is found in registry path or not
  *
  ******
  */
   std::string strName;
   tS32        as32Data[256];

   strName = (std::string)SPM_REG_BASE_PATH + strPath;

   if ( oReg.bOpen( strName.c_str( ) ) ){
      if ( oReg.bQueryString( strKey.c_str( ), (tChar*)as32Data, sizeof( as32Data ) ) ){
         strResult = (tChar*)as32Data;
         return( TRUE );
      }
   }
   return( FALSE );
} // bGetStringFromRegistry

tBool spm_tclRegistry::bGetNumberFromRegistry( const std::string& strPath,
                                               const std::string& strKey,
                                               tU32             & u32Result ){
/*!
  * \fn
  *  \brief
  *    check whether key corresponds with the given registry path and gets result in string tU32
  *
  *  \param[in]  strPath    relative path of registry file
  *  \param[in]  strKey     name of key
  *  \param[out] u32Result  result from bQueryU32
  *
  *  \return			 whether is key is found in registry path or not
  *
  ******
  */
   std::string strName;

   strName = (std::string)SPM_REG_BASE_PATH + strPath;

   if ( oReg.bOpen( strName.c_str( ) ) ){
      return( oReg.bQueryU32( strKey.c_str( ), &u32Result ) );
   }
   return( FALSE );
}

tVoid spm_tclRegistry::vTraceOutRegistry( const std::string& strBaseName ){
/*!
  * \fn
  *  \brief
  *    vTraceOutRegistry
  *
  *  \param[in] strBaseName name of base path.
  *
  ******
  */
   std::string           strName;
   tS32                  as32Data[255];

   OSAL_trIOCtrlDir      rDir  = { 0,0,0 };
   OSAL_trIOCtrlDir      rDirV = { 0,0,0 };
   OSAL_trIOCtrlRegistry rReg;
   OSAL_tIODescriptor    fd;

   ETG_TRACE_FATAL( ( "PATH: %s", strBaseName.c_str( ) ) );

   fd = OSAL_IOOpen( strBaseName.c_str( ), OSAL_EN_READONLY );
   if ( fd != OSAL_ERROR ){

      /* First, recursively show all the subkeys */
      rDir.fd        = fd;
      rDir.s32Cookie = 0;

      while ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGREADDIR, ( intptr_t )&rDir ) != OSAL_ERROR ){
         strName = strBaseName + "/" + (tString)rDir.dirent.s8Name;

         vTraceOutRegistry( strName.c_str( ) );
      }

      /* Now show all the values of this key */
      rDirV.fd        = fd;
      rDirV.s32Cookie = 0;

      while ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGENUMVALUE, ( intptr_t )&rDirV ) != OSAL_ERROR ){
         rReg.pcos8Name = rDirV.dirent.s8Name;
         rReg.ps8Value  = (tU8*)as32Data;
         rReg.u32Size   = sizeof( as32Data );

         if ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGGETVALUE, ( intptr_t )&rReg ) != OSAL_ERROR ){
            // This is correct because not all traces come out if this is missing
            if ( rReg.s32Type == OSAL_C_S32_VALUE_S32 ){
               ETG_TRACE_FATAL( ( "VAL number: '%32s' 0x%08x (%d)", (const tChar*)rDirV.dirent.s8Name, as32Data[0], as32Data[0] ) );
            } else if ( rReg.s32Type == OSAL_C_S32_VALUE_STRING ){
               ETG_TRACE_FATAL( ( "VAL string: '%32s' '%s'", (const tChar*)rDirV.dirent.s8Name, (const tChar*)as32Data ) );
            }
         }
      }
      OSAL_s32IOClose( fd );
   }
} // vTraceOutRegistry

tBool spm_tclRegistry::bRemoveSoftwareBlock( const std::string& strKeyName ){
/*!
  * \fn
  *  \brief
  *        removes softwareblock from registry
  *
  *  \param[in] name of key
  *
  *  \return    Whether software block removal is success or not.
  ******
  */
   std::string           strName;

   OSAL_trIOCtrlDir      rDir  = { 0,0,0 };
   OSAL_trIOCtrlDir      rDirV = { 0,0,0 };
   OSAL_tIODescriptor    fd;
   OSAL_trIOCtrlRegistry rReg;

   fd = OSAL_IOOpen( strKeyName.c_str( ), OSAL_EN_READWRITE );

   if ( fd != OSAL_ERROR ){
      rDir.fd        = fd;
      rDir.s32Cookie = 0;

      // dive recursive in the leafes
      while ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGREADDIR, ( intptr_t )&rDir ) != OSAL_ERROR ){
         rDir.s32Cookie = 0;
         strName        = strKeyName + "/" + (tString)rDir.dirent.s8Name;

         if ( FALSE == bRemoveSoftwareBlock( strName ) ){
            return( FALSE );
         }
      }

      rDirV.fd        = fd;
      rDirV.s32Cookie = 0;

      while ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGENUMVALUE, ( intptr_t )&rDirV ) != OSAL_ERROR ){
         rReg.pcos8Name  = rDirV.dirent.s8Name;
         rDirV.s32Cookie = 0;
         if ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGREMOVEVALUE, ( intptr_t )&rReg ) != OSAL_OK ){
            // ET_TRACE_ERROR(0, "spm_tclRegistry::bRemoveSoftwareBlock, can't remove Value: %s", rReg.pcos8Name);
         } else {
            // ET_TRACE_ERROR(0, "spm_tclRegistry::bRemoveSoftwareBlock, Removed Value: %s", rReg.pcos8Name);
         }
      }

      if ( OSAL_s32IOClose( fd ) != OSAL_OK ){
         ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
      }
      if ( OSAL_s32IORemove( strKeyName.c_str( ) ) != OSAL_OK ){
         // ET_TRACE_ERROR(0, "spm_tclRegistry::bRemoveSoftwareBlock, can't remove Key: %s", strKeyName);
         return( FALSE );
      } else {
         // ET_TRACE_ERROR(0, "spm_tclRegistry::bRemoveSoftwareBlock, Removed Key: %s", strKeyName);
      }
   }
   return( TRUE );
} // bRemoveSoftwareBlock

tVoid spm_tclRegistry::vRemoveEntry( const std::string& strPath,
                                     const std::string& strKey ) const {
/*!
  * \fn
  *  \brief
  *    removes the key from registry path
  *
  *  \param[in]  strPath    relative path of registry file
  *  \param[in]  strKey     name of key
  *
  ******
  */
   OSAL_tIODescriptor fd;
   std::string        strName;

   strName = (std::string)SPM_REG_PROCESS_BASE_PATH + strPath;

   fd      = OSAL_IOOpen( strName.c_str( ), OSAL_EN_READWRITE );
   if ( fd != OSAL_ERROR ){
      OSAL_trIOCtrlRegistry hReg;

      hReg.pcos8Name = (tPCS8)strKey.c_str( );
      hReg.s32Type   = OSAL_C_S32_VALUE_S32; // not really needed for deleting a key
      hReg.u32Size   = 0;
      hReg.ps8Value  = NULL;

      // remove a key in the registry
      if ( OSAL_ERROR == OSAL_s32IOControl( fd,
                                            OSAL_C_S32_IOCTRL_REGREMOVEVALUE,
                                            ( intptr_t )&hReg ) ){
         ETG_TRACE_USR4( ( "vRemoveEntry: Could not remove the key %s from registry", strName.c_str( ) ) );
      } else {
         ETG_TRACE_USR4( ( "vRemoveEntry:: Key %s was successfully removed from registry", strName.c_str( ) ) );
      }
         OSAL_s32IOClose( fd );
   } else {
      ETG_TRACE_FATAL( ( "vRemoveEntry: Could not open registry %s", strName.c_str( ) ) );
   }
} // vRemoveEntry

// //////////////////////////////////////////////////////////////////////////////
// leave this function at the end of the file, do not try to use ETG traces
// in there. It will not work!. We're using backslashes in that function which
// bring the ETG parsing out of control. Therefore from this function on ETG
// is not working anymore. Ask the Gods why is that in the tool and still not
// fixed.!!!

tBool spm_tclRegistry::bImportDataByFile( const std::string& strFileName,
                                          tBool              bFileWithPath,
                                          const std::string& strKeyToCheck ){
/*!
  * \fn
  *  \brief
  *    parses registry file and fills /dev/registry
  *    this should work for TSIM, too.
  *
  *  \param[in] strFileName    Name of file
  *  \param[in] bFileWithPath  Whether absolute path or relative (currently not used)
  *  \param[in] strKeyToCheck  name of key
  *
  *  \return   Always return true
  ******
  */
   (tVoid)bFileWithPath;
   OSAL_tIODescriptor fd;

   std::string        strPathBuffer;
   tBool              bSuccess            = FALSE;

   tU32               u32RegLoadStartTime = OSAL_ClockGetElapsedTime( );

   if ( !strKeyToCheck.empty( ) ){
      tU32        u32Dummy;
      std::string strPath = "PROCESS/" + strKeyToCheck;

      if ( bGetNumberFromRegistry( strPath, "REGLOADED", u32Dummy ) ){
         (void)u32Dummy;
         bSuccess = TRUE;
         ETG_TRACE_USR2( ( "bImportData(): import registry '%s' skipped (already loaded via RFD active)", strFileName.c_str( ) ) );
         {
            // store in startup investigation
            SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(_poclStartupInvest, ISpmStartupInvestigationServer);
            SPM_STARTUPINVEST_INIT_STARTUPITEM
            SPM_STARTUPINVEST_FORCE_ENTRY( "REGISTRY", ( (std::string)"registry '" + strFileName + "' already loaded via RFD active" ) );
         }
      }
   }

   if ( !bSuccess ){

      // check for need to add default reg path "/"
      if ( NULL == OSAL_ps8StringSubString( strFileName.c_str( ), "/" ) ){
         // reg file w/o path detected. add registry path
         ETG_TRACE_USR4( ( "Conversion of registry path needed: '%s'", strFileName.c_str( ) ) );
         strPathBuffer = (std::string)SPM_REGISTRY_ROOT_PATH_OSAL + "/" + strFileName;

         std::string strReg;
         strReg        = (std::string)SPM_REGISTRY_ROOT_PATH_OSAL + "/" + strFileName;

         if ( !spm_bIsFileInFs( strReg ) ){
            strPathBuffer = (std::string)SPM_REGISTRY_DYN_PATH_OSAL + "/" + strFileName;

         }
      } else {
         strPathBuffer = strFileName;
      }
      tS32 s32RegIoctrl = OSAL_C_S32_IOCTRL_BUILD_REG;
      // check if registry is encrypted
      if ( ( std::string::npos != strFileName.find( ".ERG" ) )
           || ( std::string::npos != strFileName.find( ".erg" ) )
           ){
         s32RegIoctrl = OSAL_C_S32_IOCTRL_BUILD_ERG;
      }
      if ( ( fd = OSAL_IOOpen( OSAL_C_STRING_DEVICE_REGISTRY, OSAL_EN_READWRITE ) ) != OSAL_ERROR ){
         if ( ( OSAL_s32IOControl( fd, s32RegIoctrl, ( intptr_t )strPathBuffer.c_str( ) ) ) == OSAL_OK ){
            ETG_TRACE_USR1( ( "spm_tclRegistry::bImportDataByFile(): REGISTRY loaded %s!", strPathBuffer.c_str( ) ) );
         } else {
            ETG_TRACE_FATAL( ( "spm_tclRegistry::bImportDataByFile(): FAILED to load REGISTRY %s!", strPathBuffer.c_str( ) ) );
         }
         OSAL_s32IOClose( fd );
      }

   }

   ETG_TRACE_USR1( ( "spm_tclRegistry::bImportDataByFile(): Duration to load reg-file '%70s': %d ms.", strFileName.c_str( ), OSAL_ClockGetElapsedTime( ) - u32RegLoadStartTime ) );

   return( TRUE );
} // bImportDataByFile

