/*!
  * \file spm_CcaSupplierHandler.cpp
  *  \brief
  *    Implements the cca supplier server
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 23.05.12  | CM-AI/VW32 kollai | Initial version
  ******
  */

/*****************************************************************************
  *  ServiceSupplier:
  *  In order to register for a service a client application has to know
  *  which server application (service supplier) offers the required service.
  *  That is, it has to know the AppID of the potential server.
  *
  *  The client registers for SrvSupplierStatus messages at the power manager
  *  (SPM, AppID = 0) by announcing the desired ServiceID. In return it will
  *  get a SrvSupplierStatus message which announces the availability of an
  *  application. If there is currently no appropriate application known, the
  *  status message contains the unknown application ID (0xFFFF) and the status
  *  unknown.
  *
  *  ApplicationInfo:
  *  Surveying applications is useful when an application wants to know the
  *  availability status of another (known by AppID) application. This service
  *  is offered by the power manager (AppID = 0). A server application needs
  *  this service to be informed about forced shutdown of a client. However,
  *  this service is not limited for that case.
  *
  *  The requesting application registers by announcing the AppID of the
  *  application of interest by sending the AppStatusRegister message. In
  *  return it will get an AppStatus message which announces the availability
  *  of the application. If the application is currently not available it gets
  *  the status unavailable or unknown depending on, whether the application
  *  is loaded but not started or the application is unknown at all.
  *
  *  Whenever the application is started the status changes to available.
  *******************************************************************************/

#include <set>


#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_CcaSupplierHandler.h"

// spm class definitions (wo if class)

// interfaces class definitions
#include "spm_IApplicationDatabase.h"
#include "spm_ICcaServer.h"
#include "spm_IFactory.h"

// spm helper

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_CSUH
#include "trcGenProj/Header/spm_CcaSupplierHandler.cpp.trc.h"
#endif

// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
// #define SPM_TRACE_FILE_ID   SPM_FILE_CCASUPPLIERHANDLER

spm_tclCcaSupplierHandler::spm_tclCcaSupplierHandler( const ISpmFactory& factory )
   : ISpmCcaSupplierServer( factory )
   , _poclCcaMsgHandler( NULL )
   , _poclWorkerServer( NULL ){
}

spm_tclCcaSupplierHandler::~spm_tclCcaSupplierHandler( ){
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   _poclWorkerServer->vRemoveClient( this );

   _poclWorkerServer  = NULL;
   _poclCcaMsgHandler = NULL;
}

tVoid spm_tclCcaSupplierHandler::vGetReferences( ){
   SPM_GET_IF_REFERENCE_USE_VAR( _poclCcaMsgHandler, ISpmCcaServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer,  ISpmWorkerServer );
}

tVoid spm_tclCcaSupplierHandler::vStartCommunication( ){
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   _poclWorkerServer->vAddClient( this );
}

tVoid spm_tclCcaSupplierHandler::vSupplierStateChanged( tU16 u16SupplierAppId,
                                                        tU32 u32AppState,
                                                        tU16 /*u16CoreId*/ ){
/*!
  * \fn
  *  \brief
  *    State change of an application is detected.
  *    State change to "UNAVAILABLE" when application acknowledges TERMINATE
  *    State change to "AVAILABLE" when application acknowledges INITIALIZE
  *
  *  \param
  *    tU16 u16SupplierAppId:  CCA Application ID of changed application
  *    tU32 u32AppState:       AMT_C_U8_CCAMSG_SUPPLIER_STATE_UNAVAILABLE   (tU8)0
  *                            AMT_C_U8_CCAMSG_SUPPLIER_STATE_AVAILABLE     (tU8)1
  *                            AMT_C_U8_CCAMSG_SUPPLIER_STATE_UNKNOWN       (tU8)2
  *                            AMT_C_U8_CCAMSG_SUPPLIER_STATE_ERROR         (tU8)3
  *    tU16 u16CoreId:         CORE number where SupplierApplication is located, not used
  *                            in current implementation (may be removed)
  *
  *  \version
  *    1.0   - Initial
  ******
  */
   std::map < tU32, TNotificationEntry >::iterator it;

   ETG_TRACE_USR4( ( "vSupplierStateChanged, app %04x changed service state to %u",
                     ETG_ENUM( ail_u16AppId,           u16SupplierAppId ),
                     ETG_ENUM( SPM_CCA_SUPPLIER_STATE, u32AppState )
                     ) );

   // is an entry for that supplier id already available?
   it = _mapSupplierNotificationTab.find( u16SupplierAppId );

   if ( it == _mapSupplierNotificationTab.end( ) ){
      TNotificationEntry tNewApp;
      tNewApp.u8AppStatus                           = (tU8)u32AppState;
      // the constructor of tNotificationList is automatically called
      // so that we start with an empty TAppList

      _mapSupplierNotificationTab[u16SupplierAppId] = tNewApp;
   } else {
      it->second.u8AppStatus = (tU8)u32AppState;
   }
   vSendNotification( u16SupplierAppId );

   if ( u32AppState == AMT_C_U8_CCAMSG_SUPPLIER_STATE_UNAVAILABLE ){
      // application is now disconnected -> remove all references
      if ( !bRemoveNotification( u16SupplierAppId ) ){
         ETG_TRACE_COMP( ( "SPM: !!!!!! Warning detected !!!!!!" ) );
      }
   }
} // vSupplierStateChanged

tBool spm_tclCcaSupplierHandler::bRemoveNotification( tU16 u16DisconnectedAppId ){
   /*!
     * \fn
     *  \brief
     *    All entries for references for disconnected application
     *
     *  \param
     *    tU16 u16DisconnectedAppId:  CCA Application ID of disconnected app
     *
     *  \version
     *    1.0   - Initial
     ******
     */

   // and now check for unknown applications / service mapping
   std::map < tU32, TNotificationEntry >::iterator it;

                     ETG_TRACE_USR4( ( "bRemoveNotification, disconnected app id: %04x", ETG_ENUM( ail_u16AppId, u16DisconnectedAppId ) ) );

   for ( it = _mapSupplierNotificationTab.begin( ); it != _mapSupplierNotificationTab.end( ); ++it ){
      std::list < TNotApp >::iterator it2 = it->second.tNotificationList.begin();

      while (it2 != it->second.tNotificationList.end()) {
         if ((it2->u16DstAppId) == u16DisconnectedAppId)
           it2 = it->second.tNotificationList.erase(it2);
         else
           it2++;
      }
   }
   return( TRUE );
} // bRemoveNotification

tBool spm_tclCcaSupplierHandler::bAddNotificationEntry( tU16 u16SupplierAppId,
                                                        tU16 u16AppIdToBeNotified,
                                                        tU16 u16ServiceId,
                                                        tU16 u16SourceSubId,
                                                        tU16 u16TargetSubId ){
/*!
  * \fn
  *  \brief
  *    New supplier notification is added to the notification map.
  *
  *  \param
  *    tU16 u16SupplierAppId:     CCA Application ID -> Service Owner
  *    tU16 u16AppIdToBeNotified: CCA Application ID -> Application to be notified if state change detected
  *    tU16 u16ServiceId:         Service ID
  *    tU16 u16SourceSubId:       Sub ID of the source
  *    tU16 u16TargetSubId:       Sub ID of the Target
  *
  *  \version
  *    1.0   - Initial
  ******
  */

   std::map < tU32, TNotificationEntry >::iterator it;

                     ETG_TRACE_USR4( ( "bAddNotificationEntry, supplier app: %04x, client app: %04x, service ID: 0x%04x, SourceSubId: %02x, TargetSubId: %02x",
                     ETG_ENUM( ail_u16AppId, u16SupplierAppId ),
                     ETG_ENUM( ail_u16AppId, u16AppIdToBeNotified ),
                     u16ServiceId,
                     u16SourceSubId,
                     u16TargetSubId
                     ) );

   it                     = _mapSupplierNotificationTab.find( u16SupplierAppId );

   TNotApp tNotApp;
   tNotApp.u16DstAppId    = u16AppIdToBeNotified;
   tNotApp.u16SourceSubId = u16SourceSubId;
   tNotApp.u16TargetSubId = u16TargetSubId;
   tNotApp.u16ServiceId   = u16ServiceId;

   if ( it == _mapSupplierNotificationTab.end( ) ){
      // new app found -> addto map
      TNotificationEntry tNewApp;

      tNotApp.u8LastSendState                       = AMT_C_U8_CCAMSG_SUPPLIER_STATE_ERROR;

      tNewApp.u8AppStatus                           = AMT_C_U8_CCAMSG_SUPPLIER_STATE_UNKNOWN;
      tNewApp.tNotificationList.push_back( tNotApp );

      _mapSupplierNotificationTab[u16SupplierAppId] = tNewApp;
   } else {
      // app already in list --> add application to list and send update
      tNotApp.u8LastSendState = AMT_C_U8_CCAMSG_SUPPLIER_STATE_ERROR;

      _mapSupplierNotificationTab[u16SupplierAppId].tNotificationList.push_back( tNotApp );
   }

   vSendNotification( u16SupplierAppId );

   return( TRUE );
} // bAddNotificationEntry

tBool spm_tclCcaSupplierHandler::bFindNotificationEntry( tU16 u16SupplierAppId, tU16 u16ServiceId ){
/*!
  * \fn
  *  \brief
  *    Find if a service is offered by a specific supplier application.
  *
  *  \param
  *    tU16 u16SupplierAppId:     CCA Application ID -> Service Owner
  *    tU16 u16ServiceId:         Service ID
  *
  *  \version
  *    1.0   - Initial
  ******
  */

   std::map  < tU32, TNotificationEntry > ::iterator it;
   std::list < TNotApp >                  ::iterator it2;

   it = _mapSupplierNotificationTab.find( u16SupplierAppId );

   if ( it == _mapSupplierNotificationTab.end( ) ){
      return( FALSE );
   }

   it2 = it->second.tNotificationList.begin();

   for (it2 = it->second.tNotificationList.begin(); it2 != it->second.tNotificationList.end(); ++it2){
      if ( it2->u16ServiceId == u16ServiceId ){
         return( TRUE );
      }
   }

   return( FALSE );
} // bFindNotificationEntry

tBool spm_tclCcaSupplierHandler::bRemoveNotificationEntry( tU16 u16SupplierAppId,
                                                           tU16 u16AppIdToBeNotified,
                                                           tU16 u16ServiceId,
                                                           tU16 u16SourceSubId,
                                                           tU16 u16TargetSubId ){
/*!
  * \fn
  *  \brief
  *    Existing supplier notification is removed from the notification map.
  *
  *  \param
  *    tU16 u16SupplierAppId:     CCA Application ID -> Service Owner
  *    tU16 u16AppIdToBeNotified: CCA Application ID -> Application to be notified if state change detected
  *    tU16 u16ServiceId:         Service ID
  *    tU16 u16SourceSubId:       Sub ID of the source
  *    tU16 u16TargetSubId:       Sub ID of the Target
  *
  *  \version
  *    1.0   - Initial
  ******
  */

   std::map < tU32, TNotificationEntry >::iterator it;

   ETG_TRACE_USR4( ( "bRemoveNotificationEntry, supplier app: %04x, client app: %04x, service ID: 0x%04x, SourceSubId: %02x, TargetSubId: %02x",
                     ETG_ENUM( ail_u16AppId, u16SupplierAppId ),
                     ETG_ENUM( ail_u16AppId, u16AppIdToBeNotified ),
                     u16ServiceId,
                     u16SourceSubId,
                     u16TargetSubId
                     ) );

   it = _mapSupplierNotificationTab.find( u16SupplierAppId );

   if ( it != _mapSupplierNotificationTab.end( ) ){
      std::list < TNotApp >::iterator it2 = it->second.tNotificationList.begin();

      while (it2 != it->second.tNotificationList.end()) {
         if (((it2->u16DstAppId) == u16AppIdToBeNotified) && ((it2->u16ServiceId) == u16ServiceId))
           it2 = it->second.tNotificationList.erase(it2);
         else
           it2++;
      }
   }

   return( TRUE );
} // bRemoveNotificationEntry

tBool spm_tclCcaSupplierHandler::bChangeServiceNotification( const amt_tclServiceSupplierUnregister& crSrvSupplierReg,
                                                             tBool                                   bAddEntry ){
/*!
  * \fn
  *  \brief
  *    New service notification is added to the notification map. Received by CCA message AMT_C_U8_CCAMSGTYPE_SUPPLIER_STATE_REGISTER
  *
  *  \param
  *    tU16 u16AppIdToBeNotified: CCA Application ID -> Application to be notified if state change detected
  *    tU16 u16ServiceId:         Service ID
  *
  *  \version
  *    1.0   - Initial
  ******
  */
   tU16                                                     u16ServiceId         = crSrvSupplierReg.u16GetServiceID( );
   tU16                                                     u16AppIdToBeNotified = crSrvSupplierReg.u16GetSourceAppID( );
   tU16                                                     u16SourceSubId       = crSrvSupplierReg.u16GetSourceSubID( );
   tU16                                                     u16TargetSubId       = crSrvSupplierReg.u16GetTargetSubID( );

   std::map < tU16, std::list < TAppListEntry > >::iterator it;
   tU16                                                     u16SupplierAppId     = AMT_C_U16_APPID_INVALID;

   ETG_TRACE_USR4( ( "bAddServiceNotification(bAddEntry=%d), request from app %04x (SubId %02x), inform when supplier of service ID 0x%04x changes state",
                     bAddEntry,
                     ETG_ENUM( ail_u16AppId, u16AppIdToBeNotified ),
                     u16SourceSubId,
                     u16ServiceId
                     ) );

   it = _ServiceId2AppIdMappingTab.find( u16ServiceId );

   if ( it != _ServiceId2AppIdMappingTab.end( ) ){
      std::list < TAppListEntry >::iterator it2;

      for ( it2 = it->second.begin( ); it2 != it->second.end( ); ++it2 ){
         // App (u16AppIdToBeNotified) will be informed when app (u16SupplierAppId) (owner of service ID (u16ServiceId)) changes state
         u16SupplierAppId = it2->u16ServiceApp;
         if ( bAddEntry ){
            bAddNotificationEntry( u16SupplierAppId, u16AppIdToBeNotified, u16ServiceId, u16SourceSubId, u16TargetSubId );
         } else {
            bRemoveNotificationEntry( u16SupplierAppId, u16AppIdToBeNotified, u16ServiceId, u16SourceSubId, u16TargetSubId );
         }
      }
   } else {
      // application unknown -> remember as unknown
      if ( bAddEntry ){
         bAddNotificationEntry( u16SupplierAppId, u16AppIdToBeNotified, u16ServiceId, u16SourceSubId, u16TargetSubId );
      } else {
         bRemoveNotificationEntry( u16SupplierAppId, u16AppIdToBeNotified, u16ServiceId, u16SourceSubId, u16TargetSubId );
      }
   }

   return( TRUE );
} // bChangeServiceNotification

tVoid spm_tclCcaSupplierHandler::vSendNotification( tU16 u16SupplierAppId ){
/*!
  * \fn
  *  \brief
  *    State change of supplier application detected. Send update to all (for services of this supplier) registered application.
  *
  *  \param
  *    tU16 u16SupplierAppId: CCA Application ID of Application with state change
  *
  *  \version
  *    1.0   - Initial
  ******
  */
   std::list < TNotApp >::iterator it;

   for ( it = _mapSupplierNotificationTab[u16SupplierAppId].tNotificationList.begin( ); it != _mapSupplierNotificationTab[u16SupplierAppId].tNotificationList.end( ); ++it ){
      if ( it->u8LastSendState != _mapSupplierNotificationTab[u16SupplierAppId].u8AppStatus ){
         ETG_TRACE_USR4( ( "vSendNotification, State of App %04x changed to %u (last: %u). Msg is send to app %04x, SourceSubId %02x, TargetSubId %02x, ",
                           ETG_ENUM( ail_u16AppId,           u16SupplierAppId ),
                           ETG_ENUM( SPM_CCA_SUPPLIER_STATE, _mapSupplierNotificationTab[u16SupplierAppId].u8AppStatus ),
                           ETG_ENUM( SPM_CCA_SUPPLIER_STATE, it->u8LastSendState ),
                           ETG_ENUM( ail_u16AppId,           it->u16DstAppId ),
                           it->u16SourceSubId,
                           it->u16TargetSubId
                           ) );

         it->u8LastSendState = _mapSupplierNotificationTab[u16SupplierAppId].u8AppStatus;

         if ( it->u16ServiceId != AMT_C_U16_SERVICE_UNDEF ){
            amt_tclServiceSupplierStatus oSendSupplierState( CCA_C_U16_APP_SPM,
                                                             it->u16DstAppId,
                                                             it->u16ServiceId,
                                                             u16SupplierAppId,
                                                             _mapSupplierNotificationTab[u16SupplierAppId].u8AppStatus,
                                                             it->u16TargetSubId,
                                                             it->u16SourceSubId );
            SPM_NULL_POINTER_CHECK( _poclCcaMsgHandler );
            if ( !_poclCcaMsgHandler->bPostMessage( &oSendSupplierState ) ){
               ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
            }
         } else {
            amt_tclApplicationInfoStatus oSendAppInfoState( CCA_C_U16_APP_SPM,
                                                            it->u16DstAppId,
                                                            u16SupplierAppId,
                                                            _mapSupplierNotificationTab[u16SupplierAppId].u8AppStatus );
            SPM_NULL_POINTER_CHECK( _poclCcaMsgHandler );
            if ( !_poclCcaMsgHandler->bPostMessage( &oSendAppInfoState ) ){
               ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
            }
         }
      }
   }
} // vSendNotification

tVoid spm_tclCcaSupplierHandler::vAddNewService( tU16 u16ServiceId,
                                                 tU16 u16SupplierAppId,
                                                 tU16 u16CoreId,
                                                 tU16 u16SourceSubId,
                                                 tU16 u16TargetSubId ){
/*!
  * \fn
  *  \brief
  *    New service detected by scanning registry.
  *
  *  \param
  *    tU16 u16ServiceId:         Service ID
  *    tU16 u16SupplierAppId:     CCA Application ID -> Service Owner
  *    tU16 u16CoreId:            CORE number where SupplierApplication is located
  *
  *  \version
  *    1.0   - Initial
  ******
  */

   std::map < tU16, std::list < TAppListEntry > > ::iterator  it;
   std::list < TAppListEntry >                     ::iterator it2;
   std::map < tU32, TNotificationEntry >          ::iterator  it3;
   std::list < TNotApp >                           ::iterator it4;
   std::set < spm_tclSupplierClient >              ::iterator it5;

   std::set < spm_tclSupplierClient >                         setSupplierClients;

   ETG_TRACE_USR4( ( "bAddNewService, add new service 0x%04x from supplier %04x to list",
                     u16ServiceId,
                     ETG_ENUM( ail_u16AppId, u16SupplierAppId )
                     ) );

   // ---------------------------------------------------------------------------
   // Update Service-Id-To-Application-Id mapping table.
   // ---------------------------------------------------------------------------
   it = _ServiceId2AppIdMappingTab.find( u16ServiceId );
   if ( it == _ServiceId2AppIdMappingTab.end( ) ){
      // service not found -> addto map
      std::list < TAppListEntry > tNewService;

      TAppListEntry               tNewListEntry = { u16SupplierAppId, u16CoreId, u16SourceSubId, u16TargetSubId };

      tNewService.push_back( tNewListEntry );
      _ServiceId2AppIdMappingTab[u16ServiceId] = tNewService;
   } else {
      // service found -> check if the application ID is also the same
      for ( it2 = _ServiceId2AppIdMappingTab[u16ServiceId].begin( ); it2 != _ServiceId2AppIdMappingTab[u16ServiceId].end( ); ++it2 ){
         if ( it2->u16ServiceApp == u16SupplierAppId ){
            // exact the same entry (service + application ID) is already in list -> nothing to do
            return;
         }
      }

      // service is already in list but with another application ID --> add service with another application to the list
      TAppListEntry tNewListEntry = { u16SupplierAppId, u16CoreId, u16SourceSubId, u16TargetSubId };
      _ServiceId2AppIdMappingTab[u16ServiceId].push_back( tNewListEntry );
   }

   // ---------------------------------------------------------------------------
   // Update service supplier notification table.
   // ---------------------------------------------------------------------------

   // Check if another service supplier client application has already registered
   // for the SAME service at another service supplier application. If so ... add
   // THIS application to the service supplier notification table as well because
   // the service supplier client could be interested to register for this service
   // at THIS just added application and not the already available one.
   for ( it3 = _mapSupplierNotificationTab.begin( ); it3 != _mapSupplierNotificationTab.end( ); ++it3 ){
      for ( it4 = it3->second.tNotificationList.begin( ); it4 != it3->second.tNotificationList.end( ); ++it4 ){
         if ( it4->u16ServiceId == u16ServiceId ){
            spm_tclSupplierClient oSupplierClient( it4->u16DstAppId, it4->u16ServiceId, it4->u16SourceSubId, it4->u16TargetSubId );
            setSupplierClients.insert( oSupplierClient ); // We avoid possible duplicates by intentionally using a SET and not a MULTISET.
         }
      }
   }
   for ( it5 = setSupplierClients.begin( ); it5 != setSupplierClients.end( ); ++it5 ){
      (tVoid)bAddNotificationEntry( u16SupplierAppId, it5->u16GetAppId( ), it5->u16GetServiceId( ), it5->u16GetSourceSubId( ), it5->u16GetTargetSubId( ) );
   }
   if ( !bMoveWaitingServiceRegistration( u16ServiceId, u16SupplierAppId, u16SourceSubId, u16TargetSubId ) ){
      ETG_TRACE_COMP( ( "SPM: !!!!!! Warning detected !!!!!!" ) );
   }
} // vAddNewService

tBool spm_tclCcaSupplierHandler::bMoveWaitingServiceRegistration( tU16 u16ServiceId,
                                                                  tU16 u16SupplierAppId,
                                                                  tU16 u16SourceSubId,
                                                                  tU16 u16TargetSubId ){
/*!
  * \fn
  *  \brief
  *    Called if a new service is added to update table. Move notification entries for this service received before service is found in registry.
  *
  *  \param
  *    tU16 u16ServiceId:         Service ID
  *    tU16 u16SupplierAppId:     CCA Application ID -> Service Owner
  *
  *  \version
  *    1.0   - Initial
  ******
  */
   (tVoid)u16SourceSubId;
   (tVoid)u16TargetSubId;

   tBool                                           bRet            = TRUE;
   tBool                                           bWait           = TRUE;

   // and now check for unknown applications / service mapping
   std::map < tU32, TNotificationEntry >::iterator it;
   tU16                                            u16InvalidAppId = AMT_C_U16_APPID_INVALID;

   for ( tUInt i = 0; ( ( i < 20 ) && ( bWait ) ); ++i ){
      it = _mapSupplierNotificationTab.find( u16InvalidAppId );

      if ( it != _mapSupplierNotificationTab.end( ) ){
         // yes we have unknown App / Service mapping -> check if service is now in system
         std::list < TNotApp >::iterator it2;
         for ( it2 = it->second.tNotificationList.begin( ); it2 != it->second.tNotificationList.end( ); ++it2 ){
            // get service ID and check if already in system
            tU16 u16StoredServiceId   = it2->u16ServiceId;
            tU16 u16StoredDstId       = it2->u16DstAppId;
            tU16 u16StoredSourceSubId = it2->u16SourceSubId;
            tU16 u16StoredTargetSubId = it2->u16TargetSubId;

            if ( u16StoredServiceId == u16ServiceId ){
               // someone already waiting for service --> add to notfication table
               if ( it2 != it->second.tNotificationList.end( ) ){
                  it->second.tNotificationList.erase( it2 );
                  // and now at to mapping table
               }
               if ( !bAddNotificationEntry( u16SupplierAppId, u16StoredDstId, u16ServiceId, u16StoredSourceSubId, u16StoredTargetSubId ) ){
                  ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
               }
               break;
            }
         }
      } else {
         bWait = FALSE;
      }
   }

   return( bRet );
} // bMoveWaitingServiceRegistration

tVoid spm_tclCcaSupplierHandler::vTraceSupplierInfo( ){
/*!
  * \fn
  *  \brief
  *    HElper function to trace current supplier information via TTFIS.
  *
  *  \version
  *    1.0   - Initial
  ******
  */
                            ETG_TRACE_FATAL( ( "SUPPLIER NOTIFICATION TABLE" ) );
   std::map < tU32, TNotificationEntry >::iterator it;
   for ( it = _mapSupplierNotificationTab.begin( ); it != _mapSupplierNotificationTab.end( ); ++it ){
      tU16                            u16SupplierAppId = (tU16)it->first;

      std::list < TNotApp >::iterator it2;
      for ( it2 = it->second.tNotificationList.begin( ); it2 != it->second.tNotificationList.end( ); ++it2 ){
                            ETG_TRACE_FATAL( ( "SupplierApp %04x, current state %u | ClientAppId %04x, SourceSubId 0x%04x, TargetSubId 0x%04x, ServiceId 0x%04x, last sent state %u",
                            ETG_ENUM( ail_u16AppId,           u16SupplierAppId ),
                            ETG_ENUM( SPM_CCA_SUPPLIER_STATE, it->second.u8AppStatus ),
                            ETG_ENUM( ail_u16AppId,           it2->u16DstAppId ),
                            it2->u16SourceSubId,
                            it2->u16TargetSubId,
                            it2->u16ServiceId,
                            ETG_ENUM( SPM_CCA_SUPPLIER_STATE, it2->u8LastSendState )
                            ) );
      }
   }

                            ETG_TRACE_FATAL( ( "SUPPLIER SERVICE TABLE" ) );
   std::map < tU16, std::list < TAppListEntry > >::iterator it3;
   for ( it3 = _ServiceId2AppIdMappingTab.begin( ); it3 != _ServiceId2AppIdMappingTab.end( ); ++it3 ){
      tU16                                  u16ServiceId = it3->first;

      std::list < TAppListEntry >::iterator it2;
      for ( it2 = it3->second.begin( ); it2 != it3->second.end( ); ++it2 ){
                            ETG_TRACE_FATAL( ( "Service ID 0x%04x, supplier %04x",
                            u16ServiceId,
                            ETG_ENUM( ail_u16AppId, it2->u16ServiceApp )
                            ) );
      }
   }
} // vTraceSupplierInfo

tVoid spm_tclCcaSupplierHandler::vTraceErrors( ){
   std::map < tU32, TNotificationEntry >::iterator it;

   it = _mapSupplierNotificationTab.find( AMT_C_U16_APPID_INVALID );

   if ( ( it != _mapSupplierNotificationTab.end( ) )
        && ( it->second.tNotificationList.empty( ) == FALSE ) ){
      // some services are not found
                            ETG_TRACE_FATAL( ( "SUPPLIER NOTIFICATION TABLE with services which were requested but are unavailable: %u", (tUInt)it->second.tNotificationList.size( ) ) );

      tU16                            u16SupplierAppId = (tU16)it->first;
      std::list < TNotApp >::iterator it2;
      for ( it2 = it->second.tNotificationList.begin( ); it2 != it->second.tNotificationList.end( ); ++it2 ){
                            ETG_TRACE_FATAL( ( "SupplierApp %04x, current state %u | ClientAppId %04x, SourceSubId 0x%04x, TargetSubId 0x%04x, ServiceId 0x%04x, last sent state %u",
                            ETG_ENUM( ail_u16AppId,           u16SupplierAppId ),
                            ETG_ENUM( SPM_CCA_SUPPLIER_STATE, it->second.u8AppStatus ),
                            ETG_ENUM( ail_u16AppId,           it2->u16DstAppId ),
                            it2->u16SourceSubId,
                            it2->u16TargetSubId,
                            it2->u16ServiceId,
                            ETG_ENUM( SPM_CCA_SUPPLIER_STATE, it2->u8LastSendState )
                            ) );
      }
   } else {
                            ETG_TRACE_FATAL( ( "All services which were requested are now in SUPPLIER_STATE_AVAILABLE" ) );
   }
} // vTraceErrors

tVoid spm_tclCcaSupplierHandler::vCheckForUnavailableServices( ){
   std::map < tU32, TNotificationEntry >::iterator it;

   it = _mapSupplierNotificationTab.find( AMT_C_U16_APPID_INVALID );

   if ( it != _mapSupplierNotificationTab.end( ) ){
      // some services are not found
      tU16                            u16SupplierAppId = (tU16)it->first;

      std::list < TNotApp >::iterator it2;
      for ( it2 = it->second.tNotificationList.begin( ); it2 != it->second.tNotificationList.end( ); ++it2 ){
         // Update status to DOES_NOT_EXIST as there is no application which offers the requested service.
         it->second.u8AppStatus = AMT_C_U8_CCAMSG_SUPPLIER_STATE_DOES_NOT_EXIST;
                            ETG_TRACE_FATAL( ( "vCheckForUnavailableServices: SupplierApp %04x, current state %u | ClientAppId %04x, SourceSubId 0x%04x, TargetSubId 0x%04x, ServiceId 0x%04x, last sent state %u",
                            ETG_ENUM( ail_u16AppId,           u16SupplierAppId ),
                            ETG_ENUM( SPM_CCA_SUPPLIER_STATE, it->second.u8AppStatus ),
                            ETG_ENUM( ail_u16AppId,           it2->u16DstAppId ),
                            it2->u16SourceSubId,
                            it2->u16TargetSubId,
                            it2->u16ServiceId,
                            ETG_ENUM( SPM_CCA_SUPPLIER_STATE, it2->u8LastSendState )
                            ) );

         it2->u8LastSendState = it->second.u8AppStatus;

         amt_tclServiceSupplierStatus oSendSupplierState( CCA_C_U16_APP_SPM,
                                                          it2->u16DstAppId,
                                                          it2->u16ServiceId,
                                                          u16SupplierAppId,
                                                          it->second.u8AppStatus,
                                                          it2->u16TargetSubId,
                                                          it2->u16SourceSubId );

         SPM_NULL_POINTER_CHECK( _poclCcaMsgHandler );
         if ( !_poclCcaMsgHandler->bPostMessage( &oSendSupplierState ) ){
            ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
         }
      }
   } else {
      ETG_TRACE_FATAL( ( "All services which were requested are now in SUPPLIER_STATE_AVAILABLE" ) );
   }
} // vCheckForUnavailableServices

tBool spm_tclCcaSupplierHandler::bHandleSynchrounousCall( tU32   u32Message,
                                                          tVoid *args ){

   (tVoid)args;
   switch ( u32Message ){
      case SPM_U32_WORKER_BROADCAST_STARTUP_DELAY:
      {
         vCheckForUnavailableServices( );
      }
      break;

      default:
         // nothing to do
         break;
   }

   return( FALSE );
} // bHandleSynchrounousCall

