/*****************************************************************************
 * @file:        spm_CcaServiceHandlerAccess.cpp
 * PROJECT:      NextGen
 * SW-COMPONENT: FC SPM
 * ----------------------------------------------------------------------------
 *
 * @brief:  Automated generation of method and property variables from
 *          System Power Management Service, Application Functional Interface
 *          Version : $Revision:   1.15  $
 *          Author  : Klaus-Peter Kollai
 *          Title   : SPM_CORE_FI
 *          Date    : $Date:  08.08.2018  $
 *
 * ----------------------------------------------------------------------------
 * @copyright   (C) 2017-2020 Robert Bosch GmbH.
 *              The reproduction, distribution and utilization of this file as
 *              well as the communication of its contents to others without express
 *              authorization is prohibited. Offenders will be held liable for the
 *              payment of damages. All rights reserved in the event of the grant
 *              of a patent, utility model or design.
 *****************************************************************************/

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

// FI-zone starts
#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_STDVISITORS
#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_FUNCTIONIDS
#include "spm_fi_if.h"

#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "common_fi_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

//-----------------------------------------------------------------------------
// includes FC SPM
//-----------------------------------------------------------------------------

#include "spm_Config.h"
#include "spm_CcaServiceHandlerAccess.h"

// Spm trace specific code can be put here
// spm_tclCcaServiceHandlerAccess_include_spm: user specific code start
#include "spm_ISubStateClient.h"
#include "spm_IOsalProxy.h"
#include "spm_ISystemStateManager.h"
#include "spm_ISuperVisionClient.h"
#include "spm_IOsLinux.h"
#include "spm_ICcaServiceServer.h"
#include "spm_IWakeupHandler.h"
#include "spm_IStartupSupervisor.h"
#include "spm_IStartupSystem.h"
#include "spm_ISystemPowerManager.h"
#include "spm_IApplicationDatabase.h"
#include "spm_IApplicationConfiguration.h"
#include "spm_IStartupSystem.h"
#include "spm_IPowerSupplyStatistics.h"
#include "spm_ISystemLoadStatistics.h"
#include "spm_ILateServiceHandler.h"
#include "spm_ISoftwareVersionStatistics.h"
#include "spm_IStartupCommon.h"

#include "spm_ResetStatistics.h"
#include "spm_SystemStateStatistics.h"
#include "spm_PowerOnCount.h"
#include "spm_SoftwareBlock.h"

#include "timeConvert.h"

#include "spm_IFactory.h"

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_CSH
#include "trcGenProj/Header/spm_CcaServiceHandlerAccess.cpp.trc.h"
#endif

// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

#ifndef SPM_PRJ_SYSTEM_STATE_DOWNLOAD
   #define SPM_PRJ_SYSTEM_STATE_DOWNLOAD    SPM_SYSTEM_DOWNLOAD
#endif

#ifndef SPM_CONFIGURED_APP_2_STOPPROCESS
#define SPM_CONFIGURED_APP_2_STOPPROCESS { }
#endif
#ifndef SPM_CONFIGURED_APP_2_RESTARTPROCESS
#define SPM_CONFIGURED_APP_2_RESTARTPROCESS { }
#endif

static void writeErrMemTraceSpmCcaServiceElementIsObsolete( std::string strElement, tU16 AppId ){
      std::string strErrmemBuf = strElement + " ==> SPM Error: OBSOLETE FI !!! DO NOT USE ANYMORE." ;
      ETG_TRACE_ERRMEM( ( "App %u has requested for: %s", ETG_ENUM( ail_u16AppId, AppId ), strErrmemBuf.c_str( ) ) ) ;
}
#define SPM_CCA_SERVICE_ELEMENT_IS_OBSOLETE(strElement)\
      writeErrMemTraceSpmCcaServiceElementIsObsolete( #strElement, poSrvData->u16GetSourceAppID() );


#define SPM_CCA_SERVICE_ELEMENT_IS_OBSOLETE_NEW_VAR(strElement)\
      SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY(poclSystemPowerManager, ISpmSystemPowerManager);\
      SPM_CCA_SERVICE_ELEMENT_IS_OBSOLETE(strElement)


#define SPM_CCA_SERVICE_ELEMENT_IS_OBSOLETE_NEW_VAR_VAL(strElement)\
      SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclSystemPowerManager, ISpmSystemPowerManager);\
      SPM_CCA_SERVICE_ELEMENT_IS_OBSOLETE(strElement)


// spm_tclCcaServiceHandlerAccess_include_spm: user specific code end

///////////////////////////////////////////////////////////////////////////////
// properties need to be overloaded, if set function is available
///////////////////////////////////////////////////////////////////////////////

// spm_tclCcaServiceHandlerAccess_refactorCode_spm: user specific code start
tU8 u8ErrStateHistorySetOperation(){
   SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poHistory, ISpmSystemStateStatistics);
   poHistory->vEraseErrorHistory();
   return( IProperty::IPROPERTY_RETURN_SUCCESS );
} // u8ErrStateHistorySetOperation
template<typename T>
tVoid vErrStateHistoryGetOperation(T oHistory,spm_fi_tcl_StateChangeElement &oStateElement,tUInt pos){
      TSpmSystemStateStatisticsData  tElem;
      tclTimeConvert                oTimeConv;
      OSAL_trTimeDate               tTimeDate = { 0,0,0,0,0,0,0,0,0 };
      oHistory.s32GetElem(pos, tElem);
      oTimeConv.vGetDateFromElapsedSeconds(tElem.u32Time, &tTimeDate);
      oStateElement.tWakeupDate.s16Year  = (tS16)tTimeDate.s32Year;
      oStateElement.tWakeupDate.u8Month  = (tU8)tTimeDate.s32Month;
      oStateElement.tWakeupDate.u8Day    = (tU8)tTimeDate.s32Day;
      oStateElement.tWakeupDate.u8Hour   = (tU8)tTimeDate.s32Hour;
      oStateElement.tWakeupDate.u8Minute = (tU8)tTimeDate.s32Minute;
      oStateElement.tWakeupDate.u8Second = (tU8)tTimeDate.s32Second;
      oStateElement.tSysState.enType     = (spm_fi_tcl_SPM_e32_SYSTEM_STATES::tenType)tElem.u8SystemState;
      if (tElem.u8EntryType == SPM_STATE_HISTORY_ENTRY_WAKEUP){
         oStateElement.tElemType.enType = spm_fi_tcl_e8_StateEntryType::FI_EN_SPM_E8_HISTORYENTRYWAKEUP;
         oStateElement.tWakeup.enType   = (spm_fi_tcl_SPM_e32_WAKEUP_REASON::tenType)tElem.u.u8Wakeup;
      } else if (tElem.u8EntryType == SPM_STATE_HISTORY_ENTRY_SUBSTATE){
         oStateElement.tElemType.enType                   = spm_fi_tcl_e8_StateEntryType::FI_EN_SPM_E8_HISTORYENTRYSUBSTATE;
         oStateElement.tSubStates.tSubStateMask0.u32Value = tElem.u.tTrigger.u32TriggerUpdate[0];
         oStateElement.tSubStates.tSubStateMask1.u32Value = tElem.u.tTrigger.u32TriggerUpdate[1];
         oStateElement.tSubStates.tSubStateMask2.u32Value = tElem.u.tTrigger.u32TriggerUpdate[2];
         oStateElement.tSubStates.tSubStateData0.u32Value = tElem.u.tTrigger.u32Trigger[0];
         oStateElement.tSubStates.tSubStateData1.u32Value = tElem.u.tTrigger.u32Trigger[1];
         oStateElement.tSubStates.tSubStateData2.u32Value = tElem.u.tTrigger.u32Trigger[2];
      } else if (tElem.u8EntryType == SPM_STATE_HISTORY_ENTRY_RESET){
         oStateElement.tElemType.enType = spm_fi_tcl_e8_StateEntryType::FI_EN_SPM_E8_HISTORYENTRYRESET;
         oStateElement.tReset.enType    = (spm_fi_tcl_e8_LcmResetReason::tenType)tElem.u.u8Reset;
      }
}//vErrStateHistoryGetOperation

tVoid vSystemStateElementToFiType( TSpmSystemStateStatisticsData *tSystemStateData, spm_fi_tcl_StateChangeElement *fiStateChangeElement ){
   tclTimeConvert                oTimeConv;
   OSAL_trTimeDate               tTimeDate = { 0,0,0,0,0,0,0,0,0 };

   oTimeConv.vGetDateFromElapsedSeconds(tSystemStateData->u32Time, &tTimeDate);
   fiStateChangeElement->tWakeupDate.s16Year  = (tS16)tTimeDate.s32Year;
   fiStateChangeElement->tWakeupDate.u8Month  = (tU8)tTimeDate.s32Month;
   fiStateChangeElement->tWakeupDate.u8Day    = (tU8)tTimeDate.s32Day;
   fiStateChangeElement->tWakeupDate.u8Hour   = (tU8)tTimeDate.s32Hour;
   fiStateChangeElement->tWakeupDate.u8Minute = (tU8)tTimeDate.s32Minute;
   fiStateChangeElement->tWakeupDate.u8Second = (tU8)tTimeDate.s32Second;
   fiStateChangeElement->tSysState.enType     = (spm_fi_tcl_SPM_e32_SYSTEM_STATES::tenType)tSystemStateData->u8SystemState;
   if (tSystemStateData->u8EntryType == SPM_STATE_HISTORY_ENTRY_WAKEUP){
      fiStateChangeElement->tElemType.enType = spm_fi_tcl_e8_StateEntryType::FI_EN_SPM_E8_HISTORYENTRYWAKEUP;
      fiStateChangeElement->tWakeup.enType   = (spm_fi_tcl_SPM_e32_WAKEUP_REASON::tenType)tSystemStateData->u.u8Wakeup;
   } else if (tSystemStateData->u8EntryType == SPM_STATE_HISTORY_ENTRY_SUBSTATE){
      fiStateChangeElement->tElemType.enType                   = spm_fi_tcl_e8_StateEntryType::FI_EN_SPM_E8_HISTORYENTRYSUBSTATE;
      fiStateChangeElement->tSubStates.tSubStateMask0.u32Value = tSystemStateData->u.tTrigger.u32TriggerUpdate[0];
      fiStateChangeElement->tSubStates.tSubStateMask1.u32Value = tSystemStateData->u.tTrigger.u32TriggerUpdate[1];
      fiStateChangeElement->tSubStates.tSubStateMask2.u32Value = tSystemStateData->u.tTrigger.u32TriggerUpdate[2];
      fiStateChangeElement->tSubStates.tSubStateData0.u32Value = tSystemStateData->u.tTrigger.u32Trigger[0];
      fiStateChangeElement->tSubStates.tSubStateData1.u32Value = tSystemStateData->u.tTrigger.u32Trigger[1];
      fiStateChangeElement->tSubStates.tSubStateData2.u32Value = tSystemStateData->u.tTrigger.u32Trigger[2];
   } else if (tSystemStateData->u8EntryType == SPM_STATE_HISTORY_ENTRY_RESET){
      fiStateChangeElement->tElemType.enType = spm_fi_tcl_e8_StateEntryType::FI_EN_SPM_E8_HISTORYENTRYRESET;
      fiStateChangeElement->tReset.enType    = (spm_fi_tcl_e8_LcmResetReason::tenType)tSystemStateData->u.u8Reset;
   }
}//vSystemStateElementToFiType

tVoid vResetInfoToFiType( TSpmResetStatisticsData *tResetInfoData, spm_fi_tcl_SPM_ResetInfoStatisticData *fiResetInfo ){
   tclTimeConvert                      oTimeConv;
   OSAL_trTimeDate                     tTimeDate = { 0,0,0,0,0,0,0,0,0 };

   oTimeConv.vGetDateFromElapsedSeconds( tResetInfoData->tHmiReset.u32LastResetTime, &tTimeDate );

   // update reset reason status information
   fiResetInfo->tLastResetDate.u8Hour    = (tU8)tTimeDate.s32Hour;
   fiResetInfo->tLastResetDate.u8Minute  = (tU8)tTimeDate.s32Minute;
   fiResetInfo->tLastResetDate.u8Second  = (tU8)tTimeDate.s32Second;
   fiResetInfo->tLastResetDate.u8Day     = (tU8)tTimeDate.s32Day;
   fiResetInfo->tLastResetDate.u8Month   = (tU8)tTimeDate.s32Month;
   fiResetInfo->tLastResetDate.s16Year   = (tS16)tTimeDate.s32Year;

   fiResetInfo->tLastResetReason.enType  = (spm_fi_tcl_e8_LcmResetReason::tenType)tResetInfoData->tHmiReset.u32LastResetType;

   fiResetInfo->u32ApApplicationError    = tResetInfoData->tHmiReset.u32ApplicationError;
   fiResetInfo->u32ApSoftware            = tResetInfoData->tHmiReset.u32Software;
   fiResetInfo->u32ApEcuReset            = tResetInfoData->tHmiReset.u32EcuReset;
   fiResetInfo->u32ApDownload            = tResetInfoData->tHmiReset.u32Download;
   fiResetInfo->u32ApDownloadError       = tResetInfoData->tHmiReset.u32DownloadError;
   fiResetInfo->u32ApOvertemperature     = tResetInfoData->tHmiReset.u32Overtemperature;
   fiResetInfo->u32ApLcmIntern           = tResetInfoData->tHmiReset.u32LcmIntern;
   fiResetInfo->u32ApUser                = tResetInfoData->tHmiReset.u32User;
   fiResetInfo->u32ApPowerOnCount        = tResetInfoData->tHmiReset.u32PowerOn;
   fiResetInfo->u32ApPowerLossCount      = tResetInfoData->tHmiReset.u32PowerLoss;
   fiResetInfo->u32ApPowerOffCount       = tResetInfoData->tHmiReset.u32PowerOff;
   fiResetInfo->u32ApSccCount            = tResetInfoData->tHmiReset.u32Scc;
   fiResetInfo->u32ApUnknownCount        = tResetInfoData->tHmiReset.u32Unknown;

   fiResetInfo->u32SccHwWdtCount         = tResetInfoData->tSccReset.u32HwWdg;
   fiResetInfo->u32SccPowerOnCount       = tResetInfoData->tSccReset.u32PowerOn;
   fiResetInfo->u32SccColdstartCount     = tResetInfoData->tSccReset.u32Coldstart;
   fiResetInfo->u32SccAppmodeChangeCount = tResetInfoData->tSccReset.u32AppmodeChange;
   fiResetInfo->u32SccDuringLpwCount     = tResetInfoData->tSccReset.u32DuringLpw;
   fiResetInfo->u32SccDuringPllOszCount  = tResetInfoData->tSccReset.u32DuringPllOsz;
   fiResetInfo->u32SccDuringSWCount      = tResetInfoData->tSccReset.u32DuringSw;
   fiResetInfo->u32SccWarmstartCount     = tResetInfoData->tSccReset.u32Warmstart;

}//vResetInfoToFiType

tVoid vResetElementToFiType( TSpmResetElem *tResetElemData, spm_fi_tcl_ResetElement *fiResetElement ){
   tclTimeConvert          oTimeConv;
   OSAL_trTimeDate         tTimeDate = { 0,0,0,0,0,0,0,0,0 };

   oTimeConv.vGetDateFromElapsedSeconds( tResetElemData->u32Time, &tTimeDate );
   fiResetElement->tResetDate.s16Year                = (tS16)tTimeDate.s32Year;
   fiResetElement->tResetDate.u8Month                = (tU8)tTimeDate.s32Month;
   fiResetElement->tResetDate.u8Day                  = (tU8)tTimeDate.s32Day;
   fiResetElement->tResetDate.u8Hour                 = (tU8)tTimeDate.s32Hour;
   fiResetElement->tResetDate.u8Minute               = (tU8)tTimeDate.s32Minute;
   fiResetElement->tResetDate.u8Second               = (tU8)tTimeDate.s32Second;

   fiResetElement->tSwVersionInfo.tSwBuildVersion    = (tString)tResetElemData->strBuildVersion;
   fiResetElement->tSwVersionInfo.tSwCustomerVersion = (tString)tResetElemData->strCustomerVersion;

   fiResetElement->eReset.enType                     = (spm_fi_tcl_e8_LcmResetReason::tenType)tResetElemData->u16ResetType;
}//vResetElementToFiType

tVoid vWakeupElementToFiType( TSpmWakeupStatisticsData *tWakeupData, spm_fi_tcl_WakeupElement *fiWakeupElement ){
   tclTimeConvert           oTimeConv;
   OSAL_trTimeDate          tTimeDate = { 0,0,0,0,0,0,0,0,0 };

   oTimeConv.vGetDateFromElapsedSeconds( tWakeupData->u32Time, &tTimeDate );

   fiWakeupElement->tWakeupDate.s16Year  = (tS16)tTimeDate.s32Year;
   fiWakeupElement->tWakeupDate.u8Month  = (tU8)tTimeDate.s32Month;
   fiWakeupElement->tWakeupDate.u8Day    = (tU8)tTimeDate.s32Day;
   fiWakeupElement->tWakeupDate.u8Hour   = (tU8)tTimeDate.s32Hour;
   fiWakeupElement->tWakeupDate.u8Minute = (tU8)tTimeDate.s32Minute;
   fiWakeupElement->tWakeupDate.u8Second = (tU8)tTimeDate.s32Second;

   fiWakeupElement->eWakeupReason.enType = (spm_fi_tcl_SPM_e32_WAKEUP_REASON::tenType)( tWakeupData->u32WakeUpType );
}//vWakeupElementToFiType

tVoid vSwitchOnElemToFiType( TSpmSwitchOnStatisticsData *tSwitchOnData, spm_fi_tcl_SwitchOnElement *fiSwitchOnElement ){
   tclTimeConvert             oTimeConv;
   OSAL_trTimeDate            tTimeDate = { 0,0,0,0,0,0,0,0,0 };

   oTimeConv.vGetDateFromElapsedSeconds( tSwitchOnData->u32Time, &tTimeDate );

   fiSwitchOnElement->tSwitchOnDate.s16Year  = (tS16)tTimeDate.s32Year;
   fiSwitchOnElement->tSwitchOnDate.u8Month  = (tU8)tTimeDate.s32Month;
   fiSwitchOnElement->tSwitchOnDate.u8Day    = (tU8)tTimeDate.s32Day;
   fiSwitchOnElement->tSwitchOnDate.u8Hour   = (tU8)tTimeDate.s32Hour;
   fiSwitchOnElement->tSwitchOnDate.u8Minute = (tU8)tTimeDate.s32Minute;
   fiSwitchOnElement->tSwitchOnDate.u8Second = (tU8)tTimeDate.s32Second;

   fiSwitchOnElement->eSwitchOnReason.enType = (spm_fi_tcl_SPM_e32_SubStateType::tenType)tSwitchOnData->u32WakeUpType;
}//vSwitchOnElemToFiType

tVoid vSystemLoadStatistcToFiType( TSpmSystemLoadStatisticsData *tSystemLoadData, spm_fi_tcl_SPM_SystemStatisticsData *fiSystemLoadData ){
   if ( ( tSystemLoadData != NULL ) && ( fiSystemLoadData != NULL ) ){
      fiSystemLoadData->MaxValue = tSystemLoadData->u32MaxValue;
      fiSystemLoadData->AvgValue = tSystemLoadData->u32AvgValue;
      fiSystemLoadData->MinValue = tSystemLoadData->u32MinValue;
      fiSystemLoadData->StatisticDataDisabled = tSystemLoadData->bStatisticDataDisabled;
   }
}//vSystemLoadStatistcToFiType

tVoid vSWVersionStatisticToFiType( TSpmSoftwareVersionStatisticsData* tSWVersionData, spm_fi_tcl_SPM_SWVersionStatisticData *fiSWVersionData ){
   if ( ( tSWVersionData != NULL ) && ( fiSWVersionData != NULL ) ) {
      fiSWVersionData->strBuildVersion = tSWVersionData->tSWVersion.strBuildVersion;
      fiSWVersionData->strBuildVersionCustomer = tSWVersionData->tSWVersion.strBuildVersionCustomer;
      fiSWVersionData->strCustomerVersion = tSWVersionData->tSWVersion.strCustomerVersion;
      vResetInfoToFiType( &( tSWVersionData->tResetStatisticsData ), &( fiSWVersionData->tResetInfo ) );
      for( tUInt i = 0; i < SPM_U32_STATISTIC_MAX_RESET_HISTORY_LIST; ++i ){
         spm_fi_tcl_ResetElement oFiResetElement;
         vResetElementToFiType( &( tSWVersionData->tResetHistory[i] ), &oFiResetElement );
         fiSWVersionData->tResetHistoryList.push_back( oFiResetElement );
      }
      for( tUInt i = 0; i < SPM_U32_STATISTIC_MAX_WAKEUP_HISTORY_LIST; ++i ){
         spm_fi_tcl_WakeupElement oFiWakeupElement;
         vWakeupElementToFiType( &( tSWVersionData->tWakeupStatisticsData[i] ), &oFiWakeupElement );
         fiSWVersionData->tWakeupHistoryList.push_back( oFiWakeupElement );
      }
      for( tUInt i = 0; i < SPM_U32_STATISTIC_MAX_SWITCH_ON_HISTORY_LIST; ++i ){
         spm_fi_tcl_SwitchOnElement oFiSwitchOnElement;
         vSwitchOnElemToFiType( &( tSWVersionData->tSwitchOnStatisticsData[i] ), &oFiSwitchOnElement );
         fiSWVersionData->tSwitchOnList.push_back( oFiSwitchOnElement );
      }
      for( tUInt i = 0; i < SPM_U32_STATISTIC_MAX_SYS_STATE_HISTORY_LIST; ++i ){
         spm_fi_tcl_StateChangeElement oFiStateChangeElement;
         vSystemStateElementToFiType( &( tSWVersionData->tSystemStateHistory[i] ), &oFiStateChangeElement );
         fiSWVersionData->tStateHistoryList.push_back( oFiStateChangeElement );
      }
      for( tUInt i = 0; i < SPM_U32_STATISTIC_MAX_ERR_ENTRIES_PER_STACKS; ++i ){
         spm_fi_tcl_StateChangeElement oFiStateChangeElementErr01;
         vSystemStateElementToFiType( &( tSWVersionData->tSystemStateErrorHistory0[i] ), &oFiStateChangeElementErr01 );
         fiSWVersionData->tErrStateHistory01List.push_back( oFiStateChangeElementErr01 );
      }

      for( tUInt i = 0; i < SPM_U32_STATISTIC_MAX_ERR_ENTRIES_PER_STACKS; ++i ){
         spm_fi_tcl_StateChangeElement oFiStateChangeElementErr02;
         vSystemStateElementToFiType( &( tSWVersionData->tSystemStateErrorHistory1[i] ), &oFiStateChangeElementErr02 );
         fiSWVersionData->tErrStateHistory02List.push_back( oFiStateChangeElementErr02 );
      }
      for( tUInt i = 0; i < SPM_U32_STATISTIC_MAX_ERR_ENTRIES_PER_STACKS; ++i ){
         spm_fi_tcl_StateChangeElement oFiStateChangeElementErr03;
         vSystemStateElementToFiType( &( tSWVersionData->tSystemStateErrorHistory2[i] ), &oFiStateChangeElementErr03 );
         fiSWVersionData->tErrStateHistory03List.push_back( oFiStateChangeElementErr03 );
      }

      for( tUInt i = 0; i < SPM_U32_STATISTIC_MAX_ERR_ENTRIES_PER_STACKS; ++i ){
         spm_fi_tcl_StateChangeElement oFiStateChangeElementErr04;
         vSystemStateElementToFiType( &( tSWVersionData->tSystemStateErrorHistory3[i] ), &oFiStateChangeElementErr04 );
         fiSWVersionData->tErrStateHistory04List.push_back( oFiStateChangeElementErr04 );
      }
      for( tUInt i = 0; i < SPM_U32_STATISTIC_MAX_ERR_ENTRIES_PER_STACKS; ++i ){
         spm_fi_tcl_StateChangeElement oFiStateChangeElementErr05;
         vSystemStateElementToFiType( &( tSWVersionData->tSystemStateErrorHistory4[i] ), &oFiStateChangeElementErr05 );
         fiSWVersionData->tErrStateHistory05List.push_back( oFiStateChangeElementErr05 );
      }
      fiSWVersionData->u32SystemOperationTime = tSWVersionData->u32SystemOperationTime;
      vSystemLoadStatistcToFiType( &( tSWVersionData->tCpuUsage ), &( fiSWVersionData->tCpuUsage ) );
      vSystemLoadStatistcToFiType( &( tSWVersionData->tRamUsage ), &( fiSWVersionData->tRamUsage ) );
      vSystemLoadStatistcToFiType( &( tSWVersionData->tStartupTime ), &( fiSWVersionData->tStartupTime ) );
   }
}//vSWVersionStatisticToFiType

// spm_tclCcaServiceHandlerAccess_refactorCode_spm: user specific code end

const spm_PropertySubStates& spm_PropertySubStates::operator=(const spm_corefi_tclMsgSubStatesStatus& op){
   _property = op;
   return( * this );
}

tU8 spm_PropertySubStates::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgSubStatesSet oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'SubStates': user specific code start
      SPM_CCA_SERVICE_ELEMENT_IS_OBSOLETE_NEW_VAR_VAL("Property Substates --> Set");

      SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclSubStateHandler, ISpmSubStateClient);

      tU32                u32SubStateType     = (tU32)oResultData.SubStateType.enType;
      tBool               bSubStateData       = oResultData.SubStateData;

      poclSubStateHandler->vTriggerSubStateType( u32SubStateType, bSubStateData );
      return( IPROPERTY_RETURN_SUCCESS );
      // u8SetOperation 'SubStates': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertyPhoneMuteState::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'PhoneMuteState': user specific code start
   SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY(poclSubStateHandler, ISpmSubStateClient);

   if ( poclSubStateHandler->bIsTriggerSet( SPM_U32_PHONE_MUTE ) ){
      _property.PhoneMuteState.enType = spm_fi_tcl_SPM_e32_PHONE_MUTE_STATE::FI_EN_SPM_U32_PHONEMUTE_ACTIVE;
   } else {
      _property.PhoneMuteState.enType = spm_fi_tcl_SPM_e32_PHONE_MUTE_STATE::FI_EN_SPM_U32_PHONEMUTE_INACTIVE;
   }
   // vGetOperation 'PhoneMuteState': user specific code end
} // vGetOperation

tVoid spm_PropertyOnTipperState::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'OnTipperState': user specific code start

   SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY(poclSubStateHandler, ISpmSubStateClient);

   if ( poclSubStateHandler->bIsTriggerSet( SPM_U32_ON_TIPPER ) ){
      _property.OnTipperState.enType = spm_fi_tcl_SPM_e32_KEY_STATE::FI_EN_SPM_U32_PRESSED;
   } else {
      _property.OnTipperState.enType = spm_fi_tcl_SPM_e32_KEY_STATE::FI_EN_SPM_U32_RELEASED;
   }

   // vGetOperation 'OnTipperState': user specific code end
} // vGetOperation

tVoid spm_PropertyMapDownloadLockState::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'MapDownloadLockState': user specific code start
   dp_tclSpmDpInternDataMapDlLockApp oDpMapDlLcokApp;
   _property.u16AppIdLockOwner = oDpMapDlLcokApp.tGetData( );
   // vGetOperation 'MapDownloadLockState': user specific code end
} // vGetOperation

tU8 spm_PropertyStandbyTime::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgStandbyTimeSet oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'StandbyTime': user specific code start
      SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclSystemStateManager, ISpmSystemStateManager);

      poclSystemStateManager->vSetStateTime( spm_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_STANDBY, oResultData.StandbyTime * 60 * 1000 );

      return( IPROPERTY_RETURN_SUCCESS );
      // u8SetOperation 'StandbyTime': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertyStandbyTime::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'StandbyTime': user specific code start
   SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY(poclSystemStateManager, ISpmSystemStateManager);

   _property.StandbyTime = 0;

   //getting value in milliseconds
   _property.StandbyTime = poclSystemStateManager->u32GetStateTime( spm_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_STANDBY );
   //converting StandbyTime to minutes
   _property.StandbyTime = ( _property.StandbyTime ) / ( 60 * 1000 );

   // vGetOperation 'StandbyTime': user specific code end
} // vGetOperation

tU8 spm_PropertyPersitentStandbyTime::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgPersitentStandbyTimeSet oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'PersitentStandbyTime': user specific code start
      tU32 u32StandbyPersTime = oResultData.PersitentStandbyTime;

      if ( SPM_STDBY_TIME_SEC_MAX > u32StandbyPersTime ){
         dp_tclSpmDpConfigStandbyTime    oStandby;
         dp_tclSpmDpConfigDevStandbyTime oDevStandby;
         oStandby << u32StandbyPersTime;
         // oStandby.vSetData(u32StandbyPersTime);
         oDevStandby << u32StandbyPersTime;
         // oDevStandby.vSetData(u32StandbyPersTime);

         // set standby time now
         SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclSystemStateManager, ISpmSystemStateManager);

         poclSystemStateManager->vSetStateTime( spm_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_STANDBY, u32StandbyPersTime * 1000 );

         return( IPROPERTY_RETURN_SUCCESS );
      }
      // u8SetOperation 'PersitentStandbyTime': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertyPersitentStandbyTime::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'PersitentStandbyTime': user specific code start
   dp_tclSpmDpConfigStandbyTime oStandby;
   // _property.PersitentStandbyTime = oStandby.u32StandbyTime;
   oStandby >> _property.PersitentStandbyTime;
   // vGetOperation 'PersitentStandbyTime': user specific code end
} // vGetOperation

tU8 spm_PropertyIgnitionOffTime::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgIgnitionOffTimeSet oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'IgnitionOffTime': user specific code start
      tU32 u32IgnitionOffTime = oResultData.IgnitionOffTime;

      if ( SPM_IGNITION_OFF_TIME_MIN_MAX > u32IgnitionOffTime ){
         dp_tclSpmDpConfigIgnitionOffTime oIgnOff;
         oIgnOff.vSetData( u32IgnitionOffTime );

         #if 0
            SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclSubStateHandler, ISpmSubStateClient);

            poclSubStateHandler->bSetIgnitionOffTime( u32IgnitionOffTime );
         #endif

         ISpmSupervisionClient *poclSupervisionClientIgnOff = dynamic_cast < ISpmSupervisionClient* >( ISpmFactory::_poFactoryRef->getSpmObjHandler( "spm_tclIgnitionOffSupervisor" ) );
         if ( poclSupervisionClientIgnOff ){
            poclSupervisionClientIgnOff->vTriggerSupervisionState( );
         }
         return( IPROPERTY_RETURN_SUCCESS );
      }
      // u8SetOperation 'IgnitionOffTime': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertyIgnitionOffTime::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'IgnitionOffTime': user specific code start
   dp_tclSpmDpConfigIgnitionOffTime oIgnOff;
   _property.IgnitionOffTime = oIgnOff.tGetData( );
   // vGetOperation 'IgnitionOffTime': user specific code end
} // vGetOperation

tU8 spm_PropertyDoorOpenTime::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgDoorOpenTimeSet oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'DoorOpenTime': user specific code start
      tU32 u32DoorOpenTime = oResultData.DoorOpenTime;

      if ( SPM_DOOR_OPEN_TIME_MIN_MAX > u32DoorOpenTime ){
         // set DoorOpen time
         dp_tclSpmDpConfigDoorOpenTime oDoorOpen;
         oDoorOpen.vSetData( u32DoorOpenTime * 60 );

         SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclSystemStateManager, ISpmSystemStateManager);

         poclSystemStateManager->vSetStateTime( spm_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_DOOR_OPEN, u32DoorOpenTime * 1000 * 60 );

         return( IPROPERTY_RETURN_SUCCESS );
      }
      // u8SetOperation 'DoorOpenTime': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertyDoorOpenTime::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'DoorOpenTime': user specific code start
   dp_tclSpmDpConfigDoorOpenTime oDoorOpen;
   _property.DoorOpenTime = oDoorOpen.tGetData( ) / 60;
   // vGetOperation 'DoorOpenTime': user specific code end
} // vGetOperation

tU8 spm_PropertyClockDisplayTime::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgClockDisplayTimeSet oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'ClockDisplayTime': user specific code start
      tU32 u32ClockDisplayTime = oResultData.ClockDisplayTime;

      if ( SPM_CLOCK_DISPLAY_TIME_MIN_MAX > u32ClockDisplayTime ){
         // set ClockDisplay time
         dp_tclSpmDpConfigClockDisplayTime oClckDisplay;
         oClckDisplay.vSetData( u32ClockDisplayTime * 60 );
         return( IPROPERTY_RETURN_SUCCESS );
      }
      // u8SetOperation 'ClockDisplayTime': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertyClockDisplayTime::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'ClockDisplayTime': user specific code start
   dp_tclSpmDpConfigClockDisplayTime oClckDisplay;
   _property.ClockDisplayTime = oClckDisplay.tGetData( ) / 60;
   // vGetOperation 'ClockDisplayTime': user specific code end
} // vGetOperation

tU8 spm_PropertySystemStateTime::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgSystemStateTimeSet oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'SystemStateTime': user specific code start
      SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclSystemStateManager, ISpmSystemStateManager);
      poclSystemStateManager->vSetStateTime( (tU32)oResultData.tSystemState.enType, oResultData.tStateTimerValue * 1000 );

      return( IPROPERTY_RETURN_SUCCESS );
      // u8SetOperation 'SystemStateTime': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertySystemStateTime::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'SystemStateTime': user specific code start
   fi_tclVisitorMessage                oMsg( poSrvData );

   // Create Object..
   spm_corefi_tclMsgSystemStateTimeGet oResultData;

   if ( OSAL_OK == oMsg.s32GetData( oResultData ) ){
      SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY(poclSystemStateManager, ISpmSystemStateManager);
      tU32                    u32Time                = poclSystemStateManager->u32GetStateTime( (tU32)oResultData.tSystemState.enType );
      _property.tSystemState.enType = oResultData.tSystemState.enType;
      if ( u32Time != ( tU32 ) - 1 ){
         u32Time = u32Time / 1000;
      }
      _property.tStateTimerValue    = u32Time;
   }
   // vGetOperation 'SystemStateTime': user specific code end
} // vGetOperation

tVoid spm_PropertyProcessList::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'ProcessList': user specific code start
   _property.tProcesses.tProcesses.clear( );


   SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY(poclStartupCommon, ISpmStartupCommon);
   SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY(poclStartupSystem, ISpmStartupSystem);

   if ( poclStartupCommon ){
       const TMapProcessConfiguration& oMapProcessConfig = poclStartupCommon->oGetProcessConfig( );
       for ( TMapProcessConfiguration::const_iterator posProc = oMapProcessConfig.begin( ); posProc != oMapProcessConfig.end( ); ++posProc ){
           tclTimeConvert            oTimeConv;
           OSAL_trTimeDate           tTimeDate = { 0,0,0,0,0,0,0,0,0 };

           oTimeConv.vGetDateFromElapsedSeconds( posProc->second.u32StartTimeAbs, &tTimeDate );

           spm_fi_tcl_ProcessElement oProcElement;

           oProcElement.tSwitchOnDate.s16Year  = (tS16)tTimeDate.s32Year;
           oProcElement.tSwitchOnDate.u8Month  = (tU8)tTimeDate.s32Month;
           oProcElement.tSwitchOnDate.u8Day    = (tU8)tTimeDate.s32Day;
           oProcElement.tSwitchOnDate.u8Hour   = (tU8)tTimeDate.s32Hour;
           oProcElement.tSwitchOnDate.u8Minute = (tU8)tTimeDate.s32Minute;
           oProcElement.tSwitchOnDate.u8Second = (tU8)tTimeDate.s32Second;

           oProcElement.u32StartTimeSinceStart = posProc->second.u32StartTimeRel;

           oProcElement.tProcessName           = posProc->second.strProcessLocation.c_str( );
           oProcElement.tConfiguredProcessName = posProc->second.strProcessName.c_str( );

           oProcElement.eProcState.enType = spm_fi_tcl_e8_ProcessState::FI_EN_SPM_E8_PROCESSTOSTART;
           if ( posProc->second.bProcSpawned ){
               oProcElement.eProcState.enType = spm_fi_tcl_e8_ProcessState::FI_EN_SPM_E8_PROCESSRUNNING;
           } else {
               if ( posProc->second.u32StartTimeRel == 0){
                   if ( poclStartupSystem->bAlternativeProcConfig( (tChar*)posProc->second.strProcessName.c_str( ), TRUE) ) {
                       oProcElement.eProcState.enType = spm_fi_tcl_e8_ProcessState::FI_EN_SPM_E8_PROCESSTOSTART;
                   } else {
                       oProcElement.eProcState.enType = spm_fi_tcl_e8_ProcessState::FI_EN_SPM_E8_PROCESSCONFIGUREDOUT;
                   }
               } else {
                   oProcElement.eProcState.enType = spm_fi_tcl_e8_ProcessState::FI_EN_SPM_E8_PROCESSDOWN;
               }
           }

           _property.tProcesses.tProcesses.push_back( oProcElement );

           ETG_TRACE_USR1( ( "spm_PropertyProcessList::vGetOperation(): %60s: State: %u, %04d-%02d-%02d  %02d:%02d:%02d (relative: %dms)", posProc->second.strProcessName.c_str( ),
                   ETG_ENUM(SPM_PROCESSSTATE , (tU32)oProcElement.eProcState.enType),
                   oProcElement.tSwitchOnDate.s16Year, oProcElement.tSwitchOnDate.u8Month, oProcElement.tSwitchOnDate.u8Day,
                   oProcElement.tSwitchOnDate.u8Hour, oProcElement.tSwitchOnDate.u8Minute, oProcElement.tSwitchOnDate.u8Second,
                   oProcElement.u32StartTimeSinceStart ) );

       }

   }


   // vGetOperation 'ProcessList': user specific code end
} // vGetOperation

const spm_PropertyWakeupReason& spm_PropertyWakeupReason::operator=(const spm_corefi_tclMsgWakeupReasonStatus& op){
   _property = op;
   return( * this );
}

tVoid spm_PropertyWakeupReason::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'WakeupReason': user specific code start
   tU32               u32WakeUpReason   = 0;

   SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY(poclWakeupHandler, ISpmWakeupHandler);

   u32WakeUpReason               = poclWakeupHandler->u32GetWakeupReason( );
   _property.WakeupReason.enType = (spm_fi_tcl_SPM_e32_WAKEUP_REASON::tenType)u32WakeUpReason;
   // vGetOperation 'WakeupReason': user specific code end
} // vGetOperation

tVoid spm_PropertyStartupReason::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'StartupReason': user specific code start
   SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY(poclSubStateHandler, ISpmSubStateClient);

   tU32                u32StartReasons     = poclSubStateHandler->u32GetStartupOnReason( );
   _property.StartupReason.enType = (spm_fi_tcl_SPM_e32_STARTUP_REASON::tenType)0;
   if ( SPM_BC0_EJECT & u32StartReasons ){
      _property.StartupReason.enType = spm_fi_tcl_SPM_e32_STARTUP_REASON::FI_EN_SPM_U32_STARTUP_ON_EJECT;
   }
   ETG_TRACE_USR1( ( "spm_PropertyStartupReason::vGetOperation(): Startup Reason requested: %d.", _property.StartupReason.enType ) );
   // vGetOperation 'StartupReason': user specific code end
} // vGetOperation

tVoid spm_PropertyConnectToBattery::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'ConnectToBattery': user specific code start
   dp_tclSpmDpInternDataConnectToBattery      oConnect2Bat;
   dp_tclSpmDpInternDataConnectToBatteryCount oConnBatCount;

   _property.ConnectToBatteryState = oConnect2Bat.tGetData( );
   _property.ConnectToBatteryCount = oConnBatCount.tGetData( );

   // vGetOperation 'ConnectToBattery': user specific code end
} // vGetOperation

tVoid spm_PropertyUBatSense::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'UBatSense': user specific code start

   SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY(poclOsalProxy, ISpmOsalProxy);

   poclOsalProxy->bUbatSense( &_property.UBatSense );
   // vGetOperation 'UBatSense': user specific code end
} // vGetOperation

tU8 spm_PropertyDevicePowerOnTime::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgDevicePowerOnTimeSet oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'DevicePowerOnTime': user specific code start
      ISpmPowerOnStatistics *poPwrOn = dynamic_cast < ISpmPowerOnStatistics* >( ISpmFactory::_poFactoryRef->getSpmObjHandler( "spm_tclPowerOnCount" ) );
      SPM_NULL_POINTER_CHECK_VAL( poPwrOn );
      poPwrOn->vSetPwrOnTime( (tU32)oResultData.DevicePowerOnTime );
      return( IPROPERTY_RETURN_SUCCESS );
      // u8SetOperation 'DevicePowerOnTime': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertyDevicePowerOnTime::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'DevicePowerOnTime': user specific code start
   dp_tclSpmDpPowOnPowerOnCount oPwrOn;
   _property.DevicePowerOnTime = oPwrOn.tGetData( );
   // vGetOperation 'DevicePowerOnTime': user specific code end
} // vGetOperation

tVoid spm_PropertyLastSystemRuntime::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'LastSystemRuntime': user specific code start
   dp_tclSpmDpPowOnLastSystemRunTime oSysOnTime;
   _property.u32RunTime = oSysOnTime.tGetData( );
   // vGetOperation 'LastSystemRuntime': user specific code end
} // vGetOperation

tVoid spm_PropertyCurrentSystemRuntime::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'CurrentSystemRuntime': user specific code start
   _property.u32RunTime = OSAL_ClockGetElapsedTime( ) / 1000;
   // vGetOperation 'CurrentSystemRuntime': user specific code end
} // vGetOperation

tU8 spm_PropertyResetList::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgResetListSet oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'ResetList': user specific code start
	  SPM_CCA_SERVICE_ELEMENT_IS_OBSOLETE_NEW_VAR_VAL("Property ResetList --> Set");

      ISpmStatistics *poRstStat = dynamic_cast < ISpmStatistics* >( ISpmFactory::_poFactoryRef->getSpmObjHandler( "spm_tclResetStatistics" ) );
      SPM_NULL_POINTER_CHECK_VAL( poRstStat );
      poRstStat->vErase( );
      return( IPROPERTY_RETURN_SUCCESS );
      // u8SetOperation 'ResetList': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertyResetList::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'ResetList': user specific code start
   SPM_CCA_SERVICE_ELEMENT_IS_OBSOLETE_NEW_VAR("Property ResetList --> Get");

   TSpmResetStatisticsData tResetStatistics = TSpmResetStatisticsData( );
   TSpmResetElem           tRstElem         = {
      0, 0, { 0 }, { 0 }
   };
   OSAL_trTimeDate         tTimeDate        = { 0,0,0,0,0,0,0,0,0 };

   ISpmStatistics         *poRstStat        = dynamic_cast < ISpmStatistics* >( ISpmFactory::_poFactoryRef->getSpmObjHandler( "spm_tclResetStatistics" ) );
   SPM_NULL_POINTER_CHECK( poRstStat );
   poRstStat->vGetLastReset( &tRstElem );

   tclTimeConvert          oTimeConv;
   oTimeConv.vGetDateFromElapsedSeconds( tRstElem.u32Time, &tTimeDate );

   _property.LastResetReason.enType   = (spm_fi_tcl_e8_LcmResetReason::tenType)tRstElem.u16ResetType;

   _property.DateHour                 = (tU8)tTimeDate.s32Hour;
   _property.DateMinute               = (tU8)tTimeDate.s32Minute;
   _property.DateDay                  = (tU8)tTimeDate.s32Day;
   _property.DateMonth                = (tU8)tTimeDate.s32Month;
   _property.DateYear                 = (tU16)tTimeDate.s32Year;

   // not used in GEN2 -> no COPRO / FGS
   _property.CoproResetByHmi          = 0;
   _property.CoproResetByUndervoltage = 0;
   _property.HmiResetByCoproStartup   = 0;
   _property.HmiResetByCoproRun       = 0;
   _property.HmiResetByCoproShutdown  = 0;
   _property.FgsResetByHmi            = 0;

   poRstStat->vGetData( &tResetStatistics, sizeof( tResetStatistics ) );

   _property.HmiResetByHmiWdg         = 0;
   _property.HmiResetByHmiApp         = (tU8)tResetStatistics.tHmiReset.u32ApplicationError;
   _property.HmiResetByHmiException   = (tU8)( tResetStatistics.tHmiReset.u32Software + tResetStatistics.tHmiReset.u32Unknown + tResetStatistics.tHmiReset.u32User );
   _property.HmiResetByHmiExtern      = 0;
   _property.HmiResetByHmiDownload    = (tU8)( tResetStatistics.tHmiReset.u32Download + tResetStatistics.tHmiReset.u32DownloadError );
   _property.HmiResetByHmiDiag        = (tU8)tResetStatistics.tHmiReset.u32EcuReset;

   // vGetOperation 'ResetList': user specific code end
} // vGetOperation

tU8 spm_PropertyResetCounter::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgResetCounterSet oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'ResetCounter': user specific code start
      ISpmStatistics *poRstStat = dynamic_cast < ISpmStatistics* >( ISpmFactory::_poFactoryRef->getSpmObjHandler( "spm_tclResetStatistics" ) );
      SPM_NULL_POINTER_CHECK_VAL( poRstStat );
      poRstStat->vErase( );
      return( IPROPERTY_RETURN_SUCCESS );
      // u8SetOperation 'ResetCounter': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tU8 spm_PropertyTransportModeList::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgTransportModeListSet oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'TransportModeList': user specific code start
      ISpmStatistics *poTransport = dynamic_cast < ISpmStatistics* >( ISpmFactory::_poFactoryRef->getSpmObjHandler( "spm_tclTransportmodeStatistics" ) );
      SPM_NULL_POINTER_CHECK_VAL( poTransport )
      poTransport->vErase( );
      // u8SetOperation 'TransportModeList': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertyTransportModeList::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'TransportModeList': user specific code start
   _property.TransportModeList.TransPortModeList.clear( );

   dp_tclSpmDpEngDataTpHistory oHistory;
   tU32                        u32Count = oHistory.u32GetCount( );
   for ( tUInt i = 0; i < u32Count; ++i ){
      tU32                         u32Seconds = 0;
      tclTimeConvert               oTimeConv;
      OSAL_trTimeDate              tTimeDate = { 0,0,0,0,0,0,0,0,0 };

      oHistory.s32GetElem( u32Count - i - 1, u32Seconds );
      oTimeConv.vGetDateFromElapsedSeconds( u32Seconds, &tTimeDate );
      spm_fi_tcl_SPM_TransPortMode oTransPortElem;
      oTransPortElem.YEAR   = (tU32)tTimeDate.s32Year;
      oTransPortElem.MONTH  = (tU32)tTimeDate.s32Month;
      oTransPortElem.DAY    = (tU32)tTimeDate.s32Day;
      oTransPortElem.HOUR   = (tU32)tTimeDate.s32Hour;
      oTransPortElem.MINUTE = (tU32)tTimeDate.s32Minute;
      _property.TransportModeList.TransPortModeList.push_back( oTransPortElem );
   }
   // vGetOperation 'TransportModeList': user specific code end
} // vGetOperation

tU8 spm_PropertyResetHistory::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgResetHistorySet oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'ResetHistory': user specific code start
      ISpmStatistics *poRstStat = dynamic_cast < ISpmStatistics* >( ISpmFactory::_poFactoryRef->getSpmObjHandler( "spm_tclResetStatistics" ) );
      SPM_NULL_POINTER_CHECK_VAL( poRstStat );
      poRstStat->vErase( );
      return( IPROPERTY_RETURN_SUCCESS );
      // u8SetOperation 'ResetHistory': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertyResetHistory::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'ResetHistory': user specific code start
   _property.tResetList.clear( );

   dp_tclSpmDpEngDataResetHistory oHistory;
   tU32                           u32Count = oHistory.u32GetCount( );
   for ( tUInt i = 0; i < u32Count; ++i ){
      TSpmResetElem           tRstElem  = {
         0, 0, { 0 }, { 0 }
      };
      tclTimeConvert          oTimeConv;
      OSAL_trTimeDate         tTimeDate = { 0,0,0,0,0,0,0,0,0 };

      oHistory.s32GetElem( u32Count - i - 1, tRstElem );
      oTimeConv.vGetDateFromElapsedSeconds( tRstElem.u32Time, &tTimeDate );

      spm_fi_tcl_ResetElement oResetElem;
      oResetElem.tResetDate.s16Year                = (tS16)tTimeDate.s32Year;
      oResetElem.tResetDate.u8Month                = (tU8)tTimeDate.s32Month;
      oResetElem.tResetDate.u8Day                  = (tU8)tTimeDate.s32Day;
      oResetElem.tResetDate.u8Hour                 = (tU8)tTimeDate.s32Hour;
      oResetElem.tResetDate.u8Minute               = (tU8)tTimeDate.s32Minute;
      oResetElem.tResetDate.u8Second               = (tU8)tTimeDate.s32Second;

      oResetElem.tSwVersionInfo.tSwBuildVersion    = (tString)tRstElem.strBuildVersion;
      oResetElem.tSwVersionInfo.tSwCustomerVersion = (tString)tRstElem.strCustomerVersion;

      oResetElem.eReset.enType                     = (spm_fi_tcl_e8_LcmResetReason::tenType)tRstElem.u16ResetType;

      _property.tResetList.push_back( oResetElem );
   }
   // vGetOperation 'ResetHistory': user specific code end
} // vGetOperation

tU8 spm_PropertyWakeupHistory::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgWakeupHistorySet oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'WakeupHistory': user specific code start
      ISpmStatistics *poWup = dynamic_cast < ISpmStatistics* >( ISpmFactory::_poFactoryRef->getSpmObjHandler( "spm_tclWakeupStatistics" ) );
      SPM_NULL_POINTER_CHECK_VAL( poWup );
      poWup->vErase( );
      return( IPROPERTY_RETURN_SUCCESS );
      // u8SetOperation 'WakeupHistory': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertyWakeupHistory::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'WakeupHistory': user specific code start
   _property.tWakeupList.clear( );

   dp_tclSpmDpEngDataWupHistory oHistory;

   tU32                         u32Count = oHistory.u32GetCount( );
   for ( tUInt i = 0; i < u32Count; ++i ){
      TSpmWakeupStatisticsData tWupElem ={0,0};
      tclTimeConvert           oTimeConv;
      OSAL_trTimeDate          tTimeDate = { 0,0,0,0,0,0,0,0,0 };

      //oHistory.s32GetElem(i, tWupElem);
      oHistory.s32GetElem( u32Count - i - 1, tWupElem );

      oTimeConv.vGetDateFromElapsedSeconds( tWupElem.u32Time, &tTimeDate );

      spm_fi_tcl_WakeupElement oWakeupElem;
      oWakeupElem.tWakeupDate.s16Year  = (tS16)tTimeDate.s32Year;
      oWakeupElem.tWakeupDate.u8Month  = (tU8)tTimeDate.s32Month;
      oWakeupElem.tWakeupDate.u8Day    = (tU8)tTimeDate.s32Day;
      oWakeupElem.tWakeupDate.u8Hour   = (tU8)tTimeDate.s32Hour;
      oWakeupElem.tWakeupDate.u8Minute = (tU8)tTimeDate.s32Minute;
      oWakeupElem.tWakeupDate.u8Second = (tU8)tTimeDate.s32Second;

      oWakeupElem.eWakeupReason.enType = (spm_fi_tcl_SPM_e32_WAKEUP_REASON::tenType)tWupElem.u32WakeUpType;
      _property.tWakeupList.push_back( oWakeupElem );
   }
   // vGetOperation 'WakeupHistory': user specific code end
} // vGetOperation

tU8 spm_PropertyStateHistory::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgStateHistorySet oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'StateHistory': user specific code start
      ISpmSystemStateStatistics *poHistory = dynamic_cast < ISpmSystemStateStatistics* >( ISpmFactory::_poFactoryRef->getSpmObjHandler( "spm_tclSystemStateStatistics" ) );
      if ( poHistory ){
         poHistory->vErase( );
      }
      return( IPROPERTY_RETURN_SUCCESS );
      // u8SetOperation 'StateHistory': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertyStateHistory::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'StateHistory': user specific code start
   _property.tStateList.clear( );

   dp_tclSpmDpEngStateHistory oHistory;

   tU32                       u32Count = oHistory.u32GetCount( );
   for ( tUInt i = 0; i < u32Count; ++i ){
      TSpmSystemStateStatisticsData tElem;
      tclTimeConvert                oTimeConv;
      OSAL_trTimeDate               tTimeDate = { 0,0,0,0,0,0,0,0,0 };

      oHistory.s32GetElem( u32Count - i - 1, tElem );
      oTimeConv.vGetDateFromElapsedSeconds( tElem.u32Time, &tTimeDate );

      spm_fi_tcl_StateChangeElement oStateElement;

      oStateElement.tWakeupDate.s16Year  = (tS16)tTimeDate.s32Year;
      oStateElement.tWakeupDate.u8Month  = (tU8)tTimeDate.s32Month;
      oStateElement.tWakeupDate.u8Day    = (tU8)tTimeDate.s32Day;
      oStateElement.tWakeupDate.u8Hour   = (tU8)tTimeDate.s32Hour;
      oStateElement.tWakeupDate.u8Minute = (tU8)tTimeDate.s32Minute;
      oStateElement.tWakeupDate.u8Second = (tU8)tTimeDate.s32Second;

      oStateElement.tSysState.enType     = (spm_fi_tcl_SPM_e32_SYSTEM_STATES::tenType)tElem.u8SystemState;

      if ( tElem.u8EntryType == SPM_STATE_HISTORY_ENTRY_WAKEUP ){
         oStateElement.tElemType.enType = spm_fi_tcl_e8_StateEntryType::FI_EN_SPM_E8_HISTORYENTRYWAKEUP;
         oStateElement.tWakeup.enType   = (spm_fi_tcl_SPM_e32_WAKEUP_REASON::tenType)tElem.u.u8Wakeup;
      } else if ( tElem.u8EntryType == SPM_STATE_HISTORY_ENTRY_SUBSTATE ){
         oStateElement.tElemType.enType                   = spm_fi_tcl_e8_StateEntryType::FI_EN_SPM_E8_HISTORYENTRYSUBSTATE;
         oStateElement.tSubStates.tSubStateMask0.u32Value = tElem.u.tTrigger.u32TriggerUpdate[0];
         oStateElement.tSubStates.tSubStateMask1.u32Value = tElem.u.tTrigger.u32TriggerUpdate[1];
         oStateElement.tSubStates.tSubStateMask2.u32Value = tElem.u.tTrigger.u32TriggerUpdate[2];
         oStateElement.tSubStates.tSubStateData0.u32Value = tElem.u.tTrigger.u32Trigger[0];
         oStateElement.tSubStates.tSubStateData1.u32Value = tElem.u.tTrigger.u32Trigger[1];
         oStateElement.tSubStates.tSubStateData2.u32Value = tElem.u.tTrigger.u32Trigger[2];

      } else if ( tElem.u8EntryType == SPM_STATE_HISTORY_ENTRY_RESET ){
         oStateElement.tElemType.enType = spm_fi_tcl_e8_StateEntryType::FI_EN_SPM_E8_HISTORYENTRYRESET;
         oStateElement.tReset.enType    = (spm_fi_tcl_e8_LcmResetReason::tenType)tElem.u.u8Reset;
      }
      _property.tStateList.push_back( oStateElement );
   }
   // vGetOperation 'StateHistory': user specific code end
} // vGetOperation

tU8 spm_PropertyErrStateHistory01::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgErrStateHistory01Set oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'ErrStateHistory01': user specific code start
           return(u8ErrStateHistorySetOperation());
      // u8SetOperation 'ErrStateHistory01': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertyErrStateHistory01::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'ErrStateHistory01': user specific code start
   _property.tStateList.clear( );

   dp_tclSpmDpEngStateHistoryStack01 oHistory;



      spm_fi_tcl_StateChangeElement oStateElement;
    for (tUInt i = 0; i < oHistory.u32GetCount(); ++i){
    vErrStateHistoryGetOperation(oHistory,oStateElement,i);


      _property.tStateList.push_back( oStateElement );
   }
   // vGetOperation 'ErrStateHistory01': user specific code end
} // vGetOperation

tU8 spm_PropertyErrStateHistory02::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgErrStateHistory02Set oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'ErrStateHistory02': user specific code start
        return(u8ErrStateHistorySetOperation());
      // u8SetOperation 'ErrStateHistory02': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertyErrStateHistory02::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'ErrStateHistory02': user specific code start
   _property.tStateList.clear( );

   dp_tclSpmDpEngStateHistoryStack02 oHistory;



      spm_fi_tcl_StateChangeElement oStateElement;
    for (tUInt i = 0; i < oHistory.u32GetCount(); ++i){
    vErrStateHistoryGetOperation(oHistory,oStateElement,i);


      _property.tStateList.push_back( oStateElement );
   }
   // vGetOperation 'ErrStateHistory02': user specific code end
} // vGetOperation

tU8 spm_PropertyErrStateHistory03::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgErrStateHistory03Set oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'ErrStateHistory03': user specific code start
        return(u8ErrStateHistorySetOperation());
      // u8SetOperation 'ErrStateHistory03': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertyErrStateHistory03::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'ErrStateHistory03': user specific code start
   _property.tStateList.clear( );

   dp_tclSpmDpEngStateHistoryStack03 oHistory;



      spm_fi_tcl_StateChangeElement oStateElement;
    for (tUInt i = 0; i < oHistory.u32GetCount(); ++i){
    vErrStateHistoryGetOperation(oHistory,oStateElement,i);


      _property.tStateList.push_back( oStateElement );
   }
   // vGetOperation 'ErrStateHistory03': user specific code end
} // vGetOperation

tU8 spm_PropertyErrStateHistory04::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgErrStateHistory04Set oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'ErrStateHistory04': user specific code start
      return(u8ErrStateHistorySetOperation());
      // u8SetOperation 'ErrStateHistory04': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertyErrStateHistory04::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'ErrStateHistory04': user specific code start
   _property.tStateList.clear( );

   dp_tclSpmDpEngStateHistoryStack04 oHistory;



      spm_fi_tcl_StateChangeElement oStateElement;
    for (tUInt i = 0; i < oHistory.u32GetCount(); ++i){
    vErrStateHistoryGetOperation(oHistory,oStateElement,i);


      _property.tStateList.push_back( oStateElement );
   }
   // vGetOperation 'ErrStateHistory04': user specific code end
} // vGetOperation

tU8 spm_PropertyErrStateHistory05::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgErrStateHistory05Set oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'ErrStateHistory05': user specific code start
       return(u8ErrStateHistorySetOperation());
      // u8SetOperation 'ErrStateHistory05': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertyErrStateHistory05::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'ErrStateHistory05': user specific code start
   _property.tStateList.clear( );

   dp_tclSpmDpEngStateHistoryStack05 oHistory;



      spm_fi_tcl_StateChangeElement oStateElement;
    for (tUInt i = 0; i < oHistory.u32GetCount(); ++i){
    vErrStateHistoryGetOperation(oHistory,oStateElement,i);


      _property.tStateList.push_back( oStateElement );
   }
   // vGetOperation 'ErrStateHistory05': user specific code end
} // vGetOperation

tU8 spm_PropertySwitchOnHistory::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgSwitchOnHistorySet oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'SwitchOnHistory': user specific code start
      ISpmStatistics *poStat = dynamic_cast < ISpmStatistics* >( ISpmFactory::_poFactoryRef->getSpmObjHandler( "spm_tclSwitchOnStatistics" ) );
      SPM_NULL_POINTER_CHECK_VAL( poStat );
      poStat->vErase( );
      return( IPROPERTY_RETURN_SUCCESS );
      // u8SetOperation 'SwitchOnHistory': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertySwitchOnHistory::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'SwitchOnHistory': user specific code start
   _property.tSwitchOnList.clear( );

   dp_tclSpmDpEngDataSwitchOnHistory oHistory;

   tU32                              u32Count = oHistory.u32GetCount( );
   for ( tUInt i = 0; i < u32Count; ++i ){
      TSpmSwitchOnStatisticsData tSwitchOnElem ={0,0};
      tclTimeConvert             oTimeConv;
      OSAL_trTimeDate            tTimeDate = { 0,0,0,0,0,0,0,0,0 };

      oHistory.s32GetElem( i, tSwitchOnElem );
      oTimeConv.vGetDateFromElapsedSeconds( tSwitchOnElem.u32Time, &tTimeDate );

      spm_fi_tcl_SwitchOnElement oSwitchOnElem;
      oSwitchOnElem.tSwitchOnDate.s16Year  = (tS16)tTimeDate.s32Year;
      oSwitchOnElem.tSwitchOnDate.u8Month  = (tU8)tTimeDate.s32Month;
      oSwitchOnElem.tSwitchOnDate.u8Day    = (tU8)tTimeDate.s32Day;
      oSwitchOnElem.tSwitchOnDate.u8Hour   = (tU8)tTimeDate.s32Hour;
      oSwitchOnElem.tSwitchOnDate.u8Minute = (tU8)tTimeDate.s32Minute;
      oSwitchOnElem.tSwitchOnDate.u8Second = (tU8)tTimeDate.s32Second;

      oSwitchOnElem.eSwitchOnReason.enType = (spm_fi_tcl_SPM_e32_SubStateType::tenType)tSwitchOnElem.u32WakeUpType;
      _property.tSwitchOnList.push_back( oSwitchOnElem );
   }
   // vGetOperation 'SwitchOnHistory': user specific code end
} // vGetOperation

tVoid spm_PropertyDeviceOnCounter::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'DeviceOnCounter': user specific code start
   dp_tclSpmDpEngDataDeviceWakeupCount oDevWakeUpCount;
   dp_tclSpmDpEngDataDeviceOnCount     oDevOnCount;

   _property.u32DeviceWakeupCount = oDevWakeUpCount.tGetData( );
   _property.u32DeviceUserOnCount = oDevOnCount.tGetData( );

   // vGetOperation 'DeviceOnCounter': user specific code end
} // vGetOperation

tVoid spm_PropertyVoltageEventCounter::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'VoltageEventCounter': user specific code start
   TSpmSubStatesStatisticsData    s;

   dp_tclSpmDpEngDataStatisticEng oStat;

   s = oStat.tGetData( );
   _property.tVoltageEventCounter.LowVoltageCount = s._u32LowVoltageCount;
   _property.tVoltageEventCounter.CrLowVoltageCount = s._u32CrLowVoltageCount;
   _property.tVoltageEventCounter.HighVoltageCount = s._u32HighVoltageCount;
   _property.tVoltageEventCounter.CrHighVoltageCount = s._u32CrHighVoltageCount;

   // vGetOperation 'VoltageEventCounter': user specific code end
} // vGetOperation

tVoid spm_PropertyVoltageRangeHistogram::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'VoltageRangeHistogram': user specific code start
   SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY(poclSpmPowerSupplyStatistics, ISpmPowerSupplyStatistics);
   TSpmVoltageHistogramRangeStatisticsData pVoltageRangeHistogramStatistics [SPM_U32_STATISTIC_VOLT_RANGE_HISTORAM_LENGTH + 1];

   poclSpmPowerSupplyStatistics->vGetData( pVoltageRangeHistogramStatistics, (tU32) sizeof( pVoltageRangeHistogramStatistics ) );
   _property.listVoltageRangeHistogram.clear( );

   for ( tUInt i = 0; i < SPM_U32_STATISTIC_VOLT_RANGE_HISTORAM_LENGTH + 1; ++i ) {
      spm_fi_tcl_SPM_VoltageRange oVoltageRange;
      oVoltageRange.VoltageValue = pVoltageRangeHistogramStatistics[i].u16VoltageValue;
      oVoltageRange.OperationTime = pVoltageRangeHistogramStatistics[i].u32OperationTime;
      _property.listVoltageRangeHistogram.push_back( oVoltageRange );
   }
   // vGetOperation 'VoltageRangeHistogram': user specific code end
} // vGetOperation

tVoid spm_PropertyCpuUsageStatistic::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'CpuUsageStatistic': user specific code start
   SPM_GET_CLASS_REFERENCE_NEW_VAR_STATIC_FACTORY(poclSpmCpuUsageStatistics, spm_tclCpuUsageStatistics, ISpmSystemLoadStatistics);
   TSpmSystemLoadStatisticsData           tCpuUsageStatisticsData;
   poclSpmCpuUsageStatistics->vGetData( &tCpuUsageStatisticsData, sizeof( tCpuUsageStatisticsData ) );
   _property.tCpuUsageStatisticsData.MaxValue = tCpuUsageStatisticsData.u32MaxValue;
   _property.tCpuUsageStatisticsData.AvgValue = tCpuUsageStatisticsData.u32AvgValue;
   _property.tCpuUsageStatisticsData.MinValue = tCpuUsageStatisticsData.u32MinValue;
   _property.tCpuUsageStatisticsData.StatisticDataDisabled = tCpuUsageStatisticsData.bStatisticDataDisabled;
   // vGetOperation 'CpuUsageStatistic': user specific code end
} // vGetOperation

tVoid spm_PropertyRamUsageStatistic::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'RamUsageStatistic': user specific code start
   SPM_GET_CLASS_REFERENCE_NEW_VAR_STATIC_FACTORY(poclSpmRamUsageStatistics, spm_tclRamUsageStatistics, ISpmSystemLoadStatistics);
   TSpmSystemLoadStatisticsData           tRamUsageStatisticsData;
   poclSpmRamUsageStatistics->vGetData( &tRamUsageStatisticsData, sizeof( tRamUsageStatisticsData ) );
   _property.tRamUsageStatisticsData.MaxValue = tRamUsageStatisticsData.u32MaxValue;
   _property.tRamUsageStatisticsData.AvgValue = tRamUsageStatisticsData.u32AvgValue;
   _property.tRamUsageStatisticsData.MinValue = tRamUsageStatisticsData.u32MinValue;
   _property.tRamUsageStatisticsData.StatisticDataDisabled = tRamUsageStatisticsData.bStatisticDataDisabled;
   // vGetOperation 'RamUsageStatistic': user specific code end
} // vGetOperation

tVoid spm_PropertyStartupTimeStatistic::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'StartupTimeStatistic': user specific code start
   SPM_GET_CLASS_REFERENCE_NEW_VAR_STATIC_FACTORY(poclSpmStartupTimeStatistics, spm_tclStartupTimeStatistics, ISpmSystemLoadStatistics);
   TSpmSystemLoadStatisticsData           tStartupTimeStatisticsData;
   poclSpmStartupTimeStatistics->vGetData( &tStartupTimeStatisticsData, sizeof( tStartupTimeStatisticsData ) );
   _property.tStartupTimeStatisticsData.MaxValue = tStartupTimeStatisticsData.u32MaxValue;
   _property.tStartupTimeStatisticsData.AvgValue = tStartupTimeStatisticsData.u32AvgValue;
   _property.tStartupTimeStatisticsData.MinValue = tStartupTimeStatisticsData.u32MinValue;
   _property.tStartupTimeStatisticsData.StatisticDataDisabled = tStartupTimeStatisticsData.bStatisticDataDisabled;
   // vGetOperation 'StartupTimeStatistic': user specific code end
} // vGetOperation

tVoid spm_PropertySWVersionStatistic::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'SWVersionStatistic': user specific code start
   SPM_GET_CLASS_REFERENCE_NEW_VAR_STATIC_FACTORY(poclSpmSoftwareVersionStatistics, spm_tclSoftwareVersionStatistics, ISpmSoftwareVersionStatistics);
   _property.listSWVersionStatistic.clear( );
   for ( tUInt i = 0; i < SPM_U32_STATISTIC_SW_VERSION_MAX_INDEX; ++i ){
      spm_fi_tcl_SPM_SWVersionStatisticData oSWVersionStatisticsData;
      TSpmSoftwareVersionStatisticsData      tSoftwareVersionStatisticsData;
      if ( poclSpmSoftwareVersionStatistics->vGetDataWithIndex( &tSoftwareVersionStatisticsData, i ) == TRUE ) {
         vSWVersionStatisticToFiType( &tSoftwareVersionStatisticsData, &oSWVersionStatisticsData );
         _property.listSWVersionStatistic.push_back( oSWVersionStatisticsData );
      } else {
         ETG_TRACE_USR1( ( "spm_PropertySWVersionStatistic::vGetOperation() Fail to get SWVersionStatisticsData with index %u ", i ) );
      }
   }
   // vGetOperation 'SWVersionStatistic': user specific code end
} // vGetOperation

const spm_PropertyRestart& spm_PropertyRestart::operator=(const spm_corefi_tclMsgRestartStatus& op){
   _property = op;
   return( * this );
}

tU8 spm_PropertyRestart::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgRestartSet oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'Restart': user specific code start
      SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclSystemPowerManager, ISpmSystemPowerManager);
      SPM_CCA_SERVICE_ELEMENT_IS_OBSOLETE("Property Restart --> Set");

      spm_fi_tcl_SPM_e32_RESTART_TYPE tRestart               = oResultData.Restart;
      ETG_TRACE_ERRMEM(("RESTART_TYPE: Restart type %u trigger received from app %u",
                           ETG_ENUM (SPM_RESTART_TYPE, tRestart.enType),
                           ETG_ENUM(ail_u16AppId, poSrvData->u16GetSourceAppID())));

      poclSystemPowerManager->vSetRestartType( (tU32)tRestart.enType );
      return( IPROPERTY_RETURN_SUCCESS );
      // u8SetOperation 'Restart': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertyRestart::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'Restart': user specific code start
   SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY(poclSystemPowerManager, ISpmSystemPowerManager);
   SPM_CCA_SERVICE_ELEMENT_IS_OBSOLETE("Property Restart --> Get");

   tU32                    u32RestartType         = poclSystemPowerManager->u32GetRestartType( );

   _property.Restart.enType = (spm_fi_tcl_SPM_e32_RESTART_TYPE::tenType)u32RestartType;

   // vGetOperation 'Restart': user specific code end
} // vGetOperation

const spm_PropertyOnOffReason& spm_PropertyOnOffReason::operator=(const spm_corefi_tclMsgOnOffReasonStatus& op){
   _property = op;
   return( * this );
}

tU8 spm_PropertyOnOffReason::u8SetOperation(amt_tclServiceData *poSrvData){
   tU8 u8Ret = u8SetOperationPrj(poSrvData);

   if (u8Ret == IPROPERTY_RETURN_UNKNOWN){
      //handle property only if project depending function returns error code IPROPERTY_RETURN_UNKNOWN
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgOnOffReasonSet oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'OnOffReason': user specific code start
      SPM_CCA_SERVICE_ELEMENT_IS_OBSOLETE_NEW_VAR_VAL("Property OnOffReason --> Set");
      return( IPROPERTY_RETURN_SUCCESS );

      // u8SetOperation 'OnOffReason': user specific code end
   }
   }
   return( u8Ret );
} // u8SetOperation

tVoid spm_PropertyOnOffReason::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'OnOffReason': user specific code start
   SPM_CCA_SERVICE_ELEMENT_IS_OBSOLETE_NEW_VAR("Property OnOffReason --> Get");

   // vGetOperation 'OnOffReason': user specific code end
} // vGetOperation

tU8 spm_PropertyWakeupList::u8SetOperation(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgWakeupListSet oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // u8SetOperation 'WakeupList': user specific code start

      // u8SetOperation 'WakeupList': user specific code end
   }
   return( IPROPERTY_RETURN_FAILED );
} // u8SetOperation

tVoid spm_PropertyWakeupList::vGetOperation(amt_tclServiceData *poSrvData){
   (tVoid)poSrvData;
   // vGetOperation 'WakeupList': user specific code start

   // vGetOperation 'WakeupList': user specific code end
} // vGetOperation

///////////////////////////////////////////////////////////////////////////////
// generic part of methods
///////////////////////////////////////////////////////////////////////////////

// implementation of interface IMethod
tU32 spm_MethodSystemMode::u32MethodStart(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgSystemModeMethodStart oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // vMethodStart 'SystemMode': user specific code start
      SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclCcaServiceHandler, ISpmCcaServiceServer);
      if ( poclCcaServiceHandler->bGetSysModeState( ) ){
         // currently state change is active --> deny request
         return( IMETHOD_RETURN_FAILED );
      } else {
         SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclSubStateHandler, ISpmSubStateClient);
         if ( oResultData.eSystemMode.enType == spm_fi_tcl_e8_StateModes::FI_EN_SPM_E8_SYSMODEDOWNLOAD ){
            SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclSystemStateManager, ISpmSystemStateManager);

            if ( poclSystemStateManager->u32GetNewSystemState( ) == SPM_PRJ_SYSTEM_STATE_DOWNLOAD ){
               // we are already in state DOWNLOAD --> send success message result
               return( IMETHOD_RETURN_SUCCESS );
            } else {

               poclCcaServiceHandler->vSetSysModeState( TRUE );
               poclSubStateHandler->vTriggerSubStateType( SPM_U32_DOWNLOAD, TRUE );

               return( IMETHOD_RETURN_DELAYED );
            }
         } else if ( oResultData.eSystemMode.enType == spm_fi_tcl_e8_StateModes::FI_EN_SPM_E8_SYSMODEDOWNLOADEXIT ){
            poclSubStateHandler->vTriggerSubStateType( SPM_U32_DOWNLOAD, FALSE );

            return( IMETHOD_RETURN_SUCCESS );
         } else if ( oResultData.eSystemMode.enType == spm_fi_tcl_e8_StateModes::FI_EN_SPM_E8_SYSMODECUSTOMERDOWNLOADSTART ){
            poclSubStateHandler->vTriggerSubStateType( SPM_U32_CUSTOMER_DOWNLOAD, TRUE );

            return( IMETHOD_RETURN_SUCCESS );
         } else if ( oResultData.eSystemMode.enType == spm_fi_tcl_e8_StateModes::FI_EN_SPM_E8_SYSMODECUSTOMERDOWNLOADEXIT ){
            poclSubStateHandler->vTriggerSubStateType( SPM_U32_CUSTOMER_DOWNLOAD, FALSE );

            return( IMETHOD_RETURN_SUCCESS );
         } else if ( oResultData.eSystemMode.enType == spm_fi_tcl_e8_StateModes::FI_EN_SPM_E8_SYSMODEDIAGACTIVITYSTART ){
            poclSubStateHandler->vTriggerSubStateType( SPM_U32_DIAG_ACTIVITY, TRUE );

            return( IMETHOD_RETURN_SUCCESS );
         } else if ( oResultData.eSystemMode.enType == spm_fi_tcl_e8_StateModes::FI_EN_SPM_E8_SYSMODEDIAGACTIVITYEXIT ){
            poclSubStateHandler->vTriggerSubStateType( SPM_U32_DIAG_ACTIVITY, FALSE );

            return( IMETHOD_RETURN_SUCCESS );
         } else if ( oResultData.eSystemMode.enType == spm_fi_tcl_e8_StateModes::FI_EN_SPM_E8_SYSMODEMAPDOWNLOADLOCK ){
            dp_tclSpmDpInternDataMapDlLockApp oDpMapDlLockApp;
            tU16                              u16LockApp     = oDpMapDlLockApp.tGetData( );
            tU16                              u16SourceAppID = poSrvData->u16GetSourceAppID( );

            if ( ( u16LockApp != AMT_C_U16_APPID_INVALID ) && ( u16LockApp != u16SourceAppID ) ){
               // LOCK already set by another application
               ETG_TRACE_FATAL( ( "spm_MethodSystemMode::u32MethodStart() MAPDOWNLOADLOCK: Request lock by %u. MapDlLock already locked by %u!!!", ETG_ENUM( ail_u16AppId, u16SourceAppID ), ETG_ENUM( ail_u16AppId, u16LockApp ) ) );
               return( IMETHOD_RETURN_FAILED );
            } else if ( u16LockApp == u16SourceAppID ){
               ETG_TRACE_FATAL( ( "spm_MethodSystemMode::u32MethodStart() MAPDOWNLOADLOCK: MapDlLock already locked by same application '%u'!!!", ETG_ENUM( ail_u16AppId, u16SourceAppID ) ) );
            } else {
               ETG_TRACE_FATAL( ( "spm_MethodSystemMode::u32MethodStart() MAPDOWNLOADLOCK: Set MapDlLock for %u!!!", ETG_ENUM( ail_u16AppId, u16SourceAppID ) ) );
               oDpMapDlLockApp << u16SourceAppID;

               spm_corefi_tclMsgMapDownloadLockStateStatus oMapDownloadLockState;
               oMapDownloadLockState.u16AppIdLockOwner = u16SourceAppID;
               poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_MAPDOWNLOADLOCKSTATE, &oMapDownloadLockState );
            }
            return( IMETHOD_RETURN_SUCCESS );
         } else if ( oResultData.eSystemMode.enType == spm_fi_tcl_e8_StateModes::FI_EN_SPM_E8_SYSMODEMAPDOWNLOADUNLOCK ){

            dp_tclSpmDpInternDataMapDlLockApp oDpMapDlLcokApp;
            tU16                              u16LockApp     = oDpMapDlLcokApp.tGetData( );
            tU16                              u16SourceAppID = poSrvData->u16GetSourceAppID( );

            if ( ( u16LockApp != AMT_C_U16_APPID_INVALID ) && ( u16LockApp != u16SourceAppID ) ){
               // LOCK already set by another application
               ETG_TRACE_FATAL( ( "spm_MethodSystemMode::u32MethodStart() MAPDOWNLOADUNLOCK: Request unlock by %u. MapDlLock was requested by %u!!!", ETG_ENUM( ail_u16AppId, u16SourceAppID ), ETG_ENUM( ail_u16AppId, u16LockApp ) ) );
               return( IMETHOD_RETURN_FAILED );
            } else {
               ETG_TRACE_FATAL( ( "spm_MethodSystemMode::u32MethodStart() MAPDOWNLOADUNLOCK: clear MapDlLock for %u!!!", ETG_ENUM( ail_u16AppId, u16SourceAppID ) ) );
               oDpMapDlLcokApp << AMT_C_U16_APPID_INVALID;

               spm_corefi_tclMsgMapDownloadLockStateStatus oMapDownloadLockState;
               oMapDownloadLockState.u16AppIdLockOwner = AMT_C_U16_APPID_INVALID;
               poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_MAPDOWNLOADLOCKSTATE, &oMapDownloadLockState );
            }

            return( IMETHOD_RETURN_SUCCESS );
         } else if ( oResultData.eSystemMode.enType == spm_fi_tcl_e8_StateModes::FI_EN_SPM_E8_SYSMODEMAPDOWNLOADMASTERUNLOCK ){

            dp_tclSpmDpInternDataMapDlLockApp oDpMapDlLcokApp;
            tU16                              u16SourceAppID = poSrvData->u16GetSourceAppID( );

            if ( CCA_C_U16_APP_FC_SW_UPDATE != u16SourceAppID ){
               // LOCK already set by another application
               ETG_TRACE_FATAL( ( "spm_MethodSystemMode::u32MethodStart() MAPDOWNLOADMASTERUNLOCK: Request unlock by %u. Only CCA_C_U16_APP_FC_SW_UPDATE is allowed to use master unlock!!!", ETG_ENUM( ail_u16AppId, u16SourceAppID ) ) );
               return( IMETHOD_RETURN_FAILED );
            } else {
               ETG_TRACE_FATAL( ( "spm_MethodSystemMode::u32MethodStart() MAPDOWNLOADMASTERUNLOCK: clear MasterLock MapDlLock for %u!!!", ETG_ENUM( ail_u16AppId, u16SourceAppID ) ) );
               oDpMapDlLcokApp << AMT_C_U16_APPID_INVALID;

               spm_corefi_tclMsgMapDownloadLockStateStatus oMapDownloadLockState;
               oMapDownloadLockState.u16AppIdLockOwner = AMT_C_U16_APPID_INVALID;
               poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_MAPDOWNLOADLOCKSTATE, &oMapDownloadLockState );
            }

            return( IMETHOD_RETURN_SUCCESS );
         } else {
            // currently only dl is supported
            return( IMETHOD_RETURN_FAILED );
         }
      }
      // vMethodStart 'SystemMode': user specific code end
   }
   return( IMETHOD_RETURN_FAILED );
} // u32MethodStart

// implementation of interface IMethod
tU32 spm_MethodDownloadLockState::u32MethodStart(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgDownloadLockStateMethodStart oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // vMethodStart 'DownloadLockState': user specific code start
       SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclCcaServiceHandler, ISpmCcaServiceServer);
       SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclSystemPowerManager, ISpmSystemPowerManager);
       SPM_CCA_SERVICE_ELEMENT_IS_OBSOLETE("Method DownloadLockState");

       tU16 u16SourceAppID = poSrvData->u16GetSourceAppID( );
       tBool bRet = poclCcaServiceHandler->bSetDlLockStateRequest(oResultData.strRestrictedDlState.szValue, u16SourceAppID, (tU32)oResultData.eRestrictedDlMode.enType);
       if (!bRet) {
           return( IMETHOD_RETURN_FAILED );
       }

       return( IMETHOD_RETURN_SUCCESS );

      // vMethodStart 'DownloadLockState': user specific code end
   }
   return( IMETHOD_RETURN_FAILED );
} // u32MethodStart

// implementation of interface IMethod
tU32 spm_MethodUpdateLock::u32MethodStart(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgUpdateLockMethodStart oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // vMethodStart 'UpdateLock': user specific code start
      SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL ( poclCcaServiceHandler, ISpmCcaServiceServer );
      tU16 u16SourceAppID = poSrvData->u16GetSourceAppID( );
      tBool bRet = poclCcaServiceHandler->bUpdateLock( (std::string)oResultData.strLockName.szValue, (tU32)oResultData.eLockRequest.enType, (tU16)u16SourceAppID );
      if (bRet) {
      return( IMETHOD_RETURN_SUCCESS );
      }
      // vMethodStart 'UpdateLock': user specific code end
   }
   return( IMETHOD_RETURN_FAILED );
} // u32MethodStart

// implementation of interface IMethod
tU32 spm_MethodOnOffTrigger::u32MethodStart(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgOnOffTriggerMethodStart oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // vMethodStart 'OnOffTrigger': user specific code start
      SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclSystemPowerManager, ISpmSystemPowerManager);

      //complete restart/reset handling is done in SystemPowerManager
      if ( spm_fi_tcl_SPM_e32_SubStateType::FI_EN_SPM_U32_SUBSTATE_WAIT_FOR_FSM_SHUTDOWN_ACK == (tU32)oResultData.eSwitchOnTrigger.enType ){
         SPM_CCA_SERVICE_ELEMENT_IS_OBSOLETE("Method OnOffTrigger -> enum FI_EN_SPM_U32_SUBSTATE_WAIT_FOR_FSM_SHUTDOWN_ACK");
	   }
      if ( spm_fi_tcl_SPM_e32_SubStateType::FI_EN_SPM_U32_SUBSTATE_WAIT_FOR_FSM_RESTART_ACK == (tU32)oResultData.eSwitchOnTrigger.enType){
         SPM_CCA_SERVICE_ELEMENT_IS_OBSOLETE("Method OnOffTrigger -> enum FI_EN_SPM_U32_SUBSTATE_WAIT_FOR_FSM_RESTART_ACK");
	   }

      SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclSubStateHandler, ISpmSubStateClient);
      poclSubStateHandler->vTriggerSubStateType( (tU32)oResultData.eSwitchOnTrigger.enType, oResultData.bState );

      return( IMETHOD_RETURN_SUCCESS );

      // vMethodStart 'OnOffTrigger': user specific code end
   }
   return( IMETHOD_RETURN_FAILED );
} // u32MethodStart

// implementation of interface IMethod
tU32 spm_MethodStartProcess::u32MethodStart(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgStartProcessMethodStart oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // vMethodStart 'StartProcess': user specific code start
      const tU16 au16ConfigApps[] = SPM_CONFIGURED_APP_2_STARTPROCESS;
      tU8 u8ArrayCount = ELEMENTE( au16ConfigApps );

      // Check whether Sender(SourceAppID) is in list of Configured Apps or not
      const tU16 *iter = std::find( au16ConfigApps,
                                     au16ConfigApps + u8ArrayCount,
                                     poSrvData->u16GetSourceAppID( ) );
      if( iter != ( au16ConfigApps + u8ArrayCount ) ){
         // Sender(SourceAppID) exists in Configured Apps, so it allowed to trigger StartProcess
         ETG_TRACE_USR4( ( "spm_MethodStartProcess::u32MethodStart: \
                            Received trigger to StartProcess from AppId: %u(%x).",
                            ETG_ENUM( ail_u16AppId, poSrvData->u16GetSourceAppID( ) ),
                            ETG_ENUM( ail_u16AppId, poSrvData->u16GetSourceAppID( ) ) ) );

         if ( oResultData.u32Type.enType == spm_fi_tcl_SPM_e32_START_PROC_TYPE::FI_EN_SPM_U32_START_PROC_TYPE_SYSTEMD ){
            ETG_TRACE_USR4( ( "spm_MethodStartProcess::u32MethodStart: StartProcess with StartProcType: SYSTEMD, Location: '%s'",
                               oResultData.strLocation.szValue ) );

            // Get reference to spm_tclSpmLateServiceHandler object
            SPM_GET_CLASS_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL( poDbus,
                                                                spm_tclSpmLateServiceHandler,
                                                                spm_ISpmLateServiceHandler );
            // forward StartProcess request to LcmLate via spm_tclSpmLateServiceHandler
            (tVoid)poDbus->s32StartProcess( static_cast< std::string >( oResultData.strLocation.szValue ),
                                            static_cast< tU32 >( oResultData.u32Type.enType ) );
                     // Get reference to spm_tclStartupCommon object
            SPM_GET_CLASS_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL( poclStartupCommon,
                                                                spm_tclStartupCommon,
                                                                ISpmStartupCommon );
            poclStartupCommon->vUpdateStartedService( static_cast< std::string >( oResultData.strLocation.szValue ) );
            return( IMETHOD_RETURN_SUCCESS );

         } else {
            // StartProcType is not SYSTEMD
            ETG_TRACE_ERRMEM( ( "START_PROCESS: spm_MethodStartProcess::u32MethodStart: \
                                Ignore trigger to StartProcess with StartProcType is not SYSTEMD, Type: %d, Location: '%s'",
                                oResultData.u32Type.enType,
                                oResultData.strLocation.szValue ) );
         }
      } else {
         // Sender(SourceAppID) is not allowed to trigger StartProcess
         ETG_TRACE_ERRMEM( ( "START_PROCESS: spm_MethodStartProcess::u32MethodStart:\
                              Sender(SourceAppID) is not allowed to trigger StartProcess: %u(%x)",
                              ETG_ENUM( ail_u16AppId, poSrvData->u16GetSourceAppID( ) ),
                              ETG_ENUM( ail_u16AppId, poSrvData->u16GetSourceAppID( ) ) ) );
      }
      // vMethodStart 'StartProcess': user specific code end
   }
   return( IMETHOD_RETURN_FAILED );
} // u32MethodStart

// implementation of interface IMethod
tU32 spm_MethodStopProcess::u32MethodStart(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgStopProcessMethodStart oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // vMethodStart 'StopProcess': user specific code start
      const tU16 au16ConfigApps[] = SPM_CONFIGURED_APP_2_STOPPROCESS;
      tU8 u8ArrayCount = ELEMENTE( au16ConfigApps );

      // Check whether Sender(SourceAppID) is in list of Configured Apps or not
      const tU16 *iter = std::find( au16ConfigApps,
                                    au16ConfigApps + u8ArrayCount,
                                    poSrvData->u16GetSourceAppID( ) );
      if( iter != ( au16ConfigApps + u8ArrayCount ) ){
         // Sender(SourceAppID) exists in Configured Apps, so it allowed to trigger StopProcess
         ETG_TRACE_USR4( ( "spm_MethodStopProcess::u32MethodStart: Received trigger to StopProcess from AppId: %u(%x), ProcId: %d, Location: '%s'",
                            ETG_ENUM( ail_u16AppId, poSrvData->u16GetSourceAppID( ) ),
                            ETG_ENUM( ail_u16AppId, poSrvData->u16GetSourceAppID( ) ),
                            oResultData.u32ProcId,
                            oResultData.strLocation.szValue ) );

         // Get reference to spm_tclSpmLateServiceHandler object
         SPM_GET_CLASS_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL( poDbus,
                                                             spm_tclSpmLateServiceHandler,
                                                             spm_ISpmLateServiceHandler );
         // forward StopProcess request to LcmLate via spm_tclSpmLateServiceHandler
         (tVoid)poDbus->bStopProcess( static_cast< std::string >( oResultData.strLocation.szValue ),
                                      oResultData.u32ProcId );
         // Get reference to spm_tclStartupCommon object
         SPM_GET_CLASS_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL( poclStartupCommon,
                                                             spm_tclStartupCommon,
                                                             ISpmStartupCommon );
         poclStartupCommon->vRemoveStoppedService( static_cast< std::string >( oResultData.strLocation.szValue ) );
         return( IMETHOD_RETURN_SUCCESS );
      } else {
         // Sender(SourceAppID) is not allowed to trigger StopProcess
         ETG_TRACE_ERRMEM( ( "STOP_PROCESS: spm_MethodStopProcess::u32MethodStart:\
                              Sender(SourceAppID) is not allowed to trigger StopProcess: %u(%x)",
                              ETG_ENUM( ail_u16AppId, poSrvData->u16GetSourceAppID( ) ),
                              ETG_ENUM( ail_u16AppId, poSrvData->u16GetSourceAppID( ) ) ) );
      }
      // vMethodStart 'StopProcess': user specific code end
   }
   return( IMETHOD_RETURN_FAILED );
} // u32MethodStart

// implementation of interface IMethod
tU32 spm_MethodRestartProcess::u32MethodStart(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgRestartProcessMethodStart oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // vMethodStart 'RestartProcess': user specific code start
        const tU16 au16ConfigApps[] = SPM_CONFIGURED_APP_2_RESTART;
        tU8 u8ArrayCount = ELEMENTE( au16ConfigApps );
        const tU16 *iter = std::find( au16ConfigApps,
                                    au16ConfigApps + u8ArrayCount,
                                    poSrvData->u16GetSourceAppID( ) );
        if( iter != ( au16ConfigApps + u8ArrayCount ) ){
            ETG_TRACE_USR4( ( "spm_MethodRestartProcess::RestartProcess: Received trigger to RestartProcess from AppId: %u(%x), ProcId: %d, Location: '%s'",
                             ETG_ENUM( ail_u16AppId, poSrvData->u16GetSourceAppID( ) ),
                             ETG_ENUM( ail_u16AppId, poSrvData->u16GetSourceAppID( ) ),
                             oResultData.u32ProcId,
                             oResultData.strLocation.szValue ) );
            // Get reference to spm_tclSpmLateServiceHandler object
            SPM_GET_CLASS_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL( poDbus,
                                                                 spm_tclSpmLateServiceHandler,
                                                                 spm_ISpmLateServiceHandler );
            // forward RestartProcess request to LcmLate via spm_tclSpmLateServiceHandler
            (tVoid)poDbus->bRestartProcess( static_cast< std::string >( oResultData.strLocation.szValue ),
                                                                oResultData.u32ProcId );
            return( IMETHOD_RETURN_SUCCESS );
        } else {
            // Sender(SourceAppID) is not allowed to trigger RestartProcess
            ETG_TRACE_ERRMEM( ( "RESTART_PROCESS: spm_MethodRestartProcess::RestartProcess:\
                              Sender(SourceAppID) is not allowed to trigger RestartProcess: %u(%x)",
                              ETG_ENUM( ail_u16AppId, poSrvData->u16GetSourceAppID( ) ),
                              ETG_ENUM( ail_u16AppId, poSrvData->u16GetSourceAppID( ) ) ) );
        }
      // vMethodStart 'RestartProcess': user specific code end
   }
   return( IMETHOD_RETURN_FAILED );
} // u32MethodStart

// implementation of interface IMethod
tU32 spm_MethodStartUpSyncPointReached::u32MethodStart(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgStartUpSyncPointReachedMethodStart oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // vMethodStart 'StartUpSyncPointReached': user specific code start
      SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclStartupSystem, ISpmStartupSystem);
      poclStartupSystem->vOnSyncPointReached( oResultData.strNewSyncPoint.szValue );
      return( IMETHOD_RETURN_SUCCESS );
      // vMethodStart 'StartUpSyncPointReached': user specific code end
   }
   return( IMETHOD_RETURN_FAILED );
} // u32MethodStart

// implementation of interface IMethod
tU32 spm_MethodClearList::u32MethodStart(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgClearListMethodStart oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // vMethodStart 'ClearList': user specific code start

      ISpmSystemStateStatistics *poStateHistory = dynamic_cast < ISpmSystemStateStatistics* >( ISpmFactory::_poFactoryRef->getSpmObjHandler( "spm_tclSystemStateStatistics" ) );
      ISpmStatistics            *poRstStat      = dynamic_cast < ISpmStatistics* >( ISpmFactory::_poFactoryRef->getSpmObjHandler( "spm_tclResetStatistics" ) );
      ISpmStatistics            *poWup          = dynamic_cast < ISpmStatistics* >( ISpmFactory::_poFactoryRef->getSpmObjHandler( "spm_tclWakeupStatistics" ) );
      ISpmStatistics            *poStat         = dynamic_cast < ISpmStatistics* >( ISpmFactory::_poFactoryRef->getSpmObjHandler( "spm_tclSwitchOnStatistics" ) );
      ISpmStatistics            *poTransStat    = dynamic_cast < ISpmStatistics* >( ISpmFactory::_poFactoryRef->getSpmObjHandler( "spm_tclTransportmodeStatistics" ) );
      ISpmStatistics            *poSubStateStat = dynamic_cast < ISpmStatistics* >( ISpmFactory::_poFactoryRef->getSpmObjHandler( "spm_tclSwitchOnStatistics" ) );

      SPM_NULL_POINTER_CHECK_VAL( poStateHistory );
      SPM_NULL_POINTER_CHECK_VAL( poRstStat );
      SPM_NULL_POINTER_CHECK_VAL( poWup );
      SPM_NULL_POINTER_CHECK_VAL( poStat );
      SPM_NULL_POINTER_CHECK_VAL( poSubStateStat );
      SPM_NULL_POINTER_CHECK_VAL( poTransStat );

      tBool bClearAll = FALSE;

      if ( oResultData.eList2Clear.enType == spm_fi_tcl_SPM_e32_CLEAR_LIST::FI_EN_SPM_U32_CLEAR_LIST_ALL ){
         bClearAll = TRUE;
      }
      if ( bClearAll || ( oResultData.eList2Clear.enType == spm_fi_tcl_SPM_e32_CLEAR_LIST::FI_EN_SPM_U32_CLEAR_LIST_ERROR_HISTORY ) ){
         poStateHistory->vEraseErrorHistory( );
      }
      if ( bClearAll || ( oResultData.eList2Clear.enType == spm_fi_tcl_SPM_e32_CLEAR_LIST::FI_EN_SPM_U32_CLEAR_LIST_RESET_HISTORY ) ){
         poRstStat->vErase( );
      }
      if ( bClearAll || ( oResultData.eList2Clear.enType == spm_fi_tcl_SPM_e32_CLEAR_LIST::FI_EN_SPM_U32_CLEAR_LIST_WAKEUP_HISTORY ) ){
         poWup->vErase( );
      }
      if ( bClearAll || ( oResultData.eList2Clear.enType == spm_fi_tcl_SPM_e32_CLEAR_LIST::FI_EN_SPM_U32_CLEAR_LIST_STATE_HISTORY ) ){
         poStateHistory->vErase( );
      }
      if ( bClearAll || ( oResultData.eList2Clear.enType == spm_fi_tcl_SPM_e32_CLEAR_LIST::FI_EN_SPM_U32_CLEAR_LIST_TRANSPORT_HISTORY ) ){
         poTransStat->vErase( );
      }
      if ( bClearAll || ( oResultData.eList2Clear.enType == spm_fi_tcl_SPM_e32_CLEAR_LIST::FI_EN_SPM_U32_CLEAR_LIST_SWITCH_ON_HISTORY ) ){
         poStat->vErase( );
      }
      if ( bClearAll ){
         poSubStateStat->vErase( );
      }
      dp_tclSrvIf dp;
      dp.s32StoreNow( );
      return( IMETHOD_RETURN_SUCCESS );

      // vMethodStart 'ClearList': user specific code end
   }
   return( IMETHOD_RETURN_FAILED );
} // u32MethodStart

// implementation of interface IMethod
tU32 spm_MethodSpecificSWVersionStatistic::u32MethodStart(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgSpecificSWVersionStatisticMethodStart oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // vMethodStart 'SpecificSWVersionStatistic': user specific code start
      SPM_GET_CLASS_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclSpmSoftwareVersionStatistics, spm_tclSoftwareVersionStatistics, ISpmSoftwareVersionStatistics);
      TSpmSoftwareVersionStatisticsData      tSoftwareVersionStatisticsData;
      memset( &tSoftwareVersionStatisticsData, 0, sizeof( TSpmSoftwareVersionStatisticsData ) );
      if ( TRUE == poclSpmSoftwareVersionStatistics->vGetDataWithIndex( &tSoftwareVersionStatisticsData, oResultData.u32Index ) ){
         vSWVersionStatisticToFiType( &tSoftwareVersionStatisticsData, &( _methodresult.tSWVersionStatistic ) );
         return( IMETHOD_RETURN_SUCCESS );
      } else {
         ETG_TRACE_FATAL( ( "spm_MethodSpecificSWVersionStatistic::u32MethodStart fail to get data with index %u", oResultData.u32Index ) );
         return( IMETHOD_RETURN_FAILED );
      }
      // vMethodStart 'SpecificSWVersionStatistic': user specific code end
   }
   return( IMETHOD_RETURN_FAILED );
} // u32MethodStart

// implementation of interface IMethod
tU32 spm_MethodRestartSystem::u32MethodStart(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgRestartSystemMethodStart oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // vMethodStart 'RestartSystem': user specific code start
      SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclSystemPowerManager, ISpmSystemPowerManager);

      //complete restart/reset handling is done in SystemPowerManager
      if ( spm_fi_tcl_SPM_e32_RESTART_TYPE::FI_EN_SPM_U32_SYNC_WITHOUT_RESTART == (tU32)oResultData.eRestartType.enType ){
		  SPM_CCA_SERVICE_ELEMENT_IS_OBSOLETE("Method RestartSystem -> enum SPM_U32_SYNC_WITHOUT_RESTART");
	  }

      spm_fi_tcl_SPM_e32_RESTART_TYPE tRestart = oResultData.eRestartType;
      ETG_TRACE_ERRMEM(("RESTART_TYPE: Restart type %u trigger received from app %u",
                          ETG_ENUM (SPM_RESTART_TYPE, tRestart.enType),
                          ETG_ENUM(ail_u16AppId, poSrvData->u16GetSourceAppID())));

      poclSystemPowerManager->vSetRestartType( (tU32)oResultData.eRestartType.enType );

      return( IMETHOD_RETURN_SUCCESS );
      // vMethodStart 'RestartSystem': user specific code end
   }
   return( IMETHOD_RETURN_FAILED );
} // u32MethodStart

// implementation of interface IMethod
tU32 spm_MethodAudioSourceChange::u32MethodStart(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgAudioSourceChangeMethodStart oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // vMethodStart 'AudioSourceChange': user specific code start
      dp_tclSpmDpInternDataStartupConfig oCfgConfig;
      tU32 u32Config = SPM_STARTUP_CONFIGURATION_DEFAULT;

      SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclStartupSystem, ISpmStartupSystem);

      ETG_TRACE_USR1( ( "spm_MethodAudioSourceChange::u32MethodStart(): Receive Audio source change: %u.", ETG_ENUM( CFC_FI_SPM_AUDIO_SOURCE, oResultData.eAudioSrc.enType ) ) );
      if (
         ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_RADIO_FM1 )
         || ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_RADIO_FM2 )
         || ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_RADIO_AM )
         || ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_RADIO_DAB )
         ){
         // start configuration "default" --> optimize startup for FM and MAP
          u32Config = SPM_STARTUP_CONFIGURATION_DEFAULT;

         #ifdef SPM_STARTUP_CONFIGURATION_BLUETOOTH
      } else if ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_MEDIA_BTAUDIO ){
         // start configuration "BLUETOOTH" --> optimize startup for playing bt sources
          u32Config = SPM_STARTUP_CONFIGURATION_BLUETOOTH;
         #endif // ifdef SPM_STARTUP_CONFIGURATION_BLUETOOTH

         #ifdef SPM_STARTUP_CONFIGURATION_MEDIA_CD
      } else if (
         ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_MEDIA_CDDA )
         || ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_MEDIA_CDC )
         || ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_MEDIA_CDCOMP )
         ){
         // start configuration "BLUETOOTH" --> optimize startup for playing bt sources
          u32Config = SPM_STARTUP_CONFIGURATION_MEDIA_CD;
         #endif // ifdef SPM_STARTUP_CONFIGURATION_MEDIA_CD

         #ifdef SPM_STARTUP_CONFIGURATION_MEDIA
      } else if (
         ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_MEDIA_USB )
         || ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_MEDIA_IPOD )
         || ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_MEDIA_SD )
         #ifndef SPM_STARTUP_CONFIGURATION_BLUETOOTH
            || ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_MEDIA_BTAUDIO )
         #endif
         || ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_MEDIA_MDI )
         || ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_MEDIA_AUXPLATE )
         #ifndef SPM_STARTUP_CONFIGURATION_MEDIA_CD
            || ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_MEDIA_CDDA )
            || ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_MEDIA_CDC )
            || ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_MEDIA_CDCOMP )
         #endif
         || ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_MEDIA_PANODRA )
         || ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_MEDIA_DB_PLAYER )
         ){
         // start configuration "MEDIA" --> optimize startup for playing media sources
          u32Config = SPM_STARTUP_CONFIGURATION_MEDIA;
         #endif // ifdef SPM_STARTUP_CONFIGURATION_MEDIA
         #ifdef SPM_STARTUP_CONFIGURATION_XM
      } else if (
         oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_RADIO_XM
         // ||   oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_MEDIA_IPOD
         ){
         // start configuration "XM" --> optimize startup for XM
          u32Config = SPM_STARTUP_CONFIGURATION_XM;
         #endif
         #ifdef SPM_STARTUP_CONFIGURATION_SPI
      } else if (
         ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_SPI_BT_AUDIO )
         || ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_SPI_USB )
         || ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_SPI_VR_USB )
         || ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_SPI_VR_BT_AUDIO )
         || ( oResultData.eAudioSrc.enType == spm_fi_tcl_SPM_e32_AUDIO_SOURCE::FI_EN_SPM_U32_AUDIOSRC_SPI_TTS )
         ){
         // start configuration "SPI" --> optimize startup for SmartphoneIntegration
          u32Config = SPM_STARTUP_CONFIGURATION_SPI;
         #endif
      }

      oCfgConfig << poclStartupSystem->u32CalculateStartupRegistry(u32Config);
      return( IMETHOD_RETURN_SUCCESS );
      // vMethodStart 'AudioSourceChange': user specific code end
   }
   return( IMETHOD_RETURN_FAILED );
} // u32MethodStart






// implementation of interface IMethod
tU32 spm_MethodShutdownAndTerminateCcaApps::u32MethodStart(amt_tclServiceData *poSrvData){
   //Extract CCA Message..
   fi_tclVisitorMessage oMsg(poSrvData);

   //Create Object..
   spm_corefi_tclMsgShutdownAndTerminateCcaAppsMethodStart oResultData;

   if ( OSAL_OK == oMsg.s32GetData(oResultData) ){
      // vMethodStart 'ShutdownAndTerminateCcaApps': user specific code start
      SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclApplicationDatabaseHandler, ISpmApplicationDatabase);
      SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclApplicationConfigurationHandler, ISpmApplicationConfiguration);
      SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poclSubStateHandler, ISpmSubStateClient);
      spm_tclSoftwareBlock         *poSoftwareBlock                     = NULL;
      for ( std::vector < tU16 >::const_iterator it = oResultData.tAppIdList.tAppIdList.begin( ); it != oResultData.tAppIdList.tAppIdList.end( ); ++it ){
         poclApplicationConfigurationHandler->bSetNewAppStateTable(
            ( tU32 ) * it,
            spm_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_ON,
            AMT_C_U32_STATE_OFF,
            TRUE );
         poSoftwareBlock = poclApplicationDatabaseHandler->poGetSwBlockByAppID( ( tU32 ) * it );
         poSoftwareBlock->vSetKillBlockEnabled( );
      }
      poclSubStateHandler->vTriggerSubStateType( spm_fi_tcl_SPM_e32_SubStateType::FI_EN_SPM_U32_SUBSTATE_IGNITION, TRUE );
      for ( std::vector < tU16 >::const_iterator it = oResultData.tAppIdList.tAppIdList.begin( ); it != oResultData.tAppIdList.tAppIdList.end( ); ++it ){
         poclApplicationConfigurationHandler->bSetNewAppStateTable(
            ( tU32 ) * it,
            spm_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_ON,
            AMT_C_U32_STATE_NORMAL,
            TRUE );
      }
      poclSubStateHandler->vTriggerSubStateType( spm_fi_tcl_SPM_e32_SubStateType::FI_EN_SPM_U32_SUBSTATE_IGNITION, TRUE );
      return( IMETHOD_RETURN_SUCCESS );
      // vMethodStart 'ShutdownAndTerminateCcaApps': user specific code end
   }
   return( IMETHOD_RETURN_FAILED );
} // u32MethodStart










