/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_CcaMethods.h
  * @brief        This is header file contains interfaces to work with CcaMethods.
  * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#include "spm_CcaProperties.h"
#include "spm_ICcaServer.h"

// /////////////////////////////////////////////////////////////////////////////
// interface for abstract methods

/*!
  *  \class IMethod
  *  \brief This abstract class to work with CCA methods. These methods return enum
          IMETHOD_RETURN_SUCCESS,  IMETHOD_RETURN_FAILED and IMETHOD_RETURN_DELAYED values in response to method start.
  */
class IMethod
{
public:
virtual tVoid vSetMethodResult( const fi_tclTypeBase *pMsg ) = 0;

virtual ~IMethod( ){}

// The results that a method start call can return:
// SUCCESS means that the method result is available and can be sent
// FAILED  means that the method call failed, send an error code
// DELAYED means that the method result(s) are sent later, not by
// this framework.
enum
{
   IMETHOD_RETURN_SUCCESS,
   IMETHOD_RETURN_FAILED,
   IMETHOD_RETURN_DELAYED
};

virtual tU32 u32MethodStart( amt_tclServiceData *poSrvData ) = 0;

virtual const fi_tclTypeBase*poGetResult( ) const            = 0;

};

/*!
  *  \class spm_Method
  *  \brief This template class to work with CCA methods. It provides interfaces to set and get  method result IMETHOD_RETURN_SUCCESS,  IMETHOD_RETURN_FAILED and
  *         IMETHOD_RETURN_DELAYED.


  */
template < typename T >
class spm_Method : public IMethod
{
protected:
T _methodresult;

public:
spm_Method < T >( ){
}
spm_Method < T >( const spm_Property < T > &op ){
   _methodresult = op;
}
const spm_Method < T >& operator=( const spm_Method < T >& op ){
   _methodresult = op;
   return( * this );
}

const spm_Method < T >& operator=( const T& op ){
   _methodresult = op;
   return( * this );
}

virtual ~spm_Method( ){
}

// Implementation of interface IProperty
virtual tVoid vSetMethodResult( const fi_tclTypeBase *pMsg ){
   const fi_tclTypeBase      & val = * pMsg; // ->corfoGetTypeBase();

   _methodresult = ( const T & )val;
}

// Implementation of interface IMethod
virtual const fi_tclTypeBase*poGetResult( ) const {
   return( (const fi_tclTypeBase*)&_methodresult );
}

};


// values to define the functionality of a method
#define SPM_METHOD_ADMIN_METHODRESULT       0x00000001

// convenience macros
#define SPM_PROP_ADMIN_GET_SET  ( SPM_PROP_ADMIN_GET | SPM_PROP_ADMIN_SET )
#define SPM_PROP_ADMIN_GET_UPREG ( SPM_PROP_ADMIN_GET | SPM_PROP_ADMIN_UPREG )
#define SPM_PROP_ADMIN_GET_SET_UPREG ( SPM_PROP_ADMIN_GET | SPM_PROP_ADMIN_SET | SPM_PROP_ADMIN_UPREG )

struct spm_AdminMethod
{
   IMethod *_pMethod;
   tU32 _u32Configuration;
   tCString _sName;
};

// derived from ahl_tNotification but stripped from unnecessary information
struct spm_tNotification
{
   tU16 u16AppID;
   tU16 u16CmdCounter;
   tU16 u16RegisterID;
   tU16 u16SubID;
   tU8 u8Act;
};


typedef std::map < tU16, spm_AdminMethod* > TMapMethods;
typedef std::map < tU16, spm_tNotification > TMapMethodNotification;

/*!
  *  \class spm_MethodHandling
  *  \brief This class to work with CCA methods. It provide interfaces to add and remove function ID from method MAP.

  */
class spm_MethodHandling
{
private:
TMapMethods            _methods;
TMapMethodNotification _notifications;
tU16                   _u16ServiceId;

public:
spm_MethodHandling( tU16 u16ServiceId );
virtual ~spm_MethodHandling( );

tVoid vAddMethod( tU16             u16Fid,
                  spm_AdminMethod *m ){
   TMapMethods::iterator it;

   it = _methods.find( u16Fid );
   if ( it == _methods.end( ) ){
      _methods[u16Fid] = m;
   }
}

tVoid vRemoveMethod( tU16 u16Fid ){

   _methods.erase( _methods.find( u16Fid ) );
}

const fi_tclTypeBase*poGetResult( tU16 u16Fid ) const {
   const spm_AdminMethod *m = poFindMethod( u16Fid );

   if ( m == NULL ){
      return( NULL );  // method not found
   }
   return( m->_pMethod->poGetResult( ) );
}

private:
const spm_AdminMethod*poFindMethod( tU16 u16Fid ) const {
   TMapMethods::const_iterator it;

   it = _methods.find( u16Fid );

   if ( it != _methods.end( ) ){
      return( it->second );
   }
   return( NULL );
}

const spm_AdminMethod*poFindMethod( const amt_tclServiceData *poSrvData ) const {
   TMapMethods::const_iterator it;

   it = _methods.find( poSrvData->u16GetFunctionID( ) );

   if ( it != _methods.end( ) ){
      return( it->second );
   }
   return( NULL );
}

public:
tVoid vUpdateMethodResult( ISpmCcaServer        *pServer,
                           tU16                  u16Fid,
                           const fi_tclTypeBase *pMsgBase,
                           tU8                   opCode = AMT_C_U8_CCAMSG_OPCODE_METHODRESULT );

tVoid vResultMessage( ISpmCcaServer           *pServer,
                      IMethod                 *m,
                      tU16                     u16Fid,
                      const spm_tNotification& notif,
                      tU8                      opCode = AMT_C_U8_CCAMSG_OPCODE_METHODRESULT );

tVoid vHandleMethod( ISpmCcaServer      *pServer,
                     amt_tclServiceData *poSrvData );

private:
tVoid vErrorMessage( ISpmCcaServer     *pServer,
                     tU16               u16Fid,
                     spm_tNotification *pNot,
                     tU16               u16ErrorData );

};

