/*!
  * \file spm_factory_handler.cpp
  *  \brief
  *   Creates the FC_SPM components, client handlers and performs the startup of processes needed in the project.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * 1.0 | 28.12.12  | CM-AI/VW32 kollai | initial version \n
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_if.h"
#include "dp_generic_if.h"
#include "dp_spm_if.h"

#include "spm_Config.h"

#include "spm_factory.h"

#include "spm_IStartupSystem.h"
#include "spm_StartupInvestigation.h"
#include "spm_LocalApplicationManagerConfig.h"
#include "spm_StartupCommon.h"

#include "spm_OsalProxyConfig.h"
#include "spm_WorkerServerConfig.h"
#include "spm_LamAppManagerConfig.h"
#include "spm_OsLinux.h"
#include "spm_OsalWupOnOffEvents.h"
#include "spm_OsalWupSupervisionErrorWarning.h"
#include "spm_OsalCvm.h"
#include "spm_SystemPowerManagerConfig.h"
#include "spm_SystemStateManagerConfig.h"
#include "spm_SuperVisionManager.h"
#include "spm_SupervisionEnableSupervisor.h"
#include "spm_SysStateSupervisor.h"
#include "spm_IgnitionOffSupervisor.h"
#include "spm_CpuLoadSupervisor.h"
#include "spm_RamLoadSupervisor.h"
#include "spm_StartupSupervisor.h"
#include "spm_ApplicationRequestSupervisor.h"
#include "spm_GlobalApplicationManager.h"
#include "spm_SubStateHandlerConfig.h"
#include "spm_CcaMsgHandler.h"
#include "spm_CcaServiceHandlerConfig.h"
#include "spm_CcaSupplierHandlerConfig.h"
#include "spm_ApplicationConfigurationConfig.h"
#include "spm_ApplicationErrorHandler.h"
#include "spm_CriticalVoltageManagerIntern.h"
#include "spm_ApplicationDatabase.h"
#include "spm_ResetStatistics.h"
#include "spm_SubStatesStatistics.h"
#include "spm_TransportmodeStatistics.h"
#include "spm_PowerOnCount.h"
#include "spm_SystemStateStatistics.h"
#include "spm_WakeupStatistics.h"
#include "spm_SwitchOnStatistics.h"
#include "spm_PowerSupplyStatistics.h"
#include "spm_SystemLoadStatistics.h"
#include "spm_SoftwareVersionStatistics.h"
#include "spm_VersionInfo.h"
#include "spm_WakeupHandler.h"
#include "spm_TimerHandler.h"
#include "spm_SyncHandler.h"
#include "spm_IRegistry.h"

#ifdef VARIANT_S_FTR_ENABLE_INC_SYSTEMSTATEMACHINE
#include "spm_SystemStateMachineServiceHandlerConfig.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_INC_THERMALMANAGEMENT
#include "spm_ThermalManagementServiceHandlerConfig.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_INC_SUPPLYMANAGEMENT
#include "spm_SupplyManagementServiceHandlerConfig.h"
#endif

#include "spm_ProcessDatabase.h"
#include "spm_ProcessSupervision.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_SSM
#include "trcGenProj/Header/spm_factory_handler.cpp.trc.h"
#endif

#include "spm_trace.h"

#define PROC_SUPERVISION_THREAD_PRIO                                  100
#define PROC_SUPERVISION_THREAD_STACKSIZE                             2048

tVoid spm_tclFactory::vCreateComponents( ){
   /*!
     * \fn
     *  \brief
     *   Creates FC_SPM components.
     ******
     */

   SPM_STARTUPINVEST_INIT_STARTUPITEM

      SPM_STARTUPINVEST_INIT_TIME

   {
      // first create trace to be able to add trace clients
      SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclTrace, CreateTrace, spm_tclTrace );
   }

   {
      // the bridge between FC SPM and startup processes which shall not know about
      // this component
      SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclStartupCommon, CreateStartupCommon, spm_tclStartupCommon );
   }

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclRegistry, CreateRegistry, spm_tclRegistry );

   {
      // import registry now --> needed for configurations
      _poclRegistry->bImportData( "prjversion.reg" );

      _poclStartupCommon->vReadBaseRegistry( );
      // _poclStartupCommon->vReadStartupConfiguration();

      SPM_STARTUPINVEST_FORCE_ENTRY( "FACTORY", "FC SPM: 'base.reg' is read." );
   }

   // the spm worker is used to distribute events back to the spm components
   // if a trigger come from another context (we have that if we use callbacks)
   // spm worker is running
   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poWorker, CreateWorkerServer, spm_tclWorkerServerConfig );

   // Set the number of start-up processes
   // If another number of startup processes is used then change this number.
   // The mechanism here is to determine the end of the startup phase where all
   // startup processes have finished. In that case the shutdown is allowed from
   // that moment on.
   _poWorker->vSetNumberOfStartupProcesses( SPM_STARTUP_CONFIGURATION_NUM_OF_SPMS );

   // create the database for all applications (softwareblock and connectedapps)
   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclDb, CreateApplicationDatabase, spm_tclApplicationDatabase );

   // functionalities are needed by the spm components. E.g. cvm initialization
   // is only done if the callback function is called.
   // The proxy is the first spm component to be started. It builds an abstraction
   // layer to all platform drivers needed by the spm components. It shall not
   // have any dependencies to any other spm components. Therefore the data from
   // datapool is directly set to proxy.
   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclOsalProxy, CreateOsalProxy, spm_tclOsalProxyConfig );


   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poWakeupHandler, CreateWakeupHandler, spm_tclWakeupHandler );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclSyncHdl, CreateSyncHandler, spm_tclSyncHandler );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poSPM, CreateSystemPowerManager, spm_tclSystemPowerManagerConfig );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclLocalAppManager, CreateLocalApplicationManager, spm_tclLocalApplicationManagerConfig );


   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclAppErrorHdl, CreateApplicationErrorHandler, spm_tclApplicationErrorHandler );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclApplicationRequestSupervisor, CreateApplicationRequestSupervisor, spm_tclApplicationRequestSupervisor );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poLamAppCfg, CreateLamAppManager, spm_tclLamAppManagerConfig );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclGlobalApplicationManager, CreateGlobalApplicationManager, spm_tclGlobalApplicationManager );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclCcaMsgHandler, CreateCcaMsgHandler, spm_tclCcaMsgHandler );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclDownloadLockListManager, CreateDownloadLockListManager, spm_tclDownloadLockListManager );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poSubStateHdl, CreateSubStateHandler, spm_tclSubStateHandlerConfig );

   #ifndef VARIANT_S_FTR_ENABLE_VARIANT_MULTIPLE_FSM
      SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poSystemStateManager, CreateSystemStateManager, spm_tclSystemStateManagerConfig );
   #endif

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclAppCfg, CreateApplicationConfiguration, spm_tclApplicationConfigurationConfig );
   _poclAppCfg->vReadStateConfigurations( );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclCcaSupplierHandler, CreateCcaSupplierHandler, spm_tclCcaSupplierHandlerConfig );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poOsLinuxProxy, CreateLinuxProxy, spm_tclOsLinux );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclOsalWupOnOffEvents, CreateOsalWupOnOffEvents, spm_tclOsalWupOnOffEvents );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclOsalWupSupervisionErrorWarning, CreateOsalWupSupervisionErrorWarning, spm_tclOsalWupSupervisionErrorWarning );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclOsalCvm, CreateOsalCvm, spm_tclOsalCvm );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclSupervisionManager, CreateSupervisionManager, spm_tclSupervisionManager );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclSysStateSupervisor, CreateSysStateSupervisor, spm_tclSysStateSupervisor );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclStartupSupervisor, CreateStartupSupervisor, spm_tclStartupSupervisor );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclIgnitionOffSupervisor, CreateIgnitionOffSupervisor, spm_tclIgnitionOffSupervisor );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclCpuLoadSupervisor, CreateCpuLoadSupervisor, spm_tclCpuLoadSupervisor );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclRamLoadSupervisor, CreateRamLoadSupervisor, spm_tclRamLoadSupervisor );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poSupervisionEnableSupervisor, CreateSupervisionEnableSupervisor, spm_tclSupervisionEnableSupervisor );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poProcessDatabase, CreateProcessDatabase, spm_tclProcessDatabase );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poProcessSupervision, CreateProcessSupervision, spm_tclProcessSupervision );

   // Do not add _actProcessSupervision(Process Supervision Thread) in the list of factory clients.

   /*_actProcessSupervision = new spm_tclActive(_poProcessSupervision);
   */

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclCriticalVoltageManagerIntern, CreateCriticalVoltageManagerIntern, spm_tclCriticalVoltageManagerIntern );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poServiceHdl, CreateCcaServiceHandler, spm_tclCcaServiceHandlerConfig );

   // create the startup process for T-Engine
   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poStartupLocal, CreateStartupLocal, spm_tclStartupLocal );
   _poclStartupCommon->vAddStartupSystem( _poStartupLocal );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poSWVersionStatistic, CreateSWVersionStatistics, spm_tclSoftwareVersionStatistics );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poSWVersionResetStatistic, CreateSWVersionResetStatistics, spm_tclSWVersionResetStatistics );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poResetStatistic, CreateResetStatistics, spm_tclResetStatistics );

   #ifdef VARIANT_S_FTR_ENABLE_VARIANT_HANDLING
      SPM_CREATE_NEW_SPM_CLASS_BASE( _variant, CreateStartupSystemVariant, spm_tclStartupSystemVariant );
      // group evaluation used in startup order
   #endif
   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poSWVersionSubStateStatistic, CreateSWVersionSubStateStatistics, spm_tclSWVersionSubStateStatistics );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poSWVersionPowerOnStatistic, CreateSWVersionPowerOnCount, spm_tclSWVersionPowerOnCount );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poSWVersionSystemStateStatistic, CreateSWVersionSystemStateStatistics, spm_tclSWVersionSystemStateStatistics );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poSWVersionWakeupStatistic, CreateSWVersionWakeupStatistics, spm_tclSWVersionWakeupStatistics );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poSWVersionSwitchOnStatistic, CreateSWVersionSwitchOnStatistics, spm_tclSWVersionSwitchOnStatistics );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poSWVersionCpuUsageStatistic, CreateSWVersionCpuUsageStatistics, spm_tclSWVersionCpuUsageStatistics );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poSWVersionRamUsageStatistic, CreateSWVersionRamUsageStatistics, spm_tclSWVersionRamUsageStatistics );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poSWVersionStartupTimeStatistic, CreateSWVersionStartupTimeStatistics, spm_tclSWVersionStartupTimeStatistics );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poSubStateStatistic, CreateSubStateStatistics, spm_tclSubStateStatistics );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poPowerOnStatistic, CreatePowerOnCount, spm_tclPowerOnCount );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poTransportModeStatistic, CreateTransportmodeStatistics, spm_tclTransportmodeStatistics );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poSystemStateStatistic, CreateSystemStateStatistics, spm_tclSystemStateStatistics );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclWakeupStatistic, CreateWakeupStatistics, spm_tclWakeupStatistics );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclSwitchOnStatistic, CreateSwitchOnStatistics, spm_tclSwitchOnStatistics );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poPowerSupplyStatistic,     CreatePowerSupplyStatistics,     spm_tclPowerSupplyStatistics );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poCpuUsageStatistic, CreateCpuUsageStatistics, spm_tclCpuUsageStatistics );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poRamUsageStatistic, CreateRamUsageStatistics, spm_tclRamUsageStatistics );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poStartupTimeStatistic, CreateStartupTimeStatistics, spm_tclStartupTimeStatistics );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclVersionInfo, CreateVersionInfo, spm_tclVersionInfo );

   SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poTimerHandler, CreateTimerHandler, spm_tclTimerHandler );


   #ifdef VARIANT_S_FTR_ENABLE_INC_THERMALMANAGEMENT
      // Thermal Management Client Handler.
      SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclThermalMgmtServiceHandler, CreateThermalMgmtServiceHandler, spm_tclThermalMgmtServiceHandler );
   #endif

   #ifdef VARIANT_S_FTR_ENABLE_INC_SUPPLYMANAGEMENT
      // Supply Management Client Handler.
      SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclSupplyMgmtServiceHandler, CreateSupplyMgmtServiceHandler, spm_tclSupplyMgmtServiceHandler );
   #endif

   #ifdef VARIANT_S_FTR_ENABLE_INC_SYSTEMSTATEMACHINE
      // System State Machine Client Handler.
      SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( _poclSystemStateMachineServiceHandler, CreateSystemStateMachineServiceHandler, spm_tclSystemStateMachineServiceHandler );
   #endif

   #ifdef VARIANT_S_FTR_ENABLE_PROCESS_CFG
      {
         ISpmWorkerClient *processShutdown;
         SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( processShutdown, CreateProcessShutdown, spm_tclProcessShutdown );
      } //lint !e429 Custodial pointer 'xxx' (line xx) has not been freed or returned --> pointer is added to list
   #endif
   {
      ISpmClientHandlerBase *poClient;
      SPM_CREATE_NEW_SPM_CLASS_CLIENT_HANDLER( poClient, CreateSpmLateServiceHandler, spm_tclSpmLateServiceHandler );
   } // lint !e429 Custodial pointer 'xxx' (line xx) has not been freed or returned --> pointer is added to list
   #ifdef VARIANT_S_FTR_ENABLE_USE_SPM_DIAGLOG
   {
      ISpmClientHandlerBase *poClient;
      SPM_CREATE_NEW_SPM_CLASS_CLIENT_HANDLER( poClient, CreateVdDiagLogCcaServiceHandler, spm_tclVdDiagLogServiceHandler );
   }
   #endif
}       // vCreateSpmComponents

tVoid spm_tclFactory::vStartComponents( ){
/*!
  * \fn
  *  \brief
  *   Starts threads of FC_SPM components.
  ******
  */
   SPM_NULL_POINTER_CHECK( _poclStartupInvest );
   SPM_NULL_POINTER_CHECK( _poclOsalProxy );
   SPM_NULL_POINTER_CHECK( _poWorker );

   // to make the startup investigation more precise get a delta between earliest
   // startup time and from t-engine startup time.
   // if the startup investigation is output then this difference is considered.
   // It augments the timer values by the difference.

   /* todo
      tU32 u32TmrDiff;
      _poclOsalProxy->bGetTimerDifference(&u32TmrDiff);
      _poclStartupInvest->vSetTimerOffset(u32TmrDiff);
     */

   SPM_NULL_POINTER_CHECK( _poclSupervisionManager );
   dp_tclSpmDpConfigWdtCcaHeartbeatTime         oWdtCcaHBeatTime;

   _poclSupervisionManager->vSetCCAHeartBeatInterval( oWdtCcaHBeatTime.tGetData( ) );

   _poclSupervisionManager->vSetPrewarnTime( 3 );     // 3sec earlier a warning comes up for heartbeat missing

   #ifdef VARIANT_S_FTR_ENABLE_USE_SPM_SLV
      // todo _poclSupervisionManager->vSetPreWarnTimeSlv(10); // 10sec earlier a warning comes up for heartbeat from slv missing
   #endif

   dp_tclSpmDpThreadThreadConfPrioWdtHw  oHwPrio;
   dp_tclSpmDpThreadThreadConfStackWdtHw oHWStacksize;

   _poclSupervisionManager->vStartThread( "SpmWdtHw", oHwPrio.tGetData( ), oHWStacksize.tGetData( ) );

   /* \todo: Use own values for PRIO&STACK size.
            Check if start of the process can be moved to the point when we start the shutdown
     */

   SPM_NULL_POINTER_CHECK( _poProcessSupervision );
   _poProcessSupervision->vStartThread( "SpmProcSup", PROC_SUPERVISION_THREAD_PRIO, PROC_SUPERVISION_THREAD_STACKSIZE );

   SPM_NULL_POINTER_CHECK( _poclGlobalApplicationManager );
   _poclGlobalApplicationManager->bAddNewLam( SPM_GAM_CCA_LOCAL_LAM, _poclLocalAppManager );

   SPM_NULL_POINTER_CHECK( _poclCcaMsgHandler );
   dp_tclSpmDpThreadThreadConfPrioCcaIn          oCcaInPrio;
   dp_tclSpmDpThreadThreadConfStackCcaIn         oCcaInStacksize;
   _poclCcaMsgHandler->vStart( "SpmCcaIn", oCcaInPrio.tGetData( ), oCcaInStacksize.tGetData( ), 120 );

   _poclAppErrorHdl->vStart( "SpmAppErr" );

} // vStartSpmComponents

tVoid spm_tclFactory::vPreStart( ){
/*!
  * \fn
  *  \brief Reads values from registry and evaluate process startup information.
  *
  *  This method is invoked before the startup of processes.
  ******
  */

   _poclRegistry->   bImportData(    "BuildVersion.reg" );

   const std::string strCustomerVersionRegFile = (std::string)SPM_REGISTRY_PERS_PATH_OSAL + "/customerversion.reg";
   _poclRegistry->   bImportData(    strCustomerVersionRegFile.c_str( ) );
   #ifdef VARIANT_S_FTR_ENABLE_VARIANT_HANDLING
      _poclRegistry->bImportData(    "varianthandling.reg" );
   #endif

   // get other values from registry, mark them accordingly
   if ( !_poclRegistry->bGetStringFromRegistry( "VERSIONS/BOARDCFG", "BOARD_TYP", _boardtype ) ){
      std::string strBuf;
      if ( 0 <= _poOsLinuxProxy->s32ReadFileContent( "/proc/device-tree/model", strBuf, 255 ) ){
         ETG_TRACE_USR1( ( "spm_tclFactory::vPreStart(): Boardtype:         %s'", strBuf.c_str( ) ) );
         _boardtype = strBuf.c_str( );

         dp_tclregVersionsBoardCfgName oBoardName;
         oBoardName.vSetData( &strBuf[0] );
      }
   } else {
      // nothing to do
   }

   if ( !_poclRegistry->bGetNumberFromRegistry( "VERSIONS/BOARDCFG", "SDRAM_SIZE", _sdram ) ){
      std::string strBuf;
      if ( 100 <= _poOsLinuxProxy->s32ReadFileContent( "/proc/meminfo", strBuf, 255 ) ){
         std::size_t pos = strBuf.find( " kB", strlen( "MemTotal:" ) );

         if ( pos != std::string::npos ){
            strBuf.insert( pos, 1, 0 );
            _sdram = (tU32)OSAL_s32AsciiToS32( &strBuf[strlen( "MemTotal:" )] ) / 1000;

            ETG_TRACE_USR1( ( "spm_tclFactory::vPreStart(): SD RAM size:         %d MB'", _sdram ) );

            dp_tclregVersionsBoardCfgRamSize oRamSize;
            oRamSize.vSetData( _sdram );
         }
      }
   } else {
      // nothing to do
   }
} // vPreStart

tVoid spm_tclFactory::vPostStart( ){
/*!
  * \fn
  *  \brief
  *
  *  This method is invoked after the startup of processes.
  ******
  */
   SPM_NULL_POINTER_CHECK( _poclStartupSupervisor );
   SPM_NULL_POINTER_CHECK( _poclCcaSupplierHandler );

   // check consecutive reset counter
   _poclStartupSupervisor->vStartupSpmProcessed( );

   _poclCcaSupplierHandler->vAddNewService( CCA_C_U16_SRV_SPM, CCA_C_U16_APP_SPM );
   _poclCcaSupplierHandler->vSupplierStateChanged( CCA_C_U16_APP_SPM, AMT_C_U8_CCAMSG_SUPPLIER_STATE_AVAILABLE );

   // maybe boardtype was patched away but provide a special variable for that
   if (!_poclRegistry->bGetStringFromRegistry( "VERSIONS/BOARDCFG", "BOARD_TYP", _boardtype2 )) {
      _boardtype2 = "not readable";
   }

} // vPostStart

tVoid spm_tclFactory::vSystemProcessStart( ){
/*!
  * \fn
  *  \brief Starts all implicit processes and starts thread of spm_tclStartupSystem* .
  ******
  */
   SPM_STARTUPINVEST_INIT_STARTUPITEM

      SPM_NULL_POINTER_CHECK( _poclStartupInvest );

   SPM_NULL_POINTER_CHECK( _poclStartupCommon );

   // read in further registries which are defined in base.reg
   _poclStartupCommon->vReadStartupConfiguration( );
   std::string strStartcfg = "STARTCFG";
   _poStartupLocal->bAlternativeProcConfig( &strStartcfg[0] );

   _poclStartupCommon->vReadRegistries( );
   _poclStartupCommon->vReadEarlyRegistries( );

   SPM_STARTUPINVEST_FORCE_ENTRY( "FACTORY", "FC SPM: Further registries read." );


   _poclStartupCommon->vEvaluateProcessStartup( );

   SPM_STARTUPINVEST_FORCE_ENTRY( "FACTORY", "FC SPM: start process evaluated." );


   SPM_NULL_POINTER_CHECK( _poStartupLocal );
   _poStartupLocal->vSystemStart( );
} // vSystemProcessStart

tVoid spm_tclFactory::vShowSystemInformation( ){
/*!
  * \fn
  *  \brief Shows system information.
  ******
  */
   vShowSystemInformationPrj( );

   ETG_TRACE_FATAL( ( "Board Type:          %s", _boardtype.c_str( ) ) );
   if ( _boardtype != _boardtype2 ){
      ETG_TRACE_FATAL( ( "Board Type 2:        %s", _boardtype2.c_str( ) ) );
   }
   ETG_TRACE_FATAL( ( "SDRAM Size:          %dkB", _sdram ) );
}

tVoid spm_tclFactory::vDpCallBackFunc( tU32 u32Event ){
   ETG_TRACE_FATAL( ( "vDpCallBackFunc(): Callback is called with parameter: %d", u32Event ) );
}

tVoid spm_tclFactory::vCreateCustomerVersionReg( const std::string& strCustomerVersionRegFile,
                                                 const std::string& strCustomerVersion ) const {
   const std::string strVersionReplace             = "__VERSION__";
   std::string       strCustomerVersionRegContent  = (std::string)"REGEDIT4\n\
; This file is the registry file for the customer version string\n\
[HKEY_LOCAL_MACHINE\\SOFTWARE\\BLAUPUNKT\\VERSIONS]\n\
\"CUSTOMERVERSION\"=\"" + strVersionReplace + "\"\n";

   std::string       strCustomerVersionRegOsalFile = (std::string)SPM_OSAL_FS_PREFIX + strCustomerVersionRegFile;

   strCustomerVersionRegContent.replace( strCustomerVersionRegContent.find( strVersionReplace ),
                                         strVersionReplace.length( ),
                                         strCustomerVersion );

   OSAL_tIODescriptor fd = OSAL_IOCreate( strCustomerVersionRegOsalFile.c_str( ), OSAL_EN_READWRITE );
   ETG_TRACE_USR1( ( "write new customer version %s", strCustomerVersion.c_str( ) ) );
   if ( fd != OSAL_ERROR ){
      std::string strTrace  = "file " + strCustomerVersionRegFile + " created.";
      ETG_TRACE_USR1( ( "%s", strTrace.c_str( ) ) );
      // write content to new customerversion.reg
      tS32        s32Result = OSAL_s32IOWrite( fd, (tPCS8)strCustomerVersionRegContent.c_str( ), (tU32)strCustomerVersionRegContent.length( ) );
      if ( (std::size_t)s32Result != strCustomerVersionRegContent.length( ) ){
         ETG_TRACE_USR1( ( "spm_tclFactory::vCreateCustomerVersionReg: Failed to write file '%s'.", strCustomerVersionRegFile.c_str( ) ) );
      } else {
         // all is OK
      }
      if ( OSAL_s32IOClose( fd ) != OSAL_OK ){
      }
   }
} // vCreateCustomerVersionReg

tVoid spm_tclFactory::vHandleCustomerVersionReg( ) const {
   std::string       strCustomerVersion           = "XXXX";
   const std::string strCustomerVersionRegFile    = (std::string)SPM_REGISTRY_PERS_PATH_LX + "/customerversion.reg";
   const std::string strCustomerVersionRegCfgFile = (std::string)SPM_REGISTRY_PERS_PATH_LX + "/customerversion_reg.cfg";

   if ( !_poOsLinuxProxy->bMkDir( SPM_REGISTRY_PERS_PATH_LX ) ){
      ETG_TRACE_FATAL( ( "spm_tclFactory::vHandleCustomerVersionReg: cannot create directory '%s'.", SPM_REGISTRY_PERS_PATH_LX ) );
   }

   // if customerversion.reg is not there create one with default values
   if ( !_poOsLinuxProxy->bIsFileInFs( strCustomerVersionRegFile ) ){
      // get the CustomerVerstion from prjversion.reg or take the default
      if ( _poclRegistry->bGetStringFromRegistry( "VERSIONS", "CUSTOMERVERSION_DEFAULT", strCustomerVersion ) ){
         ETG_TRACE_USR1( ( "spm_tclFactory::vHandleCustomerVersionReg: use default customer version '%s'.", strCustomerVersion.c_str( ) ) );
      }
      vCreateCustomerVersionReg( strCustomerVersionRegFile, strCustomerVersion );
   }

   // if SWU or DIAG have created a new customer version
   if ( _poOsLinuxProxy->bIsFileInFs( strCustomerVersionRegCfgFile ) ){
      std::string        strTrace                         = "found File" + strCustomerVersionRegCfgFile + ".";

      std::string        strCustomerVersionRegCfgOsalFile = (std::string)SPM_OSAL_FS_PREFIX + strCustomerVersionRegCfgFile;
      OSAL_tIODescriptor fd                               = OSAL_IOOpen( strCustomerVersionRegCfgOsalFile.c_str( ), OSAL_EN_READONLY );
      if ( fd != OSAL_ERROR ){
         tS32 s32Result = OSALUTIL_s32FGetSize( fd );
         if ( s32Result > 0 ){
            tChar *pu8FileBuffer = new tChar[(tU32)s32Result];
            if ( pu8FileBuffer ){
               // read the line containing the build information
               // <CustomerVersion>_<BuildCustomerVersion>_<BuildVersionLabel>
               tS32 s32ReadLen = OSAL_s32IORead( fd, (tPS8)pu8FileBuffer, (tU32)s32Result );
               if ( s32ReadLen != 0 ){
                  std::string CustomerVersionConfigRegLine( pu8FileBuffer );
                  std::string strBuildVersionLabel;
                  std::string strBuildVersionCustomerVersion;
                  std::string strBuildVersionSearchString;
                  // get the actuall build version
                  if ( ( _poclRegistry->bGetStringFromRegistry( "VERSIONS", "BUILDVERSION_LABEL", strBuildVersionLabel ) )
                       && ( _poclRegistry->bGetStringFromRegistry( "VERSIONS", "BUILDVERSION_CUSTVERSTRING", strBuildVersionCustomerVersion ) )
                       ){
                     strBuildVersionSearchString = "_" + strBuildVersionCustomerVersion + "_" + strBuildVersionLabel;
                     size_t StartPos = CustomerVersionConfigRegLine.find( strBuildVersionSearchString );
                     // tU32 Test1 = strBuildVersionSearchString.length();
                     // tU32 Test2 = CustomerVersionConfigRegLine.length();
                     if ( StartPos != std::string::npos ){
                        strCustomerVersion = CustomerVersionConfigRegLine.substr( 0, StartPos );
                     }
                     vCreateCustomerVersionReg( strCustomerVersionRegFile, strCustomerVersion );
                  }
               }
            }
            if ( pu8FileBuffer != NULL ){
               delete[] pu8FileBuffer;
            }
         }
         if ( OSAL_s32IOClose( fd ) != OSAL_OK ){
            ETG_TRACE_FATAL( ( "spm_tclFactory::vHandleCustomerVersionReg: Failed to close file '%s'.", strCustomerVersionRegCfgFile.c_str( ) ) );
         }
      }
      // delete file customerversion_reg.cfg
      if ( OSAL_s32IORemove( strCustomerVersionRegCfgOsalFile.c_str( ) ) != OSAL_OK ){
         ETG_TRACE_FATAL( ( "spm_tclFactory::vHandleCustomerVersionReg: Failed to delete file '%s'.", strCustomerVersionRegCfgFile.c_str( ) ) );
      }
   }

   {
      const std::string strCustomerVersionRegFile2 = (std::string)SPM_REGISTRY_PERS_PATH_OSAL + "/customerversion.reg";
      _poclRegistry->bImportData( strCustomerVersionRegFile2.c_str( ) );
   }
} // vHandleCustomerVersionReg

// EOF

