/*!
  * \file spm_factory.cpp
  *  \brief
  *   Creates the FC_SPM components, client handlers and performs the startup of processes needed in the project.
  *
  *  \note
  *  \b PROJECT: NextGen \n
  *  \b SW-COMPONENT: FC SPM \n
  *  \b COPYRIGHT:    (c) 2011-2017 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * 1.0 | 27.01.11  | TMS Fischer       | initial version \n
  * 1.1 | 03.04.12  | CM-AI/VW32 kollai | Adaptation for NISSAN LCN2 KAI
  ******
  */

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"

#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h"

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"
#include "dp_generic_if.h"

#include "spm_Config.h"

#include "spm_factory.h"

#include "spm_StartupInvestigation.h"
#include "spm_LocalApplicationManagerConfig.h"
#include "spm_StartupCommon.h"
#include "spm_StartupLocal.h"
#include "spm_IOsalProxy.h"
#include "spm_OsalWupOnOffEvents.h"
#include "spm_OsalWupSupervisionErrorWarning.h"
#include "spm_OsalCvm.h"
#include "spm_WorkerServerConfig.h"
#include "spm_LamAppManagerConfig.h"
#include "spm_ClientHandlerBase.h"
#include "spm_OsLinux.h"
#include "spm_SystemPowerManagerConfig.h"
#include "spm_SuperVisionManager.h"
#include "spm_SysStateSupervisor.h"
#include "spm_StartupSupervisor.h"
#include "spm_ApplicationRequestSupervisor.h"
#include "spm_GlobalApplicationManager.h"
#include "spm_SubStateHandlerConfig.h"
#include "spm_CcaMsgHandler.h"
#include "spm_CcaServiceHandlerConfig.h"
#include "spm_CcaSupplierHandlerConfig.h"
#include "spm_ApplicationConfigurationConfig.h"
#include "spm_CriticalVoltageManagerIntern.h"
#include "spm_ApplicationDatabase.h"
#include "spm_VersionInfo.h"
#include "spm_ProcessDatabase.h"
#include "spm_ProcessSupervision.h"
#include "spm_WakeupHandler.h"
#include "spm_TimerHandler.h"
#include "spm_ResetStatistics.h"
#include "spm_SubStatesStatistics.h"
#include "spm_SwitchOnStatistics.h"
#include "spm_SystemStateStatistics.h"
#include "spm_WakeupStatistics.h"
#include "spm_ApplicationErrorHandler.h"
#include "spm_SyncHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_INC_SYSTEMSTATEMACHINE
#include "spm_SystemStateMachineServiceHandlerConfig.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_INC_THERMALMANAGEMENT
#include "spm_ThermalManagementServiceHandlerConfig.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_INC_SUPPLYMANAGEMENT
#include "spm_SupplyManagementServiceHandlerConfig.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_SSM
#include "trcGenProj/Header/spm_factory.cpp.trc.h"
#endif

#include "spm_trace.h"


/******************************************************************************
 | local #define (scope: module-local)
 |*****************************************************************************/

ISpmFactory*ISpmFactory::_poFactoryRef = NULL;

spm_tclFactory::spm_tclFactory( )
/*!
  * \fn
  *  \brief
  *    Constructor
  ******
  */
   : ISpmFactory( * this )
   , _sdram(0)
   , _bSpecialProjectStartRegistry(FALSE)
   , _bUnconfigProjectStartRegistry(FALSE)
   , _bIsSWVersionUpdate(FALSE)
   , _poclStartupInvest(NULL)
   , _poclTrace(NULL)
   , _poclGlobalApplicationManager(NULL)
   , _poclCcaMsgHandler(NULL)
   , _poclDownloadLockListManager(NULL)
   , _poOsLinuxProxy(NULL)
   , _poWakeupHandler(NULL)
   , _poTimerHandler(NULL)
   , _poSPM(NULL)
   , _poWorker(NULL)
   , _poSubStateHdl(NULL)
   , _poServiceHdl(NULL)
   , _poLamAppCfg(NULL)
   , _poclLocalAppManager(NULL)
   , _poclAppErrorHdl(NULL)
   , _poclSupervisionManager(NULL)
   , _poclIgnitionOffSupervisor(NULL)
   , _poclSysStateSupervisor(NULL)
   , _poclCpuLoadSupervisor(NULL)
   , _poclRamLoadSupervisor(NULL)
   , _poclStartupSupervisor(NULL)
   , _poclOsalProxy(NULL)
   , _poclOsalWupOnOffEvents(NULL)
   , _poclOsalWupSupervisionErrorWarning(NULL)
   , _poclOsalCvm(NULL)
   , _poclDb(NULL)
   , _poclSyncHdl(NULL)
   , _poclWakeupStatistic(NULL)
   , _poclSwitchOnStatistic(NULL)
   , _poResetStatistic(NULL)
   , _poSubStateStatistic(NULL)
   , _poPowerOnStatistic(NULL)
   , _poTransportModeStatistic(NULL)
   , _poSystemStateStatistic(NULL)
   , _poPowerSupplyStatistic(NULL)
   , _poCpuUsageStatistic(NULL)
   , _poRamUsageStatistic(NULL)
   , _poStartupTimeStatistic(NULL)
   , _poSWVersionWakeupStatistic(NULL)
   , _poSWVersionSwitchOnStatistic(NULL)
   , _poSWVersionResetStatistic(NULL)
   , _poSWVersionSubStateStatistic(NULL)
   , _poSWVersionPowerOnStatistic(NULL)
   , _poSWVersionSystemStateStatistic(NULL)
   , _poSWVersionCpuUsageStatistic(NULL)
   , _poSWVersionRamUsageStatistic(NULL)
   , _poSWVersionStartupTimeStatistic(NULL)
   , _poSWVersionStatistic(NULL)
   , _poStartupLocal(NULL)
   , _poclAppCfg(NULL)
   , _poclCcaSupplierHandler(NULL)
   , _poclApplicationRequestSupervisor(NULL)
   , _poclStartupCommon(NULL)
   , _poclVersionInfo(NULL)
   , _poclCriticalVoltageManagerIntern(NULL)
   , _poclRegistry(NULL)
#ifdef VARIANT_S_FTR_ENABLE_VARIANT_HANDLING
   , _variant(NULL)
#endif
   , _poProcessDatabase(NULL)
   , _poProcessSupervision(NULL)
   , _poSupervisionEnableSupervisor(NULL)
#ifdef VARIANT_S_FTR_ENABLE_INC_THERMALMANAGEMENT
   , _poclThermalMgmtServiceHandler(NULL)
#endif
#ifdef VARIANT_S_FTR_ENABLE_INC_SUPPLYMANAGEMENT
   , _poclSupplyMgmtServiceHandler(NULL)
#endif
#ifdef VARIANT_S_FTR_ENABLE_INC_SYSTEMSTATEMACHINE
   , _poclSystemStateMachineServiceHandler(NULL)
#endif
   , _poSystemStateManager(NULL)
   {
   _poFactoryRef = this;

   SpmComponentPointerList.clear();
   SpmClientHdlList.clear();
}

tVoid spm_tclFactory::vStart( ){
/*!
  * \fn
  *  \brief
  *    Starts FC_SPM components, client handlers and starts all implicit processes
  ******
  */
   SPM_STARTUPINVEST_INIT_STARTUPITEM

        SPM_NULL_POINTER_CHECK( _poclStartupCommon );

   SPM_NULL_POINTER_CHECK( _poclOsalProxy );

   tU32 u32StartTime = OSAL_ClockGetElapsedTime( );
   vPreStart( );
   vPrePrjStart( );
   vPreStartEvaluateProcessStartup( );

   SPM_STARTUPINVEST_FORCE_ENTRY( "FACTORY", "FC SPM: PreStart called." );

   ETG_TRACE_USR1( ( "spm_tclFactory::vStart(): Duration 'PreStart' -> %d ms.", OSAL_ClockGetElapsedTime( ) - u32StartTime ) );


   u32StartTime = OSAL_ClockGetElapsedTime( );

   SPM_STARTUPINVEST_INIT_TIME
      SPM_STARTUPINVEST_SET_STARTTIME;
   vAddAllClientHandler( );
   SPM_STARTUPINVEST_CHECK_TIME_MEASURE( "FC SPM: clients added." );
   SPM_STARTUPINVEST_FORCE_ENTRY( "FACTORY", "FC SPM: clients added." );

   // call StartCommunication for all clients
   vStartAllCommunication( );
   SPM_STARTUPINVEST_FORCE_ENTRY( "FACTORY", "FC SPM: vStartAllCommunication called." );

   // start the spm components
   vStartComponents( );
   SPM_STARTUPINVEST_FORCE_ENTRY( "FACTORY", "FC SPM: vStartComponents called." );

   vStartPrjComponents( );
   SPM_STARTUPINVEST_FORCE_ENTRY( "FACTORY", "FC SPM: vStartPrjComponents called." );

   ETG_TRACE_USR1( ( "spm_tclFactory::vStart(): Duration 'StartComponents' -> %d ms.", OSAL_ClockGetElapsedTime( ) - u32StartTime ) );

   u32StartTime = OSAL_ClockGetElapsedTime( );
   vPostStart( );
   vPostPrjStart( );
   ETG_TRACE_USR1( ( "spm_tclFactory::vStart(): Duration 'PostStart' -> %d ms.", OSAL_ClockGetElapsedTime( ) - u32StartTime ) );

   // add startup investigation information
   SPM_STARTUPINVEST_FORCE_ENTRY( "FACTORY", "FC SPM: Initialization finished." );


   if ( _poclStartupCommon ){
      _poclStartupCommon->vOnSyncPointReached( SPM_STR_SYNC_POINT_LCMEARLY_STARTED );
   }

   vSystemProcessStart( );

   #ifdef SPM_ENABLE_ADDITIONAL_DEBUG_INFO // FOR DEBUG PURPOSE
      std::list < ISpmBase* >::const_iterator iter;
      int                                     i = 0;
      for ( iter = SpmComponentPointerList.begin( ); iter != SpmComponentPointerList.end( ); iter++ ){
         ETG_TRACE_FATAL( ( "%d: 0x%p %s", i, ( tPVoid ) ( * iter), ( * iter )->getInterfaceName( ) ) );
         ETG_TRACE_FATAL( ( "             %s", ( * iter )->getName( ) ) );
         i++;
      }
   #endif
} // vStart

tVoid spm_tclFactory::vRunEnv( ){
   //main thread (similar to AsfClient thread) is running in process main loop
   SPM_NULL_POINTER_CHECK( _poclCcaMsgHandler );
   while ( 1 ){
      _poclCcaMsgHandler->main( );
   }

} // vStart

tBool spm_tclFactory::bGetFastShutdown( ) const
/*!
  * \fn
  *  \brief
  *    returns whether fast shutdown is active or not.
  ******
  */
{
   SPM_NULL_POINTER_CHECK_VAL( _poSubStateHdl );
   return( _poSubStateHdl->bIsFastShutdownActive( ) );
}

tBool spm_tclFactory::bCreate( ){
/*!
  * \fn
  *  \brief
  *   Initialize AMT framework and create all FC_SPM components and client handlers.
  ******
  */
   SPM_STARTUPINVEST_INIT_STARTUPITEM
   SPM_STARTUPINVEST_INIT_TIME

   tU32 u32StartTime = 0;

   //first initial startup measurement
   _poclStartupInvest = CreateStartupInvestigation( );
   SPM_NULL_POINTER_CHECK_VAL( _poclStartupInvest );
   vAddFactoryClient( _poclStartupInvest );

   SPM_STARTUPINVEST_FORCE_ENTRY( "FACTORY", "FC SPM: Starting ...." );

   // Initialise AMT framework (tracing, static variables, ...)
   // Must be called before using AMT framework.
   SPM_STARTUPINVEST_SET_STARTTIME
   amt_bInit( );
   scd_init( );
   SPM_STARTUPINVEST_CHECK_TIME_MEASURE( "FC SPM: amt/scd initialized." )

   u32StartTime = OSAL_ClockGetElapsedTime( );

   // if SWU or DIAG have created a new customer version
   const std::string strCustomerVersionRegCfgFile = (std::string)SPM_REGISTRY_PERS_PATH_OSAL + "/customerversion_reg.cfg";
   OSAL_tIODescriptor hCustomerVersionRegCfgFile = OSAL_IOOpen( strCustomerVersionRegCfgFile.c_str( ), OSAL_EN_READONLY );
   if ( OSAL_ERROR != hCustomerVersionRegCfgFile ){
      _bIsSWVersionUpdate = TRUE;
      if ( OSAL_ERROR == OSAL_s32IOClose( hCustomerVersionRegCfgFile ) ){
         ETG_TRACE_FATAL( ( "spm_tclFactory::bCreate: Fail to close %s", strCustomerVersionRegCfgFile.c_str( ) ) );
      }
   }

   // create all spm components
   SPM_STARTUPINVEST_SET_STARTTIME
   vCreateComponents( );
   SPM_STARTUPINVEST_CHECK_TIME_MEASURE( "FC SPM: vCreateComponents ready." )

   SPM_STARTUPINVEST_SET_STARTTIME
   vCreatePrjComponents( );
   SPM_STARTUPINVEST_CHECK_TIME_MEASURE( "FC SPM: vCreatePrjComponents ready." )

   SPM_STARTUPINVEST_SET_STARTTIME
   vCreateClientHandling( );
   SPM_STARTUPINVEST_CHECK_TIME_MEASURE( "FC SPM: vCreateClientHandling ready." )

   ETG_TRACE_USR1( ( "spm_tclFactory::bCreate(): Duration 'CreateComponents' -> %d ms.", OSAL_ClockGetElapsedTime( ) - u32StartTime ) );


   u32StartTime = OSAL_ClockGetElapsedTime( );
   vGetAllReferences( );
   ETG_TRACE_USR1( ( "spm_tclFactory::bCreate(): Duration 'GetAllReferences' -> %d ms.", OSAL_ClockGetElapsedTime( ) - u32StartTime ) );

   // add startup investigation information
   SPM_STARTUPINVEST_FORCE_ENTRY( "FACTORY", "FC SPM: SPM components are up and running." );

   // Register all clients with trace possibilities
   _poclTrace->vAddTraceClient( this ); // as a workaround or for complex trace commands which need several components

   return( TRUE );
} // bCreate

tVoid spm_tclFactory::vAddFactoryClient( ISpmBase *client ){
/*!
  * \fn
  *  \brief
  *    Adds a given factory client to FC_SPM components list.
  *
  *  \param[in] client: Pointer to any object derived from ISpmbase.
  ******
  */
//ETG_TRACE_USR4(("spm_tclFactory::vAddFactoryClient(): '%s'", client->getName() ));

   #ifdef SPM_ENABLE_ADDITIONAL_DEBUG_INFO // FOR DEBUG PURPOSE TO FIND MULTIPLE CLIENT INSTANCES WITH THE SAME NAME
      if ( getSpmObjHandler( client->getName( ) ) != NULL ){
         ETG_TRACE_ERRMEM( ( "ALARM! Double Name! %s", client->getName( ) ) );
         NORMAL_M_ASSERT_ALWAYS( );
         std::list < ISpmBase* >::const_iterator iter;
         for ( iter = SpmComponentPointerList.begin( ); iter != SpmComponentPointerList.end( ); iter++ ){
            // first check if the interface name matches
            if ( 0 == OSAL_s32StringCompare( ( * iter )->getInterfaceName( ), client->getInterfaceName( ) ) ){
               ETG_TRACE_ERRMEM( ( "WARNING: Double INTERFACE client! name=%s", client->getName( ) ) );
               ETG_TRACE_ERRMEM( ( "         InterfaceName=%s", client->getInterfaceName( ) ) );
               ETG_TRACE_ERRMEM( ( "         Allready available clientName=%s", ( * iter )->getName( ) ) );
               ETG_TRACE_ERRMEM( ( "         Allready available InterfaceName=%s", ( * iter )->getInterfaceName( ) ) );
               NORMAL_M_ASSERT_ALWAYS( );
               // then check for specific implementation
            }
         }
      }
   #endif // ifdef SPM_ENABLE_ADDITIONAL_DEBUG_INFO
   SpmComponentPointerList.push_back( client );
} //lint !e429 Custodial pointer 'xxx' (line xx) has not been freed or returned --> pointer is added to list

// vAddFactoryClient

tVoid spm_tclFactory::vRemoveFactoryClient( ISpmBase *client ){
/*!
  * \fn
  *  \brief
  *    Removes a given factory client from FC_SPM components list.
  *
  *  \param[in] client: Pointer to any object derived from ISpmbase.
  ******
  */
   std::list < ISpmBase* >::iterator iter;

   for ( iter = SpmComponentPointerList.begin( ); iter != SpmComponentPointerList.end( ); iter++ ){
      if ( * iter == client ){
         ETG_TRACE_USR4( ( "spm_tclFactory::vRemoveFactoryClient(): '%s'", client->getName( ) ) );
         ETG_TRACE_USR4( ( "spm_tclFactory::vRemoveFactoryClientInterfaceName(): '%s'", client->getInterfaceName( ) ) );
         iter = SpmComponentPointerList.erase( iter );
      }
   }
}

tVoid spm_tclFactory::vGetAllReferences( ) const
/*!
  * \fn
  *  \brief
  *   Initializes the dependencies of all FC_SPM components and client handlers.
  ******
  */
{
   tU32                                    u32StartTime = OSAL_ClockGetElapsedTime( );

   SPM_STARTUPINVEST_INIT_STARTUPITEM

   std::list < ISpmBase* >::const_iterator iter;

   for ( iter = SpmComponentPointerList.begin( ); iter != SpmComponentPointerList.end( ); iter++ ){
      u32StartTime = OSAL_ClockGetElapsedTime( );
      ( * iter )->vGetReferences( );
      ETG_TRACE_USR1( ( "spm_tclFactory::vGetAllReferences(): Duration %d ms for '%50s'.", OSAL_ClockGetElapsedTime( ) - u32StartTime, ( * iter )->getName( ) ) );
      if ( ( OSAL_ClockGetElapsedTime( ) - u32StartTime ) > 50 ){
         SPM_STARTUPINVEST_FORCE_ENTRY( "FACTORY", ( (std::string)"FC SPM: vGetAllReferences: " + ( * iter )->getName( ) ) );
      }

   }
   std::list < ISpmClientHandlerBase* >::const_iterator iterClient;
   for ( iterClient = SpmClientHdlList.begin( ); iterClient != SpmClientHdlList.end( ); iterClient++ ){
      u32StartTime = OSAL_ClockGetElapsedTime( );
      ( * iterClient )->vGetReferences( );
      ETG_TRACE_USR1( ( "spm_tclFactory::vGetAllReferences(): Duration %d ms for '%50s'.", OSAL_ClockGetElapsedTime( ) - u32StartTime, ( * iterClient )->getName( ) ) );
      if ( ( OSAL_ClockGetElapsedTime( ) - u32StartTime ) > 50 ){
         SPM_STARTUPINVEST_FORCE_ENTRY( "FACTORY", ( (std::string)"FC SPM: vGetAllReferences: " + ( * iterClient )->getName( ) ) );
      }
   }
} // vGetAllReferences

ISpmBase*spm_tclFactory::getSpmObjHandler( const std::string& strName ) const
/*!
  * \fn
  *  \brief
  *   Gets an SPM object from FC_SPM components list based on the given name.
  *
  *  \param[in] strName: Name of the SPM object.
  ******
  */
{
   std::list < ISpmBase* >::const_iterator iter;
   tU8                                     u8MatchCount = 0;
   ISpmBase                               *poClassMatch = NULL;

   for ( iter = SpmComponentPointerList.begin( ); iter != SpmComponentPointerList.end( ); iter++ ){
      // first check if the interface name matches
      if ( ( * iter )->getInterfaceName( ) == strName ){
         poClassMatch = * iter;
         u8MatchCount++;
         // then check for specific implementation
      }
      if ( ( * iter )->getName( ) == strName ){
         poClassMatch = * iter;
         u8MatchCount++;
      }
   }

   std::list < ISpmClientHandlerBase* >::const_iterator iterClient;
   for ( iterClient = SpmClientHdlList.begin( ); iterClient != SpmClientHdlList.end( ); iterClient++ ){
      if ( ( * iterClient )->getInterfaceName( ) == strName ){
         poClassMatch = * iterClient;
         u8MatchCount++;
      }
      if ( ( * iterClient )->getName( ) == strName ){
         poClassMatch = * iterClient;
         u8MatchCount++;
      }
   }

   if ( u8MatchCount > 1 ){
      ETG_TRACE_ERRMEM( ( "getSpmObjHandler ALARM! Double Name! %s", strName.c_str( ) ) );
   }

   return( poClassMatch );
} // getSpmObjHandler

tVoid spm_tclFactory::vStartAllCommunication( ) const
/*!
  * \fn
  *  \brief
  *   Starts communication of FC_SPM components and client handlers.
  ******
  */
{
   tU32                                    u32StartTime = OSAL_ClockGetElapsedTime( );

   SPM_STARTUPINVEST_INIT_STARTUPITEM

   std::list < ISpmBase* >::const_iterator iter;

   for ( iter = SpmComponentPointerList.begin( ); iter != SpmComponentPointerList.end( ); iter++ ){
      u32StartTime = OSAL_ClockGetElapsedTime( );
      ( * iter )->vStartCommunication( );
      ETG_TRACE_USR1( ( "spm_tclFactory::vStartAllCommunication(): Duration %d ms for '%50s'.", OSAL_ClockGetElapsedTime( ) - u32StartTime, ( * iter )->getName( ) ) );
      if ( ( OSAL_ClockGetElapsedTime( ) - u32StartTime ) > 50 ){
         SPM_STARTUPINVEST_FORCE_ENTRY( "FACTORY", ( (std::string)"FC SPM: vStartCommunication: " + ( * iter )->getName( ) ) );
      }
   }

   std::list < ISpmClientHandlerBase* >::const_iterator iterClient;
   for ( iterClient = SpmClientHdlList.begin( ); iterClient != SpmClientHdlList.end( ); iterClient++ ){
      u32StartTime = OSAL_ClockGetElapsedTime( );
      ( * iterClient )->vStartCommunication( );
      ETG_TRACE_USR1( ( "spm_tclFactory::vStartAllCommunication(): Duration %d ms for '%50s'.", OSAL_ClockGetElapsedTime( ) - u32StartTime, ( * iterClient )->getName( ) ) );
      if ( ( OSAL_ClockGetElapsedTime( ) - u32StartTime ) > 50 ){
         SPM_STARTUPINVEST_FORCE_ENTRY( "FACTORY", ( (std::string)"FC SPM: vStartCommunication: " + ( * iterClient )->getName( ) ) );
      }
   }
} // vStartAllCommunication

tVoid spm_tclFactory::vStartAllLateCommunication( ) const
/*!
  * \fn
  *  \brief
  *   Starts communication of FC_SPM components and client handlers.
  ******
  */
{
   tU32                                    u32StartTime = OSAL_ClockGetElapsedTime( );

   vHandleCustomerVersionReg( );

   std::list < ISpmBase* >::const_iterator iter;
   for ( iter = SpmComponentPointerList.begin( ); iter != SpmComponentPointerList.end( ); iter++ ){
      u32StartTime = OSAL_ClockGetElapsedTime( );
      ( * iter )->vStartLateCommunication( );
         ETG_TRACE_USR1( ( "spm_tclFactory::vStartAllLateCommunication(): Duration %d ms for '%50s'.", OSAL_ClockGetElapsedTime( ) - u32StartTime, ( * iter )->getName( ) ) );
   }

   std::list < ISpmClientHandlerBase* >::const_iterator iterClient;
   for ( iterClient = SpmClientHdlList.begin( ); iterClient != SpmClientHdlList.end( ); iterClient++ ){
      u32StartTime = OSAL_ClockGetElapsedTime( );
      ( * iterClient )->vStartLateCommunication( );
      ETG_TRACE_USR1( ( "spm_tclFactory::vStartAllLateCommunication(): Duration %d ms for '%50s'.", OSAL_ClockGetElapsedTime( ) - u32StartTime, ( * iterClient )->getName( ) ) );
   }

#ifdef SPM_FEATURE_SD_NOTIFY_BY_PROJECT_CONFIG
   if (_poclSupervisionManager != NULL) _poclSupervisionManager->vStartUpFinished();
#endif

} // vStartAllLateCommunication

   tVoid spm_tclFactory::vAddFactoryClientHandler( ISpmClientHandlerBase *client ){
/*!
  * \fn
  *  \brief
  *   Adds a given client handler to List of Client Handlers.
  *
  *  \param[in] client: pointer to any object derived from ISpmClientHandlerBase.
  ******
  */
            ETG_TRACE_USR4( ( "spm_tclFactory::vAddFactoryClientHandler(): '%s'", client->getName( ) ) );
      SpmClientHdlList.push_back( client );

   }

   tVoid spm_tclFactory::vRemoveFactoryClientHandler( ISpmClientHandlerBase *client ){
      /*!
        * \fn
        *  \brief
        *   Removes a given client handler from List of Client Handlers
        *
        *  \param[in] client: pointer to any object derived from ISpmClientHandlerBase.
        ******
        */
      std::list < ISpmClientHandlerBase* >::iterator iter;

      for ( iter = SpmClientHdlList.begin( ); iter != SpmClientHdlList.end( ); iter++ ){
         if ( * iter == client ){
            ETG_TRACE_USR4( ( "spm_tclFactory::vRemoveFactoryClientHandler(): '%s'", client->getName( ) ) );
            ETG_TRACE_USR4( ( "spm_tclFactory::vRemoveFactoryClientHandler(): '%s'", client->getInterfaceName( ) ) );
            iter = SpmClientHdlList.erase( iter );
         }
      }

   }

   tVoid spm_tclFactory::vAddAllClientHandler( ) const
/*!
  * \fn
  *  \brief
  *   Adds all client handlers to CCA Service Handler.
  ******
  */
   {
      std::list < ISpmClientHandlerBase* >::const_iterator iter;

      SPM_NULL_POINTER_CHECK( _poServiceHdl );

      for ( iter = SpmClientHdlList.begin( ); iter != SpmClientHdlList.end( ); iter++ ){
         // ETG_TRACE_USR4(("spm_tclFactory::vAddAllClientHandler(): '%s'", (*iter)->getName() ));
         _poServiceHdl->vAddClientHandler( ( * iter ) );
      }
   }

   ISpmClientHandlerBase*spm_tclFactory::getClientHandler( const std::string& strName ) const
/*!
  * \fn
  *  \brief
  *   Gets a client handler from List of Client Handlers List on given name.
  *
  *  \param[in] strName: Name of the client handler.
  ******
  */
   {
      std::list < ISpmClientHandlerBase* >::const_iterator iter;

      for ( iter = SpmClientHdlList.begin( ); iter != SpmClientHdlList.end( ); iter++ ){
         // first check if the interface name matches
         if ( ( * iter )->getInterfaceName( ) == strName ){
            return ( * iter );
            // then check for specific implementation
         }
         if ( ( * iter )->getName( ) == strName ){
            return ( * iter );
         }
      }
      return( NULL );
   }

// EOF

