/*!
  * \file spm_SystemStateStatistics.cpp
  *  \brief
  *    SystemState statistics of the system
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  *  Date      | Author              | Modification
  * 09.11.12  | CM-AI/PJ-G33 Rossner | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// -----------------------------------------------------------------------------
// includes FC SPM
// -----------------------------------------------------------------------------
#include "spm_Config.h"
#include "spm_SystemStateStatistics.h"
#include "spm_IFactory.h"
#include "spm_SubStateHandler.h"
#include "timeConvert.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
 #include "trcGenProj/Header/spm_SystemStateStatistics.cpp.trc.h"
#endif

// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
#define SPM_U32_STATISTIC_MAX_ERR_STACKS                    5

template < typename T1, typename T2,typename T3, typename T4, typename T5, typename T6, typename T7 >
spm_tclSystemStateStatisticsTemplate<T1, T2, T3, T4, T5, T6, T7>::spm_tclSystemStateStatisticsTemplate( const ISpmFactory& factory, enStatisticsType eStatType )
   : ISpmSystemStateStatistics( factory ){
   _eStatType = eStatType;
}

template < typename T1, typename T2,typename T3, typename T4, typename T5, typename T6, typename T7 >
spm_tclSystemStateStatisticsTemplate<T1, T2, T3, T4, T5, T6, T7>::~spm_tclSystemStateStatisticsTemplate( ){
}

template < typename T1, typename T2,typename T3, typename T4, typename T5, typename T6, typename T7 > template < typename S >
tVoid spm_tclSystemStateStatisticsTemplate<T1, T2, T3, T4, T5, T6, T7>::vCopySysStateHistory( S &oStateHistorySrc ,TSpmSystemStateStatisticsData *pStatistics, tU32 u32ObjectSize ) const{
   if ( pStatistics != NULL ){
      tU32  u32Count = oStateHistorySrc.u32GetCount( );
      std::list< TSpmSystemStateStatisticsData > tSystemStateList;
      if ( u32ObjectSize >= u32Count * sizeof( TSpmSystemStateStatisticsData ) ){
         for ( tUInt i = 0; i < u32Count; ++i ){
            TSpmSystemStateStatisticsData tSystemStateElem = TSpmSystemStateStatisticsData( );

            oStateHistorySrc.s32GetElem( i, tSystemStateElem );
            tSystemStateList.push_back( tSystemStateElem );
         }
         memset( pStatistics, 0, u32ObjectSize );
         std::copy( tSystemStateList.begin(), tSystemStateList.end(), pStatistics );
      } else {
         ETG_TRACE_FATAL( ("ResetStatistics::vGetData(): buffer size of SpmResetElem is not valid") );
      }
   } else {
	   ETG_TRACE_ERRMEM( ("ResetStatistics::vGetData() pStatistics buffer of SpmResetElem is NULL") );
   }
}

template < typename T1, typename T2,typename T3, typename T4, typename T5, typename T6, typename T7 >
tVoid spm_tclSystemStateStatisticsTemplate<T1, T2, T3, T4, T5, T6, T7>::vUpdateSystemStateHistory( TSpmSystemStateStatisticsData& oStateElem ) const{
/*!
  * \fn
  *  \brief
  *    store the system state to data pool.
  *  \param[in]  TSpmSystemStateStatisticsData the system state history element need to store in datapool
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   T7 oStateHistory;
   oStateHistory.vPushBack( oStateElem );
   switch ( _eStatType )
   {
      case SPM_U32_STATISTIC_GLOBAL:
      {
         SPM_GET_CLASS_REFERENCE_NEW_VAR( poclSWVersionSystemStateStatistics, spm_tclSWVersionSystemStateStatistics, ISpmSystemStateStatistics );
         poclSWVersionSystemStateStatistics->vUpdateSystemStateHistory( oStateElem );
         break;
      }
      default:
         break;
   }
}

template < typename T1, typename T2,typename T3, typename T4, typename T5, typename T6, typename T7 >
tVoid spm_tclSystemStateStatisticsTemplate<T1, T2, T3, T4, T5, T6, T7>::vGetDataSysStateHistory( TSpmSystemStateStatisticsData *pStatistics, tU32 u32ObjectSize ) const{
/*!
  * \fn
  *  \brief
  *    get the system state history.
  *  \param[out] pStatistics:  pointer to TSpmSystemStateStatisticsData buffer of system state history.
  *  \param[in]  u32ObjectSize: buffer size.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  *  \details  All the exceed buffer data will be set to 0 if the u32ObjectSize > size of system state history list.
  ******
  */
   T7 oStateHistory;
   vCopySysStateHistory( oStateHistory, pStatistics, u32ObjectSize );
}

template < typename T1, typename T2,typename T3, typename T4, typename T5, typename T6, typename T7 >
tVoid spm_tclSystemStateStatisticsTemplate<T1, T2, T3, T4, T5, T6, T7>::vGetDataErrSysStateStateHistoryStack01( TSpmSystemStateStatisticsData *pStatistics, tU32 u32ObjectSize ) const{
/*!
  * \fn
  *  \brief
  *    get the error system state history stack 01.
  *  \param[out] pStatistics:  pointer to TSpmSystemStateStatisticsData buffer of error system state history.
  *  \param[in]  u32ObjectSize: buffer size.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  *  \details  All the exceed buffer data will be set to 0 if the u32ObjectSize > size of error system state history list.
  ******
  */
   T1 oStateHistory;
   vCopySysStateHistory( oStateHistory, pStatistics, u32ObjectSize );
}

template < typename T1, typename T2,typename T3, typename T4, typename T5, typename T6, typename T7 >
tVoid spm_tclSystemStateStatisticsTemplate<T1, T2, T3, T4, T5, T6, T7>::vGetDataErrSysStateStateHistoryStack02( TSpmSystemStateStatisticsData *pStatistics, tU32 u32ObjectSize ) const{
/*!
  * \fn
  *  \brief
  *    get the error system state history stack 02.
  *  \param[out] pStatistics:  pointer to TSpmSystemStateStatisticsData buffer of error system state history.
  *  \param[in]  u32ObjectSize: buffer size.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  *  \details  All the exceed buffer data will be set to 0 if the u32ObjectSize > size of error system state history list.
  ******
  */
   T2 oStateHistory;
   vCopySysStateHistory( oStateHistory, pStatistics, u32ObjectSize );
}

template < typename T1, typename T2,typename T3, typename T4, typename T5, typename T6, typename T7 >
tVoid spm_tclSystemStateStatisticsTemplate<T1, T2, T3, T4, T5, T6, T7>::vGetDataErrSysStateStateHistoryStack03( TSpmSystemStateStatisticsData *pStatistics, tU32 u32ObjectSize ) const{
/*!
  * \fn
  *  \brief
  *    get the error system state history stack 03.
  *  \param[out] pStatistics:  pointer to TSpmSystemStateStatisticsData buffer of error system state history.
  *  \param[in]  u32ObjectSize: buffer size.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  *  \details  All the exceed buffer data will be set to 0 if the u32ObjectSize > size of error system state history list.
  ******
  */
   T3 oStateHistory;
   vCopySysStateHistory( oStateHistory, pStatistics, u32ObjectSize );
}

template < typename T1, typename T2,typename T3, typename T4, typename T5, typename T6, typename T7 >
tVoid spm_tclSystemStateStatisticsTemplate<T1, T2, T3, T4, T5, T6, T7>::vGetDataErrSysStateStateHistoryStack04( TSpmSystemStateStatisticsData *pStatistics, tU32 u32ObjectSize ) const{
/*!
  * \fn
  *  \brief
  *    get the error system state history stack 04.
  *  \param[out] pStatistics:  pointer to TSpmSystemStateStatisticsData buffer of error system state history.
  *  \param[in]  u32ObjectSize: buffer size.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  *  \details  All the exceed buffer data will be set to 0 if the u32ObjectSize > size of error system state history list.
  ******
  */
   T4 oStateHistory;
   vCopySysStateHistory( oStateHistory, pStatistics, u32ObjectSize );
}

template < typename T1, typename T2,typename T3, typename T4, typename T5, typename T6, typename T7 >
tVoid spm_tclSystemStateStatisticsTemplate<T1, T2, T3, T4, T5, T6, T7>::vGetDataErrSysStateStateHistoryStack05( TSpmSystemStateStatisticsData *pStatistics, tU32 u32ObjectSize ) const{
/*!
  * \fn
  *  \brief
  *    get the error system state history stack 05.
  *  \param[out] pStatistics:  pointer to TSpmSystemStateStatisticsData buffer of error system state history.
  *  \param[in]  u32ObjectSize: buffer size.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  *  \details  All the exceed buffer data will be set to 0 if the u32ObjectSize > size of error system state history list.
  ******
  */
   T5 oStateHistory;
   vCopySysStateHistory( oStateHistory, pStatistics, u32ObjectSize );
}

template < typename T1, typename T2,typename T3, typename T4, typename T5, typename T6, typename T7 >
tVoid spm_tclSystemStateStatisticsTemplate<T1, T2, T3, T4, T5, T6, T7>::vErase( ) const {
   T7 oHistory;

   oHistory.vClearList( );
}

template < typename T1, typename T2,typename T3, typename T4, typename T5, typename T6, typename T7 >
tVoid spm_tclSystemStateStatisticsTemplate<T1, T2, T3, T4, T5, T6, T7>::vTrace( ) const {

   T7 oHistory;

                         ETG_TRACE_FATAL( ( "-----------------------------------------------------------------" ) );
                         ETG_TRACE_FATAL( ( "SystemState History:" ) );
                         ETG_TRACE_FATAL( ( "-----------------------------------------------------------------" ) );

   tU32 u32Count = oHistory.u32GetCount( );
   for ( tUInt i = 0; i < u32Count; ++i ){
      TSpmSystemStateStatisticsData tElem;
      tclTimeConvert                oTimeConv;
      OSAL_trTimeDate               tTimeDate = { 0,0,0,0,0,0,0,0,0 };

      oHistory.s32GetElem( u32Count - i - 1, tElem );
      oTimeConv.vGetDateFromElapsedSeconds( tElem.u32Time, &tTimeDate );

                         ETG_TRACE_FATAL( ( "System State %2d:                %3d (%04d-%02d-%02d  %02d:%02d:%02d)(UTC)",
                         i + 1,
                         ETG_ENUM( SPM_SYSTEM_STATES, tElem.u8SystemState ),
                         tTimeDate.s32Year, tTimeDate.s32Month, tTimeDate.s32Day,
                         tTimeDate.s32Hour, tTimeDate.s32Minute, tTimeDate.s32Second
                         ) );
      if ( tElem.u8EntryType == SPM_STATE_HISTORY_ENTRY_WAKEUP ){
                            ETG_TRACE_FATAL( ( "WakeUp reason:                  %3d", ETG_ENUM( SPM_WAKEUP_REASON, tElem.u.u8Wakeup ) ) );
      } else if ( tElem.u8EntryType == SPM_STATE_HISTORY_ENTRY_SUBSTATE ){
                            ETG_TRACE_FATAL( ( "State trigger:" ) );
         spm_tclSubStateHandler::vTraceTrigger( SSM_vHistoryTrigger, &tElem.u.tTrigger, TR_LEVEL_FATAL );
      } else if ( tElem.u8EntryType == SPM_STATE_HISTORY_ENTRY_RESET ){
                            ETG_TRACE_FATAL( ( "Reset reason :                  %3d", ETG_ENUM( CFC_FI_LCMRESETREASON, tElem.u.u8Reset ) ) );
      }
                            ETG_TRACE_FATAL( ( "-----------------------------------------------------------------" ) );
   }

} // vTrace

template < typename T1, typename T2,typename T3, typename T4, typename T5, typename T6, typename T7 >
tVoid spm_tclSystemStateStatisticsTemplate<T1, T2, T3, T4, T5, T6, T7>::vTraceErrorHistory( tU8 u8Number ) const {

   if ( u8Number < SPM_U32_STATISTIC_MAX_ERR_STACKS ){

      T7 oHistory;
      tU32                       u32Count = oHistory.u32GetCount( );
      for ( tUInt i = 0; ( i < u32Count ) && ( i < SPM_U32_STATISTIC_MAX_ERR_ENTRIES_PER_STACKS ); ++i ){
         TSpmSystemStateStatisticsData tElem  ;
         tclTimeConvert                oTimeConv;
         OSAL_trTimeDate               tTimeDate = { 0,0,0,0,0,0,0,0,0 };

         if ( u8Number == 0 ){
            T1 oErrHistory1;
            oErrHistory1.s32GetElem( i, tElem );
         } else if ( u8Number == 1 ){
            T2 oErrHistory2;
            oErrHistory2.s32GetElem( i, tElem );
         } else if ( u8Number == 2 ){
            T3 oErrHistory3;
            oErrHistory3.s32GetElem( i, tElem );
         } else if ( u8Number == 3 ){
            T4 oErrHistory4;
            oErrHistory4.s32GetElem( i, tElem );
         } else if ( u8Number == 4 ){
            T5 oErrHistory5;
            oErrHistory5.s32GetElem( i, tElem );
         }
         oTimeConv.vGetDateFromElapsedSeconds( tElem.u32Time, &tTimeDate );

                            ETG_TRACE_FATAL( ( "System State %2d:                %3d (%04d-%02d-%02d  %02d:%02d:%02d)(UTC)",
                            i + 1,
                            ETG_ENUM( SPM_SYSTEM_STATES, tElem.u8SystemState ),
                            tTimeDate.s32Year, tTimeDate.s32Month, tTimeDate.s32Day,
                            tTimeDate.s32Hour, tTimeDate.s32Minute, tTimeDate.s32Second
                            ) );
         if ( tElem.u8EntryType == SPM_STATE_HISTORY_ENTRY_WAKEUP ){
            ETG_TRACE_FATAL( ( "WakeUp reason:                  %3d", ETG_ENUM( SPM_WAKEUP_REASON, tElem.u.u8Wakeup ) ) );
         } else if ( tElem.u8EntryType == SPM_STATE_HISTORY_ENTRY_SUBSTATE ){
            ETG_TRACE_FATAL( ( "State trigger:" ) );
            spm_tclSubStateHandler::vTraceTrigger( SSM_vHistoryTrigger, &tElem.u.tTrigger, TR_LEVEL_FATAL );
         } else if ( tElem.u8EntryType == SPM_STATE_HISTORY_ENTRY_RESET ){
            ETG_TRACE_FATAL( ( "Reset reason :                  %3d", ETG_ENUM( CFC_FI_LCMRESETREASON, tElem.u.u8Reset ) ) );
         }
            ETG_TRACE_FATAL( ( "-----------------------------------------------------------------" ) );
      }
   }
} // vTraceErrorHistory

/*!
  * \fn
  *  \brief
  *   Function copies the content from one ErrorStateHistory Datapool object to another.
  *  \param[in] oErrStateHistorySrc: dp_tclSpmDpEngStateHistoryStack**(source)
  *  \param[in] oErrStateHistoryDest: dp_tclSpmDpEngStateHistoryStack**(destination)
  *  \param[in] bCopyInReverseOrder: boolean
  *  \details
  *   If the source obj is of type dp_tclSpmDpEngStateHistory then the order
  *   of contents copied needs to be reversed to print the latest ErrorState first.
  ******
  */
template < typename T1, typename T2,typename T3, typename T4, typename T5, typename T6, typename T7 > template < typename S, typename T >
tVoid spm_tclSystemStateStatisticsTemplate<T1, T2, T3, T4, T5, T6, T7>::vCopyErrStateHistoryElements( S   & oErrStateHistorySrc,
                                                                  T   & oErrStateHistoryDest,
                                                                  tBool bCopyInReverseOrder ) const {
   TSpmSystemStateStatisticsData tElem;
   tU32                           u32Count = oErrStateHistorySrc.u32GetCount( );

   for ( tUInt j = 0; ( j < u32Count ) && ( j < SPM_U32_STATISTIC_MAX_ERR_ENTRIES_PER_STACKS ); ++j ){
      if ( bCopyInReverseOrder == TRUE ){
         oErrStateHistorySrc.s32GetElem( u32Count - j - 1, tElem );
      } else {
         oErrStateHistorySrc.s32GetElem( j, tElem );
      }
      oErrStateHistoryDest.vPushBack( tElem );
   }
} //vCopyErrStateHistoryElements

/*!
  * \fn
  *  \brief
  *   Function copies State History into corresponding ErrorStateHistory datapool.  
  *  \return 
  *    tU32  - Stack Index of last State History List
  *  \details
  *   The StateHistoryStack01,StateHistoryStack02 are persistent will be written only 1 time.
  *   StateHistoryStack03, StateHistoryStack04, StateHistoryStack05 can be overwritten
  *   StateHistoryStack05 is newest . The older StateHistoryStack05  will be shifted to StateHistoryStack04 and StateHistoryStack03
  ******
  */
template < typename T1, typename T2,typename T3, typename T4, typename T5, typename T6, typename T7 >
tU32 spm_tclSystemStateStatisticsTemplate<T1, T2, T3, T4, T5, T6, T7>::u32AddErrorHistory( ) const {
   tU32                                u32CurErrStack;
   T6 oCurStack;

   oCurStack >> u32CurErrStack;
   if ( u32CurErrStack >= SPM_U32_STATISTIC_MAX_ERR_STACKS ){
       ETG_TRACE_ERRMEM( ( "spm_tclSystemStateStatistics::u32AddErrorHistory, Error Detected ! Datapool dp_tclSpmDpEngStateLastHistoryStack corrupted, now reset u32CurErrStack to 4") );
       u32CurErrStack =4;
   } 
   
      T7        oHistory;
      T1 oErrHistory1;
      T2 oErrHistory2;
      T3 oErrHistory3;
      T4 oErrHistory4;
      T5 oErrHistory5;
      ETG_TRACE_FATAL( ( "Add ErrorHistory stack %d!", u32CurErrStack ) );

      switch ( u32CurErrStack )
      {
        case 0 :
        {
            vCopyErrStateHistoryElements( oHistory, oErrHistory1, TRUE );
            if (oErrHistory1.u32GetCount( )  > 0 )
            {
            u32CurErrStack = 1;
            }
            break;
        }
        case 1 :
        {
            vCopyErrStateHistoryElements( oHistory, oErrHistory2, TRUE );
            if (oErrHistory2.u32GetCount( )  > 0 )
            {
                u32CurErrStack = 2;
            }
            break;
        }
        case 2 :
        {
               vCopyErrStateHistoryElements( oHistory, oErrHistory3, TRUE );
            if (oErrHistory3.u32GetCount( )  > 0 )
            {
               u32CurErrStack =3;
            }
            break;
        }
        case 3 :
        {
               vCopyErrStateHistoryElements( oHistory, oErrHistory4, TRUE );
            if (oErrHistory4.u32GetCount( )  > 0 )
            {
               u32CurErrStack =4;
            }
            break;
        }
        case 4 :
        {

            if (oErrHistory5.u32GetCount( )  > 0 )
            {
               oErrHistory3.vClearList();
               vCopyErrStateHistoryElements( oErrHistory4, oErrHistory3, FALSE );
               oErrHistory4.vClearList();
               vCopyErrStateHistoryElements( oErrHistory5, oErrHistory4, FALSE );
               // Clear Stack 5 and prepare to copy data from oHistory next
               oErrHistory5.vClearList();
            }
            vCopyErrStateHistoryElements( oHistory, oErrHistory5, TRUE );
            break;
        }
        default :
        {
            break;
        }
      }
   
   oCurStack << u32CurErrStack;
   switch ( _eStatType )
   {
      case SPM_U32_STATISTIC_GLOBAL:
      {
         SPM_GET_CLASS_REFERENCE_NEW_VAR_VAL( poclSWVersionSystemStateStatistics, spm_tclSWVersionSystemStateStatistics, ISpmSystemStateStatistics );
         poclSWVersionSystemStateStatistics->u32AddErrorHistory(  );
         break;
      }
      default:
         break;
   }
   return u32CurErrStack;
} // u32AddErrorHistory

template < typename T1, typename T2,typename T3, typename T4, typename T5, typename T6, typename T7 >
tVoid spm_tclSystemStateStatisticsTemplate<T1, T2, T3, T4, T5, T6, T7>::vEraseErrorHistory( ) const {
   {
      T1 oHistory;
      oHistory.vClearList( );
   }
   {
      T2 oHistory;
      oHistory.vClearList( );
   }
   {
      T3 oHistory;
      oHistory.vClearList( );
   }
   {
      T4 oHistory;
      oHistory.vClearList( );
   }
   {
      T5 oHistory;
      oHistory.vClearList( );
   }
   {
      T6 oCurStack;
      oCurStack << 0;
   }
} // vEraseErrorHistory


spm_tclSystemStateStatistics::spm_tclSystemStateStatistics( const ISpmFactory& factory )
   : spm_tclSystemStateStatisticsTemplate<dp_tclSpmDpEngStateHistoryStack01, dp_tclSpmDpEngStateHistoryStack02, dp_tclSpmDpEngStateHistoryStack03, dp_tclSpmDpEngStateHistoryStack04, dp_tclSpmDpEngStateHistoryStack05, dp_tclSpmDpEngStateLastHistoryStack ,dp_tclSpmDpEngStateHistory >( factory, SPM_U32_STATISTIC_GLOBAL )
{
}

spm_tclSWVersionSystemStateStatistics::spm_tclSWVersionSystemStateStatistics( const ISpmFactory& factory )
   : spm_tclSystemStateStatisticsTemplate < dp_tclSpmDpEngStateCurrentSWVersionHistoryStack01, dp_tclSpmDpEngStateCurrentSWVersionHistoryStack02, dp_tclSpmDpEngStateCurrentSWVersionHistoryStack03, dp_tclSpmDpEngStateCurrentSWVersionHistoryStack04, dp_tclSpmDpEngStateCurrentSWVersionHistoryStack05, dp_tclSpmDpEngStateCurrentSWVersionLastHistoryStack ,dp_tclSpmDpEngStateCurrentSWVersionHistory >( factory, SPM_U32_STATISTIC_SOFTWARE_VERSION )
{
}


// EOF

