/**
  * @swcomponent   Life Cycle Management
  * @{
  * @file          spm_SystemLoadStatistics.cpp
  * @PROJECT:      CMD project
  * @brief         implementation of spm_tclSystemLoadStatistics class
  *
  * ----------------------------------------------------------------------------
  * @copyright     (C) 2018 Robert Bosch GmbH.
  *                The reproduction, distribution and utilization of this file as well as the
  *                communication of its contents to others without express authorization is prohibited.
  *                Offenders will be held liable for the payment of damages.
  *                All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_STDVISITORS
#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_FUNCTIONIDS
#include "spm_fi_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

#include <algorithm>
#include <cstring>

// -----------------------------------------------------------------------------
// includes FC SPM
// -----------------------------------------------------------------------------
#include "spm_Config.h"
#include "spm_SystemLoadStatistics.h"
#include "spm_IFactory.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
 #include "trcGenProj/Header/spm_SystemLoadStatistics.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
template < typename T >
spm_tclSystemLoadStatistics < T >::spm_tclSystemLoadStatistics( const ISpmFactory& factory, enSystemLoadStatisticsType eSysLoadType, enStatisticsType eStatType )
   : ISpmSystemLoadStatistics( factory )
   , _tSystemLoadStatisticsData( )
   , _poSystemLoadStatistics( NULL )
   , _poSpmSupervisionClient( NULL ){
/*!
  * \fn
  *  \brief
  *    Constructor
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  *  \details load the system load statistics data from data pool.
  ******
  */
   T oSystemLoadStatisticsDataDataPool;
   oSystemLoadStatisticsDataDataPool >> _tSystemLoadStatisticsData;

   _eStatType     = eStatType;
   _eSysLoadType  = eSysLoadType;
} // spm_tclSystemLoadStatistics

template < typename T >
spm_tclSystemLoadStatistics< T >::~spm_tclSystemLoadStatistics( ){
   if ( _poSystemLoadStatistics ){
      _poSystemLoadStatistics = NULL;
   }
   if ( _poSpmSupervisionClient ){
      _poSpmSupervisionClient = NULL;
   }
} // ~spm_tclSystemLoadStatistics

template < typename T >
tVoid spm_tclSystemLoadStatistics< T >::vGetReferences( ){
   switch ( _eStatType )
   {
      case SPM_U32_STATISTIC_GLOBAL:
      {
         switch( _eSysLoadType )
         {
            case SPM_U32_STATISTIC_CPU_USAGE:
            {
               SPM_GET_CLASS_REFERENCE_USE_VAR( _poSystemLoadStatistics, spm_tclSWVersionCpuUsageStatistics, ISpmSystemLoadStatistics );
               break;
            }
            case SPM_U32_STATISTIC_RAM_USAGE:
            {
               SPM_GET_CLASS_REFERENCE_USE_VAR( _poSystemLoadStatistics, spm_tclSWVersionRamUsageStatistics, ISpmSystemLoadStatistics );
               break;
            }
            case SPM_U32_STATISTIC_STARTUP_TIME:
            {
               SPM_GET_CLASS_REFERENCE_USE_VAR( _poSystemLoadStatistics, spm_tclSWVersionStartupTimeStatistics, ISpmSystemLoadStatistics );
               break;
            }
            default:
            break;
         }
         break;
      }
      default:
         break;
   }
   switch( _eSysLoadType )
   {
      case SPM_U32_STATISTIC_CPU_USAGE:
      {
         SPM_GET_CLASS_REFERENCE_USE_VAR( _poSpmSupervisionClient, spm_tclCpuLoadSupervisor, ISpmSupervisionClient );
         break;
      }
      case SPM_U32_STATISTIC_RAM_USAGE:
      {
         SPM_GET_CLASS_REFERENCE_USE_VAR( _poSpmSupervisionClient, spm_tclRamLoadSupervisor, ISpmSupervisionClient );
         break;
      }
      case SPM_U32_STATISTIC_STARTUP_TIME:
      {
         SPM_GET_CLASS_REFERENCE_USE_VAR( _poSpmSupervisionClient, spm_tclStartupSupervisor, ISpmStartupSupervisor );
         break;
      }
      default:
      break;
   }
} // vGetReferences

template < typename T >
tVoid spm_tclSystemLoadStatistics< T >::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *    called during start-up.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   _tSystemLoadStatisticsData.bStatisticDataDisabled = ( !( _poSpmSupervisionClient->bSupervisionEnabled( ) ) );
} // vStartCommunication

template < typename T >
tVoid spm_tclSystemLoadStatistics< T >::bUpdateLoading( tU32 u32Load ){
/*!
  * \fn
  *  \brief
  *    Update min/max/avg values of system load statistics data. This is called in supervisor of system loading.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   if ( FALSE == _tSystemLoadStatisticsData.bStatisticDataDisabled ){
      tU32 u32Counter = _tSystemLoadStatisticsData.u32CounterValue;
      _tSystemLoadStatisticsData.u64SumOfLoading += u32Load;
      if ( u32Counter == 0 ){
         _tSystemLoadStatisticsData.u32MaxValue = _tSystemLoadStatisticsData.u32MinValue = _tSystemLoadStatisticsData.u32AvgValue = u32Load;
      } else {
         _tSystemLoadStatisticsData.u32MaxValue = std::max( _tSystemLoadStatisticsData.u32MaxValue, u32Load );
         _tSystemLoadStatisticsData.u32MinValue = std::min( _tSystemLoadStatisticsData.u32MinValue, u32Load );
         _tSystemLoadStatisticsData.u32AvgValue = ( tU32 )( ( _tSystemLoadStatisticsData.u64SumOfLoading ) / ( u32Counter + 1 ) );
      }
      _tSystemLoadStatisticsData.u32CounterValue++;

      if ( _poSystemLoadStatistics != NULL ){
         _poSystemLoadStatistics->bUpdateLoading( u32Load );
      }
   } else {
      ETG_TRACE_FATAL( ( "%s Statistics Data is disable!!!", getName( ) ) );
   }
} // bUpdateLoading

template < typename T >
tVoid spm_tclSystemLoadStatistics< T >::vSetNewData( tU32 u32StatisticType ) const {
/*!
  * \fn
  *  \brief
  *    write current system load statistics data to data pool.
  *
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   (void) u32StatisticType;
   T oSystemLoadStatisticsDataDataPool;
   oSystemLoadStatisticsDataDataPool << _tSystemLoadStatisticsData;
   if ( _poSystemLoadStatistics != NULL ){
      _poSystemLoadStatistics->vSetNewData( u32StatisticType );
   }
} // vSetNewData

template < typename T >
tVoid spm_tclSystemLoadStatistics< T >::vGetData( void *pStatistics,
                                             tU32  u32ObjectSize ) const {
/*!
  * \fn
  *  \brief
  *  Get the system load statistics data.
  *
  *  \param [out] void *pStatistics: buffer for getting the system load statistics.
  *  \param [in] tU32  u32ObjectSize: size of buffer.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   if ( pStatistics != NULL ){
      if ( u32ObjectSize >= sizeof( _tSystemLoadStatisticsData ) ){
         std::memcpy( pStatistics, &_tSystemLoadStatisticsData, sizeof( _tSystemLoadStatisticsData ) );
      } else {
         ETG_TRACE_FATAL( ( "spm_tclSystemLoadStatistics::vGetData(): %s buffer size is not valid", getName( ) ) );
      }
   } else {
	   ETG_TRACE_ERRMEM( ( "spm_tclSystemLoadStatistics::vGetData(): %s pStatistics buffer is NULL", getName( ) ) );
   }
} // vGetData

template < typename T >
tVoid spm_tclSystemLoadStatistics< T >::vEraseLoading( ) {
/*!
  * \fn
  *  \brief
  *  Erase system load data in data pool.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   T oSystemLoadStatisticsDataDataPool;
   std::memset( &_tSystemLoadStatisticsData, 0, sizeof( _tSystemLoadStatisticsData ) );
   oSystemLoadStatisticsDataDataPool << _tSystemLoadStatisticsData;
} // vEraseLoading

template < typename T >
tVoid spm_tclSystemLoadStatistics< T >::vTrace( ) const {
/*!
  * \fn
  *  \brief
  *   Trace of the system load statistics.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   if ( FALSE == _tSystemLoadStatisticsData.bStatisticDataDisabled ) {
      ETG_TRACE_FATAL( ( "---------------------------------------------------------------------------" ) );
      ETG_TRACE_FATAL( ( "%s Statistics Data", getName( ) ) );
      ETG_TRACE_FATAL( ( "Max Value:         %u", _tSystemLoadStatisticsData.u32MaxValue     ) );
      ETG_TRACE_FATAL( ( "Avg Value:         %u", _tSystemLoadStatisticsData.u32AvgValue     ) );
      ETG_TRACE_FATAL( ( "Min Value:         %u", _tSystemLoadStatisticsData.u32MinValue     ) );
      ETG_TRACE_FATAL( ( "Sum of loading:    %u", _tSystemLoadStatisticsData.u64SumOfLoading ) );
      ETG_TRACE_FATAL( ( "Counter Value:     %u", _tSystemLoadStatisticsData.u32CounterValue ) );
      ETG_TRACE_FATAL( ( "---------------------------------------------------------------------------" ) );
   } else {
      ETG_TRACE_FATAL( ( "---------------------------------------------------------------------------" ) );
      ETG_TRACE_FATAL( ( "%s Statistics Data is disable!!!", getName( ) ) );
      ETG_TRACE_FATAL( ( "---------------------------------------------------------------------------" ) );
   }
} // vTrace

spm_tclCpuUsageStatistics::spm_tclCpuUsageStatistics( const ISpmFactory& factory )
   : spm_tclSystemLoadStatistics< dp_tclSpmDpEngDataCpuUsageStatistics >( factory, SPM_U32_STATISTIC_CPU_USAGE, SPM_U32_STATISTIC_GLOBAL )
{
}

spm_tclRamUsageStatistics::spm_tclRamUsageStatistics( const ISpmFactory& factory )
   : spm_tclSystemLoadStatistics< dp_tclSpmDpEngDataRamUsageStatistics >(factory, SPM_U32_STATISTIC_RAM_USAGE, SPM_U32_STATISTIC_GLOBAL  )
{
}

spm_tclStartupTimeStatistics::spm_tclStartupTimeStatistics( const ISpmFactory& factory )
   : spm_tclSystemLoadStatistics< dp_tclSpmDpEngDataStartupTimeStatistics >(factory, SPM_U32_STATISTIC_STARTUP_TIME, SPM_U32_STATISTIC_GLOBAL )
{
}

spm_tclSWVersionCpuUsageStatistics::spm_tclSWVersionCpuUsageStatistics( const ISpmFactory& factory )
   : spm_tclSystemLoadStatistics< dp_tclSpmDpEngDataCurrentSWVersionCpuUsageStatistics >( factory, SPM_U32_STATISTIC_CPU_USAGE, SPM_U32_STATISTIC_SOFTWARE_VERSION )
{
}

spm_tclSWVersionRamUsageStatistics::spm_tclSWVersionRamUsageStatistics( const ISpmFactory& factory )
   : spm_tclSystemLoadStatistics< dp_tclSpmDpEngDataCurrentSWVersionRamUsageStatistics >(factory, SPM_U32_STATISTIC_RAM_USAGE, SPM_U32_STATISTIC_SOFTWARE_VERSION )
{
}

spm_tclSWVersionStartupTimeStatistics::spm_tclSWVersionStartupTimeStatistics( const ISpmFactory& factory )
   : spm_tclSystemLoadStatistics< dp_tclSpmDpEngDataCurrentSWVersionStartupTimeStatistics >(factory, SPM_U32_STATISTIC_STARTUP_TIME, SPM_U32_STATISTIC_SOFTWARE_VERSION )
{
}
