/*!
  * \file spm_SwitchOnStatistics.cpp
  *  \brief
  *    Wakeup statistics of the system
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 09.11.11  | TMS Fischer       | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_STDVISITORS
#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_FUNCTIONIDS
#include "spm_fi_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// -----------------------------------------------------------------------------
// includes FC SPM
// -----------------------------------------------------------------------------
#include "spm_Config.h"
#include "spm_SwitchOnStatistics.h"

#include "spm_IOsalProxy.h"
#include "timeConvert.h"

#include "spm_IFactory.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
 #include "trcGenProj/Header/spm_SwitchOnStatistics.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"


/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/

// //#define SPM_TRACE_FILE_ID   SPM_FILE_WAKEUPSTATISTICS
template < typename T1, typename T2 >
spm_tclSwitchOnStatisticsTemplate<T1, T2>::spm_tclSwitchOnStatisticsTemplate( const ISpmFactory& factory, enStatisticsType eStatType )
   : ISpmStatistics( factory ){
   _eStatType = eStatType;
}

template < typename T1, typename T2 >
spm_tclSwitchOnStatisticsTemplate<T1, T2>::~spm_tclSwitchOnStatisticsTemplate( ){
}

template < typename T1, typename T2 >
tVoid spm_tclSwitchOnStatisticsTemplate<T1, T2>::vSetNewData( tU32 u32StatisticType ) const {
   OSAL_trTimeDate rCurrentTime = { 0,0,0,0,0,0,0,0,0 };

   SPM_GET_IF_REFERENCE_NEW_VAR( poclOsalProxy, ISpmOsalProxy );
   (tVoid)poclOsalProxy->bGetUtcTime( &rCurrentTime );

   rCurrentTime.s32Year += 1900;

   tclTimeConvert                    oTimeConv;
   T1 oHistory;

   tU32                              u32Seconds = oTimeConv.u32GetSeconds( &rCurrentTime );
   TSpmSwitchOnStatisticsData        tSwitchOnElem;
   tSwitchOnElem.u32Time       = u32Seconds;
   tSwitchOnElem.u32WakeUpType = u32StatisticType;
   oHistory.vPushBack( tSwitchOnElem );

   ETG_TRACE_USR4( ( "Detected switch on reason: %d: %04d-%02d-%02d  %02d:%02d:%02d (seconds since 1970: %d)", ETG_ENUM( spmTriggerTypes, u32StatisticType ),
                     rCurrentTime.s32Year, rCurrentTime.s32Month, rCurrentTime.s32Day,
                     rCurrentTime.s32Hour, rCurrentTime.s32Minute, rCurrentTime.s32Second,
                     u32Seconds ) );

   T2 oDevOnCount;
   tU32                            u32Count = oDevOnCount.tGetData( ) + 1;
   oDevOnCount << u32Count;
   switch ( _eStatType )
   {
      case SPM_U32_STATISTIC_GLOBAL:
      {
         SPM_GET_CLASS_REFERENCE_NEW_VAR( poclSWVersionSwitchOnStatistics, spm_tclSWVersionSwitchOnStatistics, ISpmStatistics );
         poclSWVersionSwitchOnStatistics->vSetNewData( u32StatisticType );
         break;
      }
      default:
         break;
   }
} // vSetNewData

template < typename T1, typename T2 >
tVoid spm_tclSwitchOnStatisticsTemplate<T1, T2>::vGetData( void *pStatistics, tU32 u32ObjectSize ) const {
/*!
  * \fn
  *  \brief
  *    Get the SwitchOn statistics from data pool.
  *  \param[out] pStatistics:  pointer to TSpmSwitchOnStatisticsData buffer of SwitchOn statistics from history.
  *  \param[in]  u32ObjectSize: buffer size.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  *  \details  All the exceed buffer data will be set to 0 if the u32ObjectSize > size of SwitchOn statistics from history list.
  ******
  */
   if ( pStatistics != NULL ){
      T1    oSwitchOn;
      tU32  u32Count = oSwitchOn.u32GetCount( );
      std::list< TSpmSwitchOnStatisticsData > tSwitchOnList;
      if ( u32ObjectSize >= u32Count * sizeof( TSpmSwitchOnStatisticsData ) ){
         for ( tUInt i = 0; i < u32Count; ++i ){
            TSpmSwitchOnStatisticsData tSwitchOnElem ={0,0};

            oSwitchOn.s32GetElem( i, tSwitchOnElem );
            tSwitchOnList.push_back( tSwitchOnElem );
         }
         memset( pStatistics, 0, u32ObjectSize );
         std::copy( tSwitchOnList.begin(), tSwitchOnList.end(), ( TSpmSwitchOnStatisticsData * )pStatistics );
      } else {
         ETG_TRACE_FATAL( ("SwitchOnStatistics::vGetData(): buffer size is not valid") );
      }
   } else {
	   ETG_TRACE_ERRMEM( ("SwitchOnStatistics::vGetData() pStatistics buffer is NULL") );
   }
} // vGetData

template < typename T1, typename T2 >
tVoid spm_tclSwitchOnStatisticsTemplate<T1, T2>::vErase( ) const {
   T1 oHistory;

   oHistory.vClearList( );
}

template < typename T1, typename T2 >
tVoid spm_tclSwitchOnStatisticsTemplate<T1, T2>::vTrace( ) const {
   T2   oDevOnCount;

                         ETG_TRACE_FATAL( ( "Device ON overall count :                 %d", oDevOnCount.tGetData( ) ) );

   T1 oHistory;

   tU32                              u32Count = oHistory.u32GetCount( );

   for ( tUInt i = 0; i < u32Count; ++i ){
      TSpmSwitchOnStatisticsData tSwitchOnElem;
      tSwitchOnElem.u32Time = 0;
      //tSwitchOnElem.u32Time = 0;  Removed why need it 2 times.
      tSwitchOnElem.u32WakeUpType =0;
      
      tclTimeConvert             oTimeConv;
      OSAL_trTimeDate            tTimeDate = { 0,0,0,0,0,0,0,0,0 };

      oHistory.s32GetElem( i, tSwitchOnElem );
      oTimeConv.vGetDateFromElapsedSeconds( tSwitchOnElem.u32Time, &tTimeDate );

                         ETG_TRACE_FATAL( ( "switch on reason %2d:                %3d (%04d.%02d.%02d %02d:%02d:%02d (UTC))",
                         i + 1,
                         ETG_ENUM( spmTriggerTypes, tSwitchOnElem.u32WakeUpType ),
                         tTimeDate.s32Year, tTimeDate.s32Month, tTimeDate.s32Day,
                         tTimeDate.s32Hour, tTimeDate.s32Minute, tTimeDate.s32Second
                         ) );
   }

} // vTrace

spm_tclSwitchOnStatistics::spm_tclSwitchOnStatistics( const ISpmFactory& factory )
   : spm_tclSwitchOnStatisticsTemplate< dp_tclSpmDpEngDataSwitchOnHistory, dp_tclSpmDpEngDataDeviceOnCount >( factory, SPM_U32_STATISTIC_GLOBAL )
{
}

spm_tclSWVersionSwitchOnStatistics::spm_tclSWVersionSwitchOnStatistics( const ISpmFactory& factory )
   : spm_tclSwitchOnStatisticsTemplate< dp_tclSpmDpEngDataCurrentSWSwitchOnHistory, dp_tclSpmDpEngDataCurrentSWDeviceOnCount >( factory, SPM_U32_STATISTIC_SOFTWARE_VERSION )
{
}
