/*!
  * \file spm_ResetStatistics.cpp
  *  \brief
  *    Reset statistics of the system.
  *
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011-2018 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 09.11.11  | TMS Fischer       | initial version
  * 02..11    | TMS Fischer       | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

// -----------------------------------------------------------------------------
// includes FC SPM
// -----------------------------------------------------------------------------
// SPM  configuration
#include "spm_Config.h"

#include "spm_ResetStatistics.h"
#include "spm_IOsalProxy.h"
#include "timeConvert.h"
#include "spm_GlobDefs.h"
#include "spm_IWakeupHandler.h"
#include "spm_ISystemPowerManager.h"
#include "spm_ISystemStateManager.h"
#include "spm_ICcaServiceServer.h"
#include "spm_SystemStateStatistics.h"
#include "spm_IRegistry.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_ResetStatistics.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"
#include "spm_IFactory.h"

/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/

// //#define SPM_TRACE_FILE_ID   SPM_FILE_RESETSTATISTICS

template < typename T1, typename T2, typename T3 >
spm_tclResetStatisticsTemplate< T1, T2, T3 >::spm_tclResetStatisticsTemplate( const ISpmFactory& factory, tBool bUpdateProperty, enStatisticsType eStatType )
   : ISpmResetStatistics( factory )
   , _bUpdateProperty( FALSE ){
   _bUpdateProperty =  bUpdateProperty ;
   _eStatType = eStatType;
}
template < typename T1, typename T2, typename T3 >
spm_tclResetStatisticsTemplate< T1, T2, T3 >::~spm_tclResetStatisticsTemplate( ){
}
template < typename T1, typename T2, typename T3 >
tVoid spm_tclResetStatisticsTemplate< T1, T2, T3 >::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *    called during start-up.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
}
template < typename T1, typename T2, typename T3 >
tVoid spm_tclResetStatisticsTemplate< T1, T2, T3 >::vStartLateCommunication( ){
/*!
  * \fn
  *  \brief
  *    called during start-up and the SPM_COREFI_C_U16_RESETCOUNTER status update to registered components.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   if ( _bUpdateProperty ){
      vUpdateOnInterface( );
   }
}
template < typename T1, typename T2, typename T3 >
tVoid spm_tclResetStatisticsTemplate< T1, T2, T3 >::vUpdateOnInterface( ) const {

   TSpmResetStatisticsData             ResetStatisticData;

   T1    oRstData;

   ResetStatisticData = oRstData.tGetData( );

   tclTimeConvert                      oTimeConv;
   OSAL_trTimeDate                     tTimeDate = { 0,0,0,0,0,0,0,0,0 };

   oTimeConv.vGetDateFromElapsedSeconds( ResetStatisticData.tHmiReset.u32LastResetTime, &tTimeDate );

   // update reset reason status information
   spm_corefi_tclMsgResetCounterStatus tResetCount;
   tResetCount.tLastResetDate.u8Hour    = (tU8)tTimeDate.s32Hour;
   tResetCount.tLastResetDate.u8Minute  = (tU8)tTimeDate.s32Minute;
   tResetCount.tLastResetDate.u8Second  = (tU8)tTimeDate.s32Second;
   tResetCount.tLastResetDate.u8Day     = (tU8)tTimeDate.s32Day;
   tResetCount.tLastResetDate.u8Month   = (tU8)tTimeDate.s32Month;
   tResetCount.tLastResetDate.s16Year   = (tS16)tTimeDate.s32Year;

   tResetCount.tLastResetReason.enType  = (spm_fi_tcl_e8_LcmResetReason::tenType)ResetStatisticData.tHmiReset.u32LastResetType;

   tResetCount.u32ApApplicationError    = ResetStatisticData.tHmiReset.u32ApplicationError;
   tResetCount.u32ApSoftware            = ResetStatisticData.tHmiReset.u32Software;
   tResetCount.u32ApEcuReset            = ResetStatisticData.tHmiReset.u32EcuReset;
   tResetCount.u32ApDownload            = ResetStatisticData.tHmiReset.u32Download;
   tResetCount.u32ApDownloadError       = ResetStatisticData.tHmiReset.u32DownloadError;
   tResetCount.u32ApOvertemperature     = ResetStatisticData.tHmiReset.u32Overtemperature;
   tResetCount.u32ApLcmIntern           = ResetStatisticData.tHmiReset.u32LcmIntern;
   tResetCount.u32ApUser                = ResetStatisticData.tHmiReset.u32User;
   tResetCount.u32ApPowerOnCount        = ResetStatisticData.tHmiReset.u32PowerOn;
   tResetCount.u32ApPowerLossCount      = ResetStatisticData.tHmiReset.u32PowerLoss;
   tResetCount.u32ApPowerOffCount       = ResetStatisticData.tHmiReset.u32PowerOff;
   tResetCount.u32ApSccCount            = ResetStatisticData.tHmiReset.u32Scc;
   tResetCount.u32ApUnknownCount        = ResetStatisticData.tHmiReset.u32Unknown;

   tResetCount.u32SccHwWdtCount         = ResetStatisticData.tSccReset.u32HwWdg;
   tResetCount.u32SccPowerOnCount       = ResetStatisticData.tSccReset.u32PowerOn;
   tResetCount.u32SccColdstartCount     = ResetStatisticData.tSccReset.u32Coldstart;
   tResetCount.u32SccAppmodeChangeCount = ResetStatisticData.tSccReset.u32AppmodeChange;
   tResetCount.u32SccDuringLpwCount     = ResetStatisticData.tSccReset.u32DuringLpw;
   tResetCount.u32SccDuringPllOszCount  = ResetStatisticData.tSccReset.u32DuringPllOsz;
   tResetCount.u32SccDuringSWCount      = ResetStatisticData.tSccReset.u32DuringSw;
   tResetCount.u32SccWarmstartCount     = ResetStatisticData.tSccReset.u32Warmstart;

   SPM_GET_IF_REFERENCE_NEW_VAR( poclCcaServiceHandler, ISpmCcaServiceServer );
   poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_RESETCOUNTER, &tResetCount );
} // vUpdateOnInterface
template < typename T1, typename T2, typename T3 >
tVoid spm_tclResetStatisticsTemplate< T1, T2, T3 >::vSetNewData( tU32 u32StatisticType ) const {
/*!
  * \fn
  *  \brief
  *    update the reset information, reset history, reset state statistics to data pool and update status of SPM_COREFI_C_U16_RESETCOUNTER property.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   tBool                   bAddEmTraceEntry = FALSE;

   TSpmResetStatisticsData ResetStatisticData;

   OSAL_trTimeDate         rCurrentTime     = { 0,0,0,0,0,0,0,0,0 };

   SPM_GET_IF_REFERENCE_NEW_VAR( poOsalProxy, ISpmOsalProxy );
   (tVoid)poOsalProxy->bGetUtcTime( &rCurrentTime );
   rCurrentTime.s32Year += 1900;

   T1 oRstData;
   ResetStatisticData    = oRstData.tGetData( );

   ETG_TRACE_USR4( ( "spm_tclResetStatisticsTemplate::vSetNewData(): detected reset reason %u", ETG_ENUM( CFC_FI_LCMRESETREASON, u32StatisticType ) ) );

   {
      tU32           u32TypeCount      = 0;

      tU32           u32SystemState    = 0;
      tU32           u32NewSystemState = 0;
      SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemStateManager, ISpmSystemStateManager );
      u32SystemState    = poclSystemStateManager->u32GetSystemState( );
      u32NewSystemState = poclSystemStateManager->u32GetNewSystemState( );

      tclTimeConvert oTimeConv;
      tU32           u32Seconds = oTimeConv.u32GetSeconds( &rCurrentTime );
      SPM_GET_IF_REFERENCE_NEW_VAR( poclRegistry, ISpmRegistry );
      std::string    strBuildVersion;
      std::string    strCustomerVersion;
      if ( false == poclRegistry->bGetStringFromRegistry( "VERSIONS", "BUILDVERSION_LABEL", strBuildVersion ) ){
         ETG_TRACE_ERR( ( "BuildVersion not available in registry" ) );
      } else {
         ETG_TRACE_USR1( ( "BuildVersion=%s", strBuildVersion.c_str( ) ) );
      }
      if ( false == poclRegistry->bGetStringFromRegistry( "VERSIONS", "BUILDVERSION_CUSTVERSTRING", strCustomerVersion ) ){
         ETG_TRACE_ERR( ( "CustomerVersion not available in registry" ) );
      } else {
         ETG_TRACE_USR1( ( "CustomerVersion=%s", strCustomerVersion.c_str( ) ) );
      }
      if ( ( u32StatisticType != spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_WARMSTART )
           && ( u32StatisticType != spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_WARMSTART ) ){

         T2 oHistory;
         TSpmResetElem                  tRstElem = {
            u32Seconds,
            (tU16)u32StatisticType,
            { 0 },
            { 0 }
         };
         OSAL_szStringNCopy( tRstElem.strBuildVersion,    strBuildVersion.c_str( ),    SPM_RESET_STATISTIC_MAX_LENGTH_SW_VERSION - 1 );
         OSAL_szStringNCopy( tRstElem.strCustomerVersion, strCustomerVersion.c_str( ), SPM_RESET_STATISTIC_MAX_LENGTH_SW_VERSION - 1 );
         oHistory.vPushBack( tRstElem );
      }

      TSpmSystemStateStatisticsData oStateElem;

      oStateElem.u32Time       = u32Seconds;
      oStateElem.u32TimeMs     = OSAL_ClockGetElapsedTime( );
      oStateElem.u8SystemState = (tU8)SPM_SYSTEM_INVALID;
      oStateElem.u8EntryType   = SPM_STATE_HISTORY_ENTRY_RESET;
      oStateElem.u.u8Reset     = (tU8)u32StatisticType;

      T3 oHisState;
      oHisState.vPushBack( oStateElem );

      if ( u32StatisticType < spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_WARMSTART ){
         ResetStatisticData.tHmiReset.u32ResetCount++;
         ResetStatisticData.tHmiReset.u32LastResetType = u32StatisticType;
         ResetStatisticData.tHmiReset.u32LastResetTime = u32Seconds;
         bAddEmTraceEntry                              = TRUE;
      } else if ( u32StatisticType > spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_WARMSTART ){
         ResetStatisticData.tSccReset.u32ResetCount++;
         ResetStatisticData.tSccReset.u32LastResetType = u32StatisticType;
         ResetStatisticData.tSccReset.u32LastResetTime = u32Seconds;
      }

      switch ( u32StatisticType ){
         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_UNKNOWN:
            ResetStatisticData.tHmiReset.u32Unknown++;
            u32TypeCount                           = ResetStatisticData.tHmiReset.u32Unknown;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_USER:
            ResetStatisticData._bMyOwnResetTrigger = TRUE;
            ResetStatisticData.tHmiReset.u32User++;
            u32TypeCount                           = ResetStatisticData.tHmiReset.u32User;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_SOFTWARE:
            ResetStatisticData.tHmiReset.u32Software++;
            u32TypeCount                           = ResetStatisticData.tHmiReset.u32Software;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_APP_ERROR:
            ResetStatisticData._bMyOwnResetTrigger = TRUE;
            ResetStatisticData.tHmiReset.u32ApplicationError++;
            u32TypeCount                           = ResetStatisticData.tHmiReset.u32ApplicationError;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_DOWNLOAD_ERROR:
            ResetStatisticData._bMyOwnResetTrigger = TRUE;
            ResetStatisticData.tHmiReset.u32DownloadError++;
            u32TypeCount                           = ResetStatisticData.tHmiReset.u32DownloadError;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_DOWNLOAD:
            ResetStatisticData._bMyOwnResetTrigger = TRUE;
            ResetStatisticData.tHmiReset.u32Download++;
            u32TypeCount                           = ResetStatisticData.tHmiReset.u32Download;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_ECU:
            ResetStatisticData._bMyOwnResetTrigger = TRUE;
            ResetStatisticData.tHmiReset.u32EcuReset++;
            u32TypeCount                           = ResetStatisticData.tHmiReset.u32EcuReset;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_LCM_INTERN:
            ResetStatisticData._bMyOwnResetTrigger = TRUE;
            ResetStatisticData.tHmiReset.u32LcmIntern++;
            u32TypeCount                           = ResetStatisticData.tHmiReset.u32LcmIntern;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_OVERTEMPERATURE:
            ResetStatisticData.tHmiReset.u32Overtemperature++;
            u32TypeCount                           = ResetStatisticData.tHmiReset.u32Overtemperature;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_POWER_ON_RESET:
            ResetStatisticData.tHmiReset.u32PowerOn++;
            u32TypeCount                           = ResetStatisticData.tHmiReset.u32PowerOn;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_POWER_LOSS:
            ResetStatisticData.tHmiReset.u32PowerLoss++;
            u32TypeCount                           = ResetStatisticData.tHmiReset.u32PowerLoss;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_POWER_OFF:
            ResetStatisticData.tHmiReset.u32PowerOff++;
            u32TypeCount                           = ResetStatisticData.tHmiReset.u32PowerOff;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_SCC:
            ResetStatisticData.tHmiReset.u32Scc++;
            u32TypeCount                           = ResetStatisticData.tHmiReset.u32Scc;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_HW_WATCHDOG:
            ResetStatisticData.tSccReset.u32HwWdg++;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_POWER_ON:
            ResetStatisticData.tSccReset.u32PowerOn++;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_COLDSTART:
            ResetStatisticData.tSccReset.u32Coldstart++;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_APPMODE_CHANGE:
            ResetStatisticData.tSccReset.u32AppmodeChange++;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_DURING_LPW:
            ResetStatisticData.tSccReset.u32DuringLpw++;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_DURING_PLL_OSZ:
            ResetStatisticData.tSccReset.u32DuringPllOsz++;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_DURING_SW:
            ResetStatisticData.tSccReset.u32DuringSw++;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_USER:
            ResetStatisticData.tSccReset.u32User++;
            break;

         case spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_WARMSTART:
            ResetStatisticData.tSccReset.u32Warmstart++;
            break;

         default:
            ResetStatisticData.tHmiReset.u32Unknown++;
            u32TypeCount = ResetStatisticData.tHmiReset.u32Unknown;
            break;

      } // switch

      if ( bAddEmTraceEntry ){  // and now write emtrace entry

         std::string strResetCounterBuf;

         strResetCounterBuf.insert( 0, 1, (tChar)u32StatisticType );
         SPM_M_INSERT_STRING_AT_INDEX_T32( strResetCounterBuf, 1, u32TypeCount - 1 )
         SPM_M_INSERT_STRING_AT_INDEX_T32( strResetCounterBuf, 5, u32TypeCount )

         std::string strSystemStatesBuf;

         strSystemStatesBuf.insert( 0, 1, (tChar)u32StatisticType );
         SPM_M_INSERT_STRING_AT_INDEX_T32( strSystemStatesBuf, 1, u32SystemState )
         SPM_M_INSERT_STRING_AT_INDEX_T32( strSystemStatesBuf, 5, u32NewSystemState )

         SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemPowerManager, ISpmSystemPowerManager );
         std::string strDummy;
         poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_RESET_TYPE_COUNTER ), (const tU8*)strResetCounterBuf.c_str( ), (tU16)strResetCounterBuf.length( ), FALSE );
         poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STATE ), (const tU8*)strSystemStatesBuf.c_str( ), (tU16)strSystemStatesBuf.length( ), SPM_WRITE_DIRECTLY );
         poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (const tU8*)strDummy.c_str( ), (tU16)strDummy.length( ), FALSE );
         poclSystemPowerManager->vWriteProjectSpecificShutdownInfoToErrMem( u32StatisticType );
      }
   }

   oRstData.vSetData( ResetStatisticData );
   if ( _bUpdateProperty ){
      vUpdateOnInterface( );
   }

   switch ( _eStatType )
   {
      case SPM_U32_STATISTIC_GLOBAL:
      {
         SPM_GET_CLASS_REFERENCE_NEW_VAR( poclSWVersionResetStatistics, spm_tclSWVersionResetStatistics, ISpmResetStatistics );
         poclSWVersionResetStatistics->vSetNewData( u32StatisticType );
         break;
      }
      default:
         break;
   }
} // vSetNewData

template < typename T1, typename T2, typename T3 >
tVoid spm_tclResetStatisticsTemplate< T1, T2, T3 >::vGetData( void *pStatistics,
                                        tU32  u32ObjectSize ) const {
/*!
  * \fn
  *  \brief
  *    get the reset information.
  *  \param[out] pStatistics:  pointer to TSpmResetStatisticsData buffer of reset information.
  *  \param[in]  u32ObjectSize: buffer size.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   if ( pStatistics != NULL ){
      T1 oRstData;
      TSpmResetStatisticsData          tRstTmp = oRstData.tGetData( );
      if ( u32ObjectSize >= sizeof( TSpmResetStatisticsData ) ){
         memcpy( pStatistics, &tRstTmp, sizeof( TSpmResetStatisticsData ) );
      }
   }
}

template < typename T1, typename T2, typename T3 >
tVoid spm_tclResetStatisticsTemplate< T1, T2, T3 >::vGetData( TSpmResetElem *pStatistics,
                                        tU32  u32ObjectSize ) const {
/*!
  * \fn
  *  \brief
  *    get the reset history.
  *  \param[out] pStatistics:  pointer to TSpmResetElem buffer of reset history.
  *  \param[in]  u32ObjectSize: buffer size.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  *  \details  All the exceed buffer data will be set to 0 if the u32ObjectSize > size of reset history list.
  ******
  */
   if ( pStatistics != NULL ){
      T2    oHistory;
      tU32  u32Count = oHistory.u32GetCount( );
      std::list< TSpmResetElem > tResetElemList;
      if ( u32ObjectSize >= u32Count * sizeof( TSpmResetElem ) ){
         for ( tUInt i = 0; i < u32Count; ++i ){
            TSpmResetElem tResetElem ={0, 0, { 0 }, { 0 }};

            oHistory.s32GetElem( i, tResetElem );
            tResetElemList.push_back( tResetElem );
         }
         memset( pStatistics, 0, u32ObjectSize );
         std::copy( tResetElemList.begin(), tResetElemList.end(), ( TSpmResetElem * )pStatistics );
      } else {
         ETG_TRACE_FATAL( ("ResetStatistics::vGetData(): buffer size of SpmResetElem is not valid") );
      }
   } else {
	   ETG_TRACE_ERRMEM( ("ResetStatistics::vGetData() pStatistics buffer of SpmResetElem is NULL") );
   }
}

template < typename T1, typename T2, typename T3 >
tVoid spm_tclResetStatisticsTemplate< T1, T2, T3 >::vErase( ) const {
/*!
  * \fn
  *  \brief
  *    Clear the reset data information and reset history in data pool.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   TSpmResetStatisticsData          ResetStatisticData = TSpmResetStatisticsData( );
   T1 oRstData;

   oRstData.vSetData( ResetStatisticData );

   T2   oHistory;
   oHistory.vClearList( );
   if ( _bUpdateProperty ){
      vUpdateOnInterface( );
   }
}
template < typename T1, typename T2, typename T3 >
tVoid spm_tclResetStatisticsTemplate< T1, T2, T3 >::vTrace( ) const {
/*!
  * \fn
  *  \brief
  *    Trace info of reset information, reset history.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   TSpmResetStatisticsData          ResetStatisticData;

   T1 oRstData;

   ResetStatisticData = oRstData.tGetData( );

   T2   oHistory;

                         ETG_TRACE_FATAL( ( "Reset History:" ) );

   tU32                             u32Count = oHistory.u32GetCount( );
   for ( tUInt i = 0; i < u32Count; ++i ){
      TSpmResetElem   tRstElem  = {
         0, 0, { 0 }, { 0 }
      };
      tclTimeConvert  oTimeConv;
      OSAL_trTimeDate tTimeDate = { 0,0,0,0,0,0,0,0,0 };

      oHistory.s32GetElem( u32Count - i - 1, tRstElem );
      oTimeConv.vGetDateFromElapsedSeconds( tRstElem.u32Time, &tTimeDate );

                         ETG_TRACE_FATAL( ( "Reset reason %2d:                %3d (%04d-%02d-%02d  %02d:%02d:%02d(UTC))",
                         i + 1,
                         ETG_ENUM( CFC_FI_LCMRESETREASON, tRstElem.u16ResetType ),
                         tTimeDate.s32Year, tTimeDate.s32Month, tTimeDate.s32Day,
                         tTimeDate.s32Hour, tTimeDate.s32Minute, tTimeDate.s32Second ) );
   }
   ETG_TRACE_FATAL( ( "---------------------------------------------------------------------------" ) );
   ETG_TRACE_FATAL( ( "Reset counter ApplicationController (i.mx6, ..)" ) );
   ETG_TRACE_FATAL( ( "---------------------------------------------------------------------------" ) );
   ETG_TRACE_FATAL( ( "Reset count:                     %d", ResetStatisticData.tHmiReset.u32ResetCount ) );
   ETG_TRACE_FATAL( ( "---------------------------------------------------------------------------" ) );

   ETG_TRACE_FATAL( ( "Application error count:         %d", ResetStatisticData.tHmiReset.u32ApplicationError ) );
   ETG_TRACE_FATAL( ( "Software count:                  %d", ResetStatisticData.tHmiReset.u32Software ) );
   ETG_TRACE_FATAL( ( "ECU Reset count:                 %d", ResetStatisticData.tHmiReset.u32EcuReset ) );
   ETG_TRACE_FATAL( ( "AppDownload count:               %d", ResetStatisticData.tHmiReset.u32Download ) );
   ETG_TRACE_FATAL( ( "AppDownloadError count:          %d", ResetStatisticData.tHmiReset.u32DownloadError ) );
   ETG_TRACE_FATAL( ( "Overtemperature count:           %d", ResetStatisticData.tHmiReset.u32Overtemperature ) );
   ETG_TRACE_FATAL( ( "LCM internal reset count:        %d", ResetStatisticData.tHmiReset.u32LcmIntern ) );
   ETG_TRACE_FATAL( ( "User reset count:                %d", ResetStatisticData.tHmiReset.u32User ) );
   ETG_TRACE_FATAL( ( "Unknown reset count:             %d", ResetStatisticData.tHmiReset.u32Unknown ) );
   ETG_TRACE_FATAL( ( "Software reset count:            %d", ResetStatisticData.tHmiReset.u32Software ) );
   ETG_TRACE_FATAL( ( "Power On Reset count:            %d", ResetStatisticData.tHmiReset.u32PowerOn ) );
   ETG_TRACE_FATAL( ( "Power Loss Reset count:          %d", ResetStatisticData.tHmiReset.u32PowerLoss ) );
   ETG_TRACE_FATAL( ( "Power Off Reset count:           %d", ResetStatisticData.tHmiReset.u32PowerOff ) );
   ETG_TRACE_FATAL( ( "Reset by SCC count:              %d", ResetStatisticData.tHmiReset.u32Scc ) );

   ETG_TRACE_FATAL( ( "---------------------------------------------------------------------------" ) );
   ETG_TRACE_FATAL( ( "Reset counter System Communication Controller (V850, ..)" ) );
   ETG_TRACE_FATAL( ( "---------------------------------------------------------------------------" ) );
   ETG_TRACE_FATAL( ( "Reset count:                     %d", ResetStatisticData.tSccReset.u32ResetCount ) );
   ETG_TRACE_FATAL( ( "---------------------------------------------------------------------------" ) );
   ETG_TRACE_FATAL( ( "PowerOn count:                   %d", ResetStatisticData.tSccReset.u32PowerOn ) );
   ETG_TRACE_FATAL( ( "HW watchdog count:               %d", ResetStatisticData.tSccReset.u32HwWdg ) );
   ETG_TRACE_FATAL( ( "Coldstart count:                 %d", ResetStatisticData.tSccReset.u32Coldstart ) );
   ETG_TRACE_FATAL( ( "Application mode change count:   %d", ResetStatisticData.tSccReset.u32AppmodeChange ) );
   ETG_TRACE_FATAL( ( "During LPW count:                %d", ResetStatisticData.tSccReset.u32DuringLpw ) );
   ETG_TRACE_FATAL( ( "During PLL OSZ count:            %d", ResetStatisticData.tSccReset.u32DuringPllOsz ) );
   ETG_TRACE_FATAL( ( "During SW count:                 %d", ResetStatisticData.tSccReset.u32DuringSw ) );
   ETG_TRACE_FATAL( ( "User reset count:                %d", ResetStatisticData.tSccReset.u32User ) );
   ETG_TRACE_FATAL( ( "Warmstart count:                 %d", ResetStatisticData.tSccReset.u32Warmstart ) );
   ETG_TRACE_FATAL( ( "---------------------------------------------------------------------------" ) );

   #ifdef SPM_FEATURE_ENABLE_ADR3
      {
         tU32               u32ResetCount   = 0;
         OSAL_tIODescriptor hAdr3CtrlDevice = OSAL_IOOpen( "/dev/adr3ctrl", OSAL_EN_READWRITE );
         if ( OSAL_ERROR != hAdr3CtrlDevice ){
            if ( OSAL_s32IOControl( hAdr3CtrlDevice, OSAL_C_S32_IOCTRL_ADR3CTRL_READ_CLEAR_RESET_COUNT, ( intptr_t )&u32ResetCount ) == OSAL_ERROR ){
               ETG_TRACE_ERR( ( "SPM: Get ADR reset count failed." ) );
            }
            OSAL_s32IOClose( hAdr3CtrlDevice );
         }
         ResetStatisticData.tAdrReset._u32Reset += u32ResetCount;
         oRstData.vSetData( ResetStatisticData );
      }

      ETG_TRACE_FATAL( ( "---------------------------------------------------------------------------" ) );
      ETG_TRACE_FATAL( ( "Reset counter ADR" ) );
      ETG_TRACE_FATAL( ( "ADR reset count:                 %d", ResetStatisticData.tAdrReset._u32Reset ) );
      ETG_TRACE_FATAL( ( "---------------------------------------------------------------------------" ) );
   #endif // ifdef SPM_FEATURE_ENABLE_ADR3

} // vTrace
template < typename T1, typename T2, typename T3 >
tVoid spm_tclResetStatisticsTemplate< T1, T2, T3 >::vUpdateResetCounter( tVoid ) const {
   #ifdef SPM_FEATURE_ENABLE_ADR3

      TSpmResetStatisticsData          ResetStatisticData;
      T1 oRstData;

      tU32                             u32ResetCount   = 0;
      OSAL_tIODescriptor               hAdr3CtrlDevice = OSAL_IOOpen( "/dev/adr3ctrl", OSAL_EN_READWRITE );
      if ( OSAL_ERROR != hAdr3CtrlDevice ){
         if ( OSAL_s32IOControl( hAdr3CtrlDevice, OSAL_C_S32_IOCTRL_ADR3CTRL_READ_CLEAR_RESET_COUNT, ( intptr_t )&u32ResetCount ) == OSAL_ERROR ){
            ETG_TRACE_ERR( ( "SPM: Get ADR reset count failed." ) );
         }
         OSAL_s32IOClose( hAdr3CtrlDevice );
      }

      ResetStatisticData                      = oRstData.tGetData( );
      ResetStatisticData.tAdrReset._u32Reset += u32ResetCount;
      oRstData.vSetData( ResetStatisticData );

      switch ( _eStatType )
      {
         case SPM_U32_STATISTIC_GLOBAL:
         {
            SPM_GET_CLASS_REFERENCE_NEW_VAR( poclSWVersionResetStatistics, spm_tclSWVersionResetStatistics, ISpmResetStatistics );
            poclSWVersionResetStatistics->vUpdateResetCounter( );
            break;
         }
         default:
            break;
      }
   #endif // ifdef SPM_FEATURE_ENABLE_ADR3
} // vUpdateResetCounter
template < typename T1, typename T2, typename T3 >
tVoid spm_tclResetStatisticsTemplate< T1, T2, T3 >::vGetLastReset( TSpmResetElem *pLastResetData ) const {
   if ( pLastResetData != NULL ){
      T2 oHistory;
      tU32                           u32Count = oHistory.u32GetCount( );
      if ( u32Count > 0 ){
         oHistory.s32GetElem( u32Count - 1, * pLastResetData );
      }
   }
}
template < typename T1, typename T2, typename T3 >
tBool spm_tclResetStatisticsTemplate< T1, T2, T3 >::bChkUnknownReset( tU32 u32StatisticType ) const {
   if ( ( spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_SOFTWARE == u32StatisticType )
        || ( spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_POWER_ON_RESET == u32StatisticType )
        ){
      return( TRUE );
   } else {
      return( FALSE );
   }
}

spm_tclResetStatistics::spm_tclResetStatistics( const ISpmFactory& factory )
   : spm_tclResetStatisticsTemplate< dp_tclSpmDpEngDataResetStatistic, dp_tclSpmDpEngDataResetHistory, dp_tclSpmDpEngStateHistory >( factory, TRUE, SPM_U32_STATISTIC_GLOBAL )
{
}

spm_tclSWVersionResetStatistics::spm_tclSWVersionResetStatistics( const ISpmFactory& factory )
   : spm_tclResetStatisticsTemplate< dp_tclSpmDpEngDataCurrentSWVersionResetStatistic, dp_tclSpmDpEngDataCurrentSWVersionResetHistory, dp_tclSpmDpEngStateCurrentSWVersionHistory >( factory, FALSE, SPM_U32_STATISTIC_SOFTWARE_VERSION )
{
}
