/**
  * @swcomponent   Life Cycle Management
  * @{
  * @file          spm_PowerSupplyStatistics.cpp
  * @PROJECT:      CMD project
  * @brief         implementation of spm_tclPowerSupplyStatistics class
  *
  * ----------------------------------------------------------------------------
  * @copyright     (C) 2018 Robert Bosch GmbH.
  *                The reproduction, distribution and utilization of this file as well as the
  *                communication of its contents to others without express authorization is prohibited.
  *                Offenders will be held liable for the payment of damages.
  *                All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_STDVISITORS
#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_FUNCTIONIDS
#include "spm_fi_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// -----------------------------------------------------------------------------
// includes FC SPM
// -----------------------------------------------------------------------------
#include "spm_Config.h"
#include "spm_PowerSupplyStatistics.h"

#include "spm_IFactory.h"

#include <cstring>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
 #include "trcGenProj/Header/spm_PowerSupplyStatistics.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/

#define MILLI_SECOND                   (tU32)1000

spm_tclPowerSupplyStatistics::spm_tclPowerSupplyStatistics( const ISpmFactory& factory )
   : ISpmPowerSupplyStatistics( factory )
   , _vecVoltageHistogram( SPM_U32_STATISTIC_VOLT_RANGE_HISTORAM_LENGTH + 1 )
   , _u32LastUpdateVoltageCall( 0 ){
}

spm_tclPowerSupplyStatistics::~spm_tclPowerSupplyStatistics( ){
}

tVoid spm_tclPowerSupplyStatistics::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *    called during start-up. Load the voltage histogram data from data pool. 
  *    Init the voltage value of histogram in the first time.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   dp_tclSpmDpEngDataVoltageRangeHistogram oVoltageRangeHistogram;

   for ( tUInt i = 0; i <= SPM_U32_STATISTIC_VOLT_RANGE_HISTORAM_LENGTH; ++i ){
      TSpmVoltageHistogramRangeStatisticsData v = {0, 0, 0};
      TSpmVoltageHistogramRangeStatisticsData tVoltageRangeDataPool = {0, 0, 0};
      v.u16VoltageValue = tU16 ( i * SPM_U32_STATISTIC_VOLTAGE_RANGE_RESOLUTION );
      v.u32OperationTime      = 0;
      v.u32OperationTimeMs    = 0;

      oVoltageRangeHistogram.s32GetElem( i, tVoltageRangeDataPool );
      if ( v.u16VoltageValue != tVoltageRangeDataPool.u16VoltageValue ){
         oVoltageRangeHistogram.vPushBack( v );
         _vecVoltageHistogram[i] = v;
      } else {
         _vecVoltageHistogram[i] = tVoltageRangeDataPool;
      }
   }
} // vStartCommunication

tVoid spm_tclPowerSupplyStatistics::vUpdateVoltageHistogram( tU16 tVoltageValue ){
/*!
  * \fn
  *  \brief
  *    Update the operation time correspond to voltage value in voltage histogram.
  *    This is called in every second by SuperVisionManager.
  *
  *  \param [in] tU16 tVoltageValue: voltage value (mv).
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   tU32 u32VoltageRangeIndex = ( tVoltageValue - ( tVoltageValue % SPM_U32_STATISTIC_VOLTAGE_RANGE_RESOLUTION ) ) / SPM_U32_STATISTIC_VOLTAGE_RANGE_RESOLUTION;
   tU32 u32CurrentTime  = OSAL_ClockGetElapsedTime( );
   tU32 u32OperationTimeMsToSec = 0;
   if ( u32VoltageRangeIndex  <= SPM_U32_STATISTIC_VOLT_RANGE_HISTORAM_LENGTH ) {
      TSpmVoltageHistogramRangeStatisticsData v;
      v = _vecVoltageHistogram[u32VoltageRangeIndex];
      v.u32OperationTimeMs += ( ( u32CurrentTime - _u32LastUpdateVoltageCall ) % MILLI_SECOND );
      if ( v.u32OperationTimeMs >= MILLI_SECOND ) {
         u32OperationTimeMsToSec++;
         v.u32OperationTimeMs = ( ( v.u32OperationTimeMs - MILLI_SECOND ) % MILLI_SECOND );
      }
      v.u32OperationTime += ( ( u32CurrentTime - _u32LastUpdateVoltageCall ) / MILLI_SECOND ) + u32OperationTimeMsToSec;
      _u32LastUpdateVoltageCall = u32CurrentTime;
      _vecVoltageHistogram[u32VoltageRangeIndex] = v;
   } else {
	   ETG_TRACE_ERRMEM( ( "spm_tclPowerSupplyStatistics::vUpdateVoltageHistogram can not get the index %u", u32VoltageRangeIndex ) );
   }
}

tVoid spm_tclPowerSupplyStatistics::vSetNewData( tU32 u32StatisticType ) const {
/*!
  * \fn
  *  \brief
  *    write voltage histogram to data pool.
  *
  *  \param [in] tU32 u32StatisticType: type of statistic data.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   switch ( u32StatisticType ){
      case SPM_U32_STATISTIC_VOLTAGE_RANGE_HISTOGRAM:
      {
         dp_tclSpmDpEngDataVoltageRangeHistogram oVoltageRangeHistogram;

         for ( tUInt i = 0; i <= SPM_U32_STATISTIC_VOLT_RANGE_HISTORAM_LENGTH; ++i ){
            TSpmVoltageHistogramRangeStatisticsData v;
            v = _vecVoltageHistogram[i];
            oVoltageRangeHistogram.vPushBack( v );
         }
         break;
      }
      default:
         ETG_TRACE_FATAL( ("spm_tclPowerSupplyStatistics::vSetNewData(): u32StatisticType is not valid") );
         break;
   } // switch
} // vSetNewData

tVoid spm_tclPowerSupplyStatistics::vGetData( TSpmVoltageHistogramRangeStatisticsData *pStatistics,
                                              tU32  u32ObjectSize ) const {
/*!
  * \fn
  *  \brief
  *    Update the operation time in the voltage histogram.
  *
  *  \param [out] TSpmVoltageHistogramRangeStatisticsData *pStatistics: buffer for getting the voltage histogram.
  *  \param [in] tU32  u32ObjectSize: size of buffer.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   if ( pStatistics != NULL ){
      dp_tclSpmDpEngDataVoltageRangeHistogram oVoltageRangeHistogram;

      if ( u32ObjectSize >= _vecVoltageHistogram.size( ) * sizeof( TSpmVoltageHistogramRangeStatisticsData ) ){
         std::copy( _vecVoltageHistogram.begin(), _vecVoltageHistogram.end(), pStatistics );
      } else {
         ETG_TRACE_FATAL( ("spm_tclPowerSupplyStatistics::vGetData(): buffer size is not valid") );
      }
   } else {
	   ETG_TRACE_ERRMEM( ("spm_tclPowerSupplyStatistics::vGetData() pStatistics buffer is NULL") );
   }
} // vGetData

tVoid spm_tclPowerSupplyStatistics::vEraseVoltageHistogram( ) {
/*!
  * \fn
  *  \brief
  *    Erase operation time in voltage histogram data pool.
  *
  *  \param
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
  // clear operation time of private class voltage histogram
   std::vector < TSpmVoltageHistogramRangeStatisticsData >::iterator itVoltageRange;
   for ( itVoltageRange = _vecVoltageHistogram.begin( ); itVoltageRange != _vecVoltageHistogram.end( ); ++itVoltageRange ){
     ( * itVoltageRange ).u32OperationTime = 0;
     ( * itVoltageRange ).u32OperationTimeMs = 0;
   }

  // clear operation time of data pool
   dp_tclSpmDpEngDataVoltageRangeHistogram oVoltageRangeHistogram;
   for ( tUInt i = 0; i <= SPM_U32_STATISTIC_VOLT_RANGE_HISTORAM_LENGTH; ++i ){
      TSpmVoltageHistogramRangeStatisticsData v = {0,0,0};
      v.u16VoltageValue = _vecVoltageHistogram[i].u16VoltageValue;
      oVoltageRangeHistogram.vPushBack( v );
   }
} // vEraseVoltageHistogram

tVoid spm_tclPowerSupplyStatistics::vTrace( ) const {
/*!
  * \fn
  *  \brief
  *    Trace of the voltage histogram.
  *
  *  \param
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   ETG_TRACE_FATAL( ( "---------------------------------------------------------------------------" ) );
   for ( tUInt i = 0; i <= SPM_U32_STATISTIC_VOLT_RANGE_HISTORAM_LENGTH; ++i ){
      TSpmVoltageHistogramRangeStatisticsData v;
      v = _vecVoltageHistogram[i];
      ETG_TRACE_FATAL( ( "Voltage Value:  %u mV",     v.u16VoltageValue ) );
      ETG_TRACE_FATAL( ( "Operation Time: %u sec",    v.u32OperationTime ) );
      ETG_TRACE_FATAL( ( "Operation Time: %u msec",   v.u32OperationTimeMs ) );
   }
   ETG_TRACE_FATAL( ( "---------------------------------------------------------------------------" ) );
} // vTrace

