/*!
  * \file spm_PowerOnCount.cpp
  *  \brief
  *    Power on counter of the system
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 07.11.11  | TMS Fischer       | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

// -----------------------------------------------------------------------------
// includes FC SPM
// -----------------------------------------------------------------------------
// SPM  configuration
#include "spm_Config.h"
#include "spm_PowerOnCount.h"
#include "spm_IFactory.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
 #include "trcGenProj/Header/spm_PowerOnCount.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/

// //#define SPM_TRACE_FILE_ID   SPM_FILE_POWERONCOUNT
template < typename T1, typename T2 >
spm_tclPowerOnCountTemplate<T1, T2>::spm_tclPowerOnCountTemplate( const ISpmFactory& factory, enStatisticsType eStatType )
   : ISpmPowerOnStatistics( factory )
   , _u32PwrOnTimeAtStartup( 0 )
   , _u32CurrentPwrOnTime( 0 )
   , _u32PwrOnStoredTime( 0 )
   , _u32SystemTimeSetOffset( 0 )
   , _bPwrTimeValid( FALSE )
   , _pclsPowerOnStatistics( NULL ){
   _eStatType = eStatType;
}

template < typename T1, typename T2 >
spm_tclPowerOnCountTemplate<T1, T2>::~spm_tclPowerOnCountTemplate( ){
   _pclsPowerOnStatistics = NULL;
}

template < typename T1, typename T2 >
tVoid spm_tclPowerOnCountTemplate<T1, T2>::vGetReferences( ){
   switch ( _eStatType )
   {
      case SPM_U32_STATISTIC_GLOBAL:
      {
         SPM_GET_CLASS_REFERENCE_USE_VAR( _pclsPowerOnStatistics, spm_tclSWVersionPowerOnCount, ISpmPowerOnStatistics );
         break;
      }
      default:
         break;
   }
}

template < typename T1, typename T2 >
tVoid spm_tclPowerOnCountTemplate<T1, T2>::vCheckPwrOnTime( ){
   if ( _bPwrTimeValid ){
      _u32CurrentPwrOnTime = _u32PwrOnTimeAtStartup + ( ( OSAL_ClockGetElapsedTime( ) / 1000 ) - _u32SystemTimeSetOffset ); // all times are in s

      T1 oPwrOn;
      oPwrOn.vSetData( _u32CurrentPwrOnTime );
   }
   if ( _pclsPowerOnStatistics != NULL ){
      _pclsPowerOnStatistics->vCheckPwrOnTime( );
   }
}

template < typename T1, typename T2 >
tVoid spm_tclPowerOnCountTemplate<T1, T2>::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *    called during start-up. Sets the current power OnTime to the one after last shutdown.
  *    This time is potentially needed to show in the additional reset info (see szAsString())
  *
  *  \param none
  ******
  */
   T1 oPwrOn;

   _u32PwrOnTimeAtStartup = oPwrOn.tGetData( );
   _u32CurrentPwrOnTime   = _u32PwrOnTimeAtStartup;

   _bPwrTimeValid         = TRUE;
}

template < typename T1, typename T2 >
tVoid spm_tclPowerOnCountTemplate<T1, T2>::vStartLateCommunication( ){
/*!
  * \fn
  *  \brief
  *    Called in late start-up phase and sets the first real power on time of this life cycle.
  *
  *  \param none
  ******
  */
   vCheckPwrOnTime( );
}

template < typename T1, typename T2 >
tVoid spm_tclPowerOnCountTemplate<T1, T2>::vSetPwrOnTime( tU32 u32PwrOnTime ){
   if ( _bPwrTimeValid ){
      _u32SystemTimeSetOffset = OSAL_ClockGetElapsedTime( ) / 1000;

      _u32PwrOnTimeAtStartup  = u32PwrOnTime;
      _u32PwrOnStoredTime     = u32PwrOnTime;
      _u32CurrentPwrOnTime    = u32PwrOnTime;

      T1 oPwrOn;
      oPwrOn.vSetData( _u32CurrentPwrOnTime );
   }
}


template < typename T1, typename T2 >
tVoid spm_tclPowerOnCountTemplate<T1, T2>::vSetPwrOnLastSystemRunTime( tU32 u32LastSystemRunTime ){
/*!
  * \fn
  *  \brief
  *    Update the last system runtime to the datapool
  *
  *  \param none
  ******
  */
   T2 oLastSystemRunTime;
   oLastSystemRunTime << u32LastSystemRunTime;
   if ( _pclsPowerOnStatistics != NULL ){
      _pclsPowerOnStatistics->vSetPwrOnLastSystemRunTime( u32LastSystemRunTime );
   }
}

/*!
  * \fn
  *  \brief
  *    traces out all time stamps
  *
  *  \param none
  ******
  */
template < typename T1, typename T2 >
tVoid spm_tclPowerOnCountTemplate<T1, T2>::vTrace( ) const {
   {
      ETG_TRACE_FATAL( ( "Device Power On Time Count:      %s", szAsString( ).c_str( ) ) );
   }

   {
      tU32                              u32Ticks = 0;
      T2                                oSysOnTime;
      oSysOnTime >> u32Ticks;

      tU32                              days     = u32Ticks;
      tU32                              seconds  = days % 60;
      days /= 60;
      tU32                              minutes  = days % 60;
      days /= 60;
      tU32                              hours    = days % 24;
      days /= 24;     // days
      ETG_TRACE_FATAL( ( "Last System Runtime Count :      %dsec (%dd, %02d:%02d:%02d)", u32Ticks, days, hours, minutes, seconds ) );
   }

   {
      tU32 u32Ticks = OSAL_ClockGetElapsedTime( ) / 1000;
      tU32 days     = u32Ticks;
      tU32 seconds  = days % 60;
      days /= 60;
      tU32 minutes  = days % 60;
      days /= 60;
      tU32 hours    = days % 24;
      days /= 24;     // days
      ETG_TRACE_FATAL( ( "Cur. System Runtime Count :      %dsec (%dd, %02d:%02d:%02d)", u32Ticks, days, hours, minutes, seconds ) );
   }
} // vTrace

/*!
  * \fn
  *  \brief
  *    returns the current power on time (day/hours/minutes/seconds) as string
  *
  *  \param none
  ******
  */
template < typename T1, typename T2 >
std::string spm_tclPowerOnCountTemplate<T1, T2>::szAsString( ) const {
   std::string   buffer;

   tU32          poc     = _u32CurrentPwrOnTime;
   tU32          days    = poc;
   tU32          seconds = days % 60;

   days  /= 60;
   tU32          minutes = days % 60;
   days  /= 60;
   tU32          hours   = days % 24;
   days  /= 24;     // days

   buffer = std::to_string(poc);
   buffer += "sec (";
   buffer += std::to_string(days);
   buffer += "d, ";
   if(hours < 10) {
      buffer += "0";
   }
   buffer += std::to_string(hours);
   buffer += ":";
   if(minutes < 10) {
      buffer += "0";
   }
   buffer += std::to_string(minutes);
   buffer += ":";
   if(seconds < 10) {
      buffer += "0";
   }
   buffer += std::to_string(seconds);
   buffer += ")";

   return( buffer );
} // szAsString

spm_tclPowerOnCount::spm_tclPowerOnCount( const ISpmFactory& factory )
   : spm_tclPowerOnCountTemplate< dp_tclSpmDpPowOnPowerOnCount, dp_tclSpmDpPowOnLastSystemRunTime >( factory, SPM_U32_STATISTIC_GLOBAL )
{
}

spm_tclSWVersionPowerOnCount::spm_tclSWVersionPowerOnCount( const ISpmFactory& factory )
   : spm_tclPowerOnCountTemplate< dp_tclSpmDpPowOnCurrentSWVersionPowerOnCount, dp_tclSpmDpPowOnCurrentSWVersionLastSystemRunTime >( factory, SPM_U32_STATISTIC_SOFTWARE_VERSION )
{
}
