/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_SoftwareBlock.h
  * @brief        This is header file contains functions for SoftwareBlock.
  * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"

#include <algorithm>
#include <map>
#include <string>

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#include "spm_IDbClient.h"

class ISpmApplicationConfiguration;
class spm_tclConnectedApp;
class ISpmCcaSupplierServer;


// -----------------------------------------------------------------------------
// includes
// -----------------------------------------------------------------------------


// -----------------------------------------------------------------------------
// a SoftwareBlock can contain several applications (CCA components)
// -----------------------------------------------------------------------------

/*!
  *  \class spm_tclSoftwareBlock
  *  \brief This class contains information about a softwareblock which is the information unit of a process containing several applications.
          This class does operations to set all application to some profile , kill all applications , reset all aplications
        and send updates with/without block mode. If the connected application are grouped in a software block then they are in block mode otherwise
        in non block mode.

        This class is used by spm_tclApplicationConfiguration, spm_tclLocalApplicationManager, spm_tclLocalApplicationManagerConfig, spm_tclApplicationDatabase .

          This class send updates requested state with/without block and critical voltage events to spm_tclLocalApplicationManagerConfig.
  */

typedef std::map < tU32, spm_tclConnectedApp > TMapApplicationList;

class spm_tclSoftwareBlock
{
private:
std::string         _name;
std::string         _strExecPath;
std::string         _strSwBlockName;
std::string         _strStartType;
tBool               _bBlockMode;
tBool               _bBlockEnabled;
tBool               _bBlockConnected;
tBool               _bKillBlock;
tBool               _bBlockLoaded;
tBool               _bBlockUp;
tBool               _bIsLocal;
tU32                _u32PID;

tBool               _bForceBlock;
tU32                _u32ForceBlockState;

TMapApplicationList _mapApplicationList;

tBool bSetAllApplicationsToProfileWithBlockMode( ISpmApplicationConfiguration *pConfig,
                                                 ISpmDbClient                 *pClient );

tBool bSetAllApplicationsToProfileWithoutBlockMode( ISpmApplicationConfiguration *pConfig,
                                                    ISpmDbClient                 *pClient );

tBool bSendUpdateWithBlockMode( tU32          u32AppStateRequest,
                                ISpmDbClient *pClient );

tBool bSendUpdateWithoutBlockMode( spm_tclConnectedApp& app,
                                   ISpmDbClient        *pClient );

tBool bForceAllApplicationsToProfileWithBlockMode( tU32                          u32Profile,
                                                   tBool                         bForceAll,
                                                   ISpmApplicationConfiguration *pConfig,
                                                   ISpmDbClient                 *pClient );

tBool bForceAllApplicationsToProfileWithoutBlockMode( tU32                          u32Profile,
                                                      tBool                         bForceAll,
                                                      ISpmApplicationConfiguration *pConfig,
                                                      ISpmDbClient                 *pClient );

tBool bResetForceModeWithBlockMode( tU32 u32AppId );

tBool bResetForceModeWithoutBlockMode( tU32 u32AppId );

public:
spm_tclSoftwareBlock( const std::string& name = "" );
spm_tclSoftwareBlock( const spm_tclSoftwareBlock& op );
spm_tclSoftwareBlock& operator= (const spm_tclSoftwareBlock& roRight);
virtual ~spm_tclSoftwareBlock( );

spm_tclConnectedApp*poFind( tU32 u32AppId );

tU32 u32NumberOfConnectedApplications( ) const;

tU32  u32GetNumberOfApplications( ) const {
   return( (tU32)_mapApplicationList.size( ) );
}

tBool bIsDisconnected( ) const;

tBool bIsInSystem( ) const;

tBool bIsConnected( ) const { return( bIsInSystem( ) ); }

tBool bAllApplicationsInState( tU32 u32AppState ) const;

tBool bAllApplicationsInRequestedState( ) const;

tBool bAllApplicationsInProfile( ISpmApplicationConfiguration *pConfig ) const;

tVoid vCheckForSwBlocksToKill( ISpmDbClient *pClient );

tBool bCheckIfWholeBlockIsConnected( ISpmDbClient *pClient );

tBool bCheckIfWholeBlockIsInForcedState( ){
   tBool                                            bFound = FALSE;

   std::map < tU16, spm_tclConnectedApp >::iterator it;

   if ( bAllApplicationsInState( _u32ForceBlockState ) ){
      // ETG_TRACE_USR4(("spm_tclSoftwareBlock::bCheckIfWholeBlockIsInForcedState "));
      bFound = TRUE;
   } else {
      // ETG_TRACE_USR4(("spm_tclSoftwareBlock::bCheckIfWholeBlockIsInForcedState) -> ALL APPS NOT IN OFF"));
   }
   return( bFound );
}

tBool bCheckIfWholeBlockIsInRequestedState( ISpmDbClient *pClient );

tBool bAllApplicationsUpAndRunning( tU32& u32AppId ) const;

tBool bAllRequestsProcessed( tU32 u32AppState ) const;

tVoid vKill( ISpmDbClient *pClient );

tBool bSetAllApplicationsToProfile( ISpmApplicationConfiguration *pConfig,
                                    ISpmDbClient                 *pClient );

tBool bForceAllApplicationsToProfile( tU32                          u32Profile,
                                      tBool                         bForceAll,
                                      ISpmApplicationConfiguration *pConfig,
                                      ISpmDbClient                 *pClient );

tBool bSendUpdate( tU32          u32AppId,
                   ISpmDbClient *pClient );

tBool bResetForceMode( tU32 u32AppId );

tVoid vSetName( const std::string& name ){
   _name = name;
}

std::string pcGetName( ) const {
   return( _name );
}

tVoid vSetWholeBlockEnabled( tBool bBlockEnabled = TRUE ){
   _bBlockEnabled = bBlockEnabled;
}

tVoid vSetWholeBlockConnected( tBool bBlockConnect = TRUE ){
   _bBlockConnected = bBlockConnect;
}

tVoid vSetWholeBlockLoaded( tBool bBlockLoaded = TRUE ){
   _bBlockLoaded = bBlockLoaded;
}

tVoid vSetWholeBlockUp( tBool bBlockUp = TRUE ){
   _bBlockUp = bBlockUp;
}

tBool bIsWholeBlockConnected( ) const {
   return( _bBlockConnected );
}

tBool bIsWholeBlockEnabled( ) const {
   return( _bBlockEnabled );
}

tBool bIsWholeBlockLoaded( ) const {
   return( _bBlockLoaded );
}

tBool bIsWholeBlockUp( ) const {
   return( _bBlockUp );
}

tBool bIsWholeBlockTobeKilled( ) const {
   return( _bKillBlock );
}

tVoid vSetKillBlockEnabled( tBool bKillBlock = TRUE ){
   _bKillBlock = bKillBlock;
}

tBool bIsWholeBlockToBeForced( ) const {
   return( _bForceBlock );
}

tVoid vSetForceBlockEnabled( tU32  u32ForceState,
                             tBool bForceBlock = TRUE ){
   _u32ForceBlockState = u32ForceState;
   _bForceBlock        = bForceBlock;
}

tBool bKillAllSwBlockApplications( ISpmDbClient *pClient );

tBool bDeleteAllSwBlockApplicationsFromSupplier( ISpmCcaSupplierServer *poclCcaSupplierHandler );

tVoid vSetBlockMode( tBool bBlockMode = TRUE ){
   _bBlockMode = bBlockMode;
}

tBool bIsBlockMode( ) const {
   return( _bBlockMode );
}

tVoid vSetPID( tU32 u32PID ){
   _u32PID = u32PID;
}

tU32 u32GetPID( ) const {
   return( _u32PID );
}

tBool bCheckRequests( ISpmDbClient *pClient,
                      tBool       & bMissingResponse,
                      tU32        & u32CulpritAppID,
                      tU32        & u32UnacknowledgedState,
                      tU32        & u32ActiveRequests );

tBool bCheckLifeSignals( );

tVoid vOnWatchdogError( ISpmDbClient *pClient );

tVoid vSendCriticalVoltageEvent( tU32          u32Cvm,
                                 ISpmDbClient *pClient );

tVoid vTrace( ISpmDbClient* );

tVoid vAddConnectedApp( tU32  u32AppId,
                        tBool bIsLocal );

tVoid vSetExecPath( const std::string& str ){ _strExecPath = str; }
const std::string& oGetExecPath( ) const { return( _strExecPath ); }

tVoid vSetStartType( const std::string& str ){ _strStartType = str; }
const std::string& oGetStartType( ) const { return( _strStartType ); }

tVoid vSetSwBlockName( const std::string& str ){ _strSwBlockName = str; }
const std::string& oGetSwBlockName( ) const { return( _strSwBlockName ); }
tVoid vWriteQueueStatus( tChar *szName );

};

