/*!
  * \file spm_SoftwareBlock.cpp
  *  \brief
  *    Contains information about a softwareblock which is the
  *    information unit of a process containing several applications
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * 1.0 |  06.05.13  | Bipin Krishnan(RBEI/ECG4)     | initial version\n
  * 1.1 |  26.06.13  | Bipin Krishnan(RBEI/ECG4)     | redesigned SCC Client Handlers
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#include "spm_Config.h"

#include "spm_SoftwareBlock.h"

#include "spm_ConnectedApps.h"
#include "spm_ApplicationConfiguration.h"
#include "spm_ICcaSupplierServer.h"
#include "spm_IStartupCommon.h" //to get SPM_STARTUP_VALUE_OSAL_START_TYPE

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
    #include "trcGenProj/Header/spm_SoftwareBlock.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"


/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
// //#define SPM_TRACE_FILE_ID   SPM_FILE_SOFTWAREBLOCK


spm_tclSoftwareBlock::spm_tclSoftwareBlock( const std::string& pcName )
   : _name( pcName )
   , _strExecPath( "" )
   , _strSwBlockName( "" )
   , _strStartType( SPM_STARTUP_VALUE_OSAL_START_TYPE )
   , _bBlockMode( FALSE )
   , _bBlockEnabled( FALSE )
   , _bBlockConnected( FALSE )
   , _bKillBlock( FALSE )
   , _bBlockLoaded( FALSE )
   , _bBlockUp( FALSE )
   , _bIsLocal( TRUE )
   , _u32PID( 0 )
   , _bForceBlock( FALSE )
   , _u32ForceBlockState( 0 ){
}

spm_tclSoftwareBlock::spm_tclSoftwareBlock( const spm_tclSoftwareBlock& op ){
   _name               = op._name;
   _strExecPath        = op._strExecPath;
   _strSwBlockName     = op._strSwBlockName;
   _strStartType       = op._strStartType;
   _bBlockMode         = op._bBlockMode;
   _bBlockEnabled      = op._bBlockEnabled;
   _bBlockConnected    = op._bBlockConnected;
   _bKillBlock         = op._bKillBlock;
   _bBlockLoaded       = op._bBlockLoaded;
   _bBlockUp           = op._bBlockUp;
   _bIsLocal           = op._bIsLocal;
   _u32PID             = op._u32PID;
   _bForceBlock        = op._bForceBlock;
   _u32ForceBlockState = op._u32ForceBlockState;
   _mapApplicationList = op._mapApplicationList;
}

spm_tclSoftwareBlock& spm_tclSoftwareBlock::operator= (const spm_tclSoftwareBlock& roRight)
{
   if ( this != &roRight )
   {
      _name               = roRight._name;
      _strExecPath        = roRight._strExecPath;
      _strSwBlockName     = roRight._strSwBlockName;
      _strStartType       = roRight._strStartType;
      _bBlockMode         = roRight._bBlockMode;
      _bBlockEnabled      = roRight._bBlockEnabled;
      _bBlockConnected    = roRight._bBlockConnected;
      _bKillBlock         = roRight._bKillBlock;
      _bBlockLoaded       = roRight._bBlockLoaded;
      _bBlockUp           = roRight._bBlockUp;
      _bIsLocal           = roRight._bIsLocal;
      _u32PID             = roRight._u32PID;
      _bForceBlock        = roRight._bForceBlock;
      _u32ForceBlockState = roRight._u32ForceBlockState;
      _mapApplicationList = roRight._mapApplicationList;
   }
   return *this;
}

spm_tclSoftwareBlock::~spm_tclSoftwareBlock( ){
}

spm_tclConnectedApp*spm_tclSoftwareBlock::poFind( tU32 u32AppId ){
   TMapApplicationList::iterator it;

   it = _mapApplicationList.find( u32AppId );
   if ( it == _mapApplicationList.end( ) ){
      return( NULL );
   }
   return( &it->second );
}

tU32 spm_tclSoftwareBlock::u32NumberOfConnectedApplications( ) const {
   TMapApplicationList::const_iterator it;
   tU32                                                   u32Number = 0;

   for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
      if ( it->second.bIsConnected( ) ){
         ++u32Number;
      }
   }
   return( u32Number );
}

tBool spm_tclSoftwareBlock::bIsDisconnected( ) const {
   TMapApplicationList::const_iterator it;

   for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
      if ( it->second.bIsConnected( ) ){
         return( FALSE );
      }
   }
   return( TRUE );
}

tBool spm_tclSoftwareBlock::bIsInSystem( ) const {
   TMapApplicationList::const_iterator it;

   for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
      if ( !it->second.bIsConnected( ) ){
         return( FALSE );
      }
   }
   return( TRUE );
}

tBool spm_tclSoftwareBlock::bAllApplicationsInState( tU32 u32AppState ) const {
   TMapApplicationList::const_iterator it;

   for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
      if ( it->second.u32GetAppStateCurrent( ) != u32AppState ){
         return( FALSE );
      }
   }
   return( TRUE );
} // bAllApplicationsInState

tBool spm_tclSoftwareBlock::bAllApplicationsInRequestedState( ) const {
   TMapApplicationList::const_iterator it;

   for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
      if ( it->second.u32GetAppStateCurrent( ) != it->second.u32GetAppStateRequest( ) ){
         return( FALSE );
      }
   }
   return( TRUE );
}

tBool spm_tclSoftwareBlock::bAllApplicationsInProfile( ISpmApplicationConfiguration *pConfig ) const {
   TMapApplicationList::const_iterator it;

   for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
      if ( it->second.u32GetAppStateCurrent( ) != pConfig->u32GetNewAppState( it->second.u32GetAppId( ) ) ){
         return( FALSE );
      }
   }
   return( TRUE );
}

tBool spm_tclSoftwareBlock::bAllApplicationsUpAndRunning( tU32& u32AppId ) const {
   TMapApplicationList::const_iterator it;

   for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
      if ( ( it->second.u32GetAppStateCurrent( ) == it->second.u32GetUninitializedState( ) )
           || ( it->second.u32GetAppStateCurrent( ) == it->second.u32GetInitializedState( ) )
           ){
         u32AppId = it->second.u32GetAppId( );
         return( FALSE );
      }
   }
   return( TRUE );
}

tBool spm_tclSoftwareBlock::bAllRequestsProcessed( tU32 u32AppState ) const {
   TMapApplicationList::const_iterator it;

   for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
      if ( ( it->second.u32GetAppStateRequest( ) != u32AppState )
           || ( it->second.u32GetAppStateCurrent( ) != u32AppState ) ){
         return( FALSE );
      }
   }
   return( TRUE );
}

tVoid spm_tclSoftwareBlock::vCheckForSwBlocksToKill( ISpmDbClient *pClient ){
   TMapApplicationList::iterator it;

   for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
      if ( it->second.bIsConnected( ) && ( it->second.u32GetAppStateRequest( ) != AMT_C_U16_PWR_PROXY_END_APP ) ){
         it->second.vSetAppStateRequest( AMT_C_U16_PWR_PROXY_END_APP );

         pClient->vOnCheckForSwBlocksToKill( it->second.u32GetAppId( ) );
      }
   }
}

tBool spm_tclSoftwareBlock::bSetAllApplicationsToProfileWithBlockMode( ISpmApplicationConfiguration *pConfig,
                                                                       ISpmDbClient                 *pClient ){
   TMapApplicationList::iterator it;

   tBool                                            bFirst            = TRUE;
   tBool                                            bRequestProcessed = FALSE;

   for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
      // to check if all applications are in the same mode now, get the first
      // state request status from the first application and see if all other
      // applications are in this state, too.
      if ( bFirst ){
         bFirst            = FALSE;
         bRequestProcessed = bAllRequestsProcessed( it->second.u32GetAppStateRequest( ) );
      }
      tU32 u32AppState = AMT_C_U32_STATE_OFF;

      it->second.vSetAppStateNewRequest( ( bIsWholeBlockTobeKilled( ) || bIsWholeBlockToBeForced( ) )
                                         ? u32AppState
                                         : pConfig->u32GetNewAppState( it->second.u32GetAppId( ) ) );

      ETG_TRACE_USR1( ( "bSetAllApplicationsToProfileWithoutBlockMode, app: %u, check new request state: %u ()", ETG_ENUM( ail_u16AppId, it->second.u32GetAppId( ) ), ETG_ENUM( SPM_APPSTATE_DEF, it->second.u32GetAppStateNewRequest( ) ) ) );

      if ( bRequestProcessed ){
         // only if the current sw block is completely in its requested state then
         // the new state request is put into the state request variable.
         if ( ( it->second.u32GetAppStateCurrent( ) != it->second.u32GetAppStateNewRequest( ) )
              && ( it->second.u32GetAppStateRequest( ) == it->second.u32GetAppStateCurrent( ) )
              && it->second.bIsConnected( ) ){
            it->second.vSetAppStateRequest( it->second.u32GetAppStateNewRequest( ) );
            // after a new state request the block is no longer completely up
               vSetWholeBlockUp( FALSE );
            pClient->vOnSetAllApplicationsToProfile( it->second.u32GetAppId( ),
                                                     it->second.u32GetAppStateNewRequest( ),
                                                     TRUE );
         }
      }
   }
   return( TRUE );
} // bSetAllApplicationsToProfileWithBlockMode

tBool spm_tclSoftwareBlock::bSetAllApplicationsToProfileWithoutBlockMode( ISpmApplicationConfiguration *pConfig,
                                                                          ISpmDbClient                 *pClient ){
   TMapApplicationList::iterator it;

   for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
      tU32 u32AppState = AMT_C_U32_STATE_OFF;
      it->second.vSetAppStateNewRequest( ( bIsWholeBlockTobeKilled( ) || bIsWholeBlockToBeForced( ) )
                                         ? u32AppState
                                         : pConfig->u32GetNewAppState( it->second.u32GetAppId( ) ) );

      if ( ( it->second.u32GetAppStateCurrent( ) != it->second.u32GetAppStateNewRequest( ) )
           && ( it->second.u32GetAppStateRequest( ) == it->second.u32GetAppStateCurrent( ) )
           && it->second.bIsConnected( ) ){
         it->second.vSetAppStateRequest( it->second.u32GetAppStateNewRequest( ) );
         // after a new state request the block is no longer completely up
               vSetWholeBlockUp( FALSE );
         pClient->vOnSetAllApplicationsToProfile( it->second.u32GetAppId( ),
                                                  it->second.u32GetAppStateNewRequest( ),
                                                  FALSE );
      }
   }
   return( TRUE );
} // bSetAllApplicationsToProfileWithoutBlockMode

tBool spm_tclSoftwareBlock::bSetAllApplicationsToProfile( ISpmApplicationConfiguration *pConfig,
                                                          ISpmDbClient                 *pClient ){
   if ( bIsBlockMode( ) ){
      return( bSetAllApplicationsToProfileWithBlockMode( pConfig, pClient ) );
   }
   return( bSetAllApplicationsToProfileWithoutBlockMode( pConfig, pClient ) );
}

/*tBool spm_tclSoftwareBlock::bForceAllApplicationsToProfileWithBlockMode(
    tU32 u32Profile,
    tBool bForceAll,
    spm_tclApplicationConfiguration* pConfig,
    ISpmDbClient* pClient)*/
tBool spm_tclSoftwareBlock::bForceAllApplicationsToProfileWithBlockMode( tU32                          u32Profile,
                                                                         tBool                         bForceAll,
                                                                         ISpmApplicationConfiguration *pConfig,
                                                                         ISpmDbClient                 *pClient ){
   TMapApplicationList::iterator it;

   tBool                                            bFirst            = TRUE;
   tBool                                            bRequestProcessed = FALSE;

   for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
      // to check if all applications are in the same mode now, get the first
      // state request status from the first application and see if all other
      // applications are in this state, too.
      if ( bFirst ){
         bFirst            = FALSE;
         bRequestProcessed = bAllRequestsProcessed( it->second.u32GetAppStateRequest( ) );
      }
      tU32 u32ProfileControl = pConfig->u32GetProfileAppState( SPM_SYSTEM_STATE_PROFILE, it->second.u32GetAppId( ) );

      if ( bForceAll || ( u32ProfileControl == (tU32)TRUE ) ){  // weird comparison but old code
         tU32 u32NewAppState = pConfig->u32GetProfileAppState( u32Profile, it->second.u32GetAppId( ) );

         if ( bRequestProcessed ){
            if ( ( it->second.u32GetAppStateCurrent( ) != u32NewAppState )
                 && ( it->second.u32GetAppStateRequest( ) == it->second.u32GetAppStateCurrent( ) )
                 && it->second.bIsConnected( ) ){
               it->second.vSetAppStateRequest( u32NewAppState );

               vSetWholeBlockUp( FALSE );

               pClient->vOnForceAllApplicationsToProfile( it->second.u32GetAppId( ),
                                                          u32NewAppState,
                                                          it->second.u32GetAppStateRequest( ),
                                                          TRUE );
            } else {
               it->second.vSetAppStateForceReq( u32NewAppState );
            }
         } else {
            it->second.vSetAppStateForceReq( u32NewAppState );
         }
      }
   }
   return( TRUE );
} // bForceAllApplicationsToProfileWithBlockMode

tBool spm_tclSoftwareBlock::bForceAllApplicationsToProfileWithoutBlockMode( tU32                          u32Profile,
                                                                            tBool                         bForceAll,
                                                                            ISpmApplicationConfiguration *pConfig,
                                                                            ISpmDbClient                 *pClient ){
   TMapApplicationList::iterator it;

   for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
      tU32 u32ProfileControl = pConfig->u32GetProfileAppState( SPM_SYSTEM_STATE_PROFILE, it->second.u32GetAppId( ) );

      if ( bForceAll || ( u32ProfileControl == (tU32)TRUE ) ){  // weird comparison, but old code
         tU32 u32NewAppState = pConfig->u32GetProfileAppState( u32Profile, it->second.u32GetAppId( ) );
         if ( ( it->second.u32GetAppStateCurrent( ) != u32NewAppState )
              && ( it->second.u32GetAppStateRequest( ) == it->second.u32GetAppStateCurrent( ) )
              && it->second.bIsConnected( ) ){
            it->second.vSetAppStateRequest( u32NewAppState );

               vSetWholeBlockUp( FALSE );

            pClient->vOnForceAllApplicationsToProfile( it->second.u32GetAppId( ),
                                                       u32NewAppState,
                                                       it->second.u32GetAppStateRequest( ),
                                                       FALSE );
         } else {
            it->second.vSetAppStateForceReq( u32NewAppState );
         }
      }
   }
   return( TRUE );
} // bForceAllApplicationsToProfileWithoutBlockMode

tBool spm_tclSoftwareBlock::bForceAllApplicationsToProfile( tU32                          u32Profile,
                                                            tBool                         bForceAll,
                                                            ISpmApplicationConfiguration *pConfig,
                                                            ISpmDbClient                 *pClient ){
   if ( bIsBlockMode( ) ){
      return( bForceAllApplicationsToProfileWithBlockMode( u32Profile, bForceAll, pConfig, pClient ) );
   }
   return( bForceAllApplicationsToProfileWithoutBlockMode( u32Profile, bForceAll, pConfig, pClient ) );
}

tBool spm_tclSoftwareBlock::bKillAllSwBlockApplications( ISpmDbClient *pClient ){
   TMapApplicationList::iterator it;
   tBool                                            bReturn = FALSE;

   SPM_NULL_POINTER_CHECK_VAL( pClient );
   for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
      if ( it->second.bIsConnected( ) ){
         it->second.vSetAppStateRequest( AMT_C_U16_PWR_PROXY_END_APP );
         bReturn = TRUE;
         pClient->vOnKillAllSwBlockApplications( it->second.u32GetAppId( ) );
      }
   }
   return( bReturn );
}

tBool spm_tclSoftwareBlock::bDeleteAllSwBlockApplicationsFromSupplier( ISpmCcaSupplierServer *poclCcaSupplierHandler ){
   TMapApplicationList::iterator it;
   tBool                                            bReturn = FALSE;

   SPM_NULL_POINTER_CHECK_VAL( poclCcaSupplierHandler );
   for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
      if ( it->second.bIsConnected( ) ){
         it->second.vSetAppStateRequest( AMT_C_U16_PWR_PROXY_END_APP );
         bReturn = TRUE;

         poclCcaSupplierHandler->vSupplierStateChanged( (tU16)it->second.u32GetAppId( ), AMT_C_U8_CCAMSG_SUPPLIER_STATE_UNAVAILABLE );
      }
   }
   return( bReturn );
}

class spm_tclFindAppId
{
private:
tU32 _u32AppId;

public:
spm_tclFindAppId( tU32 u32AppID ) : _u32AppId( u32AppID ){}

bool operator()( const std::pair < tU32, spm_tclConnectedApp >& op ) const {
   return( op.second.u32GetAppId( ) == _u32AppId );
}

};


tBool spm_tclSoftwareBlock::bSendUpdateWithBlockMode( tU32          u32AppStateRequest,
                                                      ISpmDbClient *pClient ){
   TMapApplicationList::iterator it;

   if ( bAllRequestsProcessed( u32AppStateRequest ) ){
      for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
         tU32 u32ReqState;

         if ( it->second.u32GetUninitializedState( ) == it->second.u32GetAppStateForceReq( ) ){
            u32ReqState = it->second.u32GetAppStateNewRequest( );
            it->second.vSetAppStateRequest( it->second.u32GetAppStateNewRequest( ) );
         } else {
            u32ReqState = it->second.u32GetAppStateForceReq( );
            it->second.vSetAppStateRequest( u32ReqState );
            it->second.vSetAppStateForceReq( it->second.u32GetUninitializedState( ) );
         }

         vSetWholeBlockUp( FALSE );

         pClient->vOnSendUpdate( it->second.u32GetAppId( ),
                                 u32ReqState,
                                 it->second.u32GetAppStateRequest( ),
                                 it->second.u32GetAppStateNewRequest( ),
                                 TRUE );
      }
   }
   return( TRUE );
} // bSendUpdateWithBlockMode

tBool spm_tclSoftwareBlock::bSendUpdateWithoutBlockMode( spm_tclConnectedApp& app,
                                                         ISpmDbClient        *pClient ){
   tU32 u32ReqState;

   if ( app.u32GetUninitializedState( ) == app.u32GetAppStateForceReq( ) ){
      u32ReqState = app.u32GetAppStateNewRequest( );
      app.vSetAppStateRequest( app.u32GetAppStateNewRequest( ) );
   } else {
      u32ReqState = app.u32GetAppStateForceReq( );
      app.vSetAppStateRequest( u32ReqState );
      app.vSetAppStateForceReq( app.u32GetUninitializedState( ) );
   }

         vSetWholeBlockUp( FALSE );

   pClient->vOnSendUpdate( app.u32GetAppId( ),
                           u32ReqState,
                           app.u32GetAppStateRequest( ),
                           app.u32GetAppStateNewRequest( ),
                           FALSE );
   return( TRUE );
} // bSendUpdateWithoutBlockMode

tBool spm_tclSoftwareBlock::bSendUpdate( tU32          u32AppId,
                                         ISpmDbClient *pClient ){
   TMapApplicationList::iterator it;
   spm_tclFindAppId                                 f( u32AppId ); // predicate for find_if

   it = std::find_if( _mapApplicationList.begin( ), _mapApplicationList.end( ), f );
   if ( it != _mapApplicationList.end( ) ){
      // found the application with the app id in the softwareblock
      if ( bIsBlockMode( ) ){
         return( bSendUpdateWithBlockMode( it->second.u32GetAppStateRequest( ), pClient ) );
      }
      return( bSendUpdateWithoutBlockMode( it->second, pClient ) );
   }
   return( FALSE );
}

tBool spm_tclSoftwareBlock::bResetForceModeWithoutBlockMode( tU32 u32AppId ){
   TMapApplicationList::iterator it;
   spm_tclFindAppId                                 f( u32AppId ); // predicate for find_if

   it = std::find_if( _mapApplicationList.begin( ), _mapApplicationList.end( ), f );
   if ( it != _mapApplicationList.end( ) ){
      it->second.vSetAppStateForceReq( it->second.u32GetUninitializedState( ) );
      return( TRUE ); // application found, stop iteration
   }
   return( FALSE );
}

tBool spm_tclSoftwareBlock::bResetForceModeWithBlockMode( tU32 u32AppId ){
   TMapApplicationList::iterator it;
   spm_tclFindAppId                                 f( u32AppId ); // predicate for find_if

   it = std::find_if( _mapApplicationList.begin( ), _mapApplicationList.end( ), f );
   if ( it != _mapApplicationList.end( ) ){
      TMapApplicationList::iterator it2;

      for ( it2 = _mapApplicationList.begin( ); it2 != _mapApplicationList.end( ); ++it2 ){
         it2->second.vSetAppStateForceReq( it2->second.u32GetUninitializedState( ) );
      }
      return( TRUE );
   }
   return( FALSE );
}

tBool spm_tclSoftwareBlock::bResetForceMode( tU32 u32AppId ){
   if ( bIsBlockMode( ) ){
      return( bResetForceModeWithBlockMode( u32AppId ) );
   }
   return( bResetForceModeWithoutBlockMode( u32AppId ) );
}

tBool spm_tclSoftwareBlock::bCheckRequests( ISpmDbClient *pClient,
                                            tBool       & bMissingResponse,
                                            tU32        & u32CulpritAppID,
                                            tU32        & u32UnacknowledgedState,
                                            tU32        & u32ActiveRequests ){
   TMapApplicationList::iterator it;
   tBool                                            bReturn        = TRUE;

   tU32                                             u32CurrentTime = OSAL_ClockGetElapsedTime( );


   for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
      tU32 u32AppStateReqStartTime = it->second.u32GetAppStateReqStartTime( );
      tU32 u32AppStateReqEndTime   = it->second.u32GetAppStateReqEndTime( );

      if ( ( it->second.u32GetAppStateRequest( ) != it->second.u32GetAppStateCurrent( ) )
           && ( u32CurrentTime > u32AppStateReqStartTime ) ){
         pClient->vOnCheckRequests( u32ActiveRequests );

         ETG_TRACE_USR1( ( "spm_tclSoftwareBlock::bCheckRequests(): missing ack: app: %u (diff: %d, end: %d, imm: %d -> %d)", ETG_ENUM( ail_u16AppId, it->second.u32GetAppId( ) ), u32CurrentTime - u32AppStateReqStartTime, u32AppStateReqEndTime, u32AppStateReqEndTime - SPM_APPREQMON_WARNING_INTERVAL, ( u32AppStateReqEndTime - ( SPM_APPREQMON_WARNING_INTERVAL - SPM_APPREQMON_CHECK_INTERVAL ) ) ) );

         if ( ( u32CurrentTime - u32AppStateReqStartTime ) > u32AppStateReqEndTime ){
            pClient->vOnUnacknowledgeState( it->second );

            u32CulpritAppID        = it->second.u32GetAppId( );
            u32UnacknowledgedState = it->second.u32GetAppStateRequest( );
            bMissingResponse       = TRUE;

            bReturn                = FALSE;
         } else if ( ( ( u32CurrentTime - u32AppStateReqStartTime ) >= ( u32AppStateReqEndTime - SPM_APPREQMON_WARNING_INTERVAL ) )
                     && ( ( u32CurrentTime - u32AppStateReqStartTime ) < ( u32AppStateReqEndTime - ( SPM_APPREQMON_WARNING_INTERVAL - SPM_APPREQMON_CHECK_INTERVAL ) ) )
                     ){
            // trigger change state immediately once
            if ( ( it->second.u32GetAppStateRequest( ) != AMT_C_U16_PWR_PROXY_END_APP )
                 && ( it->second.u32GetInitializedState( ) != it->second.u32GetAppStateRequest( ) ) ){
               pClient->vOnSendImmediateStateChange( it->second );
               bMissingResponse = TRUE;
            } else if ( it->second.u32GetAppStateRequest( ) == AMT_C_U16_PWR_PROXY_END_APP ){
               // already in TERMINATE state, do not wait for a response
               pClient->vOnTerminateMissing( it->second );
               bMissingResponse = TRUE;
            }
         }
      } else {
         it->second.vUpdateAppStateReqStartTime( );
         it->second.vSetAppStateReqEndTime( SPM_APPREQMON_DEFAULT_INTERVAL );
      }
   }
   return( bReturn );
} // bCheckRequests

tBool spm_tclSoftwareBlock::bCheckLifeSignals( ){
   TMapApplicationList::iterator it;
   tBool                                            bReturn        = TRUE;
   tU32                                             u32CurrentTime = OSAL_ClockGetElapsedTime( );

   for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
      if ( it->second.bIsWatchdogActive( ) ){
         if ( ( u32CurrentTime > it->second.u32GetLastLifeSignalTime( ) )
              && ( it->second.u32GetNotifyInterval( ) > 0 ) ){
            // > u32NotifyInterval+1000 (ms) because framework needs some time
            // to send the watchdog-signal
            if ( u32CurrentTime - it->second.u32GetLastLifeSignalTime( ) > it->second.u32GetNotifyInterval( ) + 1000 ){
               it->second.vSetWatchdogError( );
               bReturn = FALSE;
            } else {
               it->second.vSetWatchdogError( FALSE );
            }
         }
      }
   }
   return( bReturn );
} // bCheckLifeSignals

tVoid spm_tclSoftwareBlock::vOnWatchdogError( ISpmDbClient *pClient ){
   TMapApplicationList::iterator it;

   for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
      if ( it->second.bIsWatchdogError( ) ){
         pClient->vOnWatchdogError( it->second );
         if ( it->second.bHasNotifiedProblemRequest( ) ){
            pClient->vOnWatchdogErrorNotifyProblemReport( it->second );
         } else {
            pClient->vOnWatchdogErrorNotifyProblemReportNew( it->second );
            it->second.vSetNotifyProblemRequest( );
         }
      }
   }
}

tVoid spm_tclSoftwareBlock::vSendCriticalVoltageEvent( tU32          u32Cvm,
                                                       ISpmDbClient *pClient ){
   TMapApplicationList::iterator it;

   for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
      if ( it->second.bIsConnected( ) ){
         pClient->vOnSendCriticalVoltageEvent( it->second, u32Cvm );
      }
   }
}

tVoid spm_tclSoftwareBlock::vTrace( ISpmDbClient *pClient ){
   TMapApplicationList::iterator it;

   for ( it = _mapApplicationList.begin( ); it != _mapApplicationList.end( ); ++it ){
      pClient->vOnTraceApplicationInfo( it->second );
   }
}

tVoid spm_tclSoftwareBlock::vAddConnectedApp( tU32  u32AppId,
                                              tBool bIsLocal ){
   _mapApplicationList[u32AppId] = spm_tclConnectedApp( u32AppId, bIsLocal );
}

tVoid spm_tclSoftwareBlock::vKill( ISpmDbClient *pClient ){
   if ( !bIsWholeBlockTobeKilled( ) ){
      vSetKillBlockEnabled( );

      if ( bAllApplicationsInState( AMT_C_U32_STATE_OFF ) ){
         vCheckForSwBlocksToKill( pClient );
      } else {
         pClient->vOnKill( );
      }
   }
}

tBool spm_tclSoftwareBlock::bCheckIfWholeBlockIsInRequestedState( ISpmDbClient *pClient ){
   tBool bResult = bAllApplicationsInRequestedState( );

   // all applications are available in this sw block and if these are all
   // in the requested state then notify client about that status change.
   //
   if ( bResult
        && ( u32GetNumberOfApplications( ) != 0 )
        && !bIsWholeBlockUp( ) ){
      vSetWholeBlockUp( );

      pClient->vOnWholeSwBlockIsInRequestedState( pcGetName( ) );

      if ( bIsWholeBlockToBeForced( ) && bCheckIfWholeBlockIsInForcedState( ) ){
         pClient->vOnWholeSwBlockIsInForcedState( pcGetName( ) );
      }
   }

   return( bResult );
} // bCheckIfWholeBlockIsInRequestedState

tBool spm_tclSoftwareBlock::bCheckIfWholeBlockIsConnected( ISpmDbClient *pClient ){
   if ( bIsConnected( ) && !bIsWholeBlockConnected( ) ){
      vSetWholeBlockConnected( );
      pClient->vOnWholeSwBlockIsConnected( pcGetName( ) );
   }
   return( bIsWholeBlockConnected( ) );
}

// EOF

