/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_IApplicationDatabase.h
  * @brief        This is header file contains interfaces for handling ApplicationDatabase.
  * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#include <list>

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#include "spm_ISpmBase.h"

class spm_tclConnectedApp;
class spm_tclSoftwareBlock;
class spm_tclConnectedApp;
class ISpmDbClient;
class ISpmApplicationConfiguration;


// //////////////////////////////////////////////////////////////////////////////
// Interface to ApplicationDatabase

/*!
  *  \class ISpmApplicationDatabase
  *  \brief This abstract class provides interfaces to Application Database. This class provides pure virtual functions to
          remove software blocks and get software block of process. It also supplies pure virtual functions to get number of connected applications,
        to check for software blocks to kill and force all application to specific profile.
  */
class ISpmApplicationDatabase : public ISpmBase
{
public:
ISpmApplicationDatabase( const ISpmFactory& factory ) : ISpmBase( factory ){ }
virtual ~ISpmApplicationDatabase( ){ }

// top level access functions.
virtual spm_tclConnectedApp*poGetByAppID( tU32 u32AppID )                                        = 0;

virtual spm_tclSoftwareBlock*poGetSwBlockByAppID( tU32 u32AppID )                                = 0;

virtual spm_tclSoftwareBlock*poGetSoftwareBlock( const std::string& strName )                    = 0;

virtual spm_tclSoftwareBlock*poGetSwBlockByExecPath( const std::string& strExecPath )            = 0;

virtual spm_tclSoftwareBlock*poAddSoftwareBlock( const std::string& strName )                    = 0;

virtual tVoid vRemoveSoftwareBlock( const std::string& strName )                                 = 0;

virtual tVoid vGetSwBlocksOfProcess( const std::string        & strProcessName,
                                     std::list < std::string > *poSwBlockList )                  = 0;

virtual tU32 u32NumberOfConnectedApplications( )                                                 = 0;

virtual tBool bAllApplicationsInProfile( ISpmApplicationConfiguration *pConfig )                 = 0;

virtual tVoid vCheckForSwBlocksToKill( ISpmDbClient *pClient )                                   = 0;

virtual tBool bAllApplicationsUpAndRunning( tU32& u32AppId )                                     = 0;

virtual tBool bAllApplicationsInRequestedState( ISpmDbClient *pClient )                          = 0;

virtual tBool bSetAllSwBlocksToNonBlockMode( )                                                   = 0;

virtual tBool bSetAllApplicationsToProfile( ISpmApplicationConfiguration *pConfig,
                                            ISpmDbClient                 *pClient )              = 0;


virtual tBool bKillApplications( ISpmDbClient *pClient )                                         = 0;

virtual tBool bSendUpdate( tU32          u32AppId,
                           ISpmDbClient *pClient )                                               = 0;

virtual tBool bForceAllApplicationsToProfile( tU32                          u32Profile,
                                              tBool                         bForceAll,
                                              ISpmApplicationConfiguration *pConfig,
                                              ISpmDbClient                 *pClient )            = 0;

virtual tVoid vResetForceMode( tU32          u32AppId,
                               ISpmDbClient *pClient )                                           = 0;

virtual tBool bCheckRequests( ISpmDbClient *pClient,
                              tBool       & bMissingResponse,
                              tU32        & u32CulpritAppID,
                              tU32        & u32UnacknowledgedState,
                              tU32        & u32ActiveRequests )          = 0;

virtual tBool bCheckLifeSignals( )                                       = 0;

virtual tVoid vOnWatchdogError( ISpmDbClient *pClient )                  = 0;

virtual tVoid vSendCriticalVoltageEvent( tU32          u32Cvm,
                                         ISpmDbClient *pClient )         = 0;

virtual tVoid vTraceApplicationsInfo( ISpmDbClient *pClient )            = 0;

virtual tVoid vTraceSwBlocksInfo( ISpmDbClient *pClient )                = 0;

virtual tVoid vTraceSwBlockInfo( const std::string& pcName,
                                 ISpmDbClient      *pClient )            = 0;

virtual tBool bLockedByCaller( )                                         = 0;       // is database locked by caller

virtual const tChar*getInterfaceName( ) const { return( "ISpmApplicationDatabase" ); }
};

