/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_ApplicationDatabase.h
  * @brief        This is header file contains interface functions for ApplicationDatabase.
  * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#include <map>
#include <string>
#include <vector>

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#include "spm_ISwBlock.h"

// -----------------------------------------------------------------------------
// includes
// -----------------------------------------------------------------------------

#include "spm_ConnectedApps.h"
#include "spm_SoftwareBlock.h"
#include "spm_IApplicationDatabase.h"

class ISpmSystemPowerManager;
class ISpmOsalProxy;
class ISpmCcaSupplierServer;
class ISpmApplicationConfiguration;

// -----------------------------------------------------------------------------
// stores information about a softwareblock which contains several application
// information blocks in it.
// -----------------------------------------------------------------------------

/*!
  *  \class spm_tclApplicationDatabase
  *  \brief This class stores information about software blocks . It provides operations to add software block , get software block by application Id,
  *         remove software blocks, number of connected applications , check for software block to kill, set all software block to non block state .
  *
  *         This class uses spm_tclOsalProxyConfig to get thread information by thread id.
  *         It uses spm_tclSystemPowerManagerConfig to write error in memory and it uses spm_tclCcaSupplierHandlerConfig to remove software blocks from CCA supplierHandler.
  *
  *         The spm_tclFactory object creates an instance of this class.This class is used by spm_tclApplicationConfiguration, spm_tclAppRequestMonitor,
  *         spm_tclLocalApplicationManager, spm_tclLamAppManagerConfig for application database .
  */

class spm_tclApplicationDatabase : public ISpmApplicationDatabase
{
public:
spm_tclApplicationDatabase( tU32               u32SpmId,
                            const ISpmFactory& factory );
virtual ~spm_tclApplicationDatabase( );

// top level access functions.
virtual spm_tclConnectedApp*poGetByAppID( tU32 u32AppID );

virtual spm_tclSoftwareBlock*poGetSwBlockByAppID( tU32 u32AppID );

virtual spm_tclSoftwareBlock*poGetSoftwareBlock( const std::string& strName );

virtual spm_tclSoftwareBlock*poAddSoftwareBlock( const std::string& strName );

virtual spm_tclSoftwareBlock*poGetSwBlockByExecPath( const std::string& strExecPath );

virtual tVoid vRemoveSoftwareBlock( const std::string& strName );

virtual tVoid vGetSwBlocksOfProcess( const std::string        & strProcessName,
                                     std::list < std::string > *poSwBlockList );

virtual tU32 u32NumberOfConnectedApplications( );

virtual tBool bAllApplicationsInProfile( ISpmApplicationConfiguration *pConfig );

virtual tVoid vCheckForSwBlocksToKill( ISpmDbClient *pClient );

virtual tBool bAllApplicationsUpAndRunning( tU32& u32AppId );

virtual tBool bAllApplicationsInRequestedState( ISpmDbClient *pClient );

virtual tBool bSetAllSwBlocksToNonBlockMode( );

virtual tBool bSetAllApplicationsToProfile( ISpmApplicationConfiguration *pConfig,
                                            ISpmDbClient                 *pClient );

virtual tBool bKillApplications( ISpmDbClient *pClient );

virtual tBool bSendUpdate( tU32          u32AppId,
                           ISpmDbClient *pClient );

virtual tBool bForceAllApplicationsToProfile( tU32                          u32Profile,
                                              tBool                         bForceAll,
                                              ISpmApplicationConfiguration *pConfig,
                                              ISpmDbClient                 *pClient );

virtual tVoid vResetForceMode( tU32          u32AppId,
                               ISpmDbClient *pClient );

virtual tBool bCheckRequests( ISpmDbClient *pClient,
                              tBool       & bMissingResponse,
                              tU32        & u32CulpritAppID,
                              tU32        & u32UnacknowledgedState,
                              tU32        & u32ActiveRequests );

virtual tBool bCheckLifeSignals( );

virtual tVoid vOnWatchdogError( ISpmDbClient *pClient );

virtual tVoid vSendCriticalVoltageEvent( tU32          u32Cvm,
                                         ISpmDbClient *pClient );

virtual tVoid vTraceApplicationsInfo( ISpmDbClient *pClient );

virtual tVoid vTraceSwBlocksInfo( ISpmDbClient *pClient );

virtual tVoid vTraceSwBlockInfo( const std::string& pcName,
                                 ISpmDbClient      *pClient );

virtual tBool bLockedByCaller( ){ return( false ); }

protected:
virtual tVoid vAccessDatabaseIntern( );

virtual tVoid vReleaseDatabaseIntern( );

private:
OSAL_tSemHandle                      _hListAccess;
OSAL_tThreadID                       _idt;
OSAL_tProcessID                      _idp;
std::string                          _szThreadName;
tU32                                 _u32SpmId;
std::vector < spm_tclSoftwareBlock > _vecSwBlock;

// references to singleton objects
ISpmSystemPowerManager              *_poclSystemPowerManager;
ISpmOsalProxy                       *_poclOsalProxy;
ISpmCcaSupplierServer               *_poclCcaSupplierHandler;

public:
// ISpmBase interface functions
virtual tVoid vHandleTraceMessage( const tUChar */*puchData*/ ){} //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

protected:
virtual const tChar*getName( ) const { return( "spm_tclApplicationDatabase" ); }
virtual tVoid vGetReferences( );

virtual tVoid vStartCommunication( );

};

