/*!
  * \file spm_ApplicationDatabase.cpp
  *  \brief
  *   Application database management
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 14.07.11  | TMS Fischer       | initial version
  * 27.11.12  | CM-AI/CB32 kollai | Adaptation for GENERIC PLATFORM
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "OsalConf.h"
#include "osal_public.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_ApplicationDatabase.h"

// interfaces class definitions
#include "spm_ISystemPowerManager.h"
#include "spm_IOsalProxy.h"
#include "spm_ICcaSupplierServer.h"

#include "spm_IFactory.h"

// spm helper
#include "spm_CriticalSection.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_ApplicationDatabase.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"


/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
// #define SPM_TRACE_FILE_ID   SPM_FILE_APPLICATIONDATABASE

#define SPM_SEM_APP_DB_NAME_PREFIX          "SpmAppDb"
#define SPM_SEM_APP_DB_NAME_POSTFIX_FORMAT  "%s%010u" // %u is filled with tU32 = 10 decimal places
#define SPM_SEM_APP_DB_NAME_LENGTH          ((tU8)(sizeof(SPM_SEM_APP_DB_NAME_PREFIX)+10))

#define SPM_APP_DATABASE_SWBLOCK_ITERATOR std::vector < spm_tclSoftwareBlock >::iterator it;

spm_tclApplicationDatabase::spm_tclApplicationDatabase( tU32               u32SpmId,
                                                        const ISpmFactory& factory )
   : ISpmApplicationDatabase( factory )
   , _hListAccess( OSAL_C_INVALID_HANDLE )
   , _idt( - 1 )
   , _szThreadName( OSAL_C_U32_MAX_NAMELENGTH, 0 )
   , _u32SpmId( u32SpmId )
   , _poclSystemPowerManager( NULL )
   , _poclOsalProxy( NULL )
   , _poclCcaSupplierHandler( NULL ){
/*!
  * \fn
  *  \brief
  *    Constructor
  *
  *  \param[in]
     u32SpmId: application ID of this SPM instance
     factory: spm factory object.
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   tChar szSemName[SPM_SEM_APP_DB_NAME_LENGTH];

   _idp      = OSAL_ProcessWhoAmI( );

   _idt   = OSAL_ThreadWhoAmI( );
   (tVoid)bGetTaskName( _idp, _idt, &_szThreadName[0] );

   if (OSALUTIL_s32SaveNPrintFormat(
      szSemName,
      sizeof(szSemName),
      SPM_SEM_APP_DB_NAME_POSTFIX_FORMAT,
      SPM_SEM_APP_DB_NAME_PREFIX,
     _u32SpmId) == OSAL_ERROR) {
         tU32 u32ErrorReason = OSAL_u32ErrorCode( );
         ETG_TRACE_ERRMEM( ( "SPM: spm_tclApplicationDatabase constructor !!!!!! Error detected !!!!!! cannot create Semaphore name for _hListAccess: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
     return;
   }

   // create semaphore for parallel access to application database from
   // localapplicationmanager, startup processes, appwdtmonitor, sysstatemanager
   if ( OSAL_s32SemaphoreCreate( szSemName, &_hListAccess, (tU32)1 ) != OSAL_OK ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: spm_tclApplicationDatabase constructor !!!!!! Error detected !!!!!! cannot create Semaphore _hListAccess: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
}

spm_tclApplicationDatabase::~spm_tclApplicationDatabase( ){
/*!
  * \fn
  *  \brief
  *    Destructor
  *
  *  \param none
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   tChar szSemName[SPM_SEM_APP_DB_NAME_LENGTH];

   if (OSALUTIL_s32SaveNPrintFormat(
      szSemName,
      sizeof(szSemName),
      SPM_SEM_APP_DB_NAME_POSTFIX_FORMAT,
      SPM_SEM_APP_DB_NAME_PREFIX,
     _u32SpmId) == OSAL_ERROR) {
         tU32 u32ErrorReason = OSAL_u32ErrorCode( );
         ETG_TRACE_ERRMEM( ( "SPM: spm_tclApplicationDatabase destructor !!!!!! Error detected !!!!!! cannot create Semaphore name for _hListAccess: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
     return; 
   }

   if ( _hListAccess != OSAL_C_INVALID_HANDLE ){
      OSAL_s32SemaphoreClose( _hListAccess );
      OSAL_s32SemaphoreDelete( szSemName );
   }

   _poclOsalProxy          = NULL;
   _poclSystemPowerManager = NULL;
   _poclCcaSupplierHandler = NULL;
}

tVoid spm_tclApplicationDatabase::vGetReferences( ){
/*!
  * \fn
  *  \brief
  *    get all needed references to other SPM objects from the factory
  *
  *  \param none
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSystemPowerManager, ISpmSystemPowerManager );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclOsalProxy,          ISpmOsalProxy );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclCcaSupplierHandler, ISpmCcaSupplierServer );
}

tVoid spm_tclApplicationDatabase::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *    first time this module should interact with other SPM modules
  *
  *  \param none
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
}

tVoid spm_tclApplicationDatabase::vAccessDatabaseIntern( ){
/*!
  * \fn
  *  \brief
  *    Method to get the lock of ApplicationDatabase.
  *
  *  \param none
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   OSAL_tThreadID myThreadId = OSAL_ThreadWhoAmI( );

   if ( myThreadId != _idt ){
      std::string szTmpThreadName( OSAL_C_U32_MAX_NAMELENGTH, 0 );
      (tVoid)bGetTaskName( myThreadId, myThreadId, &szTmpThreadName[0] );

      ETG_TRACE_FATAL( ( "spm_tclApplicationDatabase::vAccessDatabase(): Wrong thread context for calling database: Thread: (tid=%d) '%s' .",
                         myThreadId,
                         szTmpThreadName.c_str( ) ) );

      NORMAL_M_ASSERT_ALWAYS( );
   }

} // vAccessDatabaseIntern

tVoid spm_tclApplicationDatabase::vReleaseDatabaseIntern( ){
/*!
  * \fn
  *  \brief
  *    Method to release the lock of ApplicationDatabase.
  *
  *  \param none
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
} // vReleaseDatabaseIntern

spm_tclConnectedApp*spm_tclApplicationDatabase::poGetByAppID( tU32 u32AppId ){
/*!
  * \fn
  *  \brief
  *    Method to get a specific application
  *
  *  \param u32AppId: application ID
  *  \return pointer to the found application or NULL if no application with given ID was found
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR

   // check for app id in all blocks
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      spm_tclConnectedApp *app = it->poFind( u32AppId );
      if ( app != NULL ){
         vReleaseDatabaseIntern( );
         return( app );
      }
   }
   vReleaseDatabaseIntern( );
   return( NULL );
} // poGetByAppID

spm_tclSoftwareBlock*spm_tclApplicationDatabase::poGetSwBlockByAppID( tU32 u32AppId ){
/*!
  * \fn
  *  \brief
  *    Method to get a specific SoftwareBlock
  *
  *  \param u32AppId: application ID
  *  \return pointer to the found SoftwareBlock the application belongs to or NULL if no application with given ID was found
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR

   // check for app id in all blocks
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      spm_tclConnectedApp *app = it->poFind( u32AppId );
      if ( app != NULL ){
         spm_tclSoftwareBlock *poSWBlock = &* it;
         vReleaseDatabaseIntern( );
         return( poSWBlock );
      }
   }
   vReleaseDatabaseIntern( );
   return( NULL );
} // poGetSwBlockByAppID

spm_tclSoftwareBlock*spm_tclApplicationDatabase::poGetSoftwareBlock( const std::string& pcName ){
/*!
  * \fn
  *  \brief
  *    Method to get a specific SoftwareBlock
  *
  *  \param pcName: Name of SoftwareBlock
  *  \return pointer to the found SoftwareBlock or NULL if no Softwareblock with given name was found
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR

   // check if a sw block in the container has the requested name
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      if ( it->pcGetName( ) == pcName ){
         spm_tclSoftwareBlock *poSWBlock = &* it;
         vReleaseDatabaseIntern( );
         return( poSWBlock );
      }
   }
         vReleaseDatabaseIntern( );
   return( NULL );
} // poGetSoftwareBlock

spm_tclSoftwareBlock*spm_tclApplicationDatabase::poGetSwBlockByExecPath( const std::string& strExecPath ){
/*!
  * \fn
  *  \brief
  *    Method to get a specific SoftwareBlock
  *
  *  \param strExecPath: Process location or Execpath  of SoftwareBlock
  *  \return pointer to the found SoftwareBlock or NULL if no Softwareblock with given location/path was found
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR

   // check if a sw block in the container has the requested location/execpath
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      if ( it->oGetExecPath( ) == strExecPath ){
         spm_tclSoftwareBlock *poSWBlock = &* it;
         vReleaseDatabaseIntern( );
         return( poSWBlock );
      }
   }
   vReleaseDatabaseIntern( );
   return( NULL );
} // poGetSwBlockByExecPath

tU32 spm_tclApplicationDatabase::u32NumberOfConnectedApplications( ){
/*!
  * \fn
  *  \brief
  *    Method to get the number of known applications
  *
  *  \param none
  *  \return number of applications that are connected (known to SPM)
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   std::vector < spm_tclSoftwareBlock >::const_iterator it;
   tU32                                                 u32Number = 0;
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      u32Number += it->u32GetNumberOfApplications( );
   }
   vReleaseDatabaseIntern( );
   return( u32Number );
}

tBool spm_tclApplicationDatabase::bAllApplicationsInProfile( ISpmApplicationConfiguration *pConfig ){
/*!
  * \fn
  *  \brief
  *    ??
  *
  *  \param pConfig: pointer to ApplicationConfiguration
  *  \return True ??
  *          False ??
  *  \note
  *  \todo: Add explanation what this function is for
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   std::vector < spm_tclSoftwareBlock >::const_iterator it;
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      if ( it->bIsWholeBlockEnabled( ) ){
         if ( !it->bAllApplicationsInProfile( pConfig ) ){
            vReleaseDatabaseIntern( );
            return( FALSE );
         }
      }
   }
   vReleaseDatabaseIntern( );
   return( TRUE );
} // bAllApplicationsInProfile

tVoid spm_tclApplicationDatabase::vCheckForSwBlocksToKill( ISpmDbClient *pClient ){
/*!
  * \fn
  *  \brief
  *    ??
  *
  *  \param pClient: pointer to DataBaseClient
  *  \note
  *  \todo Add explanation what this function is for
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      if ( it->bIsWholeBlockTobeKilled( ) ){
         it->vCheckForSwBlocksToKill( pClient );
      }
   }
   vReleaseDatabaseIntern( );
}

tBool spm_tclApplicationDatabase::bAllApplicationsUpAndRunning( tU32& u32AppId ){
/*!
  * \fn
  *  \brief
  *    Method to check if all applications are running
  *
  *  \param u32AppId: application ID -> will be filled with the first application found that is not running
  *  \return true:  if no app in state UNINITIALIZED or INITIALIZED is found
  *          false: otherwise
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR

   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      if ( !it->bAllApplicationsUpAndRunning( u32AppId ) ){
         vReleaseDatabaseIntern( );
         return( FALSE );
      }
   }
   vReleaseDatabaseIntern( );
   return( TRUE );
}

tBool spm_tclApplicationDatabase::bAllApplicationsInRequestedState( ISpmDbClient *pClient ){
/*!
  * \fn
  *  \brief
  *    Method to check if all applications have reached their requested state
  *
  *  \param pClient: pointer to database client
  *  \return true if all applications are in expected state
  *          false otherwise
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   tBool fAllAppsInReqState = TRUE;

   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR

   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      if ( !it->bCheckIfWholeBlockIsInRequestedState( pClient ) ){
         fAllAppsInReqState = FALSE;
      }
   }
   vReleaseDatabaseIntern( );
   return( fAllAppsInReqState );
} // bAllApplicationsInRequestedState

tBool spm_tclApplicationDatabase::bSetAllApplicationsToProfile( ISpmApplicationConfiguration *pConfig,
                                                                ISpmDbClient                 *pClient ){
/*!
  * \fn
  *  \brief
  *    ??
  *
  *  \param pConfig: pointer to ApplicationConfiguration
  *         pClient: Pointer to DatabaseClient
  *  \return ??
  *  \note
  *  \todo Add explanation with this function is for
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR
   tBool bReturn = TRUE;
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      if ( it->bCheckIfWholeBlockIsConnected( pClient )
           && it->bIsWholeBlockEnabled( ) ){
         it->bSetAllApplicationsToProfile( pConfig, pClient );
      }
   }
   vReleaseDatabaseIntern( );
   return( bReturn );
} // bSetAllApplicationsToProfile

tBool spm_tclApplicationDatabase::bSetAllSwBlocksToNonBlockMode( ){
/*!
  * \fn
  *  \brief
  *    Method to disable BlockMode for all SoftwareBlocks
  *
  *  \param none
  *  \return TRUE always
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      it->vSetBlockMode( FALSE );
   }
   vReleaseDatabaseIntern( );
   return( TRUE );
}

tBool spm_tclApplicationDatabase::bForceAllApplicationsToProfile( tU32                          u32Profile,
                                                                  tBool                         bForceAll,
                                                                  ISpmApplicationConfiguration *pConfig,
                                                                  ISpmDbClient                 *pClient ){
/*!
  * \fn
  *  \brief
  *    ??
  *
  *  \param u32AppId: application ID
  *         bForceAll: do it for all regardless of current state
  *         pConfig: pointer to ApplicationConfiguration
  *         pClient: pointer to DatabaseClient
  *  \return TRUE always
  *  \note
  *  \todo Add explanation what this function is for
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR
   tBool bReturn = TRUE;
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      if ( it->bCheckIfWholeBlockIsConnected( pClient )
           && it->bIsWholeBlockEnabled( ) ){
         it->bForceAllApplicationsToProfile( u32Profile, bForceAll, pConfig, pClient );
      }
   }
   vReleaseDatabaseIntern( );
   return( bReturn );
} // bForceAllApplicationsToProfile

tBool spm_tclApplicationDatabase::bKillApplications( ISpmDbClient *pClient ){
/*!
  * \fn
  *  \brief
  *    Method to kill all applications in all SoftwareBlocks
  *
  *  \param pClient: pointer to DatabaseClient
  *  \return true if all applications could be killed
  *          false if an application was not yet connected and could not be killed.
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR
   tBool bReturn = FALSE;
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      it->vSetKillBlockEnabled( );

      bReturn = it->bKillAllSwBlockApplications( pClient );
   }
   vReleaseDatabaseIntern( );
   return( bReturn );
}

tBool spm_tclApplicationDatabase::bSendUpdate( tU32          u32AppId,
                                               ISpmDbClient *pClient ){
/*!
  * \fn
  *  \brief
  *    ??
  *
  *  \param u32AppId: application ID
  *         pClient: pointer to DatabaseClient
  *  \return ??
  *  \note
  *  \todo Add explanation what this function is for
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      it->bSendUpdate( u32AppId, pClient );
   }
   vReleaseDatabaseIntern( );
   return( TRUE );
}

tVoid spm_tclApplicationDatabase::vResetForceMode( tU32          u32AppId,
                                                   ISpmDbClient *pClient ){
/*!
  * \fn
  *  \brief
  *    ??
  *
  *  \param u32AppId: application ID
  *         pClient: pointer to DatabaseClient
  *  \return ??
  *  \note
  *  \todo Add explanation what this function is for
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      if ( it->bCheckIfWholeBlockIsConnected( pClient )
           && it->bIsWholeBlockEnabled( ) ){
         if ( it->bResetForceMode( u32AppId ) ){
            break;  // stop iterating if application was found
         }
      }
   }
   vReleaseDatabaseIntern( );
} // vResetForceMode

tBool spm_tclApplicationDatabase::bCheckRequests( ISpmDbClient *pClient,
                                                  tBool       & bMissingResponse,
                                                  tU32        & u32CulpritAppID,
                                                  tU32        & u32UnacknowledgedState,
                                                  tU32        & u32ActiveRequests ){
/*!
  * \fn
  *  \brief
  *    check if all application have send the responce to the last state change request
  *
  *  \param pClient: pointer to DatabaseClient
  *         bMissingResponce: filled with true if response is missing
  *         u32CulpritAppID: filled with the first application where response is missing
  *         u32UnacknowledgedState: filled with the expected state
  *         u32ActiveRequests: number that keeps increasing with each missing responce
  *  \return true if all applications have correctly responded
  *          false if any response is missing
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR
   tBool bReturn = TRUE;
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      if ( !it->bCheckRequests( pClient, bMissingResponse, u32CulpritAppID, u32UnacknowledgedState, u32ActiveRequests ) ){
         bReturn = FALSE;
      }
   }
   vReleaseDatabaseIntern( );
   return( bReturn );
}

tBool spm_tclApplicationDatabase::bCheckLifeSignals( ){
/*!
  * \fn
  *  \brief
  *    check if all application watchdog messages were recieved
  *
  *  \param true: all configured watchdogs were received
  *         false: at least one application did not send watchdog message
  *  \return
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR
   tBool bReturn = TRUE;
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      if ( !it->bCheckLifeSignals( ) ){
         bReturn = FALSE;
      }
   }
   vReleaseDatabaseIntern( );
   return( bReturn );
}

tVoid spm_tclApplicationDatabase::vOnWatchdogError( ISpmDbClient *pClient ){
/*!
  * \fn
  *  \brief
  *    function is called in case the watchdog message was not recieved (see bCheckLifeSignals)
  *
  *  \param pClient: pointer to DatabaseClient
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      it->vOnWatchdogError( pClient );
   }
   vReleaseDatabaseIntern( );
}

tVoid spm_tclApplicationDatabase::vSendCriticalVoltageEvent( tU32          u32Cvm,
                                                             ISpmDbClient *pClient ){
/*!
  * \fn
  *  \brief
  *    function sends the received critical voltage event to all SW-blocks
  *
  *  \param u32Cvm: critical voltage event
  *         pClient: pointer to DatabaseClient
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      it->vSendCriticalVoltageEvent( u32Cvm, pClient );
   }
   vReleaseDatabaseIntern( );
}

tVoid spm_tclApplicationDatabase::vTraceApplicationsInfo( ISpmDbClient *pClient ){
/*!
  * \fn
  *  \brief
  *    helper function to get traces about applications
  *
  *  \param pClient: pointer to DatabaseClient
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      pClient->vOnTraceSwBlockInfo( * it );
      it->vTrace( pClient );
   }
   vReleaseDatabaseIntern( );
}

spm_tclSoftwareBlock*spm_tclApplicationDatabase::poAddSoftwareBlock( const std::string& name ){
/*!
  * \fn
  *  \brief
  *    add base sw block to container
  *
  *  \param name: name of the new SoftwareBlock
  *  \return pointer to the new SoftwareBlock
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR

                         it        = _vecSwBlock.insert( _vecSwBlock.end( ), name );
   spm_tclSoftwareBlock *poSWBlock = &* it;
   vReleaseDatabaseIntern( );
   return( poSWBlock );
}

tVoid spm_tclApplicationDatabase::vRemoveSoftwareBlock( const std::string& name ){
/*!
  * \fn
  *  \brief
  *    remove a SoftwareBlock
  *
  *  \param name: name of the SoftwareBlock to be removed
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      if ( it->pcGetName( ) == name ){
         // iterate through all applications to set supplier state to UNKNOWN+
         it->bDeleteAllSwBlockApplicationsFromSupplier( _poclCcaSupplierHandler );
         _vecSwBlock.erase( it );
         break;
      }
   }
   vReleaseDatabaseIntern( );
} // vRemoveSoftwareBlock

tVoid spm_tclApplicationDatabase::vGetSwBlocksOfProcess( const std::string        & strProcessName,
                                                         std::list < std::string > *poSwBlockList ){
/*!
  * \fn
  *  \brief
  *    get a list of SoftwareBlocks of a given process
  *
  *  \param strProcessName: name of the process
  *         poSwBlcokList: filled with all SoftwareBlocks found for this process name
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      std::string strProcess = it->pcGetName( );
      std::string strSwBlock = it->oGetSwBlockName( );
      ETG_TRACE_USR1( ( "spm_tclApplicationDatabase::vGetSwBlocksOfProcess(): SwBlock '%40s' Compare '%50s' <-> '%50s'", strSwBlock.c_str( ), strProcessName.c_str( ), strProcess.c_str( ) ) );
      if ( strProcess == strProcessName ){
         ETG_TRACE_USR1( ( "spm_tclApplicationDatabase::vGetSwBlocksOfProcess():Add SwBlock to list '%50s'", strSwBlock.c_str( ) ) );
         poSwBlockList->push_back( strSwBlock );
      }
   }
   vReleaseDatabaseIntern( );
} // vGetSwBlocksOfProcess

tVoid spm_tclApplicationDatabase::vTraceSwBlocksInfo( ISpmDbClient *pClient ){
/*!
  * \fn
  *  \brief
  *    helper function to trace info about all SoftwareBlocks
  *
  *  \param pClient: pointer to DatabaseClient
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      pClient->vOnTraceSwBlock( * it );
   }
   vReleaseDatabaseIntern( );
}

tVoid spm_tclApplicationDatabase::vTraceSwBlockInfo( const std::string& pcName,
                                                     ISpmDbClient      *pClient ){
/*!
  * \fn
  *  \brief
  *    helper function to trace info about a specific SoftwareBlock
  *
  *  \param name: name of the SoftwareBlock
  *         pClient: pointer to DatabaseClient
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   vAccessDatabaseIntern( );

   SPM_APP_DATABASE_SWBLOCK_ITERATOR

   // check if a sw block in the container has the requested name
   for ( it = _vecSwBlock.begin( ); it != _vecSwBlock.end( ); ++it ){
      if ( it->pcGetName( ) == pcName ){
         pClient->vOnTraceSwBlock( * it );
      }
   }
   vReleaseDatabaseIntern( );
}

