/*!
  * \file spm_SCCSocketInterface.cpp
  *  \brief
  *    Implementation of Socket Interface to communicate with System Communication Controller(SCC) through Inter Node Communication(INC).
  *
  *  \b PROJECT: NextGen \n
  *  \b SW-COMPONENT: FC SPM \n
  *  \b COPYRIGHT:    (c) 2011-2017 Robert Bosch GmbH, Hildesheim \n
  *  \version
  * 1.0 |  06.05.13  | Bipin Krishnan(RBEI/ECG4)     | initial version for thermal management \n
  * 1.1 |  14.06.13  | Bipin Krishnan(RBEI/ECG4)     | adaptation for supply management \n
  * 1.2 |  26.06.13  | Bipin Krishnan(RBEI/ECG4)     | redesign of SCC Client Handlers \n
  * 1.3 |  11.05.16  | Ravi Kiran Gundeti (RBEI/ECF5)| redesign of SCC Client Handlers
  ******
  */
#include <errno.h>

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

#include "spm_Config.h"

#include "spm_SCCIncMessage.h"
#include "spm_SCCSocketInterface.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_SCCSocketInterface.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

spm_tclSCCSocketInterface::spm_tclSCCSocketInterface( tU8                u8SocketFamily,
                                                      tS32               s32Portno ){
/*!
  * \fn
  *  \brief
  *    spm_tclSCCSocketInterface Constructor
  *
  *  \param[in] u8SocketFamily: Communication Domain/Socket Family.
  *  \param[in] s32Portno: Port Number specific to SCC Client Handler.
  ******
  */

   dp_tclSpmDpConfigFakeDeviceConfig bFakeDeviceConnect;
   dp_tclSpmDpThreadThreadConfPrioSCCRx  oPrio;
   dp_tclSpmDpThreadThreadConfStackSCCRx oStacksize;

   tBool bConnect2FakeDevice = bFakeDeviceConnect.tGetData( );
   ETG_TRACE_USR4( ( "spm_tclSCCSocketInterface:spm_tclSCCSocketInterface u8SocketFamily = %d, s32Portno = %d",u8SocketFamily, s32Portno ) );
   if ( bConnect2FakeDevice ){
      ETG_TRACE_ERRMEM( ( "spm_tclSCCSocketInterface::spm_tclSCCSocketInterface: Connecting to Fake Device." ) );
   }

   _u8SocketFamily = u8SocketFamily;
   _s32Portno = s32Portno;

   _poDatagramSocket = new DatagramSocket(this, oStacksize.tGetData( ), oPrio.tGetData( ), s32Portno, u8SocketFamily, !bConnect2FakeDevice, bConnect2FakeDevice, CONNECT_TO_CLIENT);
   SPM_NULL_POINTER_CHECK( _poDatagramSocket );
   ETG_TRACE_USR3( ( "spm_tclSCCSocketInterface::spm_tclSCCSocketInterface: bConnect2FakeDevice= %u", ETG_ENUM( SPM_BOOL_STATE, bConnect2FakeDevice ) ) );
}

spm_tclSCCSocketInterface::~spm_tclSCCSocketInterface( ){
/*!
  * \fn
  *  \brief
  *    spm_tclSCCSocketInterface Destructor
  *
  *  \details
  *   Terminates the INC communication with System Communication Controller and closes the socket connection.
  ******
  */

   if( OSAL_NULL != _poDatagramSocket) {
      OSAL_DELETE _poDatagramSocket;
   }
   _poDatagramSocket = OSAL_NULL;
}

tS32 spm_tclSCCSocketInterface::s32SendToINCSocket( spm_tclSCCMessage *poSCCMessage ){
/*!
  * \fn
  *  \brief
  *   Method to send a PDU(message) to System Communication Controller.
  *  \param[in]
  *     poSCCMessage: Message to be sent to System Communication Controller.
  *  \note
  *  spm_tclSCCSocketInterface uses dgram_service library API to send a PDU(message) to System Communication Controller.
  ******
  */
   tS32 s32Ret    = SCC_ERROR;

   SPM_NULL_POINTER_CHECK_VAL( poSCCMessage );
   tU8 *pu8Data   = poSCCMessage->pu8GetMsgData( );
   tU16 u16MsgLen = poSCCMessage->u16GetMsgLength( );

   ETG_TRACE_USR1(("spm_tclSCCSocketInterface::s32SendToINCSocket(): Called.... ."));
   if (_poDatagramSocket) {
      ETG_TRACE_USR1(("spm_tclSCCSocketInterface::s32SendToINCSocket Sent message (MsgId = 0x%x ) to SCC with %d bytes", (tU8)pu8Data[0], u16MsgLen));
      s32Ret = _poDatagramSocket->sendMessage((void*)pu8Data, u16MsgLen);
   }
   return( s32Ret );

}

void spm_tclSCCSocketInterface::onNewDatagramMessage(void* data, size_t /*length*/){
/*!
  * \fn
  *  \brief
  *   Receive new datagram message and trigger process function.
  *  \param[in]
  *     data: Received data.
  *  \note
  ******
  */

   // Process the Message from System Communication Controller
   SPM_NULL_POINTER_CHECK( data );
   vProcessSCCMessage( static_cast < tU8* > ( data ) );
}

void spm_tclSCCSocketInterface::vSetupConnectionToFD( tU8 bIsConnectToFD ){

/*!
  * \fn
  *  \brief
  *   Create connection with fake device during runtime of LCM.
  *  \param[in]
  *     bIsConnectToFD: Connect or disconnect with fake device.
  *  \note
  ******
  */

   dp_tclSpmDpThreadThreadConfPrioSCCRx  oPrio;
   dp_tclSpmDpThreadThreadConfStackSCCRx oStacksize;

   ETG_TRACE_USR4(("spm_tclSCCSocketInterface::vSetupConnectionToFD _poDatagramSocket is deleting"));
   if( OSAL_NULL != _poDatagramSocket) {
      OSAL_DELETE _poDatagramSocket;
   }
   _poDatagramSocket = OSAL_NULL;
   ETG_TRACE_USR4(("spm_tclSCCSocketInterface::vSetupConnectionToFD _poDatagramSocket is deleted"));
   _poDatagramSocket = new DatagramSocket(this, oStacksize.tGetData( ), oPrio.tGetData( ), _s32Portno, _u8SocketFamily, !bIsConnectToFD, bIsConnectToFD, CONNECT_TO_CLIENT);
   SPM_NULL_POINTER_CHECK( _poDatagramSocket );
   ETG_TRACE_USR4(("spm_tclSCCSocketInterface::vSetupConnectionToFD"));
}

