/*!
  * \file spm_LocalApplicationManagerConfig.cpp
  *  \brief
  *    Project specific part of the power message handling.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 25.07.11  | TMS Fischer       | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_STDVISITORS
#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_FUNCTIONIDS
#include "spm_fi_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// SPM  configuration
#include "spm_Config.h"
#include "spm_GlobDefs.h"

// my class header
#include "spm_LocalApplicationManagerConfig.h"

// spm class definitions
#include "spm_ApplicationRequestSupervisor.h"
#include "spm_ApplicationConfigurationConfig.h"

// interfaces class definitions
#include "spm_ISubStateClient.h"
#include "spm_ICcaSupplierServer.h"
#include "spm_ICcaServiceServer.h"
#include "spm_IStartupInvestigationServer.h"
#include "spm_IOsalProxy.h"
#include "spm_ISystemStateManager.h"
#include "spm_ISystemPowerManager.h"
#include "spm_IGlobalApplicationManager.h"
#include "spm_IApplicationErrorHandler.h"

#include "spm_IFactory.h"

// spm helper
#include "spm_SoftwareBlock.h"
#include "spm_ConnectedApps.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_LAM
    #include "trcGenProj/Header/spm_LocalApplicationManagerConfig.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"


// /////////////////////////////////////////////////////////////////////////////
// \todo: refactoring:
// check all bIf... functions if they are really const and do not have side
// effects. Its quite irritating to see that those function manipulate data
// /////////////////////////////////////////////////////////////////////////////


/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
// //#define SPM_TRACE_FILE_ID   SPM_FILE_LOCALAPPMANAGER

#ifndef SPM_PRJ_SYSTEM_STATE_DOWNLOAD
   #define SPM_PRJ_SYSTEM_STATE_DOWNLOAD    SPM_SYSTEM_DOWNLOAD
#endif

spm_tclLocalApplicationManagerConfig::spm_tclLocalApplicationManagerConfig( const ISpmFactory& factory,
                                                                            tU32               u32AppId,
                                                                            tU8                u8LamConfigID )
   : spm_tclLocalApplicationManager( factory, u32AppId, u8LamConfigID ){
   fOverTempActive         = FALSE;
   _bNaviConnected         = FALSE;
   _bDapiConnected         = FALSE;
   _bMapConnected          = FALSE;

   _bStartupEjectProcessed = FALSE;

   _poclCcaServiceHandler  = NULL;
   _bAllowedToDump         = TRUE; // Allowed to dump startup investigation once
}

spm_tclLocalApplicationManagerConfig::~spm_tclLocalApplicationManagerConfig( ){
   _poclCcaServiceHandler = NULL;
}

tVoid spm_tclLocalApplicationManagerConfig::vGetReferences( ){
   spm_tclLocalApplicationManager::vGetReferences( );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclCcaServiceHandler, ISpmCcaServiceServer );
}

tVoid spm_tclLocalApplicationManagerConfig::vOnCheckForSwBlocksToKill( tU32 u32AppId ){
   if ( _poclApplicationRequestSupervisor ){
      _poclApplicationRequestSupervisor->vStartSupervision( );
   }
   if ( _poclCcaSupplierHandler && SPM_FRAMEWORK_IS_APP_CCA( u32AppId ) ){
      _poclCcaSupplierHandler->vSupplierStateChanged( (tU16)u32AppId, AMT_C_U8_CCAMSG_SUPPLIER_STATE_UNAVAILABLE );
   }
   vPostPowerMessage( u32AppId,
                      AMT_C_U16_PWR_PROXY_END_APP,
                      0,
                      0 );

   ETG_TRACE_USR4( ( "vOnCheckForSwBlocksToKill, request AMT_C_U16_PWR_PROXY_END_APP state app: %u",
                     ETG_ENUM( ail_u16AppId, (tU16)u32AppId )
                     ) );
} // vOnCheckForSwBlocksToKill

/**
  *  \brief Called when the SW-Block is loaded/spawned
  *
  *  \param [in] strSwBlockName Parameter_Description
  *  \return none
  *
  *  \details It also adds the new SW-Block to the list of Connected SW-Blocks
  */
tVoid spm_tclLocalApplicationManagerConfig::vOnWholeSwBlockIsLoaded( const std::string& strSwBlockName ){

   _oLoadedState.insert( strSwBlockName );

   ETG_TRACE_USR4( ( "vOnWholeSwBlockIsLoaded, Block %s is loaded now (all apps have reached INIT)",
                     strSwBlockName.c_str( )
                     ) );

   std::string                    output;
   TSpmSwBlockSet::const_iterator it;

   for ( it = _oLoadedState.begin( ); it != _oLoadedState.end( ); ++it ){
      output += * it;
      output += " ";
   }

   ETG_TRACE_USR4( ( "BlockIsLoadedList: %s", output.c_str( ) ) );

   std::set < ISpmSwBlockClient* >::iterator it2;
   for ( it2 = _clients.begin( ); it2 != _clients.end( ); ++it2 ){
      ( * it2 )->vOnSwBlockLoaded( _oLoadedState );
   }
} // vOnWholeSwBlockIsLoaded

/**
  *  \brief Called when all applications in the SW-Block are registered with LCM
  *
  *  \param [in] strSwBlockName Parameter_Description
  *  \return none
  *
  *  \details this function sets a couple of internal states for specific SW-Blocks (NAVAPP, DAPIAPP and MAP).
  *  It also adds the new SW-Block to the list of Connected SW-Blocks
  */
tVoid spm_tclLocalApplicationManagerConfig::vOnWholeSwBlockIsConnected( const std::string& strSwBlockName ){
   SPM_NULL_POINTER_CHECK( _poclCcaServiceHandler );

   _oConnectedState.insert( strSwBlockName );

      ETG_TRACE_USR4( ( "vOnWholeSwBlockIsConnected, Block %s is connected now (all apps have reached INIT)",
                     strSwBlockName.c_str( )
                     ) );


   if ( strSwBlockName == "NAVAPP" ){
      _bNaviConnected = TRUE;
   }
   if ( strSwBlockName == "DAPIAPP" ){
      _bDapiConnected = TRUE;
   }
   if ( strSwBlockName == "MAP" ){
      _bMapConnected = TRUE;
      #if OSAL_CONF == OSAL_LINUX
         if ( !_bDapiConnected ){
            // set dapi to Connected for LSIM as long as there is no DAPI process
            _bDapiConnected = TRUE;
         }
      #endif
   }

   if ( _bNaviConnected && _bDapiConnected && _bMapConnected ){
      // send StartupState update -> navi is ready
      ETG_TRACE_USR4( ( "vOnWholeSwBlockIsInRequestedState, send StartupState update -> FI_EN_SPM_U32_STARTUP_STATE_NAVI is ready" ) );

      spm_corefi_tclMsgStartupStateStatus oStartupState;
      oStartupState.StartupState.enType = spm_fi_tcl_SPM_e32_STARTUP_STATE::FI_EN_SPM_U32_STARTUP_STATE_NAVI;

      _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_STARTUPSTATE, &oStartupState );
   }

   std::string                    output;
   TSpmSwBlockSet::const_iterator it;

   for ( it = _oConnectedState.begin( ); it != _oConnectedState.end( ); ++it ){
      output += * it;
      output += " ";
   }

      ETG_TRACE_USR4( ( "BlockIsConnectedList: %s", output.c_str( ) ) );

   std::set < ISpmSwBlockClient* >::iterator it2;
   for ( it2 = _clients.begin( ); it2 != _clients.end( ); ++it2 ){
      ( * it2 )->vOnSwBlockConnected( _oConnectedState );
   }
} // vOnWholeSwBlockIsConnected

/**
  *  \brief Called when all applications in the SW-Block are in the required application state of current SystemState
  *
  *  \param [in] strSwBlockName Parameter_Description
  *  \return none
  *
  *  \details this function handles MediaEject events specially for PROCHMI
  *  It also adds the new SW-Block to the list of Up SW-Blocks
  */
tVoid spm_tclLocalApplicationManagerConfig::vOnWholeSwBlockIsInRequestedState( const std::string& strSwBlockName ){
   SPM_NULL_POINTER_CHECK( _poclCcaServiceHandler );
   SPM_NULL_POINTER_CHECK( _poclSubStateHandler );

   _oStartUpRunState.insert( strSwBlockName );

      ETG_TRACE_USR4( ( "vOnWholeSwBlockIsInRequestedState, Block %s is connected now (all apps have reached NORMAL/OFF/...)",
                     strSwBlockName.c_str( )
                     ) );

   if ( ( strSwBlockName == "PROCHMI" ) && ( !_bStartupEjectProcessed ) && ( SPM_BC0_EJECT & _poclSubStateHandler->u32GetStartupOnReason( ) ) ){
      // check for EJECT as wakeup reason
      spm_corefi_tclMsgMediaEjectStateStatus oEjectStatus;
      oEjectStatus.MediaEjectState.enType = spm_fi_tcl_SPM_e32_MEDIA_EJECT_STATE::FI_EN_SPM_U32_EJECT_PRESSED;

      _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_MEDIAEJECTSTATE, &oEjectStatus );
      oEjectStatus.MediaEjectState.enType = spm_fi_tcl_SPM_e32_MEDIA_EJECT_STATE::FI_EN_SPM_U32_EJECT_RELEASED;
      _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_MEDIAEJECTSTATE, &oEjectStatus );

      _bStartupEjectProcessed             = TRUE;
   }

   TSpmSwBlockSet::const_iterator it;

   std::string                    output;
   for ( it = _oStartUpRunState.begin( ); it != _oStartUpRunState.end( ); ++it ){
      output += * it;
      output += " ";
   }

      ETG_TRACE_USR4( ( "BlockIsUpList: %s", output.c_str( ) ) );

   std::set < ISpmSwBlockClient* >::iterator it2;
   for ( it2 = _clients.begin( ); it2 != _clients.end( ); ++it2 ){
      ETG_TRACE_USR4( ( "BlockIsUpList: Trigger block up!!" ) );
      ( * it2 )->vOnSwBlockUp( _oStartUpRunState );
   }
} // vOnWholeSwBlockIsInRequestedState

tVoid spm_tclLocalApplicationManagerConfig::vOnKill( ){
   SPM_NULL_POINTER_CHECK( _poclSubStateHandler );

   _poclSubStateHandler->vSetSubStateType( spm_fi_tcl_SPM_e32_SubStateType::FI_EN_SPM_U32_SUBSTATE_SYSSTATE_CHECK, FALSE );
}

tVoid spm_tclLocalApplicationManagerConfig::vOnSetAllApplicationsToProfile( tU32  u32AppId,
                                                                            tU32  u32AppStateRequest,
                                                                            tBool bBlockMode ){
   if ( _poclApplicationRequestSupervisor ){
      _poclApplicationRequestSupervisor->vStartSupervision( );
      // support startup investigation
   }
   _poclStartupInvest->vUpdateApplicationItem( u32AppId,
                                               ISpmStartupInvestigationServer::SPM_STARTUP_APPLICATION_STATEREQ,
                                               u32AppStateRequest );
   vPostPowerMessage( u32AppId,
                      AMT_C_U16_PWR_PROXY_STATE_CHANGE_REQ,
                      u32AppStateRequest,
                      0 );

   if ( bBlockMode ){
      ETG_TRACE_USR4( ( "bHandleAppStateChangeAck, request new application state app: %u, state: %u",
                        ETG_ENUM( ail_u16AppId,     u32AppId ),
                        ETG_ENUM( SPM_APPSTATE_DEF, u32AppStateRequest )
                        ) );
   } else {
                        ETG_TRACE_USR4( ( "vOnSetAllComponentsInProfile, request new application state app: %u, state: %u",
                        ETG_ENUM( ail_u16AppId,     u32AppId ),
                        ETG_ENUM( SPM_APPSTATE_DEF, u32AppStateRequest )
                        ) );
   }
} // vOnSetAllApplicationsToProfile

tVoid spm_tclLocalApplicationManagerConfig::vOnKillAllSwBlockApplications( tU32 u32AppId ){
   if ( _poclApplicationRequestSupervisor ){
      _poclApplicationRequestSupervisor->vStartSupervision( );
   }
   if ( _poclCcaSupplierHandler && SPM_FRAMEWORK_IS_APP_CCA( u32AppId ) ){
      _poclCcaSupplierHandler->vSupplierStateChanged( (tU16)u32AppId, AMT_C_U8_CCAMSG_SUPPLIER_STATE_UNAVAILABLE );
   }
   vPostPowerMessage( u32AppId,
                      AMT_C_U16_PWR_PROXY_END_APP,
                      0,
                      0 );

   ETG_TRACE_USR4( ( "vOnKillAllSwBlockApplications, request AMT_C_U16_PWR_PROXY_END_APP state app: %u",
                     ETG_ENUM( ail_u16AppId, (tU16)u32AppId )
                     ) );
} // vOnKillAllSwBlockApplications

tVoid spm_tclLocalApplicationManagerConfig::vOnSendUpdate( tU32  u32AppId,
                                                           tU32  u32RequestedState,
                                                           tU32  u32AppStateRequest,
                                                           tU32  u32AppStateNewRequest,
                                                           tBool bBlockmode ){
   if ( _poclApplicationRequestSupervisor ){
      _poclApplicationRequestSupervisor->vStartSupervision( );
   }
   _poclStartupInvest->vUpdateApplicationItem( u32AppId,
                                               ISpmStartupInvestigationServer::SPM_STARTUP_APPLICATION_STATEREQ,
                                               bBlockmode ? u32AppStateNewRequest : u32RequestedState );

   vPostPowerMessage( u32AppId,
                      AMT_C_U16_PWR_PROXY_STATE_CHANGE_REQ,
                      u32RequestedState,
                      0 );

   ETG_TRACE_USR4( ( "vOnSendUpdate, request new application state app: %u, state: %u",
                     ETG_ENUM( ail_u16AppId,     (tU32)u32AppId ),
                     ETG_ENUM( SPM_APPSTATE_DEF, u32AppStateRequest )
                     ) );

} // vOnSendUpdate

tVoid spm_tclLocalApplicationManagerConfig::vOnForceAllApplicationsToProfile( tU32  u32AppId,
                                                                              tU32  u32NewAppState,
                                                                              tU32  u32AppStateRequest,
                                                                              tBool /*bBlockmode*/ ){
   if ( _poclApplicationRequestSupervisor ){
      _poclApplicationRequestSupervisor->vStartSupervision( );
   }
   _poclStartupInvest->vUpdateApplicationItem( u32AppId,
                                               ISpmStartupInvestigationServer::SPM_STARTUP_APPLICATION_STATEREQ,
                                               u32NewAppState );

   vPostPowerMessage( u32AppId,
                      AMT_C_U16_PWR_PROXY_STATE_CHANGE_REQ,
                      u32NewAppState,
                      0 );

   ETG_TRACE_USR4( ( "vOnForceAllApplicationsToProfile, request new application state app: %u, state: %u",
                     ETG_ENUM( ail_u16AppId,     (tU32)u32AppId ),
                     ETG_ENUM( SPM_APPSTATE_DEF, u32AppStateRequest )
                     ) );
} //lint !e715 Symbol 'bBlockmode' not referenced --> CURRENTLY not used

tVoid spm_tclLocalApplicationManagerConfig::vOnCheckRequests( tU32& u32ActiveRequests ){
   dp_tclSpmDpInternDataSystemState oNewSysState;

   // we are already waiting for shutdown
   // -> HW WDT trigger is finished
   // -> need to trigger WDT here
   // -> supervision is by application monitoring
   if ( ( oNewSysState.tGetData( ) == spm_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_OFF )
        || ( oNewSysState.tGetData( ) == spm_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_SUSPEND ) ){
      ETG_TRACE_USR4( ( "vOnCheckRequests -> Trigger WDT." ) );
      _poclOsalProxy->bTriggerWatchdog( );
   }

   ++u32ActiveRequests; // seen a request more
} // vOnCheckRequests

/*!
  * \fn
  *  \brief
  *   Function is logs the failure of the application to acknowledge state change
  *   request.
  *  \param[in] app: Connected applications.
  * \details
  *  Application error handler is triggered to log the further info for the application
  *  state change failure like the Reset Reason, Current and new System state,app state etc
  *  into errmem.If the application is found to be initialization phase then the corresponding
  *  message is logged along with the Startup investigation traces.
  *********
  */
tVoid spm_tclLocalApplicationManagerConfig::vOnUnacknowledgeState( spm_tclConnectedApp& app ){
   ETG_TRACE_FATAL( ( "vOnUnacknowledgedState, app %u didn't reply with state: %u",
                      ETG_ENUM( ail_u16AppId,     app.u32GetAppId( ) ),
                      ETG_ENUM( SPM_APPSTATE_DEF, app.u32GetAppStateRequest( ) )
                      ) );

   /* Handling to avoid multiple write of the application state into errormemory before
      reset is really performend. Its enough to write the SHUTDOWN_AFTER_NO STATE request
      error message & callstacks once.
     */
   if ( app.u32GetFatalErrorState( ) != SPM_U32_SHUTDOWN_AFTER_NO_STATE_REQUEST ){
      //vWriteApplicationErrorToErrmem(app, SPM_U32_SHUTDOWN_AFTER_NO_STATE_REQUEST, SPM_U8_ERRMEM_TYPE_STATE_REQUEST);
      SPM_GET_IF_REFERENCE_NEW_VAR( poclAppErrHandler, ISpmApplicationErrorHandler );
      poclAppErrHandler->vStateChangeFailed( app );

      app.vSetFatalErrorState( SPM_U32_SHUTDOWN_AFTER_NO_STATE_REQUEST );
   }

   // Is application still in initialization phase? Then perhaps the corresponding process
   // could not be started. Therefore make a dump of the startup investigation into errmem.
   if ( _bAllowedToDump && ( ( app.u32GetAppStateRequest( ) == AMT_C_U32_STATE_INVALID )
                             || ( app.u32GetAppStateRequest( ) == AMT_C_U32_STATE_INITIALIZED ) ) ){
      std::string strPrintingReadyThreads = "Missing application INITIALIZED during startup detected, printing out READY_THREADs:";
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strPrintingReadyThreads.c_str( ), (tU16)OSAL_u32StringLength( strPrintingReadyThreads.c_str( ) ) );
      _poclStartupInvest->vDump( );
      _bAllowedToDump = FALSE;
   }

} // vOnUnacknowledgeState

tVoid spm_tclLocalApplicationManagerConfig::vOnSendImmediateStateChange( spm_tclConnectedApp& app ){
   vPostPowerMessage( app.u32GetAppId( ),
                      AMT_C_U16_PWR_PROXY_STATE_CHANGE_IMMEDIATELY,
                      app.u32GetAppStateRequest( ),
                      0 );

   ETG_TRACE_FATAL( ( "vOnSendImmediateStateChange, request (AMT_C_U16_PWR_PROXY_STATE_CHANGE_IMMEDIATELY) new application state app: %u state: %u",
                      ETG_ENUM( ail_u16AppId,     app.u32GetAppId( ) ),
                      ETG_ENUM( SPM_APPSTATE_DEF, app.u32GetAppStateRequest( ) )
                      ) );

                      SPM_GET_IF_REFERENCE_NEW_VAR( poclAppErrHandler, ISpmApplicationErrorHandler );
   poclAppErrHandler->vStateChangeImmediately( app );

   //vWriteApplicationErrorToErrmem(app, 0, SPM_U8_ERRMEM_TYPE_STATE_REQUEST_IMM);

} // vOnSendImmediateStateChange

tVoid spm_tclLocalApplicationManagerConfig::vOnTerminateMissing( spm_tclConnectedApp& app ){
   ETG_TRACE_FATAL( ( "vOnTerminateMissing, request (AMT_C_U16_PWR_PROXY_STATE_CHANGE_IMMEDIATELY) new application state app: %u state: %u",
                      ETG_ENUM( ail_u16AppId,     app.u32GetAppId( ) ),
                      ETG_ENUM( SPM_APPSTATE_DEF, app.u32GetAppStateRequest( ) )
                      ) );

                      SPM_GET_IF_REFERENCE_NEW_VAR( poclAppErrHandler, ISpmApplicationErrorHandler );
   poclAppErrHandler->vTerminateChangeFailed( app );

} // vOnTerminateMissing

tVoid spm_tclLocalApplicationManagerConfig::vOnWatchdogError( spm_tclConnectedApp& app ){
   ETG_TRACE_FATAL( ( "vOnWatchdogError, Watchdog Error Component: %u",
                      ETG_ENUM( ail_u16AppId, app.u32GetAppId( ) )
                      ) );
}

tVoid spm_tclLocalApplicationManagerConfig::vOnWatchdogErrorNotifyProblemReport( spm_tclConnectedApp& app ){
   // it is not possible to repaire the system -> shutdown system
   ETG_TRACE_FATAL( ( "vOnWatchdogErrorNotifyProblemReport, application watchdog error detected for app: %u",
                      ETG_ENUM( ail_u16AppId, app.u32GetAppId( ) )
                      ) );

   SPM_GET_IF_REFERENCE_NEW_VAR( poclAppErrHandler, ISpmApplicationErrorHandler );
   poclAppErrHandler->vWdgNotifyProblem( app, TRUE );

} // vOnWatchdogErrorNotifyProblemReport

tVoid spm_tclLocalApplicationManagerConfig::vOnWatchdogErrorNotifyProblemReportNew( spm_tclConnectedApp& app ){
   vPostPowerMessage( app.u32GetAppId( ),
                      AMT_C_U16_PWR_SVM_APP_NOTIFY_PROBLEM,
                      0,
                      0 );
   ETG_TRACE_USR4( ( "vOnWatchdogErrorNotifyProblemReportNew, send AMT_C_U16_PWR_SVM_APP_NOTIFY_PROBLEM event to %u, time: %dms",
                     ETG_ENUM( ail_u16AppId,   app.u32GetAppId( ) ),
                     OSAL_ClockGetElapsedTime( )
                     ) );
}

tVoid spm_tclLocalApplicationManagerConfig::vOnSendCriticalVoltageEvent( spm_tclConnectedApp& app,
                                                                         tU32                 u32Cvm ){
   vPostPowerMessage( app.u32GetAppId( ),
                      AMT_C_U16_PWR_PROXY_CVM_SIGNAL_CHANGED,
                      u32Cvm,
                      u32Cvm );

   ETG_TRACE_USR4( ( "vSendCriticalVoltageEvent, send critical voltage event (%u) to %u, time: %dms",
                     ETG_ENUM( CRITICAL_EVENT, u32Cvm ),
                     ETG_ENUM( ail_u16AppId,   app.u32GetAppId( ) ),
                     OSAL_ClockGetElapsedTime( )
                     ) );
}

tVoid spm_tclLocalApplicationManagerConfig::vOnSetAlternativeApplicationConfiguration( tU32 u32SystemState,
                                                                                       tU32 u32Configuration ){

   // Write project specific stuff here if the application configuration was changed.
   // Maybe supervision.
   std::vector < std::string > output;

   if ( _poclAppCfg ){
      _poclAppCfg->vGetSpecificSupervisionConfiguration( u32SystemState, u32Configuration, output );
   }
   if ( output.size( ) == 0 ){
      ETG_TRACE_USR4( ( "Supervision set is empty" ) );

      /* Empty "output" set forces bChangeProcessSupervision() to change the supervision flag
         for all supervised processes to TRUE (TRUE=Supervision active*/
      // _poclOsalProxy->bChangeProcessSupervision(output, TRUE);

   } else {
      /* Sets the supervision active flag for all processes in the list to FALSE*/
      // _poclOsalProxy->bChangeProcessSupervision(output, FALSE);
   }


} // vOnSetAlternativeApplicationConfiguration

tVoid spm_tclLocalApplicationManagerConfig::vOnTraceApplicationInfo( spm_tclConnectedApp& app ){
   ETG_TRACE_FATAL( ( "vOnTraceApplicationInfo: %d application %u, current state %u, requested state %u, application watchdog interval %u",
                      ETG_ENUM( SPM_APPLICATION_PLACEMENT, app.bIsLocal( ) ),
                      ETG_ENUM( ail_u16AppId,              app.u32GetAppId( ) ),
                      ETG_ENUM( SPM_APPSTATE_DEF,          app.u32GetAppStateCurrent( ) ),
                      ETG_ENUM( SPM_APPSTATE_DEF,          app.u32GetAppStateRequest( ) ),
                      app.u32GetNotifyInterval( )
                      ) );
}

tVoid spm_tclLocalApplicationManagerConfig::vOnTraceSwBlock( spm_tclSoftwareBlock& sb ){
   ETG_TRACE_FATAL( ( "ProcessName                   = %s", sb.pcGetName( ).c_str( ) ) );
   ETG_TRACE_FATAL( ( "SwBlockName                   = %s", sb.oGetSwBlockName( ).c_str( ) ) );
   ETG_TRACE_FATAL( ( "ProcessID                     = %d", sb.u32GetPID( ) ) );
   ETG_TRACE_FATAL( ( "CheckLifeSignals              = %d", sb.bCheckLifeSignals( ) ) );
   ETG_TRACE_FATAL( ( "IsBlockMode                   = %d", sb.bIsBlockMode( ) ) );
   ETG_TRACE_FATAL( ( "IsConnected                   = %d", sb.bIsConnected( ) ) );
   ETG_TRACE_FATAL( ( "IsInSystem                    = %d", sb.bIsInSystem( ) ) );
   ETG_TRACE_FATAL( ( "IsWholeBlockConnected         = %d", sb.bIsWholeBlockConnected( ) ) );
   ETG_TRACE_FATAL( ( "IsWholeBlockEnabled           = %d", sb.bIsWholeBlockEnabled( ) ) );
   ETG_TRACE_FATAL( ( "IsWholeBlockToBeKilled        = %d", sb.bIsWholeBlockTobeKilled( ) ) );
   ETG_TRACE_FATAL( ( "IsWholeBlockUp                = %d", sb.bIsWholeBlockUp( ) ) );
   ETG_TRACE_FATAL( ( "NumberOfApplications          = %d", sb.u32GetNumberOfApplications( ) ) );
   ETG_TRACE_FATAL( ( "NumberOfConnectedApplications = %d", sb.u32NumberOfConnectedApplications( ) ) );
}

tVoid spm_tclLocalApplicationManagerConfig::vOnTraceSwBlockInfo( spm_tclSoftwareBlock& sb ){
   ETG_TRACE_FATAL( ( "vOnTraceSwBlockInfo, PID %d and SwBlockName = %s )",
                      sb.u32GetPID( ),
                      sb.oGetSwBlockName( ).c_str( )
                      ) );
}

tVoid spm_tclLocalApplicationManagerConfig::vAddClient( ISpmSwBlockClient *clt ){
   _clients.insert( clt );
}

tVoid spm_tclLocalApplicationManagerConfig::vRemoveClient( ISpmSwBlockClient *clt ){
   std::set < ISpmSwBlockClient* >::iterator it;

   it = _clients.find( clt );
   if ( it != _clients.end( ) ){
      _clients.erase( it );
   }
}

tBool spm_tclLocalApplicationManagerConfig::bHandleSynchrounousCall( tU32   u32Message,
                                                                     tVoid *args ){
   tBool fValetMode = FALSE;

   switch ( u32Message ){
      case SPM_U32_WORKER_BROADCAST_VALET_MODE:
      {
         if ( args ){
            fValetMode = * (tBool*)args;
         }
         ETG_TRACE_USR4( ( "spm_tclLocalApplicationManagerConfig::bHandleSynchrounousCall(SPM_U32_WORKER_BROADCAST_VALET_MODE) ValetMode=%u", fValetMode ) );
         if ( fValetMode ){
            vSetAlternativeApplicationConfiguration( spm_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_ON, 1 );
            // _poclSubStateHandler->vSetInternalSubState(SPM_U32_INTERNAL_SUBSTATE_VALET_MODE, SPM_U32_INTERNAL_SUBSTATE_VALET_MODE);
         } else {
            vSetAlternativeApplicationConfiguration( spm_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_ON, 0 );
            // _poclSubStateHandler->vSetInternalSubState(SPM_U32_INTERNAL_SUBSTATE_VALET_MODE, 0);
         }
      }
         return( TRUE );

      case SPM_U32_WORKER_BROADCAST_STARTUP_ENDED:
      {
            ETG_TRACE_USR4( ( "spm_tclLocalApplicationManagerConfig::bHandleSynchrounousCall(SPM_U32_WORKER_BROADCAST_STARTUP_ENDED)" ) );
         if ( fOverTempActive ){
            ETG_TRACE_USR4( ( "StartUp finished, and overtemp active. Send overtemp SPM_COREFI_C_U16_MOSTSTATE" ) );
         } else {
            ETG_TRACE_USR4( ( "StartUp finished, no overtemp. Nothing to do here." ) );
         }
      }
      break;

      case SPM_U32_WORKER_BROADCAST_STARTUP_DELAY:
      {
            ETG_TRACE_USR4( ( "spm_tclLocalApplicationManagerConfig::bHandleSynchrounousCall(SPM_U32_WORKER_BROADCAST_STARTUP_DELAY)" ) );
         // retrigger DL state in case of download
            SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclSystemStateManager, ISpmSystemStateManager );
         SPM_NULL_POINTER_CHECK_VAL( _poclGlobalApplicationManager );
         if ( poclSystemStateManager->u32GetNewSystemState( ) == SPM_PRJ_SYSTEM_STATE_DOWNLOAD ){
            if ( FALSE == _poclGlobalApplicationManager->bProcessSystemState( SPM_PRJ_SYSTEM_STATE_DOWNLOAD ) ){
            }
         }
      }
      break;

      default:
         // call base class if message is not recognized here
         ETG_TRACE_USR4( ( "spm_tclLocalApplicationManagerConfig::vHandleMessage() -> spm_tclLocalApplicationManager::bHandleSynchrounousCall()" ) );
         return( spm_tclLocalApplicationManager::bHandleSynchrounousCall( u32Message, args ) );
   } // switch

   return( spm_tclLocalApplicationManager::bHandleSynchrounousCall( u32Message, args ) );
} // bHandleSynchrounousCall

//
tVoid spm_tclLocalApplicationManagerConfig::vOnWholeSwBlockIsInForcedState( const std::string& strSwBlockName ){
   _oStartUpForcedState.insert( strSwBlockName );

         ETG_TRACE_USR4( ( "vOnWholeSwBlockIsInForcedState, Block %s is in fored state now (all apps have reached OFF/...)",
                     strSwBlockName.c_str( )
                     ) );

   TSpmSwBlockSet::const_iterator it;

   std::string                    output;
   for ( it = _oStartUpForcedState.begin( ); it != _oStartUpForcedState.end( ); ++it ){
      output += * it;
      output += " ";
   }

         ETG_TRACE_USR4( ( "BlockIsForcedList: %s", output.c_str( ) ) );

   std::set < ISpmSwBlockClient* >::iterator it2;
   for ( it2 = _clients.begin( ); it2 != _clients.end( ); ++it2 ){
      ( * it2 )->vOnSwBlockForced( _oStartUpForcedState );
   }
} // vOnWholeSwBlockIsInForcedState

