/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_LocalApplicationManager.h
  * @brief        This is header file contains functions for handling many things for application according to
  *                its SourceAppID like application initialisation, start and end.
  * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#include <vector>
#include <set>
#include <map>

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#include "spm_ILocalAppManager.h"

class ISpmApplicationConfiguration;
class ISpmApplicationDatabase;
class ISpmApplicationRequestSupervisor;
class ISpmCcaServer;
class ISpmCcaSupplierServer;
class ISpmGlobalApplicationManager;
class ISpmGlobalApplicationManager;
class ISpmOsalProxy;
class ISpmStartupCommon;
class ISpmStartupInvestigationServer;
class ISpmSubStateClient;
class ISpmSystemPowerManager;
class ISpmWorkerServer;

#define SPM_ALTERNATE_CONFIG_TO_MS 60000
typedef  std::vector < std::string > TProcessList;
struct TSpmProcStopItem
{
   std::string procName;
   tU16 u16Id;
   tBool fIsKilled;
};
typedef std::map < std::string, TSpmProcStopItem > TProcessStopMap;


/*!
  *  \class spm_tclLocalApplicationManager
  *  \brief This class handles many things for application according to its SourceAppID like application initialisation ,
          start , end and SVM register and unregister.

          It uses spm_tclApplicationRequestSupervisor for supervisiona and spm_tclStartupCommon to remove software blocks
        It uses spm_tclSystemStateManagerConfig to get state of application.

          The spm_tclFactory object creates an instance of this class.

          spm_tclSystemPowerManager use this Class to get culprit application ID , to get requested application state and uninitialized application state.
  */
class spm_tclLocalApplicationManager : public ISpmLocalAppManager
{
public:
spm_tclLocalApplicationManager( const ISpmFactory& factory,
                                tU32               u32AppId,
                                tU8                u8LamConfigID = 0 );

virtual tBool bCheckLifeSignals( );

virtual ~spm_tclLocalApplicationManager( );

tVoid vTerminate( );

tVoid vTraceApplicationsInfo( );

tVoid vTraceApplicationInfo( tU32 u32AppId );

tVoid vTraceSwBlocksInfo( );

tVoid vTraceSwBlockInfo( const std::string& strName );

virtual tBool bHandleAppStartReq( const tU32 u32SourceAppID );

virtual tBool bHandleAppInitialized( const tU32 u32SourceAppID );

virtual tBool bHandleAppStateChangeAck( const tU32 u32SourceAppID,
                                        const tU32 u32ChangedState,
                                        const tU32 u32TimeExtension );

virtual tBool bHandleAppEndSuccess( const tU32 u32SourceAppID );

virtual tBool bHandleAppEndFailure( const tU32 u32SourceAppID,
                                    const tU32 u32ErrorType );

virtual tBool bHandleAppQueueFull( tU32 u32SourceAppID,
                                   tU32 u32TargetAppId );

virtual tBool bHandleSvmAppRegister( const tU32 u32SourceAppID,
                                     const tU32 u32RegRecFlags,
                                     const tU32 u32NotifyInterval );

virtual tBool bHandleSvmAppNotify( const tU32 u32SourceAppID );

virtual tBool bHandleSvmAppUnRegister( const tU32 u32SourceAppID );

virtual tBool bHandleSvmAppNotifyProblemResponse( tU32 u32SourceAppID,
                                                  tU32 u32AppState );

/**
  * This function will be called if the Watchdog Thread detected a Watchdog Error. The detected error is marked in the Connected-Application-List. The function sends to all
  * corrupt applications a NOTIFY_PROBLEM - Message. This is the last chance for the application to fix the problem.
  */
virtual tVoid vWatchdogError( );

tVoid vSendCriticalVoltageEvent( tU32 u32Cvm );

virtual tU32  u32GetRequestedAppState( tU32 u32AppId );

virtual tU32  u32GetUninitializedAppState( tU32 u32AppId );

virtual tU32  u32GetCulpritAppID( ) const { return( _u32CulpritAppID ); }
tU32  u32GetUnacknowledgedState( ) const { return( _u32CulpritUnacknowledgedState ); }


tVoid vSetAlternativeApplicationConfiguration( tU32 u32SystemState,
                                               tU32 u32Configuration );


// implementation of interface ISpmLocalAppManager
virtual tBool bCheckRequests( tU32& u32ActiveRequests );

virtual tVoid vUnacknowledgedApplicationState( );

virtual tBool bAllApplicationsUpAndRunning( tU32& u32AppId );

virtual tU32  u32GetParentId( ){ return( SPM_U32_ADAPTER_APP ); }

virtual tBool bKillAllApplications( );

virtual tVoid vKillSoftwareBlock( const std::string& name );

virtual tVoid vKillSoftwareBlocksOfProcess( const std::string& name );

virtual tBool bForceSoftwareBlock( const std::string& name,
                                   tU32               u32AppState );

virtual tBool bAllRequestsProcessed( );

virtual tBool bSetAllSwBlocksToNonBlockMode( );

virtual tBool bSetAllApplicationsInProfile( );

virtual tBool bForceAllApplicationsToProfile( tU32  u32Profile,
                                              tBool bForceAll = FALSE );

virtual tVoid vHandleApplicationEndFailureTimeout( );

virtual tVoid vCheckForSwBlocksToKill( );

virtual tU32  u32GetCurrentAppState( tU32 u32AppId );

virtual tVoid vCheckForUnacknowledgedState( tU32& u32ActiveRequests );

protected:
// ISpmBase interface functions
virtual tVoid vHandleTraceMessage( const tUChar *puchData );

virtual const tChar*getName( ) const { return( "spm_tclLocalApplicationManager" ); }
virtual tVoid vGetReferences( );

virtual tVoid vStartCommunication( );

// implementation of interface ISpmWorkerClient
virtual tVoid vHandleMessage( tU32 u32Message,
                              tU32 u32Parm = 0 );

virtual tBool bHandleSynchrounousCall( tU32   u32Message,
                                       tVoid *args );


tVoid vPostPowerMessage( tU32 u32TargetID,
                         tU16 u16PowerType,
                         tU32 u32PowerData1,
                         tU32 u32PowerData2 );

#ifdef VARIANT_S_FTR_ENABLE_ALTERNATE_CFG
   static tVoid vAlternateConfigSetTimerCallback( tVoid *pArg );

#endif
#ifdef VARIANT_S_FTR_ENABLE_PROCESS_CFG
   tBool bGetProcessListForConfiguration( tU32          u32Configuration,
                                          TProcessList& oProcessList ) const;

#endif

TSpmSwBlockSet                    _oLoadedState;
TSpmSwBlockSet                    _oConnectedState;
TSpmSwBlockSet                    _oStartUpRunState;

tU32                              _u32AppEndFailureCounter;
tU32                              _u32EndRequestedAppId;
tU32                              _u32AppEndWantedTimeOut;

tU8                               _u8LamConfigID;
tU32                              _u32AppToBlock;

ISpmWorkerServer                 *_poclWorkerServer;
ISpmOsalProxy                    *_poclOsalProxy;
ISpmApplicationDatabase          *_poclDb;
ISpmStartupInvestigationServer   *_poclStartupInvest;
ISpmSystemPowerManager           *_poclSystemPowerManager;
ISpmCcaServer                    *_poclCcaMsgHandler;
ISpmCcaSupplierServer            *_poclCcaSupplierHandler;
ISpmSubStateClient               *_poclSubStateHandler;
ISpmGlobalApplicationManager     *_poclGlobalApplicationManager;
ISpmApplicationRequestSupervisor *_poclApplicationRequestSupervisor;
ISpmApplicationConfiguration     *_poclAppCfg;
ISpmStartupCommon                *_poclCommonStartup;

tU32                              _u32AppId;

tU32                              _u32CulpritAppID;
tU32                              _u32CulpritUnacknowledgedState;
tBool                             _bQueueFullHandlingActive;


#ifdef VARIANT_S_FTR_ENABLE_ALTERNATE_CFG
   tBool                          _bAlternativConfigChangeRunning;
   OSAL_tTimerHandle              _hAlternateConfigTimer;
#endif
#ifdef VARIANT_S_FTR_ENABLE_PROCESS_CFG
   TProcessStopMap                _MapOfProcessesToStop;
   OSAL_tSemHandle                _hSemHandle;
   tU16                           _procStopId;
#endif

};

