/*!
  * \file spm_LocalApplicationManager.cpp
  *  \brief
  *    handling of the power messages, state requests and acknowledges
  *               coming from the applications.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * 04.03.11  | TMS Fischer       | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_LocalApplicationManager.h"

// spm class definitions
#include "spm_ApplicationConfigurationConfig.h"
#include "spm_ApplicationRequestSupervisor.h"
#include "spm_StartupCommon.h"
#include "spm_SoftwareBlock.h"

// interfaces class definitions
#include "spm_IOsalProxy.h"
#include "spm_IGlobalApplicationManager.h"
#include "spm_ICcaServiceServer.h"
#include "spm_ISystemStateManager.h"
#include "spm_ISubStateClient.h"
#include "spm_ISystemPowerManager.h"
#include "spm_IApplicationDatabase.h"
#include "spm_IStartupInvestigationServer.h"
#include "spm_ICcaServer.h"
#include "spm_ICcaSupplierServer.h"
#include "spm_ICcaServiceServer.h"
#include "spm_ProcessShutdown.h"
#include "spm_IApplicationErrorHandler.h"
#include "spm_IStartupSystemVariant.h"
#include "spm_ILamAppManager.h"
#include "spm_ISupervisionEnableSupervisor.h"

#ifdef VARIANT_S_FTR_ENABLE_PROCESS_CFG
#include "spm_IProcessShutdown.h"
#endif
#include "spm_CcaServiceHandler.h"
#include "spm_ILateServiceHandler.h"
#include "spm_IFactory.h"
#include "spm_CriticalSection.h"
#include "spm_StringTokenizer.h"
// spm helper
#include "spm_IRegistry.h"
#include "spm_ConnectedApps.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_LAM
#include "trcGenProj/Header/spm_LocalApplicationManager.cpp.trc.h"
#endif

// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
// #define SPM_TRACE_FILE_ID   SPM_FILE_LOCALAPPLICATIONMANAGER

#define SPM_APP_MAX_TIMEEXTENSION        60000
#define SPM_APP_END_WANTED_RESET_DELAY   1000

spm_tclLocalApplicationManager::spm_tclLocalApplicationManager( const ISpmFactory& factory,
                                                                tU32               u32AppId,
                                                                tU8                u8LamConfigID )

/*!
  * \fn
  *  \brief
  *    Constructor.
  *
  *  \param
  *    poSpm: reference to main class
  *  \version
  *    1.0   - Initial
  ******
  */
   : ISpmLocalAppManager( factory )
   , _u32AppEndFailureCounter( 0 )
   , _u32EndRequestedAppId( 0 )
   , _u32AppEndWantedTimeOut( 0 )
   , _u8LamConfigID( u8LamConfigID )
   , _u32AppToBlock( AMT_C_U16_APPID_INVALID )
   , _poclWorkerServer( NULL )
   , _poclOsalProxy( NULL )
   , _poclDb( NULL )
   , _poclStartupInvest( NULL )
   , _poclSystemPowerManager( NULL )
   , _poclCcaMsgHandler( NULL )
   , _poclCcaSupplierHandler( NULL )
   , _poclSubStateHandler( NULL )
   , _poclGlobalApplicationManager( NULL )
   , _poclApplicationRequestSupervisor( NULL )
   , _poclAppCfg( NULL )
   , _poclCommonStartup( NULL )
   , _u32AppId( u32AppId )
   , _u32CulpritAppID( 0 )
   , _u32CulpritUnacknowledgedState( AMT_C_U32_STATE_UNINITALIZED )
   , _bQueueFullHandlingActive( FALSE )
#ifdef VARIANT_S_FTR_ENABLE_ALTERNATE_CFG
   , _bAlternativConfigChangeRunning(FALSE)
   ,  _hAlternateConfigTimer(OSAL_C_INVALID_HANDLE)
#endif
#ifdef VARIANT_S_FTR_ENABLE_PROCESS_CFG
   , _hSemHandle(OSAL_C_INVALID_HANDLE)
   , _procStopId(1)
#endif
   {

   #ifdef VARIANT_S_FTR_ENABLE_ALTERNATE_CFG
      _bAlternativConfigChangeRunning = FALSE;

      if ( OSAL_s32TimerCreate( vAlternateConfigSetTimerCallback, this, &_hAlternateConfigTimer ) != OSAL_OK ){
         tU32 u32ErrorReason = OSAL_u32ErrorCode( );
         ETG_TRACE_ERRMEM( ( "SPM: spm_tclCcaMsgHandler !!!!!! Error detected !!!!!! cannot create Timer _hAlternateConfigTimer: error 0x%08X (%s)",
                                (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
      }
      if ( OSAL_ERROR == OSAL_s32SemaphoreCreate( "SpmLamSt", &_hSemHandle, (tU32)1 ) ){
         tU32 u32ErrorReason = OSAL_u32ErrorCode( );
         ETG_TRACE_ERRMEM( ( "SPM: spm_tclCcaMsgHandler !!!!!! Error detected !!!!!! cannot create Semaphore _hSemHandle: error 0x%08X (%s)",
                                (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
      }
   #endif
}

spm_tclLocalApplicationManager::~spm_tclLocalApplicationManager( ){
   #ifdef VARIANT_S_FTR_ENABLE_ALTERNATE_CFG
      if ( OSAL_s32TimerSetTime( _hAlternateConfigTimer, 0, 0 ) != OSAL_OK ){
         ETG_TRACE_ERR( ( "Timer set error!" ) );
      }
      if ( OSAL_s32TimerDelete( _hAlternateConfigTimer ) != OSAL_OK ){
         ETG_TRACE_ERR( ( "Timer delete error!" ) );
      }
   #endif

   _poclCcaMsgHandler                = NULL;
   _poclGlobalApplicationManager     = NULL;
   _poclAppCfg                       = NULL;
   _poclApplicationRequestSupervisor = NULL;
   _poclCcaSupplierHandler           = NULL;
   _poclSubStateHandler              = NULL;
   _poclSystemPowerManager           = NULL;
   _poclCommonStartup                = NULL;
   _poclDb                           = NULL;
   _poclStartupInvest                = NULL;
   _poclOsalProxy                    = NULL;
   _poclWorkerServer                 = NULL;
}

tVoid spm_tclLocalApplicationManager::vGetReferences( ){
   // get all needed references now -> all SPM objects are now available
   SPM_GET_IF_REFERENCE_USE_VAR( _poclGlobalApplicationManager,     ISpmGlobalApplicationManager );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclCcaMsgHandler,                ISpmCcaServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSubStateHandler,              ISpmSubStateClient );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSystemPowerManager,           ISpmSystemPowerManager );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclDb,                           ISpmApplicationDatabase );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer,                 ISpmWorkerServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclApplicationRequestSupervisor, ISpmApplicationRequestSupervisor );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclAppCfg,                       ISpmApplicationConfiguration );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclOsalProxy,                    ISpmOsalProxy );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclCcaSupplierHandler,           ISpmCcaSupplierServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclStartupInvest,                ISpmStartupInvestigationServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclCommonStartup,                ISpmStartupCommon );
} // vGetReferences

tVoid spm_tclLocalApplicationManager::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *    Synchronize the startup. Until now the FC_SPM is fully running.
  ******
  */
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   _poclWorkerServer->vAddClient( this );
}

tBool spm_tclLocalApplicationManager::bHandleAppStartReq( const tU32 u32SourceAppID ){
/*!
  * \fn
  *  \brief
  *    First message of an CCA-application has been received
  *    -- check the application database if the application is already known
  *    -- check the registry for a new SWBlock and
  *    -- add a new SW-Block if it is found
  *    -- send INIT message back to app
  *    -- updated StartupItem for this app
  *
  *  \param
  ******
  */
   tBool                bSuccess = FALSE;
   tU16                 u16PowerType;
   spm_tclConnectedApp *poConApp = NULL;

   SPM_NULL_POINTER_CHECK_VAL( _poclDb );
   SPM_NULL_POINTER_CHECK_VAL( _poclStartupInvest );
   SPM_NULL_POINTER_CHECK_VAL( _poclCommonStartup );

   // SPM_ET_TRACE_INFO1( _u16SpmID,"SystemPowerManager::bHandleAppStartReq, received PWR_APP_START_REQ %d\n", u32SourceAppID);
   ETG_TRACE_USR1( ( "spm_tclLocalApplicationManager::bHandleAppStartReq, received PWR_APP_START_REQ %u(%d/0x%x)",
                     ETG_ENUM( ail_u16AppId, u32SourceAppID ), u32SourceAppID, u32SourceAppID
                     ) );

   poConApp = _poclDb->poGetByAppID( u32SourceAppID );
   if ( poConApp == NULL ){

      if ( SPM_FRAMEWORK_IS_APP_CCA( u32SourceAppID ) ){
         ETG_TRACE_USR1( ( "spm_tclLocalApplicationManager::bHandleAppStartReq, Application %u currently unknown --> try to find if registry was already loaded by another component.",
                           ETG_ENUM( ail_u16AppId, u32SourceAppID )
                           ) );
         //here we have to update the registry to check for exclusions
         #ifdef VARIANT_S_FTR_ENABLE_VARIANT_HANDLING
            SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poStartupSystemVariant, ISpmStartupSystemVariant );
            poStartupSystemVariant->vUpdateRegistry( );
         #endif

         // try to find if registry was already loaded by another component
            SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclRegistry, ISpmRegistry );
         std::string strSwBlockName;
         if ( poclRegistry->bFindSwBlockNameOfApp( u32SourceAppID, strSwBlockName ) ){
            std::string strExecPath    = _poclCommonStartup->strGetProcLocationByBlockName( strSwBlockName );
            std::string strProcessName = _poclCommonStartup->strGetProcNameByBlockName( strSwBlockName );
            if ( _poclCommonStartup->bAddSwBlock( strSwBlockName, FALSE, TRUE, strExecPath, strProcessName ) ){
               poConApp = _poclDb->poGetByAppID( u32SourceAppID );
            } else {
               ETG_TRACE_USR1( ( "spm_tclLocalApplicationManager::bHandleAppStartReq, Application %u(%d/0x%x): ISpmStartupCommon::bAddSwBlock failed.",
                                 ETG_ENUM( ail_u16AppId, u32SourceAppID ), u32SourceAppID, u32SourceAppID
                                 ) );
            }
         } else {
            ETG_TRACE_USR1( ( "spm_tclLocalApplicationManager::bHandleAppStartReq, Application %u(%d/0x%x): spm_tclRegistry::bFindSwBlockNameOfApp failed.",
                              ETG_ENUM( ail_u16AppId, u32SourceAppID ), u32SourceAppID, u32SourceAppID
                              ) );
         }
      }
   }

   if ( poConApp != NULL ){
      bSuccess = TRUE;

      ETG_TRACE_USR1( ( "spm_tclLocalApplicationManager::bHandleAppStartReq, send PWR_APP_START_CONF %u(%d/0x%x), time: %dms",
                        ETG_ENUM( ail_u16AppId, u32SourceAppID ), u32SourceAppID, u32SourceAppID,
                        OSAL_ClockGetElapsedTime( )
                        ) );

      // support startup investigation
      _poclStartupInvest->vUpdateApplicationItem( u32SourceAppID,
                                                  ISpmStartupInvestigationServer::SPM_STARTUP_APPLICATION_STARTREQ,
                                                  0 );

      // queue is evaluated in send function
      // poConApp->vSetQueue(scd_OpenQueue(u32SourceAppID));
      u16PowerType = AMT_C_U16_PWR_PROXY_START_CONF;
      poConApp->vSetAppStateRequest( poConApp->u32GetInitializedState( ) );

      if ( _poclApplicationRequestSupervisor ){
         _poclApplicationRequestSupervisor->vStartSupervision( );
      }
   } else {
      ETG_TRACE_ERRMEM( ( "spm_tclLocalApplicationManager::bHandleAppStartReq, unknown application %u(0x%x)", (tUInt)u32SourceAppID, (tUInt)u32SourceAppID ) );

      u16PowerType = AMT_C_U16_PWR_PROXY_START_REJ;

      ETG_TRACE_ERRMEM( ( "spm_tclLocalApplicationManager::bHandleAppStartReq, send PWR_APP_START_REJ" ) );

      // support startup investigation
      _poclStartupInvest->vUpdateApplicationItem( u32SourceAppID,
                                                  ISpmStartupInvestigationServer::SPM_STARTUP_APPLICATION_STARTREQ,
                                                  SPM_REJECTED_REQUESTED_STATE );
   }

   // support startup investigation
   _poclStartupInvest->vUpdateApplicationItem( u32SourceAppID,
                                               ISpmStartupInvestigationServer::SPM_STARTUP_APPLICATION_STARTCONF,
                                               0 );

   vPostPowerMessage( u32SourceAppID,
                      u16PowerType,
                      0,
                      0 );

   return( bSuccess );
} // bHandleAppStartReq

tBool spm_tclLocalApplicationManager::bHandleAppInitialized( const tU32 u32SourceAppID ){
   spm_tclConnectedApp *poConApp = NULL;

   SPM_NULL_POINTER_CHECK_VAL( _poclDb );
   SPM_NULL_POINTER_CHECK_VAL( _poclStartupInvest );
   SPM_NULL_POINTER_CHECK_VAL( _poclAppCfg );

   // SPM_ET_TRACE_INFO1(_u16SpmID, "SystemPowerManager::bHandleAppInitialized, received PWR_APP_START_INITIALIZED %d\n", u32SourceAppID);
   ETG_TRACE_USR1( ( "bHandleAppInitialized, received PWR_APP_START_INITIALIZED %u(%d/0x%x), time: %dms",
                     ETG_ENUM( ail_u16AppId, u32SourceAppID ), u32SourceAppID, u32SourceAppID, OSAL_ClockGetElapsedTime( )
                     ) );

   // support startup investigation
   _poclStartupInvest->vUpdateApplicationItem( u32SourceAppID,
                                               ISpmStartupInvestigationServer::SPM_STARTUP_APPLICATION_INITIALIZED,
                                               0 );

   _poclAppCfg->vNewAppInSystem( u32SourceAppID );
   poConApp = _poclDb->poGetByAppID( u32SourceAppID );

   if ( !poConApp ){
      // SPM_ET_TRACE_ERROR( u16GetSpmID(),"SystemPowerManager::bHandleAppInitialized, app not in connected-app-list\n");
      ETG_TRACE_USR1( ( "bHandleAppInitialized, app %u(%d/0x%x) not in connected-app-list",
                        ETG_ENUM( ail_u16AppId, u32SourceAppID ), u32SourceAppID, u32SourceAppID
                        ) );
      return( FALSE );
   }

   poConApp->vSetIsConnected( );
   poConApp->bHandleAppInitialized( );
   poConApp->vSetAppStateReqEndTime( SPM_APPREQMON_DEFAULT_INTERVAL );

   _poclDb->bSetAllApplicationsToProfile( _poclAppCfg, this );

   if ( _poclCcaSupplierHandler && SPM_FRAMEWORK_IS_APP_CCA( u32SourceAppID ) ){
      _poclCcaSupplierHandler->vSupplierStateChanged( (tU16)u32SourceAppID, AMT_C_U8_CCAMSG_SUPPLIER_STATE_AVAILABLE );
   }
   return( TRUE );
} // bHandleAppInitialized

tBool spm_tclLocalApplicationManager::bHandleAppStateChangeAck( const tU32 u32SourceAppID,
                                                                const tU32 u32ChangedState,
                                                                const tU32 u32TimeExtension ){
   SPM_NULL_POINTER_CHECK_VAL( _poclDb );
   SPM_NULL_POINTER_CHECK_VAL( _poclGlobalApplicationManager );
   SPM_NULL_POINTER_CHECK_VAL( _poclStartupInvest );

   spm_tclConnectedApp *poConApp = _poclDb->poGetByAppID( u32SourceAppID );

   if ( !poConApp ){
      // SPM_ET_TRACE_ERROR( u16GetSpmID(),"SystemPowerManager::bHandleAppStateChangeAck, app not in connected-app-list\n");
      ETG_TRACE_ERR( ( "bHandleAppStateChangeAck, app not in connected-app-list: app %u(%d/0x%x)",
                       ETG_ENUM( ail_u16AppId, u32SourceAppID ), u32SourceAppID, u32SourceAppID
                       ) );
      return( FALSE );
   }

   if ( u32SourceAppID == _u32AppToBlock ){
      ETG_TRACE_ERRMEM( ( "bHandleAppStateChangeAck, app is currently blocked (just to test): app %u(%d/0x%x)",
                          (tUInt)ETG_ENUM( ail_u16AppId, u32SourceAppID ), (tUInt)u32SourceAppID, (tUInt)u32SourceAppID
                          ) );
      return( FALSE );
   }

   // ignore extension of time messages (navi)
   if ( u32TimeExtension != 0 ){
      // add new timeextension if endtime is in time limit
      if ( poConApp->u32GetAppStateReqEndTime( ) < SPM_APP_MAX_TIMEEXTENSION ){
         poConApp->vExtendAppStateReqEndTime( u32TimeExtension );
      }
      ETG_TRACE_USR1( ( "bHandleAppStateChangeAck, received time extension (%dms) from app %u(%d/0x%x)",
                        poConApp->u32GetAppStateReqEndTime( ),
                        ETG_ENUM( ail_u16AppId, u32SourceAppID ), u32SourceAppID, u32SourceAppID
                        ) );
      return( TRUE );
   }

   // support startup investigation
   _poclStartupInvest->vUpdateApplicationItem( u32SourceAppID,
                                               ISpmStartupInvestigationServer::SPM_STARTUP_APPLICATION_STATECONF,
                                               u32ChangedState );

   ETG_TRACE_USR1( ( "bHandleAppStateChangeAck, received StateChangeAck, app: %u(%d/0x%x) state: %u, Transition duration: %dms, time: %dms",
                     ETG_ENUM( ail_u16AppId,     u32SourceAppID ), u32SourceAppID, u32SourceAppID,
                     ETG_ENUM( SPM_APPSTATE_DEF, u32ChangedState ),
                     OSAL_ClockGetElapsedTime( ) - poConApp->u32GetAppStateReqStartTime( ),
                     OSAL_ClockGetElapsedTime( )
                     ) );

   {
      SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclAppManagerConfig, ISpmLamAppManager );

      poclAppManagerConfig->vAppStateAcknowledge( u32SourceAppID, u32ChangedState );
   }
   // set new state in internal "connected-application-list"
   poConApp->vSetAppStateCurrent( u32ChangedState );
   poConApp->vSetAppStateReqEndTime( SPM_APPREQMON_DEFAULT_INTERVAL );

   if ( ( u32ChangedState != poConApp->u32GetAppStateNewRequest( ) )
        || ( poConApp->u32GetUninitializedState( ) != poConApp->u32GetAppStateForceReq( ) )
        ){
      if ( !_poclDb->bSendUpdate( poConApp->u32GetAppId( ), this ) ){
         ETG_TRACE_COMP( ( "SPM: !!!!!! Warning detected !!!!!!" ) );
      }
   }
   if ( poConApp->u32GetUninitializedState( ) != poConApp->u32GetAppStateForceReq( ) ){
      // force app state is requested --> check if received state is already the forced state
      if ( u32ChangedState == poConApp->u32GetAppStateForceReq( ) ){
         // ETG_TRACE_USR4(("app is already in the new state, no need to send again --> reset force mode"));
         // app is already in the new state, no need to send again --> reset force mode for this application

         _poclDb->vResetForceMode( u32SourceAppID, this );
      }
   }

   if ( bAllRequestsProcessed( ) ){
      _poclGlobalApplicationManager->vNewSystemStateReached( _u8LamConfigID );
   }
   return( TRUE );
} // bHandleAppStateChangeAck

tBool spm_tclLocalApplicationManager::bHandleAppEndSuccess( const tU32 u32SourceAppID ){
   tBool bAllAppsTerminated = FALSE;

   SPM_NULL_POINTER_CHECK_VAL( _poclGlobalApplicationManager );
   SPM_NULL_POINTER_CHECK_VAL( _poclDb );
   SPM_NULL_POINTER_CHECK_VAL( _poclCommonStartup );
   SPM_NULL_POINTER_CHECK_VAL( _poclSystemPowerManager );
   SPM_NULL_POINTER_CHECK_VAL( _poclCcaSupplierHandler );
   SPM_NULL_POINTER_CHECK_VAL( _poclCcaMsgHandler );


   spm_tclConnectedApp  *poConApp = _poclDb->poGetByAppID( u32SourceAppID );
   spm_tclSoftwareBlock *pSwBlock = _poclDb->poGetSwBlockByAppID( u32SourceAppID );

   if ( !poConApp ){
      // SPM_ET_TRACE_ERROR( u16GetSpmID(),"SystemPowerManager::bHandleAppEndSuccess, app not in connected-app-list\n");
      ETG_TRACE_ERR( ( "bHandleAppEndSuccess, app %d(0x%x) not in connected-app-list", u32SourceAppID, u32SourceAppID ) );
      return( FALSE );
   }

   ETG_TRACE_USR1( ( "bHandleAppEndSuccess, received APP_END_SUCCESSFUL from app: %u(%d/0x%x), Transition duration: %dms",
                     ETG_ENUM( ail_u16AppId, u32SourceAppID ), u32SourceAppID, u32SourceAppID,
                     OSAL_ClockGetElapsedTime( ) - poConApp->u32GetAppStateReqStartTime( )
                     ) );

   _poclCcaMsgHandler->vApplicationEndSuccessful( (tU16)u32SourceAppID );

   // set new state in internal "connected-application-list"
   poConApp->vSetAppStateCurrent( AMT_C_U16_PWR_PROXY_END_APP );
   poConApp->vSetIsConnected( FALSE );
   if ( _poclDb->u32NumberOfConnectedApplications( ) == 0 ){
      bAllAppsTerminated = TRUE;
      _poclGlobalApplicationManager->vNewSystemStateReached( _u8LamConfigID );
   }

   if ( pSwBlock->bIsWholeBlockTobeKilled( ) && pSwBlock->bIsDisconnected( ) ){
      pSwBlock->vSetWholeBlockConnected( FALSE );

      TSpmSwBlockSet::iterator it = _oConnectedState.find( pSwBlock->pcGetName( ) );
      if ( it != _oConnectedState.end( ) ){
         _oConnectedState.erase( it );
      }
      pSwBlock->vSetWholeBlockUp( FALSE );
      it = _oStartUpRunState.find( pSwBlock->pcGetName( ) );
      if ( it != _oStartUpRunState.end( ) ){
         _oStartUpRunState.erase( it );
         // here we have to shutdown the process
      }
      _poclCommonStartup->vRemoveSwBlock( pSwBlock->pcGetName( ) );
   }

   if ( bAllAppsTerminated ){
      _poclSystemPowerManager->vKillAllApplicationsReady( );
   }
   if ( SPM_FRAMEWORK_IS_APP_CCA( u32SourceAppID ) ){
      _poclCcaSupplierHandler->vSupplierStateChanged( (tU16)u32SourceAppID, AMT_C_U8_CCAMSG_SUPPLIER_STATE_UNAVAILABLE );
   }

   return( TRUE );
} // bHandleAppEndSuccess

tBool spm_tclLocalApplicationManager::bHandleAppEndFailure( const tU32 u32SourceAppID,
                                                            const tU32 u32ErrorType ){
   std::string strFirstEntry( "APP_END_WANTED: 1. Thread information  (Request received by AIL)." );

   SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclSupervisionEnableSupervisor, ISpmSupervisionEnableSupervisor );

   if ( TRUE == poclSupervisionEnableSupervisor->bIsSupervisionStopped( ) ){
      ETG_TRACE_ERRMEM( ( "spm_tclApplicationErrorHandler::bHandleAppEndFailure: Triggering Callstack disabled due to Supervision settings (WD_OFF/disable_reset.txt/TTFis-SpmResetSupervisionDisable)" ) );
      return( FALSE );
   }

   SPM_NULL_POINTER_CHECK_VAL( _poclDb );
   SPM_NULL_POINTER_CHECK_VAL( _poclOsalProxy );

   if ( _u32EndRequestedAppId != u32SourceAppID ){
      spm_tclConnectedApp *poConApp = _poclDb->poGetByAppID( u32SourceAppID );

      if ( poConApp != NULL ){
         SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclAppErrHandler, ISpmApplicationErrorHandler );

         poclAppErrHandler->vAppEndFailure( * poConApp, u32ErrorType );
         _u32EndRequestedAppId = u32SourceAppID;
      }
   }

   if ( _u32AppEndFailureCounter == 0 ){
      // for the first trigger a getp_ready is triggered
      _poclOsalProxy->bDumpThreadInfoReady( SPM_S32_THREAD_PRIO_LOW );

      _u32AppEndWantedTimeOut = OSAL_ClockGetElapsedTime( ) + SPM_APP_END_WANTED_RESET_DELAY; // trigger timeout

      // resume AppRequest thread to activate delay time
      if ( _poclApplicationRequestSupervisor ){
         _poclApplicationRequestSupervisor->vStartSupervision( );
      }
   }
   _u32AppEndFailureCounter++;

   return( TRUE );
} // bHandleAppEndFailure

/*!
  * \fn
  *  \brief
  *   The function logs the thread and queue information in case of
  *   application failure due to queue full..
  *  \param[in] u32SourceAppID: Source application ID.
  *  \param[in] u16TargetAppId: Target application ID.
  *  \details
  *   If the _bQueueFullHandlingActive is active then based on the list of
  *  connected applications ApplicationErrohandler is triggered to log the
  *  QueueStatus and reason for application failure into errmem. If more than
  *  one queue full in encountered then the corresponding msg is logged into errmem.
  *********
  */
tBool spm_tclLocalApplicationManager::bHandleAppQueueFull( tU32 u32SourceAppID,
                                                           tU32 u32TargetAppId ){

   ETG_TRACE_USR4( ( "spm_tclLocalApplicationManager::bHandleAppQueueFull(%d,%d)", u32SourceAppID, u32TargetAppId ) );
   std::string strFirstEntry( "QUEUE_FULL ERROR: Getting Thread&Queue information" );
   std::string strErrmemBuf;
   tU8         u8Dummy = 0;
   SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclAppErrHandler, ISpmApplicationErrorHandler );

   SPM_NULL_POINTER_CHECK_VAL( _poclSystemPowerManager );
   SPM_NULL_POINTER_CHECK_VAL( _poclDb );
   SPM_NULL_POINTER_CHECK_VAL( _poclWorkerServer );
   SPM_NULL_POINTER_CHECK_VAL( _poclAppCfg );
   SPM_NULL_POINTER_CHECK_VAL( _poclOsalProxy );

   if ( !_bQueueFullHandlingActive ){
      _bQueueFullHandlingActive = TRUE;

      spm_tclConnectedApp *poConApp;

      poConApp                  = _poclDb->poGetByAppID( u32TargetAppId );
      if ( poConApp != NULL ){
         poclAppErrHandler->vMsgQueueFull( * poConApp, FALSE );
      } else {
         poclAppErrHandler->vMsgQueueFull( u32TargetAppId, FALSE );
      }

      OSAL_s32ThreadWait( 1000 );

      poConApp = _poclDb->poGetByAppID( u32SourceAppID );
      if ( poConApp != NULL ){
         poclAppErrHandler->vMsgQueueFull( * poConApp, TRUE, TRUE );
      } else {
         poclAppErrHandler->vMsgQueueFull( u32SourceAppID, TRUE, TRUE );
      }
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (const tU8*)&u8Dummy, sizeof( u8Dummy ) );
   } else {
      ETG_TRACE_USR4( ( "spm_tclLocalApplicationManager::bHandleAppQueueFull(%d,%d). No callstack generation, because other queuefull handling is allready running", u32SourceAppID, u32TargetAppId ) );
      std::stringstream ssRevQueueFull;
      ssRevQueueFull << "QUEUE_FULL : Received another QUEUE_FULL (TargetAppId=" << std::to_string(u32TargetAppId) << "(0x" << std::hex << u32TargetAppId << ") SourceAppId=" << std::to_string(u32SourceAppID) << "(0x" << std::hex << u32SourceAppID << ")) while other queue full is allready handled.";
      strErrmemBuf = ssRevQueueFull.str();
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strErrmemBuf.c_str( ), (tU16)OSAL_u32StringLength( strErrmemBuf.c_str( ) ) );
   }

   return( TRUE );
} // bHandleAppQueueFull

tBool spm_tclLocalApplicationManager::bHandleSvmAppRegister( const tU32 u32SourceAppID,
                                                             const      tU32, /*u32RegRecFlags*/
                                                             const tU32 u32NotifyInterval ){
/*
   -------------------------------------------------------------------------
   Handler "SVM_APP_REGISTER"
   application register, to monitor by the SPM  (Watchdog)
   -------------------------------------------------------------------------
  */
   SPM_NULL_POINTER_CHECK_VAL( _poclDb );

   spm_tclConnectedApp *poApp = _poclDb->poGetByAppID( u32SourceAppID );

   if ( poApp != NULL ){
      poApp->vSetWatchdogActive( u32NotifyInterval );

      // SPM_ET_TRACE_INFO2( _u16SpmID, "SystemPowerManager::bHandleSvmAppRegister, monitor %d %d\n", u32SourceAppID, poApp->_u32NotifyInterval);
      ETG_TRACE_USR1( ( "bHandleSvmAppRegister, monitor %u(%d/0x%x) 0x%04x",
                        ETG_ENUM( ail_u16AppId, u32SourceAppID ), u32SourceAppID, u32SourceAppID,
                        poApp->u32GetNotifyInterval( )
                        ) );
      return( TRUE );
   }
   return( FALSE );
} // bHandleSvmAppRegister

tBool spm_tclLocalApplicationManager::bHandleSvmAppUnRegister( const tU32 u32SourceAppID ){
/*
   -------------------------------------------------------------------------
   Handler "SVM_APP_UNREGISTER"
   unregister application to monitor by the SPM  (Watchdog)
   -------------------------------------------------------------------------
  */
   SPM_NULL_POINTER_CHECK_VAL( _poclDb );

   spm_tclConnectedApp *poApp = _poclDb->poGetByAppID( u32SourceAppID );

   if ( poApp != NULL ){
      poApp->vSetWatchdogInactive( );
      return( TRUE );
   }
   return( FALSE );
}

tBool spm_tclLocalApplicationManager::bHandleSvmAppNotify( const tU32 u32SourceAppID ){
/*
   -------------------------------------------------------------------------
   Handler "SVM_APP_NOTIFY"
   Handles the notify messages of the applications.
   If the spm recieved a notify message the current time will stored in the
   connected-application-list
   -------------------------------------------------------------------------
  */
   ETG_TRACE_USR1( ( "bHandleSvmAppNotify, received AMT_C_U16_PWR_SVM_APP_NOTIFY %u(%d/0x%x)",
                     ETG_ENUM( ail_u16AppId, u32SourceAppID ), u32SourceAppID, u32SourceAppID
                     ) );

   spm_tclConnectedApp *poApp = NULL;

   SPM_NULL_POINTER_CHECK_VAL( _poclDb );

   poApp = _poclDb->poGetByAppID( u32SourceAppID );
   if ( poApp != NULL ){
      poApp->vUpdateLastLifeSignalTime( );
      return( TRUE );
   }
   return( FALSE );
} // bHandleSvmAppNotify

tBool spm_tclLocalApplicationManager::bHandleSvmAppNotifyProblemResponse( tU32 u32SourceAppID,
                                                                          tU32 u32AppState ){
/*
   -------------------------------------------------------------------------
   Handler "SVM_APP_NOTIFYPROBLEMRESPONSE"
   If the SPM send a NotifyProblem-Message the applicaton must answer with a
   NotifyProblemResponse-Message. After that, the SPM could handle the
   system behaviour.
   -------------------------------------------------------------------------
  */
   tBool                bSuccess = TRUE;
   spm_tclConnectedApp *poApp    = NULL;

   SPM_NULL_POINTER_CHECK_VAL( _poclDb );
   SPM_NULL_POINTER_CHECK_VAL( _poclWorkerServer );

   ETG_TRACE_USR1( ( "bHandleSvmAppNotifyProblemResponse, received PWR_SVM_APP_NOTIFY_PROBLEM_RESPONSE %d %u(%d/0x%x)", ETG_ENUM( ail_u32PWRAppState, u32AppState ),
                     ETG_ENUM( ail_u16AppId, u32SourceAppID ), u32SourceAppID, u32SourceAppID
                     ) );

   switch ( u32AppState ){
      case AMT_C_U32_PWR_APP_STATE_OK:
      {
         // application could remove the problem
         poApp = _poclDb->poGetByAppID( u32SourceAppID );
         if ( poApp != NULL ){
            poApp->vSetPowerStateOk( );
         }
      }
      break;

      case AMT_C_U32_PWR_LPM_APP_STATE_RESTART:
      case AMT_C_U32_PWR_APP_STATE_UNDEFINED:
      {
         // application couldn't remove the problem
         // reboot system: KILL KILL KILL Hahaha
         std::string strErrmemBuf;

         std::stringstream ssDetectNofityProblemResponse;
         ssDetectNofityProblemResponse << "SPM: Application AppNotifyProblemResponse detected NOTIFY_PROBLEM_RESPONSE: application " << std::to_string(u32SourceAppID) << "(0x" << std::hex << u32SourceAppID << ")";
         strErrmemBuf = ssDetectNofityProblemResponse.str();
         if ( _poclSystemPowerManager ){
            _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strErrmemBuf.c_str( ), (tU16)OSAL_u32StringLength( strErrmemBuf.c_str( ) ) );
         }
         _poclWorkerServer->bPostMessageToWorker( SPM_U32_WORKER_SPM_SHUTDOWN, SPM_U32_SHUTDOWN_NOTIFY_PROBLEM );
      }
      break;

      default:
      {
         ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!" ) );
      }
      break;
   } // switch

   // deliver status if watchdog signal did not come
   return( bSuccess );
} // bHandleSvmAppNotifyProblemResponse

tU32 spm_tclLocalApplicationManager::u32GetRequestedAppState( tU32 u32AppId ){

   tU32                 u32RequestedState = AMT_C_U32_STATE_INVALID;

   SPM_NULL_POINTER_CHECK_VAL( _poclDb );

   spm_tclConnectedApp *poConApp          = _poclDb->poGetByAppID( u32AppId );

   if ( poConApp != NULL ){
      u32RequestedState = poConApp->u32GetAppStateRequest( );
   }
   return( u32RequestedState );
}

tU32 spm_tclLocalApplicationManager::u32GetUninitializedAppState( tU32 u32AppId ){
   tU32                 u32UninitializedAppState = AMT_C_U32_STATE_INVALID;

   SPM_NULL_POINTER_CHECK_VAL( _poclDb );

   spm_tclConnectedApp *poConApp                 = _poclDb->poGetByAppID( u32AppId );

   if ( poConApp != NULL ){
      u32UninitializedAppState = poConApp->u32GetUninitializedState( );
   }
   return( u32UninitializedAppState );
}

// //////////////////////////////////////////////////////////////////////////////
// implementation of interface spm_InterfaceLocalApplicationManager
// //////////////////////////////////////////////////////////////////////////////

tBool spm_tclLocalApplicationManager::bCheckRequests( tU32& u32ActiveRequests ){
   tBool         bRetVal;
   tBool         bMissingResponse = FALSE;
   OSAL_tMSecond u32CurrentTime;

   bRetVal           = TRUE;
   u32ActiveRequests = 0; // count requests which are still active
   u32CurrentTime    = OSAL_ClockGetElapsedTime( );

   SPM_NULL_POINTER_CHECK_VAL( _poclDb );
   SPM_NULL_POINTER_CHECK_VAL( _poclOsalProxy );
   SPM_NULL_POINTER_CHECK_VAL( _poclWorkerServer );

   // check if ApplicationEndWanted was received in the last 4s
   if ( ( _u32AppEndWantedTimeOut != 0 ) && ( _u32AppEndWantedTimeOut < u32CurrentTime ) ){
      _u32AppEndWantedTimeOut = 0;

      // trigger TO to reset system
      _poclWorkerServer->bPostMessage( "ISpmGlobalApplicationManager", SPM_U32_WORKER_LAM_APP_END_WANTED_TO );
   }

   bRetVal = _poclDb->bCheckRequests( this, bMissingResponse, _u32CulpritAppID, _u32CulpritUnacknowledgedState, u32ActiveRequests );

   if ( bMissingResponse ){
      SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclSupervisionEnableSupervisor, ISpmSupervisionEnableSupervisor );

      if ( TRUE == poclSupervisionEnableSupervisor->bIsSupervisionStopped( ) ){
         ETG_TRACE_ERRMEM( ( "spm_tclLocalApplicationManager::bCheckRequests(): bDumpThreadInfoReady blocked by SupervisionDisableSettings (WD_OFF/disable_reset.txt/TTFis-SpmResetSupervisionDisable)" ) );
      } else {
         _poclOsalProxy->bDumpThreadInfoReady( SPM_S32_THREAD_PRIO_LOW );
      }
   }
   return( bRetVal );
} // bCheckRequests

tVoid spm_tclLocalApplicationManager::vUnacknowledgedApplicationState( ){
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   _poclWorkerServer->bPostMessageToWorker( SPM_U32_WORKER_SPM_SHUTDOWN, SPM_U32_SHUTDOWN_AFTER_NO_STATE_REQUEST );
}

tVoid spm_tclLocalApplicationManager::vCheckForUnacknowledgedState( tU32& u32ActiveRequests ){
   if ( !bCheckRequests( u32ActiveRequests ) ){
      // at least one application has not acknowledged state transition
      vUnacknowledgedApplicationState( );
   }
}

tBool spm_tclLocalApplicationManager::bAllApplicationsUpAndRunning( tU32& u32AppId ){
   SPM_NULL_POINTER_CHECK_VAL( _poclDb );
   return( _poclDb->bAllApplicationsUpAndRunning( u32AppId ) );
}

tBool spm_tclLocalApplicationManager::bKillAllApplications( ){
   SPM_NULL_POINTER_CHECK_VAL( _poclDb );
   return( _poclDb->bKillApplications( this ) );
}

tVoid spm_tclLocalApplicationManager::vKillSoftwareBlock( const std::string& name ){
      SPM_NULL_POINTER_CHECK( _poclDb );
      SPM_NULL_POINTER_CHECK( _poclSubStateHandler );

   spm_tclSoftwareBlock *sb = _poclDb->poGetSoftwareBlock( name );

   if ( ( sb != NULL ) && !sb->bIsWholeBlockTobeKilled( ) ){
      sb->vSetKillBlockEnabled( );
      if ( sb->bAllApplicationsInState( AMT_C_U32_STATE_OFF ) ){
         _poclDb->vCheckForSwBlocksToKill( this );
      } else {
         _poclSubStateHandler->vSetSubStateType( spm_fi_tcl_SPM_e32_SubStateType::FI_EN_SPM_U32_SUBSTATE_SYSSTATE_CHECK, FALSE );
      }
   }
}

tVoid spm_tclLocalApplicationManager::vKillSoftwareBlocksOfProcess( const std::string& name ){
      SPM_NULL_POINTER_CHECK( _poclDb );

      ETG_TRACE_USR1( ( "spm_tclLocalApplicationManager::vKillSoftwareBlocksOfProcess(): terminate process '%s' now.", name.c_str( ) ) );
   std::list < std::string >           oSwBlockList;
   _poclDb->vGetSwBlocksOfProcess( name.c_str( ), &oSwBlockList );

   std::list < std::string >::iterator itSwBlock;
   for ( itSwBlock = oSwBlockList.begin( ); itSwBlock != oSwBlockList.end( ); ++itSwBlock ){
      ETG_TRACE_USR1( ( "spm_tclLocalApplicationManager::vKillSoftwareBlocksOfProcess(): kill SW Block '%s'.", itSwBlock->c_str( ) ) );
      vKillSoftwareBlock( itSwBlock->c_str( ) );
   }
}

tBool spm_tclLocalApplicationManager::bForceSoftwareBlock( const std::string& name,
                                                           tU32               u32AppState ){
   tBool                 bSwBlockInForcedState = FALSE;

   SPM_NULL_POINTER_CHECK_VAL( _poclDb );
   SPM_NULL_POINTER_CHECK_VAL( _poclSubStateHandler );

   spm_tclSoftwareBlock *sb                    = _poclDb->poGetSoftwareBlock( name );

   if ( ( sb != NULL ) && !sb->bIsWholeBlockToBeForced( ) ){
      sb->vSetForceBlockEnabled( u32AppState );
      if ( sb->bAllApplicationsInState( u32AppState ) ){
         ETG_TRACE_USR4( ( "spm_tclLocalApplicationManager::bForceSoftwareBlock() -> bAllApplicationsInState in state %d (SWBLOCK=%s)", ETG_ENUM( ail_u32CCAState, u32AppState ), name.c_str( ) ) );
         bSwBlockInForcedState = TRUE;
      } else {
         ETG_TRACE_USR4( ( "spm_tclLocalApplicationManager::bForceSoftwareBlock() -> bAllApplicationsInState NOT in %d, sending substate trigger (SWBLOCK=%s)", ETG_ENUM( ail_u32CCAState, u32AppState ), name.c_str( ) ) );
         bSetAllApplicationsInProfile( );
         _poclSubStateHandler->vSetSubStateType( spm_fi_tcl_SPM_e32_SubStateType::FI_EN_SPM_U32_SUBSTATE_SYSSTATE_CHECK, FALSE );
      }
   }
   return( bSwBlockInForcedState );
} // bForceSoftwareBlock

tBool spm_tclLocalApplicationManager::bAllRequestsProcessed( ){
   SPM_NULL_POINTER_CHECK_VAL( _poclDb );
   return( _poclDb->bAllApplicationsInRequestedState( this ) );
}

tBool spm_tclLocalApplicationManager::bSetAllApplicationsInProfile( ){
   SPM_NULL_POINTER_CHECK_VAL( _poclDb );
   return( _poclDb->bSetAllApplicationsToProfile( _poclAppCfg, this ) );
}

tBool spm_tclLocalApplicationManager::bSetAllSwBlocksToNonBlockMode( ){
   SPM_NULL_POINTER_CHECK_VAL( _poclDb );
   return( _poclDb->bSetAllSwBlocksToNonBlockMode( ) );
}

tBool spm_tclLocalApplicationManager::bForceAllApplicationsToProfile( tU32  u32Profile,
                                                                      tBool bForceAll ){
   SPM_NULL_POINTER_CHECK_VAL( _poclDb );
   return( _poclDb->bForceAllApplicationsToProfile( u32Profile, bForceAll, _poclAppCfg, this ) );
}

tVoid spm_tclLocalApplicationManager::vHandleApplicationEndFailureTimeout( ){
   std::string cSecondEntry( "APP_END_WANTED: 2. Thread information (after max. 2s timeout) " );

   SPM_GET_IF_REFERENCE_NEW_VAR( poclSupervisionEnableSupervisor, ISpmSupervisionEnableSupervisor );

   if ( TRUE == poclSupervisionEnableSupervisor->bIsSupervisionStopped( ) ){
      ETG_TRACE_ERRMEM( ( "spm_tclLocalApplicationManager::vHandleApplicationEndFailureTimeout(): bDumpThreadInfoReady blocked by SupervisionDisableSettings (WD_OFF/disable_reset.txt/TTFis-SpmResetSupervisionDisable)" ) );
      return;
   }

   SPM_NULL_POINTER_CHECK( _poclSystemPowerManager );
   SPM_NULL_POINTER_CHECK( _poclOsalProxy );
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );

   _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)cSecondEntry.c_str( ), (tU16)cSecondEntry.size( ) );

   _poclOsalProxy->bDumpThreadInfoReady( SPM_S32_THREAD_PRIO_LOW );
   _poclWorkerServer->bPostMessageToWorker( SPM_U32_WORKER_SPM_SHUTDOWN, SPM_U32_SHUTDOWN_APP_END_WANTED );
} // spm_tclLocalApplicationManager::vHandleApplicationEndFailureTimeout

tVoid spm_tclLocalApplicationManager::vCheckForSwBlocksToKill( ){
   SPM_NULL_POINTER_CHECK( _poclDb );
   return( _poclDb->vCheckForSwBlocksToKill( this ) );
}

tU32 spm_tclLocalApplicationManager::u32GetCurrentAppState( tU32 u32AppId ){

   tU32                 u32CurrentState = AMT_C_U32_STATE_INVALID;
   spm_tclConnectedApp *poConApp        = NULL;

   SPM_NULL_POINTER_CHECK_VAL( _poclDb );

   poConApp = _poclDb->poGetByAppID( u32AppId );
   if ( poConApp != NULL ){
      u32CurrentState = poConApp->u32GetAppStateCurrent( );
   }
   return( u32CurrentState );
}

tBool spm_tclLocalApplicationManager::bCheckLifeSignals( ){
   SPM_NULL_POINTER_CHECK_VAL( _poclDb );
   return( _poclDb->bCheckLifeSignals( ) );
}

tVoid spm_tclLocalApplicationManager::vWatchdogError( ){
      SPM_NULL_POINTER_CHECK( _poclDb );
   _poclDb->vOnWatchdogError( this );
}

tVoid spm_tclLocalApplicationManager::vSendCriticalVoltageEvent( tU32 u32Cvm ){
      SPM_NULL_POINTER_CHECK( _poclDb );
   _poclDb->vSendCriticalVoltageEvent( u32Cvm, this );
}

tVoid spm_tclLocalApplicationManager::vPostPowerMessage( tU32 u32TargetID,
                                                         tU16 u16PowerType,
                                                         tU32 u32PowerData1,
                                                         tU32 u32PowerData2 ){
   if ( SPM_FRAMEWORK_IS_APP_CCA( u32TargetID ) ){
      SPM_NULL_POINTER_CHECK( _poclCcaMsgHandler );

      if ( !_poclCcaMsgHandler->bPostPowerMessage( (tU16)u32TargetID,
                                                   u16PowerType,
                                                   u32PowerData1,
                                                   u32PowerData2 ) ){
         ETG_TRACE_ERRMEM( ( "spm_tclLocalApplicationManager::vPostPowerMessage() Cant post message" ) );
      }
   } else if ( SPM_FRAMEWORK_IS_APP_LCM( u32TargetID ) ){
      if ( AMT_C_U16_PWR_PROXY_STATE_CHANGE_REQ == u16PowerType ){
         SPM_GET_CLIENT_HANDLER_IF_REFERENCE_NEW_VAR(poDbusTmp, spm_ISpmLateServiceHandler);
         if ( poDbusTmp ){
            poDbusTmp->s32AppStateChange( u32TargetID, u32PowerData1 );
         }
      }
   }
} // vPostPowerMessage

tVoid spm_tclLocalApplicationManager::vTerminate( ){
   if ( _poclApplicationRequestSupervisor ){
      _poclApplicationRequestSupervisor->vSetTerminate( );
   }
}

tVoid spm_tclLocalApplicationManager::vTraceApplicationsInfo( ){
   SPM_NULL_POINTER_CHECK( _poclDb );
   _poclDb->vTraceApplicationsInfo( this );
}

tVoid spm_tclLocalApplicationManager::vTraceApplicationInfo( tU32 u32AppId ){
   spm_tclConnectedApp *poConApp = NULL;

   SPM_NULL_POINTER_CHECK( _poclDb );

   poConApp = _poclDb->poGetByAppID( u32AppId );

   if ( poConApp == NULL ){
      ETG_TRACE_FATAL( ( "vTraceAppInfo, unknown app %u(%d/0x%x). )",
                         ETG_ENUM( ail_u16AppId, (tU16)u32AppId ), (tU16)u32AppId, (tU16)u32AppId
                         ) );
      return;
   }

   ETG_TRACE_FATAL( ( "Current state of app %u(%d/0x%x)",
                      ETG_ENUM( ail_u16AppId, poConApp->u32GetAppId( ) ), poConApp->u32GetAppId( ), poConApp->u32GetAppId( )
                      ) );
   ETG_TRACE_FATAL( ( "AppStateCurrent      %u",
                      ETG_ENUM( SPM_APPSTATE_DEF, poConApp->u32GetAppStateCurrent( ) )
                      ) );
   ETG_TRACE_FATAL( ( "AppStateRequest      %u",
                      ETG_ENUM( SPM_APPSTATE_DEF, poConApp->u32GetAppStateRequest( ) )
                      ) );
   ETG_TRACE_FATAL( ( "AppStateNewRequest   %u",
                      ETG_ENUM( SPM_APPSTATE_DEF, poConApp->u32GetAppStateNewRequest( ) )
                      ) );
   ETG_TRACE_FATAL( ( "AppStateForceReq     %u",
                      ETG_ENUM( SPM_APPSTATE_DEF, poConApp->u32GetAppStateForceReq( ) )
                      ) );
                      ETG_TRACE_FATAL( ( "AppStateReqStartTime %u",
                      poConApp->u32GetAppStateReqStartTime( )
                      ) );
                      ETG_TRACE_FATAL( ( "AppStateReqEndTime   %u",
                      poConApp->u32GetAppStateReqEndTime( )
                      ) );
                      ETG_TRACE_FATAL( ( "CurrentTimeOsalTime  %u",
                      OSAL_ClockGetElapsedTime( )
                      ) );

   if ( _poclAppCfg ){
      _poclAppCfg->vPrintAppInfo( u32AppId );
   }
} // vTraceApplicationInfo

tVoid spm_tclLocalApplicationManager::vTraceSwBlocksInfo( ){
   SPM_NULL_POINTER_CHECK( _poclDb );
   _poclDb->vTraceSwBlocksInfo( this );
}

/*!
  * \fn
  *  \brief
  *   The functiom provides trace info about a specific SoftwareBlock
  *  \param[in] strName: name of the SoftwareBlock
  *********
  */
tVoid spm_tclLocalApplicationManager::vTraceSwBlockInfo( const std::string& strName ){
   SPM_NULL_POINTER_CHECK( _poclDb );
   _poclDb->vTraceSwBlockInfo( strName, this );
}

tVoid spm_tclLocalApplicationManager::vSetAlternativeApplicationConfiguration( tU32 u32SystemState,
                                                                               tU32 u32Configuration ){
   // first set all application states back to the original default mapping
   if ( _poclAppCfg ){
      _poclAppCfg->vSetDefaultStateTable( u32SystemState );
      if ( !_poclAppCfg->bSetSpecificStateConfiguration( u32SystemState, u32Configuration ) && ( u32Configuration > 0 ) ){
         ETG_TRACE_FATAL( ( "New Configuration %d %d was not set, try to set configuration 0",
                            ETG_ENUM( SPM_SYSTEM_STATES, u32SystemState ),
                            u32Configuration
                            ) );
         // set the default configuration if available
         _poclAppCfg->bSetSpecificStateConfiguration( u32SystemState, 0 );
      } else {
         ETG_TRACE_USR1( ( "New Configuration %d %d was set",
                           ETG_ENUM( SPM_SYSTEM_STATES, u32SystemState ),
                           u32Configuration
                           ) );
      }
   }
} // vSetAlternativeApplicationConfiguration

tVoid spm_tclLocalApplicationManager::vHandleMessage( tU32 u32Message,
                                                      tU32 u32Parm ){

   (void)u32Parm; // currently not used

   if ( SPM_U32_WORKER_LAM_APP_CHECK_REQ == u32Message ){
      ETG_TRACE_USR4( ( "spm_tclLocalApplicationManager::vHandleMessage(): Handle 'SPM_U32_WORKER_LAM_APP_CHECK_REQ'." ) );
      if ( _poclDb ){
         {

            if ( FALSE == _poclDb->bCheckLifeSignals( ) ){
               _poclDb->vOnWatchdogError( this );
            }
         }

         {
            tU32 u32ActiveRequests;
            // check request by LocalApplicationManager and remember failed appId and failed state transition
            vCheckForUnacknowledgedState( u32ActiveRequests );
            #ifdef VARIANT_S_FTR_ENABLE_ALTERNATE_CFG
               if ( _bAlternativConfigChangeRunning ){
                  ETG_TRACE_USR4( ( "SPM_U32_WORKER_LAM_APP_CHECK_REQ received during _bAlternativConfigChangeRunning=1." ) );
                  if ( u32ActiveRequests == 0 ){
                     {
                        _bAlternativConfigChangeRunning = FALSE;
                        if ( OSAL_s32TimerSetTime( _hAlternateConfigTimer, 0, 0 ) != OSAL_OK ){
                           ETG_TRACE_ERRMEM( ( "TRACE_SPM_ERROR" ) );
                        }
                        ETG_TRACE_USR4( ( "AlternativeOn config change was active, sending method result." ) );
                        #ifdef VARIANT_S_FTR_ENABLE_SET_ON_CONFIGURATION
                           spm_corefi_tclMsgSetOnConfigurationMethodResult methodResult;
                           methodResult.Result = 1;
                           SPM_GET_IF_REFERENCE_NEW_VAR( poCcaServiceHandler, ISpmCcaServiceServer );
                           poCcaServiceHandler->vUpdateMethodResult( SPM_COREFI_C_U16_SETONCONFIGURATION, &methodResult );
                        #endif // #ifdef VARIANT_S_FTR_ENABLE_SET_ON_CONFIGURATION
                     }
                  } else {
                     ETG_TRACE_USR4( ( "AlternativeOn config change still active (_u32ActiveRequests=%d), yet no send of method result.", u32ActiveRequests ) );
                  }
               }
            #endif // ifdef VARIANT_S_FTR_ENABLE_ALTERNATE_CFG

         }
      }
   }
}    // vHandleMessage

tBool spm_tclLocalApplicationManager::bHandleSynchrounousCall( tU32   u32Message,
                                                               tVoid *args ){
   SPM_NULL_POINTER_CHECK_VAL( _poclDb );
   SPM_NULL_POINTER_CHECK_VAL( _poclStartupInvest );
   SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poCcaServiceHandler, ISpmCcaServiceServer );

   #ifdef VARIANT_S_FTR_ENABLE_PROCESS_CFG
      spm_ProcessShutdown *poProcShutdown = dynamic_cast < spm_ProcessShutdown* >( _crfFactory.getSpmObjHandler( "spm_ProcessShutdown" ) );
      SPM_NULL_POINTER_CHECK_VAL( poProcShutdown );
   #endif

   switch ( u32Message ){
      case SPM_U32_WORKER_BROADCAST_STARTUP_DELAY:
      {
         if ( !* (tBool*)args ){
            // parameter was false, that means that still some application did not come
            // up with the start state.
            // Here we check again which application is the one which causes the startup
            // to fail.
            // Formerly this functionality was in AppReqMonitor (now here)
                                  ETG_TRACE_FATAL( ( "spm_tclLocalApplicationManager::Startup was not successful" ) );
            // ETG_TRACE_FATAL(("entered critical section"));
            if ( !bAllApplicationsUpAndRunning( _u32CulpritAppID ) ){
               _u32CulpritUnacknowledgedState = AMT_C_U32_STATE_INITIALIZED;
                                  ETG_TRACE_FATAL( ( "The application %u(%d/0x%x) did not acknowledge the state request to INITIALIZED",
                                  ETG_ENUM( ail_u16AppId, (tU16)_u32CulpritAppID ), (tU16)_u32CulpritAppID, (tU16)_u32CulpritAppID
                                  ) );
               // OSAL_s32ThreadWait(3000);
               // vUnacknowledgedApplicationState();
               vOnUnacknowledgeState( * _poclDb->poGetByAppID( _u32CulpritAppID ) );
            } else {
               // ETG_TRACE_FATAL(("All applications seem to be initialized"));
            }
            // ETG_TRACE_FATAL(("Left critical section"));
         } else {
            // otherwise the startup was successfully terminated. Indicate that to the
            // startup investigation.
            ETG_TRACE_USR1( ( "spm_tclLocalApplicationManager::Startup successfully terminated" ) );
            _poclStartupInvest->vStartupPerformed( );
         }
      }
      break;

         #ifdef VARIANT_S_FTR_ENABLE_ALTERNATE_CFG
            case SPM_U32_WORKER_SET_ALTERNATIVE_CONFIG:
            {
               tU32 _u32ActiveRequests = 0;
               ETG_TRACE_USR4( ( "SPM_U32_WORKER_SET_ALTERNATIVE_CONFIG received." ) );
               bSetAllApplicationsInProfile( );
               vCheckForUnacknowledgedState( _u32ActiveRequests );

               if ( _u32ActiveRequests == 0 ){
                  ETG_TRACE_USR4( ( "SPM_U32_WORKER_SET_ALTERNATIVE_CONFIG: No active requests running." ) );
                  {
                     if ( OSAL_s32TimerSetTime( _hAlternateConfigTimer, 0, 0 ) != OSAL_OK ){
                        ETG_TRACE_ERRMEM( ( "TRACE_SPM_ERROR" ) );
                     }
                     #ifdef VARIANT_S_FTR_ENABLE_SET_ON_CONFIGURATION
                        spm_corefi_tclMsgSetOnConfigurationMethodResult methodResult;
                        methodResult.Result = 1;
                        ETG_TRACE_USR4( ( "calling vUpdateMethodResult(SPM_COREFI_C_U16_SETONCONFIGURATION(1)" ) );
                        poCcaServiceHandler->vUpdateMethodResult( SPM_COREFI_C_U16_SETONCONFIGURATION, &methodResult );
                     #else
                        ETG_TRACE_USR4( ( "VARIANT_S_FTR_ENABLE_SET_ON_CONFIGURATION not set, no calling of vUpdateMethodResult(SPM_COREFI_C_U16_SETONCONFIGURATION(1)" ) );
                        poCcaServiceHandler->vUpdateMethodResult( SPM_COREFI_C_U16_SETONCONFIGURATION, &methodResult );
                     #endif // #ifdef VARIANT_S_FTR_ENABLE_SET_ON_CONFIGURATION
                  }
               } else {
                  _bAlternativConfigChangeRunning = TRUE;
                  if ( OSAL_s32TimerSetTime( _hAlternateConfigTimer, SPM_ALTERNATE_CONFIG_TO_MS, 0 ) != OSAL_OK ){
                     ETG_TRACE_ERRMEM( ( "TRACE_SPM_ERROR" ) );
                  }
                     ETG_TRACE_USR4( ( "SPM_U32_WORKER_SET_ALTERNATIVE_CONFIG: %d active request(s) running.", _u32ActiveRequests ) );
               }
            }
            break;

            case SPM_U32_WORKER_ALTERNATIVE_CONFIG_TO:
               if ( _bAlternativConfigChangeRunning ){
                  _bAlternativConfigChangeRunning = FALSE;
                  ETG_TRACE_ERR( ( "ERROR: AlternativeOn config set timeout received. Sending method result." ) );
                  {
                     if ( OSAL_s32TimerSetTime( _hAlternateConfigTimer, 0, 0 ) != OSAL_OK ){
                        ETG_TRACE_ERRMEM( ( "TRACE_SPM_ERROR" ) );
                     }
                     #ifdef VARIANT_S_FTR_ENABLE_SET_ON_CONFIGURATION
                        spm_corefi_tclMsgSetOnConfigurationMethodResult methodResult;
                        methodResult.Result = 1;
                        poCcaServiceHandler->vUpdateMethodResult( SPM_COREFI_C_U16_SETONCONFIGURATION, &methodResult );
                     #endif  // #ifdef VARIANT_S_FTR_ENABLE_SET_ON_CONFIGURATION
                  }

               } else {
                  ETG_TRACE_ERR( ( "ERROR: AlternativeOn config set timeout received, but _bAlternativConfigChangeRunning is not set?" ) );
               }
               break;

                     #endif // ifdef VARIANT_S_FTR_ENABLE_ALTERNATE_CFG
                     #ifdef VARIANT_S_FTR_ENABLE_PROCESS_CFG
                        case SPM_U32_FORCED_PROCESS_CONFIG_SET:
                        {
                           tU32                   u32ProcessConfig = * (tU32*)args;
                           TProcessList           oProcessList;

                                          ETG_TRACE_USR4( ( "LAM: SPM_U32_FORCED_PROCESS_CONFIG_SET (%d)", u32ProcessConfig ) );

                           spm_tclHandleSemaphore sem( _hSemHandle ); // critical section

                           if ( bGetProcessListForConfiguration( u32ProcessConfig, oProcessList ) ){
                                          ETG_TRACE_USR4( ( "LAM: SPM_U32_FORCED_PROCESS_CONFIG_SET  config found for (%d)!", u32ProcessConfig ) );
                              if ( oProcessList.size( ) != 0 ){
                                 TProcessList::iterator it;
                                 tBool                  fNewKillPosted = FALSE;
                                 for ( it = oProcessList.begin( ); it != oProcessList.end( ); ++it ){
                                    TProcessStopMap::iterator itProc = _MapOfProcessesToStop.find( * it );
                                          ETG_TRACE_USR4( ( "Process: %s", it->c_str( ) ) );
                                    if ( itProc != _MapOfProcessesToStop.end( ) ){
                                          ETG_TRACE_USR4( ( "Process: %s is allready in kill list!", it->c_str( ) ) );

                                       /* Process allready in kill list, but not sucessfully killed?
                                          -> Send kill request again.
                                         */
                                       if ( itProc->second.fIsKilled == FALSE ){
                                          ETG_TRACE_USR4( ( "Posting (again) process stop request for: %s", it->c_str( ) ) );
                                          poProcShutdown->bPostStopProcessRequest( * it, itProc->second.u16Id );
                                          fNewKillPosted = TRUE;
                                       } else {
                                          ETG_TRACE_USR4( ( "Process: %s is allready killed", it->c_str( ) ) );
                                       }
                                    } else {
                                       /* New entry*/
                                       TSpmProcStopItem newItem;
                                       newItem.fIsKilled           = FALSE;
                                       newItem.procName            = * it;
                                       newItem.u16Id               = _procStopId++;

                                          ETG_TRACE_USR4( ( "_MapOfProcessesToStop[*it] : %s", it->c_str( ) ) );
                                       _MapOfProcessesToStop[* it] = newItem;

                                       poProcShutdown->bPostStopProcessRequest( * it, newItem.u16Id );
                                       fNewKillPosted              = TRUE;
                                    }

                                 }
                                 /* The processes to kill are allready killed by a previous kill, so send kill sucessfull answer directly here*/
                                 if ( !fNewKillPosted ){
                                          ETG_TRACE_USR4( ( "All processes are allready killed, nothing to do here -> Sending method result." ) );
                                    //#ifdef SPM_COREFI_C_U16_SETPROCESSCONFIGURATION
                                    spm_corefi_tclMsgSetProcessConfigurationMethodResult methodResult;
                                    methodResult.Result = 1;
                                    poCcaServiceHandler->vUpdateMethodResult( SPM_COREFI_C_U16_SETPROCESSCONFIGURATION, &methodResult );
                                    //#endif
                                 }
                              } else {
                                          ETG_TRACE_USR4( ( "No process config found. Sending error method result." ) );
                                 //#ifdef SPM_COREFI_C_U16_SETPROCESSCONFIGURATION
                                 spm_corefi_tclMsgSetProcessConfigurationMethodResult methodResult;
                                 methodResult.Result = 0;
                                 poCcaServiceHandler->vUpdateMethodResult( SPM_COREFI_C_U16_SETPROCESSCONFIGURATION, &methodResult );
                                 //#endif
                              }

                           } else {
                              ETG_TRACE_ERRMEM( ( "LAM: SPM_U32_WORKER_SET_PROCESS_CONFIG not config found for (%d)!", u32ProcessConfig ) );
                              //#ifdef SPM_COREFI_C_U16_SETPROCESSCONFIGURATION
                              spm_corefi_tclMsgSetProcessConfigurationMethodResult methodResult;
                              methodResult.Result = 0;
                              poCcaServiceHandler->vUpdateMethodResult( SPM_COREFI_C_U16_SETPROCESSCONFIGURATION, &methodResult );
                              //#endif
                           }
                        }
                        break;

                        case SPM_U32_FORCED_PROCESS_SET_FINISHED:
                        {
                           ETG_TRACE_USR4( ( "LAM:SPM_U32_FORCED_PROCESS_SET_FINISHED" ) );

                           spm_tclHandleSemaphore    sem( _hSemHandle ); // critical section

                           tU32                      u32Par        = * (tU32*)args;
                           tU16                      u16Id         = ( u32Par >> 16 ) & 0xFFFF;
                           tU16                      u16StopResult = u32Par & 0xFFFF;

                           TProcessStopMap::iterator it;
                           if ( u16StopResult ){
                              for ( it = _MapOfProcessesToStop.begin( ); it != _MapOfProcessesToStop.end( ); ++it ){
                                 if ( it->second.u16Id == u16Id ){
                                    it->second.fIsKilled = TRUE;
                                    ETG_TRACE_ERR( ( "LAM:SPM_U32_FORCED_PROCESS_SET_FINISHED. Process (id=%d) marked as stopped!", u16Id ) );
                                    break;
                                 }
                              }
                           } else {
                              ETG_TRACE_ERRMEM( ( "LAM:SPM_U32_FORCED_PROCESS_SET_FINISHED error. Process (id=%d) cant be stopped!", u16Id ) );
                              /* Send error method result*/
                              //#ifdef SPM_COREFI_C_U16_SETPROCESSCONFIGURATION
                              spm_corefi_tclMsgSetProcessConfigurationMethodResult methodResult;
                              methodResult.Result = 0;
                              poCcaServiceHandler->vUpdateMethodResult( SPM_COREFI_C_U16_SETPROCESSCONFIGURATION, &methodResult );
                              //#endif
                           }

                           tBool fAllProcessesKilled = TRUE;
                           for ( it = _MapOfProcessesToStop.begin( ); it != _MapOfProcessesToStop.end( ); ++it ){
                              if ( it->second.fIsKilled == FALSE ){
                                 fAllProcessesKilled = FALSE;
                              }
                           }
                           if ( fAllProcessesKilled ){
                              ETG_TRACE_USR4( ( "LAM:SPM_U32_FORCED_PROCESS_SET_FINISHED-> All processed stopped, sending method result!" ) );
                              //#ifdef SPM_COREFI_C_U16_SETPROCESSCONFIGURATION
                              spm_corefi_tclMsgSetProcessConfigurationMethodResult methodResult;
                              methodResult.Result = 1;
                              poCcaServiceHandler->vUpdateMethodResult( SPM_COREFI_C_U16_SETPROCESSCONFIGURATION, &methodResult );
                              //#endif
                           }
                           break;
                        }
                     #endif // ifdef VARIANT_S_FTR_ENABLE_PROCESS_CFG

               default:
                  // nothing to do
                  break;
   } // switch
   return( FALSE );
}    // bHandleSynchrounousCall

#ifdef VARIANT_S_FTR_ENABLE_ALTERNATE_CFG
   tVoid spm_tclLocalApplicationManager::vAlternateConfigSetTimerCallback( tVoid *pArg ){
      spm_tclLocalApplicationManager *poLam = (spm_tclLocalApplicationManager*)pArg;

                              ETG_TRACE_USR4( ( "spm_tclLocalApplicationManager::vAlternateConfigSetTimerCallback()" ) );
      poLam->_poclWorkerServer->bPostMessageToWorker( SPM_U32_WORKER_ALTERNATIVE_CONFIG_TO, 0 );
   }

#endif

#ifdef VARIANT_S_FTR_ENABLE_PROCESS_CFG
   tBool spm_tclLocalApplicationManager::bGetProcessListForConfiguration( tU32          u32Configuration,
                                                                          TProcessList& oProcessList ) const {
      tU32                                           as32Data[255];
      OSAL_trIOCtrlDir                               rDirV = { 0 };
      OSAL_tIODescriptor                             fd;
      OSAL_trIOCtrlRegistry                          rReg;

      tBool                                          bUpdated = FALSE;

      std::map < tU32, std::string >::const_iterator it;
      // this result in "DOWNLOADxx"
      OSALUTIL_s32SaveNPrintFormat( (tChar*)as32Data, sizeof( as32Data ), "STOP%d", u32Configuration );

      std::string                                    confname = (const tChar*)as32Data;

      fd = OSAL_IOOpen( SPM_REG_PROCESS_BASE_SPM_PROCESSES_BASE_PATH, OSAL_EN_READONLY );
      if ( fd != OSAL_ERROR ){
         rDirV.fd        = fd;
         rDirV.s32Cookie = 0;

         while ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGENUMVALUE, ( intptr_t )&rDirV ) != OSAL_ERROR ){
            rReg.pcos8Name = rDirV.dirent.s8Name;
            rReg.ps8Value  = (tU8*)as32Data;
            rReg.u32Size   = sizeof( as32Data );

            if ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGGETVALUE, ( intptr_t )&rReg ) != OSAL_ERROR ){
               if ( rReg.s32Type == OSAL_C_S32_VALUE_STRING ){
                  if ( confname == std::string( (const tChar*)rDirV.dirent.s8Name ) ){
                     spm_tclStringTokenizer st( (tChar*)as32Data, ", ", FALSE );
                     while ( st.bHasMoreTokens( ) ){
                        oProcessList.push_back( st.oNextToken( ) );
                        bUpdated = TRUE;
                     }
                     break;
                  }
               }
            }
         }
         OSAL_s32IOClose( fd );
      }
      return( bUpdated );
   } // bGetProcessListForConfiguration

#endif // ifdef VARIANT_S_FTR_ENABLE_PROCESS_CFG

