/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_GlobalApplicationManager.h
  * @brief        This is header file contains functions to control different state of application.
  * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#include <map>

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#include "spm_GlobDefs.h"
#include "spm_IGlobalApplicationManager.h"

class ISpmCcaServiceServer;
class ISpmSubStateClient;
class ISpmSystemPowerManager;
class ISpmLamAppManager;
class ISpmLocalAppManager;
class spm_tclFactory;

// predefined value for the localapplicationmanager
#define SPM_GAM_CCA_LOCAL_LAM                   0

#define SPM_GAM_LAM_STATE_INIT                  0x00
#define SPM_GAM_LAM_STATE_CHANGE_IN_PROGRESS    0x03
#define SPM_GAM_LAM_STATE_CHANGE_PROCESSED      0x04

struct TLamState
{
   TLamState( ){
      _u8State  = SPM_GAM_LAM_STATE_INIT;
      _poRefLam = NULL;
   }

   TLamState( ISpmLocalAppManager *poLam ){
      _u8State  = SPM_GAM_LAM_STATE_INIT;
      _poRefLam = poLam;
   }

   TLamState( const TLamState& op ){
      _u8State  = op._u8State;
      _poRefLam = op._poRefLam;
   }

   TLamState& operator= (const TLamState& roRight)
   {
      if ( this != &roRight )
      {
         _u8State  = roRight._u8State;
         _poRefLam = roRight._poRefLam;
      }
      return *this;
   }

   tU8 _u8State;
   ISpmLocalAppManager *_poRefLam;
};


/*!
  *  \class spm_tclGlobalApplicationManager
  *  \brief This class control different state of application.

          It is used to force applications to some intermidiate state, to kill application , and triger applications to some new state .

        The spm_tclFactory object creates an instance of this class. It is used by substate client to get info.

        It uses theses classes spm_tclLamAppManagerConfig, spm_tclWorkerServerConfig, spm_tclSystemStateManagerConfig,
        spm_tclSubStateHandlerConfig, spm_tclSystemPowerManagerConfig for state related operations.

          spm_tclSubStateHandler uses this class to triger apllication to new state.
  */
class spm_tclGlobalApplicationManager : public ISpmGlobalApplicationManager
{

private:
tU32                         _u32SystemState;
tU32                         _u32SubState;

ISpmLamAppManager           *_poclAppManagerConfig;
ISpmSubStateClient          *_poclSubStateHandler;
ISpmSystemPowerManager      *_poclSystemPowerManager;
ISpmCcaServiceServer        *_poclCcaServiceHandler;
ISpmWorkerServer            *_poclWorkerServer;

std::map < tU32, TLamState > _mapLam;

public:
spm_tclGlobalApplicationManager( const ISpmFactory& factory );
virtual ~spm_tclGlobalApplicationManager( );

virtual tBool bShutdownApplications( );

virtual tVoid vNewSystemStateReached( tU32 u32LamId );

virtual tBool bProcessSystemState( tU32 u32NewSystemState,
                                   tU32 u32SystemStateReached = SPM_SYSTEM_INVALID );

virtual tVoid vForceIntermediateState( tU32  u32Profile,
                                       tBool bForceAll );

virtual tVoid vTriggerAppStateChange( TTriggerMsg *tNewTrigger );

virtual tVoid  vTraceRemainingTime( );

virtual tVoid  vAllSubStatesProcessed( );

virtual tBool  bAddNewLam( tU32                 u32LamConfigId,
                           ISpmLocalAppManager *poRefLam );

protected:
// ISpmBase interface functions
virtual tVoid vHandleTraceMessage( const tUChar *puchData ); //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

virtual const tChar*getName( ) const { return( "spm_tclGlobalApplicationManager" ); }
virtual tVoid vGetReferences( );

virtual tVoid vStartCommunication( );

// Implementation of interface ISpmWorkerClient
virtual tBool bHandleSynchrounousCall( tU32   /*u32Message*/,
                                       tVoid */*args*/ ){ return( FALSE ); }  //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used
virtual tVoid vHandleMessage( tU32 u32Message,
                              tU32 u32Parm = 0 );

};

