/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         systemd_interface.cpp
  * @brief        Implementation of the systemd notify and watchdog abstraction
  * @copyright    (C) 2014 - 2016 Robert Bosch Engineering and Business Solutions Limited.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#include "systemd_interface.h"

//COMPONENT ID currently is SPM, cross check???
#define TR_CLASS_SYSTEMD_INTERFACE ( TR_COMP_SPM + 0x001C )

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SYSTEMD_INTERFACE
    #include "trcGenProj/Header/systemd_interface.cpp.trc.h"
#endif


//prevent c++ name mangling so that symbol can be looked up by dl_sym() for dynamic loading
#ifdef __cplusplus
   extern "C"
   {
   systemd_interface*GetSystemdInterface( SystemdIF *component,
                                          uint32_t   Appid ){
      return( new systemd_interface( component, Appid ) );

   }

   }
#endif

systemd_interface::systemd_interface( SystemdIF *AppSystemdIF,
                                      uint32_t   CcaAppId ) :
   u_CcaAppId( CcaAppId ),
   watchdog_interval( 0 ),
   _poenv_config( NULL ),
   timerID( 0 ),
   _poAppSystemdIF( AppSystemdIF ){
   //First clear up the garbage values in the memory locations
   memset( &SignalEvent,  0, sizeof( SignalEvent ) );
   memset( &SignalAction, 0, sizeof( SignalAction ) );
   memset( &timerSpecs,   0, sizeof( timerSpecs ) );

   ETG_TRACE_USR4( ( "get the pre-configured watchdog interval for APPID %d with processid %d", u_CcaAppId, getpid( ) ) );

   //Get the environment variable from the process environment,it is in microseconds
   _poenv_config = getenv( "WATCHDOG_USEC" );

   if ( NULL == _poenv_config ){
      ETG_TRACE_FATAL( ( "Check if you have configured it in .service file for APPID %d with processid %d", u_CcaAppId, getpid( ) ) );
   } else {
      watchdog_interval = atoi( _poenv_config ) / ( 1000000 ) / 2;

      ETG_TRACE_USR1( ( "atoi output is %d for AppId %d", watchdog_interval, u_CcaAppId ) );

      if ( watchdog_interval > 0 ){
         // Define the timer specification
         //first occurence
         timerSpecs.it_value.tv_sec     = watchdog_interval;
         timerSpecs.it_value.tv_nsec    = 0;
         //repeated occurences
         timerSpecs.it_interval.tv_sec  = watchdog_interval;
         timerSpecs.it_interval.tv_nsec = 0;
         /* Install timer_handler as the signal handler for SIGVTALRM. */

         // Clear the sa_mask
         sigemptyset( &SignalAction.sa_mask );

         // set the SA_SIGINFO flag to use the extended signal-handler function
         SignalAction.sa_flags             = SA_SIGINFO;

         // This method is installed as a callback for the interested signal
         SignalAction.sa_sigaction         = systemd_interface::timer_expiry_handler;

         // Set the notification method as that of signalling
         SignalEvent.sigev_notify          = SIGEV_SIGNAL;

         // Pass a pointer to the object(This is done for 'C' compatibility)
         //Use static cast whenever you come across 'c' compatibility in a c++ application
         SignalEvent.sigev_value.sival_ptr = static_cast < void* >( _poAppSystemdIF );

         // Let the signal be given periodically
         SignalEvent.sigev_signo           = SIGALRM;

         // Create the Timer
         if ( timer_create( CLOCK_MONOTONIC, &SignalEvent, &timerID ) != 0 ){
            ETG_TRACE_FATAL( ( "Timer creation fails for APPID %d with processid %d", u_CcaAppId, getpid( ) ) );
         }

         // Specify the action to be taken once the event occurs
         if ( sigaction( SIGALRM, &SignalAction, NULL ) ){
            ETG_TRACE_FATAL( ( "Failed to specify a action for event for APPID %d with processid %d", u_CcaAppId, getpid( ) ) );
         }
      } else {
         ETG_TRACE_FATAL( ( "Check your environment configuration" ) );
      }
   }
}

systemd_interface::~systemd_interface( ){
   ETG_TRACE_USR4( ( "systemd_interface destructor called for APPID %d with process id %d", u_CcaAppId, getpid( ) ) );

   if ( timer_delete( timerID ) < 0 ){
      ETG_TRACE_USR4( ( "Timer deletion failed for APPID %d with process id %d", u_CcaAppId, getpid( ) ) );
   }
   timerID         = 0; //lint fix
   _poAppSystemdIF = NULL;
   _poenv_config   = NULL;
}

void systemd_interface::systemd_notify_ready( ){

   ETG_TRACE_USR4( ( "Notifying service state as ready for APPID %d with processid %d", u_CcaAppId, getpid( ) ) );

   if ( sd_notify( 0, "READY=1" ) < 0 ){
      ETG_TRACE_FATAL( ( "systemd notification failed for APPID %d", u_CcaAppId ) );
   }

   //proceed only if configuration is appropriate

   if ( watchdog_interval > 0 ){
      //Start the watchdog notification timer
      if ( timer_settime( timerID, 0, &timerSpecs, NULL ) == - 1 ){
         ETG_TRACE_FATAL( ( "Could not start timer for AppId:%d", u_CcaAppId ) );
      }
   }
} // systemd_interface::systemd_notify_ready

void systemd_interface::timer_expiry_handler( int        sigNumb,
                                              siginfo_t *si,
                                              void      *uc ){
   (void )sigNumb; //lint fix
   uc=uc;          //lint fix

   // get the pointer out of the siginfo structure and type cast it,stick to static_cast
   SystemdIF *__poAppSystemdIF = static_cast < SystemdIF* > ( si->si_value.sival_ptr );

   if ( NULL != __poAppSystemdIF ){
      // call the member function
      if ( __poAppSystemdIF->OnAppWatchdog( ) ){
         //ping systemd with watchdog notification
         (void) sd_notify( 0, "WATCHDOG=1" );
      }
   }
} // systemd_interface::timer_expiry_handler

