/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         StartupControllerComponent.cpp
  * @brief        Implementation of the StartupControllerComponent component
  * @copyright    (C) 2015 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#include "StartupControllerComponent.h"
#include "asf/core/Application.h"
#include <systemd/sd-daemon.h> // you also need to add -lsystemd-daemon to the linker settings
#include <algorithm>
#include <iterator>
#define EARLY_TIMEOUT_START_USEC 10000000 // 10 000 000 usec = 10 sec
// \todo: boost::shared_ptr is used here cause by ASF interface.
namespace org {
namespace bosch {
namespace cm  {
namespace lcm {
namespace startupcontroller {

using namespace ::asf::core;
using namespace ::org::bosch::cm::lcm;

/*! \fn
  *  \brief function to escape the unit names
  *    specified in
  *    copied from systemd sources at: systemd-217\src\shared\unit-name.c
  *  \param UnitName original name of the unit
  *  \exception none.
  *  \return new escaped unitname
  */
std::string StartupControllerComponent::unitNameEscape( const std::string UnitName ){
   std::locale loc;
   std::string EscapedUnitName;

   for ( std::string::const_iterator it = UnitName.begin( ); it != UnitName.end( ); ++it ){
      if ( std::isalnum( * it, loc ) ){
         EscapedUnitName += * it;
      } else {
         int               character = * it;
         std::stringstream strstream;
         strstream << "_" << std::hex << character;
         EscapedUnitName += strstream.str( );
      }
   }
   return( EscapedUnitName );
}              // unitNameEscape

/**
  *  \brief Initialize supported command line parameters
  *
  *  currently these command line parameters are supported:
  *  - StartupTime=<sec>
  */
const StartupControllerComponent::CliParamItem StartupControllerComponent::_SupportedCliParams[] = {
   { StartupTimeType, std::string( "StartupTime=" ) }
};

/**
  *  \brief Constructor, initialize variables
  *
  *  \param [in]
  *  \return
  */
StartupControllerComponent::StartupControllerComponent( )
   : BaseComponent( ),
   StartupcontrollerStub( "StartupControllerPort" ),
   startupStatus( StartupState__STARTUP_INIT ),
   _StartupTimeSec( 0 ),
   _TimeoutStartUSec( 0 ),
   _PreTimeoutStartSec( 0 ){
   LOG_INFO( "StartupControllerComponent called" );
   // get arguments from command line
   const std::vector < std::string > arguments = asf::core::Application::getApplication( )->getArguments( );
   _ServiceName = "StartupController.service";
   std::string escapedServiceName = unitNameEscape( _ServiceName );
   std::string objectPath         = "/org/freedesktop/systemd1/unit/" + escapedServiceName;
   LOG_INFO( "Create ServiceProxy for objectpath - %s", objectPath.c_str( ) );
   _serviceProxy = ServiceProxy::createProxy( "DBusServiceClientPort",    // portName
                                              "org.freedesktop.systemd1", // busName
                                              objectPath,                 // objectPath
                                              DBUS_BUS_SYSTEM,            // busType
                                              * this                      // ServiceAvailableIF
                                              );

   LOG_INFO( "ArgumentListSize = %i", arguments.size( ) );
   if ( arguments.size( ) > 0 ){
      ProcessCommandLineArguments( arguments );
   }
}



/**
  *  \brief perform processing command line parameters
  *
  *  \param [in] arguments contain all command line parameters
  *  \return
  */
void StartupControllerComponent::ProcessCommandLineArguments( const std::vector < std::string >& arguments ){
   std::vector < std::string >::const_iterator cit;
   for ( cit = arguments.begin( ); cit != arguments.end( ); ++cit ){
      std::string  argument = ( * cit );
      LOG_INFO( "Argument%i = %s", ( cit - arguments.begin( ) ), argument.c_str( ) );
      CliParamType pType    = GetParamType( argument );
      if ( pType != CliParamTypeMax ){
         size_t                pos         = _SupportedCliParams[pType].KeyString.length( );
         // get characters of the value-part of input argument
         std::string           ValueString = argument.substr( pos );

         std::string::iterator it;
         switch ( pType ){
            case StartupTimeType:
            {
               // only in debug mode
               it = find_if( ValueString.begin( ), ValueString.end( ), std::not1( IsDigit( ) ) );
               if ( ( ValueString.empty( ) == FALSE ) && ( it == ValueString.end( ) ) ){
                  // all characters are digits => StartupTime has a valid value
                  _StartupTimeSec = std::atoi( argument.substr( pos ).c_str( ) );
                  if ( _StartupTimeSec > 0 ){
                     _StartupTimeEmergencyTimer.start( * this, _StartupTimeSec * 1000, 1 );
                     LOG_INFO( "StartupTimeType::Auto notify READY to systemd after '%d' sec", _StartupTimeSec );
                  }else{
                     LOG_ERROR( "Parameter: 'StartupTime' must be a positive value: '%s' ", ValueString.c_str( ) );
                  }
               } else {
                  LOG_ERROR( "Parameter: 'StartupTime' has invalid value: %s", ValueString.c_str( ) );
               }
               break;
            }
            default:
               break;
         } // switch
      } else {
         LOG_ERROR( "Unknown parameter: %s", argument.c_str( ) );
      }
   }
} // StartupControllerComponent::ProcessCommandLineArguments

/**
  *  \brief get the type of command line parameter
  *
  *  \param [in] argument contains specific command line parameter
  *  \return the type of specific command line parameter
  */
StartupControllerComponent::CliParamType StartupControllerComponent::GetParamType( std::string argument ){
   CliParamType   ret = CliParamTypeMax;
   unsigned short idx;
   for ( idx = 0; idx < CliParamTypeMax; ++idx ){
      std::string::iterator resultIt = std::search( argument.begin( ),
                                                    argument.end( ),
                                                    _SupportedCliParams[idx].KeyString.begin( ),
                                                    _SupportedCliParams[idx].KeyString.end( ) );
      if ( ( resultIt != argument.end( ) ) && ( resultIt == argument.begin( ) ) ){
         ret = _SupportedCliParams[idx].Type;
         break;
      }
   }

   return( ret );
} // StartupControllerComponent::GetParamType

/**
  *  \brief Destructor - currenlty nothing is done
  *
  *  \param [in]
  *  \return
  */
StartupControllerComponent::~StartupControllerComponent( ){
   LOG_INFO( "~StartupControllerComponent called" );
}

/**
  *  \brief handler of Get method for Fi_Version
  *
  *  \param [in] returns current version of the interface - automatically generated from fidl file
  *  \return
  */
void StartupControllerComponent::onFI_VersionGet( const ::boost::shared_ptr < FI_VersionGet >& payload ){
   LOG_INFO( "StartupControllerComponent::onFI_VersionGet called: %s", payload->getFI_Version( ).c_str( ) );
}

/**
  *  \brief handler of Get method for StartupStatus
  *
  *  \param [in] returns "undefined" all the time, as program will exit when state "Startup_Done" is received
  *  \return
  */
void StartupControllerComponent::onStartupStatusGet( const ::boost::shared_ptr < StartupStatusGet >& payload ){
   std::string StartupStateString( "undefined" );

   if ( payload->hasStartupStatus( ) ){
      StartupState startupState     = payload->getStartupStatus( );
      const char  *StartupStateName = StartupState_Name( startupState );
      if ( StartupStateName != NULL ){
         StartupStateString = std::string( StartupStateName );
      } else {
         // nothing to do, already set to "undefined"
      }
   }
   LOG_INFO( "StartupControllerComponent::onStartupStatusGet called - %s", StartupStateString.c_str( ) );
   sendStartupStatusGetUpdate( startupStatus );
}

/**
  *  \brief handler of Set method for StartupStatus
  *
  *  \param [in] payload information string of LCM
  *  \return
  */
void StartupControllerComponent::onStartupStatusSet( const ::boost::shared_ptr < StartupStatusSet >& payload ){
   if ( payload->hasStartupStatus( ) ){
      std::string  StateString      = "LCM reported State as ";
      StartupState _startupState    = payload->getStartupStatus( );
      const char  *startupStateName = StartupState_Name( _startupState );
      if ( startupStateName != NULL ){
         std::string stateString( startupStateName );
         StateString  += stateString;
         startupStatus = _startupState;
         if ( StartupState__STARTUP_DONE == startupStatus ){
             StopStartupController( );
         }
      } else {
            StateString  += "undefined - resetting internal state to STARTUP_INIT";
            startupStatus = StartupState__STARTUP_INIT;
      }
   } else {
        LOG_ERROR( "StartupControllerComponent::onStartupStatusSet(): No startupState found" );
   }
}              // onStartupStatusSet

/**
  *  \brief TimerCallbackIF implementation
  *
  *  the only timer currently used is the Startup simulation timer
  *  comandline parameter "StartupTime=<sec>"
  *
  *  \param [in] timer reference to the expired timer
  *  \param [in] payload passed payload of started timer. Currently not used
  *  \return
  */
void StartupControllerComponent::onExpired( asf::core::Timer                            & timer,
                                            boost::shared_ptr < asf::core::TimerPayload > payload ){
    (void)payload;  // payload is not needed here.
    LOG_INFO( "StartupControllerComponent::onExpired ..........." );
    if ( ( timer.getAct( ) == _StartupTimeEmergencyTimer.getAct( ) ) && ( &timer == &_StartupTimeEmergencyTimer ) ){
        LOG_INFO( "StartupControllerComponent:: Debug mode, StartupTime=%d is expired ! ", _StartupTimeSec );
        // set FailureIndication to indicate timer of "StartupTime=<Sec>" expired
        StartupFailureInfo failureInfo( startupStatus,
                                        StartupFailure__STARTUP_FAILURE_DEBUGGING_STARTUPTIME_EXPIRED,
                                        _StartupTimeSec );
        setStartupFailureIndication( failureInfo );
        // auto set to startup done status.
        StopStartupController( );
    } else if(( timer.getAct( ) == _TimeoutStartTimer.getAct( ) )&& ( &timer == &_TimeoutStartTimer )){
        LOG_INFO( "StartupControllerComponent:: Normal mode, _TimeoutStartTimer=%d is expired ! ", _PreTimeoutStartSec );
        // set FailureIndication to indicate timer of "_TimeoutStartTimer=<Sec>" expired
        StartupFailureInfo failureInfo( startupStatus,
                                        StartupFailure__STARTUP_FAILURE_STARTUPTIME_EXPIRED,
                                        _PreTimeoutStartSec );
        setStartupFailureIndication( failureInfo );
    }else {
        LOG_WARN( "unknown timer" );
    }
}              // onExpired

/**
  *  \brief Stops the application
  *
  *  systemd status is set to keep track of the current application status
  *  application exit is called. This function never returns
  *
  *  \param [in]
  *  \return
  */
void StartupControllerComponent::StopStartupController( void ){
    LOG_INFO( "StartupControllerComponent::StopStartupController is triggered ! " );
    sd_notifyf( 0, "READY=1\n"
                   "STATUS=Startup finished\n"
                   "MAINPID=%lu\nPARENTID=%lu",
                   (unsigned long)getpid( ),
                   (unsigned long)getppid( ) );
    sd_notifyf( 0, "STOPPING=1\n"                            // Tells the init system that service is beginning its shutdown.
                   "STATUS=Startup Controller was stopped\n" // Passes a single-line status string back to the init system that describes the daemon state.
                                                            // This is free-form and can be used for various purposes
                   "MAINPID=%lu\nPARENTID=%lu",
                   (unsigned long)getpid( ),
                   (unsigned long)getppid( ) );

    if ( FALSE == asf::core::Application::getApplication( )->shutdownCurrentApplication( 0 /* exitcode */ ) ){
        LOG_FATAL( "Shutdown failed" );
        // set FailureIndication to indicate shutdown of StartupController failed
        StartupFailureInfo failureInfo( startupStatus,
                                        StartupFailure__STARTUP_FAILURE_SHUTDOWN_FAILED,
                                        getpid( ) );
        setStartupFailureIndication( failureInfo );
        _exit( 0 );
    }
    _exit( 0 );
}              // onExpired



// ServiceAvailableIF implementation
void StartupControllerComponent::onAvailable( const boost::shared_ptr < Proxy >& proxy, const ServiceStateChange         & stateChange ){
    (void)stateChange;
    LOG_INFO( " StartupControllerComponent::  - ServiceProxy is onAvailable" );
   if ( proxy && ( _serviceProxy == proxy ) ){
      LOG_INFO( "  - ServiceProxy is connected" );

      // set FailureIndication to indicate the cli paramerter "StartupTime=<Sec>" is used
      if(_StartupTimeSec > 0){
          LOG_INFO( "PreTimeoutSecType::Auto notify READY to systemd after '%d'sec", _StartupTimeSec );
          StartupFailureInfo failureInfo( startupStatus,
                                          StartupFailure__STARTUP_FAILURE_DEBUGGING_STARTUPTIME_AVAILABLE,
                                          _StartupTimeSec );
          setStartupFailureIndication( failureInfo );
      }else{
          _serviceProxy->sendTimeoutStartUSecGet( * this );
      }
   }
}

void StartupControllerComponent::onUnavailable( const boost::shared_ptr < Proxy >& proxy, const ServiceStateChange         & stateChange ){
    (void)stateChange;
    LOG_ERROR( " StartupControllerComponent::  - ServiceProxy is onUnavailable" );
   if ( proxy && ( _serviceProxy == proxy ) ){
      LOG_ERROR( "  - ServiceProxy is connected" );
      _serviceProxy->sendTimeoutStartUSecDeregisterAll();
   }
}

// Callback 'TimeoutStartUSecCallbackIF'

void StartupControllerComponent::onTimeoutStartUSecError(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< TimeoutStartUSecError >& error) {
   if ( ( _serviceProxy == proxy ) && error->hasName( ) && error->hasMessage( ) ){
      LOG_ERROR( "Received onTimeoutStartUSecError: ErrorName=%s, ErrorMessage=%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
   } else {
      LOG_ERROR( "StartupControllerComponent::onTimeoutStartUSecError: proxy not available" );
   }
}

void StartupControllerComponent::onTimeoutStartUSecUpdate(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< TimeoutStartUSecUpdate >& update) {
    LOG_INFO( "StartupControllerComponent::onTimeoutStartUSecUpdate: proxy is updated" );
    if ( ( _serviceProxy == proxy ) && update->hasTimeoutStartUSec( ) ){
      _TimeoutStartUSec = (uint32)update->getTimeoutStartUSec( );
      LOG_INFO( "StartupControllerComponent::Received onTimeoutStartUSecUpdate: _TimeoutStartUSec: %d(uSec) ", _TimeoutStartUSec);
      if(_TimeoutStartUSec > EARLY_TIMEOUT_START_USEC){
          _PreTimeoutStartSec = (_TimeoutStartUSec - EARLY_TIMEOUT_START_USEC)/1000000;
          LOG_INFO( " StartupControllerComponent:: PreTimeoutStart = %d(sec) ", _PreTimeoutStartSec );
          _TimeoutStartTimer.start(*this, _PreTimeoutStartSec * 1000, 1);
          StartupFailureInfo failureInfo( startupStatus,
                                         StartupFailure__STARTUP_FAILURE_STARTUPTIME_AVAILABLE,
                                         _PreTimeoutStartSec );
          setStartupFailureIndication( failureInfo );
      }else {
          LOG_ERROR( " StartupControllerComponent:: TimeoutStart should be more than 10 seconds, currently TimeoutStart = %d ", _TimeoutStartUSec );
     }
   } else {
      LOG_ERROR( "StartupControllerComponent::onTimeoutStartUSecUpdate: proxy not available" );
   }
}

DEFINE_CLASS_LOGGER_AND_LEVEL( "org.bosch.cm.lcm.startupcontroller", StartupControllerComponent, Info );

}
}
}
}
}

