/*!
  * \file     LcmRecoveryClientComponent.cpp
  * \brief    Base class to handle failures of systemd units. This class will be the
  *           base of the GenericRecoveryClient app that is started via
  *           tag "OnFailure="
  *
  * \author   rmi2hi
  *
  * \par Copyright:
  * (C) 2013 - 2016 Robert Bosch GmbH.
  *   The reproduction, distribution and utilization of this file as well as the
  *   communication of its contents to others without express authorization is prohibited.
  *   Offenders will be held liable for the payment of damages.
  *   All rights reserved in the event of the grant of a patent, utility model or design.
  ***************************************************************************/

#include "LcmRecoveryClientComponent.h"
#include "asf/core/Application.h"
#include <systemd/sd-daemon.h> // you also need to add -lnotify to the linker settings
#include <cstdlib>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>
#include <locale>
#include <signal.h>
#include <fstream>
// \todo: boost::shared_ptr is used here cause by ASF interface.
namespace org {
namespace bosch {
namespace cm  {
namespace lcm {
namespace genericrecoveryclient {

using namespace ::asf::core;
using namespace ::org::bosch::cm::lcm;
using namespace ::org::bosch::cm::lcm::genericrecoveryclient;
using namespace ::org::bosch::cm::lcm::genericrecoveryclient::Generic_RecoveryClient;
using namespace ::org::freedesktop::systemd1::Unit;
using namespace ::org::freedesktop::systemd1::Service;
using namespace ::org::freedesktop::systemd1::Manager;

#define LCMRECOVERYCLIENT_MAX_CHECK_COUNTER  3

LcmRecoveryClientComponent::LcmRecoveryClientComponent( )
   : iCounter( 0 ){
   LOG_INFO( "LcmRecoveryClientComponent called" );

   // ERROR should be dumped to ErrMem
   LOG_ERROR( "Error in LCM detected. Collecting debug info and reset." );
}

LcmRecoveryClientComponent::~LcmRecoveryClientComponent( ){
   LOG_INFO( "~LcmRecoveryClientComponent called" );
}

// ServiceAvailableIF implementation
void LcmRecoveryClientComponent::onAvailable( const boost::shared_ptr < Proxy >& proxy,
                                              const ServiceStateChange         & stateChange ){
   if ( stateChange.hasCurrentState( ) ){
      LOG_INFO( "LcmRecoveryClientComponent::onAvailable: ServiceState: %s", ServiceState_Name( stateChange.getCurrentState( ) ) );
      GenericRecoveryClientComponent::onAvailable( proxy, stateChange );
   } else {
      LOG_ERROR( "LcmRecoveryClientComponent::onAvailable: service unknown" );
   }
}              // onAvailable

void LcmRecoveryClientComponent::onUnavailable( const boost::shared_ptr < Proxy >& proxy,
                                                const ServiceStateChange         & stateChange ){
   if ( stateChange.hasCurrentState( ) ){
      LOG_INFO( "LcmRecoveryClientComponent::onUnavailable: ServiceState: %s", ServiceState_Name( stateChange.getCurrentState( ) ) );
      GenericRecoveryClientComponent::onUnavailable( proxy, stateChange );
   } else {
      LOG_ERROR( "LcmRecoveryClientComponent::onUnavailable: service unknown" );
   }
}              // onUnavailable

// *************
// UserCallbacks
// *************

/**
  * callback from GenericRecoveryClientComponent::onActiveStateUpdate to do application specific stuff
  */
void LcmRecoveryClientComponent::_onActiveStateUpdate( const ::boost::shared_ptr < UnitProxy >        & proxy,
                                                       const ::boost::shared_ptr < ActiveStateUpdate >& update,
                                                       uint8                                            CallStackSignalCounter ){
   if ( ( _unitProxy == proxy ) && update->hasActiveState( ) ){
      // ERROR should be dumped to ErrMem
      LOG_ERROR( "LcmRecoveryClientComponent::Received _onActiveStateUpdate: Service=%s, Activestate=%s, CallStackSignalCounter=%d", _ServiceName.c_str( ), update->getActiveState( ).c_str( ), CallStackSignalCounter );
      LOG_ERROR( "LcmRecoveryClientComponent::Received Callstack generation is triggered" );
   } else {
      LOG_ERROR( "LcmRecoveryClientComponent::_onActiveStateUpdate: proxy not available" );
   }
}

/**
  * callback from GenericRecoveryClientComponent::onActiveStateUpdate to do application specific stuff
  */
void LcmRecoveryClientComponent::_onActiveStateError( const ::boost::shared_ptr < UnitProxy >       & proxy,
                                                      const ::boost::shared_ptr < ActiveStateError >& error,
                                                      uint8                                           CallStackSignalCounter ){
   if ( ( _unitProxy == proxy ) && error->hasName( ) && error->hasMessage( ) ){
      LOG_ERROR( "LcmRecoveryClientComponent::Received onJobRemovedError: ErrorName=%s, ErrorMessage=%s, CallStackSignalCounter=%d", error->getName( ).c_str( ), error->getMessage( ).c_str( ), CallStackSignalCounter );
   } else {
      LOG_INFO( "LcmRecoveryClientComponent::_onActiveStateError: proxy not available" );
   }
}

// *************
// Timer
// *************
// TimerCallbackIF implementation

/**
  * implementation of virtual function.
  * BaseFunction in GenericRecoveryClientComponent needs to be called
  */
void LcmRecoveryClientComponent::onExpired( asf::core::Timer                            & timer,
                                            boost::shared_ptr < asf::core::TimerPayload > payload ){
   LOG_INFO( "LcmRecoveryClientComponent::onExpired: Timer:%u", timer.getAct( ) );
   if ( payload->hasReason( ) && payload->hasRepeatCount( ) ){
      LOG_INFO( "LcmRecoveryClientComponent::onExpired: TimerPayload Reason: %s, Repeat Count:%u", TimerPayload_Reason_Name( payload->getReason( ) ), payload->getRepeatCount( ) );
   }
   if ( ( timer.getAct( ) == _CheckTimer.getAct( ) )
        && ( &timer == &_CheckTimer )
        ){
      iCounter++;
      if ( iCounter == 1 ){
         // ERROR should be dumped to ErrMem
         LOG_ERROR( "dumping status info of service file" );
         std::string strCmdLine = "systemctl status " + _ServiceName + " > /dev/errmem";
         LOG_ERROR( "command=%s", strCmdLine.c_str( ) );
         system( strCmdLine.c_str( ) ); // OK => This system() call runs WITHOUT root permission. Required group memberships : 'shell', 'adit_errmem_log'.
         strCmdLine = "journalctl --unit=" + _ServiceName + " > /dev/errmem";
         LOG_ERROR( "command=%s", strCmdLine.c_str( ) );
         system( strCmdLine.c_str( ) ); // OK => This system() call runs WITHOUT root permission. Required group memberships : 'shell', 'adit_errmem_log'.
      }
      if ( iCounter >= LCMRECOVERYCLIENT_MAX_CHECK_COUNTER ){
         // ERROR should be dumped to ErrMem
         LOG_ERROR( "Exit - service file resets now" );
         exit( EXIT_FAILURE );  // just quit. The service file is configured to reset
      }
   }
   // just to be paranoid. If exit fails do not call notify anymore
   if ( iCounter < LCMRECOVERYCLIENT_MAX_CHECK_COUNTER ){
      GenericRecoveryClientComponent::onExpired( timer, payload );
   }
}              // onExpired

DEFINE_CLASS_LOGGER_AND_LEVEL( "org.bosch.cm.lcm.RecoveryClient", LcmRecoveryClientComponent, Info );

}
}
}
}
}

