/****************************************************************************
* Copyright (C) Robert Bosch Car Multimedia GmbH, 2018
* This software is property of Robert Bosch GmbH. Unauthorized
* duplication and disclosure to third parties is prohibited.
*
* FILE                : PluginClientHandler.h
* COMPONENT Name      : PluginClientHandler
* DESCRIPTION         : DBus client for plugins
*                       Receives command requests from HMI and sends it to plugin
*                       Receives updates from plugin and sends it to HMI
* AUTHOR              : Preethi Alagappan
* Date                : 14.09.2018
* Revision History    : 0.1
* Date 14.09.2018     : Initial version
* Revision History    : 0.2
* Date 25.10.2018     : Removed condition check for plugin and control block name.
*                       Moved the code, which converts EventDataUtility to Dbus variant
*                       and vice versa, to PluginDataConverter
* Revision History    : 0.3
* Date 15.11.2018     : Implementation for PluginListCommandRequest and PluginListUpdateSignal
*                       Register for plugin signal updates based on plugin and control block names
****************************************************************************/

#ifndef PLUGIN_CLIENT_HANDLER_HEADER
#define PLUGIN_CLIENT_HANDLER_HEADER

/*****************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------*/

/*****************************************************************
| includes
|----------------------------------------------------------------*/
#include "AppBase/ServiceAvailableIF.h"
#include "PluginDbusInterfaceProxy.h"
#include "ProjectPluginMsgs.h"

using namespace ::asf::core;
using namespace ::bosch::cm::ai::PluginDbusInterface;

class PluginClientHandler
   : public hmibase::ServiceAvailableIF
   , public StartupSync::PropertyRegistrationIF
   , public ::bosch::cm::ai::PluginDbusInterface::PluginCommandCallbackIF
   , public ::bosch::cm::ai::PluginDbusInterface::PluginUpdateCallbackIF
   , public ::bosch::cm::ai::PluginDbusInterface::PluginListCommandCallbackIF
   , public ::bosch::cm::ai::PluginDbusInterface::PluginListUpdateCallbackIF
   , public ::bosch::cm::ai::PluginDbusInterface::EndStationListUpdateCallbackIF
   , public ::bosch::cm::ai::PluginDbusInterface::EntityDetailsUpdateCallbackIF
{
   public:
      static PluginClientHandler* getInstance();
      virtual ~PluginClientHandler();

      bool registerPluginSignals(const std::map<std::string, std::vector<std::string> >& signalMap);

      void onPluginCommandError(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< PluginCommandError >& error);
      void onPluginCommandResponse(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< PluginCommandResponse >& response);
      void onPluginCommandRequest(const std::string& pluginName,
                                  const std::string& ctrlBlockName,
                                  const boost::shared_ptr<EventDataUtility>& eventData);

      void onPluginListCommandError(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< PluginListCommandError >& error);
      void onPluginListCommandResponse(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< PluginListCommandResponse >& response);
      void onPluginListCommandRequest(const std::string& pluginName,
                                      const std::string& ctrlBlockName,
                                      const boost::shared_ptr<EventDataUtility>& eventData,
                                      const boost::shared_ptr<EventListDataUtility>& listData);

      void onPluginUpdateError(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< PluginUpdateError >& error);
      void onPluginUpdateSignal(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< PluginUpdateSignal >& signal);

      void onPluginListUpdateError(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< PluginListUpdateError >& error);
      void onPluginListUpdateSignal(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< PluginListUpdateSignal >& signal);
      virtual void onEndStationListUpdateError(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< EndStationListUpdateError >& error);
      virtual void onEndStationListUpdateUpdate(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< EndStationListUpdateUpdate >& update);
      virtual void onEntityDetailsUpdateError(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< EntityDetailsUpdateError >& error);
      virtual void onEntityDetailsUpdateUpdate(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< EntityDetailsUpdateUpdate >& update);
      void vRequestRouterStatusUpdate();
      void vRequestCamportStatusUpdate();

      void sendEntityDetailsUpdateGetRequest()
      {
         if (NULL != m_poPluginDbusProxy.get())
         {
            m_poPluginDbusProxy->sendEntityDetailsUpdateGet(*this);
         }
      }

      COURIER_MSG_MAP_BEGIN(0)
      ON_COURIER_MESSAGE(PluginCommandReqMsg)
      ON_COURIER_MESSAGE(PluginListCommandReqMsg)
      COURIER_MSG_MAP_END()

      bool onCourierMessage(const PluginCommandReqMsg& oMsg);
      bool onCourierMessage(const PluginListCommandReqMsg& oMsg);

   protected:

      void registerProperties(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange);
      void deregisterProperties(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange);

   private:
      PluginClientHandler();
      PluginClientHandler(const PluginClientHandler&);
      PluginClientHandler& operator = (const PluginClientHandler&);
      bool isSignalRegistered(const std::string& pluginName, const std::string& ctrlBlockName);

      std::map<std::string, std::vector<std::string> > pluginCtrlBlockMap;
      ::boost::shared_ptr<PluginDbusInterfaceProxy> m_poPluginDbusProxy;
};


#endif
