/**
 * @file         : VehicleServiceClient.cpp
 * @author       :
 * @addtogroup   : AppHmi_App_PRJ_Common
 * @brief        : VehicleServiceClient is to handle the vehicle service
 *                 client implementation
 * @copyright    : (C) 2016 Robert Bosch GmbH
 *                 (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#include "VehicleServiceClient.h"
#include "unistd.h"
#include "CommonTrace.h"
#include "hmi_trace_if.h"
#include <Trace/TraceUtils.h>


using namespace ::hmibase::trace;
using namespace ::VEHICLE_MAIN_FI;
using namespace ::vehicle_main_fi_types;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_COMMON
#include "trcGenProj/Header/VehicleServiceClient.cpp.trc.h"
#endif /* VARIANT_S_FTR_ENABLE_TRC_GEN */


namespace languageHandler {


VehicleServiceClient::VehicleServiceClient()
{
   ETG_TRACE_USR1(("VehicleServiceClient: INFO: APP: %25s : PID: %d : CTOR",  getAppName().c_str(), getpid()));
   clear();
}


VehicleServiceClient::~VehicleServiceClient()
{
   ETG_TRACE_USR1(("VehicleServiceClient: INFO: APP: %25s : PID: %d : DTOR",  getAppName().c_str(), getpid()));
   clear();
}


VehicleServiceClient::VehicleServiceClient(const VehicleServiceClient& obj)
{
   ETG_TRACE_USR1(("VehicleServiceClient: INFO: APP: %25s : PID: %d : CTOR",  getAppName().c_str(), getpid()));
   clear();
   *this = obj;
}


VehicleServiceClient& VehicleServiceClient::operator= (const VehicleServiceClient& obj)
{
   if (this != &obj)
   {
      clear();
   }
   return *this;
}


void VehicleServiceClient::clear()
{
   _languageStatus_RegId = 0;
   _configurator         = NULL;
   _languageStatusPropertyUpdateCBInfo.clear();
}


void VehicleServiceClient::initVehicleServiceProxy()
{
   //INFO: Proxy is instantiated on first activation request (lazy instantiation) afterwards registration / deregistration is controlled.
   //       Because proxy life time is valid till the life cycle of process after the first instantiation and its not erasable
   //       till the life cycle of process.
   if (NULL == _vehicleProxy.get())
   {
      if (NULL == _configurator)
      {
         return;
      }
      bool isWiredRunTime    = false;
      ::std::string portName = "";
      _configurator->configureVehicleServicePortName(portName, isWiredRunTime);
      if (!portName.empty())
      {
         if (isWiredRunTime)
         {
            _vehicleProxy = VEHICLE_MAIN_FIProxy::createProxy(portName, 47, 1, 0, (*this), ::asf::cca::CcaProxy::SPM_SERVICE_SUPPLIER);
         }
         else
         {
            _vehicleProxy = VEHICLE_MAIN_FIProxy::createProxy(portName, (*this));
         }
      }
   }
}


void VehicleServiceClient::registerLanguageStatusPropertyUpdate(ILanguageStatusPropertyUpdateCB& imp)
{
   ::std::vector< ILanguageStatusPropertyUpdateCB* >::const_iterator itr = ::std::find(_languageStatusPropertyUpdateCBInfo.begin(), _languageStatusPropertyUpdateCBInfo.end(), (&imp));
   if (itr == _languageStatusPropertyUpdateCBInfo.end())
   {
      _languageStatusPropertyUpdateCBInfo.push_back((&imp));
      uint32 tCurrentLanguageId = 0;
      if ((NULL != (&imp)) && (fetchLanguageStatusInfo(tCurrentLanguageId)))
      {
         imp.onLanguageStatusUpdate(tCurrentLanguageId);
      }
   }
}


void VehicleServiceClient::deregisterLanguageStatusPropertyUpdate(ILanguageStatusPropertyUpdateCB& imp)
{
   ::std::vector< ILanguageStatusPropertyUpdateCB* >::iterator itr = ::std::find(_languageStatusPropertyUpdateCBInfo.begin(), _languageStatusPropertyUpdateCBInfo.end(), (&imp));
   if (itr != _languageStatusPropertyUpdateCBInfo.end())
   {
      (void)_languageStatusPropertyUpdateCBInfo.erase(itr);
   }
}


void VehicleServiceClient::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ::hmibase::ServiceAvailableIF::onAvailable(proxy, stateChange);
   if (isMyVehicleServiceProxy(proxy))
   {
      _languageStatus_RegId = _vehicleProxy->sendLanguageUpReg(*this);
   }
}


void VehicleServiceClient::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ::hmibase::ServiceAvailableIF::onUnavailable(proxy, stateChange);
   if (isMyVehicleServiceProxy(proxy))
   {
      _vehicleProxy->sendLanguageRelUpReg(_languageStatus_RegId);
      _languageStatus_RegId = 0;
   }
}


void VehicleServiceClient::onLanguageError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< LanguageError >& error)
{
   if ((isMyVehicleServiceProxy(proxy)) && (NULL != error.get()) && (error->getAct() == _languageStatus_RegId))
   {
      ETG_TRACE_ERR(("VehicleServiceClient: ERROR: APP: %25s : PID: %d : onLanguageError", getAppName().c_str(), getpid()));
   }
}


void VehicleServiceClient::onLanguageStatus(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< LanguageStatus >& status)
{
   if ((isMyVehicleServiceProxy(proxy)) && (NULL != status.get()) && (status->getAct() == _languageStatus_RegId))
   {
      uint32 tLanguageId     = getLanguageId((*(status.get())));
      ETG_TRACE_USR1(("VehicleServiceClient: INFO: APP: %25s : PID: %d : onLanguageStatus: LanguageId: %d",  getAppName().c_str(), getpid(), tLanguageId));
      for (::std::vector< ILanguageStatusPropertyUpdateCB* >::const_iterator itr = _languageStatusPropertyUpdateCBInfo.begin(); (itr != _languageStatusPropertyUpdateCBInfo.end()); ++itr)
      {
         if (NULL != (*itr))
         {
            (*itr)->onLanguageStatusUpdate(tLanguageId);
         }
      }
   }
}


bool VehicleServiceClient::fetchLanguageStatusInfo(uint32& languageId) const
{
   bool isFetched = false;
   if ((NULL != _vehicleProxy.get()) && (_vehicleProxy->hasLanguage()))
   {
      isFetched  = true;
      languageId = getLanguageId(_vehicleProxy->getLanguage());
   }
   return isFetched;
}


uint32 VehicleServiceClient::getLanguageId(const LanguageStatus& info) const
{
   uint32 tLanguageId = 0;
   for (::std::vector< T_Language_SourceTable >::const_iterator itr = info.getLangTable().begin(); (itr != info.getLangTable().end()); ++itr)
   {
      ETG_TRACE_USR4(("VehicleServiceClient: INFO: APP: %25s : PID: %d : getLanguageId: SourceId: %d : HasChange: %d : EnLanguage: %d :  LanguageId: %d", \
                      getAppName().c_str(), getpid(), (*itr).getEnLangSrcId(), (*itr).hasBHasChanged(), (*itr).getEnLanguage(), tLanguageId));
      if ((*itr).hasBHasChanged())
      {
         tLanguageId = (((*itr).getEnLangSrcId() == T_e8_Language_SourceId__Diag) && ((*itr).getEnLanguage() > 0)) ? (*itr).getEnLanguage() : info.getLanguage();
         break;
      }
   }
   return tLanguageId;
}


} // namespace languageHandler
