/**
 * @file         : HmiInfoServiceClient.cpp
 * @author       :
 * @addtogroup   : AppHmi_App_PRJ_Common
 * @brief        : HmiInfoServiceClient is to handle the HmiInfo service
 *                 client implementation
 * @copyright    : (C) 2016 Robert Bosch GmbH
 *                 (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#include "HmiInfoServiceClient.h"
#include "unistd.h"
#include "CommonTrace.h"
#include "hmi_trace_if.h"
#include <Trace/TraceUtils.h>


using namespace ::hmibase::trace;
using namespace ::bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_COMMON
#include "trcGenProj/Header/HmiInfoServiceClient.cpp.trc.h"
#endif /* VARIANT_S_FTR_ENABLE_TRC_GEN */


namespace languageHandler {


HmiInfoServiceClient::HmiInfoServiceClient()
{
   ETG_TRACE_USR1(("HmiInfoServiceClient: INFO: APP: %25s : PID: %d : CTOR",  getAppName().c_str(), getpid()));
   clear();
}


HmiInfoServiceClient::~HmiInfoServiceClient()
{
   ETG_TRACE_USR1(("HmiInfoServiceClient: INFO: APP: %25s : PID: %d : DTOR",  getAppName().c_str(), getpid()));
   clear();
}


HmiInfoServiceClient::HmiInfoServiceClient(const HmiInfoServiceClient& obj)
{
   ETG_TRACE_USR1(("HmiInfoServiceClient: INFO: APP: %25s : PID: %d : CTOR",  getAppName().c_str(), getpid()));
   clear();
   *this = obj;
}


HmiInfoServiceClient& HmiInfoServiceClient::operator= (const HmiInfoServiceClient& obj)
{
   if (this != &obj)
   {
      clear();
   }
   return *this;
}


void HmiInfoServiceClient::clear()
{
   _languageStatusInfoGet_RegId    = 0;
   _languageStatusInfoUpdate_RegId = 0;
   _configurator                   = NULL;
   _languageStatusPropertyUpdateCBInfo.clear();
}


void HmiInfoServiceClient::initHmiInfoServiceProxy()
{
   //INFO: Proxy is instantiated on first activation request (lazy instantiation) afterwards registration / deregistration is controlled.
   //       Because proxy life time is valid till the life cycle of process after the first instantiation and its not erasable
   //       till the life cycle of process.
   if (NULL == _hmiInfoProxy.get())
   {
      if (NULL == _configurator)
      {
         return;
      }
      bool isWiredRunTime    = false;
      ::std::string portName = "";
      _configurator->configureHmiInfoServicePortName(portName, isWiredRunTime);
      if (!portName.empty())
      {
         if (isWiredRunTime)
         {
            ::std::string busName       = "App.Core.AppHmi_Master";
            ::std::string objectPath    = "/bosch/cm/ai/hmi/hmiinfoservice/hmiinfoservice";
            ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector connectorOptions;
            connectorOptions.setBusType(::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector_BusType__Session);
            _hmiInfoProxy = HmiInfoServiceProxy::createProxy(portName, busName, objectPath, connectorOptions, (*this));
         }
         else
         {
            _hmiInfoProxy = HmiInfoServiceProxy::createProxy(portName, (*this));
         }
      }
   }
}


void HmiInfoServiceClient::registerLanguageStatusPropertyUpdate(ILanguageStatusPropertyUpdateCB& imp)
{
   ::std::vector< ILanguageStatusPropertyUpdateCB* >::const_iterator itr = ::std::find(_languageStatusPropertyUpdateCBInfo.begin(), _languageStatusPropertyUpdateCBInfo.end(), (&imp));
   if (itr == _languageStatusPropertyUpdateCBInfo.end())
   {
      _languageStatusPropertyUpdateCBInfo.push_back((&imp));
      uint32 tCurrentLanguageId = 0;
      if ((NULL != (&imp)) && (fetchLanguageStatusInfo(tCurrentLanguageId)))
      {
         imp.onLanguageStatusUpdate(tCurrentLanguageId);
      }
   }
}


void HmiInfoServiceClient::deregisterLanguageStatusPropertyUpdate(ILanguageStatusPropertyUpdateCB& imp)
{
   ::std::vector< ILanguageStatusPropertyUpdateCB* >::iterator itr = ::std::find(_languageStatusPropertyUpdateCBInfo.begin(), _languageStatusPropertyUpdateCBInfo.end(), (&imp));
   if (itr != _languageStatusPropertyUpdateCBInfo.end())
   {
      (void)_languageStatusPropertyUpdateCBInfo.erase(itr);
   }
}


void HmiInfoServiceClient::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ::hmibase::ServiceAvailableIF::onAvailable(proxy, stateChange);
   if (isMyHmiInfoServiceProxy(proxy))
   {
      _languageStatusInfoUpdate_RegId = _hmiInfoProxy->sendLanguageStatusInfoRegister(*this);
      _languageStatusInfoGet_RegId    = _hmiInfoProxy->sendLanguageStatusInfoGet(*this);
   }
}


void HmiInfoServiceClient::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ::hmibase::ServiceAvailableIF::onUnavailable(proxy, stateChange);
   if (isMyHmiInfoServiceProxy(proxy))
   {
      _hmiInfoProxy->sendLanguageStatusInfoDeregister(_languageStatusInfoUpdate_RegId);
      _languageStatusInfoGet_RegId    = 0;
      _languageStatusInfoUpdate_RegId = 0;
   }
}


void HmiInfoServiceClient::onLanguageStatusInfoError(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr< LanguageStatusInfoError >& error)
{
   if ((isMyHmiInfoServiceProxy(proxy)) && (NULL != error.get()) && ((error->getAct() == _languageStatusInfoGet_RegId) || (error->getAct() == _languageStatusInfoUpdate_RegId)))
   {
      ETG_TRACE_ERR(("HmiInfoServiceClient: ERROR: APP: %25s : PID: %d : onLanguageStatusInfoError: Msg: %s", getAppName().c_str(), getpid(), error->getMessage().c_str()));
   }
}


void HmiInfoServiceClient::onLanguageStatusInfoUpdate(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr< LanguageStatusInfoUpdate >& update)
{
   if ((isMyHmiInfoServiceProxy(proxy)) && (NULL != update.get()) && ((update->getAct() == _languageStatusInfoGet_RegId) || (update->getAct() == _languageStatusInfoUpdate_RegId)))
   {
      uint32 tLanguageId = update->getLanguageStatusInfo();
      ETG_TRACE_USR1(("HmiInfoServiceClient: INFO: APP: %25s : PID: %d : onLanguageStatusInfoUpdate: LanguageId: %d",  getAppName().c_str(), getpid(), tLanguageId));
      for (::std::vector< ILanguageStatusPropertyUpdateCB* >::const_iterator itr = _languageStatusPropertyUpdateCBInfo.begin(); (itr != _languageStatusPropertyUpdateCBInfo.end()); ++itr)
      {
         if (NULL != (*itr))
         {
            (*itr)->onLanguageStatusUpdate(tLanguageId);
         }
      }
   }
}


bool HmiInfoServiceClient::fetchLanguageStatusInfo(uint32& languageId) const
{
   bool isFetched = false;
   if ((NULL != _hmiInfoProxy.get()) && (_hmiInfoProxy->hasLanguageStatusInfo()))
   {
      isFetched  = true;
      languageId = _hmiInfoProxy->getLanguageStatusInfo();
   }
   return isFetched;
}


} // namespace languageHandler
