/* ***************************************************************************************
* FILE:          DefaultFocusConsistencyChecker.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  DefaultFocusConsistencyChecker.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

#include "ProjectBaseMsgs.h"
#include "DefaultFocusConsistencyChecker.h"

#include <Focus/Default/FDefaultCrtAppExchange.h>
#include <Focus/Default/FDefaultAvgBuilder.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_FOCUS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/DefaultFocusConsistencyChecker.cpp.trc.h"
#endif

COURIER_LOG_SET_REALM(Candera::Diagnostics::LogRealm::User);


///////////////////////////////////////////////////////////////////////////////
DefaultFocusConsistencyChecker::DefaultFocusConsistencyChecker(Focus::FManager& manager) : _manager(manager), _timer(NULL)
{
   static bool registered = false;
   if (!registered)
   {
      registered = true;
      //this message receiver receives the messages through the focus manager which belongs to the view component
      Courier::ActivationResMsg::Subscribe(Courier::ComponentType::View);                 //lint !e641
      Courier::ViewResMsg::Subscribe(Courier::ComponentType::View);                       //lint !e641
   }

   _timer = FOCUS_NEW(Util::Timer);
}


///////////////////////////////////////////////////////////////////////////////
DefaultFocusConsistencyChecker::~DefaultFocusConsistencyChecker()
{
   if (_timer != NULL)
   {
      _timer->stop();
      FOCUS_DELETE(_timer);
      _timer = NULL;
   }
}


///////////////////////////////////////////////////////////////////////////////
void DefaultFocusConsistencyChecker::restartTimer()
{
   if (_timer != NULL)
   {
      _timer->setTimeout(0, 100);
      _timer->restart();
   }
}


///////////////////////////////////////////////////////////////////////////////
void DefaultFocusConsistencyChecker::stopTimer()
{
   if (_timer != NULL)
   {
      _timer->stop();
   }
}


///////////////////////////////////////////////////////////////////////////////
bool DefaultFocusConsistencyChecker::onMessage(const Focus::FMessage& msg)
{
   switch (msg.GetId())
   {
      case Courier::ActivationResMsg::ID:
      case Courier::ViewResMsg::ID:
         //restart the timer everytime a view is hidden or shown
         if ((_manager.getCurrentFocus().ViewId != Focus::Constants::InvalidViewId) || _manager.isFocusVisible())
         {
            restartTimer();
         }
         break;

      case TimerExpiredMsg::ID:
      {
         const TimerExpiredMsg* timerMsg = Courier::message_cast<const TimerExpiredMsg*>(&msg);
         if ((timerMsg != NULL) && (timerMsg->GetTimer() != NULL) && (timerMsg->GetTimer() == _timer))
         {
            checkFocus();
         }
         break;
      }

      default:
         break;
   }
   return false;
}


bool DefaultFocusConsistencyChecker::checkFocus()
{
   Focus::FTaskManager* taskManager = _manager.getTaskManager();
   Focus::FSessionManager* sessionManager = _manager.getSessionManager();
   Focus::FAvgManager* avgManager = _manager.getAvgManager();

   if ((taskManager != NULL) && (sessionManager != NULL) && (sessionManager->getSession() == NULL)
         && ((_manager.getCurrentFocus().ViewId != Focus::Constants::InvalidViewId) || _manager.isFocusVisible()))
   {
      Focus::FAppCurrentFocusInfo currentFocus = _manager.getCurrentFocus();
      ETG_TRACE_USR1_CLS((TR_CLASS_HMI_FW_FOCUS, "DefaultFocusConsistencyChecker::checkConsistency widget=%s", FID_STR(currentFocus.WidgetId)));

      //begin a new session
      if ((sessionManager->beginSession(Focus::FSession::Master)) && (sessionManager->getSession() != NULL))
      {
         //add tasks to collect data from the visible views, merge the state info and to build the active view group
         taskManager->clearTasks();
         taskManager->addTask(FOCUS_NEW(Focus::FDefaultCourierInfoCollector)(_manager));
         taskManager->addTask(FOCUS_NEW(Focus::FDefaultCrtAppState2SessionUpdater)(_manager));
         taskManager->addTask(FOCUS_NEW(Focus::FDefaultAvgBuilder)(_manager));
         if (taskManager->executeTasks() == Focus::FTask::Completed)
         {
            _manager.printSessionDebugInfo();

            //check if the current focus in still available
            bool isCurrentFocusAvailable = avgManager->isCurrentFocusAvailable(*sessionManager->getSession());
            ETG_TRACE_USR1_CLS((TR_CLASS_HMI_FW_FOCUS, "DefaultFocusConsistencyChecker::checkConsistency isCurrentFocusAvailable=%u", isCurrentFocusAvailable));

            if (isCurrentFocusAvailable)
            {
               taskManager->addTask(FOCUS_NEW(Focus::FAvg2FocusInfoUpdater)(_manager));
               taskManager->addTask(FOCUS_NEW(Focus::FDefaultSession2CrtAppStateUpdater)(_manager));
               taskManager->addTask(FOCUS_NEW(Focus::FDefaultCourierInfoPublisher)(_manager));
               taskManager->executeTasks();
            }
            else
            {
               _manager.hideFocus();
            }
         }
      }
      sessionManager->endSession();
   }
   return true;
}
