/* ***************************************************************************************
* FILE:          DefaultFocus.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  DefaultFocus.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

#include "ProjectBaseConstants.h"

#include "DefaultFocus.h"
#include "DefaultFocusInput.h"
#include "DefaultFocusReqHandler.h"
#include "DefaultListRotaryController.h"
#include "DefaultFocusConsistencyChecker.h"
#include "FocusOnTouchHandler.h"

#include <Focus/Default/FDefaultAnimationManager.h>
#include <Focus/Default/FDefaultIpcManager.h>
#include <Focus/Default/FDefaultEnterKeyController.h>
#include <Focus/Default/FDefaultJoystickController.h>
#include <Focus/Default/FDefaultRotaryController.h>
#include <Focus/Default/FDefaultVisibilityManager.h>
#include <Focus/Default/FDefaultReqController.h>

#include "Widgets/2D/List/ListWidget2D.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_FOCUS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/DefaultFocus.cpp.trc.h"
#endif

COURIER_LOG_SET_REALM(Candera::Diagnostics::LogRealm::User);

///////////////////////////////////////////////////////////////////////////////
class DefaultActivityTimerListener : public Focus::FTimerListener
{
   public:
      DefaultActivityTimerListener(Focus::FManager& manager) : _manager(manager) {}

      virtual void onRestarted() {}
      virtual void onStopped() {}
      virtual void onExpired()
      {
         _manager.hideFocus();
      }

   private:
      DefaultActivityTimerListener(const DefaultActivityTimerListener&);
      DefaultActivityTimerListener& operator=(const DefaultActivityTimerListener&);

      Focus::FManager& _manager;
};


///////////////////////////////////////////////////////////////////////////////
class ScreenCoordinateController : public Focus::FController
{
public:
   virtual bool configureWidget(Focus::FSession&, Focus::FWidgetConfig& handle, Focus::FFrameworkWidget& widget)
   {
      BaseWidget2D* baseWidget = dynamic_cast<BaseWidget2D*>(&widget);
      if ((baseWidget != NULL)
         && (baseWidget->GetNode() != NULL)
         && (baseWidget->GetParentView() != NULL)
         && (baseWidget->GetParentView()->Is2D()))
      {
         Focus::FRectangle* rectangle = handle.getOrCreateData<Focus::FRectangle>();
         Candera::Node2D* node = baseWidget->GetNode();
         Courier::ViewScene2D* viewScene = baseWidget->GetParentView()->ToViewScene2D();
         if ((rectangle != NULL) && (viewScene != NULL) && (!viewScene->GetCameraPtrVector().Empty()))
         {
            Candera::Vector2 position(node->GetWorldPosition());

            Candera::Camera2D* camera = viewScene->GetCameraPtrVector()[0];
            if ((camera != NULL) && (camera->GetRenderTarget() != NULL))
            {
               position = Candera::Math2D::TransformSceneToViewport(*camera, position);
               position = Candera::Math2D::TransformViewportToRenderTarget(*camera, position);
               position = Candera::Math2D::TransformRenderTargetToScreen(*camera->GetRenderTarget(), position);
            }
            node->GetEffectiveBoundingRectangle(*rectangle);
            rectangle->SetLeft(position.GetX());
            rectangle->SetTop(position.GetY());
         }
      }
      return true;
   }
};


///////////////////////////////////////////////////////////////////////////////
DefaultFocusManagerConfigurator::DefaultFocusManagerConfigurator(Focus::FManager& manager, AppViewHandler& viewHandler, const Focus::FAppId& currentAppId) :
   Base(manager, &viewHandler),
   _activityTimerListener(NULL),
   _ipcAdapter(NULL),
   _hkConverter(NULL),
   _onTouchHandler(NULL)
{
   setCurrentAppId(currentAppId);
}


///////////////////////////////////////////////////////////////////////////////
DefaultFocusManagerConfigurator::~DefaultFocusManagerConfigurator()
{
   _activityTimerListener = NULL;
   _ipcAdapter = NULL;
   _hkConverter = NULL;
   _onTouchHandler = NULL;
}


///////////////////////////////////////////////////////////////////////////////
bool DefaultFocusManagerConfigurator::initialize()
{
   bool result = Base::initialize();

   if (_hkConverter == NULL)
   {
      _hkConverter = FOCUS_NEW(DefaultHKtoEnterKeyConverter)(_manager);
      if (_hkConverter != NULL)
      {
         _manager.addMsgReceiver(*_hkConverter);
      }
   }

   if (_onTouchHandler == NULL)
   {
      _onTouchHandler = FOCUS_NEW(FocusOnTouchHandler)(_manager);
      if (_onTouchHandler != NULL)
      {
         _manager.addMsgReceiver(*_onTouchHandler);
      }
   }

   return result;
}


///////////////////////////////////////////////////////////////////////////////
bool DefaultFocusManagerConfigurator::finalize()
{
   if (_onTouchHandler != NULL)
   {
      _manager.removeMsgReceiver(*_onTouchHandler);
      FOCUS_DELETE(_onTouchHandler);
      _onTouchHandler = NULL;
   }

   if (_hkConverter != NULL)
   {
      _manager.removeMsgReceiver(*_hkConverter);
      FOCUS_DELETE(_hkConverter);
      _hkConverter = NULL;
   }

   bool result = Base::finalize();
   return result;
}


bool DefaultFocusManagerConfigurator::initializeConsistencyChecker()
{
   //return Base::initializeConsistencyChecker();
   setConsistencyChecker(FOCUS_NEW(DefaultFocusConsistencyChecker)(_manager));
   return true;
}


bool DefaultFocusManagerConfigurator::initializeVisibilityManager()
{
   //return Base::initializeVisibilityManager();
   setVisibilityManager(FOCUS_NEW(Focus::FViewSpecificVisibilityManager)(_manager));
   return true;
}

bool DefaultFocusManagerConfigurator::initializeAnimationManager()
{
   if (getViewHandler() == NULL)
   {
      return false;
   }

   setAnimationManager(FOCUS_NEW(Focus::FDefaultAnimationManager)(_manager, *getViewHandler()));
   return true;
}

///////////////////////////////////////////////////////////////////////////////
bool DefaultFocusManagerConfigurator::initializeTimers()
{
   bool result = Base::initializeTimers();

   if ((_manager.getActivityTimer() != NULL) && (_activityTimerListener == NULL))
   {
      _manager.getActivityTimer()->setTimeout(10000);

      //_activityTimerListener = FOCUS_NEW(DefaultActivityTimerListener)(_manager);
      //if (_activityTimerListener != NULL)
      //{
      //   _manager.getActivityTimer()->addListener(*_activityTimerListener);
      //}
   }
   return result;
}


///////////////////////////////////////////////////////////////////////////////
bool DefaultFocusManagerConfigurator::finalizeTimers()
{
   if ((_manager.getActivityTimer() != NULL) && (_activityTimerListener != NULL))
   {
      _manager.getActivityTimer()->removeListener(*_activityTimerListener);
      FOCUS_DELETE(_activityTimerListener);
      _activityTimerListener = NULL;
   }

   bool result = Base::finalizeTimers();
   return result;
}


//todo: these should come from the contract
#if !defined FOCUS_CONTROLLER_SET_ID_DEFAULT_ROOT_GROUP
#define FOCUS_CONTROLLER_SET_ID_DEFAULT_ROOT_GROUP 10
#endif
#if !defined FOCUS_CONTROLLER_SET_ID_DEFAULT_VERTICAL_LIST
#define FOCUS_CONTROLLER_SET_ID_DEFAULT_VERTICAL_LIST 11
#endif
#if !defined FOCUS_CONTROLLER_SET_ID_DEFAULT_SPELLER
#define FOCUS_CONTROLLER_SET_ID_DEFAULT_SPELLER 12
#endif
#if !defined FOCUS_CONTROLLER_SET_ID_DEFAULT_GROUP
#define FOCUS_CONTROLLER_SET_ID_DEFAULT_GROUP 13
#endif
#if !defined FOCUS_CONTROLLER_SET_ID_SCREEN_COORDINATE
#define FOCUS_CONTROLLER_SET_ID_SCREEN_COORDINATE 1000
#endif

///////////////////////////////////////////////////////////////////////////////
bool DefaultFocusManagerConfigurator::initializeControllers()
{
   //don't inherit controllers from the default implementation
   //Base::initializeControllers();

   registerController(FOCUS_CONTROLLER_SET_ID_DEFAULT_ROOT_GROUP, FOCUS_NEW(Focus::FDefaultEnterKeyController)(_manager));
   registerController(FOCUS_CONTROLLER_SET_ID_DEFAULT_ROOT_GROUP, FOCUS_NEW(Focus::FDefaultRotaryController)(_manager));
   registerController(FOCUS_CONTROLLER_SET_ID_DEFAULT_ROOT_GROUP, FOCUS_NEW(Focus::FDefaultJoystickController)(_manager));
   registerController(FOCUS_CONTROLLER_SET_ID_DEFAULT_ROOT_GROUP, FOCUS_NEW(Focus::FDefaultReqController)(_manager));
   //registerController(FOCUS_CONTROLLER_SET_ID_DEFAULT_ROOT_GROUP, FOCUS_NEW(DefaultFocusReqHandler)(_manager));
   setRootGroupControllerSetId(FOCUS_CONTROLLER_SET_ID_DEFAULT_ROOT_GROUP);

   registerController(FOCUS_CONTROLLER_SET_ID_DEFAULT_VERTICAL_LIST, FOCUS_NEW(DefaultListRotaryController));
   ListWidget2D::SetDefaultFocusControllers(FOCUS_CONTROLLER_SET_ID_DEFAULT_VERTICAL_LIST);

   registerController(FOCUS_CONTROLLER_SET_ID_SCREEN_COORDINATE, FOCUS_NEW(ScreenCoordinateController));

   return true;
}


///////////////////////////////////////////////////////////////////////////////
bool DefaultFocusManagerConfigurator::finalizeControllers()
{
   //don't inherit controllers from the default implementation
   //Base::finalizeControllers();

   clearControllerSet(FOCUS_CONTROLLER_SET_ID_DEFAULT_ROOT_GROUP, true);
   clearControllerSet(FOCUS_CONTROLLER_SET_ID_DEFAULT_VERTICAL_LIST, true);
   clearControllerSet(FOCUS_CONTROLLER_SET_ID_SCREEN_COORDINATE, true);

   return true;
}


///////////////////////////////////////////////////////////////////////////////
bool DefaultFocusManagerConfigurator::initializeInputMsgChecker()
{
   setInputMsgChecker(FOCUS_NEW(DefaultInputMsgChecker));
   return true;
}


///////////////////////////////////////////////////////////////////////////////
bool DefaultFocusManagerConfigurator::finalizeInputMsgChecker()
{
   if (_manager.getInputMsgChecker() != NULL)
   {
      FOCUS_DELETE(_manager.getInputMsgChecker());
      setInputMsgChecker(NULL);
   }
   return true;
}


///////////////////////////////////////////////////////////////////////////////
bool DefaultFocusManagerConfigurator::initializeIpcManager()
{
   //if (_ipcAdapter == NULL)
   //{
   //   //create ipc adapter
   //   _ipcAdapter = FOCUS_NEW(DefaultIpcAdapter);
   //   if (_ipcAdapter != NULL)
   //   {
   //      //create ipc manager
   //      Focus::FDefaultIpcManager* ipcManager = FOCUS_NEW(Focus::FDefaultIpcManager)(_manager, *_ipcAdapter);
   //      if (ipcManager != NULL)
   //      {
   //         setIpcManager(ipcManager);
   //      }
   //   }
   //}
   return true;
}


///////////////////////////////////////////////////////////////////////////////
bool DefaultFocusManagerConfigurator::finalizeIpcManager()
{
   //if (_ipcAdapter != NULL)
   //{
   //   if (_manager.getIpcManager() != NULL)
   //   {
   //      FOCUS_DELETE(_manager.getIpcManager);
   //      setIpcManager(NULL);
   //   }

   //   //delete ipc adapter
   //   FOCUS_DELETE(_ipcAdapter);
   //   _ipcAdapter = NULL;
   //}
   return true;
}


Focus::FDefaultManagerConfigurator* DefaultFocus::createConfigurator(AppViewHandler& viewHandler, const Focus::FAppId& currentAppId)
{
   return FOCUS_NEW(DefaultFocusManagerConfigurator)(Focus::FManager::getInstance(), viewHandler, currentAppId);
}
