/* ***************************************************************************************
* FILE:          ListRegistry.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ListRegistry.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"
#include "ListRegistry.h"


/**************************************************************************//**
* Destructor
******************************************************************************/
ListRegistry::~ListRegistry()
{
}


/**************************************************************************//**
* Constructor
******************************************************************************/
ListRegistry::ListRegistry()
{
}


/**************************************************************************//**
*
******************************************************************************/
void ListRegistry::addListImplementation(uint32_t listid, ListImplementation* imp)
{
   if (_theInstance == 0)
   {
      s_initialize();
   }
   _listCallbackMap.insert(std::pair< uint32_t, ListImplementation*>(listid, imp));
}


void ListRegistry::removeListImplementation(uint32_t listid)
{
   std::map <uint32_t, ListImplementation*>::iterator iter = _listCallbackMap.find(listid);
   if (iter != _listCallbackMap.end())
   {
      _listCallbackMap.erase(iter);
   }
   if (_listCallbackMap.empty())
   {
      s_destroy();
   }
}


/**************************************************************************//**
*
******************************************************************************/
bool ListRegistry::updateList(const ListDateProviderReqMsg& oMsg)
{
   std::map <uint32_t, ListImplementation*>::iterator iter = _listCallbackMap.find(oMsg.GetListId());
   if (iter != _listCallbackMap.end())
   {
      tSharedPtrDataProvider dataProvider = (iter->second)->getListDataProvider(oMsg);
      if (!dataProvider.PointsToNull())
      {
         COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)->Post();
         return true;
      }
   }
   return false;
}


/**************************************************************************//**
*
******************************************************************************/
ListRegistry* ListRegistry::_theInstance = 0;

bool ListRegistry::s_initialize()
{
   if (_theInstance == 0)
   {
      _theInstance = new ListRegistry();
      return true;
   }
   HMI_APP_ASSERT_ALWAYS();
   return false;
}


void ListRegistry::s_destroy()
{
   if (_theInstance)
   {
      delete _theInstance;
      _theInstance = 0;
   }
}


ListRegistry& ListRegistry::s_getInstance()
{
   if (_theInstance == 0)
   {
      s_initialize();
   }
   HMI_APP_ASSERT(_theInstance != 0);
   return *_theInstance;
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

/**************************************************************************//**
* Constructor
******************************************************************************/
ListImplementation::ListImplementation()
   : _expandableListViewModels()
{
}


/**************************************************************************//**
* Destructor
******************************************************************************/
ListImplementation::~ListImplementation()
{
   for (std::map<unsigned int, ExpandableListViewModel*>::iterator it = _expandableListViewModels.begin(); it != _expandableListViewModels.end(); ++it)
   {
      delete it->second;
   }
   _expandableListViewModels.clear();
}


/**************************************************************************//**
*
******************************************************************************/
ExpandableListViewModel* ListImplementation::AddNewExpandableListViewModel(unsigned int expandableListId, unsigned int rawListId)
{
   //todo: add error handling in case an ExpandableListViewModel is already registered for the specified expandableListId
   ExpandableListViewModel* expandableListViewModel = CANDERA_NEW(ExpandableListViewModel)(expandableListId, rawListId);
   _expandableListViewModels[expandableListId] = expandableListViewModel;
   return expandableListViewModel;
}


/**************************************************************************//**
*
******************************************************************************/
bool ListImplementation::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   if (delegateCourierMessageToExpandableListViewModels(oMsg))
   {
      return true;
   }

   //Based on the Sender parameter of the ButtonReactionMsg we can determine if the button is part of list item inside a FlexListWidget2D.
   //For those buttons we can obtain ListId, HdlRow, HdlCol and Flags which were associated to that ListItem.
   //See ListDataProviderBuilder
   ListProviderEventInfo info;
   if (ListProviderEventInfo::GetItemIdentifierInfo(oMsg.GetSender(), info))
   {
      unsigned int listId      = info.getListId();     // the list id for generic access
      unsigned int hdlRow      = info.getHdlRow();     // normaly the index
      unsigned int hdlCol      = info.getHdlCol();     // if more than 1 active element in one list row, e.g. Button in a button line

      //For each ButtonReactionMsg for buttons which are part of list items we post a ButtonListItemMsg with detailed info about the list item.
      ButtonListItemUpdMsg* msg = COURIER_MESSAGE_NEW(ButtonListItemUpdMsg)(listId, hdlRow, hdlCol, oMsg.GetEnReaction());
      if (msg != NULL)
      {
         msg->Post();
      }

      //We provide automatic toggle for comboboxes inside lists and also automatic closing of standalone pushups and comboboxes.
      if ((listId != 0u) && (oMsg.GetEnReaction() == enRelease))
      {
         unsigned int ddListId;
         Candera::DropdownChangeType ddChange;
         //clicking expandable items should toggle their associated sublists which have the id stored inside hdlCol
         if ((info.getFlags() & ListDataProviderBuilder::ListItem::Expandable) == ListDataProviderBuilder::ListItem::Expandable)
         {
            ddListId = static_cast<unsigned int>(hdlCol);
            ddChange = Candera::DropdownChangeToggle;
         }
         //clicking non expandable items should close their parent list (assuming that it is a dropdown list)
         else
         {
            ddListId = listId;
            ddChange = Candera::DropdownChangeClose;
         }

         DropdownChangeMsg* ddToggleMsg = COURIER_MESSAGE_NEW(DropdownChangeMsg)(ddChange, ddListId);
         if (ddToggleMsg != NULL)
         {
            ddToggleMsg->Post();
         }
      }
      return true;
   }

   return false;
}


/**************************************************************************//**
* This message is sent by the list widget when it requires new data.
* This happens when the list is displayed or scrolled.
******************************************************************************/
bool ListImplementation::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   if (delegateCourierMessageToExpandableListViewModels(oMsg))
   {
      return true;
   }

   return ListRegistry::s_getInstance().updateList(oMsg);
}


bool ListImplementation::onCourierMessage(const ListDateProviderResMsg& oMsg)
{
   if (delegateCourierMessageToExpandableListViewModels(oMsg))
   {
      return true;
   }

   return false;
}


/**************************************************************************//**
*
******************************************************************************/
bool ListImplementation::delegateCourierMessageToExpandableListViewModels(const Courier::Message& oMsg)
{
   for (std::map<unsigned int, ExpandableListViewModel*>::iterator it = _expandableListViewModels.begin(); it != _expandableListViewModels.end(); ++it)
   {
      if (it->second->OnMessage(oMsg))
      {
         return true;
      }
   }

   return false;
}
