/**************************************************************************************
* @file         : TunerCommonHandler.cpp
* @author       : ECV2-Rohini SR
* @addtogroup   : Tuner HALL
* @brief        : Class contains necessary business logic required to TunerCommonHandler List data.
* @copyright    : (c) 2017-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/
#include "hall_std_if.h"
#include "App/Core/Utility/TunerCommonHandler.h"

#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_hmi_02_if.h"
#include "dp_generic_if.h"
#include "dp_tuner_if.h"
#endif


#include "App/datapool/TunerDataPoolConfig.h" //wst4kor
#include "App/Core/Tuner_Defines.h"
#include "App/Core/HmiTranslation_TextIds.h"
#include "App/Core/LanguageDefines.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_TUNER_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_TUNER
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_TUNER_"
#define ETG_I_FILE_PREFIX                 App::Core::TunerCommonHandler::
#include "trcGenProj/Header/TunerCommonHandler.cpp.trc.h"
#endif

namespace App {
namespace Core {

static const Category _ProgramType[] =
{
#define CATEGORY_MAP(ptyTextId) {ptyTextId},
#include "TunerCategory.dat"
#undef CATEGORY_MAP
};


/**
 * Description     : Destructor of TunerCommonHandler
 */
TunerCommonHandler::~TunerCommonHandler()
{
}


/**
 * Description     : Constructor of TunerCommonHandler
 */
TunerCommonHandler::TunerCommonHandler()
   : _isHDMounted(false)
   , _isFMHDSwitchEnabled(false)
   , _isAMHDSwitchEnabled(false)
{
   initaliseHDKDSValue();
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1_R
   setAmAvailability();
#endif
}


void TunerCommonHandler::initaliseHDKDSValue()
{
   getHDHWConfigurationData();
   getFMHDConfigurationSwitchData();
   getAMHDConfigurationSwitchData();
}


void TunerCommonHandler::getHDHWConfigurationData()
{
   dp_tclKdsCMVariantCoding KDSHDHWMounted;
   tU8 isHDMounted = 0x0F;
   KDSHDHWMounted.u8GetHDRadio(isHDMounted);
   _isHDMounted = isHDMounted;
}


void TunerCommonHandler::getFMHDConfigurationSwitchData()
{
   dp_tclKdsAMFMTunerParameter KDSHDSwitchEnabled;
   tU8 isFMHDSwitchEnabled = 0x0F;
   KDSHDSwitchEnabled.u8GetDemodAudioTuner1FMHD(isFMHDSwitchEnabled);
   _isFMHDSwitchEnabled = isFMHDSwitchEnabled;
}


void TunerCommonHandler::getAMHDConfigurationSwitchData()
{
   dp_tclKdsAMFMTunerParameter KDSHDSwitchEnabled;
   tU8 isAMHDSwitchEnabled = 0x0F;
   KDSHDSwitchEnabled.u8GetDemodAudioTuner1AMHD(isAMHDSwitchEnabled);
   _isAMHDSwitchEnabled = isAMHDSwitchEnabled;
}


void TunerCommonHandler::updateAudioPopupData(std::string& scanText, std::string psName, std::string& freqService, uint8 Band, std::string HDprogramNumber)
{
   /*Check if there is any change in the content before updating UI */
   /* (*_audioPopupInfo).mBandIconSwitch = Band;
    std::string secureString = TunerUtility::getSecureString((*_audioPopupInfo).mScanText);
    if (strcmp(secureString.c_str(), scanText.c_str()) != 0)
    {
       (*_audioPopupInfo).mScanText = scanText.c_str();
    }
    secureString = TunerUtility::getSecureString((*_audioPopupInfo).mStationName);
    if (strcmp(secureString.c_str(), psName.c_str()) != 0)
    {
       (*_audioPopupInfo).mStationName = psName.c_str();
    }
    secureString = TunerUtility::getSecureString((*_audioPopupInfo).mFrequencyServiceInfo);
    if (strcmp(secureString.c_str(), freqService.c_str()) != 0)
    {
       (*_audioPopupInfo).mFrequencyServiceInfo = freqService.c_str();
    }
    secureString = TunerUtility::getSecureString((*_audioPopupInfo).mHDProgramNumber);
    if(strcmp(secureString.c_str(), HDprogramNumber.c_str()) != 0)
    {
       (*_audioPopupInfo).mHDProgramNumber = HDprogramNumber.c_str();
    }
    _audioPopupInfo.MarkAllItemsModified();
    _audioPopupInfo.SendUpdate();*/
}


bool TunerCommonHandler::isHDFeatureEnabled()
{
   ETG_TRACE_USR4(("TunerCommonHandler::isHDFeatureEnabled():%d %d %d", _isHDMounted, _isFMHDSwitchEnabled, _isAMHDSwitchEnabled));
   return (_isHDMounted && _isFMHDSwitchEnabled && _isAMHDSwitchEnabled);
}


bool TunerCommonHandler::isNARVariant()
{
   return ((TunerDataPoolConfig::getInstance()->readRegionType() == NAM_REGION)
           || (TunerDataPoolConfig::getInstance()->readRegionType() == CAN_REGION)
           || (TunerDataPoolConfig::getInstance()->readRegionType() == MEX_REGION));
}


/**
* Description: Function to get the status of scan baseed on region
* @return     : bool
*/

//bool TunerCommonHandler::isScanFeatureEnabled()
//{
//   //EUR doesnt support scan feature
//   //   return (!(TunerDataPoolConfig::getInstance()->readRegionType() == EUR_REGION));
//}


#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2
/**
* Callback to handle Courier Message TunerFooterListIdReqMsg Request
* @param [in] : TunerFooterListIdReqMsg
* @return     : bool true if message is consumed
*/
bool TunerCommonHandler::onCourierMessage(const TunerFooterListIdReqMsg& oMsg)
{
   _totalListSize = oMsg.GetListSize();
   _focusedIndex = -1;
   _CurrentStartIndex = 0;
   _currentRequestedListID = oMsg.GetListId();
   updateFooterInfo(_CurrentStartIndex);
   return true;
}


/**
 * Callback to handle Courier Message ListChangedUpdMsg Request
 * @param [in] : Reference of Courier::ListChangedUpdMsg
 * @return     :  bool true if message is consumed
 */
bool TunerCommonHandler::onCourierMessage(const ListChangedUpdMsg& oMsg)
{
   _CurrentStartIndex = oMsg.GetFirstVisibleIndex() + 1;
   if (_focusedIndex == -1)
   {
      updateFooterInfo(_CurrentStartIndex);
   }
   return false;
}


/**
* Callback to handle Courier Message FocusChangedUpdMsg Request
* @param [in] : FocusChangedUpdMsg
* @return     : bool true if message is consumed
*/
bool TunerCommonHandler::onCourierMessage(const FocusChangedUpdMsg& oMsg)
{
   ListProviderEventInfo info;
   if (ListProviderEventInfo::GetItemIdentifierInfo(oMsg.GetWidget(), info))
   {
      uint32 listId = info.getListId();
      if (_currentRequestedListID == listId)
      {
         _focusedIndex = static_cast<uint32>(info.getHdlRow() + 1); // normally the index
         updateFooterInfo(_focusedIndex);
         return true;
      }
   }
   else
   {
      _focusedIndex = -1;
      updateFooterInfo(_CurrentStartIndex);
   }

   return false;
}


/**
* Description: Helper Function to update footer info for a list.
* @param [in] : current stating index of the list
* @return     : void
*/
void TunerCommonHandler::updateFooterInfo(int32 index)
{
   DataBindingItem<DTVUpdateFooterDataBindingSource> _dTVUpdateFooter;
   if (index != -1)
   {
      std::ostringstream footerData;
      footerData << index << "/" << _totalListSize;
      (*_dTVUpdateFooter).mFooterText = footerData.str().c_str();
   }
   _dTVUpdateFooter.SendUpdate(true);
}


/**
* Description: Helper Function to get the DTV module mount status
* @param [in] : void
* @return     : uint8
*/
uint8 TunerCommonHandler::isDTVModuleMounted()
{
   dp_tclKdsCMVariantCoding oKdsCMVariantCoding;
   oKdsCMVariantCoding.u8GetDTV(_u8DtvMountStatus);
   return _u8DtvMountStatus;
}


#endif
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1_R
bool TunerCommonHandler::getSimulCastStatus()
{
   return _bIsSimulCastEnabled;
}


void TunerCommonHandler::setSimulCastStatus(bool bSimulCast)
{
   _bIsSimulCastEnabled = bSimulCast;
}


bool TunerCommonHandler::getAmStatus()
{
   dp_tclTmcTunerDPAMAvailability AmCurrentStatus;
   return AmCurrentStatus.tGetData();
}


void TunerCommonHandler::setAmAvailability()
{
   dp_tclKdsHMIConfig AmAvailability;
   tU8 isAmAvailable = 0xFF;
   AmAvailability.u8GetCFGAM2(isAmAvailable);
   _isAmAvailable = static_cast<bool>(isAmAvailable);
}


bool TunerCommonHandler::getAmAvailability()
{
   return _isAmAvailable;
}


#endif
/**
* Description     : Function called to get Category Name
*
* @param[in]      : ptyIndex
* @return         : std::string
*/
std::string TunerCommonHandler::getCategoryName(uint8 ptyIndex)
{
   std::string category = "";
   if (ptyIndex < PTYRDS_RBDS_SIZE)
   {
//#if  defined(VARIANT_S_FTR_ENABLE_AIVI_SCOPE2)
      if (ptyIndex > 0)
      {
         ptyIndex = ptyIndex - 1;
         category = TunerUtility::getSecureString(Candera::String(_ProgramType[ptyIndex].ptyTextId));
      }
      else
      {
         category = TunerUtility::getSecureString(Candera::String(""));
      }
//#endif
   }
   return category;
}


} // namespace Core
} // namespace App
