/**************************************************************************************
* @file         : AMFMTestMode.cpp
* @author       : ECV2-Priyanka B
* @addtogroup   : Tuner HALL
* @brief        : Class contains necessary business logic required to handle AM FM Testmode data.
* @copyright    : (c) 2017-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "AMFMTestMode.h"
#include "App/Core/Utility/TunerUtility.h"
#include "CourierTunnelService/CourierMessageReceiverStub.h"
#include "MIDW_TUNER_FIProxy.h"
#include "midw_tuner_fi_types.h"
#include "App/Core/Tuner_Defines.h"
#include "App/Core/LanguageDefines.h"
#include "AnnoHandler.h"
#include "App/Core/Utility/TunerCommonHandler.h"
#include "List/Data/ListDataProviderDistributor.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_TUNER_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_TUNER
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_Tuner_"
#define ETG_I_FILE_PREFIX                 App::Core::AMFMTestMode::
#include "trcGenProj/Header/AMFMTestMode.cpp.trc.h"
#endif

#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#endif


using namespace ::MIDW_TUNER_FI;
using namespace ::MIDW_TUNERMASTER_FI;
using namespace ::MIDW_EXT_DABTUNER_FI;
using namespace ::midw_tuner_fi_types;
using namespace ::midw_tunermaster_fi_types;
using namespace ::midw_ext_dabtuner_fi_types;

//const uint8 AMFMTestMode::DAB_QUERY_MAX_COUNT = 8;


#define SHARX_MAX 6
#define SHARX_MIN 0
#define HICUT_MIN 0
#define HICUT_MAX 7
#define STEP_SIZE 1

namespace App {
namespace Core {
DEFINE_CLASS_LOGGER_AND_LEVEL("App/Core/Tuner_TestMode", AMFMTestMode, Info);

AMFMTestMode* AMFMTestMode::_AMFMTestModeInst = NULL;

/**
* Description     : Destructor of class AMFMTestMode
*/
AMFMTestMode::~AMFMTestMode()
{
   // ListRegistry::s_getInstance().removeListImplementation(LIST_ID_DTM_HD_MONITOR); //wst4kor
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_DTM_HD_PARAM_TUNING);
   _tunerMasterFiProxy.reset();
   _tunerFiProxy.reset();
   _dabFiProxy.reset();
   _DabExpertAPIInfoList.clear();
   ETG_I_UNREGISTER_FILE();
}


/**
* Description     : Constructor of class AMFMTestMode
*/
AMFMTestMode::AMFMTestMode()
   : _tunerFiProxy(MIDW_TUNER_FIProxy::createProxy("tunerFiPort", *this)),
     _tunerMasterFiProxy(MIDW_TUNERMASTER_FIProxy::createProxy("tunermasterFiPort", *this)),
     _dabFiProxy(MIDW_EXT_DABTUNER_FIProxy::createProxy("dabFiPort", *this)),
     _antennaStatus(""),
     _fieldStrength(""),
     _signalQuality(""),
     _audioMode(""),
     _audioQuality(""),
     _audioBitrate(""),
     _audioSamplingRate(""),
     _ensembleChannel(""),
     _ensembleName(""),
     _ensembleID(""),
     _serviceName(""),
     _serviceID(""),
     _fMFrequency(""),
     _expertIDIndex(EXPERTID_INDEX_1),
     _DabDTMDBsceneId(0),
     _DRCStatus(false),
     _fMPICode(""),
     _phaseDiversity(""),
     _physicalTuner(0),
     _band(0),
     _audibleTuner(0xFF),
     _hiCut(0),
     _sharxLevel(0),
     _secPhysicalTuner(0),
     _frequency(0),
     _taActive(false),
     _measureIndex(false),
     _concealmentLevel(0),
     _afIndex(false),
     _isRegActive(true),
     _phaseDiversityIndex(0),
     _linkingMode(T_e8_DAB_ServiceLinkingMode__DAB_SERVICELINKING_DAB),
     _ddaState("OFF"),
     _regionType(0),
     _ActiveViewListID(0),
     //Added for HD wst4kor
     _dtmHDScreenStatus(false),
     _hdAudio(""),
     _blendStatus(""),
     _dtmHdStatus(""),
     _dtmHdAudio(""),
     _dtmBlendStatus(""),
     _dtmSis(""),
     _callSign(""),
     _stationId(""),
     _cd_No(""),
     _isHdActive(""),
     _activeHDAudioPrgm(0),
     _primaryHdTuner(T_e8_Tun_PhysicalTuner__INVALID_TYPE),
     _hdActiveStatus(false),
     _secondaryHdTuner(T_e8_Tun_PhysicalTuner__INVALID_TYPE),
     _splitModeStatus(T_e8_Tun_AudioSplitMode__TUN_DEACTIVATE_AUDIO_SPLIT_MODE),
     _isHdParameterSent(false),
     _CurrentSelectedIndex(0),
     _FocusElemValue(0),
     _screenEntryRequest(false),
     _requestCount(0)

{
   if (_tunerFiProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _tunerFiProxy->getPortName());
   }
   if (_tunerMasterFiProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _tunerMasterFiProxy->getPortName());
   }
   if (_dabFiProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _dabFiProxy->getPortName());
   }
   _feature_DDA = false;
   _measureMode = "";
   m_sceneStatus = 0;
   _AMFMTestModeInst = this;
   //wst4kor Added for HD
   // ListRegistry::s_getInstance().addListImplementation(LIST_ID_DTM_HD_MONITOR, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_DTM_HD_PARAM_TUNING, this);
   initDtmHdParamterList();
   m_hddtmsceneStatus = 0;
   _audioPrgmList.setTUN_HD_AUDIOPROGRAM_MPS(false);
   _audioPrgmList.setTUN_HD_AUDIOPROGRAM_SPS1(false);
   _audioPrgmList.setTUN_HD_AUDIOPROGRAM_SPS2(false);
   _audioPrgmList.setTUN_HD_AUDIOPROGRAM_SPS3(false);
   _audioPrgmList.setTUN_HD_AUDIOPROGRAM_SPS4(false);
   _audioPrgmList.setTUN_HD_AUDIOPROGRAM_SPS5(false);
   _audioPrgmList.setTUN_HD_AUDIOPROGRAM_SPS6(false);
   _audioPrgmList.setTUN_HD_AUDIOPROGRAM_SPS7(false);
   _tunerDataBindingHandler = TunerDataBindingHandler::getDataBindingHandlerInstance();

   readKdsForRegion();
   ETG_I_REGISTER_FILE();
}


/**
* Description     : Request from Framework to start registering for properties. This request means GUI is ready now.
*
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : stateChange : represents a state transition of a service
* @return         : void
*/
void AMFMTestMode::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   //
}


/**
* Description     : Request from Framework to start de-registering for properties.
*
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : stateChange : represents a state transition of a service
* @return         : void
*/
void AMFMTestMode::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   //
}


void AMFMTestMode::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_GET_DATA_TESTMODE_DDADDSRelUpRegAll();
      _tunerFiProxy->sendFID_TUN_G_SET_SHARXLEVELRelUpRegAll();
      _tunerFiProxy->sendFID_TUN_G_SET_HICUTRelUpRegAll();
      _tunerFiProxy->sendFID_TUN_G_SET_MEASUREMODERelUpRegAll();
      _tunerFiProxy->sendFID_TUN_G_CALIBRATION_DATARelUpRegAll();
      _tunerFiProxy->sendFID_TUN_G_SET_RDSREGRelUpRegAll();
      _tunerFiProxy->sendFID_TUN_G_SET_AFRelUpRegAll();
      _tunerFiProxy->sendFID_TUN_G_SET_DDARelUpRegAll();

      //wst4kor Added for HD
      _tunerFiProxy->sendFID_TUN_G_AUDIO_SPLIT_MODERelUpRegAll();
      if (TunerCommonHandler::instance()->isHDFeatureEnabled())
      {
         _tunerFiProxy->sendFID_TUN_G_HD_SECTION_PARAMETERRelUpRegAll();
         _tunerFiProxy->sendFID_TUN_G_HD_TEST_MODE_DATARelUpRegAll();
      }
   }
   if ((proxy == _tunerMasterFiProxy) && (_tunerMasterFiProxy.get()))
   {
      _tunerMasterFiProxy->sendFID_TUNMSTR_G_RF_ANTENNA_TESTRelUpRegAll();
      _tunerMasterFiProxy->sendFID_TUNMSTR_G_TP_STATUSRelUpRegAll();
      _tunerMasterFiProxy->sendFID_TUNMSTR_G_ANNO_BUTTON_STATUSRelUpRegAll();
   }
   if (_dabFiProxy == proxy && _dabFiProxy.get())
   {
      _dabFiProxy->sendFID_DAB_G_TESTMODE_DATARelUpRegAll();
      _dabFiProxy->sendFID_DAB_G_BG_TESTMODE_DATARelUpRegAll();
      _dabFiProxy->sendFID_DAB_G_TUNER_STATUSRelUpRegAll();
      _dabFiProxy->sendFID_DAB_G_ACTIVE_SERVICE_INFORelUpRegAll();
      _dabFiProxy->sendFID_DAB_G_ACTIVE_ENSEMBLE_INFORelUpRegAll();
      _dabFiProxy->sendFID_DAB_G_ACTIVE_COMPONENT_INFORelUpRegAll();
      // _dabFiProxy->sendFID_DAB_G_ANNO_STATUSRelUpRegAll();
   }
}


void AMFMTestMode::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("AMFMTestMode : onAvailable"));
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_GET_DATA_TESTMODE_DDADDSUpReg(*this);
      _tunerFiProxy->sendFID_TUN_G_SET_SHARXLEVELUpReg(*this);
      _tunerFiProxy->sendFID_TUN_G_SET_HICUTUpReg(*this);
      _tunerFiProxy->sendFID_TUN_G_SET_MEASUREMODEUpReg(*this);
      _tunerFiProxy->sendFID_TUN_G_CALIBRATION_DATAUpReg(*this);
      _tunerFiProxy->sendFID_TUN_G_SET_RDSREGUpReg(*this);
      _tunerFiProxy->sendFID_TUN_G_SET_AFUpReg(*this);
      _tunerFiProxy->sendFID_TUN_G_SET_DDAUpReg(*this);

      //wst4kor Added for HD
      _tunerFiProxy->sendFID_TUN_G_AUDIO_SPLIT_MODEUpReg(*this);
      if (TunerCommonHandler::instance()->isHDFeatureEnabled())
      {
         _tunerFiProxy->sendFID_TUN_G_HD_TEST_MODE_DATAUpReg(*this);
         _tunerFiProxy->sendFID_TUN_G_HD_SECTION_PARAMETERUpReg(*this);
      }
   }

   if (_tunerMasterFiProxy == proxy && _tunerMasterFiProxy.get())
   {
      _tunerMasterFiProxy->sendFID_TUNMSTR_G_RF_ANTENNA_TESTUpReg(*this);
      _tunerMasterFiProxy->sendFID_TUNMSTR_G_ANNO_BUTTON_STATUSUpReg(*this);
      _tunerMasterFiProxy->sendFID_TUNMSTR_G_TP_STATUSUpReg(*this);
   }

   if (_dabFiProxy == proxy && _dabFiProxy.get())
   {
      _dabFiProxy->sendFID_DAB_G_TESTMODE_DATAUpReg(*this);
      _dabFiProxy->sendFID_DAB_G_BG_TESTMODE_DATAUpReg(*this);
      _dabFiProxy->sendFID_DAB_G_TUNER_STATUSUpReg(*this);
      _dabFiProxy->sendFID_DAB_G_ACTIVE_SERVICE_INFOUpReg(*this);
      _dabFiProxy->sendFID_DAB_G_ACTIVE_ENSEMBLE_INFOUpReg(*this);
      _dabFiProxy->sendFID_DAB_G_ACTIVE_COMPONENT_INFOUpReg(*this);
      // _dabFiProxy->sendFID_DAB_G_ANNO_STATUSUpReg(*this);
   }
}


void AMFMTestMode::onFID_DAB_G_BG_TESTMODE_DATAError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::FID_DAB_G_BG_TESTMODE_DATAError >& /*error*/) {}
void AMFMTestMode::onFID_DAB_G_BG_TESTMODE_DATAStatus(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::FID_DAB_G_BG_TESTMODE_DATAStatus >& status)
{
   uint32 u32BGEnsembleFreg;
   const ::midw_ext_dabtuner_fi_types::T_DAB_BG_TestModeData& bgTestModeData = status->getBG_TestModeData();
   tChar szData[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
   u32BGEnsembleFreg = bgTestModeData.getU32BgFrequency();
   if (TUNER_ZERO_VALUE == (u32BGEnsembleFreg % M_DIVIDE_BY_HUNDRED))
   {
      sprintf(szData, "%.1f", ((tF32)u32BGEnsembleFreg / M_CONVERSION_KHZ_TO_MHZ));
   }
   else
   {
      sprintf(szData, "%.2f", ((tF32)u32BGEnsembleFreg / M_CONVERSION_KHZ_TO_MHZ));
   }
   (*oDABTestModePointer).mBg_tuner_ens_freq = szData;
   _DTMFieldMap[BG_EnsFreq] = TunerUtility::u32ConversionToString(bgTestModeData.getU32BgFrequency());
   _DTMFieldMap[CurrentEnsID] = TunerUtility::u32ConversionToString(bgTestModeData.getU32BgCurrentEnsID());
   _DTMFieldMap[BgFICBER] = TunerUtility::u32ConversionToString(bgTestModeData.getU32BgFICBER());
   // T_b16_DAB_TunerStatusFlags b16_DAB_TunerStatus = T_b16_DAB_TunerStatusFlags__DAB_TUNERSTATUSFLAGS_AUDIO_SERVICE_VALID;
   //_DTMFieldMap[DabMode] = bgTestModeData.getsBgDabMode();
   // (*oDABTestModePointer).mBGSync_flag = (Candera::String) b16_DAB_TunerStatus;
   (*oDABTestModePointer).mBg_Ensemble_ID = TunerUtility::u32ConversionToString(bgTestModeData.getU32BgCurrentEnsID()).c_str();
   (*oDABTestModePointer).mBg_FIC = TunerUtility::u32ConversionToString(bgTestModeData.getU32BgFICBER()).c_str();
   ETG_TRACE_USR4(("Tuner:Bg_tuner_ens_freq  : %s", szData));
   ETG_TRACE_USR4(("Tuner:Bg_Ensemble_ID: %s", TunerUtility::u32ConversionToString(bgTestModeData.getU32BgCurrentEnsID()).c_str()));
   ETG_TRACE_USR4(("Tuner:Bg_FIC : %s", TunerUtility::u32ConversionToString(bgTestModeData.getU32BgFICBER()).c_str()));
   oDABTestModePointer.MarkAllItemsModified();
   oDABTestModePointer.SendUpdate(true);
}


void AMFMTestMode::onFID_DAB_G_TESTMODE_DATAError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::FID_DAB_G_TESTMODE_DATAError >& /*error*/) {}
void AMFMTestMode::onFID_DAB_G_TESTMODE_DATAStatus(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::FID_DAB_G_TESTMODE_DATAStatus >& status)
{
   const ::midw_ext_dabtuner_fi_types::T_DAB_TestModeData& testModeData = status->getTestModeData();
   _fieldStrength = TunerUtility::u8ConversionToString(testModeData.getS8FieldStrength());
   _fMPICode = TunerUtility::u16ConversionToString(testModeData.getU16AltFMPI());
   _fMFrequency = TunerUtility::u32ConversionToString(testModeData.getU32AltFMFrequency());
   _DTMFieldMap[FieldStrength] = _fieldStrength;
   _DTMFieldMap[FMPI] = _fMPICode;
   _DTMFieldMap[FM] = _fMFrequency;
   _DTMFieldMap[FIC] = TunerUtility::u32ConversionToString(testModeData.getU32FICBER());
   _DTMFieldMap[NetbitErroRate] = TunerUtility::u32ConversionToString(testModeData.getU32NETBER());
   _DTMFieldMap[MSC] = TunerUtility::u32ConversionToString(testModeData.getU32MSCBER());
   _DTMFieldMap[FMQuality] = TunerUtility::u8ConversionToString(testModeData.getU8AltFMQuality());
   _DTMFieldMap[PL] = TunerUtility::u8ConversionToString(testModeData.getU8ProtectionLevel());
   _DTMFieldMap[RS_FEC] = TunerUtility::u8ConversionToString(testModeData.getU8RSFEC());
   _DTMFieldMap[SyncFlag] = testModeData.getBSync() ? "1" : "0";
   _linkingMode = testModeData.getE8ServiceLinkingMode();
   _concealmentLevel = testModeData.getU8ConcealmentLevel();
   ETG_TRACE_USR4(("Tuner:AMFMTestMode : %d", _concealmentLevel));

   (*oDABTestModeGTestmodeInfo).mFM = (Candera::String)_fMFrequency.c_str();
   (*oDABTestModeGTestmodeInfo).mPI = (Candera::String)_fMPICode.c_str();
   (*oDABTestModeGTestmodeInfo).mQFM = TunerUtility::u8ConversionToString(testModeData.getU8AltFMQuality()).c_str();
   (*oDABTestModeGTestmodeInfo).mPL = TunerUtility::u8ConversionToString(testModeData.getU8ProtectionLevel()).c_str();

   (*oDABTestModePointer).mSync_flag = testModeData.getBSync() ? "1" : "0";
   (*oDABTestModePointer).mFIC = TunerUtility::u32ConversionToString(testModeData.getU32FICBER()).c_str();;
   // (*oDABTestModePointer).mNetbitErrorrate = TunerUtility::u32ConversionToString(testModeData.getU32NETBER()).c_str();;
   (*oDABTestModePointer).mField_Strength = (Candera::String)_fieldStrength.c_str();
   (*oDABTestModePointer).mMSC = TunerUtility::u32ConversionToString(testModeData.getU32MSCBER()).c_str();;
   (*oDABTestModePointer).mRS_FEC = TunerUtility::u8ConversionToString(testModeData.getU8RSFEC()).c_str();;
   //(*oDABTestModePointer).mNET=szData;
   ETG_TRACE_USR4(("Tuner:onField_Strength : %s", _fieldStrength.c_str()));
   ETG_TRACE_USR4(("Tuner:onMSC : %s", TunerUtility::u32ConversionToString(testModeData.getU32MSCBER()).c_str()));
   ETG_TRACE_USR4(("Tuner:onFIC : %s", TunerUtility::u32ConversionToString(testModeData.getU32FICBER()).c_str()));
   //ETG_TRACE_USR4(("Tuner:onmNetbitErrorrate : %s", TunerUtility::u32ConversionToString(testModeData.getU32NETBER()).c_str()));
   oDABTestModePointer.MarkAllItemsModified();
   oDABTestModeGTestmodeInfo.MarkAllItemsModified();
   oDABTestModeGTestmodeInfo.SendUpdate(true);
   oDABTestModePointer.SendUpdate(true);
   ETG_TRACE_USR4(("AMFMTestMode : onFID_DAB_G_TESTMODE_DATAStatus"));
   ETG_TRACE_USR4(("Tuner:onField_Strength : %s", _fieldStrength.c_str()));
   ETG_TRACE_USR4(("Tuner:onMSC : %s", TunerUtility::u32ConversionToString(testModeData.getU32MSCBER()).c_str()));
   ETG_TRACE_USR4(("Tuner:onFIC : %s", TunerUtility::u32ConversionToString(testModeData.getU32FICBER()).c_str()));
}


void AMFMTestMode::onFID_DAB_S_SET_TESTMODEError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::FID_DAB_S_SET_TESTMODEError >& /*error*/) {}
void AMFMTestMode::onFID_DAB_S_SET_TESTMODEResult(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::FID_DAB_S_SET_TESTMODEResult >& /*result*/) {}

/**
* Description     : Virtual function implemented to get update of FID_TUNMSTR_G_ANNO_BUTTON_STATUS Error
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNERMASTER_FIProxy'
* @param[in]      : error : The error message of 'FID_TUNMSTR_G_ANNO_BUTTON_STATUS'
* @return         : void
*/
//void AMFMTestMode::onFID_TUNMSTR_G_ANNO_BUTTON_STATUSError(const ::boost::shared_ptr< MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUNMSTR_G_ANNO_BUTTON_STATUSError >& /*error*/) */
//{
//}

/**
* Description     : Virtual function implemented to get update of FID_TUNMSTR_G_ANNO_BUTTON_STATUS Status
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNERMASTER_FIProxy'
* @param[in]      : status : The Result message of the property "FID_TUNMSTR_G_ANNO_BUTTON_STATUS"
* @return         : void
*/


void AMFMTestMode::onFID_TUN_G_SET_AFError(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_SET_AFError >& /*error*/)
{
   ETG_TRACE_USR4(("RadioSetting : onFID_TUN_G_SET_AFError"));
}


void AMFMTestMode::onFID_TUN_G_SET_AFStatus(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_SET_AFStatus >& status)
{
   bool bDataUpdated = false;
   ETG_TRACE_USR4(("RadioSetting : onFID_TUN_G_SET_AFStatus %d", status->getU8AF()));
   if (status->getU8AF())
   {
      //_afStatus = "ON";
      _afIndex = 1;
   }
   else
   {
      //_afStatus = "OFF";
      _afIndex = 0;
   }

   (*oFMSetupMenuInfo).mToggleAF = _afIndex;
   oFMSetupMenuInfo.MarkAllItemsModified();
   bDataUpdated = oFMSetupMenuInfo.SendUpdate(true);
   //update GUI
}


/**
* Description     : Virtual function implemented to get update of FID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENT Error
*
* @param[in]      : proxy: the client side representation of the CCA Functional InteETG_TRACE_USR4(("onFID_TUNMSTR_G_ANNOUNCEMENT_STATUSStatus: src = %d miAnnoType = %d", src,miAnnoType));
rface 'MIDW_TUNERMASTER_FIProxy'
* @param[in]      : error : The error message of 'FID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENT'
* @return         : void
*/
void AMFMTestMode::onFID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENTError(const ::boost::shared_ptr< MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENTError >& /*error*/)
{
}


/**
* Description     : Virtual function implemented to get update of FID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENT Result
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNERMASTER_FIProxy'
* @param[in]      : result : The Result message of the property "FID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENT"
* @return         : void
*/
void AMFMTestMode::onFID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENTResult(const ::boost::shared_ptr< MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENTResult >& /*result*/)
{
}


/**
* Description     : Virtual function implemented to get update of FID_TUNMSTR_S_STOP_ANNOUNCEMENT Error
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNERMASTER_FIProxy'
* @param[in]      : error : The error message of 'FID_TUNMSTR_S_STOP_ANNOUNCEMENT'
* @return         : void
*/
void AMFMTestMode::onFID_TUNMSTR_S_STOP_ANNOUNCEMENTError(const ::boost::shared_ptr< MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUNMSTR_S_STOP_ANNOUNCEMENTError >& /*error*/)
{
}


/**
* Description     : Virtual function implemented to get update of FID_TUNMSTR_S_STOP_ANNOUNCEMENT Result
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNERMASTER_FIProxy'
* @param[in]      : result : The Result message of the property "FID_TUNMSTR_S_STOP_ANNOUNCEMENT"
* @return         : void
*/
void AMFMTestMode::onFID_TUNMSTR_S_STOP_ANNOUNCEMENTResult(const ::boost::shared_ptr< MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUNMSTR_S_STOP_ANNOUNCEMENTResult >& /*result*/)
{
}


/**
* Description     : Virtual function implemented to get update of FID_TUN_G_ANNOUNCEMENT_INFO Error
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : error : The error message of 'FID_TUN_G_ANNOUNCEMENT_INFO'
* @return         : void
*/
void AMFMTestMode::onFID_TUN_G_ANNOUNCEMENT_INFOError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_G_ANNOUNCEMENT_INFOError >& /*error*/)
{
}


void AMFMTestMode::onFID_TUN_G_ANNOUNCEMENT_INFOStatus(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_G_ANNOUNCEMENT_INFOStatus >& status)
{
}


void AMFMTestMode::onSelectPhaseDiversity()
{
   T_TunerFeature testMode;
   T_b32_Tun_Features feature;
   T_b32_Tun_Features feature_mask;

   feature_mask.setTUN_FEATURE_DDA_MODE(true);
   testMode.setB32MaskFeatures(feature_mask);
   feature.setTUN_FEATURE_DDA_MODE(!_feature_DDA);
   testMode.setB32Features(feature);
   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_FEATUREStart(*this, testMode)));
}


void AMFMTestMode::onFID_TUN_G_SET_DDAError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_G_SET_DDAError >& /*error*/)
{
}


/**
* Description     : Virtual function implemented to get update of FID_TUN_G_SET_DDA Status
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : status : The status message of 'FID_TUN_G_SET_DDA'
* @return         : void
*/
void AMFMTestMode::onFID_TUN_G_SET_DDAStatus(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_G_SET_DDAStatus >& status)
{
   bool bDataUpdated = false;
   _feature_DDA = status->getU8DDA();
   if (_feature_DDA == false)
   {
      _ddaState = "OFF";
      _phaseDiversityIndex = 0;
   }
   else
   {
      _ddaState = "ON";
      _phaseDiversityIndex = 1;
   }
   //char szData[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
   //sprintf(szData, "%d", _sharxLevel);
   (*oFMSetupMenuInfo).mTogglePhaseDiversity = _phaseDiversityIndex;
   oFMSetupMenuInfo.MarkAllItemsModified();
   bDataUpdated = oFMSetupMenuInfo.SendUpdate(true);
}


void AMFMTestMode::onFID_TUN_S_SETUP_DDADDSError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_S_SETUP_DDADDSError >& /*error*/)
{
}


/**
* Description     : Virtual function implemented to get update of FID_TUN_G_SET_DDA Result
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : status : The status message of 'FID_TUN_G_SET_DDA'
* @return         : void
*/
void AMFMTestMode::onFID_TUN_S_SETUP_DDADDSResult(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_S_SETUP_DDADDSResult >& /*result*/)
{
}


void AMFMTestMode::onFID_TUNMSTR_G_RF_ANTENNA_TESTError(const ::boost::shared_ptr< ::MIDW_TUNERMASTER_FI::MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< ::MIDW_TUNERMASTER_FI::FID_TUNMSTR_G_RF_ANTENNA_TESTError >& /*error*/) {}
void AMFMTestMode::onFID_TUNMSTR_G_RF_ANTENNA_TESTStatus(const ::boost::shared_ptr< ::MIDW_TUNERMASTER_FI::MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< ::MIDW_TUNERMASTER_FI::FID_TUNMSTR_G_RF_ANTENNA_TESTStatus >& status)
{
   ETG_TRACE_USR4(("AMFMTestMode : onFID_TUNMSTR_G_RF_ANTENNA_TESTStatus"));
   using namespace ::midw_tunermaster_fi_types;
   uint8 antennaTest = status->getTunmstr_RF_Antenna().getAntennaTest();
   if (antennaTest != T_e8_AntennaTest__DAB_Antenna1)
   {
      return;
   }

   uint8 antennaStatus = status->getTunmstr_RF_Antenna().getRFAntennaStatus();
   std::string antennaStatusString = "Unknown";
   switch (antennaStatus)
   {
      case T_e8_RFAntenna__TUN_MSTR_RF_ANTENNA_STATE_OK:
         antennaStatusString = "Ok";
         break;
      case T_e8_RFAntenna__TUN_MSTR_RF_ANTENNA_STATE_OPEN:
         antennaStatusString = "Open";
         break;
      case T_e8_RFAntenna__TUN_MSTR_RF_ANTENNA_STATE_SHORT:
         antennaStatusString = "Short";
         break;
      case T_e8_RFAntenna__TUN_MSTR_RF_ANTENNA_STATE_DEACTIVATED:
         antennaStatusString = "Deactivated";
         break;
      case T_e8_RFAntenna__TUN_MSTR_RF_ANTENNA_STATE_DEACTIVATED_LOW_VG:
         antennaStatusString = "Deactivated_Lvg";
         break;
      case T_e8_RFAntenna__TUN_MSTR_RF_ANTENNA_STATE_DEACTIVATED_HIGH_VG:
         antennaStatusString = "Deactivated_Hvg";
         break;
      case T_e8_RFAntenna__TUN_MSTR_RF_ANTENNA_STATE_OPEN_AMPLIFIER:
         antennaStatusString = "Open_Amp";
         break;
      case T_e8_RFAntenna__TUN_MSTR_RF_ANTENNA_STATE_PASSIVE:
         antennaStatusString = "Passive";
         break;
      default:
         antennaStatusString = "Unknown";
         break;
   }
   _antennaStatus = antennaStatusString;
}


void AMFMTestMode::onFID_TUNMSTR_S_SRCCHG_TESTMODEError(const ::boost::shared_ptr< ::MIDW_TUNERMASTER_FI::MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< ::MIDW_TUNERMASTER_FI::FID_TUNMSTR_S_SRCCHG_TESTMODEError >& /*error*/) {}
void AMFMTestMode::onFID_TUNMSTR_S_SRCCHG_TESTMODEResult(const ::boost::shared_ptr< ::MIDW_TUNERMASTER_FI::MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< ::MIDW_TUNERMASTER_FI::FID_TUNMSTR_S_SRCCHG_TESTMODEResult >& /*result*/) {}
void AMFMTestMode::onFID_TUNMSTR_G_TP_STATUSError(const ::boost::shared_ptr< MIDW_TUNERMASTER_FI::MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< MIDW_TUNERMASTER_FI::FID_TUNMSTR_G_TP_STATUSError >& /*error*/) {}
void AMFMTestMode::onFID_TUNMSTR_G_TP_STATUSStatus(const ::boost::shared_ptr< MIDW_TUNERMASTER_FI::MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< MIDW_TUNERMASTER_FI::FID_TUNMSTR_G_TP_STATUSStatus >& status)
{
   bool bDataUpdated = false;
   bool tpStatus = status->getTP_STATE();
   T_e8_Source Source_Active = status->getTP_Source();
   if (Source_Active == T_e8_Source__TUN_MSTR_SOURCE_DAB)
   {
      ETG_TRACE_USR4(("Tuner:onFID_TUNMSTR_G_TP_STATUSStatus tpStatus : %d", tpStatus));
      TunerDataBindingHandler::getDataBindingHandlerInstance()->TunerPLMsg(EV_MSG_TO_PL_TA_ICON_TYPE, tpStatus, "");
   }
}


void AMFMTestMode::onFID_TUN_S_CALIBRATION_DATAError(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_S_CALIBRATION_DATAError >& /*error*/) {}
void AMFMTestMode::onFID_TUN_S_CALIBRATION_DATAResult(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_S_CALIBRATION_DATAResult >& /*result*/) {}
void AMFMTestMode::onFID_TUN_G_CALIBRATION_DATAError(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_CALIBRATION_DATAError >& /*error*/) {}
void AMFMTestMode::onFID_TUN_G_CALIBRATION_DATAStatus(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_CALIBRATION_DATAStatus >& status)
{
   ETG_TRACE_USR4(("AMFMTestMode : onFID_TUN_G_CALIBRATION_DATAStatus"));

   //uint16 calibFMTun1 = status->getCalibrationDataValues().getCalibrationValues()[0];
   uint16 calibFMTun1 = status->getCalibrationDataValues().getPhysicalTunerOne().getU16AMCalibrationValue();
   ETG_TRACE_USR4(("AMFMTestMode : onFID_TUN_G_CALIBRATION_DATAStatus %d", calibFMTun1));
   std::string s_calibFMTun1 = TunerUtility::u16ConversionToString(calibFMTun1);

   //uint16 calibFMTun2 = status->getCalibrationData().getCalibrationValues()[1];
   uint16 calibFMTun2 = status->getCalibrationDataValues().getPhysicalTunerOne().getU16FMCalibrationValue();
   ETG_TRACE_USR4(("AMFMTestMode : onFID_TUN_G_CALIBRATION_DATAStatus %d", calibFMTun2));
   std::string s_calibFMTun2 = TunerUtility::u16ConversionToString(calibFMTun2);

   //uint16 calibFMTun3 = status->getCalibrationData().getCalibrationValues()[2];
   uint16 calibFMTun3 = status->getCalibrationDataValues().getPhysicalTunerTwo().getU16FMCalibrationValue();
   std::string s_calibFMTun3 = TunerUtility::u16ConversionToString(calibFMTun3);

   //uint16 calibFMTun4 = status->getCalibrationData().getCalibrationValues()[3];
   uint16 calibFMTun4 = status->getCalibrationDataValues().getPhysicalTunerThree().getU16FMCalibrationValue();
   std::string s_calibFMTun4 = TunerUtility::u16ConversionToString(calibFMTun4);

   //uint16 calibFMTun5 = status->getCalibrationData().getCalibrationValues()[4];
   uint16 calibFMTun5 = status->getCalibrationDataValues().getPhysicalTunerFour().getU16AMCalibrationValue();
   std::string s_calibFMTun5 = TunerUtility::u16ConversionToString(calibFMTun5);

   bool bDataUpdated = false;

   (*oEngineeringMenuData2).mAMTuner = s_calibFMTun5.c_str();

   (*oEngineeringMenuData2).mFMTuner1 = s_calibFMTun1.c_str();

   (*oEngineeringMenuData2).mFMTuner2 = s_calibFMTun2.c_str();

   (*oEngineeringMenuData2).mFMTuner3 = s_calibFMTun3.c_str();

   (*oEngineeringMenuData2).mFMTuner4 = s_calibFMTun4.c_str();

   oEngineeringMenuData2.MarkAllItemsModified();

   bDataUpdated = oEngineeringMenuData2.SendUpdate(true);

   //TunerDataBindingHandler::getDataBindingHandlerInstance()->updateCalibrationData(s_calibFMTun1, s_calibFMTun2, s_calibFMTun3, s_calibFMTun4, s_calibFMTun5);
}


void AMFMTestMode::onFID_TUN_G_SET_MEASUREMODEError(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_SET_MEASUREMODEError >& /*error*/) {}
//void AMFMTestMode::onFID_TUN_G_SET_MEASUREMODEStatus(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_SET_MEASUREMODEStatus >& status){}


void AMFMTestMode::onFID_TUN_S_SET_SHARXLEVELError(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_S_SET_SHARXLEVELError >& /*error*/) {}
void AMFMTestMode::onFID_TUN_S_SET_SHARXLEVELResult(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_S_SET_SHARXLEVELResult >& /*result*/) {}

void AMFMTestMode::onFID_TUN_G_SET_SHARXLEVELError(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_SET_SHARXLEVELError >& /*error*/) {}
void AMFMTestMode::onFID_TUN_G_SET_SHARXLEVELStatus(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_SET_SHARXLEVELStatus >&  status)
{
   bool bDataUpdated = false;
   _sharxLevel = status->getU8SharxStep();
   ETG_TRACE_USR4(("onFID_TUN_G_SET_SHARXLEVELStatus _sharxLevel = %d", status->getU8SharxStep()));
   //TunerDataBindingHandler::getDataBindingHandlerInstance()->TunerPLMsg(EV_MSG_TO_PL_SHARXABS_STATUS, _sharxLevel, "");
   char szData[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
   sprintf(szData, "%d", _sharxLevel);
   (*oEngineeringMenuData1).mSzSharx = szData;
   oEngineeringMenuData1.MarkAllItemsModified();
   bDataUpdated = oEngineeringMenuData1.SendUpdate(true);
}


void AMFMTestMode::onFID_TUN_G_SET_MEASUREMODEStatus(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_G_SET_MEASUREMODEStatus >& status)
{
   bool bDataUpdated = false;
   if (status->getU8Measuremode() == 1)
   {
      _measureMode = "ON";
      _measureIndex = 1;
   }
   else if (status->getU8Measuremode() == 0)
   {
      _measureMode = "OFF";
      _measureIndex = 0;
   }
   (*oFMSetupMenuInfo).mToggleMM = _measureIndex;
   ETG_TRACE_USR3(("mMeasureOnOff: %d", (*oFMSetupMenuInfo).mToggleMM));
   oFMSetupMenuInfo.MarkAllItemsModified();
   bDataUpdated = oFMSetupMenuInfo.SendUpdate(true);
}


void AMFMTestMode::onFID_TUN_G_SET_RDSREGError(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_SET_RDSREGError >& /*error*/)
{
}


void AMFMTestMode::onFID_TUN_G_SET_RDSREGStatus(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_SET_RDSREGStatus >& status)
{
   bool bDataUpdated = false;
   ETG_TRACE_USR4(("RadioSetting : onFID_TUN_G_SET_RDSREGStatus %d", status->getU8RDSreg()));
   if (status->getU8RDSreg() == 0)
   {
      _isRegActive = true;
   }
   else if (status->getU8RDSreg() == 1)
   {
      _isRegActive = false;
   }
   (*oFMSetupMenuInfo).mToggleRDS = _isRegActive;//Display RDS info
   oFMSetupMenuInfo.MarkAllItemsModified();
   bDataUpdated = oFMSetupMenuInfo.SendUpdate(true);
}


void AMFMTestMode::vSetHiCutValue(uint8 value)
{
   ETG_TRACE_USR4(("AMFMTestMode::vSetHiCutValue _hiCut"));
   if (value == 0)
   {
      _hiCut = decreaseRadioParam(_hiCut, HICUT_MIN);
   }
   else if (value == 1)
   {
      _hiCut = increaseRadioParam(_hiCut, HICUT_MAX);
   }
   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_HICUTStart(*this, _hiCut)));
}


void AMFMTestMode::onFID_TUN_G_SET_HICUTError(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_SET_HICUTError >& /*error*/) {}
void AMFMTestMode::onFID_TUN_G_SET_HICUTStatus(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_SET_HICUTStatus >& status)
{
   bool bDataUpdated = false;
   _hiCut = status->getU8AbsVal();
   ETG_TRACE_USR4(("AMFMTestMode::onFID_TUN_G_SET_HICUTStatus _hiCut = %d", status->getU8AbsVal()));
   //TunerDataBindingHandler::getDataBindingHandlerInstance()->TunerPLMsg(EV_MSG_TO_PL_HICUTABS_STATUS, _hiCut, "");
   char szData[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
   sprintf(szData, "%d", _hiCut);
   (*oEngineeringMenuData1).mSzHicut = szData;//_hiCut;
   oEngineeringMenuData1.MarkAllItemsModified();
   bDataUpdated = oEngineeringMenuData1.SendUpdate(true);
}


void AMFMTestMode::vSetSharxValue(uint8 value)
{
   ETG_TRACE_USR4(("vSetSharxValue _sharxLevel"));
   if (value == 0)
   {
      _sharxLevel = decreaseRadioParam(_sharxLevel, SHARX_MIN);
   }
   else if (value == 1)
   {
      _sharxLevel = increaseRadioParam(_sharxLevel, SHARX_MAX);
   }

   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_SHARXLEVELStart(*this, _sharxLevel)));
}


/**
* Description     : Function called on press of '-' option in AM FM Settings screen.
*
* @return         : uint8
*/
uint8 AMFMTestMode::decreaseRadioParam(uint8 paramValue, uint8 valueMin)
{
   if (paramValue > valueMin)
   {
      paramValue = paramValue - STEP_SIZE;
   }
   else
   {
      paramValue = valueMin;
   }
   return paramValue;
}


/**
* Description     : Function called on press of '+' option in AM FM Settings screen.
*
* @return         : uint8
*/
uint8 AMFMTestMode::increaseRadioParam(uint8 paramValue, uint8 valueMax)
{
   if (paramValue < valueMax)
   {
      paramValue = paramValue + STEP_SIZE;
   }
   else
   {
      paramValue = valueMax;
   }
   return paramValue;
}


/**
* Description     : Virtual function implemented to get update of FID_TUN_S_SET_FEATURE Error
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : error : The error message of 'FID_TUN_S_SET_FEATURE'
* @return         : void
*/
void AMFMTestMode::onFID_TUN_S_SET_FEATUREError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_SET_FEATUREError >& /*error*/)
{
}


/**
* Description     : Virtual function implemented to get update of FID_TUNMSTR_S_TEST_MODE Error
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNERMASTER_FIProxy'
* @param[in]      : error : The error message of 'FID_TUNMSTR_S_TEST_MODE'
* @return         : void
*/
void AMFMTestMode::onFID_TUNMSTR_S_TEST_MODEError(const ::boost::shared_ptr< MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUNMSTR_S_TEST_MODEError >& /*error*/)
{
}


/**
* Description     : Virtual function implemented to get update of FID_TUNMSTR_S_TEST_MODE Result
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNERMASTER_FIProxy'
* @param[in]      : result : The status message of 'FID_TUNMSTR_S_TEST_MODE'
* @return         : void
*/
void AMFMTestMode::onFID_TUNMSTR_S_TEST_MODEResult(const ::boost::shared_ptr< MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUNMSTR_S_TEST_MODEResult >& /*result*/)
{
}


/**
* Description     : Virtual function implemented to get update of FID_TUN_S_SET_FEATURE Result
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : result : The status message of 'FID_TUN_S_SET_FEATURE'
* @return         : void
*/
void AMFMTestMode::onFID_TUN_S_SET_FEATUREResult(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_SET_FEATUREResult >& /*result*/)
{
}


void AMFMTestMode::onFID_TUN_S_SET_HICUTError(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& proxy, const boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_S_SET_HICUTError >& /*error*/) {}
void AMFMTestMode::onFID_TUN_S_SET_HICUTResult(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& proxy, const boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_S_SET_HICUTResult >& /*result*/) {}

void AMFMTestMode::onFID_TUN_G_GET_DATA_TESTMODE_DDADDSError(const ::boost::shared_ptr< MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_TUNER_FI::FID_TUN_G_GET_DATA_TESTMODE_DDADDSError >& /*error*/) {}
void AMFMTestMode::onFID_TUN_G_GET_DATA_TESTMODE_DDADDSStatus(const ::boost::shared_ptr< MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_TUNER_FI::FID_TUN_G_GET_DATA_TESTMODE_DDADDSStatus >& status)
{
   ETG_TRACE_USR4(("AMFMTestMode : onFID_TUN_G_GET_DATA_TESTMODE_DDADDSStatus"));

   T_TunerTestModeData_DDADDS TMstatus = status->getTunerTestModeData();

   _physicalTuner = TMstatus.getU8PhysicalTuner();
   if (TMstatus.getU8TestmodeBits().getTUN_AUDIO_TUNER_BIT() == TESTMODEBITACTIVE)
   {
      _band = TMstatus.getU8Source();
   }
   _frequency = TMstatus.getU32AFFrequency();

   ////Update Quad Monitor
   sendQuadMonitorData(status);

   if (_audibleTuner == PHYSICAL_TUNER_TWO)
   {
      _secPhysicalTuner = PHYSICAL_TUNER_THREE;
   }
   else if (_audibleTuner == PHYSICAL_TUNER_THREE)
   {
      _secPhysicalTuner = PHYSICAL_TUNER_TWO;
   }
   uint8 fieldStrength = TMstatus.getU8Fieldstrength();
   uint32 frequency = TMstatus.getU32AFFrequency();
   std::string fs = "";
   fs = TunerUtility::u8ConversionToString(fieldStrength);
   fs = fs.append(" dBmV");
   //wst4kor Added for HD
   if (_physicalTuner == _secondaryHdTuner)
   {
      TunerDataBindingHandler::getDataBindingHandlerInstance()->updateHdDTMSecPhysicalTunerInfo(TunerUtility::formatTestmodeFrequency(frequency, getBand(_band)), fs);
   }
   else if (_physicalTuner == _primaryHdTuner)
   {
      TunerDataBindingHandler::getDataBindingHandlerInstance()->updateHdDTMAudiblePhysicalTunerInfo(TunerUtility::formatTestmodeFrequency(frequency, getBand(_band)), fs);
   }
   //Update STM Screen
   sendStmScreenData(status);
}


void AMFMTestMode::sendQuadMonitorData(const boost::shared_ptr< FID_TUN_G_GET_DATA_TESTMODE_DDADDSStatus >& status)
{
   T_TunerTestModeData_DDADDS TMstatus = status->getTunerTestModeData();

   //TunerDataBindingHandler* _testModeDataBinding = TunerDataBindingHandler::getDataBindingHandlerInstance();

   if ((TMstatus.getU8TestmodeBits().getTUN_AUDIO_TUNER_BIT()))
   {
      _audibleTuner = TMstatus.getU8PhysicalTuner();
      updateTMactiveTunerdata(TMstatus.getU8PhysicalTuner());
   }
   uint8 bandOfPhysicalTuner;

   //if (NULL != _testModeDataBinding)
   //{
   uint32 FREQ = TMstatus.getU32AFFrequency();
   std::string freq = TunerUtility::formatTestmodeFrequency(FREQ, getBand(TMstatus.getU8Source()));
   ETG_TRACE_USR3(("getU32AFFrequency: %d", FREQ));
   ETG_TRACE_USR3(("getU32AFFrequency string: %s", freq.c_str()));

   uint16 PI = TMstatus.getU16CurrentPI();
   std::string pi = TunerUtility::u16ConversionToHexString(PI);
   ETG_TRACE_USR3(("getU16CurrentPI: %d", PI));
   ETG_TRACE_USR3(("getU16CurrentPI string: %s", pi.c_str()));

   std::string psName = TMstatus.getSCurrentStationName();
   ETG_TRACE_USR3(("getU16CurrentPI string: %s", psName.c_str()));

   uint8 FS = TMstatus.getU8Fieldstrength();
   ETG_TRACE_USR3(("getU8Fieldstrength: %d", FS));

   uint8 QUALITY = TMstatus.getU8Quality();
   std::string fs_q = TunerUtility::quadConcatenate(FS, QUALITY);
   ETG_TRACE_USR3(("getU8Quality: %d", QUALITY));
   ETG_TRACE_USR3(("getU8Quality string: %s", fs_q.c_str()));

   uint8 HC = TMstatus.getU8HiCut();

   uint8 HcExt = TMstatus.getU8HicutExtender();

   std::string hiCut = TunerUtility::quadConcatenate(HC, HcExt);

   uint8 SharxValue = TMstatus.getU8Sharx();
   std::string sharx = TunerUtility::u8ConversionToString(SharxValue);

   uint8 TRAFPRG = (uint8)TMstatus.getU8TestmodeBits().getTUN_TP_BIT();
   std::string TP = TunerUtility::u8ConversionToString(TRAFPRG);
   uint8 TraficAnnounce = (uint8)TMstatus.getU8TestmodeBits().getTUN_TA_BIT();
   std::string TA = TunerUtility::u8ConversionToString(TraficAnnounce);
   uint8 rds = TMstatus.getU8RdsErrorRate();
   std::string RDS = TunerUtility::u8ConversionToString(rds);
   uint8 afLen = TMstatus.getU8AFListLength();
   std::string afListLen = TunerUtility::u8ConversionToString(afLen);
   uint8 MP = TMstatus.getU8Multipath();
   uint8 Neigh = TMstatus.getU8Neighbour();
   std::string mp_nc = TunerUtility::quadConcatenate(MP, Neigh);
   uint8 dev = TMstatus.getU8FMDeviation();
   std::string fmDev = TunerUtility::u8ConversionToString(dev);
   uint8 CS = TMstatus.getU8ChannelSeparation();
   std::string cs = TunerUtility::u8ConversionToString(CS);

   ETG_TRACE_USR4(("AMFMTestMode : _physicalTuner %d", _physicalTuner));
   switch (_physicalTuner)
   {
      case T_e8_Tun_PhysicalTuner__TUNER_0:
      {
         bandOfPhysicalTuner = TMstatus.getU8Source();
         updateTun1QuadData(getBand(bandOfPhysicalTuner), freq, pi, psName, fs_q, hiCut, sharx, _phaseDiversity, TP, TA, RDS, afListLen, mp_nc, fmDev, cs);
      }
      break;
      case  T_e8_Tun_PhysicalTuner__TUNER_1:
      {
         bandOfPhysicalTuner = TMstatus.getU8Source();
         updateTun2QuadData(getBand(bandOfPhysicalTuner), freq, pi, psName, fs_q, hiCut, sharx, _phaseDiversity, TP, TA, RDS, afListLen, mp_nc, fmDev, cs);
      }
      break;
      case T_e8_Tun_PhysicalTuner__TUNER_2:
      {
         bandOfPhysicalTuner = TMstatus.getU8Source();
         updateTun3QuadData(getBand(bandOfPhysicalTuner), freq, pi, psName, fs_q, hiCut, sharx, _phaseDiversity, TP, TA, RDS, afListLen, mp_nc, fmDev, cs);
      }
      break;
      case  T_e8_Tun_PhysicalTuner__TUNER_3:
      {
         bandOfPhysicalTuner = TMstatus.getU8Source();
         //if ((TunerDataPoolConfig::getInstance()->readRegionType() == EUR_REGION))
         updateTun4QuadData(getBand(bandOfPhysicalTuner), freq, pi, psName, fs_q, hiCut, sharx, _phaseDiversity, TP, TA, RDS, afListLen, mp_nc, fmDev, cs);
      }
      break;
      default:
         break;
   }
   // }
}


/**
* Description     : Function to update STM screen Data.
*
* @param[in]      : The status message of 'FID_TUN_G_GET_DATA_TESTMODE_DDADDS'
* @return         : void
*/
void AMFMTestMode::sendStmScreenData(const boost::shared_ptr< FID_TUN_G_GET_DATA_TESTMODE_DDADDSStatus >& status)
{
   T_TunerTestModeData_DDADDS TMstatus = status->getTunerTestModeData();
   //TunerDataBindingHandler* _testModeDataBinding = TunerDataBindingHandler::getDataBindingHandlerInstance();

   if ((_physicalTuner == getActivePhysicalTuner()))
   {
      uint32 FREQ = TMstatus.getU32AFFrequency();
      std::string freq = TunerUtility::formatTestmodeFrequency(FREQ, getBand(TMstatus.getU8Source()));
      uint16 PI = TMstatus.getU16CurrentPI();
      std::string pi = TunerUtility::u16ConversionToHexString(PI);
      std::string psName = TMstatus.getSCurrentStationName();
      uint8 FS = TMstatus.getU8Fieldstrength();
      std::string fieldStrength = TunerUtility::u8ConversionToString(FS);
      uint8 QUALITY = TMstatus.getU8Quality();
      std::string fieldQuality = TunerUtility::u8ConversionToString(QUALITY);

      std::string fs_q = TunerUtility::quadConcatenate(FS, QUALITY);

      uint8 HC = TMstatus.getU8HiCut();
      uint8 HcExt = TMstatus.getU8HicutExtender();
      std::string hiCut = TunerUtility::quadConcatenate(HC, HcExt);
      uint8 SharxValue = TMstatus.getU8Sharx();
      std::string sharx = TunerUtility::u8ConversionToString(SharxValue);
      uint8 dev = TMstatus.getU8FMDeviation();
      std::string fmDev = TunerUtility::u8ConversionToString(dev);
      uint8 CS = TMstatus.getU8ChannelSeparation();
      std::string cs = TunerUtility::u8ConversionToString(CS);

      if (_band == T_e8_Tun_Testmode_TunerBand__TUN_BAND_AM)
      {
         updateEngineeringMenu(freq, psName, pi, fieldStrength, fieldQuality, hiCut, sharx, fmDev, cs);
      }
      if ((_band == T_e8_Tun_Testmode_TunerBand__TUN_BAND_FM) || (_band == T_e8_Tun_Testmode_TunerBand__TUN_BAND_FM_NON_RDS))
      {
         updateEngineeringMenu(freq, psName, pi, fieldStrength, fieldQuality, hiCut, sharx, fmDev, cs);
      }
   }
}


/**
* Description     : Function returns if the band is AM, FM or DAB.
*
* @param[in]      : band
* @return         : void
*/
std::string AMFMTestMode::getBand(uint8 band) const
{
   std::string sband = "";

   if ((band == T_e8_Tun_Testmode_TunerBand__TUN_BAND_FM) || (_band == T_e8_Tun_Testmode_TunerBand__TUN_BAND_FM_NON_RDS))
   {
      sband = "FM";
   }
   else if (band == T_e8_Tun_Testmode_TunerBand__TUN_BAND_AM)
   {
      sband = "AM";
   }
   else if (band == T_e8_Tun_Testmode_TunerBand__TUN_BAND_DAB)
   {
      sband = "DAB";
   }
   else
   {
      sband = " ";
   }
   return sband;
}


/**
* Description     : Function returns the audible tuner.
*
* @return         : uint8
*/
uint8 AMFMTestMode::getActivePhysicalTuner()
{
   return _audibleTuner;
}


void AMFMTestMode::onDTMScreenEntry(uint8 screenState)
{
   m_sceneStatus = screenState;
   sendTestModeONOFF(screenState);

   if (screenState == SCENE_ENTRY)
   {
      onScreenEntry();
   }
   else if (screenState == SCENE_EXIT)
   {
      onScreenExit();
   }
}


/**
* Description     : Function called to send TestMode ON/OFF to middleware.
*
* @param[in]      : screenState: The entry/exit of the screen.
* @return         : void
*/
void AMFMTestMode::sendTestModeONOFF(uint8 screenState)
{
   T_TunerFeature testMode;
   T_b32_Tun_Features feature;
   T_b32_Tun_Features feature_mask;
   feature.setTUN_FEATURE_TM_DATA_MODE((bool)screenState);
   feature_mask.setTUN_FEATURE_TM_DATA_MODE(true);
   testMode.setB32Features(feature);
   testMode.setB32MaskFeatures(feature_mask);

   if (_tunerFiProxy.get() && _tunerFiProxy->isAvailable())
   {
      //_tunerFiProxy->sendFID_TUN_S_SET_FEATUREStart(*this, testMode);
      ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_FEATUREStart(*this, testMode)));
   }
   ((_tunerMasterFiProxy.get()) && (_tunerMasterFiProxy->sendFID_TUNMSTR_S_TEST_MODEStart(*this, (bool)screenState)));

   bool screen = getscenestatus();
   ETG_TRACE_USR4(("AMFMTestMode  screen: %d", getscenestatus()));
   if (screen == 1)
   {
      //onTestmodeAvailable();
   }
   else
   {
      //onTestmodeUnAvailable();
   }
   bool tmOnOff = (screen == SCENE_ENTRY) ? true : false;
   ETG_TRACE_USR4(("sendTestModeONOFF screenState: %d", tmOnOff));
   ((_dabFiProxy.get()) && (_dabFiProxy->sendFID_DAB_S_SET_TESTMODEStart(*this, (bool)tmOnOff)));
}


//void AMFMTestMode::sendTestModeONOFFDAB()
//{
// bool screenState = getscenestatus();
// ETG_TRACE_USR4(("AMFMTestMode  screenstate: %d", getscenestatus()));
// bool tmOnOff = (screenState == SCENE_ENTRY) ? true : false;
// ETG_TRACE_USR4(("sendTestModeONOFF screenState: %d", tmOnOff));
// ((_dabFiProxy.get()) && (_dabFiProxy->sendFID_DAB_S_SET_TESTMODEStart(*this, tmOnOff)));


//}
void AMFMTestMode::onScreenEntry()
{
   T_TunerFeature testMode, AMFMtestMode;
   T_b32_Tun_Features feature, AMFMfeature;
   T_b32_Tun_Features feature_mask, AMFMfeature_mask;
   feature_mask.setTUN_FEATURE_PERSISTENTSTORAGE_MODE(true);
   feature.setTUN_FEATURE_PERSISTENTSTORAGE_MODE(false);
   testMode.setB32Features(feature);
   testMode.setB32MaskFeatures(feature_mask);
   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_FEATUREStart(*this, testMode)));
   AMFMfeature_mask.setTUN_FEATURE_AMFMSETUP_MODE(true);
   AMFMfeature.setTUN_FEATURE_AMFMSETUP_MODE(true);
   AMFMtestMode.setB32Features(AMFMfeature);
   AMFMtestMode.setB32MaskFeatures(AMFMfeature_mask);
   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_FEATUREStart(*this, AMFMtestMode)));
   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_TESTMODEStart(*this, T_e8_Tun_TestModeSource__TUN_TEST_MODE_SOURCE_HD, T_e8_Tun_TestModeActivity__TUN_TEST_MODE_NOTIFICATION_ON)));
   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_CALIBRATION_DATAStart(*this)));
}


void AMFMTestMode::onScreenExit()
{
   T_TunerFeature testMode, measuremode;
   T_b32_Tun_Features feature, measuremodefeature;
   T_b32_Tun_Features feature_mask, featuremask;
   feature_mask.setTUN_FEATURE_PERSISTENTSTORAGE_MODE(true);
   feature.setTUN_FEATURE_PERSISTENTSTORAGE_MODE(true);
   testMode.setB32Features(feature);
   testMode.setB32MaskFeatures(feature_mask);
   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_FEATUREStart(*this, testMode)));
   featuremask.setTUN_FEATURE_MEASURE_MODE(true);
   measuremodefeature.setTUN_FEATURE_MEASURE_MODE(false);
   measuremode.setB32Features(measuremodefeature);
   measuremode.setB32MaskFeatures(featuremask);
   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_FEATUREStart(*this, measuremode)));
   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_TESTMODEStart(*this, T_e8_Tun_TestModeSource__TUN_TEST_MODE_SOURCE_HD, T_e8_Tun_TestModeActivity__TUN_TEST_MODE_NOTIFICATION_OFF)));
   //((_dabFiProxy.get()) && (_dabFiProxy->sendFID_DAB_S_SET_TESTMODEStart(*this, false)));
}


bool AMFMTestMode::onCourierMessage(const FMMonitorMessage& corfoMsg)
{
   ETG_TRACE_USR4(("AMFMTestMode : onCourierMessage - FMMonitorMessage"));

   const FMMonitorMessage* copoFMMonitorMessage = Courier::message_cast<const FMMonitorMessage*>(&corfoMsg);

   if (NULL != copoFMMonitorMessage)
   {
      uint8 arguement1 = copoFMMonitorMessage->GetFirstArg();
      uint8 arguement2 = copoFMMonitorMessage->GetSecondArg();

      switch (arguement1)
      {
         case SHARX:
            vSetSharxValue(arguement2);
            break;

         case HI_CUT:
            vSetHiCutValue(arguement2);
            break;
         default:
            break;
      }
   }
   return true;
}


uint8 AMFMTestMode::getscenestatus()
{
   return m_sceneStatus;
}


/**
* Description     : Gives status message for FID_TUNMSTR_G_TA_STATUS property
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : error : actual status of the FID_TUNMSTR_G_TA_STATUS property
*/
void AMFMTestMode::onFID_TUNMSTR_G_ANNO_BUTTON_STATUSError(const ::boost::shared_ptr< MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUNMSTR_G_ANNO_BUTTON_STATUSError >& /*error*/)
{}

void AMFMTestMode::onFID_TUNMSTR_G_ANNO_BUTTON_STATUSStatus(const ::boost::shared_ptr< MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUNMSTR_G_ANNO_BUTTON_STATUSStatus >& status)
{
   bool bDataUpdated = false;
   _taActive = status->getB32AnnouncementType().getTUNMSTR_ANNOTYPE_TA();
   ETG_TRACE_USR4(("Tuner:onFID_TUNMSTR_G_TA_STATUSStatus _taActive : %d", _taActive));

   (*oFMSetupMenuInfo).mToggleTA_status = _taActive;
   oFMSetupMenuInfo.MarkAllItemsModified();
   bDataUpdated = oFMSetupMenuInfo.SendUpdate(true);
}


/**
* Description     : Function called by GUI, when user selects different tuners in Quad monitor screen.
*
* @param[in]      : tunerIndex : The index of the selected tuner.
* @return         : void
*/
void AMFMTestMode::onQuadMonitorSelect(uint8 tunerIndex)
{
   ETG_TRACE_USR4(("AMFMTestMode : onQuadMonitorSelect %d", tunerIndex));
   // TunerDataBindingHandler* _testModeDataBinding = TunerDataBindingHandler::getDataBindingHandlerInstance();

   switch (tunerIndex)
   {
      case TUN_TM_REQ_ACTIVE_TUNER_0:
      {
         ((_tunerMasterFiProxy.get()) && (_tunerMasterFiProxy->sendFID_TUNMSTR_S_SRCCHG_TESTMODEStart(*this, T_e8_WhichTuner__TUN_MSTR_TUNER_0)));
         updateQuadMonitorStatus(true, false, false, false,
                                 true, false, false, false,
                                 true, false, false, false,
                                 true, true, true, true);
         break;
      }
      case TUN_TM_REQ_ACTIVE_TUNER_1:
      {
         ((_tunerMasterFiProxy.get()) && (_tunerMasterFiProxy->sendFID_TUNMSTR_S_SRCCHG_TESTMODEStart(*this, T_e8_WhichTuner__TUN_MSTR_TUNER_1)));
         updateQuadMonitorStatus(false, true, false, false,
                                 false, true, false, false,
                                 false, true, false, false,
                                 true, true, true, true);
         break;
      }
      case TUN_TM_REQ_ACTIVE_TUNER_2:
      {
         ((_tunerMasterFiProxy.get()) && (_tunerMasterFiProxy->sendFID_TUNMSTR_S_SRCCHG_TESTMODEStart(*this, T_e8_WhichTuner__TUN_MSTR_TUNER_2)));
         updateQuadMonitorStatus(false, false, true, false,
                                 false, false, true, false,
                                 false, false, true, false,
                                 true, true, true, true);
         break;
      }
      case TUN_TM_REQ_ACTIVE_TUNER_3:
      {
         ((_tunerMasterFiProxy.get()) && (_tunerMasterFiProxy->sendFID_TUNMSTR_S_SRCCHG_TESTMODEStart(*this, T_e8_WhichTuner__TUN_MSTR_TUNER_3)));
         updateQuadMonitorStatus(false, false, false, true,
                                 false, false, false, true,
                                 false, false, false, true,
                                 true, true, true, true);
         break;
      }
      default:
         break;
   }
}


/**
* Description     : Function called to update Active Tuner Data in Quad monitor
*
* @param[in]      : activeBand: Active Physical Tuner
* @return         : void
*/
void AMFMTestMode::updateTMactiveTunerdata(uint8 activeBand)
{
   (*oTTTunerSource_Band).mTunerSource1 = false;
   (*oTTTunerSource_Band).mTunerSource2 = false;
   (*oTTTunerSource_Band).mTunerSource3 = false;
   (*oTTTunerSource_Band).mTunerSource4 = false;
   (*oTTTunerSource_Band).mBandEnable1 = false;
   (*oTTTunerSource_Band).mBandEnable2 = false;
   (*oTTTunerSource_Band).mBandEnable3 = false;
   (*oTTTunerSource_Band).mBandEnable4 = false;
   if (activeBand == 0)
   {
      (*oTTTunerSource_Band).mTunerSource1 = true;
      (*oTTTunerSource_Band).mBandEnable1 = true;
   }
   else if (activeBand == 1)
   {
      (*oTTTunerSource_Band).mTunerSource2 = true;
      (*oTTTunerSource_Band).mBandEnable2 = true;
   }
   else if (activeBand == 2)
   {
      (*oTTTunerSource_Band).mTunerSource3 = true;
      (*oTTTunerSource_Band).mBandEnable3 = true;
   }
   else if (activeBand == 3)
   {
      (*oTTTunerSource_Band).mTunerSource4 = true;
      (*oTTTunerSource_Band).mBandEnable4 = true;
   }
   oTTTunerSource_Band.MarkAllItemsModified();
   oTTTunerSource_Band.SendUpdate(true);
}


/**
* Description     : Function called to update Quad Monitor Column 1 Data
*
* @param[in]      : band: band in the corresponding physical tuner
* @param[in]      : freq: frequency
* @param[in]      : PI: PI code of the tuned frequency
* @param[in]      : psName: PS name of the tuned station
* @param[in]      : FS_Q: field strength and quality of the station
* @param[in]      : hCut: HiCut value
* @param[in]      : sharx: Sharx Value
* @param[in]      : PD: Phase diversity
* @param[in]      : TP: TP Station
* @param[in]      : TA: Traffic Announcement
* @param[in]      : RDS: RDS Error Rate
* @param[in]      : AFlength: AF List length
* @param[in]      : MP_NC: MultiPath and Neighbour Channel
* @param[in]      : deviation: Frequency deviation
* @param[in]      : CS: Channel Separation
* @return         : void
*/
void AMFMTestMode::updateTun1QuadData(std::string band, std::string freq, std::string PI, std::string psName, std::string FS_Q, std::string hCut, std::string sharx,
                                      std::string PD, std::string TP, std::string TA, std::string RDS, std::string AFlength, std::string MP_NC, std::string deviation, std::string CS)
{
   ETG_TRACE_USR3(("AMFMTestMode::updateTun1QuadData 1"));

   (*oTTMonitor1).mFreq = freq.c_str();
   (*oTTMonitor1).mPS = psName.c_str();
   (*oTTMonitor1).mPI = PI.c_str();
   (*oTTMonitor1).mFSQ = FS_Q.c_str();
   (*oTTMonitor1).mSharx = sharx.c_str();
   (*oTTMonitor1).mHicut = hCut.c_str();
   (*oTTMonitor1).mDev = deviation.c_str();
   (*oTTMonitor1).mCS = CS.c_str();
   (*oTTMonitor1).mRDS = RDS.c_str();
   (*oTTMonitor1).mTP = TP.c_str();
   (*oTTMonitor1).mTA = TA.c_str();
   (*oTTMonitor1).mAF = AFlength.c_str();
   (*oTTMonitor1).mMPNC = MP_NC.c_str();
   (*oTTMonitor1).mBand = band.c_str();

   oTTMonitor1.MarkAllItemsModified();
   oTTMonitor1.SendUpdate();
}


/**
* Description     : Function called to update Quad Monitor Column 2 Data
*
* @param[in]      : band: band in the corresponding physical tuner
* @param[in]      : freq: frequency
* @param[in]      : PI: PI code of the tuned frequency
* @param[in]      : psName: PS name of the tuned station
* @param[in]      : FS_Q: field strength and quality of the station
* @param[in]      : hCut: HiCut value
* @param[in]      : sharx: Sharx Value
* @param[in]      : PD: Phase diversity
* @param[in]      : TP: TP Station
* @param[in]      : TA: Traffic Announcement
* @param[in]      : RDS: RDS Error Rate
* @param[in]      : AFlength: AF List length
* @param[in]      : MP_NC: MultiPath and Neighbour Channel
* @param[in]      : deviation: Frequency deviation
* @param[in]      : CS: Channel Separation
* @return         : void
*/
void AMFMTestMode::updateTun2QuadData(std::string band, std::string freq, std::string PI, std::string psName, std::string FS_Q, std::string hCut, std::string sharx,
                                      std::string PD, std::string TP, std::string TA, std::string RDS, std::string AFlength, std::string MP_NC, std::string deviation, std::string CS)
{
   ETG_TRACE_USR3(("AMFMTestMode::updateTun1QuadData 2"));

   (*oTTMonitor2).mFreq = freq.c_str();
   (*oTTMonitor2).mPS = psName.c_str();
   (*oTTMonitor2).mPI = PI.c_str();
   (*oTTMonitor2).mFSQ = FS_Q.c_str();
   (*oTTMonitor2).mSharx = sharx.c_str();
   (*oTTMonitor2).mHicut = hCut.c_str();
   (*oTTMonitor2).mDev = deviation.c_str();
   (*oTTMonitor2).mCS = CS.c_str();
   (*oTTMonitor2).mRDS = RDS.c_str();
   (*oTTMonitor2).mTP = TP.c_str();
   (*oTTMonitor2).mTA = TA.c_str();
   (*oTTMonitor2).mAF = AFlength.c_str();
   (*oTTMonitor2).mMPNC = MP_NC.c_str();
   (*oTTMonitor2).mBand = band.c_str();

   oTTMonitor2.MarkAllItemsModified();
   oTTMonitor2.SendUpdate();
}


/**
* Description     : Function called to update Quad Monitor Column 3 Data
*
* @param[in]      : band: band in the corresponding physical tuner
* @param[in]      : freq: frequency
* @param[in]      : PI: PI code of the tuned frequency
* @param[in]      : psName: PS name of the tuned station
* @param[in]      : FS_Q: field strength and quality of the station
* @param[in]      : hCut: HiCut value
* @param[in]      : sharx: Sharx Value
* @param[in]      : PD: Phase diversity
* @param[in]      : TP: TP Station
* @param[in]      : TA: Traffic Announcement
* @param[in]      : RDS: RDS Error Rate
* @param[in]      : AFlength: AF List length
* @param[in]      : MP_NC: MultiPath and Neighbour Channel
* @param[in]      : deviation: Frequency deviation
* @param[in]      : CS: Channel Separation
* @return         : void
*/
void AMFMTestMode::updateTun3QuadData(std::string band, std::string freq, std::string PI, std::string psName, std::string FS_Q, std::string hCut, std::string sharx,
                                      std::string PD, std::string TP, std::string TA, std::string RDS, std::string AFlength, std::string MP_NC, std::string deviation, std::string CS)
{
   ETG_TRACE_USR3(("AMFMTestMode::updateTun1QuadData 3"));

   (*oTTMonitor3).mFreq = freq.c_str();
   (*oTTMonitor3).mPS = psName.c_str();
   (*oTTMonitor3).mPI = PI.c_str();
   (*oTTMonitor3).mFSQ = FS_Q.c_str();
   (*oTTMonitor3).mSharx = sharx.c_str();
   (*oTTMonitor3).mHicut = hCut.c_str();
   (*oTTMonitor3).mDev = deviation.c_str();
   (*oTTMonitor3).mCS = CS.c_str();
   (*oTTMonitor3).mRDS = RDS.c_str();
   (*oTTMonitor3).mTP = TP.c_str();
   (*oTTMonitor3).mTA = TA.c_str();
   (*oTTMonitor3).mAF = AFlength.c_str();
   (*oTTMonitor3).mMPNC = MP_NC.c_str();
   (*oTTMonitor3).mBand = band.c_str();

   oTTMonitor3.MarkAllItemsModified();
   oTTMonitor3.SendUpdate();
}


/**
* Description     : Function called to update Quad Monitor Column 4 Data
*
* @param[in]      : band: band in the corresponding physical tuner
* @param[in]      : freq: frequency
* @param[in]      : PI: PI code of the tuned frequency
* @param[in]      : psName: PS name of the tuned station
* @param[in]      : FS_Q: field strength and quality of the station
* @param[in]      : hCut: HiCut value
* @param[in]      : sharx: Sharx Value
* @param[in]      : PD: Phase diversity
* @param[in]      : TP: TP Station
* @param[in]      : TA: Traffic Announcement
* @param[in]      : RDS: RDS Error Rate
* @param[in]      : AFlength: AF List length
* @param[in]      : MP_NC: MultiPath and Neighbour Channel
* @param[in]      : deviation: Frequency deviation
* @param[in]      : CS: Channel Separation
* @return         : void
*/
void AMFMTestMode::updateTun4QuadData(std::string band, std::string freq, std::string PI, std::string psName, std::string FS_Q, std::string hCut, std::string sharx,
                                      std::string PD, std::string TP, std::string TA, std::string RDS, std::string AFlength, std::string MP_NC, std::string deviation, std::string CS)
{
   ETG_TRACE_USR3(("AMFMTestMode::updateTun1QuadData 4"));
   (*oTTMonitor4).mFreq = freq.c_str();
   (*oTTMonitor4).mPS = psName.c_str();
   (*oTTMonitor4).mPI = PI.c_str();
   (*oTTMonitor4).mFSQ = FS_Q.c_str();
   (*oTTMonitor4).mSharx = sharx.c_str();
   (*oTTMonitor4).mHicut = hCut.c_str();
   (*oTTMonitor4).mDev = deviation.c_str();
   (*oTTMonitor4).mCS = CS.c_str();
   (*oTTMonitor4).mRDS = RDS.c_str();
   (*oTTMonitor4).mTP = TP.c_str();
   (*oTTMonitor4).mTA = TA.c_str();
   (*oTTMonitor4).mAF = AFlength.c_str();
   (*oTTMonitor4).mMPNC = MP_NC.c_str();
   (*oTTMonitor4).mBand = band.c_str();

   oTTMonitor4.MarkAllItemsModified();
   oTTMonitor4.SendUpdate();
}


void AMFMTestMode::updateQuadMonitorStatus(bool tuner1Status, bool tuner2Status, bool tuner3Status, bool tuner4Status,
      bool band1Status, bool band2Status, bool band3Status, bool band4Status,
      bool isBand1Enabled, bool isBand2Enabled, bool isBand3Enabled, bool isBand4Enabled, bool isTuner1Enabled, bool isTuner2Enabled, bool isTuner3Enabled, bool isTuner4Enabled)
{
}


void AMFMTestMode::updateEngineeringMenu(std::string freq, std::string psName, std::string PI, std::string FS_Q, std::string Quality, std::string hCut, std::string sharx,
      std::string deviation, std::string CS)
{
   (*oEngineeringMenuData).m_szFrequency = freq.c_str();
   (*oEngineeringMenuData).m_szPSName = psName.c_str();
   (*oEngineeringMenuData).m_szPI = PI.c_str();
   (*oEngineeringMenuData).m_szFieldstrength = FS_Q.c_str();
   (*oEngineeringMenuData).m_szQuality = Quality.c_str();
   (*oEngineeringMenuData).m_szSharx = sharx.c_str();
   (*oEngineeringMenuData).m_szHicut = hCut.c_str();
   (*oEngineeringMenuData).m_szFMDeviation = deviation.c_str();
   (*oEngineeringMenuData).m_szChannelSeparation = CS.c_str();

   oEngineeringMenuData.MarkAllItemsModified();
   oEngineeringMenuData.SendUpdate();
}


/**
* Description     : Function called on press of onSelectMeasureMode option in AM FM Settings screen.
*
* @return         : void
*/
void AMFMTestMode::onSelectMeasureMode()
{
   T_TunerFeature testMode;
   T_b32_Tun_Features feature;
   T_b32_Tun_Features feature_mask;

   feature_mask.setTUN_FEATURE_MEASURE_MODE(true);
   feature.setTUN_FEATURE_MEASURE_MODE(!_measureIndex);
   testMode.setB32Features(feature);
   testMode.setB32MaskFeatures(feature_mask);
   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_FEATUREStart(*this, testMode)));
}


/**
* Description     : Function called on press of AF Following option in AM FM Settings screen.
*
* @return         : void
*/
void AMFMTestMode::onSelectAFFollowing()
{
   ETG_TRACE_USR3(("onSelectAFFollowing"));
   T_TunerFeature testMode;
   T_b32_Tun_Features feature;
   T_b32_Tun_Features feature_mask;
   bool AF = false;
   if (_tunerFiProxy.get())
   {
      AF = _tunerFiProxy->getFID_TUN_G_SET_AF().getU8AF();
   }
   feature_mask.setTUN_FEATURE_AF_MODE(true);
   feature.setTUN_FEATURE_AF_MODE(!AF);
   testMode.setB32Features(feature);
   testMode.setB32MaskFeatures(feature_mask);
   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_FEATUREStart(*this, testMode)));
}


/**
* Description     : Function called on press of RDS REG option in AM FM Settings screen.
*
* @return         : void
*/
void AMFMTestMode::onSelectRDSREG()
{
   T_TunerFeature testMode;
   T_b32_Tun_Features feature;
   T_b32_Tun_Features feature_mask;
   bool bRDS = false;
   if (_tunerFiProxy.get())
   {
      bRDS = _tunerFiProxy->getFID_TUN_G_SET_RDSREG().getU8RDSreg();
   }
   feature_mask.setTUN_FEATURE_REG_AUTO_MODE(true);
   feature.setTUN_FEATURE_REG_AUTO_MODE(!bRDS);
   testMode.setB32Features(feature);
   testMode.setB32MaskFeatures(feature_mask);
   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_FEATUREStart(*this, testMode)));
}


void AMFMTestMode::onToggleTA()
{
   setTAflag(!_taActive);
}


void AMFMTestMode::setTAflag(bool tastatus)
{
   ETG_TRACE_USR3(("AnnoHandler setTAflag is %d ", tastatus));
   if (_taActive != tastatus)
   {
      T_b8_SourceAnno source;
      T_b32_AnnoType anno;

      anno.setTUNMSTR_ANNOTYPE_TA(tastatus);
      anno.setTUNMSTR_ANNOTYPE_PTY(true);
      source.setTUNMSTR_ANNOSRC_FM(true);
      source.setTUNMSTR_ANNOSRC_DAB(true);

      ((_tunerMasterFiProxy.get())
       && (_tunerMasterFiProxy->sendFID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENTStart(
              *this, anno, source)));
   }
}


void AMFMTestMode::readKdsForRegion()
{
   dp_tclKdsVehicleInformation _KdsVehicleInformationInst;
   _KdsVehicleInformationInst.u8GetDestinationRegion1(_regionType);
}


//added from dab

void AMFMTestMode::onFID_DAB_G_TUNER_STATUSError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::FID_DAB_G_TUNER_STATUSError >& /*error*/)
{
}


/**
* Description     : Gives status message for FID_DAB_G_MTC_TUNER_STATUS property
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : status : actual status of the FID_DAB_G_MTC_TUNER_STATUS property
*/
void AMFMTestMode::onFID_DAB_G_TUNER_STATUSStatus(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::FID_DAB_G_TUNER_STATUSStatus >& status)
{
   const ::midw_ext_dabtuner_fi_types::T_DAB_TunerStatus& tunerStatus = status->getTunerStatus();
   _audioQuality = TunerUtility::u8ConversionToString(tunerStatus.getU8AudioQuality());
   _signalQuality = TunerUtility::u8ConversionToString(tunerStatus.getU8SignalQuality());
   _DTMFieldMap[AudioQuality] = _audioQuality;
   _DTMFieldMap[SignalQuality] = _signalQuality;
   _ensembleChannel = tunerStatus.getFreqTuned().getSFreqLabel();
   _DTMFieldMap[ChannelNo] = _ensembleChannel;
   // ETG_TRACE_USR3(("AMFMTestMode::mFreq_Table: %s", tunerStatus.getSFrequencyTable().c_str()));
   // ETG_TRACE_USR3(("AMFMTestMode::mEnsemble_freq: %s", tunerStatus.getFreqTuned().getSFreqLabel().c_str()));
   (*oDABTestModePointer).mFreq_Table = tunerStatus.getSFrequencyTable().c_str();
   (*oDABTestModePointer).mchannel_Num = _ensembleChannel.c_str();
   (*oDABTestModePointer).mQA = (Candera::String)_audioQuality.c_str();
   (*oDABTestModePointer).mQS = (Candera::String)_signalQuality.c_str();
   (*oDABTestModePointer).mEnsemble_freq = (Candera::String)_ensembleChannel.c_str();

   oDABTestModePointer.MarkAllItemsModified();
   oDABTestModePointer.SendUpdate(true);
}


/**
* Description     : Gives error message for FID_DAB_G_ACTIVE_SERVICE_INFO property
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : error : report error
*/
void AMFMTestMode::onFID_DAB_G_ACTIVE_SERVICE_INFOError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::FID_DAB_G_ACTIVE_SERVICE_INFOError >& /*error*/)
{
}


/**
* Description     : Gives status message for FID_DAB_G_ACTIVE_SERVICE_INFO property
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : status : actual status of the FID_DAB_G_ACTIVE_SERVICE_INFO property
*/
void AMFMTestMode::onFID_DAB_G_ACTIVE_SERVICE_INFOStatus(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::FID_DAB_G_ACTIVE_SERVICE_INFOStatus >& status)
{
   ETG_TRACE_USR3(("AMFMTestMode::onFID_DAB_G_ACTIVE_SERVICE_INFOStatus"));
   const ::midw_ext_dabtuner_fi_types::T_DAB_ServiceInfo& serviceInfo = status->getServiceInfo();
   ::midw_ext_dabtuner_fi_types::T_b8_DAB_ServiceInfoFlags flag;
   _audioBitrate = TunerUtility::u16ConversionToString(serviceInfo.getU16BitRate());
   _audioSamplingRate = TunerUtility::u8ConversionToString(serviceInfo.getU8SampleRate());
   _audioMode = serviceInfo.getSAudioMode();
   _DTMFieldMap[BitRate] = _audioBitrate;
   _DTMFieldMap[SamplingRate] = _audioSamplingRate;
   _DTMFieldMap[ShortLabel] = serviceInfo.getServiceLabel().getSLabelShort();
   _DTMFieldMap[ACCFlag] = _audioMode + "/" + serviceInfo.getSAudioCodec();
   _DRCStatus = flag.getDAB_SERVICEINFO_DRC();
   _DTMFieldMap[NoOfComponent] = TunerUtility::u8ConversionToString(serviceInfo.getU8NumberOfAudioComponents()) + "/" + TunerUtility::u8ConversionToString(serviceInfo.getU8NumberOfDataComponents());

   (*oDABTestModePointer).mAudio_Mode = (Candera::String)_audioMode.c_str();
   (*oDABTestModePointer).mService_Lbl = (Candera::String)(serviceInfo.getServiceLabel().getSLabelShort()).c_str();
   (*oDABTestModePointer).mAud_Data_Bit_Rate = (Candera::String)_audioBitrate.c_str();

   (*oDABTestModeActivertunerInfo).mDRC = flag.getDAB_SERVICEINFO_DRC();
   (*oDABTestModeActivertunerInfo).mSM_ACC = (Candera::String)/* _audioMode + "/" +  */serviceInfo.getSAudioCodec().c_str();
   (*oDABTestModeActivertunerInfo).mSR = (Candera::String)_audioSamplingRate.c_str();
   //(*oDABTestModeActivertunerInfo).mTP= szData;
   ETG_TRACE_USR4(("Tuner:_audioBitrate _audioBitrate : %s", _audioBitrate.c_str()));
   ETG_TRACE_USR4(("Tuner:_audioBitrate mService_Lbl : %s", serviceInfo.getServiceLabel().getSLabelShort().c_str()));
   oDABTestModePointer.MarkAllItemsModified();
   oDABTestModeActivertunerInfo.MarkAllItemsModified();

   oDABTestModePointer.SendUpdate(true);
   oDABTestModeActivertunerInfo.SendUpdate(true);
}


/**
* Description     : Gives error message for FID_DAB_G_ACTIVE_ENSEMBLE_INFO property
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : error : report error
*/
void AMFMTestMode::onFID_DAB_G_ACTIVE_ENSEMBLE_INFOError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::FID_DAB_G_ACTIVE_ENSEMBLE_INFOError >& /*error*/)
{
}


/**
* Description     : Gives status message for FID_DAB_G_ACTIVE_ENSEMBLE_INFO property
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : status : actual status of the FID_DAB_G_ACTIVE_ENSEMBLE_INFO property
*/
void AMFMTestMode::onFID_DAB_G_ACTIVE_ENSEMBLE_INFOStatus(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::FID_DAB_G_ACTIVE_ENSEMBLE_INFOStatus >& status)
{
   ETG_TRACE_USR3(("AMFMTestMode::onFID_DAB_G_ACTIVE_ENSEMBLE_INFOStatus"));
   const ::midw_ext_dabtuner_fi_types::T_DAB_EnsembleInfo& ensembleInfo = status->getEnsembleInfo();
   _ensembleName = ensembleInfo.getEnsembleLabel().getSLabel();
   _ensembleID = TunerUtility::u32ConversionToHexString(ensembleInfo.getU32EnsembleId());
   _DTMFieldMap[EnsembleLabel] = _ensembleName;
   _DTMFieldMap[EnsembleID] = _ensembleID;
   _DTMFieldMap[NoOFEnsebles] = TunerUtility::u8ConversionToString(ensembleInfo.getU8NumberOfEnsembles());
   _DTMFieldMap[NoOfServices] = TunerUtility::u8ConversionToString(ensembleInfo.getU8NumberOfAudioServices()) + "/" + TunerUtility::u8ConversionToString(ensembleInfo.getU8NumberOfDataServices());

   (*oDABTestModeActiveEnsembleInfo).mNoOfAud_DataServiceComp = TunerUtility::u8ConversionToString(ensembleInfo.getU8NumberOfDataServices()).c_str();
   (*oDABTestModeActiveEnsembleInfo).mNoOfAud_DataService = TunerUtility::u8ConversionToString(ensembleInfo.getU8NumberOfAudioServices()).c_str();
   (*oDABTestModeActiveEnsembleInfo).mNoOfEnsembles = TunerUtility::u8ConversionToString(ensembleInfo.getU8NumberOfEnsembles()).c_str();
   (*oDABTestModePointer).mEnsemble_ID = (Candera::String) _ensembleID.c_str();

   oDABTestModePointer.MarkAllItemsModified();
   oDABTestModeActiveEnsembleInfo.MarkAllItemsModified();
   oDABTestModePointer.SendUpdate(true);
   oDABTestModeActiveEnsembleInfo.SendUpdate(true);
}


/**
* Description     : Gives error message for FID_DAB_G_ACTIVE_COMPONENT_INFO property
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : error : report error
*/
void AMFMTestMode::onFID_DAB_G_ACTIVE_COMPONENT_INFOError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::FID_DAB_G_ACTIVE_COMPONENT_INFOError >& /*error*/)
{
}


/**
* Description     : Gives status message for FID_DAB_G_ACTIVE_COMPONENT_INFO property
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : status : actual status of the FID_DAB_G_ACTIVE_COMPONENT_INFO property
*/
void AMFMTestMode::onFID_DAB_G_ACTIVE_COMPONENT_INFOStatus(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::FID_DAB_G_ACTIVE_COMPONENT_INFOStatus >& status)
{
   const ::midw_ext_dabtuner_fi_types::T_DAB_ComponentInfo& componentInfo = status->getComponentInfo();
   midw_ext_dabtuner_fi_types::T_e8_DAB_ComponentType cType = componentInfo.getE8ComponentType();
   midw_ext_dabtuner_fi_types::T_e8_DAB_TMIdType tmIdType = componentInfo.getE8TMIdType();
   _DTMFieldMap[SCType] = TunerUtility::u16ConversionToString(componentInfo.getU16SCIDI());
   _DTMFieldMap[TMID] = TunerUtility::u8ConversionToString(tmIdType);
   _DTMFieldMap[SC] = TunerUtility::u8ConversionToString(componentInfo.getU8AudioDataServiceComponentType());

   (*oDABTestModeActiveCompInfo).mSC = TunerUtility::u8ConversionToString(componentInfo.getU8AudioDataServiceComponentType()).c_str();
   (*oDABTestModeActiveCompInfo).mSCType = TunerUtility::u16ConversionToString(componentInfo.getU16SCIDI()).c_str();

   oDABTestModeActiveCompInfo.MarkAllItemsModified();
   oDABTestModeActiveCompInfo.SendUpdate(true);
}


/**
* Description     : Gives error message for FID_DAB_S_GET_EXP_API_INFO method start
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : error : report error
*/
void AMFMTestMode::onFID_DAB_S_GET_EXP_API_INFOError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::FID_DAB_S_GET_EXP_API_INFOError >& /*error*/)
{
}


/**
* Description     : Gives result for FID_DAB_S_GET_EXP_API_INFO method start
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : result : result of the FID_DAB_S_GET_EXP_API_INFO method start
*/
void AMFMTestMode::onFID_DAB_S_GET_EXP_API_INFOResult(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::FID_DAB_S_GET_EXP_API_INFOResult >& result)
{
   DabExpertAPIInfo lAPIinfo;
   uint16 expertID = result->getU16ExpApiId();
   lAPIinfo._expertID = TunerUtility::u16ConversionToHexString(expertID);
   lAPIinfo._expertIDLabel = result->getSLabel();
   lAPIinfo._expertIDValue = result->getSValue();
   uint8 u8Id = (_expertIDIndex - EXPERTID_INDEX_1);

   if (u8Id < _DabExpertAPIInfoList.size())
   {
      _DabExpertAPIInfoList[u8Id] = lAPIinfo;
   }
}


/**
* Description     : Gives error message for FID_DAB_S_DB_QUERY method start
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : error : report error
*/
void AMFMTestMode::onFID_DAB_S_DB_QUERYError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::FID_DAB_S_DB_QUERYError >& /*error*/)
{
}


/**
* Description     : Gives result for FID_DAB_S_DB_QUERY method start
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : result : result of the FID_DAB_S_DB_QUERY method start
*/
void AMFMTestMode::onFID_DAB_S_DB_QUERYResult(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FI::FID_DAB_S_DB_QUERYResult >& result)
{
   /*    if (_testModeDataBinding != NULL)
   {
   _testModeDataBinding->setDabQuery(result->getSDbQuery());
   } */
}


void AMFMTestMode::onFID_DAB_S_SETUPError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_DAB_S_SETUPError >& /*error*/)
{
}


void AMFMTestMode::onFID_DAB_S_SETUPResult(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_DAB_S_SETUPResult >& /*result*/)
{
}


/**
* Description     : Gives error message for FID_TUN_S_SET_TESTMODE method start
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : error : report error
*/
void AMFMTestMode::onFID_TUN_S_SET_TESTMODEError(const ::boost::shared_ptr< MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_TUNER_FI::FID_TUN_S_SET_TESTMODEError >& /*error*/)
{
}


/**
* Description     : Gives resultfor FID_TUN_S_SET_TESTMODE method start
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : result : result of the FID_TUN_S_SET_TESTMODE method start
*/
void AMFMTestMode::onFID_TUN_S_SET_TESTMODEResult(const ::boost::shared_ptr< MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< MIDW_TUNER_FI::FID_TUN_S_SET_TESTMODEResult >& /*result*/)
{
}


/**
* Description     : Function called to update experd API ID
*/
void AMFMTestMode::updateDabExpertID()
{
}


/**
* Description     : Function called to change DAB DTM DATABASE screen
*/
void AMFMTestMode::changeDabDBScene()
{
}


::std::string AMFMTestMode::getLinkingMode(midw_ext_dabtuner_fi_types::T_e8_DAB_ServiceLinkingMode mode)
{
   ::std::string modeStr = "";
   switch (mode)
   {
      case T_e8_DAB_ServiceLinkingMode__DAB_SERVICELINKING_OFF:
         modeStr = "OFF";
         break;
      case T_e8_DAB_ServiceLinkingMode__DAB_SERVICELINKING_DAB:
         modeStr = "DAB";
         break;
      case T_e8_DAB_ServiceLinkingMode__DAB_SERVICELINKING_DAB_FM:
         modeStr = "DAB/FM";
         break;
      case T_e8_DAB_ServiceLinkingMode__DAB_SERVICELINKING_FM:
         modeStr = "FM";
         break;
      default:
         modeStr = "";
         break;
   }
   return modeStr;
}


void AMFMTestMode::toggleLinkingMode()
{
   T_b32_DAB_SetupMask setupMask;
   T_b32_DAB_SetupFlags setupFlags;
   T_b8_DAB_LinkingSetupMask b8LinkingSetupMask;
   uint8 index = _linkingMode;
   T_e8_DAB_ServiceLinkingMode modeArray[] = { T_e8_DAB_ServiceLinkingMode__DAB_SERVICELINKING_OFF, T_e8_DAB_ServiceLinkingMode__DAB_SERVICELINKING_DAB,
                                               T_e8_DAB_ServiceLinkingMode__DAB_SERVICELINKING_DAB_FM, T_e8_DAB_ServiceLinkingMode__DAB_SERVICELINKING_FM
                                             };
   index = (index + 1) % 4;            //to toggle the index value between 0 to 3(for linking mode)
   _linkingMode = modeArray[index];    //assigns the value of the array at index to the linking mode based on the previous mode

   TunerDataBindingHandler::getDataBindingHandlerInstance()->UpdateDABServiceLinkingMsg(index);
   ((_dabFiProxy.get()) && (_dabFiProxy->sendFID_DAB_S_SETUPStart(*this, setupMask, setupFlags, 0, 0, true, _linkingMode, 0, 0, 0, 0, 0, 0, 0, 0, 0, b8LinkingSetupMask, 0, 0, 0, 0, false)));
}


void AMFMTestMode::updateConcealmentValue(uint8& param, uint8 operation)
{
   tChar szData[] = { 0, 0, 0, };

   T_b32_DAB_SetupMask setupMask;
   T_b32_DAB_SetupFlags setupFlags;
   T_b8_DAB_LinkingSetupMask b8LinkingSetupMask;
   if (operation == TUN_DIRECTION_UP)
   {
      if (param < CONCEAL_MAX)
      {
         param += 1;
      }
   }
   else if (operation == TUN_DIRECTION_DOWN)
   {
      if (param > CONCEAL_MIN)
      {
         param -= 1;
      }
   }
   ETG_TRACE_USR3(("AMFMTestMode::updateConcealmentValue(%d)", _concealmentLevel));
   sprintf(szData, "%d", _concealmentLevel);
   (*oDABSetupMonitor).mConcealmentLevel = szData;
   oDABSetupMonitor.MarkItemModified(ItemKey::DAB_Setup_Screen::ConcealmentLevelItem);
   oDABSetupMonitor.SendUpdate();
   ((_dabFiProxy.get()) && (_dabFiProxy->sendFID_DAB_S_SETUPStart(*this, setupMask, setupFlags, true, _concealmentLevel, true, _linkingMode, 0, 0, 0, 0, 0, 0, 0, 0, 0, b8LinkingSetupMask, 0, 0, 0, 0, false)));
}


void AMFMTestMode::onSelectConcealmentValue(uint8 colId)
{
}


void AMFMTestMode::onPressDabSettingList(uint8 rowId, uint8 colId)
{
   switch (rowId)
   {
      case 0:
         toggleLinkingMode();
         break;
      case 1:
         onSelectConcealmentValue(colId);
         break;
      default:
         break;
   }
}


/**
* Description     : Function called to set active view name
* @param[in]      : lActiveViewID : View ID
*/
void AMFMTestMode::setActiveView(uint32 lActiveViewID)
{
}


/**
* Description     : Function implemented to calculate combined announcement type bitmask of the current service.
*
* @param[in]      : maskVal : uint16 value - to get mask of each individual type.
* @param[in]      : finalMask : uint16 value - to get the combined mask value.
* @param[in]      : type : bool value.
* @return         : void
*/

void AMFMTestMode::calcMask(uint16& maskVal, uint16& finalMask, bool type)
{
   if (type)
   {
      finalMask = finalMask | maskVal;          // combining the mask obtained till now with the current bitmask of the service, if it is supported.
   }
   maskVal = maskVal << 1;                      // traversing through the bits for each announcement type.
}


/**
* Description     : Function implemented to get the calculated bitmask of all announcement types into one.
*
* @param[in]      : type : T_b16_DAB_AnnouncementType value.
* @return         : std::string
*/

std::string AMFMTestMode::getBitMask(::midw_ext_dabtuner_fi_types::T_b16_DAB_AnnouncementType type)
{
   uint16 maskVal = 1;
   uint16 finalMask = 0;
   char stringToConv[ARRAY_SIZE] = "\0";

   calcMask(maskVal, finalMask, type.getDAB_ANNOTYPE_ALARM());
   calcMask(maskVal, finalMask, type.getDAB_ANNOTYPE_TA());
   calcMask(maskVal, finalMask, type.getDAB_ANNOTYPE_TRANSPORT());
   calcMask(maskVal, finalMask, type.getDAB_ANNOTYPE_WARNING());
   calcMask(maskVal, finalMask, type.getDAB_ANNOTYPE_NEWS());
   calcMask(maskVal, finalMask, type.getDAB_ANNOTYPE_WEATHER());
   calcMask(maskVal, finalMask, type.getDAB_ANNOTYPE_EVENT());
   calcMask(maskVal, finalMask, type.getDAB_ANNOTYPE_SPECIALEVENT());
   calcMask(maskVal, finalMask, type.getDAB_ANNOTYPE_PROGRAMME());
   calcMask(maskVal, finalMask, type.getDAB_ANNOTYPE_SPORTS());
   calcMask(maskVal, finalMask, type.getDAB_ANNOTYPE_FINANCE());
   calcMask(maskVal, finalMask, type.getDAB_ANNOTYPE_b11_reserved());
   calcMask(maskVal, finalMask, type.getDAB_ANNOTYPE_b12_reserved());
   calcMask(maskVal, finalMask, type.getDAB_ANNOTYPE_b13_reserved());
   calcMask(maskVal, finalMask, type.getDAB_ANNOTYPE_b14_reserved());
   calcMask(maskVal, finalMask, type.getDAB_ANNOTYPE_b15_reserved());

   snprintf(stringToConv, sizeof stringToConv, "%04x", finalMask);            // to convert the bitmask into hex format

   return stringToConv;
}


/**
* Description     : Function implemented to set the channelinfodata from testmode.
*
* @param[in]      : status : boost::shared_ptr< ::MIDW_EXT_DABTUNER_FI::FID_DAB_G_CHANNEL_INFOStatus > value.
* @return         : void
*/

void AMFMTestMode::setChannelInfoDataToTestMode(const boost::shared_ptr< ::MIDW_EXT_DABTUNER_FI::FID_DAB_G_CHANNEL_INFOStatus >& status)
{
   bool bDataUpdated = false;
   const ::midw_ext_dabtuner_fi_types::T_DAB_ChannelInfo& channelInfo = status->getChannelInfo();
   // T_b16_DAB_AnnouncementType mask = channelInfo.getB16AvailableAnnoTypesMask();
   _serviceID = TunerUtility::u32ConversionToHexString(channelInfo.getU32ServId());
   _serviceName = channelInfo.getServiceLabel().getSLabel();
   _DTMFieldMap[ServiceID] = _serviceID;
   _DTMFieldMap[ServiceLabel] = _serviceName;
   _DTMFieldMap[MuteFlag] = TunerUtility::u8ConversionToString(channelInfo.getBMuted());
   _DTMFieldMap[PSFlag] = channelInfo.getBComponentMode() ? "S" : "P";
   //_DTMFieldMap[ASU] = getBitMask(channelInfo.getB16AvailableAnnoTypesMask());

   _DTMFieldMap[TA_Source] = (AnnoHandler::poGetInstance()->getTASource() == T_e8_Source__TUN_MSTR_SOURCE_DAB) ? "DAB" : "FM";

   (*oDABTestModePointer).mService_ID = _serviceID.c_str();
   (*oDABTestModePointer).mMute_flag = TunerUtility::u8ConversionToString(channelInfo.getBMuted()).c_str();
   (*oDABTestModePointer).mTASrc = (AnnoHandler::poGetInstance()->getTASource() == T_e8_Source__TUN_MSTR_SOURCE_DAB) ? "DAB" : "FM";
   // (*oDABTestModeAnnoStatusInfo).mASU = getBitMask(channelInfo.getB16AvailableAnnoTypesMask()).c_str();

   oDABTestModeAnnoStatusInfo.MarkAllItemsModified();
   oDABTestModePointer.MarkAllItemsModified();
   bDataUpdated = oDABTestModeAnnoStatusInfo.SendUpdate(true);
   oDABTestModePointer.SendUpdate(true);

   //((_dabFiProxy.get()) && (_dabFiProxy->sendFID_DAB_G_ANNO_STATUSGet(*this)));
}


//wst4kor added for HD*************************************************************************************************************

void AMFMTestMode::onFID_TUN_G_HD_TEST_MODE_DATAError(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_HD_TEST_MODE_DATAError >& /*error*/)
{
}


void AMFMTestMode::onFID_TUN_G_HD_TEST_MODE_DATAStatus(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_HD_TEST_MODE_DATAStatus >& status)
{
   T_TunerHDTestModeData HDTestModeData = status->getTunerHDTestModeData();
   ETG_TRACE_USR4(("onFID_TUN_G_HD_TEST_MODE_DATAStatus"));
   _callSign = HDTestModeData.getSHDCallSign();
   _stationId = TunerUtility::u32ConversionToString(HDTestModeData.getU32HDStationId());
   _cd_No = TunerUtility::u8ConversionToString(HDTestModeData.getU8CD_NO());
   if (NULL != TunerDataBindingHandler::getDataBindingHandlerInstance()
      )
   {
      TunerDataBindingHandler::getDataBindingHandlerInstance()
      ->updateDTMHdTransitionCount(TunerUtility::u8ConversionToString(HDTestModeData.getU8TransitionCount()));
   }
   bool dtmHdStatus = false;
   if (HDTestModeData.getB8HDStationInfoBits().getRESERVED_2() && (!HDTestModeData.getB8HDStationInfoBits().getRESERVED_1()))
   {
      _primaryHdTuner = HDTestModeData.getE8PhysicalTuner();//Listening HD Tuner(HD 1.0)
      if (HDTestModeData.getB8HDAvailableAudioPrograms() != _audioPrgmList)
      {
         _audioPrgmList = HDTestModeData.getB8HDAvailableAudioPrograms();
         dtmHdStatus = true;
      }
      if (HDTestModeData.getE8HDActiveAudioProgram() != _activeHDAudioPrgm)
      {
         _activeHDAudioPrgm = HDTestModeData.getE8HDActiveAudioProgram();
         dtmHdStatus = true;
      }
      uint8 audioQuality = HDTestModeData.getU8AudioQuality();
      if (TunerDataBindingHandler::getDataBindingHandlerInstance()
            != NULL)
      {
         TunerDataBindingHandler::getDataBindingHandlerInstance()
         ->updateDTMHdInfo(_callSign, _cd_No, TunerUtility::u8ConversionToString(audioQuality));
      }
   }
   if ((HDTestModeData.getB8HDStationInfoBits().getRESERVED_2() && HDTestModeData.getB8HDStationInfoBits().getRESERVED_1()) && TunerDataBindingHandler::getDataBindingHandlerInstance()
         != NULL)
   {
      _secondaryHdTuner = HDTestModeData.getE8PhysicalTuner(); // HD Tuner which is doing background scan(HD 0.5)
      TunerDataBindingHandler::getDataBindingHandlerInstance()
      ->updateDTMHdAudibleTunerCdNo(_cd_No);
   }
   if (dtmHdStatus && _dtmHDScreenStatus)
   {
      getTMRadioHDAudioPrgmList();
      /* uint32 listid = LIST_ID_DTM_HD_MONITOR;
       updateList(listid);*/
   }
}


void AMFMTestMode::setSTMData(const ::boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_ATSTATIONINFOStatus >& status)
{
   ETG_TRACE_USR4(("AMFMTestMode::setSTMData _dtmHDScreenStatus:%d", _dtmHDScreenStatus));
   T_b8_Tun_HDInfoBits hdInfo = status->getAtstationinfo().getB8HDStationInfoBits();
   uint8 fieldStrength = status->getAtstationinfo().getU8FieldStrngth();
   std::string fs = "";
   fs = TunerUtility::u8ConversionToString(fieldStrength);
   fs = fs.append(" dBmV");
   if (_hdActiveStatus != hdInfo.getHD_MODULATION_DETECTION())
   {
      _hdActiveStatus = hdInfo.getHD_MODULATION_DETECTION();
      if (!_hdActiveStatus && _dtmHDScreenStatus)
      {
         getTMRadioHDAudioPrgmList();
         /* uint32 listid = LIST_ID_DTM_HD_MONITOR;
              updateList(listid);*/
      }
   }
   updateHdActiveStatus(_hdActiveStatus);
   updateHdAudioStatus(hdInfo.getHD_AUDIO_DECODING());
   updateBlendStatus(hdInfo.getHD_BLEND_ANALOG());
   updateSisStatus(hdInfo.getHD_DATA_DECODING());

   if ((_band == T_e8_Tun_Testmode_TunerBand__TUN_BAND_FM) || (_band == T_e8_Tun_Testmode_TunerBand__TUN_BAND_FM_NON_RDS) || _band == T_e8_Tun_Testmode_TunerBand__TUN_BAND_AM)
   {
      TunerDataBindingHandler::getDataBindingHandlerInstance()->updateHdDTMScreenInfo(_dtmHdStatus, _dtmSis, _dtmHdAudio, _dtmBlendStatus);
   }
}


void AMFMTestMode::updateHdActiveStatus(bool hdActive)
{
   if (hdActive == true)
   {
      _isHdActive = "Active";
      _dtmHdStatus = "ON";
   }
   else
   {
      _isHdActive = "InActive";
      _dtmHdStatus = "OFF";
   }
}


void AMFMTestMode::updateHdAudioStatus(bool hdAudio)
{
   if (hdAudio == true)
   {
      _hdAudio = "Yes";
      _dtmHdAudio = "ON";
   }
   else
   {
      _hdAudio = "No";
      _dtmHdAudio = "OFF";
   }
}


void AMFMTestMode::updateBlendStatus(bool blendStatus)
{
   if (blendStatus == true)
   {
      _blendStatus = "Digital";
      _dtmBlendStatus = "ON";
   }
   else
   {
      _blendStatus = "Analog";
      _dtmBlendStatus = "OFF";
   }
}


void AMFMTestMode::updateSisStatus(bool sis)
{
   if (sis == true)
   {
      _dtmSis = "ON";
   }
   else
   {
      _dtmSis = "OFF";
   }
}


/**
* Description     : Function called by GUI, when user enters DTM HD Monitor screen.
*
* @param[in]      : screenState: The entry/exit of the screen.
* @return         : void
*/
void AMFMTestMode::onDtmHdMonitorScreenEntry(uint8 screenState)
{
   ETG_TRACE_USR4(("AMFMTestMode::onDtmHdMonitorScreenEntry screenState:%d", screenState));
   if (screenState == SCENE_ENTRY)
   {
      _dtmHDScreenStatus = true;
   }
   else if (screenState == SCENE_EXIT)
   {
      _dtmHDScreenStatus = false;
      if (_splitModeStatus == T_e8_Tun_AudioSplitMode__TUN_ACTIVATE_AUDIO_SPLIT_MODE)
      {
         _splitModeStatus = T_e8_Tun_AudioSplitMode__TUN_DEACTIVATE_AUDIO_SPLIT_MODE;
         ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_AUDIO_SPLIT_MODEStart(*this, _splitModeStatus)));
      }
   }
   if (_dtmHDScreenStatus == true)
   {
      getTMRadioHDAudioPrgmList();
   }
}


/**
* Description     : Virtual function implemented to set the result of method start SELECT HD AUDIO SPLIT MODE
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : result : The result message of the method "SELECT HD AUDIO SPLIT MODE"
* @return         : void
*/
void AMFMTestMode::onFID_TUN_S_SET_AUDIO_SPLIT_MODEResult(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_SET_AUDIO_SPLIT_MODEResult >& /*result*/)
{
}


/**
* Description     : Virtual function implemented to set the error status of method start SELECT HD AUDIO SPLIT MODE
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : result : The result message of the method "SELECT HD AUDIO SPLIT MODE"
* @return         : void
*/
void AMFMTestMode::onFID_TUN_S_SET_AUDIO_SPLIT_MODEError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_SET_AUDIO_SPLIT_MODEError >& /*error*/)
{
}


/**
* Description     : Virtual function implemented to get the error status of SELECT HD AUDIO SPLIT MODE
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : result : The result message of the method "SELECT HD HD AUDIO SPLIT MODE"
* @return         : void
*/
void AMFMTestMode::onFID_TUN_G_AUDIO_SPLIT_MODEError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_G_AUDIO_SPLIT_MODEError >& /*error*/)
{
}


/**
* Description     : Virtual function implemented to get the status of SELECT HD AUDIO SPLIT MODE
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : result : The result message of the method "SELECT HD RESET TRANSITIONS"
* @return         : void
*/
void AMFMTestMode::onFID_TUN_G_AUDIO_SPLIT_MODEStatus(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_G_AUDIO_SPLIT_MODEStatus >& status)
{
   _splitModeStatus = status->getE8ParameterValue();
   ETG_TRACE_USR4(("onFID_TUN_G_AUDIO_SPLIT_MODEStatus::_splitModeStatus = %d", _splitModeStatus));
}


/**
* Description     : function implemented send hd split audio method start to midw
*@return         : void
*/
void AMFMTestMode::onDtmSplitAudio()
{
   if (_splitModeStatus == 0)
   {
      ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_AUDIO_SPLIT_MODEStart(*this, T_e8_Tun_AudioSplitMode__TUN_ACTIVATE_AUDIO_SPLIT_MODE)));
   }
   else if (_splitModeStatus == 1)
   {
      ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_AUDIO_SPLIT_MODEStart(*this, T_e8_Tun_AudioSplitMode__TUN_DEACTIVATE_AUDIO_SPLIT_MODE)));
   }
}


/**
* Description     : function implemented send hd reset transitions method start to midw
* @return         : void
*/
void AMFMTestMode::onDtmHdResetTransitons()
{
   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_HD_RESET_TRANSITIONCOUNTStart(*this)));
}


/**
* Description     : Virtual function implemented to get update of SELECT HD RESET TRANSITIONS Error
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : error : The error message of 'SELECT HD RESET TRANSITIONS'
* @return         : void
*/
void AMFMTestMode::onFID_TUN_S_HD_RESET_TRANSITIONCOUNTError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_HD_RESET_TRANSITIONCOUNTError >& /*error*/)
{
}


/**
* Description     : Virtual function implemented to set the result of method start SELECT HD RESET TRANSITIONS
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : result : The result message of the method "SELECT HD RESET TRANSITIONS"
* @return         : void
*/
void AMFMTestMode::onFID_TUN_S_HD_RESET_TRANSITIONCOUNTResult(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_HD_RESET_TRANSITIONCOUNTResult >& /*result*/)
{
}


//*****************************From here for audio program list for HD DTM Monitor screen

/**
* Description     : Function implemented to populate HD Audio Programs List
*
* @return         : tSharedPtrDataProvider
*/

/*tSharedPtrDataProvider AMFMTestMode::getTMRadioHDAudioPrgmList()
{
   const char* const RADIO_HD_AUDIO_PRGM_LIST_ITEM = "BtnAudioPgm";
   ListDataProviderBuilder listBuilder(LIST_ID_DTM_HD_MONITOR, RADIO_HD_AUDIO_PRGM_LIST_ITEM);
   bool isactive = false;
   bool AudioPrgmAvailabilityStatus = _audioPrgmList.getTUN_HD_AUDIOPROGRAM_MPS();
   if (AudioPrgmAvailabilityStatus && (_dtmHdStatus == "ON") && _activeHDAudioPrgm == 0)
   {
      isactive = true;
   }
   listBuilder.AddItem(0, 0, RADIO_HD_AUDIO_PRGM_LIST_ITEM).AddData("M").AddData(isactive).AddData(AudioPrgmAvailabilityStatus);
   for (unsigned int itr = 1; itr < DTM_HD_AUDIO_PRGM_SIZE; itr++)
   {
      isactive = false;
      AudioPrgmAvailabilityStatus = getAudioPrgmAvailability(itr);
      if (AudioPrgmAvailabilityStatus && (_dtmHdStatus == "ON") && itr == _activeHDAudioPrgm)
      {
         isactive = true;
      }
      listBuilder.AddItem(itr, 0, RADIO_HD_AUDIO_PRGM_LIST_ITEM).AddData(itr).AddData(isactive).AddData(AudioPrgmAvailabilityStatus);
   }
   return listBuilder.CreateDataProvider();
}


*/

//*****************************From here for audio program list for HD DTM Monitor screen

/**
* Description     : Function implemented to populate HD Audio Programs List
*
* @return         : tSharedPtrDataProvider
*/
void AMFMTestMode::getTMRadioHDAudioPrgmList()
{
   unsigned int initItr = 0;
   unsigned int isavailable = false;
   bool isactive = false;
   bool AudioPrgmAvailabilityStatus = _audioPrgmList.getTUN_HD_AUDIOPROGRAM_MPS();
   ETG_TRACE_USR4(("getTMRadioHDAudioPrgmList:AudioPrgmAvailabilityStatus = %d", AudioPrgmAvailabilityStatus));
   if (AudioPrgmAvailabilityStatus && (_dtmHdStatus == "ON"))
   {
      isavailable = true;
      if (_activeHDAudioPrgm == 0)
      {
         isactive = true;
      }
   }
   TunerDataBindingHandler::getDataBindingHandlerInstance()->setHDProgAvailableTestmode(initItr, isavailable);
   TunerDataBindingHandler::getDataBindingHandlerInstance()->setHDProgSizeTestmode(initItr, isactive);
   for (unsigned int itr = 1; itr < DTM_HD_AUDIO_PRGM_SIZE; itr++)
   {
      isactive = false;
      isavailable = false;
      AudioPrgmAvailabilityStatus = getAudioPrgmAvailability(itr);
      if (AudioPrgmAvailabilityStatus && (_dtmHdStatus == "ON"))
      {
         isavailable = true;
         if (itr == _activeHDAudioPrgm)
         {
            isactive = true;
         }
      }
      TunerDataBindingHandler::getDataBindingHandlerInstance()->setHDProgAvailableTestmode(itr, isavailable);
      TunerDataBindingHandler::getDataBindingHandlerInstance()->setHDProgSizeTestmode(itr, isactive);
   }
}


bool AMFMTestMode::getAudioPrgmAvailability(unsigned int num)
{
   bool status = false;
   switch (num)
   {
      case 1:
         status = _audioPrgmList.getTUN_HD_AUDIOPROGRAM_SPS1();
         break;
      case 2:
         status = _audioPrgmList.getTUN_HD_AUDIOPROGRAM_SPS2();
         break;
      case 3:
         status = _audioPrgmList.getTUN_HD_AUDIOPROGRAM_SPS3();
         break;
      case 4:
         status = _audioPrgmList.getTUN_HD_AUDIOPROGRAM_SPS4();
         break;
      case 5:
         status = _audioPrgmList.getTUN_HD_AUDIOPROGRAM_SPS5();
         break;
      case 6:
         status = _audioPrgmList.getTUN_HD_AUDIOPROGRAM_SPS6();
         break;
      case 7:
         status = _audioPrgmList.getTUN_HD_AUDIOPROGRAM_SPS7();
         break;

      default:
         status = _audioPrgmList.getTUN_HD_AUDIOPROGRAM_SPS1();
         break;
   }
   return status;
}


/**
     * Description     : Function implemented to populate DTM HD Tuning Parameter List
     *
     * @return         : tSharedPtrDataProvider
     */
tSharedPtrDataProvider AMFMTestMode::getDtmHdParameterTuningList()
{
   ETG_TRACE_USR3(("AMFMTestMode :getDtmHdParameterTuningList "));
   ListDataProviderBuilder listBuilder(LIST_ID_DTM_HD_PARAM_TUNING, "HDParamTuningTemplate");
   std::string strHDParamHexValue;
   HDParamTuningIconData item;
   for (uint index = 0; index < _listData.size(); index++)
   {
      if (index == _CurrentSelectedIndex)
      {
         ETG_TRACE_USR3(("HD Icon highlight,index = %d", index));
         item.mHD_Icon = 1;
      }
      else
      {
         ETG_TRACE_USR3(("HD Icon highlight,index = %d", index));
         item.mHD_Icon = 0;
      }
      strHDParamHexValue = TunerUtility::u16ConversionToHexString(_listData[index].value);
      listBuilder.AddItem(index, 0, "HDParamTuningTemplate").AddData(_listData.at(index).title.c_str()).AddData(strHDParamHexValue.c_str()).AddData((index == _CurrentSelectedIndex)).AddDataBindingUpdater<HDParamTuningIconDataBindingSource>(item);
   }
   return listBuilder.CreateDataProvider();
}


void AMFMTestMode::onEntryRequestHDParam(bool bScreenStatus)
{
   // _CurrentSelectedIndex = -1;
   ETG_TRACE_USR3(("AMFMTestMode :onEntryRequestHDParam bScreenStatus %d", bScreenStatus));
   if (bScreenStatus)
   {
      int8 parameterValue;
      uint8 u8Index;
      for (u8Index = 0; u8Index < _listData.size(); u8Index++)
      {
         T_e16_Tun_ParameterID eParameterId = mapHDParameterValues(u8Index);
         parameterValue = _listData.at(u8Index).value;

         ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_HD_SECTION_PARAMETERStart(*this, eParameterId, parameterValue)));
      }
      _requestCount = u8Index;
      _screenEntryRequest = true;
   }
   else
   {
      _requestCount = 0;
      _screenEntryRequest = false;
   }
}


/**
* Description     : Function implemented on courier message of dtm hs parameter tuning update
*
* @return         : void
*/
void AMFMTestMode::onDtmHdParamterTuningUpdate(uint8 tuneType)
{
   ETG_TRACE_USR3(("AMFMTestMode::onDtmHdParamterTuningUpdate: %d %d", tuneType, _CurrentSelectedIndex));
   switch (tuneType)
   {
      case HD_INCREMENT:
      {
         if (_CurrentSelectedIndex < _listData.size())
         {
            ETG_TRACE_USR3(("AMFMTestMode::onDtmHdParamterTuningUpdate: %d %d", tuneType, _CurrentSelectedIndex));
            if (_listData.at(_CurrentSelectedIndex).value < _listData.at(_CurrentSelectedIndex).upperLimit)
            {
               _FocusElemValue = _listData.at(_CurrentSelectedIndex).value + 1;
               setHdParameter(_FocusElemValue);
            }
         }
         break;
      }
      case HD_DECREMENT:
      {
         if (_CurrentSelectedIndex < _listData.size())
         {
            if (_listData.at(_CurrentSelectedIndex).value > _listData.at(_CurrentSelectedIndex).lowerLimit)
            {
               _FocusElemValue = _listData.at(_CurrentSelectedIndex).value - 1;
               setHdParameter(_FocusElemValue);
            }
         }
         break;
      }
      default:
         break;
   }
}


/**
* Description     : Function implemented set the hd parameter value and send a method start to MIDW.
*
* @return         : void
*/
void AMFMTestMode::setHdParameter(uint8 u8ItemValue)
{
   ETG_TRACE_USR3(("AMFMTestMode::setHdParameter u8ItemValue: %d _CurrentSelectedIndex: %d", u8ItemValue, _CurrentSelectedIndex));
   _isHdParameterSent = true;

   if (_CurrentSelectedIndex < _listData.size())
   {
      T_e16_Tun_ParameterID eParameterId = mapHDParameterValues(_CurrentSelectedIndex);
      int8 parameterValue = u8ItemValue;

      ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_HD_SECTION_PARAMETERStart(*this, eParameterId, parameterValue)));
   }
}


/**
* Description     : Function implemented to map the hd parameter values
*
* @return         : T_e16_Tun_ParameterID
*/
midw_tuner_fi_types::T_e16_Tun_ParameterID AMFMTestMode::mapHDParameterValues(uint8 u8Index)
{
   ETG_TRACE_USR3(("AMFMTestMode::mapHDParameterValues u8Index: %d ", u8Index));
   T_e16_Tun_ParameterID eTunParameter = T_e16_Tun_ParameterID__BLEND_THRESHOLD_FM_HYBRID;
   switch (u8Index)
   {
      case 0:
      {
         eTunParameter = T_e16_Tun_ParameterID__BLEND_THRESHOLD_FM_HYBRID;
      }
      break;
      case 1:
      {
         eTunParameter = T_e16_Tun_ParameterID__BLEND_RATE_FM_HYBRID;
      }
      break;
      case 2:
      {
         eTunParameter = T_e16_Tun_ParameterID__BLEND_THRESHOLD_AM_HYBRID;
      }
      break;
      case 3:
      {
         eTunParameter = T_e16_Tun_ParameterID__AM_MPS_BLEND_RATE;
      }
      break;

      case 4:
      {
         eTunParameter = T_e16_Tun_ParameterID__BLEND_DECISION;
      }
      break;

      case 5:
      {
         eTunParameter = T_e16_Tun_ParameterID__FM_CD_OR_N0_BLEND_DECISION;
      }
      break;

      case 6:
      {
         eTunParameter = T_e16_Tun_ParameterID__AM_CD_OR_N0_BLEND_DECISION;
      }
      break;

      case 7:
      {
         eTunParameter = T_e16_Tun_ParameterID__FM_SPS_BLEND_THRESHOLD;
      }
      break;

      case 8:
      {
         eTunParameter = T_e16_Tun_ParameterID__FM_SPS_BLEND_RATE;
      }
      break;

      default:
         break;
   }
   return eTunParameter;
}


/**
* Description     : Function called to initialize DTM HD parameter list
*
* @param[in]      :
* @return         : void
*/
void AMFMTestMode::initDtmHdParamterList()
{
   _listData.clear();
   _listData.push_back(ListItem("Blend Threshold FM Hybrid", 3, 0, 7));
   _listData.push_back(ListItem("Blend Rate FM Hybrid", 3, 1, 8));
   _listData.push_back(ListItem("Blend Threshold AM Hybrid", 3, 0, 7));
   _listData.push_back(ListItem("AM MPS Blend Rate", 3, 1, 8));
   _listData.push_back(ListItem("Blend Decision", 1, 0, 1));
   _listData.push_back(ListItem("FM Cd/N0 Blend Decision", 59, 52, 60));
   _listData.push_back(ListItem("AM Cd/N0 Blend Decision", 67, 50, 70));
   _listData.push_back(ListItem("FM SPS Blend Threshold", 4, 0, 7));
   _listData.push_back(ListItem("FM SPS Blend Rate", 1, 1, 8));
}


/**
     * Description     : Function called to set the values to active selected item in the list
     *
     * @param[in]      :
     * @return         : void
     */
void AMFMTestMode::setCurrentDtmHdParamterData(uint8 index)
{
   ETG_TRACE_USR4(("AMFMTestMode::setCurrentDtmHdParamterData: index : %d, %d", index, _listData.size()));

   if (index < _listData.size())
   {
      _CurrentSelectedIndex = index;
   }
   _isHdParameterSent = false;
}


/**
* Description     : Virtual function implemented to get update of G HD SECTION PARAMETER Error
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : error : The error message of 'G HD SECTION PARAMETER'
* @return         : void
*/

void AMFMTestMode::onFID_TUN_G_HD_SECTION_PARAMETERError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& proxy, const ::boost::shared_ptr< FID_TUN_G_HD_SECTION_PARAMETERError >& error)
{
   ETG_TRACE_USR3(("onFID_TUN_G_HD_SECTION_PARAMETERError"));
}


void AMFMTestMode::onFID_TUN_S_SET_HD_SECTION_PARAMETERError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& proxy, const ::boost::shared_ptr< FID_TUN_S_SET_HD_SECTION_PARAMETERError >& error)
{
   ETG_TRACE_USR3(("onFID_TUN_S_SET_HD_SECTION_PARAMETERError"));
}


/**
* Description     : Virtual function implemented to get update of SET HD SECTION PARAMETER Result
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : error : The error message of 'SELECT HD AUDIO PRG'
* @return         : void
*/
void AMFMTestMode::onFID_TUN_S_SET_HD_SECTION_PARAMETERResult(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& proxy, const ::boost::shared_ptr< FID_TUN_S_SET_HD_SECTION_PARAMETERResult >& result)
{
   ETG_TRACE_USR3(("onFID_TUN_S_SET_HD_SECTION_PARAMETERResult"));
}


/**
* Description     : Virtual function implemented to get update of G HD SECTION PARAMETER Status
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : error : The error message of 'G HD SECTION PARAMETER'
* @return         : void
*/
void AMFMTestMode::onFID_TUN_G_HD_SECTION_PARAMETERStatus(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& proxy, const ::boost::shared_ptr< FID_TUN_G_HD_SECTION_PARAMETERStatus >& status)
{
   uint16 hdParameterStatus = status->getE16Parameter_ID();
   tS8 hdParameterValue = status->getS8ParameterValue();
   ETG_TRACE_USR4(("AMFMTestMode:onFID_TUN_G_HD_SECTION_PARAMETERStatus hdParameterStatus: %d, hdParameterValue : %d", hdParameterStatus, hdParameterValue));
   //ETG_TRACE_USR4(("onFID_TUN_G_HD_SECTION_PARAMETERStatus hdparameter status %d hdparameter value %d ", hdParameterStatus, hdParameterValue));
   //ETG_TRACE_USR4(("onFID_TUN_G_HD_SECTION_PARAMETERStatus _CurrentSelectedIndex %d _FocusElemValue %d", _CurrentSelectedIndex, _FocusElemValue));
   //ETG_TRACE_USR4(("onFID_TUN_G_HD_SECTION_PARAMETERStatus _screenEntryRequest %d _requestCount %d", _screenEntryRequest, _requestCount));

   if (_screenEntryRequest && _requestCount == _listData.size())
   {
      updateList(LIST_ID_DTM_HD_PARAM_TUNING);
      _requestCount = 0;
      _screenEntryRequest = false;
   }
   else
   {
      if (((hdParameterStatus == mapHDParameterValues(_CurrentSelectedIndex)) && (_FocusElemValue == hdParameterValue)))
      {
         _listData[_CurrentSelectedIndex].value = hdParameterValue;
         updateList(LIST_ID_DTM_HD_PARAM_TUNING);
      }
   }
}


bool AMFMTestMode::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   bool isRetVal = false;
   ListProviderEventInfo info;
   if ((ListProviderEventInfo::GetItemIdentifierInfo(oMsg.GetSender(), info) == true))
   {
      unsigned int listId = info.getListId();     // the list id for generic access
      uint32 hdlRow = info.getHdlRow();     // normaly the itr
      unsigned int hdlCol = info.getHdlCol();     // if more than 1 active element in one list row, e.g. Button in a button line

      ETG_TRACE_USR4(("AMFMTestMode:ButtonReactionMsg Listid : %d, HdlRow : %d, hdlCol : %d", listId, hdlRow, hdlCol));
      if (oMsg.GetEnReaction() == enRelease)
      {
         switch (info.getListId())
         {
            /*  case LIST_ID_DTM_HD_MONITOR:
              {
                 onHDAudioPrgmSelect(hdlRow);
                 isRetVal = true;
                 break;
              }*/
            case LIST_ID_DTM_HD_PARAM_TUNING:
            {
               ETG_TRACE_USR4(("LIST_ID_DTM_HD_PARAM_TUNING: HdlRow : %d, hdlCol : %d", hdlRow, hdlCol));
               setCurrentDtmHdParamterData(hdlRow);
               updateList(LIST_ID_DTM_HD_PARAM_TUNING);
               isRetVal = true;
               break;
            }
            default:
               break;
         }
      }
      if (isRetVal)
      {
         POST_MSG((COURIER_MESSAGE_NEW(ButtonListItemUpdMsg)(listId, hdlRow, hdlCol, enRelease)));
      }
   }
   return isRetVal;
}


void AMFMTestMode::onHDAudioPrgmSelect(uint32 progvalue)
{
   ETG_TRACE_USR4(("AMFMTestMode::onHDAudioPrgmSelect progvalue: %d", progvalue));
   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SELECT_HD_AUDIOPRGMStart(*this, T_e8_Tun_SelectMode__TUN_SELECTMODE_ABSOLUTE, getHDAudioProgramEnumValue(progvalue), T_e8_Tun_RelativeDirection__TUN_RELATIVE_DECREMENT)));
}


T_e8_Tun_HDAudioProgram AMFMTestMode::getHDAudioProgramEnumValue(uint32 rownum)
{
   switch (rownum)
   {
      case 0:
         return T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_ONE;
         break;
      case 1:
         return T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_TWO;
         break;
      case 2:
         return T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_THREE;
         break;
      case 3:
         return T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_FOUR;
         break;
      case 4:
         return T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_FIVE;
         break;
      case 5:
         return T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_SIX;
         break;
      case 6:
         return T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_SEVEN;
         break;
      case 7:
         return T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_EIGHT;
         break;
      default:
         return T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_ONE;
   }
}


void AMFMTestMode::onFID_TUN_S_SELECT_HD_AUDIOPRGMError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_SELECT_HD_AUDIOPRGMError >& /*error*/)
{
   ETG_TRACE_USR4(("onFID_TUN_S_SELECT_HD_AUDIOPRGMError"));
}


void AMFMTestMode::onFID_TUN_S_SELECT_HD_AUDIOPRGMResult(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_SELECT_HD_AUDIOPRGMResult >& /*result*/)
{
   ETG_TRACE_USR4(("onFID_TUN_S_SELECT_HD_AUDIOPRGMResult"));
}


/**
* Description     : CallBack fuction from List implementation.
*
* @param[in]      : listId : List Id of the list to be displayed
* @return         : tSharedPtrDataProvider
*/

tSharedPtrDataProvider AMFMTestMode::getListDataProvider(const ListDateProviderReqMsg& listDataInfo)
{
   ETG_TRACE_USR3(("AMFMTestMode : getListDataProvider"));
   tSharedPtrDataProvider ptr = tSharedPtrDataProvider();
   ETG_TRACE_USR3(("AMFMTestMode : list ID %d", listDataInfo.GetListId()));
   getTMRadioHDAudioPrgmList();
   switch (listDataInfo.GetListId())
   {
      /* case LIST_ID_DTM_HD_MONITOR:
       {
          ptr = getTMRadioHDAudioPrgmList();
          break;
       }*/

      case LIST_ID_DTM_HD_PARAM_TUNING:
      {
         ptr = getDtmHdParameterTuningList();
         break;
      }
      default:
         break;
   }

   return ptr;
}


/**
* onCourierMessage - This message is received from the flexlist widget when it requires new data when the
*               list is displayed or scrolled
* @param[in] ListDateProviderReqMsg& oMsg
* @parm[out] bool
*/
bool AMFMTestMode::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   ETG_TRACE_USR3(("AMFMTestMode : onCourierMessage ListDateProviderReqMsg"));
   return ListRegistry::s_getInstance().updateList(oMsg.GetListId(), oMsg.GetStartIndex(), oMsg.GetWindowElementSize());//TODO : Confirm if this should be included after list data is received
}


/**
* Description     : Function implemented to update the list of the specified list ID
*
* @param[in]      : listID : List ID of the list to be updated
* @return         : void
*/
void AMFMTestMode::updateList(const uint32& listID)
{
   ETG_TRACE_USR3(("AMFMTestMode : updateList"));
   //ListDateProviderReqMsg listDataInfo;
   //listDataInfo.GetListId() = listID;
   ListRegistry::s_getInstance().updateList(listID);
}


}
} // namespace Core
// namespace App
