/**************************************************************************************
* @file         : RadioSetting.cpp
* @author       : Bharanidharan S
* @addtogroup   : AppHmi_Tuner
* @brief        :
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "hall_std_if.h"
#include  <vector>
#include "RadioSetting.h"
#include "AnnoHandler.h"
#include "CourierTunnelService/CourierMessageReceiverStub.h"
#include <string>
#include "AppHmi_TunerStateMachine.h"
#include "App/Core/DABInfo/DABStationList.h"
#include "App/Core/HmiTranslation_TextIds.h"
#include "ApplicationSwitchConst.h"
#include "../VehicleSrvClient/VehicleSrvClient.h"  //CAN Imp

using namespace App::Core;
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_TUNER_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_TUNER
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_Tuner_"
#define ETG_I_FILE_PREFIX                 App::Core::RadioSetting::
#include "trcGenProj/Header/RadioSetting.cpp.trc.h"
#endif

using namespace ::midw_tuner_fi_types;
using namespace ::MIDW_TUNER_FI;
using namespace ::MIDW_TUNERMASTER_FI;

#define LIST_ID_TUNER_RADIO_SETTING 1000
#define LIST_ID_TUNER_DABPTY 1001

//RadioSetting List Items
#define SOUNDSETTINGS 0
#define RADIOTEXT 1
#define TRAFFICANNOUNCEMENT 2
#define RDS 3
#define ALTERNATEFREQUENCY 4
#define DABPTYFILTER  5
#define DABDYNAMICCONTENT 6
#define DABFMLINKING 7
#define DABDABLINKING 8
#define WEATHERALERT 9


//Screen Ids for Transitions
#define DABPTYFILTERSCREENID 16

//DABPTY List IDs
#define DABPTY_INDEX0 0
#define DABPTY_INDEX1 1
#define DABPTY_INDEX2 2
#define DABPTY_INDEX3 3
#define DABPTY_INDEX4 4
#define DABPTY_INDEX5 5

#define CHECKED_DISABLE 3
#define UNCHECKED_DISABLE 2

namespace App {
namespace Core {

using namespace ::MIDW_EXT_DABTUNER_FI;
using namespace ::midw_ext_dabtuner_fi_types;


RadioSetting* RadioSetting::m_poRadioSetting = NULL;


RadioSetting::RadioSetting() :
   _tunerFiProxy(::MIDW_TUNER_FI::MIDW_TUNER_FIProxy::createProxy("tunerFiPort", *this)),
   _tunerMasterFiProxy(::MIDW_TUNERMASTER_FI::MIDW_TUNERMASTER_FIProxy::createProxy("tunermasterFiPort", *this)),
   _dabFiProxy(MIDW_EXT_DABTUNER_FIProxy::createProxy("dabFiPort", *this)),
   _isRegActive(true),
   _measureIndex(false),
   _measureMode("OFF"),
   _afIndex(false)
{
   m_bRadioSettingListUpdateRequired = false;

   m_baCheckBoxStatus[0] = false;
   m_baCheckBoxStatus[1] = false;
   m_baCheckBoxStatus[2] = false;
   m_baCheckBoxStatus[3] = false;
   m_baCheckBoxStatus[4] = false;
   m_baCheckBoxStatus[5] = false;
   m_baCheckBoxStatus[6] = false;
   m_baCheckBoxStatus[7] = false;
   m_baCheckBoxStatus[8] = false;
   m_baCheckBoxStatus[9] = false;

   m_ui32DABPTYCheckBoxStatus[0] = false;
   m_ui32DABPTYCheckBoxStatus[1] = false;
   m_ui32DABPTYCheckBoxStatus[2] = false;
   m_ui32DABPTYCheckBoxStatus[3] = false;
   m_ui32DABPTYCheckBoxStatus[4] = false;
   m_ui32DABPTYCheckBoxStatus[5] = false;
   m_bOptDABDynamicText = false;
   m_bOptRadioText = false;
   m_bOptDabFmFollowing = false;
   m_u8KDSConfigAF_Enable = false;
   m_u8KDSConfigREG_Enable = false;
   m_u8KDSConfigTA_Enable = false;
   m_ui32ListId = 0;
   m_bDABPTYLlistUpdateRequired = false;
   //used for hardcode testing of DABPTY
   dabptyforfirsttime = true;

   int region = TunerDataPoolConfig::getInstance()->readRegionType();
   ETG_TRACE_USR4(("RadioSetting::Print Region ( %d)", region));
   if (region == EUR_REGION)
   {
      ListRegistry::s_getInstance().addListImplementation(LIST_ID_TUNER_RADIO_SETTING, this);
      ListRegistry::s_getInstance().addListImplementation(LIST_ID_TUNER_DABPTY, this);
   }
   else if (region == NAM_REGION)  //todo check for DAB.
   {
      ListRegistry::s_getInstance().addListImplementation(LIST_ID_TUNER_RADIO_SETTING, this);
   }

   if (_tunerFiProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _tunerFiProxy->getPortName());
   }
   if (_tunerMasterFiProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _tunerMasterFiProxy->getPortName());
   }
   if (_dabFiProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _dabFiProxy->getPortName());
   }

   //todo check for DAB variant for EU region.
   if (region == EUR_REGION)
   {
      std::string textid = "";
      _radioSettingsList.reserve(8);
      _radioSettingsList.push_back(RadioSettingsListItem(Candera::String(TextId_0x0A70), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_BUTTON_IMAGE, SOUNDSETTINGS));
      _radioSettingsList.push_back(RadioSettingsListItem(Candera::String(TextId_0x0A6E), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, RADIOTEXT));
      _radioSettingsList.push_back(RadioSettingsListItem(Candera::String(TextId_0x007B), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, TRAFFICANNOUNCEMENT));
      _radioSettingsList.push_back(RadioSettingsListItem(Candera::String(TextId_0x0A6F), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, RDS));
      _radioSettingsList.push_back(RadioSettingsListItem(Candera::String(TextId_0x1045), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, ALTERNATEFREQUENCY));
      _radioSettingsList.push_back(RadioSettingsListItem(Candera::String(TextId_0x0A6C), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, DABDYNAMICCONTENT));
      _radioSettingsList.push_back(RadioSettingsListItem(Candera::String(TextId_0x0A6B), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, DABFMLINKING));
      _radioSettingsList.push_back(RadioSettingsListItem(Candera::String(TextId_0x0B9E), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_BUTTON, DABPTYFILTER));
      //_radioSettingsList.push_back(RadioSettingsListItem("DAB DAB Linkin,g", IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE,DABDABLINKING));

      //DAB PTY CATEGORY
      _dabradioSettingsList.reserve(6);
      _dabradioSettingsList.push_back(RadioSettingsListItem(Candera::String(TextId_0x0BBE), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_BUTTON_IMAGE, DABPTY_INDEX0));
      _dabradioSettingsList.push_back(RadioSettingsListItem(Candera::String(TextId_0x0BA0), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, DABPTY_INDEX1));
      _dabradioSettingsList.push_back(RadioSettingsListItem(Candera::String(TextId_0x0BA6), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, DABPTY_INDEX2));
      _dabradioSettingsList.push_back(RadioSettingsListItem(Candera::String(TextId_0x0BA3), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, DABPTY_INDEX3));
      _dabradioSettingsList.push_back(RadioSettingsListItem(Candera::String(TextId_0x0EA7), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, DABPTY_INDEX4));
      _dabradioSettingsList.push_back(RadioSettingsListItem(Candera::String(TextId_0x0EA8), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, DABPTY_INDEX5));
   }
   else if (region == NAM_REGION)
   {
      _radioSettingsList.reserve(3);
      _radioSettingsList.push_back(RadioSettingsListItem(Candera::String(TextId_0x090F), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_BUTTON_IMAGE, SOUNDSETTINGS));
      _radioSettingsList.push_back(RadioSettingsListItem(Candera::String(TextId_0x0A6E), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, RADIOTEXT));
      _radioSettingsList.push_back(RadioSettingsListItem(Candera::String(TextId_0x191D), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, WEATHERALERT));
   }
   /*else
   {
   _radioSettingsList.reserve(1);
   _radioSettingsList.push_back(RadioSettingsListItem("Sound Setting", IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_BUTTON_IMAGE, SOUNDSETTINGS));
   }*/
   //ReadRadioSettingKDS();
   m_u32DabPtyMask = 0;
   m_u32DabPtyFilterToClient = 0;

   ReadRadioSettingKDS();

   InitDABPTYDisableForFirstTime();

   ETG_TRACE_USR4(("RadioSetting::m_u32DabPtyMask ( %d)", m_u32DabPtyMask));
   ETG_TRACE_USR4(("RadioSetting::m_u32DabPtyFilterToClient ( %d)", m_u32DabPtyFilterToClient));

   //The below code will iterate for 6 items and m_u32DabPtyFilterToClient will be zero: for first time
   //The next time m_u32DabPtyMask will be taken from dp and according the m_u32DabPtyFilterToClient is updated with value.

   for (unsigned int u32Index = 0; u32Index < (sizeof(g_cau32DabPtyFilterMask) / sizeof(unsigned int)); u32Index++)
   {
      if (m_u32DabPtyMask & (1 << u32Index))
      {
         m_u32DabPtyFilterToClient |= g_cau32DabPtyFilterMask[u32Index];
      }
      else
      {
         m_u32DabPtyFilterToClient &= (~g_cau32DabPtyFilterMask[u32Index]);
      }

      ETG_TRACE_USR4(("RadioSetting::m_u32DabPtyFilterToClient ( %d)", m_u32DabPtyFilterToClient));
   }
}


void RadioSetting::vSetDABPtyFilter(unsigned int u32Option)
{
   ETG_TRACE_USR3(("RadioSetting::vSetDABPtyFilter u32Option %d", u32Option));

   /**
   * Determine filter mask
   */
   unsigned int  u32DabPtyMask = m_u32DabPtyMask ^ (1 << u32Option);
   vSetDpDABPtyFilter(u32DabPtyMask);
   ETG_TRACE_USR3(("RadioSetting::vSetDABPtyFilter :(%d %d)", m_u32DabPtyMask, m_u32DabPtyFilterToClient));
}


void RadioSetting::vSetDpDABPtyFilter(unsigned int u32Mask)
{
   dp_tclAppHmi_TunerPtymask objptymask;
   m_u32DabPtyMask = u32Mask;
   tS32 s32Status = objptymask.s32SetData(m_u32DabPtyMask);
   ETG_TRACE_USR3(("s32Status(%d)", (tU32)s32Status));

   ETG_TRACE_USR3(("RadioSetting::vSetDpDABPtyFilter u32Mask %d", u32Mask));
   ETG_TRACE_USR3(("RadioSetting::vSetDpDABPtyFilter m_u32DabPtyMask %d", m_u32DabPtyMask));

   vSendPTYFilter();
}


void RadioSetting::vSendPTYFilter(void)
{
   unsigned int u32PtyFilter = PTY_FILTER_EN_ALL_TO_MIDW;

   ETG_TRACE_USR3(("RadioSetting::vSendPTYFilter u32PtyFilter %d", u32PtyFilter));

   /**
   * Evaluate filter to client from HMI mask
   */
   for (unsigned int u32Index = 0; u32Index < (sizeof(g_cau32DabPtyFilterMask) / sizeof(unsigned int)); u32Index++)
   {
      if (m_u32DabPtyMask & (1 << u32Index))
      {
         m_u32DabPtyFilterToClient |= g_cau32DabPtyFilterMask[u32Index];
      }
      else
      {
         m_u32DabPtyFilterToClient &= (~g_cau32DabPtyFilterMask[u32Index]);
      }
   }
   if ((m_u32DabPtyFilterToClient & (1 << EN_PTY_FILTER_ENABLED) /*Value =1*/) &&
         (m_u32DabPtyFilterToClient & ((unsigned int)~PTY_FILTER_ENABLE_MASK))
      )
   {
      /*Loop is executed only if Enable DAB category filter
      is checked and one or more of the  5 categories  is checked */
      u32PtyFilter = m_u32DabPtyFilterToClient;
      /*[DAB] Unable to list only DAB service
      specific category instead displaying selected category + empty category
      */
      u32PtyFilter &= 0xFFFFFFFE;

      ETG_TRACE_USR3(("RadioSetting::vSendPTYFilter Loop is executed only if Enable DAB category filter u32PtyFilter %d", u32PtyFilter));
   }

   DABStationList::poGetInstance()->setDABPTYFilterToMW(u32PtyFilter);
}


RadioSetting::~RadioSetting()
{
}


/**
 * Description     : Request from Framework to start registering for properties.
 *                   This request means GUI is ready now.
 *
 * @param[in]      : proxy : reference to the base class of all proxies
 * @param[in]      : stateChange : represents a state transition of a service
 * @return         : void
 */
void RadioSetting::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("RadioSetting::registerProperties()"));

   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      //  _tunerFiProxy->sendFID_TUN_G_SET_RDSREGUpReg(*this);
      _tunerFiProxy->sendFID_TUN_G_SET_AFUpReg(*this);
      ETG_TRACE_USR4(("RadioSetting::registerProperties()sendFID_TUN_G_SET_RDSREGUpReg"));
   }
}


/**
 * Description     : Request from Framework to start de-registering for properties.
 *
 * @param[in]      : proxy : reference to the base class of all proxies
 * @param[in]      : stateChange : represents a state transition of a service
 * @return         : void
 */
void RadioSetting::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("RadioSetting::deregisterProperties()"));
}


void RadioSetting::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("RadioSetting : onAvailable"));
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_SET_RDSREGUpReg(*this);
      _tunerFiProxy->sendFID_TUN_G_SET_AFUpReg(*this);
      //_tunerFiProxy->sendFID_TUN_G_SET_MEASUREMODEUpReg(*this);

      vSendPTYFilter();
   }
}


void RadioSetting::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("RadioSetting : onUnavailable"));
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      // _tunerFiProxy->sendFID_TUN_G_SET_MEASUREMODERelUpRegAll();
      _tunerFiProxy->sendFID_TUN_G_SET_RDSREGRelUpRegAll();
      _tunerFiProxy->sendFID_TUN_G_SET_AFRelUpRegAll();
   }
}


void RadioSetting::onFID_TUN_G_SET_RDSREGError(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_SET_RDSREGError >& /*error*/)
{
}


void RadioSetting::onFID_TUN_G_SET_RDSREGStatus(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_SET_RDSREGStatus >& status)
{
   ETG_TRACE_USR4(("RadioSetting : onFID_TUN_G_SET_RDSREGStatus %d", status->getU8RDSreg()));
   if (status->getU8RDSreg() == 0)
   {
      _isRegActive = true;
   }
   else if (status->getU8RDSreg() == 1)
   {
      _isRegActive = false;
   }

   //update to GUI model and also datapool
   m_u8KDSConfigREG_Enable = _isRegActive;
   m_baCheckBoxStatus[3] = m_u8KDSConfigREG_Enable;
   //ListRegistry::s_getInstance().updateList(LIST_ID_TUNER_RADIO_SETTING);
   updateSettingsList();
}


void RadioSetting::onFID_TUN_S_SETUP_DDADDSError(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_S_SETUP_DDADDSError >& /*error*/)
{}
void RadioSetting::onFID_TUN_S_SETUP_DDADDSResult(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_S_SETUP_DDADDSResult >& /*result*/) {}

/**
* Description     : Virtual function implemented to get update of FID_TUN_S_SET_FEATURE Error
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : error : The error message of 'FID_TUN_S_SET_FEATURE'
* @return         : void
*/
void RadioSetting::onFID_TUN_S_SET_FEATUREError(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_S_SET_FEATUREError >& /*error*/)
{
}


/**
* Description     : Virtual function implemented to get update of FID_TUN_S_SET_FEATURE Result
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : result : The status message of 'FID_TUN_S_SET_FEATURE'
* @return         : void
*/
void RadioSetting::onFID_TUN_S_SET_FEATUREResult(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_S_SET_FEATUREResult >& /*result*/)
{
}


void RadioSetting::onFID_TUN_G_SET_AFError(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_SET_AFError >& /*error*/)
{
   ETG_TRACE_USR4(("RadioSetting : onFID_TUN_G_SET_AFError"));
}


void RadioSetting::onFID_TUN_G_SET_AFStatus(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_SET_AFStatus >& status)
{
   ETG_TRACE_USR4(("RadioSetting : onFID_TUN_G_SET_AFStatus %d", status->getU8AF()));
   if (status->getU8AF())
   {
      //_afStatus = "ON";
      _afIndex = 1;
   }
   else
   {
      //_afStatus = "OFF";
      _afIndex = 0;
   }
   //update GUI
   m_u8KDSConfigAF_Enable = _afIndex;
   m_baCheckBoxStatus[4] = _afIndex;

//   TunerDataBindingHandler::getDataBindingHandlerInstance()->TunerPLMsg(EV_MSG_TO_PL_OPT_ALT_FREQ, _afIndex, "");
   updateSettingsList();
}


tSharedPtrDataProvider RadioSetting::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   tSharedPtrDataProvider dataProvider;
   m_ui32ListId = oMsg.GetListId();
   ETG_TRACE_USR4(("RadioSetting::getListDataProvider(%d)", m_ui32ListId));

   switch (m_ui32ListId)
   {
      case LIST_ID_TUNER_RADIO_SETTING:
      {
         dataProvider = getListDataProviderMainList(m_ui32ListId);
         break;
      }
      case LIST_ID_TUNER_DABPTY:
      {
         dataProvider = getListDataProviderMainList(m_ui32ListId);
         break;
      }

      default:
      {
         return tSharedPtrDataProvider();
      }
   }

   return dataProvider;
}


tSharedPtrDataProvider RadioSetting::getListDataProviderMainList(::Courier::UInt32 currentListID)
{
   ETG_TRACE_USR4(("RadioSetting::getListDataProviderMainList()"));
   ListDataProviderBuilder listBuilder(currentListID);

   if (currentListID == LIST_ID_TUNER_DABPTY)
   {
      for (std::vector<dabRadioSettingsListItem>::iterator itemNumber = _dabradioSettingsList.begin(); itemNumber != _dabradioSettingsList.end(); ++itemNumber)
      {
         if (itemNumber->_listButtonType == LIST_BUTTON_TYPE_TOGGLE)
         {
            ETG_TRACE_USR4(("RadioSetting::LIST_BUTTON_TYPE_TOGGLE for DAB %d", itemNumber->_order));
            listBuilder.AddItem(
               itemNumber->_order,
               0UL,  //    - unused
               "Layer_Checkbox")  //0 - Name(Id) of the button
            .AddData(itemNumber->_text)
            .AddData(m_ui32DABPTYCheckBoxStatus[itemNumber->_order]);
         }
         else if (itemNumber->_listButtonType == LIST_BUTTON_TYPE_BUTTON_IMAGE)
         {
            ETG_TRACE_USR4(("RadioSetting::LIST_BUTTON_TYPE_TOGGLE for DAB %d", itemNumber->_order));
            listBuilder.AddItem(
               itemNumber->_order,
               0UL,  //    - unused
               "Layer_Checkbox_EnableDisable")  //0 - Name(Id) of the button
            .AddData(itemNumber->_text)
            .AddData(m_ui32DABPTYCheckBoxStatus[itemNumber->_order]);
         }
      }
   }//end of DABPTY list

   else if (currentListID == LIST_ID_TUNER_RADIO_SETTING)
   {
      for (std::vector<RadioSettingsListItem>::iterator itemNumber = _radioSettingsList.begin(); itemNumber != _radioSettingsList.end(); ++itemNumber)
      {
         if ((itemNumber->_listButtonType == LIST_BUTTON_TYPE_TOGGLE) && (itemNumber == _radioSettingsList.begin()))
         {
            ETG_TRACE_USR4(("RadioSetting::LIST_BUTTON_TYPE_SETTINGS %d", itemNumber->_order));
            listBuilder.AddItem(
               itemNumber->_order,
               0UL,  //    - unused
               "Layer_Checkbox")  //0 - Name(Id) of the button
            .AddData(itemNumber->_text)
            .AddData(true);
         }
         else
         {
            if (itemNumber->_listButtonType == LIST_BUTTON_TYPE_TOGGLE)
            {
               ETG_TRACE_USR4(("RadioSetting::LIST_BUTTON_TYPE_TOGGLE %d", itemNumber->_order));
               listBuilder.AddItem(
                  itemNumber->_order,
                  0UL,  //    - unused
                  "Layer_Checkbox")  //0 - Name(Id) of the button
               .AddData(itemNumber->_text)
               .AddData(m_baCheckBoxStatus[itemNumber->_order]);
            }
            else if (itemNumber->_listButtonType == LIST_BUTTON_TYPE_BUTTON)
            {
               ETG_TRACE_USR4(("RadioSetting::LIST_BUTTON_TYPE_BUTTON %d", itemNumber->_order));
               listBuilder.AddItem(
                  itemNumber->_order,
                  0UL,  //    - unused
                  "Layer_ListButton")  //0 - Name(Id) of the button
               .AddData(itemNumber->_text)
               .AddData(true);
            }
            else if (itemNumber->_listButtonType == LIST_BUTTON_TYPE_BUTTON_IMAGE)
            {
               ETG_TRACE_USR4(("RadioSetting::LIST_BUTTON_TYPE_BUTTON_IMAGE %d", itemNumber->_order));
               listBuilder.AddItem(
                  itemNumber->_order,
                  0UL,  //    - unused
                  "Layer_Sound")  //0 - Name(Id) of the button
               .AddData(itemNumber->_text)
               .AddData(true);   //reverted changes
            }
         }
      }
   }//end of radio setting list

   return listBuilder.CreateDataProvider();
}


bool RadioSetting::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   ETG_TRACE_USR4(("RadioSetting::ButtonListItemUpdMsg"));
   bool msgProcessed = false;
   unsigned int listID = oMsg.GetListId();
   unsigned int rowIdx = (unsigned int)oMsg.GetHdl();
   ListDataProviderBuilder modifiedListItems(LIST_ID_TUNER_DABPTY);

   //if (LISTID_DAB == listID)
   //{
   //	if (oMsg.GetReaction() == enRelease)
   //	{
   //		msgProcessed = true;
   //	}
   //}

   if (LIST_ID_TUNER_RADIO_SETTING == listID)
   {
      if (oMsg.GetReaction() == enRelease)
      {
         HandleRadiosettingButtonListItemUpdMsg(rowIdx);
         msgProcessed = true;
      }
   }
   else if (LIST_ID_TUNER_DABPTY == listID)
   {
      if (oMsg.GetReaction() == enRelease)
      {
         HandleDABPTYButtonListItemUpdMsg(rowIdx);
         msgProcessed = true;
      }
   }
   if (m_bRadioSettingListUpdateRequired)
   {
      if (LIST_ID_TUNER_DABPTY == listID)
      {
         ListRegistry::s_getInstance().updateList(LIST_ID_TUNER_DABPTY);
      }
      else if (LIST_ID_TUNER_RADIO_SETTING == listID)
      {
         ListRegistry::s_getInstance().updateList(LIST_ID_TUNER_RADIO_SETTING);
      }
   }

   return msgProcessed;
}


void RadioSetting::HandleRadiosettingButtonListItemUpdMsg(unsigned int rowIdx)
{
   switch (rowIdx)
   {
      case SOUNDSETTINGS:
      {
         ETG_TRACE_USR4(("RadioSetting::ButtonListItemUpdMsg() Sound Settings"));
         uint8 activeRegion = SrcChangeHandler::poGetInstance()->getActiveRegionValue();
         ETG_TRACE_USR3(("RadioSetting::ActiveRegionID(%d)", activeRegion));
         ApplicationSwitchClientComponent* _applicationSwitchClientComponent = ApplicationSwitchClientComponent::poGetInstance();
         if (activeRegion == 2)
         {
            _applicationSwitchClientComponent->requestContextSwitchApplication(activeRegion, APPID_APPHMI_TUNER, enActivityIDs__eActivityID_TUNER_SETTINGS_VIEW, APPID_APPHMI_SYSTEM, enActivityIDs__eActivityID_SETTINGS_VOLUME);
         }
         else if (activeRegion == 0)
         {
            _applicationSwitchClientComponent->requestContextSwitchApplication(activeRegion, APPID_APPHMI_TUNER, enActivityIDs__eActivityID_TUNER_SETTINGS_VIEW, APPID_APPHMI_SYSTEM, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)186);
         }
         else
         {
            //do nothing
         }         break;
      }
      case RADIOTEXT:
      {
         ETG_TRACE_USR4(("RadioSetting::ButtonListItemUpdMsg() RadioText Index %d", m_baCheckBoxStatus[rowIdx]));

         m_bOptRadioText = !m_bOptRadioText;
         m_baCheckBoxStatus[rowIdx] = m_bOptRadioText;
         //Write the value to datapool, update GUI to enable or disable radio text
         TunerDataPoolConfig::getInstance()->setRadioTextStatus(m_bOptRadioText);
         TunerDataBindingHandler::getDataBindingHandlerInstance()->TunerPLMsg(EV_MSG_TO_PL_OPT_RADIO_TEXT, m_bOptRadioText, "");
         VehicleSrvClient::poGetInstance()->updateRDSStatus(m_bOptRadioText); //CAN Imp
         m_bRadioSettingListUpdateRequired = true;
         break;
      }
      case TRAFFICANNOUNCEMENT:

      {
         ETG_TRACE_USR4(("RadioSetting::ButtonListItemUpdMsg() TRAFFICANNOUNCEMENT Index %d", m_baCheckBoxStatus[rowIdx]));

         AnnoHandler::poGetInstance()->onToggleTA();/*setTAflag(m_baCheckBoxStatus[rowIdx]);*/
         m_bRadioSettingListUpdateRequired = true;
         break;
      }
      case RDS:  //regional
      {
         ETG_TRACE_USR4(("RadioSetting::ButtonListItemUpdMsg() RDS Index %d", m_baCheckBoxStatus[rowIdx]));
         setREGstatus(!_isRegActive);
         m_bRadioSettingListUpdateRequired = true;
         break;
      }
      case ALTERNATEFREQUENCY:
      {
         ETG_TRACE_USR4(("RadioSetting::ButtonListItemUpdMsg() ALTERNATEFREQUENCY Index %d", m_baCheckBoxStatus[rowIdx]));
         onSelectAFFollowing();
         break;
      }

      case DABPTYFILTER:
      {
         ETG_TRACE_USR4(("RadioSetting::ButtonListItemUpdMsg() DAB Index %d", m_baCheckBoxStatus[rowIdx]));
         sm_gTargetView = DABPTYFILTERSCREENID;
         Courier::Message* lMsg = COURIER_MESSAGE_NEW(TunerViewMsg)();
         if (0 != lMsg)
         {
            lMsg->Post();
         }
         break;
      }

      case DABDYNAMICCONTENT:
      {
         ETG_TRACE_USR4(("RadioSetting::ButtonListItemUpdMsg() DAB content Index %d", m_baCheckBoxStatus[rowIdx]));
         m_bOptDABDynamicText = !m_bOptDABDynamicText;
         m_baCheckBoxStatus[rowIdx] = m_bOptDABDynamicText;
         //Write the value to datapool, update GUI to enable or disable DAB Dynamic DLS text
         TunerDataPoolConfig::getInstance()->vSetDabDynamicContentStatus(m_bOptDABDynamicText);
         TunerDataBindingHandler::getDataBindingHandlerInstance()->TunerPLMsg(EV_MSG_TO_PL_OPT_DAB_DYNCONTENT, m_bOptDABDynamicText, "");
         m_bRadioSettingListUpdateRequired = true;
         break;
      }
      case DABFMLINKING:
      {
         ETG_TRACE_USR4(("RadioSetting::ButtonListItemUpdMsg() DAB FM Index %d", m_baCheckBoxStatus[rowIdx]));
         m_bOptDabFmFollowing = !m_bOptDabFmFollowing;
         m_baCheckBoxStatus[rowIdx] = m_bOptDabFmFollowing;
         //Write the value to datapool, update GUI to enable or disable DAB FM Link Status
         TunerDataPoolConfig::getInstance()->vSetDabFmFollowingStatus(m_bOptDabFmFollowing);
         TunerDataBindingHandler::getDataBindingHandlerInstance()->TunerPLMsg(EV_MSG_TO_PL_OPT_DAB_FM_LINKING, m_bOptDabFmFollowing, "");
         m_bRadioSettingListUpdateRequired = true;
         VehicleSrvClient::poGetInstance()->updateDABFMStatus(m_bOptDabFmFollowing); //CAN Imp
         DABStationList::poGetInstance()->vRequestServiceFollowing(m_bOptDabFmFollowing, true);
         break;
      }
      case DABDABLINKING:
      {
         ETG_TRACE_USR4(("RadioSetting::ButtonListItemUpdMsg() DAB DAB Index %d", m_baCheckBoxStatus[rowIdx]));
         m_baCheckBoxStatus[rowIdx] = (m_baCheckBoxStatus[rowIdx] == true) ? false : true;
         m_bRadioSettingListUpdateRequired = true;
         break;
      }
      //weather alert
      case WEATHERALERT:

      {
         ETG_TRACE_USR4(("RadioSetting::ButtonListItemUpdMsg() WEATHERALERT Index %d", m_baCheckBoxStatus[rowIdx]));

         AnnoHandler::poGetInstance()->onToggleWB();/*setTAflag(m_baCheckBoxStatus[rowIdx]);*/
         m_bRadioSettingListUpdateRequired = true;
         break;
      }
      default:
      {
         ETG_TRACE_USR4(("RadioSetting::ButtonListItemUpdMsg() default"));
         break;
      }
   }//end of radiosetting switch
}


void RadioSetting::HandleDABPTYButtonListItemUpdMsg(unsigned int rowIdx)
{
   ETG_TRACE_USR4(("RadioSetting::HandleDABPTYButtonListItemUpdMsg m_ui32DABPTYCheckBoxStatus[0] %d", m_ui32DABPTYCheckBoxStatus[0]));
   switch (rowIdx)
   {
      case DABPTY_INDEX0:
      {
         ETG_TRACE_USR4(("RadioSetting::ButtonListItemUpdMsg() DABPTY Index %d", rowIdx));
         m_ui32DABPTYCheckBoxStatus[rowIdx] = (m_ui32DABPTYCheckBoxStatus[rowIdx] == true) ? false : true;
         HandleDABPTYOptionsEnableDisable(m_ui32DABPTYCheckBoxStatus[rowIdx]);

         if (m_ui32DABPTYCheckBoxStatus[rowIdx] == true)
         {
            (*oDabOptionsData).misChecked_EnablePTYFilter = true;
         }
         else
         {
            (*oDabOptionsData).misChecked_EnablePTYFilter = false;
         }
         oDabOptionsData.MarkAllItemsModified();
         oDabOptionsData.SendUpdate();

         //if (m_ui32DABPTYCheckBoxStatus[rowIdx] == true)
         //{
         //	(*odabptyCheckBoxData).mToggleIconSwitchIndex = true;
         //	odabptyCheckBoxData.MarkAllItemsModified();
         //	odabptyCheckBoxData.SendUpdate(true);
         //}
         //else
         //{
         //	(*odabptyCheckBoxData).mToggleIconSwitchIndex = false;
         //	odabptyCheckBoxData.MarkAllItemsModified();
         //	odabptyCheckBoxData.SendUpdate(true);

         //}
         m_bRadioSettingListUpdateRequired = true;
         break;
      }
      case DABPTY_INDEX1:
         if (m_ui32DABPTYCheckBoxStatus[0] == 1)
         {
            ETG_TRACE_USR4(("RadioSetting::ButtonListItemUpdMsg() DABPTY Index %d", rowIdx));
            m_ui32DABPTYCheckBoxStatus[rowIdx] = (m_ui32DABPTYCheckBoxStatus[rowIdx] == true) ? false : true;
            m_bRadioSettingListUpdateRequired = true;
            break;
         }
      case DABPTY_INDEX2:
         if (m_ui32DABPTYCheckBoxStatus[0] == 1)
         {
            ETG_TRACE_USR4(("RadioSetting::ButtonListItemUpdMsg() DABPTY Index %d", rowIdx));
            m_ui32DABPTYCheckBoxStatus[rowIdx] = (m_ui32DABPTYCheckBoxStatus[rowIdx] == true) ? false : true;
            m_bRadioSettingListUpdateRequired = true;
            break;
         }
      case DABPTY_INDEX3:
         if (m_ui32DABPTYCheckBoxStatus[0] == 1)
         {
            ETG_TRACE_USR4(("RadioSetting::ButtonListItemUpdMsg() DABPTY Index %d", rowIdx));
            m_ui32DABPTYCheckBoxStatus[rowIdx] = (m_ui32DABPTYCheckBoxStatus[rowIdx] == true) ? false : true;
            m_bRadioSettingListUpdateRequired = true;
            break;
         }

      case DABPTY_INDEX4:
         if (m_ui32DABPTYCheckBoxStatus[0] == 1)
         {
            ETG_TRACE_USR4(("RadioSetting::ButtonListItemUpdMsg() DABPTY Index %d", rowIdx));
            m_ui32DABPTYCheckBoxStatus[rowIdx] = (m_ui32DABPTYCheckBoxStatus[rowIdx] == true) ? false : true;
            m_bRadioSettingListUpdateRequired = true;
            break;
         }

      case DABPTY_INDEX5:
         if (m_ui32DABPTYCheckBoxStatus[0] == 1)
         {
            ETG_TRACE_USR4(("RadioSetting::ButtonListItemUpdMsg() DABPTY content Index %d", rowIdx));
            m_ui32DABPTYCheckBoxStatus[rowIdx] = (m_ui32DABPTYCheckBoxStatus[rowIdx] == true) ? false : true;
            m_bRadioSettingListUpdateRequired = true;
            break;
         }

      default:
      {
         break;
      }
   }//end of DABPTY Switch

   if (m_ui32DABPTYCheckBoxStatus[0] == true)
   {
      ETG_TRACE_USR4(("RadioSetting::ButtonListItemUpdMsg() m_ui32DABPTYCheckBoxStatus before vSetDABPtyFilter  %d", rowIdx));
      vSetDABPtyFilter(rowIdx);
   }
   else
   {
      ETG_TRACE_USR4(("setting all the filters"));
      //uint32 allFilter = 0x3FFFFFFF;
      //DABStationList::poGetInstance()->setDABPTYFilterToMW(allFilter);  // send all the filters suman
   }
}


/**
* Description     : Function called on selecting REG in Radio settings list.
* @return         : void
*/
void RadioSetting::setREGstatus(bool regStatus)
{
   if (_isRegActive != regStatus)
   {
      T_TunerFeature rdsONOFF;
      T_b32_Tun_Features feature;
      T_b32_Tun_Features feature_mask;
      feature.setTUN_FEATURE_REG_AUTO_MODE(_isRegActive);
      feature_mask.setTUN_FEATURE_REG_AUTO_MODE(true);
      rdsONOFF.setB32Features(feature);
      rdsONOFF.setB32MaskFeatures(feature_mask);
      (_tunerFiProxy.get())&&  (_tunerFiProxy->sendFID_TUN_S_SET_FEATUREStart(*this, rdsONOFF));
   }
}


void RadioSetting::setHMI_REGstatus()
{
   bool regstatus = false;
   T_TunerFeature rdsONOFF;
   T_b32_Tun_Features feature;
   T_b32_Tun_Features feature_mask;
   feature.setTUN_FEATURE_REG_AUTO_MODE(regstatus);
   feature_mask.setTUN_FEATURE_REG_AUTO_MODE(true);
   rdsONOFF.setB32Features(feature);
   rdsONOFF.setB32MaskFeatures(feature_mask);
   (_tunerFiProxy.get())&&  (_tunerFiProxy->sendFID_TUN_S_SET_FEATUREStart(*this, rdsONOFF));
}


/**
* Description     : Function returns REG status
*
* @return         : bool
*/
bool RadioSetting::isREGActive()
{
   return _isRegActive;
}


/**
* Description     : Function called to get TA ON/OFF text.
*
* @param[in]      : none.
* @return         : std::string
*/
bool RadioSetting::getTAStatus()
{
   return AnnoHandler::poGetInstance()->isTaActive();
}


/**
* Description     : Function called on press of AF Following option in AM FM Settings screen.
*
* @return         : void
*/
void RadioSetting::onSelectAFFollowing()
{
   T_TunerFeature testMode;
   T_b32_Tun_Features feature;
   T_b32_Tun_Features feature_mask;
   bool AF = false;
   if (_tunerFiProxy.get())
   {
      AF = _tunerFiProxy->getFID_TUN_G_SET_AF().getU8AF();
   }
   feature_mask.setTUN_FEATURE_AF_MODE(true);
   feature.setTUN_FEATURE_AF_MODE(!AF);
   testMode.setB32Features(feature);
   testMode.setB32MaskFeatures(feature_mask);
   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_FEATUREStart(*this, testMode)));
}


void RadioSetting::onSelectHMI_AFFollowing()
{
   T_TunerFeature testMode;
   T_b32_Tun_Features feature;
   T_b32_Tun_Features feature_mask;
   bool AF = true;

   feature_mask.setTUN_FEATURE_AF_MODE(true);
   feature.setTUN_FEATURE_AF_MODE(AF);
   testMode.setB32Features(feature);
   testMode.setB32MaskFeatures(feature_mask);
   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_FEATUREStart(*this, testMode)));
}


void RadioSetting::vUpdateTAButtonInSettings(bool TA)
{
   ETG_TRACE_USR4(("RadioSetting::vUpdateTAButtonInSettings() %d", TA));

   //m_baCheckBoxStatus[1] = m_bOptRadioText;
   m_baCheckBoxStatus[2] = TA;
   //m_baCheckBoxStatus[3] = m_u8KDSConfigREG_Enable;
   //m_baCheckBoxStatus[4] = m_u8KDSConfigAF_Enable;
   //m_baCheckBoxStatus[6] = m_bOptDabDynamicText;
   //m_baCheckBoxStatus[7] = m_bOptDabFmFollowing;
   //ListRegistry::s_getInstance().updateList(LIST_ID_TUNER_RADIO_SETTING);
   updateSettingsList();
}


//weather alert
void RadioSetting::vUpdateWBButtonInSettings(bool WB)
{
   ETG_TRACE_USR4(("RadioSetting::vUpdateWBButtonInSettings() %d", WB));

   //m_baCheckBoxStatus[1] = m_bOptRadioText;
   m_baCheckBoxStatus[9] = WB;
   //m_baCheckBoxStatus[3] = m_u8KDSConfigREG_Enable;
   //m_baCheckBoxStatus[4] = m_u8KDSConfigAF_Enable;
   //m_baCheckBoxStatus[6] = m_bOptDabDynamicText;
   //m_baCheckBoxStatus[7] = m_bOptDabFmFollowing;
   //ListRegistry::s_getInstance().updateList(LIST_ID_TUNER_RADIO_SETTING);
   updateSettingsList();
}


void RadioSetting::ReadRadioSettingKDS()
{
   dp_tclKdsAMFMTunerParameter oAMFMTunerPara;
   tU8 u8Temp = 0;
   //RDS enable/Disable
   oAMFMTunerPara.u8GetRDS(u8Temp);
   bool m_bRDSConfigEnable = u8Temp;

   oAMFMTunerPara.u8GetAF(u8Temp);//AF in the Option AM/FM view
   m_u8KDSConfigAF_Enable = u8Temp;

   oAMFMTunerPara.u8GetREG(u8Temp);// REG in the Option AM/FM view
   m_u8KDSConfigREG_Enable = u8Temp;

   oAMFMTunerPara.u8GetTA(u8Temp);// TA in the Option AM/FM view
   m_u8KDSConfigTA_Enable = u8Temp;

   //@ded6kor
   dp_tclAppHmi_TunerDPRadiotext obj;
   tS32 s32Status = obj.tGetData();
   m_bOptRadioText = s32Status;
   ETG_TRACE_USR4(("s32Status Radio Txt = %d,%d ", s32Status, m_bOptRadioText));

   dp_tclAppHmi_TunerDAB_DynamicText objDabDynText;
   s32Status = objDabDynText.tGetData();
   bool m_bOptDabDynamicText = s32Status;
   ETG_TRACE_USR4(("s32Status DAB Dyntxt = %d,%d ", s32Status, m_bOptDabDynamicText));

   dp_tclAppHmi_TunerDAB_FMFollowing objDabFmLinking;
   s32Status = objDabFmLinking.tGetData();
   bool m_bOptDabFmFollowing = s32Status;
   ETG_TRACE_USR4(("s32Status DAB FM Follow = %d,%d ", s32Status, m_bOptDabFmFollowing));

   dp_tclAppHmi_TunerDAB_DABLinking objDabDabLinking;
   s32Status = objDabDabLinking.tGetData();
   bool m_bOptDabDabFollowing = s32Status;
   ETG_TRACE_USR4(("s32Status= %d,%d DAB DAB link ", s32Status, m_bOptDabDabFollowing));

   dp_tclAppHmi_TunerPtymask objptymask;
   s32Status = objptymask.tGetData();
   m_u32DabPtyMask = s32Status;
   ETG_TRACE_USR4(("s32Status= %d DabPty Mask", s32Status));

   m_baCheckBoxStatus[1] = m_bOptRadioText;
   m_baCheckBoxStatus[2] = m_u8KDSConfigTA_Enable;
   m_baCheckBoxStatus[3] = m_u8KDSConfigREG_Enable;
   m_baCheckBoxStatus[4] = m_u8KDSConfigAF_Enable;
   m_baCheckBoxStatus[6] = m_bOptDabDynamicText;
   m_baCheckBoxStatus[7] = m_bOptDabFmFollowing;

   ETG_TRACE_USR4(("ReadRadioSettingKDS m_baCheckBoxStatus[0]= %d, m_bOptRadioText %d  ", m_baCheckBoxStatus[0], m_bOptRadioText));
}


/*This fuction is used to interate the DATPTYList and display the check box disable state(checked/unchecked)
based on the previous state. Here 0->unchecked enabled 1-->checked enabled 2-->uchecked disable 3-->checked disable
The above images are mapped in composite in scene composer the above listed order*/

void RadioSetting::HandleDABPTYOptionsEnableDisable(bool optenabledis)
{
   ETG_TRACE_USR4(("RadioSetting::HandleDABPTYOptionsEnableDisable() optenabledis %d", optenabledis));
   if (optenabledis == true)
   {
      for (int i = 1; i <= 5; i++)
      {
         if (m_ui32DABPTYCheckBoxStatus[i] == CHECKED_DISABLE)
         {
            m_ui32DABPTYCheckBoxStatus[i] = true; // CHECKED_ENABLE
            m_bDABPTYLlistUpdateRequired = true;
         }
         else if (m_ui32DABPTYCheckBoxStatus[i] == UNCHECKED_DISABLE)
         {
            m_ui32DABPTYCheckBoxStatus[i] = false; // CHECKED_DISABLE
            m_bDABPTYLlistUpdateRequired = true;
         }
         ETG_TRACE_USR4(("RadioSetting::HandleDABPTYOptionsEnableDisable() m_ui32DABPTYCheckBoxStatus[%d] %d", i, m_ui32DABPTYCheckBoxStatus[i]));
      }
   }
   else if (optenabledis == false)
   {
      for (int i = 1; i <= 5; i++)
      {
         if (m_ui32DABPTYCheckBoxStatus[i] == true) // CHECKED_ENABLE
         {
            m_ui32DABPTYCheckBoxStatus[i] = CHECKED_DISABLE;
            m_bDABPTYLlistUpdateRequired = true;
         }
         else if (m_ui32DABPTYCheckBoxStatus[i] == false) // CHECKED_DISABLE
         {
            m_ui32DABPTYCheckBoxStatus[i] = UNCHECKED_DISABLE;
            m_bDABPTYLlistUpdateRequired = true;
         }
         //HandleDABPTYButtonListItemUpdMsg(0);//added by suman
         vSetDABPtyFilter(0);
         ETG_TRACE_USR4(("RadioSetting::HandleDABPTYOptionsEnableDisable() m_ui32DABPTYCheckBoxStatus[%d] %d", i, m_ui32DABPTYCheckBoxStatus[i]));
      }
   }
   if (m_bDABPTYLlistUpdateRequired)
   {
      ListRegistry::s_getInstance().updateList(LIST_ID_TUNER_DABPTY);
   }
}


void RadioSetting::InitDABPTYDisableForFirstTime()
{
   ETG_TRACE_USR4(("RadioSetting::InitDABPTYDisableForFirstTime_ "));
   m_ui32DABPTYCheckBoxStatus[0] = m_u32DabPtyMask & (1 << EN_PTY_FILTER_ENABLED);
   m_ui32DABPTYCheckBoxStatus[1] = m_u32DabPtyMask & (1 << EN_PTY_TYPE_NEWS);
   m_ui32DABPTYCheckBoxStatus[2] = m_u32DabPtyMask & (1 << EN_PTY_TYPE_CULTURE);
   m_ui32DABPTYCheckBoxStatus[3] = m_u32DabPtyMask & (1 << EN_PTY_TYPE_SPORTS);
   m_ui32DABPTYCheckBoxStatus[4] = m_u32DabPtyMask & (1 << EN_PTY_TYPE_MUSIC);
   m_ui32DABPTYCheckBoxStatus[5] = m_u32DabPtyMask & (1 << EN_PTY_TYPE_SPECIAL);

   ETG_TRACE_USR4(("RadioSetting::InitDABPTYDisableForFirstTime_ m_ui32DABPTYCheckBoxStatus[0] %d", m_ui32DABPTYCheckBoxStatus[0]));

   if (m_ui32DABPTYCheckBoxStatus[0] == true)  // white checkbox ( checked or unchecked )
   {
      m_ui32DABPTYCheckBoxStatus[0] = 1;
      for (int i = 1; i <= 5; i++)
      {
         m_ui32DABPTYCheckBoxStatus[i] ? m_ui32DABPTYCheckBoxStatus[i] = 1 : m_ui32DABPTYCheckBoxStatus[i] = 0;
         ETG_TRACE_USR4(("RadioSetting:: white checkbox_ %d", m_ui32DABPTYCheckBoxStatus[i]));
      }
   }
   else
   {
      m_ui32DABPTYCheckBoxStatus[0] = 0;  // grey checkbox(checked or unchecked)
      for (int i = 1; i <= 5; i++)
      {
         m_ui32DABPTYCheckBoxStatus[i] ? m_ui32DABPTYCheckBoxStatus[i] = 3 : m_ui32DABPTYCheckBoxStatus[i] = 2;
         ETG_TRACE_USR4(("RadioSetting:: grey checkbox_ %d", m_ui32DABPTYCheckBoxStatus[i]));
      }
   }
   if (m_ui32DABPTYCheckBoxStatus[0] == true)
   {
      (*oDabOptionsData).misChecked_EnablePTYFilter = true;
   }
   else
   {
      (*oDabOptionsData).misChecked_EnablePTYFilter = false;
   }
   oDabOptionsData.MarkAllItemsModified();
   oDabOptionsData.SendUpdate();
}


void RadioSetting::updateSettingsList()
{
   ETG_TRACE_USR4(("RadioSetting:: updateSettingsList %d", sm_gCurrentView));
   if (VIEW_OPT_FMAM == sm_gCurrentView)
   {
      ETG_TRACE_USR4(("RadioSetting:: updateSettingsList in Settings Screen"));
      ListRegistry::s_getInstance().updateList(LIST_ID_TUNER_RADIO_SETTING);
   }
}


void RadioSetting::vUpdate_Settings_Options(bool status)
{
   m_bOptRadioText = status;
   m_baCheckBoxStatus[1] = status;
   m_baCheckBoxStatus[7] = status;
   ETG_TRACE_USR4(("RadioSetting:: vUpdate_RadioText_Settings %d", m_bOptRadioText));
   TunerDataBindingHandler::getDataBindingHandlerInstance()->TunerPLMsg(EV_MSG_TO_PL_OPT_RADIO_TEXT, m_bOptRadioText, "");
   TunerDataBindingHandler::getDataBindingHandlerInstance()->TunerPLMsg(EV_MSG_TO_PL_OPT_DAB_FM_LINKING, m_bOptDabFmFollowing, "");
}


}
}
