/**************************************************************************************
* @file         : DabInfo.cpp
* @author       : ECV2-Priyanka B
* @addtogroup   : Tuner HALL
* @brief        : Class contains necessary business logic required to handle DAB data.
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "DabInfo.h"
#include "App/Core/Tuner_Defines.h"
#include "CourierTunnelService/CourierMessageReceiverStub.h"
#include "AppBase/StartupSync/StartupSync.h"
#include "App/Core/Utility/TunerUtility.h"
#include "App/Core/PresetListHandler/PresetListHandlingIF.h"
#include "midw_ext_dabtuner_fi_types.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "App/Core/LanguageDefines.h"
#include "App/Core/Utility/TunerCommonHandler.h"
#include "CgiExtensions/ImageLoader.h"
#include "AMFMTestMode.h"
#include "App/Core/PresetListHandler/PresetListHandling.h"
#include "App/Core/AMFMInfo/AMFMStationList.h"
#include "AMFMInfo.h"
#include "MIDW_TUNER_FIProxy.h"


#define IS_DAB_VALID() ((_dabFiProxy != NULL))

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_TUNER_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_TUNER
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_Tuner_"
#define ETG_I_FILE_PREFIX                 App::Core::DabInfo::
#include "trcGenProj/Header/DabInfo.cpp.trc.h"
#endif
using namespace ::MIDW_EXT_DABTUNER_FI;
using namespace ::midw_ext_dabtuner_fi_types;


namespace App {
namespace Core {

//Array for PTY DAB
static const char* const ptyDab[] = { "", "NEWS", "AFFAIRS", "INFO", "SPORT", "EDUCATE", "DRAMA", "CULTURE",
                                      "SCIENCE", "VARIED", "POP M", "ROCK M", "EASY", "CLASSICS", "CLASSICS",
                                      "OTHER_M", "WEATHER", "FINANCE", "CHILDREN", "FACTUAL", "RELIGION", "PHONE_IN",
                                      "TRAVEL", "LEISURE", "JAZZ", "COUNTRY", "NATURAL_M", "OLDIES", "FOLK", "DOCUMENT", "", ""
                                    };

DabInfo* DabInfo::m_poDabInfo = NULL;

/* Description     : Constructor of DabInfo */
DabInfo::DabInfo() :
   _dabFiProxy(MIDW_EXT_DABTUNER_FIProxy::createProxy("dabFiPort", *this))
   , _DabInfoDataBinder(NULL)
   , _dabInfoCourierMsg(NULL)
   , enDabStatus(EN_DAB_ALL_OK) //added for coverity may 4
{
   ETG_TRACE_USR3(("DABInfo:: Constructor called"));
   if (_dabFiProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _dabFiProxy->getPortName());
   }
   ETG_I_REGISTER_FILE();
   _dabInfoCourierMsg = new DabInfoCourierMsgHandler(this, DABStationList::poGetInstance());
   _DabInfoDataBinder = TunerDataBindingHandler::getDataBindingHandlerInstance();
   _isPTYSignalOff = false;
   _presetID = 0;
   _dabPresetID = 0;
   _bankId = 0;
}


/**
* Description     : Destructor of DabInfo
*/
DabInfo::~DabInfo()
{
   _dabFiProxy.reset();
   ETG_I_UNREGISTER_FILE();

   if (NULL != _DabInfoDataBinder)
   {
      delete _DabInfoDataBinder;
      _DabInfoDataBinder = NULL;
   }

   if (NULL != _dabInfoCourierMsg)
   {
      delete _dabInfoCourierMsg;
      _dabInfoCourierMsg = NULL;
   }
   //_dabList = NULL;
}


/**
* Description     : Request from Framework to start registering for properties.
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : stateChange : represents a state transition of a service
*/
void DabInfo::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR3(("DABInfo::registerProperties "));
}


/**
* Description     : Request from Framework to start de-registering for properties.
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : stateChange : represents a state transition of a service
*/
void DabInfo::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/, const asf::core::ServiceStateChange& /*stateChange*/)
{
}


/**
* Description     : Request from Framework to start de-registering for properties.
*
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : stateChange : represents a state transition of a service
* @return         : void
*/
void DabInfo::onAvailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3(("DABInfo::onAvailable()"));

   if ((proxy == _dabFiProxy) && (_dabFiProxy.get()))
   {
      _dabFiProxy->sendFID_DAB_G_ACTIVE_SERVICE_INFOUpReg(*this);
      _dabFiProxy->sendFID_DAB_G_CHANNEL_INFOUpReg(*this);
      _dabFiProxy->sendFID_DAB_G_ACTIVE_ENSEMBLE_INFOUpReg(*this);
      _dabFiProxy->sendFID_DAB_G_PRESET_LISTUpReg(*this);
   }
}


/**
* Description     : Request from Framework to start de-registering for properties.
*
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : stateChange : represents a state transition of a service
* @return         : void
*/
void DabInfo::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   if ((proxy == _dabFiProxy) && (_dabFiProxy.get()))
   {
      _dabFiProxy->sendFID_DAB_G_ACTIVE_SERVICE_INFORelUpRegAll();
      _dabFiProxy->sendFID_DAB_G_CHANNEL_INFORelUpRegAll();
      _dabFiProxy->sendFID_DAB_G_ACTIVE_ENSEMBLE_INFORelUpRegAll();
      _dabFiProxy->sendFID_DAB_G_PRESET_LISTRelUpRegAll();
   }
}


/**
* Description     : Virtual function implemented to get update of FID_DAB_G_ACTIVE_SERVICE_INFO Error
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
* @param[in]      : error : The error message of 'FID_DAB_G_ACTIVE_SERVICE_INFO'
*/
void DabInfo::onFID_DAB_G_ACTIVE_SERVICE_INFOError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_DAB_G_ACTIVE_SERVICE_INFOError >& /*error*/)
{
}


/**
* Description     : Virtual function implemented to get update of FID_DAB_G_ACTIVE_SERVICE_INFO Status
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
* @param[in]      : status : The status message of the property "FID_DAB_G_ACTIVE_SERVICE_INFO"
*/
void DabInfo::onFID_DAB_G_ACTIVE_SERVICE_INFOStatus(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_DAB_G_ACTIVE_SERVICE_INFOStatus >& status)
{
   ETG_TRACE_USR3(("DABInfo:: onFID_DAB_G_ACTIVE_SERVICE_INFO Status "));

   T_DAB_ServiceInfo serviceInfo = status->getServiceInfo();
   uint8 index = serviceInfo.getU8PTYCode();
   std::string category = "";

   if (index < 32 && !_isPTYSignalOff)
   {
      category = ptyDab[index];
   }
   if (NULL != _DabInfoDataBinder)
   {
      //_DabInfoDataBinder->setCategory(category);
   }
   else
   {
      ETG_TRACE_USR3(("DABInfo:: Databinder null pointer "));
   }
}


/**
* Description     : Virtual function implemented to get update of FID_DAB_G_CHANNEL_INFO Error
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
* @param[in]      : error : The error message of 'FID_DAB_G_CHANNEL_INFO'
*/
void DabInfo::onFID_DAB_G_CHANNEL_INFOError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_DAB_G_CHANNEL_INFOError >& /*error*/)
{
}


/**
* Description     : Virtual function implemented to get update of FID_DAB_G_CHANNEL_INFO Status
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
* @param[in]      : status : The status message of the property "FID_DAB_G_CHANNEL_INFO"
*/
void DabInfo::onFID_DAB_G_CHANNEL_INFOStatus(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_DAB_G_CHANNEL_INFOStatus >& status)
{
   ETG_TRACE_USR3(("DABInfo:: onFID_DAB_G_CHANNEL_INFOStatus"));
   const T_DAB_ChannelInfo& channelInfo = status->getChannelInfo();
   _presetID = channelInfo.getU8PresetListElementID();

   std::string presetText = "DAB";
   std::string serviceLabel = getServiceLabel(channelInfo.getBFmServiceFollowActive(), channelInfo.getServiceLabel().getSLabel(), channelInfo.getServiceLabel().getSLabelShort());
   //m_servicename = getServiceLabel(channelInfo.getBFmServiceFollowActive(), channelInfo.getServiceLabel().getSLabel(), channelInfo.getServiceLabel().getSLabelShort());
   m_servicename =  channelInfo.getServiceLabel().getSLabel();
   std::string shortServLabel = channelInfo.getServiceLabel().getSLabelShort();
   uint16 serviceListId = channelInfo.getU16FrozenListElementID();
   T_b16_DAB_AnnouncementType annoType = channelInfo.getB16AvailableAnnoTypesMask();
   ETG_TRACE_USR3(("DABStationList : serviceListId == %d", serviceListId));

   if (channelInfo.getBComponentMode())
   {
      sm_gIsDABComponentMode = true;
   }
   else
   {
      sm_gIsDABComponentMode = false;
   }

   if (VIEW_DAB_SL == sm_gCurrentView)
   {
      ETG_TRACE_USR3(("To Check for DAB Station List"));
      AMFMStationList::poGetInstance()->updateList(LISTID_DAB);
   }

   if (_DabInfoDataBinder != NULL)
   {
      setDLSelements(status);
      updateChannelData(serviceLabel, shortServLabel, _presetID);
      if (T_e8_Band__TUN_MSTR_BAND_DAB == AMFMInfo::poGetInstance()->e8GetCurrentBand())
      {
         _DabInfoDataBinder->vUpdate_DAB_PresetFocused(_presetID);   			//Sends the Preset focus element number
      }
   }
   ETG_TRACE_USR4(("DAB Preset Id: %d", _dabPresetID));
   ETG_TRACE_USR4(("_presetID: %d", _presetID));
   AMFMTestMode::poGetInstance()->setChannelInfoDataToTestMode(status);
   if (_dabPresetID != _presetID)
   {
      _dabPresetID = _presetID;
      if (_aDAB_Presets.size())
      {
         if (_DabInfoDataBinder != NULL)
         {
            _DabInfoDataBinder->updateDABpresets(_aDAB_Presets);
         }
      }
   }
}


/**
* Description     : Virtual function implemented to get update of FID_DAB_G_PRESET_LIST Error
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
* @param[in]      : error : The error message of 'FID_DAB_G_PRESET_LIST'
*/
void DabInfo::onFID_DAB_G_PRESET_LISTError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_DAB_G_PRESET_LISTError >& /*error*/)
{
}


/**
* Description     : Virtual function implemented to get update of FID_DAB_S_ACTIVATE_PRESET_LIST_ELEM Error
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
* @param[in]      : status : The status message of the property "FID_DAB_S_ACTIVATE_PRESET_LIST_ELEM"
*/
void DabInfo::onFID_DAB_S_ACTIVATE_PRESET_LIST_ELEMError(const ::boost::shared_ptr< ::MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr<  ::MIDW_EXT_DABTUNER_FI::FID_DAB_S_ACTIVATE_PRESET_LIST_ELEMError >& /*error*/) {}

/**
* Description     : Virtual function implemented to get update of FID_DAB_S_ACTIVATE_PRESET_LIST_ELEM Result
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
* @param[in]      : status : The status message of the property "FID_DAB_S_ACTIVATE_PRESET_LIST_ELEM"
*/
void DabInfo::onFID_DAB_S_ACTIVATE_PRESET_LIST_ELEMResult(const ::boost::shared_ptr< ::MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr<  ::MIDW_EXT_DABTUNER_FI::FID_DAB_S_ACTIVATE_PRESET_LIST_ELEMResult >& /*result*/) {}

/**
* Description     : Virtual function implemented to get update of FID_DAB_G_PRESET_LIST Status
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
* @param[in]      : status : The status message of the property "FID_DAB_G_PRESET_LIST"
*/
void DabInfo::onFID_DAB_G_PRESET_LISTStatus(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_DAB_G_PRESET_LISTStatus >& status)
{
   ETG_TRACE_USR3(("DABInfo:: onFID_DAB_G_PRESET_LISTStatus"));
   clearDABPresetList();
   _dabPresetList = status->getPresetList();

   uint32 dabPresetListSize = _dabPresetList.size();
   //::std::vector<std::string> aDAB_Presets;
   _aDAB_Presets.clear();

   for (uint itr = 0; itr < dabPresetListSize; itr++)
   {
      midw_ext_dabtuner_fi_types::T_DAB_PresetListElement& listElement = _dabPresetList[itr];
      const midw_ext_dabtuner_fi_types::T_DAB_Label& labelInfo = listElement.getServiceLabel();
      std::string shortServLabel = labelInfo.getSLabelShort();
      ETG_TRACE_USR3(("DABInfo:: Dab PresetList Status Name %s", _dabPresetList[itr].getServiceLabel().getSLabelShort().c_str()));
      //aDAB_Presets.push_back(shortServLabel);
      _aDAB_Presets.push_back(shortServLabel);
   }
   if (_aDAB_Presets.size())
   {
      if (_DabInfoDataBinder != NULL)
      {
         _DabInfoDataBinder->updateDABpresets(_aDAB_Presets);
      }
   }
}


void DabInfo::updateDabPresetData()
{
   ETG_TRACE_USR3(("DABInfo:: onFID_DAB_G_PRESET_LISTStatus"));
   if (_aDAB_Presets.size())
   {
      if (_DabInfoDataBinder != NULL)
      {
         _DabInfoDataBinder->updateDABpresets(_aDAB_Presets);
      }
   }
}


/**
* Description     : Virtual function implemented to get update of FID_DAB_G_ACTIVE_ENSEMBLE_INFO Error
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
* @param[in]      : error : The error message of 'FID_DAB_G_ACTIVE_ENSEMBLE_INFO'
* @return         : void
*/
void DabInfo::onFID_DAB_G_ACTIVE_ENSEMBLE_INFOError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_DAB_G_ACTIVE_ENSEMBLE_INFOError >& /*error*/)
{
   ETG_TRACE_USR3(("DABInfo:: onFID_DAB_G_ACTIVE_ENSEMBLE_INFOError"));
}


/**
* Description     : Virtual function implemented to get update of FID_DAB_G_ACTIVE_ENSEMBLE_INFO Status
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
* @param[in]      : status : The status message of the property "FID_DAB_G_ACTIVE_ENSEMBLE_INFO"
* @return         : void
*/
void DabInfo::onFID_DAB_G_ACTIVE_ENSEMBLE_INFOStatus(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_DAB_G_ACTIVE_ENSEMBLE_INFOStatus >& status)
{
   ETG_TRACE_USR3(("DABInfo:: onFID_DAB_G_ACTIVE_ENSEMBLE_INFOStatus"));
   const T_DAB_EnsembleInfo& ensembleInfo = status->getEnsembleInfo();
   std::string ensembleLabel = ensembleInfo.getEnsembleLabel().getSLabel();
   std::string shortEnsembleLabel = ensembleInfo.getEnsembleLabel().getSLabelShort();
   uint32 ensembleId = ensembleInfo.getU32EnsembleId();
   uint8 noofAudioServicesComp = ensembleInfo.getU8NumberOfAudioServiceComponents();
   uint8 noofDataServicesComp = ensembleInfo.getU8NumberOfDataServiceComponents();
   uint8 noofEnsembles = ensembleInfo.getU8NumberOfEnsembles();
   uint8 noofAudioServices = ensembleInfo.getU8NumberOfAudioServices();
   uint8 noofDataServices = ensembleInfo.getU8NumberOfDataServices();

   ETG_TRACE_USR3(("DABInfo:: ensembleLabel== %s", ensembleLabel.c_str()));
   ETG_TRACE_USR3(("DABInfo:: ensembleLabel== %s", shortEnsembleLabel.c_str()));

   if (_DabInfoDataBinder != NULL)
   {
      _DabInfoDataBinder->setEnsembleName(ensembleLabel);
   }
}


/************************************************************************
* Description     : Updates the selected channel data
* @param[in]      : serviceLabel: present service label
* @param[in]      : radioText: present radio text
* @param[in]      : presetNum: selected preset number
************************************************************************/
void DabInfo::updateChannelData(std::string serviceLabel, std::string shortServLabel, uint8 presetNum)const
{
   //serviceLabel = "Brain Adams";
   //shortServLabel = "Justin";

   ETG_TRACE_USR3(("DABInfo:: Service Name== %s", serviceLabel.c_str()));
   ETG_TRACE_USR3(("DABInfo:: Service Name== %s", shortServLabel.c_str()));

   if (_DabInfoDataBinder != NULL)
   {
      uint8 presetIndex = PRESET_INVISIBLE;
      std::ostringstream presetNumberText;
      if (presetNum != 0)
      {
         presetIndex = PRESET_VISIBLE;
         presetNumberText << TunerUtility::u16ConversionToString(presetNum);
      }
      _DabInfoDataBinder->updateDabChannelData(serviceLabel, shortServLabel, presetNumberText.str(), presetIndex);
   }
}


/************************************************************************
*FUNCTION: 		getstationvalue
*DESCRIPTION: 	Returns the currently tuned stationname in DAB
************************************************************************/

std::string DabInfo::getstationvalue()
{
   return m_servicename;
}


/************************************************************************
*DESCRIPTION		:Returns Valid Radio Text/Advisories
*RETURNVALUE		: std::string
************************************************************************/
void DabInfo::setDLSelements(const boost::shared_ptr< FID_DAB_G_CHANNEL_INFOStatus >& status)
{
   const T_DAB_ChannelInfo& channelInfo = status->getChannelInfo();
   const T_e8_DAB_ChannelInfoState& dabState = channelInfo.getEnState();
   uint32 switchIndex = DABADVISORIES;
   std::string radioText = TunerUtility::getSecureString(Candera::String(TextId_New_Audio_9119_No_info));
   std::string advisoryText = "";
   _isPTYSignalOff = false;
   enDabStatus = EN_DAB_ALL_OK;

   if (channelInfo.getBFmServiceFollowActive())
   {
      enDabStatus = EN_DAB_FM_LINK;
   }
   if (channelInfo.getBComponentMode())
   {
      enDabStatus = EN_DAB_SECOND_AUDIO;
   }
   ETG_TRACE_USR3(("DABInfo:: Dab State ==%d", dabState));
   switch (dabState)
   {
      case T_e8_DAB_ChannelInfoState__DAB_SELECTING_WAIT:
      case T_e8_DAB_ChannelInfoState__DAB_SELECTING_SEARCH:
      {
         advisoryText = TunerUtility::getSecureString(Candera::String(TextId_IT_05016_loading_advisory));
         enDabStatus = EN_DAB_LOADING;
         _isPTYSignalOff = true;
         ETG_TRACE_USR3(("DABInfo:: Dab Search"));
         break;
      }
      case T_e8_DAB_ChannelInfoState__DAB_UNAVAIL:
      case T_e8_DAB_ChannelInfoState__DAB_INVALID:
      {
         advisoryText = TunerUtility::getSecureString(Candera::String(TextId_New_Audio_9118_Not_available_advisory));
         enDabStatus = EN_DAB_NO_RECEPTION;
         _isPTYSignalOff = true;
         ETG_TRACE_USR3(("DABInfo:: Dab Not Available"));
         break;
      }
      case T_e8_DAB_ChannelInfoState__DAB_STABLE:
      {
         ETG_TRACE_USR3(("DABInfo:: Dab Stable"));
         if (channelInfo.getRRadioText().getBValid())
         {
            radioText = channelInfo.getRRadioText().getSRadioTextData();
            if (!radioText.empty())
            {
               switchIndex = RADIOSWITCHTEXT; //added for compiler,coverity may 11
            }
         }
         break;
      }

      default:
         break;
   }

   if (NULL != _DabInfoDataBinder)
   {
      _DabInfoDataBinder->setDLSelements(advisoryText, radioText, switchIndex, (tU8)enDabStatus);
   }
}


/************************************************************************
*FUNCTION			:getServiceLabel
*DESCRIPTION		:Returns Valid service label
*RETURNVALUE		: std::string
************************************************************************/
std::string DabInfo::getServiceLabel(bool isServiceFollowActive, std::string servLabel, std::string shortservicelabel)
{
   if (isServiceFollowActive)
   {
      servLabel = (servLabel.size() > 10) ? (shortservicelabel.append("(FM)")) : (servLabel.append("(FM)"));
   }
   return servLabel;
}


/************************************************************************
*DESCRIPTION		: Clear's the DAB preset list
*RETURNVALUE		: Void
************************************************************************/
void DabInfo::clearDABPresetList()
{
   uint32 dabPresetListSize = _dabPresetList.size();
   for (uint itr = 0; itr < dabPresetListSize; itr++)
   {
      midw_ext_dabtuner_fi_types::T_DAB_PresetListElement& listElement = _dabPresetList[itr];
      listElement.clear();
   }
   _dabPresetList.clear();
}


void DabInfo::onTunerTrackCommandReq(uint32 u8tunertrackCmd)
{
   ETG_TRACE_USR4(("DabInfo::onTunerTrackCommandReq u8tunertrackCmd: %d", u8tunertrackCmd));

   switch (u8tunertrackCmd)

   {
      case TrackCommand_Next:
      {
         POST_MSG((COURIER_MESSAGE_NEW(NextIconSelectMsg)()));
         POST_MSG((COURIER_MESSAGE_NEW(DABServiceReqMsg)(TUN_INC)));
      }
      break;
      case TrackCommand_Previous:
      {
         POST_MSG((COURIER_MESSAGE_NEW(PreviousIconSelectMsg)()));
         POST_MSG((COURIER_MESSAGE_NEW(DABServiceReqMsg)(TUN_DEC)));
      }
      break;
      default:
      {
         ETG_TRACE_USR1(("DabInfo::onTunerTrackCommandReq Default case "));
      }

      break;
   }
}


///*************************************************
//* TTFis Trace commands
//*************************************************/

///**
//* Description     : Trace command to call onPresetChange
//* @param[in]      : Preset number
//*/
//ETG_I_CMD_DEFINE((TraceCmd_onPresetChange, "PresetChange %d", int))
//	void DabInfo::TraceCmd_onPresetChange(uint8 presetNum)
//{
//	if (_dabInfoInst != NULL)
//	{
//		_dabInfoInst->updateChannelData(EMPTY_STRING, EMPTY_STRING, presetNum);
//	}
//}


///**
//* Description     : Trace command to call onPresetChange
//* @param[in]      :  Direction either up or down
//*/
//ETG_I_CMD_DEFINE((TraceCmd_onPresetUpDown, "PresetUpDown %d", int))
//	void DabInfo::TraceCmd_onPresetUpDown(uint8 direction)
//{
//	if (_dabInfoInst != NULL)
//	{
//		_dabInfoInst->presetUpDown(direction);
//	}
//}


///**
//* Description     : Trace command to call ensembleSeek
//* @param[in]      : Direction either up or down
//*/
//ETG_I_CMD_DEFINE((TraceCmd_onEnsembleSeek, "ensembleSeek %d", int))
//	void DabInfo::TraceCmd_onEnsembleSeek(uint8 direction)
//{
//	if (_dabInfoInst != NULL)
//	{
//		_dabInfoInst->ensembleSeek(direction);
//	}
//}
} // namespace Core
} // namespace App
