/**************************************************************************************
* @file         : DABStationList.cpp
* @author       : ECV2-Avinash B Guledagudda
* @addtogroup   : Tuner HALL
* @brief        : Class contains necessary business logic required to handle DAB List data.
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/
#include "hall_std_if.h"
#include "DABStationList.h"
//#include "DabInfoDataBindingHandler.h"
#include "AppBase/StartupSync/StartupSync.h"
#include "List/Data/ListDataProviderDistributor.h"
#include "CourierTunnelService/CourierMessageReceiverStub.h"
#include "App/Core/Utility/TunerUtility.h"
#include "App/Core/AMFMInfo/AMFMStationListTimer.h"
#include "App/Core/Utility/TunerCommonHandler.h"
#include "App/Core/LanguageDefines.h"
#include "DabInfo.h"
#include "TunerDataBindingHandler.h"

#define SEEK_POPUP_TIMEOUT_VALUE 2000
#define AUDIO_UNMUTEPOPUP_TIMEOUT_VALUE 4000
#define APPSTATUS_DEFAULT 0

#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1_R
#define ENTRY_ANIM_TIMER 250
#endif


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_TUNER_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_TUNER
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_Tuner_"
#define ETG_I_FILE_PREFIX                 App::Core::DABStationList::
#include "trcGenProj/Header/DABStationList.cpp.trc.h"
#endif


namespace App {
namespace Core {
using namespace ::MIDW_EXT_DABTUNER_FI;
using namespace ::midw_ext_dabtuner_fi_types;
//using namespace ::org::bosch::cm::DataServices;
DABStationList* DABStationList::m_poDABStationList = NULL;
/**
 * Description     : Destructor of DABStationList
 */
DABStationList::~DABStationList()
{
   ETG_TRACE_USR4(("DABStationList:: Destrcutor"));

   ListRegistry::s_getInstance().removeListImplementation(LISTID_DAB);
   clearDabList();
   _dabFiProxy.reset();
   ETG_I_UNREGISTER_FILE();
}


/**
 * Description     : Constructor of DABStationList
 */
DABStationList::DABStationList()
   : _dabFiProxy(MIDW_EXT_DABTUNER_FIProxy::createProxy("dabFiPort", *this))
   , _dab_FmStatus(false)
   , _selectedElemID(0)
   , _tunerStatus(APPSTATUS_DEFAULT)
   , _audioPopUpStatus(false)
   , _seekActiveStatus(false)
   , _dabListStatus(false)
   , _dabLinkingToggled(false)
   , _ensSeekDirection(0)
   , _MWlistStatus(MW_LIST_INSTANCE_CLOSE)
   , bReception(false)
   , m_u32DABPTYFilter(0)
{
   ETG_I_REGISTER_FILE();
   if (_dabFiProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _dabFiProxy->getPortName());
   }
   else
   {
      ETG_TRACE_USR3(("DABStationList : startupSync failed."));
   }
   ETG_TRACE_USR4(("DABStationList:: Constructor"));
   ListRegistry::s_getInstance().addListImplementation(LISTID_DAB, this);
   clearDabList();
}


/**
 * Description     : Request from Framework to start registering for properties.
 *
 * @param[in]      : proxy : reference to the base class of all proxies
 * @param[in]      : stateChange : represents a state transition of a service
 * @return         : void
 */
void DABStationList::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR3(("DABStationList : onAvailable"));
   if ((proxy == _dabFiProxy) && (_dabFiProxy.get()))
   {
      _dabFiProxy->sendFID_DAB_G_SETUP_STATUSUpReg(*this);
   }
}


/**
* Description     : Request from Framework to start de-registering for properties.
*
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : stateChange : represents a state transition of a service
* @return         : void
*/
void DABStationList::onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   if ((proxy == _dabFiProxy) && (_dabFiProxy.get()))
   {
      _dabFiProxy->sendFID_DAB_G_SETUP_STATUSRelUpRegAll();
   }
}


void DABStationList::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
}


void DABStationList::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
tSharedPtrDataProvider DABStationList::getListDataProvider(const ListDateProviderReqMsg& listDataInfo)
{
   ETG_TRACE_USR3(("DABStationList : getListDataProvider"));
   tSharedPtrDataProvider ptr = tSharedPtrDataProvider();
   switch (listDataInfo.GetListId())
   {
      case LISTID_DAB:
      {
         ptr = getTunerDABServiceList();
         break;
      }
      default:
      {
         break;
      }
   }
   return ptr;
}


bool DABStationList::onCourierMessage(const DABStationListReqMsg& corfoMsg)
{
   ETG_TRACE_USR3(("DABStationList : DABStationListReqMsg"));
   dabListoperation(OPEN);
   _dabListStatus = true;
   return true;
}


/**
 * Description     : Function implemented to populate DAB station list
 * @return         : tSharedPtrDataProvider
 */
tSharedPtrDataProvider DABStationList::getTunerDABServiceList()
{
   StationInfoSourceData item;

   const char* const DAB_SERVICE_LIST_ITEM = "Layer_TouchArea_Service_1";
   ListDataProviderBuilder listBuilder(LISTID_DAB, DAB_SERVICE_LIST_ITEM);

   for (uint16 itr = 0; itr < _dabServiceList.size(); itr++)
   {
      if (strcmp(stationnamearray[itr].c_str(), DabInfo::poGetInstance()->getstationvalue().c_str()) == 0)
      {
         ETG_TRACE_USR3(("Icon highlight"));
         item.mDAB_Icon_station = 1;
      }
      else
      {
         ETG_TRACE_USR3(("Icon not  highlight"));
         item.mDAB_Icon_station = 0;
      }

      ETG_TRACE_USR4(("getTunerDABServiceList : At pos %d Label= %s", itr, _dabServiceList[itr]._psName.c_str()));
      listBuilder.AddItem(itr, 0, DAB_SERVICE_LIST_ITEM).AddData(_dabServiceList[itr]._psName.c_str()).AddData(_dabServiceList[itr]._catName.c_str()).AddData(_dabServiceList[itr]._isPrestListElementId.c_str()).AddDataBindingUpdater<StationInfoSourceDataBindingSource>(item);
   }
   return listBuilder.CreateDataProvider();
}


/**
 * Description     : Virtual function implemented to get update of FID_DAB_G_SETUP_STATUS Status
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
 * @param[in]      : error : The error message of 'FID_DAB_G_SETUP_STATUS'
 * @return         : void
 */

void DABStationList::onFID_DAB_G_SETUP_STATUSStatus(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_DAB_G_SETUP_STATUSStatus >& status)
{
   T_e8_DAB_ServiceLinkingMode serviceLinkingMode = status->getE8ServiceLinkingMode();
   ETG_TRACE_USR4(("serviceLinkingMode: %d", serviceLinkingMode));
   ETG_TRACE_USR4(("_dab_FmStatus: %d", _dab_FmStatus));
   if (!_dabLinkingToggled)
   {
      if (serviceLinkingMode == T_e8_DAB_ServiceLinkingMode__DAB_SERVICELINKING_DAB_FM)
      {
         _dab_FmStatus = true;
      }
      else
      {
         _dab_FmStatus = false;
      }
   }
}


/**
 * Description     : Virtual function implemented to get update of FID_DAB_S_SET_SRV_LIST Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
 * @param[in]      : error : The error message of 'FID_DAB_S_SET_SRV_LIST'
 * @return         : void
 */

void DABStationList::onFID_DAB_G_SETUP_STATUSError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_DAB_G_SETUP_STATUSError >& /*error*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_DAB_G_SETUP_STATUS Status
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
 * @param[in]      : error : The error message of 'FID_DAB_G_TUNER_STATUS'
 * @return         : void
 */

void DABStationList::onFID_DAB_G_TUNER_STATUSStatus(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_DAB_G_TUNER_STATUSStatus >& status)
{
   T_DAB_TunerStatus statusInfo = status->getTunerStatus();

   uint32 ensFrequency = statusInfo.getFreqTuned().getU32Frequency();
   uint32 tuneType = 3;

   if (_seekActiveStatus)
   {
      (_dabFiProxy.get())&&  (_dabFiProxy->sendFID_DAB_S_TEST_OPERATIONStart(*this, T_e8_DAB_TestOperation__TUNE_ENSEMBLE, tuneType, ensFrequency, _ensSeekDirection));
   }
}


/**
 * Description     : Virtual function implemented to get update of FID_DAB_G_TUNER_STATUS Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
 * @param[in]      : error : The error message of 'FID_DAB_G_TUNER_STATUS'
 * @return         : void
 */

void DABStationList::onFID_DAB_G_TUNER_STATUSError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_DAB_G_TUNER_STATUSError >& /*error*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_DAB_S_SET_SRV_LIST Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
 * @param[in]      : error : The error message of 'FID_DAB_S_SET_SRV_LIST'
 * @return         : void
 */
void DABStationList::onFID_DAB_S_SET_SRV_LISTError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_DAB_S_SET_SRV_LISTError >& /*error*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_DAB_S_SET_SRV_LIST Result
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
 * @param[in]      : result : The Result message of the property "FID_DAB_S_SET_SRV_LIST"
 * @return         : void
 */
void DABStationList::onFID_DAB_S_SET_SRV_LISTResult(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_DAB_S_SET_SRV_LISTResult >& result)
{
   ETG_TRACE_USR4(("In onFID_DAB_S_SET_SRV_LISTResult"));
   if (result->getBOpen())
   {
      clearDabList();
      ETG_TRACE_USR4(("In FID_DAB_S_SRV_SELECTResult initially clearing dab list"));
      uint32 itr = 0;
      bool serviceSelected = false;
      _selectedElemID = result->getU16InitialActivatedElementId() - 1;
      //const boost::ptr_vector<midw_ext_dabtuner_fi_types::T_DAB_ChnListElement> channelListElems = result->getChannelList();
      ::std::vector<T_DAB_ChnListElement> channelListElems = result->getChannelList();
      //ETG_TRACE_USR4(("In FID_DAB_S_SRV_SELECTResult getting channel list count %d", result->getChannelList()));
      if (channelListElems.size() != 0)
      {
         if (result->getEnServiceListType() == T_e8_DAB_ServiceListType__GlobalList)
         {
            uint8 ispreset = 0;

            for (itr = 0; itr < channelListElems.size(); itr++)
            {
               ispreset = 0;
               uint16 scid = result->getChannelList().at(itr).getU8CompScids();
               bReception = result->getChannelList().at(itr).getBReception();
               if (scid == 0)
               {
                  serviceSelected = (itr == _selectedElemID);
                  std::string strPsName = channelListElems[itr].getLabel().getSLabel();

                  stationnamearray.push_back(strPsName);
                  uint8 isEPGsupported = result->getChannelList().at(itr).getBEPGSupport();
                  uint32 serviceID = result->getChannelList().at(itr).getU32DABId();
                  uint16 scid = result->getChannelList().at(itr).getU8CompScids();
                  uint32 _isPTYFilter = result->getChannelList().at(itr).getU8PTY();
                  uint8 isPresetlistelementid = result->getChannelList().at(itr).getU8PresetListElemId();
                  std::string presetNumberString = TunerUtility::u8ConversionToString(result->getChannelList().at(itr).getU8PresetListElemId());
                  ETG_TRACE_USR3(("DABStationList:: PresetListElemId ==%d", channelListElems[itr].getU8PresetListElemId()));
                  if (channelListElems[itr].getU8PresetListElemId())
                  {
                     ispreset = 1;
                  }
                  if ((channelListElems[itr].getU8PresetListElemId() == 0) || (channelListElems[itr].getU8PresetListElemId() > 12))
                  {
                     presetNumberString = "";
                  }

                  FeatStd::String Category;
                  std::string dabCat;
                  //////////////////////////////
                  _isPTYFilter = (1 << _isPTYFilter);
                  FeatStd::String str;
                  if (_isPTYFilter & PTY_FILTER_NEWS_MASK)//News ??
                  {
                     FeatStd::TextId id("0x0BA0");
                     FeatStd::String str(id);
                     dabCat = str.GetCString();
                     FeatStd::String Category(id);
                     ETG_TRACE_USR3(("DABStationList::vCreateFileListData News str[%s]", Category.GetCString()));
                     ETG_TRACE_USR3(("DABStationList::vCreateFileListData News dabCat[%s]", dabCat.c_str()));
                     //poNewNode->vSetStringData(2, str.GetCString());
                  }
                  else if (_isPTYFilter & PTY_FILTER_CULTURE_MASK)//Culture ??
                  {
                     FeatStd::TextId id("0x0BA6");
                     FeatStd::String str(id);
                     dabCat = str.GetCString();
                     FeatStd::String Category(id);
                     ETG_TRACE_USR3(("DABStationList::vCreateFileListData Culture str[%s]", Category.GetCString()));
                     //poNewNode->vSetStringData(2, str.GetCString());
                  }
                  else if (_isPTYFilter & PTY_FILTER_SPORTS_MASK)//Sport ??
                  {
                     FeatStd::TextId id("0x0BA3");
                     FeatStd::String str(id);
                     dabCat = str.GetCString();
                     FeatStd::String Category(id);
                     ETG_TRACE_USR3(("DABStationList::vCreateFileListData Sport str[%s]", Category.GetCString()));
                     //poNewNode->vSetStringData(2, str.GetCString());
                  }
                  else if (_isPTYFilter & PTY_FILTER_MUSIC_MASK)//Music ??
                  {
                     FeatStd::TextId id("0x0EA7");
                     FeatStd::String str(id);
                     dabCat = str.GetCString();
                     FeatStd::String Category(id);
                     ETG_TRACE_USR3(("DABStationList::vCreateFileListData Music str[%s]", Category.GetCString()));
                     //poNewNode->vSetStringData(2, str.GetCString());
                  }
                  else if (_isPTYFilter & PTY_FILTER_SPECIAL_MASK)//Special ??
                  {
                     FeatStd::TextId id("0x0EA8");
                     FeatStd::String str(id);
                     dabCat = str.GetCString();
                     FeatStd::String Category(id);
                     ETG_TRACE_USR3(("DABStationList::vCreateFileListData Special str[%s]", Category.GetCString()));
                     //poNewNode->vSetStringData(2, str.GetCString());
                  }
                  else
                  {
                     // Do nothing
                  }
                  ////////////////////////////////////
                  DabServiceList_Elem odabServList_Elem(strPsName, serviceSelected, ispreset, isEPGsupported, serviceID, scid, _isPTYFilter, Category.GetCString(), dabCat, presetNumberString);
                  _dabServiceList.push_back(odabServList_Elem);
               }
            }
            //if two services have same name but different SIDs, then append Ensemble label to them...
            for (itr = 1; itr < channelListElems.size(); itr++)
            {
               if ((channelListElems[itr].getLabel().getSLabel() == channelListElems[itr - 1].getLabel().getSLabel()) && (channelListElems[itr].getU32DABId() != channelListElems[itr - 1].getU32DABId()))
               {
                  _dabServiceList[itr]._psName = _dabServiceList[itr]._psName + " (" + channelListElems[itr].getEnsembleLabel().getSLabelShort() + ")";
                  _dabServiceList[itr - 1]._psName = _dabServiceList[itr - 1]._psName + " (" + channelListElems[itr - 1].getEnsembleLabel().getSLabelShort() + ")";
               }
            }
            updateList(LISTID_DAB);
         }
      }
      dabListoperation(CLOSE);
      _dabListStatus = false;
   }
}


/**
 * Description     : Virtual function implemented to get update of FID_DAB_S_SRV_SELECT Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
 * @param[in]      : error : The error message of 'FID_DAB_S_SRV_SELECT'
 * @return         : void
 */
void DABStationList::onFID_DAB_S_SRV_SELECTError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_DAB_S_SRV_SELECTError >& /*error*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_DAB_S_SRV_SELECT Result
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
 * @param[in]      : result : The Result message of the property "FID_DAB_S_SRV_SELECT"
 * @return         : void
 */
void DABStationList::onFID_DAB_S_SRV_SELECTResult(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_DAB_S_SRV_SELECTResult >& result)
{
   T_e8_DAB_Response response = result->getE8MethodResult();
   ETG_TRACE_USR4(("onFID_DAB_S_SRV_SELECTResult: %d", response));
   if ((response == T_e8_DAB_Response__DAB_RES_OK || response == T_e8_DAB_Response__DAB_RES_INPROCESS) && (_MWlistStatus == MW_LIST_INSTANCE_OPEN) && !_dabListStatus)
   {
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2
      dabListoperation(CLOSE, MW_LIST_INSTANCE_CLOSE); // 1 for close
#endif
   }
   if (result->getE8MethodResult() == 0)
   {
      TunerDataBindingHandler::getDataBindingHandlerInstance()->vOnUpdateSearchNextPreviousIconMsg(0);
   }
   _seekActiveStatus = false;
//   _seekPopUpTimer.start();
}


void DABStationList::onFID_DAB_S_SETUPError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_DAB_S_SETUPError >& /*error*/)
{
}


void DABStationList::onFID_DAB_S_SETUPResult(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_DAB_S_SETUPResult >& result)
{
   T_e8_DAB_Response response = result->getE8MethodResult();
   ETG_TRACE_USR4(("onFID_DAB_S_SETUPResult: %d", response));
   if (result->getE8MethodResult() == T_e8_DAB_Response__DAB_RES_OK)
   {
      _dab_FmStatus = !_dab_FmStatus;
   }
}


void DABStationList::onFID_DAB_S_SEEK_SERVICEError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_DAB_S_SEEK_SERVICEError >& /*error*/)
{
}


void DABStationList::onFID_DAB_S_SEEK_SERVICEResult(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_DAB_S_SEEK_SERVICEResult >& result)
{
   T_e8_DAB_Response response = result->getE8MethodResult();
   ETG_TRACE_USR4(("onFID_DAB_S_SEEK_SERVICEResult: %d", response));

   if (response == 0)
   {
      TunerDataBindingHandler::getDataBindingHandlerInstance()->vOnUpdateSearchNextPreviousIconMsg(0);
   }
   _seekActiveStatus = false;
}


void DABStationList::onFID_DAB_S_SEEK_ENSEMBLEError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_DAB_S_SEEK_ENSEMBLEError >& /*error*/)
{
}


void DABStationList::onFID_DAB_S_SEEK_ENSEMBLEResult(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_DAB_S_SEEK_ENSEMBLEResult >& result)
{
   //
}


bool DABStationList::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   ETG_TRACE_USR3(("DABStationList : onCourierMessage ListDateProviderReqMsg"));
   return ListRegistry::s_getInstance().updateList(oMsg.GetListId(), oMsg.GetStartIndex(), oMsg.GetWindowElementSize());//TODO : Confirm if this should be included after list data is received
}


/**
 * Description     : Function to get the tuner status
 * @param[in]      : appStatus: application status
 * @return         : void
 */

void DABStationList::setTunerAppStatus(uint8& appStatus)
{
   /* _tunerStatus = appStatus;
    if (_tunerStatus == APPSTATUS_FOREGROUND)
    {
       _seekActiveStatus = false;
       if (_audioPopUpStatus)
       {
          hidePopUpData();
       }
    }*/
}


/**
 * Description    : Function to show dab audio popup
 *
 */

void DABStationList::showPopUpData()
{
   //if (_seekActiveStatus)
   //{
   //   POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("TunerPopups#Popups#RADIO__TPOP_AUDIO"))));
   //   //ETG_TRACE_USR4(("popup status: %d", _audioPopUpStatus));
   //   _audioPopUpStatus = true;
   //}
}


/**
 * Description    : Function to show dab audio popup
 *
 */

void DABStationList::hidePopUpData()
{
   //POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Hide, Courier::ViewId("TunerPopups#Popups#RADIO__TPOP_AUDIO"))));
   ////ETG_TRACE_USR4(("popup status: %d", _audioPopUpStatus));
   //_audioPopUpStatus = false;
}


/**
 * Description     : Timer expiry method on timeout
 *
 * @param[in]      : timer, data
 * @param[in]      : asf::core::Timer, boost::shared_ptr<asf::core::TimerPayload>
 * @return         : void
 */
void DABStationList::onExpired(asf::core::Timer& timer, boost::shared_ptr<asf::core::TimerPayload> data)
{
   //If expired, check for the reason and fire SM message
   /* if (((timer.getAct()  == _seekPopUpTimer.getAct()) || (timer.getAct()  == _seekAudioPopUpTimer.getAct())) && (data->getReason() == asf::core::TimerPayload_Reason__Completed))
    {
       if (_audioPopUpStatus)
       {
          hidePopUpData();
       }

   #ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1_R
       if (_NoEpgPopUpFlag)
       {
          POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Hide, Courier::ViewId("TunerPopups#Popups#TUNER__TPOP_NO_EPG"))));
          _NoEpgPopUpFlag = false;
       }
   #endif
    }*/
}


/**
 * Description     : Function to show toast pop-up when tuner is in background and Audio is unmuted.
 *
 * @return         : void
 */
void DABStationList::setEntMuteStatus(bool& entMuteStatus)
{
   /* if (!entMuteStatus)
    {
       POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("TunerPopups#Popups#RADIO__TPOP_AUDIO"))));
       _seekAudioPopUpTimer.start();
       _audioPopUpStatus = true;
    }*/
}


/**
 * Description     : generic function to call the method start for list based on list operation
 * @param[in]      : status: open or close
 * @return         : void
 */

void DABStationList::dabListoperation(uint8 status)
{
   ETG_TRACE_USR4(("dabListoperation sendFID_DAB_S_SET_SRV_LISTStart"));
   ((_dabFiProxy.get()) && (_dabFiProxy->sendFID_DAB_S_SET_SRV_LISTStart(*this,
                            static_cast<T_e8_DAB_ListOperation>(status),
                            T_e8_DAB_ServiceListType__GlobalList,
                            /*PTY filter*/ m_u32DABPTYFilter      /*0x3FFFFFFF*/,
                            /*First element iD*/ 0,
                            /*Num of Elements*/  0XFFFF,
                            T_e8_DAB_ServiceListSortingType__DAB_SERVICELIST_SORTINGTYPE_LABEL,
                            /*ensemble index*/   0)));
}


/**
* Description     : Function called by Setttings when user selects a DAB PTY filters
* @param[in]      : uint32 filter, this value is used to request filter list
* @return         : void
*/

void DABStationList::setDABPTYFilterToMW(uint32 filter)
{
   m_u32DABPTYFilter = filter;
   ETG_TRACE_USR3(("DABStationList:: setDABPTYFilterToMW %d %d", filter, m_u32DABPTYFilter));
}


/**
 * Description     : Function called by GUI when user selects a service from service list
 * @param[in]      : rowID: row ID of the list element
 * @return         : void
 */
void DABStationList::onDabServiceSelect(uint8 rowID)
{
   T_e8_DAB_ChnSelectMode selectMode = T_e8_DAB_ChnSelectMode__DAB_CHNSELECTMODE_LISTID;
   uint32 selIndex = rowID - 1;
   bool negativeSteps = false;
   ETG_TRACE_USR3(("DABStationList:: onDabServiceSelect %d ", selIndex));
   ((_dabFiProxy.get()) && (_dabFiProxy->sendFID_DAB_S_SRV_SELECTStart(*this, T_e8_DAB_ChnSelectMode__DAB_CHNSELECTMODE_PSDI, _dabServiceList.at(rowID)._servID, _dabServiceList.at(rowID)._secondaryServiceId, false)));
}


/**
 * Description     : Function called by GUI when user requests for seek service
 * @param[in]      : dabDirection: next or previous seek
 * @return         : void
 */
void DABStationList::onSeekService(uint8 dabDirection)
{
   uint32 u32Value = 1;

   _seekActiveStatus = true;
   ETG_TRACE_USR3(("DABStationList:: onSeekService %d", dabDirection));
   ((_dabFiProxy.get()) && (_dabFiProxy->sendFID_DAB_S_SRV_SELECTStart(*this, T_e8_DAB_ChnSelectMode__DAB_CHNSELECTMODE_RELATIVE, u32Value, 0, dabDirection)));
   //((_dabFiProxy.get()) && (_dabFiProxy->sendFID_DAB_S_SEEK_SERVICEStart(*this, (T_e8_DAB_TuneCmd)dabDirection)));
}


void DABStationList::onSeekReqService(uint8 SeekDirection)
{
   ETG_TRACE_USR3(("DABStationList:: onSeekReqService %d", SeekDirection));
   ((_dabFiProxy.get()) && (_dabFiProxy->sendFID_DAB_S_SEEK_SERVICEStart(*this, (T_e8_DAB_TuneCmd)SeekDirection)));
}


void DABStationList::onSeekEnsemble(uint32 dabDirection)
{
   ETG_TRACE_USR3(("DABStationList:: onSeekReqService %d", dabDirection));
   ((_dabFiProxy.get()) && (_dabFiProxy->sendFID_DAB_S_SEEK_ENSEMBLEStart(*this, (T_e8_DAB_TuneCmd)dabDirection)));
}


/**
 * Description     : Function called to clear the DAB station List.
 * @param[in]      : Void
 * @return         : void
 */
void DABStationList::clearDabList()
{
   ETG_TRACE_USR4(("clearDabList"));
   _dabServiceList.clear();
   stationnamearray.clear();
}


void DABStationList::onDabserviceLinking()
{
   T_e8_DAB_ServiceLinkingMode e8ServiceLinkingMode = T_e8_DAB_ServiceLinkingMode__DAB_SERVICELINKING_DAB;
   if (!_dab_FmStatus)
   {
      e8ServiceLinkingMode = T_e8_DAB_ServiceLinkingMode__DAB_SERVICELINKING_DAB_FM;
   }

   T_b32_DAB_SetupMask setupMask;
   T_b32_DAB_SetupFlags setupFlags;
   T_b8_DAB_LinkingSetupMask b8LinkingSetupMask;

   ((_dabFiProxy.get()) && (_dabFiProxy->sendFID_DAB_S_SETUPStart(*this, setupMask, setupFlags, 0, 0, true, e8ServiceLinkingMode, 0, 0, 0, 0, 0, 0, 0, 0, 0, b8LinkingSetupMask, 0, 0, 0, 0, false)));
   _dabLinkingToggled = true;
}


tVoid DABStationList::vRequestServiceFollowing(tBool bFmFollowingStatus, tBool bDabLinkingStatus)
{
   ETG_TRACE_USR3(("DABStationList::vRequestServiceFollowing(%d %d)", bFmFollowingStatus, bDabLinkingStatus));
   T_b32_DAB_SetupMask setupMask;
   T_b32_DAB_SetupFlags setupFlags;
   T_b8_DAB_LinkingSetupMask b8LinkingSetupMask;
   T_e8_DAB_ServiceLinkingMode enRequiredStatus = T_e8_DAB_ServiceLinkingMode__DAB_SERVICELINKING_OFF;

   if (bFmFollowingStatus && bDabLinkingStatus)
   {
      enRequiredStatus = T_e8_DAB_ServiceLinkingMode__DAB_SERVICELINKING_DAB_FM;
   }
   else if (bFmFollowingStatus)                //never enters this condition as dablinking is always true
   {
      enRequiredStatus = T_e8_DAB_ServiceLinkingMode__DAB_SERVICELINKING_FM;
   }
   else if (bDabLinkingStatus)
   {
      enRequiredStatus = T_e8_DAB_ServiceLinkingMode__DAB_SERVICELINKING_DAB;
   }
   else
   {
      enRequiredStatus = T_e8_DAB_ServiceLinkingMode__DAB_SERVICELINKING_OFF;
   }

   ((_dabFiProxy.get()) && (_dabFiProxy->sendFID_DAB_S_SETUPStart(*this, setupMask, setupFlags, 0, 0, true, enRequiredStatus, 0, 0, 0, 0, 0, 0, 0, 0, 0, b8LinkingSetupMask, 0, 0, 0, 0, false)));
}


/**
 * Description     : Function implemented to update the list of the specified list ID
 * @param[in]      : listID : List ID of the list to be updated
 * @return         : void
 */
void DABStationList::updateList(const uint32& listID)
{
   ETG_TRACE_USR3(("DABStationList : updateList"));
   //ListDataInfo listDataInfo;
   //listDataInfo.listId = listID;
   ListRegistry::s_getInstance().updateList(listID);
}


/**
 * Description    :Function for updating GUI with List Start index.
 *
 *@param[in]      :listSize: List Size
 * @return        :void
 */
void DABStationList::setListStartIndex()
{
   uint32 elemId = _selectedElemID - 1;
   uint8 listSize = _dabServiceList.size();
   uint32 startIndex = TunerUtility::getListStartIndex(listSize, elemId);
   ETG_TRACE_USR4(("setListStartIndex: %d", startIndex));
   POST_MSG((COURIER_MESSAGE_NEW(ListChangeMsg)(LISTID_DAB, ListChangeSet, startIndex)));
}


/**
 * Description     : Function called to highlight the selected service list element
 * @param[in]      : serviceLabel: the current active service label
 * @param[in]      : serviceElementId: The current selected service elementId
 * @param[in]      : listCount: the service list size
 * @return         : void
 */
void DABStationList::dabServiceListHighlight(uint16 serviceElementID)
{
   ETG_TRACE_USR3(("DABStationList : dabServiceListHighlight == %d", serviceElementID));
   if (_dabListStatus)
   {
      _selectedElemID = serviceElementID;
      for (uint16 itr = 0; itr < _dabServiceList.size(); itr++)
      {
         if ((serviceElementID != 0) && (itr == serviceElementID - 1))
         {
            _dabServiceList[itr]._serviceSelected = true;
         }
         else
         {
            _dabServiceList[itr]._serviceSelected = false;
         }
      }
      updateList(LISTID_DAB);
   }
}


/**
 * Description     : Virtual function implemented to get update of FID_DAB_S_TEST_OPERATION Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
 * @param[in]      : error : The error message of 'FID_DAB_S_TEST_OPERATION'
 * @return         : void
 */
void DABStationList::onFID_DAB_S_TEST_OPERATIONError(const ::boost::shared_ptr<  ::MIDW_EXT_DABTUNER_FI:: MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr<  ::MIDW_EXT_DABTUNER_FI:: FID_DAB_S_TEST_OPERATIONError >& /*error*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_DAB_S_TEST_OPERATION Result
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
 * @param[in]      : result : The Result message of the property "FID_DAB_S_TEST_OPERATION"
 * @return         : void
 */
void DABStationList:: onFID_DAB_S_TEST_OPERATIONResult(const ::boost::shared_ptr<  ::MIDW_EXT_DABTUNER_FI:: MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr<  ::MIDW_EXT_DABTUNER_FI:: FID_DAB_S_TEST_OPERATIONResult >& /*result*/)
{
   _seekActiveStatus = false;
}


} // namespace Core
} // namespace App
