/**************************************************************************************
* @file         : AnnoHandlerHandling.h
* @author       : ECV2-Rohini S R
* @addtogroup   : Tuner HALL
* @brief        : Class contains necessary business logic required to handle Traffic Announcement data.
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "AppHmi_TunerStateMachine.h"
#include "AnnoHandler.h"
//#include "AnnoDataBindingHandler.h"
//#include "AnnoCourierMsgHandler.h"
#include "App/datapool/TunerDataPoolConfig.h"
#include "App/Core/Utility/TunerUtility.h"
#include "App/Core/Utility/TunerCommonHandler.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "../VehicleSrvClient/VehicleSrvClient.h"  //CAN Imp

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_TUNER_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_TUNER
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_Tuner_"
#define ETG_I_FILE_PREFIX                 App::Core::AnnoHandler::
#include "trcGenProj/Header/AnnoHandler.cpp.trc.h"
#endif

#include "App/Core/LanguageDefines.h"

namespace App {
namespace Core {

using namespace ::MIDW_TUNER_FI;
using namespace ::MIDW_TUNERMASTER_FI;
using namespace ::MIDW_EXT_DABTUNER_FI;
using namespace ::midw_tunermaster_fi_types;

AnnoHandler* AnnoHandler::m_poAnnouncementHandler = NULL;

/**
 * Description     : Destructor of class AnnoHandler
 */
AnnoHandler::~AnnoHandler()
{
   _Announcements.clear();
   _tunerMasterFiProxy.reset();
   _tunerFiProxy.reset();
   _dabFiProxy.reset();
   ETG_I_UNREGISTER_FILE();
}


/**
 * Description     : Constructor of class AnnoHandler
 */
AnnoHandler::AnnoHandler()
   : _tunerMasterFiProxy(MIDW_TUNERMASTER_FIProxy::createProxy("tunermasterFiPort", *this)),
     _tunerFiProxy(MIDW_TUNER_FIProxy::createProxy("tunerFiPort", *this)),
     _dabFiProxy(MIDW_EXT_DABTUNER_FIProxy::createProxy("dabFiPort", *this)),
     _taActive(false),
     _wbActive(false),
     _annoStatus(false),
     _isPTYEnabled(false),
     _CurrentActiveSrc(T_e8_Source__TUN_MSTR_SOURCE_NONE),
     m_popupstatus(0),
     _CurrentAnnoType(T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_INVALID),
     _psName(""),
     _CurrentFreq(0)

{
   if (_tunerMasterFiProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _tunerMasterFiProxy->getPortName());
   }
   if (_tunerFiProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _tunerFiProxy->getPortName());
   }
   if (_dabFiProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _dabFiProxy->getPortName());
   }
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1_R
   initOtherAnnoArray(false);
#endif
   // _annoDataBinder = new AnnoDataBindingHandler();
   // _annoCourierMsgHandler = new AnnoCourierMsgHandler(this);
   _Announcements.clear();
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2
   _Announcements.push_back(AnnoType(false, "TunerPopups#Popups#RADIO__POPUP_TA_ACTIVE_USERCONF", static_cast<uint8>(ANNOTYPE_TA)));
   _Announcements.push_back(AnnoType(false, "TunerPopups#Popups#RADIO__POPUP_PTY31_ACTIVE_USERCONF", static_cast<uint8>(ANNOTYPE_PTY)));
#endif

#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1_R
   _Announcements.push_back(AnnoType(false, "TunerPopups#Popups#TUNER__ITPOP_TA_ACTIVE", static_cast<uint8>(ANNOTYPE_TA)));
   _Announcements.push_back(AnnoType(false, "TunerPopups#Popups#TUNER__ITPOP_DAB_IA_ACTIVE", static_cast<uint8>(ANNOTYPE_OTHER)));
   _Announcements.push_back(AnnoType(false, "TunerPopups#Popups#TUNER__ITPOP_PTY31_ACTIVE", static_cast<uint8>(ANNOTYPE_PTY)));
#endif

   _Announcements.push_back(AnnoType(false, "", static_cast<uint8>(ANNOTYPE_INVALID)));

   ETG_I_REGISTER_FILE();
}


/**
 * Description     : Request from Framework to start registering for properties. This request means GUI is ready now.
 *
 * @param[in]      : proxy : reference to the base class of all proxies
 * @param[in]      : stateChange : represents a state transition of a service
 * @return         : void
 */
void AnnoHandler::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   if ((proxy == _tunerMasterFiProxy) && (_tunerMasterFiProxy.get()))
   {
      _tunerMasterFiProxy->sendFID_TUNMSTR_G_ANNOUNCEMENT_STATUSUpReg(*this);
      _tunerMasterFiProxy->sendFID_TUNMSTR_G_ANNO_BUTTON_STATUSUpReg(*this);
   }
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_ANNOUNCEMENT_INFOUpReg(*this);
   }
   if ((proxy == _dabFiProxy) && (_dabFiProxy.get()))
   {
      _dabFiProxy->sendFID_DAB_G_ANNO_STATUSUpReg(*this);
   }
}


/**
 * Description     : Request from Framework to start de-registering for properties.
 *
 * @param[in]      : proxy : reference to the base class of all proxies
 * @param[in]      : stateChange : represents a state transition of a service
 * @return         : void
 */
void AnnoHandler::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   if ((proxy == _tunerMasterFiProxy) && (_tunerMasterFiProxy.get()))
   {
      _tunerMasterFiProxy->sendFID_TUNMSTR_G_ANNO_BUTTON_STATUSRelUpRegAll();
      _tunerMasterFiProxy->sendFID_TUNMSTR_G_ANNOUNCEMENT_STATUSRelUpRegAll();
   }
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_ANNOUNCEMENT_INFORelUpRegAll();
   }
   if ((proxy == _dabFiProxy) && (_dabFiProxy.get()))
   {
      _dabFiProxy->sendFID_DAB_G_ANNO_STATUSRelUpRegAll();
   }
}


void AnnoHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("AnnoHandler::onAvailable"));
   if ((proxy == _tunerMasterFiProxy) && (_tunerMasterFiProxy.get()))
   {
      _tunerMasterFiProxy->sendFID_TUNMSTR_G_ANNOUNCEMENT_STATUSUpReg(*this);
      _tunerMasterFiProxy->sendFID_TUNMSTR_G_ANNO_BUTTON_STATUSUpReg(*this);
   }
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_ANNOUNCEMENT_INFOUpReg(*this);
   }
   if ((proxy == _dabFiProxy) && (_dabFiProxy.get()))
   {
      _dabFiProxy->sendFID_DAB_G_ANNO_STATUSUpReg(*this);
   }
}


// ServiceAvailableIF implementation
void AnnoHandler::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   if ((proxy == _tunerMasterFiProxy) && (_tunerMasterFiProxy.get()))
   {
      _tunerMasterFiProxy->sendFID_TUNMSTR_G_ANNO_BUTTON_STATUSRelUpRegAll();
      _tunerMasterFiProxy->sendFID_TUNMSTR_G_ANNOUNCEMENT_STATUSRelUpRegAll();
   }
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_ANNOUNCEMENT_INFORelUpRegAll();
   }
   if ((proxy == _dabFiProxy) && (_dabFiProxy.get()))
   {
      _dabFiProxy->sendFID_DAB_G_ANNO_STATUSRelUpRegAll();
   }
}


/**
 * Description     : Virtual function implemented to get update of FID_TUNMSTR_G_ANNO_BUTTON_STATUS Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNERMASTER_FIProxy'
 * @param[in]      : error : The error message of 'FID_TUNMSTR_G_ANNO_BUTTON_STATUS'
 * @return         : void
 */
void AnnoHandler::onFID_TUNMSTR_G_ANNO_BUTTON_STATUSError(const ::boost::shared_ptr< MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUNMSTR_G_ANNO_BUTTON_STATUSError >& /*error*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_TUNMSTR_G_ANNO_BUTTON_STATUS Status
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNERMASTER_FIProxy'
 * @param[in]      : status : The Result message of the property "FID_TUNMSTR_G_ANNO_BUTTON_STATUS"
 * @return         : void
 */
void AnnoHandler::onFID_TUNMSTR_G_ANNO_BUTTON_STATUSStatus(const ::boost::shared_ptr< MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUNMSTR_G_ANNO_BUTTON_STATUSStatus >& status)
{
   ETG_TRACE_USR4(("onFID_TUNMSTR_G_ANNO_BUTTON_STATUSStatus: %d", status->getSource()));

   if ((status->getSource() == T_e8_Source__TUN_MSTR_SOURCE_FM1) || (status->getSource() == T_e8_Source__TUN_MSTR_SOURCE_DAB) || (status->getSource() == T_e8_Source__TUN_MSTR_SOURCE_WEATHER_ALERT))
   {
      _taActive = status->getB32AnnouncementType().getTUNMSTR_ANNOTYPE_TA();
      _wbActive = status->getB32AnnouncementType().getTUNMSTR_ANNOTYPE_WEATHERALERT();
      ETG_TRACE_USR4(("TA active : %d", _taActive));
      ETG_TRACE_USR4(("WB active : %d", _wbActive));
      RadioSetting::poGetInstance()->vUpdateWBButtonInSettings(_wbActive);
      RadioSetting::poGetInstance()->vUpdateTAButtonInSettings(_taActive);
      TunerDataBindingHandler::getDataBindingHandlerInstance()->TunerPLMsg(EV_MSG_TO_PL_TA_ICON, _taActive, "");
   }
}


/**
 * Description     : Virtual function implemented to get update of FID_TUNMSTR_G_ANNOUNCEMENT_STATUS Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNERMASTER_FIProxy'
 * @param[in]      : error : The error message of 'FID_TUNMSTR_G_ANNOUNCEMENT_STATUS'
 * @return         : void
 */
void AnnoHandler::onFID_TUNMSTR_G_ANNOUNCEMENT_STATUSError(const ::boost::shared_ptr< MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUNMSTR_G_ANNOUNCEMENT_STATUSError >& /*error*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_TUNMSTR_G_ANNOUNCEMENT_STATUS Status
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNERMASTER_FIProxy'
 * @param[in]      : status : The Result message of the property "FID_TUNMSTR_G_ANNOUNCEMENT_STATUS"
 * @return         : void
 */
void AnnoHandler::onFID_TUNMSTR_G_ANNOUNCEMENT_STATUSStatus(const ::boost::shared_ptr< MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUNMSTR_G_ANNOUNCEMENT_STATUSStatus >& status)
{
   ETG_TRACE_USR3(("AnnoHandler::onFID_TUNMSTR_G_ANNOUNCEMENT_STATUSStatus"));
   _CurrentActiveSrc = status->getSource();
   _CurrentAnnoType = status->getAnno_Types();
   _annoStatus = status->getAnno_Status();

   ETG_TRACE_USR3(("AnnoHandler::_CurrentActiveSrc %d", _CurrentActiveSrc));
   ETG_TRACE_USR3(("AnnoHandler::_CurrentAnnoType %d", _CurrentAnnoType));
   ETG_TRACE_USR3(("AnnoHandler::getAnno_Status %d", status->getAnno_Status()));

   if (status->getAnno_Status())
   {
      AnnoType& currentAnno = getAnnouncementObject(getAnnoType(_CurrentAnnoType));
      ETG_TRACE_USR3(("check for %d", currentAnno.annoType));
      ETG_TRACE_USR3(("check for PTY31 and TA announcement popups"));
      showHideAnnouncement(true, _CurrentAnnoType);
      currentAnno.status = true;
   }
   else
   {
      ETG_TRACE_USR3(("AnnoHandler::getAnno_Status to close popup %d", status->getAnno_Status()));
      POST_MSG((COURIER_MESSAGE_NEW(Popup_PFO_TAclose)()));
      VehicleSrvClient::poGetInstance()->updateTAStatus(false); //CAN Imp New
      POST_MSG((COURIER_MESSAGE_NEW(Popup_PFO_PTY31close)()));
      POST_MSG((COURIER_MESSAGE_NEW(Popup_PFO_WBclose)()));
   }
}


bool AnnoHandler::checkAnnoStatus()
{
   ETG_TRACE_USR3(("AnnoHandler::Update Anno Status== %d", _annoStatus));
   return _annoStatus;
}


/**
 * Description     : Function to close all the announcement popups
 *
 * @param[in]      : void
 * @return         : void
 */
void AnnoHandler::closeAllAnnouncementPopUps()
{
   for (uint8 i = 0; i < _Announcements.size(); i++)
   {
      if (_Announcements.at(i).status)
      {
         showHideAnnouncement(false, _Announcements.at(i).annoType);
         _Announcements.at(i).status = false;
      }
   }
}


/**
 * Description     : Virtual function implemented to get update of FID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENT Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional InteETG_TRACE_USR4(("onFID_TUNMSTR_G_ANNOUNCEMENT_STATUSStatus: src = %d miAnnoType = %d", src,miAnnoType));
		rface 'MIDW_TUNERMASTER_FIProxy'
 * @param[in]      : error : The error message of 'FID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENT'
 * @return         : void
 */
void AnnoHandler::onFID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENTError(const ::boost::shared_ptr< MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENTError >& /*error*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENT Result
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNERMASTER_FIProxy'
 * @param[in]      : result : The Result message of the property "FID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENT"
 * @return         : void
 */
void AnnoHandler::onFID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENTResult(const ::boost::shared_ptr< MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENTResult >& /*result*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_DAB_G_ANNO_STATUS Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
 * @param[in]      : error : The error message of 'FID_DAB_G_ANNO_STATUS'
 * @return         : void
 */
void AnnoHandler::onFID_DAB_G_ANNO_STATUSError(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_DAB_G_ANNO_STATUSError >& /*error*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_DAB_G_ANNO_STATUS Status
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_EXT_DABTUNER_FIProxy'
 * @param[in]      : status : The Result message of the property "FID_DAB_G_ANNO_STATUS"
 * @return         : void
 */


void AnnoHandler::onFID_DAB_G_ANNO_STATUSStatus(const ::boost::shared_ptr< MIDW_EXT_DABTUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_DAB_G_ANNO_STATUSStatus >& status)
{
   ETG_TRACE_USR3(("AnnoHandler:: onFID_DAB_G_ANNO_STATUSStatus"));
   if ((_CurrentActiveSrc == T_e8_Source__TUN_MSTR_SOURCE_DAB))
   {
      std::string annoText = status->getAnnoStatus().getAnnoServiceLabel().getSLabel();
      ETG_TRACE_USR3(("annoText is %s", annoText));

      switch (_CurrentAnnoType)
      {
         case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_TA:
         case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_PTY31:
         {
            TunerDataBindingHandler::getDataBindingHandlerInstance()->updateFmTAStnName(annoText, _CurrentAnnoType);
         }
         break;

         case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_TRANSPORT:
         case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_WARNING:
         case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_NEWS:
         case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_WEATHER:
         case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_EVENT:
         case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_SPECIALEVENT:
         case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_PROGRAMME:
         case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_SPORT:
         case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_FINANCE:
         {
            TunerDataBindingHandler::getDataBindingHandlerInstance()->updateDabOtherAnnouncementsName(status->getAnnoStatus().getAnnoServiceLabel().getSLabel());
         }
         break;

         default:
            break;
      }
   }
}


/**
 * Description     : Virtual function implemented to get update of FID_TUN_G_ANNOUNCEMENT_INFO Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : error : The error message of 'FID_TUN_G_ANNOUNCEMENT_INFO'
 * @return         : void
 */
void AnnoHandler::onFID_TUN_G_ANNOUNCEMENT_INFOError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_G_ANNOUNCEMENT_INFOError >& /*error*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_TUN_G_ANNOUNCEMENT_INFO Status
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : status : The Result message of the property "FID_TUN_G_ANNOUNCEMENT_INFO"
 * @return         : void
 */
void AnnoHandler::onFID_TUN_G_ANNOUNCEMENT_INFOStatus(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_G_ANNOUNCEMENT_INFOStatus >& status)
{
   ETG_TRACE_USR3(("AnnoHandler::onFID_TUN_G_ANNOUNCEMENT_INFOStatus"));
   /* std::string fmAnnoStationName = status->getAnnoInfoStatus().getSAnnoStationName();
    uint32 announcementFreq = status->getAnnoInfoStatus().getU32Frequency();
    if (fmAnnoStationName.empty())
    {
       fmAnnoStationName =  TunerUtility::formatFrequency(announcementFreq);
    }
   #ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2
    fmAnnoStationName = getAnnoText(_CurrentAnnoType, fmAnnoStationName);
   #endif

    if ((_CurrentActiveSrc == T_e8_Source__TUN_MSTR_SOURCE_FM1) || (_CurrentActiveSrc == T_e8_Source__TUN_MSTR_SOURCE_PTY31_FM))
    {
       TunerDataBindingHandler::getDataBindingHandlerInstance()->updateFmTAStnName(fmAnnoStationName, _CurrentAnnoType);
    }*/
}


void AnnoHandler::setPSName(std::string& psName, uint32& currentFreq)
{
   _psName = psName;
   _CurrentFreq = currentFreq;
   if (_psName.empty())
   {
      _psName = TunerUtility::formatFrequency(_CurrentFreq);
   }
   if ((_CurrentActiveSrc == T_e8_Source__TUN_MSTR_SOURCE_FM1) || (_CurrentActiveSrc == T_e8_Source__TUN_MSTR_SOURCE_PTY31_FM))
   {
      TunerDataBindingHandler::getDataBindingHandlerInstance()->updateFmTAStnName(_psName, _CurrentAnnoType);
   }
}


/**
 * Description     : Function called to get the updated announcement text based on the announcement type
 *                   Implementation includes checking the announcement type and concatenating corresponding text(TA or PTY) to the announcement text.
 * @param[in]     : annoType: the type of announcement put through (TA or PTY31)
 *
 * @param[in]      : annoText: the announcement data to be appended to
 *
 * @return         : the string annoText
 */


/**
 * Description     : Virtual function implemented to get the current TA source
 *
 * @return         : T_e8_Source
 */
midw_tunermaster_fi_types::T_e8_Source AnnoHandler::getTASource()
{
   return _CurrentActiveSrc;
}


/**
 * Description     : Virtual function implemented to get update of FID_TUNMSTR_S_STOP_ANNOUNCEMENT Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNERMASTER_FIProxy'
 * @param[in]      : error : The error message of 'FID_TUNMSTR_S_STOP_ANNOUNCEMENT'
 * @return         : void
 */
void AnnoHandler::onFID_TUNMSTR_S_STOP_ANNOUNCEMENTError(const ::boost::shared_ptr< MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUNMSTR_S_STOP_ANNOUNCEMENTError >& /*error*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_TUNMSTR_S_STOP_ANNOUNCEMENT Result
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNERMASTER_FIProxy'
 * @param[in]      : result : The Result message of the property "FID_TUNMSTR_S_STOP_ANNOUNCEMENT"
 * @return         : void
 */
void AnnoHandler::onFID_TUNMSTR_S_STOP_ANNOUNCEMENTResult(const ::boost::shared_ptr< MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUNMSTR_S_STOP_ANNOUNCEMENTResult >& /*result*/)
{
}


//WEATHER ALERT
/**
* Description     : Function called to request Toggle WB
* @return         : void
*/
void AnnoHandler::onToggleWB()
{
   setWBflag(!_wbActive);
}


/**
* Description     : Function called to set WB
*                   Implementation includes sending WB status to DataBindingHandler updateTaIndication
* @return         : void
*/
void AnnoHandler::setWBflag(bool wbstatus)
{
   ETG_TRACE_USR3(("AnnoHandler setwbflag is %d ", wbstatus));
   if (_wbActive != wbstatus)
   {
      T_b8_SourceAnno source;
      T_b32_AnnoType anno;

      anno.setTUNMSTR_ANNOTYPE_WEATHERALERT(wbstatus);
      anno.setTUNMSTR_ANNOTYPE_PTY(true);
      source.setTUNMSTR_ANNOSRC_WEATHER(true);
      source.setTUNMSTR_ANNOSRC_FM(true);

      ((_tunerMasterFiProxy.get())
       && (_tunerMasterFiProxy->sendFID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENTStart(
              *this, anno, source)));
   }
}


/**
* Description     : Function returns if TA is active
*
* @return         : bool
*/
bool AnnoHandler::isWbActive()
{
   return _wbActive;
}


/**
 * Description     : Function called to request Toggle TA
 * @return         : void
 */
void AnnoHandler::onToggleTA()
{
   setTAflag(!_taActive);
}


/**
 * Description     : Function called to set TA
 *                   Implementation includes sending TA status to DataBindingHandler updateTaIndication
 * @return         : void
 */
void AnnoHandler::setTAflag(bool tastatus)
{
   ETG_TRACE_USR3(("AnnoHandler setTAflag is %d ", tastatus));
   if (_taActive != tastatus)
   {
      T_b8_SourceAnno source;
      T_b32_AnnoType anno;

      anno.setTUNMSTR_ANNOTYPE_TA(tastatus);
      anno.setTUNMSTR_ANNOTYPE_PTY(true);
      source.setTUNMSTR_ANNOSRC_FM(true);
      source.setTUNMSTR_ANNOSRC_DAB(true);

      ((_tunerMasterFiProxy.get())
       && (_tunerMasterFiProxy->sendFID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENTStart(
              *this, anno, source)));
      //  VehicleSrvClient::poGetInstance()->updateTAStatus(tastatus); //CAN Imp
   }
}


void AnnoHandler::setHMI_TAflag()
{
   ETG_TRACE_USR3((" setHMI_TAflag "));

   T_b8_SourceAnno source;
   T_b32_AnnoType anno;

   anno.setTUNMSTR_ANNOTYPE_TA(false);
   anno.setTUNMSTR_ANNOTYPE_PTY(true);
   source.setTUNMSTR_ANNOSRC_FM(true);
   source.setTUNMSTR_ANNOSRC_DAB(true);

   ((_tunerMasterFiProxy.get()) && (_tunerMasterFiProxy->sendFID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENTStart(*this, anno, source)));
   // VehicleSrvClient::poGetInstance()->updateTAStatus(false); //CAN Imp
}


void AnnoHandler::enableDabPTY()
{
   T_b8_SourceAnno source;
   T_b32_AnnoType anno;

   anno.setTUNMSTR_ANNOTYPE_TA(_taActive);
   anno.setTUNMSTR_ANNOTYPE_PTY(true);
   source.setTUNMSTR_ANNOSRC_DAB(true);
   source.setTUNMSTR_ANNOSRC_FM(true);

   ((_tunerMasterFiProxy.get()) && (_tunerMasterFiProxy->sendFID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENTStart(*this, anno, source)));
}


#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1_R
/**
 * Description     : Function called to request Other announcements of DAB
 *                   Implementation includes sending other announcements method start to MIDW.
 * @return         : void
 */
void AnnoHandler::onDabOtherAnnouncements(uint8 hdlRow)
{
   ETG_TRACE_USR4(("onDabOtherAnnouncements : %d", hdlRow));
   T_b8_SourceAnno dabSource;
   T_b32_AnnoType dabAnno;
   if (hdlRow <= NO_OF_IANNOUNCEMENTS)
   {
      switch (hdlRow)
      {
         case IANNOUNCEMENT_TRANSPORT:
         {
            _dabOtherAnno[IANNOUNCEMENT_TRANSPORT] = !_dabOtherAnno[IANNOUNCEMENT_TRANSPORT];
         }
         break;
         case IANNOUNCEMENT_WARNING:
         {
            _dabOtherAnno[IANNOUNCEMENT_WARNING] = !_dabOtherAnno[IANNOUNCEMENT_WARNING];
         }
         break;
         case IANNOUNCEMENT_NEWS:
         {
            _dabOtherAnno[IANNOUNCEMENT_NEWS] = !_dabOtherAnno[IANNOUNCEMENT_NEWS];
         }
         break;
         case IANNOUNCEMENT_WEATHER:
         {
            _dabOtherAnno[IANNOUNCEMENT_WEATHER] = !_dabOtherAnno[IANNOUNCEMENT_WEATHER];
         }
         break;
         case IANNOUNCEMENT_EVENT:
         {
            _dabOtherAnno[IANNOUNCEMENT_EVENT] = !_dabOtherAnno[IANNOUNCEMENT_EVENT];
         }
         break;
         case IANNOUNCEMENT_MUSICAL_EVENT:
         {
            _dabOtherAnno[IANNOUNCEMENT_MUSICAL_EVENT] = !_dabOtherAnno[IANNOUNCEMENT_MUSICAL_EVENT];
         }
         break;
         case IANNOUNCEMENT_RADIO_INFO:
         {
            _dabOtherAnno[IANNOUNCEMENT_RADIO_INFO] = !_dabOtherAnno[IANNOUNCEMENT_RADIO_INFO];
         }
         break;
         case IANNOUNCEMENT_SPORT:
         {
            _dabOtherAnno[IANNOUNCEMENT_SPORT] = !_dabOtherAnno[IANNOUNCEMENT_SPORT];
         }
         break;
         case IANNOUNCEMENT_FINANCE:
         {
            _dabOtherAnno[IANNOUNCEMENT_FINANCE] = !_dabOtherAnno[IANNOUNCEMENT_FINANCE];
         }
         break;
         case IANNOUNCEMENT_TA:
         {
            _taActive = !_taActive;
            dabSource.setTUNMSTR_ANNOSRC_FM(true);
         }
         break;
         default:
            break;
      }

      sendOtherAnnouncementSettingsToMW();
   }
}


/**
 * Description     : Function sets the other announcement vector
 * @param[in]      : otherAnnoStatus
 * @return         : void
 */

void AnnoHandler::setDabOtherAnnouncements(bool otherAnnoStatus)
{
   initOtherAnnoArray(otherAnnoStatus);
   sendOtherAnnouncementSettingsToMW();
}


/**
 * Description     : Function sends the other announcement request to middleware
 *
 * @return         : void
 */

void AnnoHandler::sendOtherAnnouncementSettingsToMW()
{
   T_b8_SourceAnno dabSource;
   T_b32_AnnoType dabAnno;

   dabAnno.setTUNMSTR_ANNOTYPE_TRANSPORT(_dabOtherAnno[IANNOUNCEMENT_TRANSPORT]);
   dabAnno.setTUNMSTR_ANNOTYPE_WARNING(_dabOtherAnno[IANNOUNCEMENT_WARNING]);
   dabAnno.setTUNMSTR_ANNOTYPE_NEWS(_dabOtherAnno[IANNOUNCEMENT_NEWS]);
   dabAnno.setTUNMSTR_ANNOTYPE_WEATHER(_dabOtherAnno[IANNOUNCEMENT_WEATHER]);
   dabAnno.setTUNMSTR_ANNOTYPE_EVENT(_dabOtherAnno[IANNOUNCEMENT_EVENT]);
   dabAnno.setTUNMSTR_ANNOTYPE_SPECIALEVENT(_dabOtherAnno[IANNOUNCEMENT_MUSICAL_EVENT]);
   dabAnno.setTUNMSTR_ANNOTYPE_PROGRAMME(_dabOtherAnno[IANNOUNCEMENT_RADIO_INFO]);
   dabAnno.setTUNMSTR_ANNOTYPE_SPORTS(_dabOtherAnno[IANNOUNCEMENT_SPORT]);
   dabAnno.setTUNMSTR_ANNOTYPE_FINANCE(_dabOtherAnno[IANNOUNCEMENT_FINANCE]);
   dabAnno.setTUNMSTR_ANNOTYPE_TA(_taActive);

   dabSource.setTUNMSTR_ANNOSRC_DAB(true);
   ((_tunerMasterFiProxy.get()) && (_tunerMasterFiProxy->sendFID_TUNMSTR_S_ACTIVATE_ANNOUNCEMENTStart(*this, dabAnno, dabSource)));
}


/**
 * Description     : Function returns the status of other announcemnts of DAB
 *
 * @return         : bool
 */
bool AnnoHandler::getDabOtherAnnouncementStatus(uint8 annoRow)
{
   bool annoStatus = false;
   if (annoRow < NO_OF_IANNOUNCEMENTS)
   {
      annoStatus = _dabOtherAnno[annoRow];
   }
   return annoStatus;
}


/**
 * Description     : Function that initializes the array of DAB other announcements
 *
 * @return         : void
 */
void AnnoHandler::initOtherAnnoArray(bool otherAnnoStatus)
{
   for (unsigned int itr = 0; itr < NO_OF_IANNOUNCEMENTS; itr++)
   {
      _dabOtherAnno[itr] = otherAnnoStatus;
   }
}


#endif


/**
 * Description     : Function returns if TA is active
 *
 * @return         : bool
 */
bool AnnoHandler::isTaActive()
{
   return _taActive;
}


void AnnoHandler::showHideAnnouncement(bool status, uint8 annoType)
{
   ETG_TRACE_USR3(("AnnoHandler showHideAnnouncement is %d ", annoType));
   m_popupstatus = annoType;

   if ((_CurrentActiveSrc == T_e8_Source__TUN_MSTR_SOURCE_DAB) && (status == 1))
   {
      sm_gAudioSource = SRC_TUNER_TA_DAB;
   }

   if (status)
   {
      if ((annoType == ANNOTYPE_PTY) && (sm_gAudioSource != SRC_PHONE) && (sm_gAudioSource != SRC_PHONE_OUTBAND_RING) && (sm_gAudioSource != SRC_SPI_ALERT) && (sm_gAudioSource != SRC_SPI_PHONE) && (sm_gAudioSource != SRC_SPI_SPEECHRECOGNITION) && (sm_gAudioSource != SRC_PHONE_SPEECHRECOGNITION))
      {
         POST_MSG((COURIER_MESSAGE_NEW(Popup_PFO_TAclose)()));
         VehicleSrvClient::poGetInstance()->updateTAStatus(false); //CAN Imp New
         ETG_TRACE_USR3(("In PTY31 announcement popup"));
         usleep(500);
         POST_MSG((COURIER_MESSAGE_NEW(Popup_PFO_PTY31)()));
      }
      if ((annoType == ANNOTYPE_TA) && (sm_gAudioSource != SRC_PHONE) && (sm_gAudioSource != SRC_PHONE_OUTBAND_RING) && (sm_gAudioSource != SRC_SPI_ALERT) && (sm_gAudioSource != SRC_SPI_PHONE) && (sm_gAudioSource != SRC_SPI_SPEECHRECOGNITION) && (sm_gAudioSource != SRC_PHONE_SPEECHRECOGNITION))
      {
         POST_MSG((COURIER_MESSAGE_NEW(Popup_PFO_PTY31close)()));
         ETG_TRACE_USR3(("In  TA announcement popup"));
         usleep(500);
         POST_MSG((COURIER_MESSAGE_NEW(Popup_PFO_TA)()));
         VehicleSrvClient::poGetInstance()->updateTAStatus(true); //CAN Imp New
      }
      if ((annoType == ANNOTYPE_WB) && (sm_gAudioSource != SRC_PHONE) && (sm_gAudioSource != SRC_PHONE_OUTBAND_RING) && (sm_gAudioSource != SRC_SPI_ALERT) && (sm_gAudioSource != SRC_SPI_PHONE) && (sm_gAudioSource != SRC_SPI_SPEECHRECOGNITION) && (sm_gAudioSource != SRC_PHONE_SPEECHRECOGNITION))
      {
         // POST_MSG((COURIER_MESSAGE_NEW(Popup_PFO_PTY31close)()));
         ETG_TRACE_USR3(("In  WB announcement popup"));
         usleep(500);
         POST_MSG((COURIER_MESSAGE_NEW(Popup_PFO_WB)()));
      }
   }
}


uint8 AnnoHandler::GetPopupStatus()
{
   return m_popupstatus;
}


/**
 * Description     : Function called to cancel TA pop up
 *
 * @return         : void
 */
void AnnoHandler::onCancelAnnouncement()
{
   /* AnnoType currAnno = getAnnouncementObject(type);
    if (currAnno.annoType != static_cast<uint8>(ANNOTYPE_INVALID))
    {
       if (currAnno.status)
       {
          ((_tunerMasterFiProxy.get()) && (_tunerMasterFiProxy->sendFID_TUNMSTR_S_STOP_ANNOUNCEMENTStart(*this, T_StopAnnouncements__TUN_MSTR_STOP_CURRENT)));
          currAnno.status = false;
       }
    }*/
   ((_tunerMasterFiProxy.get()) && (_tunerMasterFiProxy->sendFID_TUNMSTR_S_STOP_ANNOUNCEMENTStart(*this, T_StopAnnouncements__TUN_MSTR_STOP_CURRENT)));
}


/**
 * Description     : Funtion creates Lists corresponding to the list IDs that it receives
 *
 * @param[in]      : listDataInfo : unique ID for lists
 * @return         : tSharedPtrDataProvider
 */
//tSharedPtrDataProvider AnnoHandler::getListDataProvider(const ListDataInfo& /*listDataInfo*/)
//{
//   return tSharedPtrDataProvider();
//}


/**
 * Description     : Function called to updateListInfo.
 *
 * @param[in]      : uint32 listID.
 * @return         : void
 */
//void AnnoHandler::updateListInfo(uint32& listID)
//{
//   ListDataInfo listDataInfo;
//   listDataInfo.listId = listID;
//   ListRegistry::s_getInstance().updateList(listDataInfo);
//}


uint8 AnnoHandler::getAnnoType(T_e8_AnnoTypes type)
{
   uint8 aType = ANNOTYPE_INVALID;
   switch (type)
   {
      case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_PTY31:
         aType = ANNOTYPE_PTY;
         break;

      case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_TA:
         aType = ANNOTYPE_TA;
         break;

      case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_WEATHER_ALERT:
         aType = ANNOTYPE_WB;
         break;

      case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_TRANSPORT:
      case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_WARNING:
      case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_NEWS:
      case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_WEATHER:
      case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_EVENT:
      case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_SPECIALEVENT:
      case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_PROGRAMME:
      case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_SPORT:
      case T_e8_AnnoTypes__TUNMSTR_ANNOTYPE_FINANCE:
         aType = ANNOTYPE_OTHER;
         break;

      default:
         aType = ANNOTYPE_INVALID;
         break;
   }
   return aType;
}


App::Core::AnnoHandler::AnnoType& AnnoHandler::getAnnouncementObject(uint8 type)
{
   uint8 pos = 0;
   for (uint8 i = 0; i < _Announcements.size(); i++)
   {
      if (type == _Announcements.at(i).annoType)
      {
         pos = i;
      }
   }
   return _Announcements.at(pos);
}


/**
 * Description     : Function called to show TA popUp
 *
 * @param[in]      : source: TA source
 * @return         : void
 */
void AnnoHandler::showTAPopUp()
{
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2
   POST_MSG((COURIER_MESSAGE_NEW(TaPopupUpdateMsg)(true)));
   POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("TunerPopups#Popups#RADIO__POPUP_TA_ACTIVE_USERCONF"))));
#endif
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1_R
   POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("TunerPopups#Popups#TUNER__ITPOP_TA_ACTIVE"))));
#endif
}


} // namespace Core
} // namespace App
