/**************************************************************************************
* @file         : AMFMInfoCourierMsgHandler.cpp
* @author       : ECV2-Rohini S R
* @addtogroup   : Tuner HALL
* @brief        : Class contains utility functions to handle courier messages required for AM FM Info.
* @copyright    : (c) 2017-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "util/StrUtf8.h"
#include "AppHmi_TunerStateMachine.h"
#include "AMFMInfoCourierMsgHandler.h"
#include "hmi_trace_if.h"
#include "AMFMStationIF.h"
#include "AMFMInfoIF.h"
#include "PresetListHandlingIF.h"
#include "AMFMInfo.h"
#include "DabInfo.h"
#include "AMFMTestMode.h"
#include "TunerDataBindingHandler.h"
#include "PresetListHandling.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "SrcChangeHandler.h"
#include "RadioSetting.h"
#include "AnnoHandler.h"
#include "Tuner_Defines.h"
#include "ApplicationSwitchConst.h"
#include "AMFMHDdata.h"
#include "../VehicleSrvClient/VehicleSrvClient.h"  //CAN Imp


#define TESTMODE_START 1
#define TESTMODE_STOP 2

#define TUN_SEEKTYPE_OFF 0
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_TUNER_DM
#include "trcGenProj/Header/AMFMInfoCourierMsgHandler.cpp.trc.h"
#endif
std::map<tenViewID, tenTunBank> m_ViewBankMap;


namespace App {
namespace Core {


/**
 * Description     : Constructor of class AMFMInfoCourierMsgHandler
 */
//AMFMInfoCourierMsgHandler::AMFMInfoCourierMsgHandler()
//{
//	ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler : Default Constructor Called"));
//}


AMFMInfoCourierMsgHandler::AMFMInfoCourierMsgHandler(AMFMInfoIF* _mCourierIF, PresetListHandlingIF* _mPresetListHandlingIF)
   : _AmFmInfoIF(_mCourierIF),
     _PresetListHandlingIF(_mPresetListHandlingIF)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler : Constructor Called"));
   ETG_I_REGISTER_FILE();

   m_ViewBankMap[EN_VIEW_SOURCE_SELECTION] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_FM1] = EN_TUN_BANK_FM1;
   m_ViewBankMap[EN_VIEW_FM2] = EN_TUN_BANK_FM2;
   m_ViewBankMap[EN_VIEW_FM_AST] = EN_TUN_BANK_FM_AST;
   m_ViewBankMap[EN_VIEW_AM] = EN_TUN_BANK_AM;
   m_ViewBankMap[EN_VIEW_AM_AST] = EN_TUN_BANK_AM_AST;
   m_ViewBankMap[EN_VIEW_OPT_FMAM] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_FM_TESTMODE] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_FM_MONITORTESTMODE] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_AM_TESTMODE] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_AM_MONITORTESTMODE] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_SOURCE_SELECTION_woNAV] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_FM_SL] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_DAB] = EN_TUN_BANK_DAB;
   m_ViewBankMap[EN_VIEW_DAB_AST] = EN_TUN_BANK_DAB_AST;
   m_ViewBankMap[EN_VIEW_DAB_SL] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_OPT_DABPTY] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_TT_MONITOR1] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_TT_MONITOR2] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_TT_MONITOR3] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_DAB_SETUP_MONITOR] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_DAB_QUALITY_TESTMODE] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_DAB_SERVICE_INFO_TESTMODE] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_DAB_EXPERT_API_TESTMODE] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_AF_LIST] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_TMC] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_TMC_LIST] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_TMC_INFO] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_TMC_STN_INFO] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_TESTMODESCREENS] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_WEATHERBAND] = EN_TUN_BANK_WX;
   m_ViewBankMap[EN_VIEW_FM_INFO] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_AM_INFO] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_HDDTM_TESTMODE] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_HDPARAMTUNING_TESTMODE] = EN_TUN_BANK_INVALID;
   m_ViewBankMap[EN_VIEW_WEATHERBAND_MONITOR] = EN_TUN_BANK_INVALID;

   _tunerDataBindingHandler = TunerDataBindingHandler::getDataBindingHandlerInstance();
   _applicationSwitchClientComponent = ApplicationSwitchClientComponent::poGetInstance();
   m_u8popupState = (tU8)EN_POPUP_DISPLAY;//Value is 0
   m_u8timerCount = 0;
   m_sceneStatus = 0;
   m_EnggModeActive = false;
   m_HdDTMActive = false;//wst4kor for HD
   m_lastActiveScreen = 0;
   activeSource = 0;
   activityID = 0;
   m_hddtmsceneStatus = 0;
   enPreviousView = TUNER_VIEW_INVALID;
}


/**
 * Description     : Destructor of class AMFMInfoCourierMsgHandler
 */
AMFMInfoCourierMsgHandler::~AMFMInfoCourierMsgHandler()
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler : Destructor Called"));
   _AmFmInfoIF = NULL;
   //_AMFMStationIF = NULL;
   _PresetListHandlingIF = NULL;
   ETG_I_UNREGISTER_FILE();
}


bool AMFMInfoCourierMsgHandler::Init()
{
   return true;
}


/**
* @brief   : Incomming message for HMI data model a message component
* @param   :
*/
bool AMFMInfoCourierMsgHandler::onCourierMessage(const Courier::StartupMsg& /*oMsg*/)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::StartupMsg()"));
   return true;
}


bool AMFMInfoCourierMsgHandler::onCourierMessage(const TunerBLMsg& corfoMsg)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::TunerBLMsg()"));
   bool bMsgConsumed = false; // Holds true if Model consumes the message
   const TunerBLMsg* poTunerBLMsg = Courier::message_cast<const TunerBLMsg*>(&corfoMsg);

   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::TunerBLMsg( %d)", poTunerBLMsg->GetEventId()));
   uint8 i = poTunerBLMsg->GetEventId();
   bool sxm_cur_status = SrcChangeHandler::poGetInstance()->getSXMStatus();
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::TunerBLMsg() sxm_cur_status:%d", sxm_cur_status));
   bool reset_back_val = SrcChangeHandler::poGetInstance()->getBackBtnStatus();
   switch (i)
   {
      case EV_SRCSEL_BACK:
      {
         /*if (sxm_cur_status == true)
         {
            ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::const TunerBLMsg(SXM)"));
            uint16 activityID = 0;
            TunerDataBindingHandler::getDataBindingHandlerInstance()->SetSXMActive(true);
            activityID = enActivityIDs__eActivityID_SXM_MAIN;
            ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::ContextToMaster==%d", activityID));
            uint8 activeRegion = SrcChangeHandler::poGetInstance()->getActiveRegionValue();

            _applicationSwitchClientComponent->requestContextSwitchApplication(activeRegion, APPID_APPHMI_TUNER, enActivityIDs__eActivityID_TUNER_SOURCE_SELECTION, APPID_APPHMI_SXM, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)activityID);
         }
         else if (reset_back_val == true)
         {
            SrcChangeHandler::poGetInstance()->setBackBtnStatus(false);
            vTriggerView(EN_VIEW_FM1);
         }
         else
         {
            vSwitchToTunerPlayView();
         }*/
         uint16 activityID = 0;
         activityID = enActivityIDs__eActivityID_HOME;
         uint8 activeRegion = SrcChangeHandler::poGetInstance()->getActiveRegionValue();
         _applicationSwitchClientComponent->requestContextSwitchApplication(activeRegion, APPID_APPHMI_TUNER, enActivityIDs__eActivityID_TUNER_SOURCE_SELECTION, APPID_APPHMI_MASTER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)activityID);
      }
      break;
      case EV_ENGMODE_START:
      {
         if (!m_EnggModeActive)
         {
            ETG_TRACE_USR3(("******************** Tuner Engineering mode start ********************"));
            m_EnggModeActive = true;
            m_sceneStatus = 1;
            AMFMTestMode::poGetInstance()->onDTMScreenEntry(m_sceneStatus);
            //AMFMTestMode::poGetInstance()->sendTestModeONOFF(m_sceneStatus);
            /*m_poTunerClientHandler->vStartStopTestModeFeature(TRUE);
            vHandleTestModeRegistration(TRUE);*/
         }
      }
      break;
      case EV_ENGMODE_STOP:
      {
         ETG_TRACE_USR3(("******************** Tuner Engineering mode end ********************"));
         m_sceneStatus = 0;
         AMFMTestMode::poGetInstance()->onDTMScreenEntry(m_sceneStatus);
         //AMFMTestMode::poGetInstance()->sendTestModeONOFF(m_sceneStatus);
         m_EnggModeActive = false;
         //vDeregisterFromTestData();
      }
      break;
      case EV_ENGMODE_MEASURE_MODE:
      {
         ETG_TRACE_USR3(("******************** Tuner Engineering EV_ENGMODE_MEASURE_MODE ********************"));
         AMFMTestMode::poGetInstance()->onSelectMeasureMode();
      }
      break;
      case EV_ENGMODE_AF_ONOFF:
      {
         ETG_TRACE_USR3(("******************** Tuner Engineering EV_ENGMODE_AF_ONOFF ********************"));
         AMFMTestMode::poGetInstance()->onSelectAFFollowing();
      }
      break;
      case EV_MSG_TO_PL_RDS_CONFIG:
      {
         ETG_TRACE_USR3(("******************** Tuner Engineering EV_ENGMODE_AF_ONOFF ********************"));
         AMFMTestMode::poGetInstance()->onSelectRDSREG();
      }
      break;
      case EV_MSG_TO_PL_TA_ICON:
      {
         ETG_TRACE_USR3(("******************** Tuner Engineering EV_ENGMODE_AF_ONOFF ********************"));
         AMFMTestMode::poGetInstance()->onToggleTA();
      }
      break;
      case EV_MSG_TO_PL_PHASEDIVERSITY:
      {
         ETG_TRACE_USR3(("******************** Tuner Engineering EV_MSG_TO_PL_PHASEDIVERSITY ********************"));
         AMFMTestMode::poGetInstance()->onSelectPhaseDiversity();
      }
      break;
      case EV_CLOSE_TA:
      {
         ETG_TRACE_USR3(("******************** Tuner Engineering EV_CLOSE_TA ********************"));
         AnnoHandler::poGetInstance()->onCancelAnnouncement();
      }
      break;
      //weather alert
      case EV_CLOSE_WB:
      {
         ETG_TRACE_USR3(("******************** Tuner Engineering EV_CLOSE_WB ********************"));
         AnnoHandler::poGetInstance()->onCancelAnnouncement();
      }
      break;
      case EV_HDDTM_START:
      {
         if (!m_HdDTMActive)
         {
            ETG_TRACE_USR3(("******************** Tuner HD DTM mode start ********************"));
            m_HdDTMActive = true;
            m_hddtmsceneStatus = 1;
            AMFMTestMode::poGetInstance()->onDtmHdMonitorScreenEntry(m_hddtmsceneStatus);
         }
      }
      break;
      case EV_HDDTM_STOP:
      {
         ETG_TRACE_USR3(("******************** Tuner HD DTM mode end ********************"));
         m_hddtmsceneStatus = 0;
         AMFMTestMode::poGetInstance()->onDtmHdMonitorScreenEntry(m_hddtmsceneStatus);
         m_HdDTMActive = false;
      }
      break;

      default:
      {
         ETG_TRACE_USR3(("TUN_ERROR: INVALID EVENT ID"));
      }
      break;
   }
   return true;
}


bool AMFMInfoCourierMsgHandler::onCourierMessage(const MediaChangeMsg& corfoMsg)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::MediaChangeMsg()"));
   bool bMsgConsumed = false; // Holds true if Model consumes the message

   const MediaChangeMsg* copoMediaChangeMsg = Courier::message_cast<const MediaChangeMsg*>(&corfoMsg);
   uint8 activeRegion = SrcChangeHandler::poGetInstance()->getActiveRegionValue();

   if (NULL == copoMediaChangeMsg)
   {
      ETG_TRACE_USR3(("TUN_ERROR: tclTunerAppLogic::MediaChangeMsg() CourierMsg Pointer Invalid"));
   }
   else
   {
      bMsgConsumed = true;
      //the activity shall be enabled when it is implemented in the Media
      //_applicationSwitchClientComponent->requestSwitchApplication(APPID_APPHMI_MEDIA);
      _applicationSwitchClientComponent->requestContextSwitchApplication(activeRegion, APPID_APPHMI_TUNER, enActivityIDs__eActivityID_TUNER_SOURCE_SELECTION, APPID_APPHMI_MEDIA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_MEDIA_SOURCE_SELECTION);
   }

   return bMsgConsumed;
}


bool AMFMInfoCourierMsgHandler::onCourierMessage(const ApplicationStateUpdMsg& oMsg)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler : ApplicationStateUpdMsg "));
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler : ApplicationStateUpdMsg  %d", sm_gTargetView));

   return false;
}


bool AMFMInfoCourierMsgHandler::onCourierMessage(const ButtonMainSrcActivate& oMsg)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler : ButtonMainSrcActivate "));

   const ButtonMainSrcActivate* copoButtonMainSrcActivate = Courier::message_cast<const ButtonMainSrcActivate*>(&oMsg);
   if ((NULL != copoButtonMainSrcActivate))
   {
      uint8 u8srcmode = copoButtonMainSrcActivate->GetSrcmode();
      ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler : u8srcmode %d", u8srcmode));

      AMFMInfo::poGetInstance()->vAbortTunerFunctionality();
      activeSource = SrcChangeHandler::poGetInstance()->getsourceInfo();
      uint8 activeRegion = SrcChangeHandler::poGetInstance()->getActiveRegionValue();
      ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler : activeSource == %d", activeSource));
      if (u8srcmode == TUNER_SRC_PLAY)
      {
         switch (activeSource)
         {
            case SRC_TUNER_FM:
               activityID = ACTIVITYID_TUNER_FM;  //FM
               break;
            case SRC_TUNER_AM:
               activityID = ACTIVITYID_TUNER_AM; //AM
               break;
            case SRC_DAB:
               activityID = ACTIVITYID_TUNER_DAB;  //DAB
               break;
            case SRC_TUNER_WX:
               activityID = ACTIVITYID_TUNER_WX;  //WX
               break;
            default:
               activityID = ACTIVITYID_TUNER_SRC_SELECT; // SOURCE_SELECTION
               break;
         }
      }
      if (u8srcmode == TUNER_SRC_SELECTION)  // SOURCE_SELECTION
      {
         activityID = ACTIVITYID_TUNER_SRC_SELECT;
      }
      if (u8srcmode == TUNER_TOGGLE)   // Media_Source_Selection
      {
         SrcChangeHandler::poGetInstance()->reqMediaScreenToggle();
         return true;
      }
      _applicationSwitchClientComponent->requestContextSwitchApplication(activeRegion, APPID_APPHMI_MASTER, enActivityIDs__eActivityID_HOME, APPID_APPHMI_TUNER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)activityID/*enActivityIDs__eActivityID_TUNER_FM1*/);
   }
   return true;
}


/**
* Description     : Function called on Courier AppStatusUpdateMsg
*
* @param[in]      : AppStatusUpdateMsg
* @return         : bool
*/
bool AMFMInfoCourierMsgHandler::onCourierMessage(const AppStatusUpdateMsg& corfoMsg)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler : AppStatusUpdateMsg "));
   uint8 tunerAppStatus = corfoMsg.GetAppStatus();
   if (NULL != _AmFmInfoIF)
   {
      _AmFmInfoIF->setTunerAppStatus(tunerAppStatus);
   }
   //if (NULL != _AMFMStationIF)
   //{
   //	_AMFMStationIF->setTunerAppStatus(tunerAppStatus);
   //}
   //if ((TunerDataPoolConfig::getInstance()->readRegionType() == EUR_REGION))
   //{
   //	//ETG_TRACE_USR4(("Tuner status in amfm: %d", tunerAppStatus));
   //	POST_MSG((COURIER_MESSAGE_NEW(DabAppStatusUpdateMsg)(tunerAppStatus)));
   //}
   return false;
}


bool AMFMInfoCourierMsgHandler::onCourierMessage(const SeekReqMsg& corfoMsg)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::bOnSeekReqMsg()"));

   tBool bMsgConsumed = FALSE;
   /**
   * Seek request from the fascia is blocked if
   * - A view change is in progress
   * - Any announcements are active
   */
#ifdef CODENOTUSED
   if (VIEW_CHANGE_IN_PROGRESS() || m_TAActive || m_PTY31Active)
   {
      ETG_TRACE_USR3(("TUN_ERROR :tclTunerAppLogic::bSendPresetStoreRequest() SEEK REQ IS ABORTED when view change is in Progress  TA Status:(%d)", m_TAActive));
      return true;
   }
#endif
   const SeekReqMsg* copoSeekReqMsg = Courier::message_cast<const SeekReqMsg*>(&corfoMsg);

   if (NULL != copoSeekReqMsg)
   {
      bMsgConsumed = TRUE;
      tU8 seekMode = copoSeekReqMsg->GetSeekMode();

#ifdef CODENOTUSED
      ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::bOnSeekReqMsg :seekMode:(%d) ,m_enSeekType:(%d)", seekMode, m_enSeekType));
      if (IS_DAB_VALID() && (PresetListHandling::poGetInstance()->IsBandDAB()))
      {
         m_poTunerDABAppLogic->vSeekOperation(DAB_SEEK_TYPE_SERVICE, (enSeekMode)seekMode);
         return bMsgConsumed;
      }
#endif
      if (NULL != _AmFmInfoIF)
      {
         ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::bOnSeekReqMsg( mode = %d)", seekMode));
         _AmFmInfoIF->seek(seekMode);
      }
      else
      {
         ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::_AmFmInfoIF NULL Pointer"));
      }
   }
   else
   {
      ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::bOnSeekReqMsg() copoSeekReqMsg NULL Pointer"));
   }

   return bMsgConsumed;
}


bool AMFMInfoCourierMsgHandler::onCourierMessage(const AutoStoreReqMsg& corfoMsg)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::bOnAutostoreReqMsg"));
   tBool bMsgConsumed = FALSE; //Holds TRUE if the message is consumed

   const AutoStoreReqMsg* copoAutoStoreReqMsg = Courier::message_cast<const AutoStoreReqMsg*>(&corfoMsg);

   if (NULL != copoAutoStoreReqMsg)
   {
      _PresetListHandlingIF->vAutostore();
      bMsgConsumed = TRUE;
   }
   else
   {
      ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::bOnAutostoreReqMsg() TunMasterClient NULL Pointer"));
   }

   return bMsgConsumed;
}


bool AMFMInfoCourierMsgHandler::onCourierMessage(const PresetSelectMsg& corfoMsg)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::PresetSelectMsg() "));
   tBool bMsgConsumed = false; //Holds TRUE if the message is consumed
#ifdef CODENOTUSED
   if (VIEW_CHANGE_IN_PROGRESS())
   {
      ETG_TRACE_USR3(("TUN_ERROR : AMFMInfoCourierMsgHandler::PresetSelectMsg PRESET RECALL IS ABORTED when view change is in Progress"));
      return true;
   }
#endif
   const PresetSelectMsg* copoPresetLoadMsg = Courier::message_cast<const PresetSelectMsg*>(&corfoMsg);
   if ((NULL != copoPresetLoadMsg))
   {
      tU16 u16PresetNumber = (tU16)copoPresetLoadMsg->GetPresetNumber();
      if (VIEW_DAB_AST == sm_gCurrentView)
      {
         u16PresetNumber = u16PresetNumber + 6;
         ETG_TRACE_USR3(("DAB PresetSelectMsg inc done"));
      }
      _PresetListHandlingIF->vSelectPreset(u16PresetNumber);
      ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler :PresetSelectMsg( %d)", u16PresetNumber));
      bMsgConsumed = true;
   }
   else
   {
      ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler :bSendPresetSelectRequest() CourierMsg Pointer Invalid"));
   }
   return bMsgConsumed;
}


bool AMFMInfoCourierMsgHandler::onCourierMessage(const PresetStoreMsg& corfoMsg)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::PresetStoreMsg"));
   tBool bMsgConsumed = false; //Holds TRUE if the message is consumed
#ifdef CODENOTUSED
   if (VIEW_CHANGE_IN_PROGRESS())
   {
      ETG_TRACE_USR3(("TUN_ERROR : AMFMInfoCourierMsgHandler::PresetStoreMsg PRESET RECALL IS ABORTED when view change is in Progress"));
      return true;
   }
#endif
   const PresetStoreMsg* copoPresetStoreMsg = Courier::message_cast<const PresetStoreMsg*>(&corfoMsg);

   if (NULL != copoPresetStoreMsg)
   {
      tU16 u16PresetNumber = (tU16)copoPresetStoreMsg->GetPresetNumber();

      ETG_TRACE_USR3(("PresetStoreMsg : Data %d", u16PresetNumber));
      _PresetListHandlingIF->vStorePreset(u16PresetNumber);
      bMsgConsumed = true;
   }
   else
   {
      ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler :bSendPresetStoreRequest() CourierMsg Pointer Invalid"));
   }
   return bMsgConsumed;
}


bool AMFMInfoCourierMsgHandler::onCourierMessage(const TuneFrequencyReqMsg& corfoMsg)
{
   ETG_TRACE_USR3(("tclTunerAppLogic::bOnTuneFrequencyReqMsg()"));
   tBool bMsgConsumed = FALSE; //Holds TRUE if the message is consumed

   const TuneFrequencyReqMsg* copoTuneFrequencyReqMsg = Courier::message_cast<const TuneFrequencyReqMsg*>(&corfoMsg);

   if ((NULL != copoTuneFrequencyReqMsg) /*&& (IS_TUNER_VALID())*/)
   {
      bMsgConsumed = TRUE;
      /* Request the service to tune to the new frequency with new ticks and direction */
      _AmFmInfoIF->vTuneFrequencyStart(copoTuneFrequencyReqMsg->GetDirection(), copoTuneFrequencyReqMsg->GetTicks());
   }
   return bMsgConsumed;
}


bool AMFMInfoCourierMsgHandler::onCourierMessage(const ScanReqMsg& corfoMsg)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::bOnScanReqMsg()"));

   tBool bMsgConsumed = FALSE; //Holds TRUE if the message is consumed

   const ScanReqMsg* poScanReqMsg = Courier::message_cast<const ScanReqMsg*>(&corfoMsg);

   if (NULL != poScanReqMsg)
   {
      bMsgConsumed = TRUE;
      _AmFmInfoIF->vScanRequestMessage();
   }

   return bMsgConsumed;
}


bool AMFMInfoCourierMsgHandler::onCourierMessage(const Slider_TunerFrequencyMsg& corfoMsg)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::bOnSliderRequest()"));

   tBool bMsgConsumed = FALSE; //Holds TRUE if the message is consumed

   const Slider_TunerFrequencyMsg* poSliderReqMsg = Courier::message_cast<const Slider_TunerFrequencyMsg*>(&corfoMsg);

   if ((NULL != poSliderReqMsg))
   {
      uint32 u32SliderValue = (uint32)poSliderReqMsg->GetCurrent_Frequency_Value();

      ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::bOnSliderRequest( %d )", u32SliderValue));
      _AmFmInfoIF->vSlider_TunerFrequencyMsg(u32SliderValue);
   }
   return bMsgConsumed;
}


bool AMFMInfoCourierMsgHandler::onCourierMessage(const TunerReqMsg& corfoMsg)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::bOnTunerReqMsg()"));

   tBool bMsgConsumed = FALSE; //Holds TRUE if the message is consumed
   /*static tU8 u8Tuner0_Source = midw_fi_tcl_e8_Band::FI_EN_TUN_MSTR_BAND_FM;
   static tU8 u8Tuner1_Source = midw_fi_tcl_e8_Band::FI_EN_TUN_MSTR_BAND_FM;*/

   const TunerReqMsg* copoTunerReqMsg =
      Courier::message_cast<const TunerReqMsg*>(&corfoMsg);

   if (NULL != copoTunerReqMsg)
   {
      bMsgConsumed = TRUE;
      tU8 u8TunerData = copoTunerReqMsg->GetTunerData();
      ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::bOnTunerReqMsg :tunerData:(%d)", u8TunerData));
      /*
      * Request for active tuner should be sent only if measure mode is enabled!
      */
      switch (u8TunerData)
      {
         case TUN_TM_REQ_ACTIVE_TUNER_0:
         case TUN_TM_REQ_ACTIVE_TUNER_1:
         case TUN_TM_REQ_ACTIVE_TUNER_2:
         case TUN_TM_REQ_ACTIVE_TUNER_3:
         case TUN_TM_REQ_AUTO_MODE:
         {
            AMFMTestMode::poGetInstance()->onQuadMonitorSelect(u8TunerData);
         }
         break;

         case TUN_TM_REQ_BAND_CHANGE_TUNER_0:
         case TUN_TM_REQ_BAND_CHANGE_TUNER_1:
         case TUN_TM_REQ_BAND_CHANGE_TUNER_2:
         case TUN_TM_REQ_BAND_CHANGE_TUNER_3:
         {
            int32 u32Band = u8TunerData;
            SrcChangeHandler::poGetInstance()->onRequestNextBand();
         }
         break;
         default:
         {
            ETG_TRACE_USR3((" AMFMInfoCourierMsgHandler ERROR: seek mode is INVALID"));
         }
         break;
      }
   }
   else
   {
      ETG_TRACE_USR3((" AMFMInfoCourierMsgHandler ERROR: AMFMInfoCourierMsgHandler::bOnSeekReqMsg() copoSeekReqMsg NULL Pointer"));
   }

   return bMsgConsumed;
}


bool AMFMInfoCourierMsgHandler::onCourierMessage(const TunerViewReqMsg& corfoMsg)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::TunerViewReqMsg()"));
   const TunerViewReqMsg* poTunerViewReqMsg = Courier::message_cast<const TunerViewReqMsg*>(&corfoMsg);

   if (NULL != poTunerViewReqMsg)
   {
      tenViewID enRequestedView = (tenViewID)poTunerViewReqMsg->GetView();
      ETG_TRACE_USR3(("tclTunerAppLoAMFMInfoCourierMsgHandlergic::bOnTunerViewReqMsg() Request: %d", enRequestedView));
      ETG_TRACE_USR3(("tclTunerAppLoAMFMInfoCourierMsgHandlergic::bOnTunerViewReqMsg() Current View: %d", sm_gCurrentView));

      ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::bOnTunerViewReqMsg() getRegion %d", TunerDataPoolConfig::getInstance()->readRegionType()));

      TunerDataBindingHandler::getDataBindingHandlerInstance()->UpdateSourceIconData((tU8)sm_gCurrentView);
      vRequestView(enRequestedView);
      TunerDataBindingHandler::getDataBindingHandlerInstance()->InitialIconHighlight();
      if (enRequestedView == VIEW_DAB)
      {
         DabInfo::poGetInstance()->updateDabPresetData();
      }
      if (VIEW_FM1 == sm_gCurrentView  || VIEW_AM == sm_gCurrentView || VIEW_MASTER_GADGET_TUNER == sm_gCurrentView || VIEW_SOURCE_SELECTION == sm_gCurrentView)
      {
         AMFMInfo::poGetInstance()->vAbortTunerFunctionality();
         ETG_TRACE_USR3(("AMFMInfo::calling abort functionality"));
      }
      else
      {
         ETG_TRACE_USR3(("AMFMInfo::bOnTunerViewReqMsg() vAbortTunerFunctionality"));
      }
   }
   return true;
}


bool AMFMInfoCourierMsgHandler::onCourierMessage(const Tuner_TestmodeViewReqMsg& corfoMsg)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::Tuner_TestmodeViewReqMsg()"));

   const Tuner_TestmodeViewReqMsg* poTuner_TestmodeViewReqMsg = Courier::message_cast<const Tuner_TestmodeViewReqMsg*>(&corfoMsg);
   if (NULL != poTuner_TestmodeViewReqMsg)
   {
      tU8 u8TestmodeReqvalue = poTuner_TestmodeViewReqMsg->GetTestmode_Value();
      switch (u8TestmodeReqvalue)
      {
         case TESTMODE_START:
         {
            //AMFMTestMode::poGetInstance()->onTestmodeAvailable();
         }
         break;
         case TESTMODE_STOP:
         {
            // AMFMTestMode::poGetInstance()->onTestmodeUnAvailable();
         }
         break;
         default:
         {
            ETG_TRACE_USR3(("ERROR: Tuner_TestmodeViewReqMsg is INVALID"));
         }
         break;
      }
   }
   return true;
}


void AMFMInfoCourierMsgHandler::vRequestView(tenViewID enRequestedView)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::vRequestView() %d", enRequestedView));
   if (enRequestedView == EN_VIEW_PREVIOUS)
   {
      enRequestedView = (tenViewID)sm_gPreviousView;
   }
   else if (enRequestedView == EN_VIEW_PLAYVIEW)
   {
      vSwitchToTunerPlayView();
   }
   if (enRequestedView < EN_NUM_VIEWS)
   {
      vTriggerView(enRequestedView);
   }
}


void AMFMInfoCourierMsgHandler::vTriggerView(tenViewID enRequestedView)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::vTriggerView (%d)", enRequestedView));
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::PreviousView (%d)", enPreviousView));

   if ((enRequestedView >= EN_NUM_VIEWS)/* || VIEW_CHANGE_IN_PROGRESS()*/)
   {
      ETG_TRACE_USR3(("TUN_ERROR :AMFMInfoCourierMsgHandler::return"));
      //return;
   }
   sm_gTargetView = (int)enRequestedView;

   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::Target (%d)", sm_gTargetView));
   //set the active preset list
   PresetListHandling::poGetInstance()->vSelectPresetList(sm_gTargetView);
   bAllocateAudioSouce(sm_gTargetView);
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::check (%d)", sm_gTargetView));
   if (enPreviousView != sm_gTargetView)
   {
      if ((sm_gTargetView != VIEW_FM1) && (sm_gTargetView != VIEW_AM) && (sm_gTargetView != VIEW_DAB) && (sm_gTargetView != VIEW_WEATHERBAND))
      {
         Courier::Message* lMsg = COURIER_MESSAGE_NEW(TunerViewMsg)();
         if (0 != lMsg)
         {
            lMsg->Post();
         }
      }
      if ((sm_gAudioSource == AUD_FM) || (sm_gAudioSource == AUD_AM) || (sm_gAudioSource == AUD_DAB) || (sm_gAudioSource == AUD_WX))
      {
         if (((sm_gTargetView == VIEW_FM1) && (sm_gAudioSource == AUD_FM)) || ((sm_gTargetView == VIEW_AM) && (sm_gAudioSource == AUD_AM)) || ((sm_gTargetView == VIEW_DAB) && (sm_gAudioSource == AUD_DAB)) || ((sm_gTargetView == VIEW_WEATHERBAND) && (sm_gAudioSource == AUD_WX)))
         {
            Courier::Message* lMsg = COURIER_MESSAGE_NEW(TunerViewMsg)();
            if (0 != lMsg)
            {
               lMsg->Post();
            }
         }
      }
      enPreviousView = sm_gTargetView;
   }
}


bool AMFMInfoCourierMsgHandler::bAllocateAudioSouce(int iViewId)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::bAllocateAudioSouce(%d)", iViewId));
   int32 iRequestedSource = SRC_INVALID;
   switch (iViewId)
   {
      case VIEW_AM:
      {
         iRequestedSource = SRC_TUNER_AM;
      }
      break;

      case VIEW_DAB:
      {
         iRequestedSource = SRC_DAB;
      }
      break;

      case VIEW_FM1:
      {
         iRequestedSource = SRC_TUNER_FM;
      }
      break;
      case VIEW_WEATHERBAND:
      {
         iRequestedSource = SRC_TUNER_WX;
      }
      break;
      case VIEW_PLAYVIEW:
      {
         //vSwitchToTunerPlayView();
      }
      break;
      default:
      {
      }
      break;
   }

   if ((iRequestedSource != SRC_INVALID) && (iViewId != 0) && (iRequestedSource != sm_gAudioSource))
   {
      SrcChangeHandler::poGetInstance()->activateSource(iRequestedSource);
   }
   return true;
}


/************************************************************************
*FUNCTION:		vSwitchToTunerPlayView
************************************************************************/

void AMFMInfoCourierMsgHandler::vSwitchToTunerPlayView()
{
   tenViewID enRequestedView = (tenViewID)EN_TUNER_VIEW_INVALID;
   for (unsigned int iIndex = 0; iIndex < EN_NUM_VIEWS; ++iIndex)
   {
      if (sm_gSwitchState == m_ViewBankMap[(tenViewID)iIndex])
      {
         enRequestedView = (tenViewID)iIndex;
         break;
      }
   }
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::vSwitchToTunerPlayView(%d)", enRequestedView));
   //sm_gTargetView = enRequestedView;
   vTriggerView(enRequestedView);
}


bool AMFMInfoCourierMsgHandler::onCourierMessage(const ContextToMaster& oMsg)
{
   const ContextToMaster* copoContextToMaster = Courier::message_cast<const ContextToMaster*>(&oMsg);
   if ((NULL != copoContextToMaster))
   {
      uint8 activityID = 0;
      uint8 activeRegion = SrcChangeHandler::poGetInstance()->getActiveRegionValue();
      switch (copoContextToMaster->GetContextScene())
      {
         case VIEW_SOURCE_SELECTION:
            activityID = enActivityIDs__eActivityID_TUNER_SOURCE_SELECTION;
            break;
         case VIEW_OPT_FMAM:
            activityID = enActivityIDs__eActivityID_TUNER_SETTINGS_VIEW;
            break;
         case VIEW_FM_SL:
            SrcChangeHandler::poGetInstance()->setSXMStatus(false);
            activityID = enActivityIDs__eActivityID_TUNER_FM_LIST;
            TunerDataBindingHandler::getDataBindingHandlerInstance()->FMAMHomeScreen();
            break;
         case VIEW_DAB_SL:
            SrcChangeHandler::poGetInstance()->setSXMStatus(false);
            activityID = enActivityIDs__eActivityID_TUNER_DAB_LIST;
            break;
         case VIEW_TESTMODESCREENS:
            activityID = enActivityIDs__eActivityID_TUNER_TESTMODESCREEN;
            break;
         default:
            activityID = enActivityIDs__eActivityID_TUNER_SOURCE_SELECTION;
            break;
      }
      ETG_TRACE_USR4(("AMFMInfoCourierMsgHandler::ContextToMaster==%d", activityID));
      // _applicationSwitchClientComponent->requestSendContext(activeRegion, APPID_APPHMI_TUNER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)activityID);
      sendStoreContextRequest(activityID);
   }
   return true;
}


//For AppFlyns
bool AMFMInfoCourierMsgHandler::onCourierMessage(const ActiveRenderedView& msg)
{
   ETG_TRACE_USR1(("AMFMInfoCourierMsgHandler::onCourierMessage ActiveRenderedView surface id %d ,view name %s", msg.GetSurfaceId(), msg.GetViewName().GetCString()));
   std::string sViewName = msg.GetViewName().GetCString();
   uint16 activityID = 0;
   activityID = getTunerActivityID(sViewName);
   sendStoreContextRequest(activityID);
   return false;
}


uint16 AMFMInfoCourierMsgHandler::getTunerActivityID(std::string viewid)
{
   uint16 activityID = 0;
   if (0 == viewid.compare("TunerModule#TunerScenes#R_FM1_S"))
   {
      SrcChangeHandler::poGetInstance()->setSXMStatus(false);
      activityID = enActivityIDs__eActivityID_TUNER_FM1;
   }
   else if (0 == viewid.compare("TunerModule#TunerScenes#R_AM_S"))
   {
      SrcChangeHandler::poGetInstance()->setSXMStatus(false);
      activityID = enActivityIDs__eActivityID_TUNER_AM;
   }
   else if (0 == viewid.compare("TunerModule#TunerScenes#R_DAB"))
   {
      SrcChangeHandler::poGetInstance()->setSXMStatus(false);
      activityID = enActivityIDs__eActivityID_TUNER_DAB;
   }
   else if (0 == viewid.compare("TunerModule#TunerScenes#R_WeatherBand"))
   {
      SrcChangeHandler::poGetInstance()->setSXMStatus(false);
      activityID = enActivityIDs__eActivityID_TUNER_WX;
      TunerDataBindingHandler::getDataBindingHandlerInstance()->WXHomeScreen();
   }

   return activityID;
}


void AMFMInfoCourierMsgHandler::sendStoreContextRequest(uint16 activityID)
{
   ETG_TRACE_USR4(("AMFMInfoCourierMsgHandler::sendStoreContextRequest: %d", activityID));
   uint8 activeRegion = SrcChangeHandler::poGetInstance()->getActiveRegionValue();
   if (activityID != 0)
   {
      _applicationSwitchClientComponent->requestSendContext(activeRegion, APPID_APPHMI_TUNER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)activityID);
   }
}


//HD button visibility wst4kor
/**
* Description     : Function called on Courier HD ON/OFF Request
*
* @param[in]      : HdOnOffReqMsg
* @return         : bool
*/
bool AMFMInfoCourierMsgHandler::onCourierMessage(const HdOnOffReqMsg& oMsg)
{
   ETG_TRACE_USR3(("inside CourierMessage HdOnOffReqMsg()"));
   if (NULL != AMFMHDdata::poGetInstance())
   {
      ETG_TRACE_USR3(("inside-if CourierMessage HdOnOffReqMsg()"));
      AMFMHDdata::poGetInstance()->onToggleHD(oMsg.GetBand());
   }
   return true;
}


/**
* Description     : Function called on press of DTM Split Audio
*
* @param[in]      : DtmSplitAudioReqMsg
* @return         : bool
*/
bool AMFMInfoCourierMsgHandler::onCourierMessage(const DtmSplitAudioReqMsg& /*oMsg*/)
{
   ETG_TRACE_USR3(("inside CourierMessage DtmSplitAudioReqMsg()"));
   if (NULL != AMFMTestMode::poGetInstance())
   {
      AMFMTestMode::poGetInstance()->onDtmSplitAudio();
   }
   return true;
}


/**
* Description     : Function called on press of DTM Reset Transitions button
*
* @param[in]      : DtmResetTransitionsReqMsg
* @return         : bool
*/
bool AMFMInfoCourierMsgHandler::onCourierMessage(const DtmResetTransitionsReqMsg& /*oMsg*/)
{
   ETG_TRACE_USR3(("inside CourierMessage DtmResetTransitionsReqMsg()"));
   if (NULL != AMFMTestMode::poGetInstance())
   {
      AMFMTestMode::poGetInstance()->onDtmHdResetTransitons();
   }
   return true;
}


/**
* Description     : Function called on recieving DTM HD Tuning Parameter request message
*
* @param[in]      : HdParameterTuningMsg
* @return         : void
*/

bool AMFMInfoCourierMsgHandler::onCourierMessage(const HdParameterTuningMsg& oMsg)
{
   ETG_TRACE_USR3(("inside CourierMessage HdParameterTuningMsg()"));

   if (NULL != AMFMTestMode::poGetInstance())
   {
      AMFMTestMode::poGetInstance()->onDtmHdParamterTuningUpdate(oMsg.GetTuneType());
   }
   return true;
}


/**
* Callback to handle Courier Message HdParameterListUpdate Request
* @param [in] : Reference of Courier::HdParameterListUpdate
* @return     : True-When Message is processed.
*/
bool AMFMInfoCourierMsgHandler::onCourierMessage(const HdParameterListUpdate& oMsg)
{
   ETG_TRACE_USR3(("inside CourierMessage HdParameterListUpdate()"));
   AMFMTestMode::poGetInstance()->onEntryRequestHDParam(oMsg.GetUpdate());
   return true;
}


/**
* Description     : Function called on recieving DTM HD Tuning Parameter request message
*
* @param[in]      : HdAudioProgSelectMsg
* @return         : void
*/

bool AMFMInfoCourierMsgHandler::onCourierMessage(const HdAudioProgSelectMsg& oMsg)
{
   ETG_TRACE_USR3(("inside CourierMessage HdAudioProgSelectMsg()"));

   if (NULL != AMFMTestMode::poGetInstance())
   {
      AMFMTestMode::poGetInstance()->onHDAudioPrgmSelect(oMsg.GetProgVal());
   }
   return true;
}


/**
* Description     : Function called on Courier HdListAccessUpdateMsg
*
* @param[in]      : HdListAccessUpdateMsg
* @return         : bool
*/
bool AMFMInfoCourierMsgHandler::onCourierMessage(const HdListAccessUpdateMsg& oMsg)
{
   if (NULL != AMFMHDdata::poGetInstance())
   {
      uint8 ScreenStatus = oMsg.GetScreenStatus();
      AMFMHDdata::poGetInstance()->onHDInfoScreenAccessStatus(ScreenStatus);
   }
   return true;
}


//Prev Next button
/**
* Description     : Function called on press Home screen
*
* @param[in]      : CloseNextPrev
* @return         : bool
*/
bool AMFMInfoCourierMsgHandler::onCourierMessage(const CloseNextPrev& /*oMsg*/)
{
   AMFMInfo::poGetInstance()->vAbortTunerFunctionality();

   return true;
}


/**
* Description     : Function called on press of SXM Button on radio source selection screen
*
* @param[in]      : SXMContextMsg
* @return         : bool
*/
bool AMFMInfoCourierMsgHandler::onCourierMessage(const SXMContextMsg& /*oMsg*/)
{
   SrcChangeHandler::poGetInstance()->setSXMStatus(true);
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::SXMContextMsg()"));
   uint8 activeRegion = SrcChangeHandler::poGetInstance()->getActiveRegionValue();
   if (activeRegion == REGION_COCKPIT)
   {
      VehicleSrvClient::poGetInstance()->updateTunerBandStatus(CAN_XM); //CAN Imp
   }
   else if (activeRegion == REGION_CABIN_A)
   {
      VehicleSrvClient::poGetInstance()->updateCabinTunerBandStatus(CAN_XM); //CAN Imp
   }
   uint16 activityID = 0;
   TunerDataBindingHandler::getDataBindingHandlerInstance()->SetSXMActive(true);
   activityID = enActivityIDs__eActivityID_SXM_MAIN;
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::ContextToMaster==%d", activityID));

   _applicationSwitchClientComponent->requestContextSwitchApplication(activeRegion, APPID_APPHMI_TUNER, enActivityIDs__eActivityID_TUNER_SOURCE_SELECTION, APPID_APPHMI_SXM, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)activityID);

   return true;
}


/**
* Description     : Function called on press of SXM Monitor Button on Radio Testmode Screen
*
* @param[in]      : SXMMonitorContextMsg
* @return         : bool
*/
bool AMFMInfoCourierMsgHandler::onCourierMessage(const SXMMonitorContextMsg& /*oMsg*/)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::SXMMonitorContextMsg()"));
   uint16 activityID = 0;
   activityID = enActivityIDs__eActivityID_SXM_MONITOR_TESTMODE;
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::ContextToMaster SXMMonitorContextMsg==%d", activityID));
   uint8 activeRegion = SrcChangeHandler::poGetInstance()->getActiveRegionValue();

   _applicationSwitchClientComponent->requestContextSwitchApplication(activeRegion, APPID_APPHMI_TUNER, enActivityIDs__eActivityID_TUNER_TESTMODESCREEN, APPID_APPHMI_SXM, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)activityID);

   return true;
}


/**
* Description     : Function called on press of SXM Settings Button on radio Testmode screen
*
* @param[in]      : SXMSettingsTestmodeContextMsg
* @return         : bool
*/
bool AMFMInfoCourierMsgHandler::onCourierMessage(const SXMSettingsTestmodeContextMsg& /*oMsg*/)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::SXMSettingsTestmodeContextMsg()"));
   uint16 activityID = 0;
   activityID = enActivityIDs__eActivityID_SXM_SETTINGS_TESTMODE;
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::ContextToMaster SXMSettingsTestmodeContextMsg==%d", activityID));
   uint8 activeRegion = SrcChangeHandler::poGetInstance()->getActiveRegionValue();

   _applicationSwitchClientComponent->requestContextSwitchApplication(activeRegion, APPID_APPHMI_TUNER, enActivityIDs__eActivityID_TUNER_TESTMODESCREEN, APPID_APPHMI_SXM, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)activityID);

   return true;
}


/**
* Description     : Function called on long press of source Button on radio source selection screen
*
* @param[in]      : TestmodeBlockContextMsg
* @return         : bool
*/
bool AMFMInfoCourierMsgHandler::onCourierMessage(const TestmodeBlockContextMsg& /*oMsg*/)
{
   tU8 testVal = TunerDataPoolConfig::getInstance()->readTestmodeValue();
   ETG_TRACE_USR3(("TunerDataBindingHandler::readTestmodeValue() testmodeShowHide %d", testVal));

   if (testVal == TESTMODE_ENABLE)
   {
      POST_MSG((COURIER_MESSAGE_NEW(TunerBLMsg)(2, 0, 0)));
      POST_MSG((COURIER_MESSAGE_NEW(TunerViewReqMsg)(35)));
   }
   else
   {
      ETG_TRACE_USR3(("Testmode is blocked. Please check KDS or enable in KDS"));
   }

   return true;
}


/**
* Description     : Function called on press of Back Button on FMAM Settings Screen
* @param[in]      : FMAMBackContextMsg
* @return         : bool
*/
bool AMFMInfoCourierMsgHandler::onCourierMessage(const FMAMBackContextMsg& /*oMsg*/)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::FMAMBackContextMsg()"));

   if (sm_gPreviousView == VIEW_SOURCE_SELECTION)
   {
      POST_MSG((COURIER_MESSAGE_NEW(TunerViewReqMsg)(0)));
   }
   else if (sm_gPreviousView == VIEW_FM_SL)
   {
      POST_MSG((COURIER_MESSAGE_NEW(TunerViewReqMsg)(12)));
   }
   else if (sm_gPreviousView == VIEW_DAB_SL)
   {
      POST_MSG((COURIER_MESSAGE_NEW(TunerViewReqMsg)(15)));
   }
   else
   {
      POST_MSG((COURIER_MESSAGE_NEW(TunerViewReqMsg)(253)));
   }

   return true;
}


/**
* Callback to handle Courier Message HdParameterListUpdate Request
* @param [in] : Reference of Courier::HdParameterListUpdate
* @return     : True-When Message is processed.
*/
bool AMFMInfoCourierMsgHandler::onCourierMessage(const ViewPreviousMsg& /*oMsg*/)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler:: viewPrevious()"));
   enPreviousView = TUNER_VIEW_INVALID;
   return true;
}


//Remote Control
bool AMFMInfoCourierMsgHandler::onCourierMessage(const onHardKeyPress& msg)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::onHardKeyPress HKCode: %d %d", msg.GetHKCode(), msg.GetHKState()));
   int hardKeyCode = msg.GetHKCode();
   int hardkeyState = msg.GetHKState();
   bool retVal = vRequestRemoteCommands(hardkeyState, hardKeyCode);
   return retVal;
}


bool AMFMInfoCourierMsgHandler::vRequestRemoteCommands(uint32 hardkeyPress, uint32 hardKeyValue)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::onHardKeyPress HKValue: %d %d", hardkeyPress, hardKeyValue));
   bool retValue = true;

   switch (hardKeyValue)
   {
      case HARDKEYCODE_REMOTECONTROL_TUNER:
      {
         if (Enum_hmibase_HARDKEYSTATE_UP == hardkeyPress)
         {
            ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::onHardKeyPress of Tuner,this function will call sourceselection or nowplaying based on activesource"));
            activeSource = SrcChangeHandler::poGetInstance()->getsourceInfo();
            if ((activeSource == SRC_TUNER_FM) || (activeSource == SRC_TUNER_AM) || (activeSource == SRC_DAB) || (activeSource == SRC_TUNER_WX))
            {
               POST_MSG((COURIER_MESSAGE_NEW(ButtonMainSrcActivate)(1)));
            }
            else if (activeSource == SRC_TUNER_XM)
            {
               POST_MSG((COURIER_MESSAGE_NEW(SXMContextMsg)()));
            }
            else
            {
               POST_MSG((COURIER_MESSAGE_NEW(ButtonMainSrcActivate)(2)));
            }
         }
         else
         {
            ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::onHardKeyPress invalid key state = %d", hardkeyPress));
         }
      }
      break;
      case HARDKEYCODE_REMOTECONTROL_TRAFFIC:
      {
         if (Enum_hmibase_HARDKEYSTATE_UP == hardkeyPress)
         {
            ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::onHardKeyPress of Button Traffic"));
            POST_MSG((COURIER_MESSAGE_NEW(ButtonListItemUpdMsg)(1000, 2, 0, enRelease)));
         }
         else
         {
            ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::onHardKeyPress invalid key state = %d", hardkeyPress));
         }
      }
      break;
      default:
      {
         retValue = false;
         break;
      }
   }
   return retValue;
}


} // namespace Core
} // namespace App
