/**************************************************************************************
* @file         : AMFMHDdata.cpp
* @author       : ECV- Rohini S R
* @addtogroup   : Tuner HALL
* @brief        : Class contains necessary business logic required to handle AM FM HD data.
* @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "AMFMHDdata.h"
#include "Tuner_Defines.h"
#include "AMFMStationIF.h"
//#include "App/Core/LanguageDefines.h"
#include "App/Core/Utility/TunerUtility.h"
#include "App/Core/Utility/TunerCommonHandler.h"
#include "App/Core/PresetListHandler/PresetListHandling.h"
//#include "AMFMInfoIF.h"
#include "../VehicleSrvClient/VehicleSrvClient.h"  //CAN Imp


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_TUNER_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_TUNER
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_Tuner_"
#define ETG_I_FILE_PREFIX                 App::Core::ATStationInfo::
#include "trcGenProj/Header/AMFMHDdata.cpp.trc.h"
#endif

namespace App {
namespace Core {

using namespace ::MIDW_TUNER_FI;
using namespace ::midw_tuner_fi_types;
using namespace ::midw_tunermaster_fi_types;


DEFINE_CLASS_LOGGER_AND_LEVEL("App/Core/Tuner_AMFM", AMFMHDdata, Info);

static const HDInfoScreenGenre _HDGenreListArray[] =
{
#define HDINFOGENRE_MAP(hdInfogenreName, hdInfogenreID) \
		{ hdInfogenreName, hdInfogenreID},
#include "TunerHDInfoGenre.dat"
#undef HDINFOGENERE_MAP
};


static const HDMainScreenGenre _HDMainScreenGenre[] =
{
#define HDMAINSCREENGENRE_MAP(hdMainScrngenre, hdMainScrngenreID) \
		{ hdMainScrngenre, hdMainScrngenreID},
#include "TunerHDMainScreenGenre.dat"
#undef HDMAINSCREENGENERE_MAP
};


/**
 * Description     : Destructor of class AMFMHDdata
 */

AMFMHDdata::~AMFMHDdata()
{
   ETG_TRACE_USR3(("AMFMHDdata : Destructor entered"));
   if (NULL != _AmFmInfoDataBinder)
   {
      delete _AmFmInfoDataBinder;
      _AmFmInfoDataBinder = NULL;
   }
}


/**
 * Description     : Constructor of class AMFMHDdata
 */
AMFMHDdata* AMFMHDdata::m_poAMFMHDdata = NULL;
AMFMHDdata::AMFMHDdata() :
   _tunerFiProxy(MIDW_TUNER_FIProxy::createProxy("tunerFiPort", *this)), enHDStatus(EN_HD_AVAILABLE)
   /*
   #ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1_R
   _RadioSettings(_mRadioSettings),
   #endif*/

{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR3(("AMFMHDdata : Param Constructor entered"));
   if (_tunerFiProxy.get())
   {
      ETG_TRACE_USR3(("AMFMHDdata : StartupSync"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _tunerFiProxy->getPortName());
   }
   _AmFmInfoDataBinder = TunerDataBindingHandler::getDataBindingHandlerInstance();//new TunerDataBindingHandler();
   // _AmFmInfoCourierMsgHandler = new AMFMInfoCourierMsgHandler(this, PresetListHandling::poGetInstance());
   _AMFMStationIF = AMFMStationList::poGetInstance();
   _AmFmInfo = AMFMInfo::poGetInstance();

   _HDDataDecoding = false;
   _HDAudioDecoding = false;
   _FMHDActive = false;
   _AMHDActive = false;
   _activeHDAudioPrg = 0;
   _switchIndex = NONHDSTATION;
   _categoryName = "";
   _band = 10;
   _HDImageFileInfo.clear();
   _timerStatus = false;
   _radiotxt = "";
   _HdInfoScreenStatus = false;
   _HDDataDecoding = false;
   _HDAudioDecoding = false;
   _CurrentFreq = 0;
   _psName = "";
   _HDCallSign = "";
   _isDemodSet = false;

   clearHDInfoData();
   clearOtherHDData();
}


/**
 * Description     : Request from Framework to start registering for properties. This request means GUI is ready now.
 *
 * @param[in]      : proxy : reference to the base class of all proxies
 * @param[in]      : stateChange : represents a state transition of a service
 * @return         : void
 */
void AMFMHDdata::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR3(("AMFMHDdata: registerProperties"));
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      /*   _tunerFiProxy->sendFID_TUN_G_DEMODULATION_SWITCH_STATUSUpReg(*this);
         _tunerFiProxy->sendFID_TUN_G_HD_PSD_DATAUpReg(*this);
         _tunerFiProxy->sendFID_TUN_G_HD_IMAGEFILE_UPDATEUpReg(*this);
         _tunerFiProxy->sendFID_TUN_G_HD_LONGNAMEUpReg(*this);
         _tunerFiProxy->sendFID_TUN_G_HD_AVISORY_INFOUpReg(*this);
         _tunerFiProxy->sendFID_TUN_G_HD_SIGNALLOST_TIMERUpReg(*this);
          _tunerFiProxy->sendFID_TUN_G_HD_AVISORY_INFOUpReg(*this);*/
   }
}


//wst4kor
void AMFMHDdata::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_DEMODULATION_SWITCH_STATUSUpReg(*this);
      _tunerFiProxy->sendFID_TUN_G_HD_PSD_DATAUpReg(*this);
      _tunerFiProxy->sendFID_TUN_G_HD_IMAGEFILE_UPDATEUpReg(*this);
      _tunerFiProxy->sendFID_TUN_G_HD_LONGNAMEUpReg(*this);
      _tunerFiProxy->sendFID_TUN_G_HD_AVISORY_INFOUpReg(*this);
      _tunerFiProxy->sendFID_TUN_G_HD_SIGNALLOST_TIMERUpReg(*this);
      _tunerFiProxy->sendFID_TUN_G_RADIO_TEXTUpReg(*this);
   }
}


/**
 * Description     : Request from Framework to start de-registering for properties.
 *
 * @param[in]      : proxy : reference to the base class of all proxies
 * @param[in]      : stateChange : represents a state transition of a service
 * @return         : void
 */
void AMFMHDdata::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR3(("AMFMHDdata: deregisterProperties"));
}


/**
 * Description     : Request from Framework to start de-registering for properties.
 *
 * @param[in]      : proxy : reference to the base class of all proxies
 * @param[in]      : stateChange : represents a state transition of a service
 * @return         : void
 */
void AMFMHDdata::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_DEMODULATION_SWITCH_STATUSRelUpRegAll();
      _tunerFiProxy->sendFID_TUN_G_HD_PSD_DATARelUpRegAll();
      _tunerFiProxy->sendFID_TUN_G_HD_IMAGEFILE_UPDATERelUpRegAll();
      _tunerFiProxy->sendFID_TUN_G_HD_LONGNAMERelUpRegAll();
      _tunerFiProxy->sendFID_TUN_G_HD_AVISORY_INFORelUpRegAll();
      _tunerFiProxy->sendFID_TUN_G_HD_SIGNALLOST_TIMERRelUpRegAll();
      _tunerFiProxy->sendFID_TUN_G_RADIO_TEXTRelUpRegAll();
   }
}


/**
 * Description     : Virtual function implemented to set HD Info bits.
 *
 * @param[in]      : HDDataDecoding : status to check Data decoding enabled/disabled
 * @param[in]      : HDAudioDecoding : status to check Audio decoding enabled/disabled
 * @return         : void
 */

void AMFMHDdata::setHDData(bool& HDDataDecoding, bool& HDAudioDecoding)
{
   ETG_TRACE_USR3(("AMFMHDdata::setHDData=%d %d ", HDDataDecoding, HDAudioDecoding));
   if ((_HDDataDecoding != HDDataDecoding) || (_HDAudioDecoding != HDAudioDecoding))
   {
      _HDDataDecoding = HDDataDecoding;
      _HDAudioDecoding = HDAudioDecoding;
      if ((NULL != _AmFmInfoDataBinder) && (!bIsHDActive()))
      {
         clearHDInfoData();
         _HDImageFileInfo.clear();
         _AmFmInfoDataBinder->updateHDAlbumArt(_HDImageFileInfo, _HDImageFileInfo.size(), false);
      }
      if (NULL != _AMFMStationIF)
      {
         _AMFMStationIF->setHDDecodingData(_HDDataDecoding, _HDAudioDecoding);
      }
      if (NULL != _AmFmInfoDataBinder)
      {
         _AmFmInfoDataBinder->updatePSDData(_radioMainData);
         if (_HdInfoScreenStatus)
         {
            _AmFmInfoDataBinder->updateHdInfoData(_radioMainData);
         }
      }
      if (_HDDataDecoding || _HDAudioDecoding)
      {
         setSwitchIndex();
      }
   }
   updateHDLogo();
}


/**
 * Description     : Function implemented to set switch index to display PSD/Advisory/Radio txt data.
 *
 * @return         : void
 */
void AMFMHDdata::setSwitchIndex()
{
   if (((bIsHDActive()) && (_HDDataDecoding && _HDAudioDecoding)))
   {
      _switchIndex = HDSTATION; //PSD
   }
   else if (((!_HDDataDecoding) && (!_HDAudioDecoding)))
   {
      _switchIndex = NONHDSTATION; //RadioText
      if (!_timerStatus)
      {
         ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_G_ATSTATIONINFOGet(*this)));
      }
   }
   if (NULL != _AmFmInfoDataBinder)
   {
      // _AmFmInfoDataBinder->setSwitchIndex(_switchIndex);
      updatePSName();
   }
}


/**
 * Description     : Virtual function implemented to get update of HD PSD Data Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : error : The error message of 'HD PSD Data'
 * @return         : void
 */
void AMFMHDdata::onFID_TUN_G_HD_PSD_DATAError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_G_HD_PSD_DATAError >& /*error*/)
{
   ETG_TRACE_USR4(("onFID_TUN_G_HD_PSD_DATAError"));
}


/**
 * Description     : Virtual function implemented to get update of StationInfo Status
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : status : The status message of the property "StationInfo"
 * @return         : void
 */

void AMFMHDdata::onFID_TUN_G_ATSTATIONINFOStatus(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_G_ATSTATIONINFOStatus >& status)
{
   uint8 ptyIndex = status->getAtstationinfo().getU8PTY();
   std::string category =  TunerCommonHandler::instance()->getCategoryName(ptyIndex);
   if ((NULL != _AmFmInfoDataBinder) && (!_timerStatus) && (_switchIndex == NONHDSTATION) && (_activeHDAudioPrg == 0))
   {
      _AmFmInfoDataBinder->setCategory(category);
   }
}


/**
 * Description     : Virtual function implemented to get update of StationInfo Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : error : The error message of 'StationInfo'
 * @return         : void
 */

void AMFMHDdata::onFID_TUN_G_ATSTATIONINFOError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_G_ATSTATIONINFOError >& /*error*/)
{
   ETG_TRACE_USR4(("onFID_TUN_G_ATSTATIONINFOError"));
}


/**
 * Description     : Virtual function implemented to get update of HS PSD data Status
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : status : The status message of the property "HD PSD Data"
 * @return         : void
 */

void AMFMHDdata::onFID_TUN_G_HD_PSD_DATAStatus(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_G_HD_PSD_DATAStatus >& status)
{
   ETG_TRACE_USR4(("onFID_TUN_G_HD_PSD_DATAStatus"));
   clearHDInfoData();
   if (_activeHDAudioPrg == status->getTunPSDData().getE8HDAudioProgram())
   {
      _radioMainData._artistName = status->getTunPSDData().getSArtist();
      _radioMainData._albumName = status->getTunPSDData().getSAlbum();
      _radioMainData._trackName = status->getTunPSDData().getSTitle();

      _radioMainData._commerialPriceString = status->getTunPSDData().getTunPSDCommercial().getSPriceString();
      _radioMainData._commerialValidity = status->getTunPSDData().getTunPSDCommercial().getSValidUntil();
      _radioMainData._commerialContactURL = status->getTunPSDData().getTunPSDCommercial().getSContactURL();
      _radioMainData._commerialRecievedAs = status->getTunPSDData().getTunPSDCommercial().getSReceivedAs();
      _radioMainData._commerialSellerName = status->getTunPSDData().getTunPSDCommercial().getSNameOfSeller();
      _radioMainData._commerialDesc = status->getTunPSDData().getTunPSDCommercial().getSDescription();
      _radioMainData._commentLanguage = status->getTunPSDData().getTunPSDComment().getSLanguage();
      _radioMainData._commentShortDec = status->getTunPSDData().getTunPSDComment().getSShortContentDescription();
      _radioMainData._commentActualTxt = status->getTunPSDData().getTunPSDComment().getSActualText();
      _radioMainData._HDInfoGenre = getPSDGenre(status->getTunPSDData().getSGenre());
   }
   if (NULL != _AmFmInfoDataBinder)
   {
      _AmFmInfoDataBinder->updatePSDData(_radioMainData);
      if (_HdInfoScreenStatus)
      {
         _AmFmInfoDataBinder->updateHdInfoData(_radioMainData);
      }
   }
   uint32 listID = LIST_ID_HD_INFO_LIST;
   updateListInfo(listID);

   if (NULL != _AMFMStationIF)
   {
      _AMFMStationIF->setradioMainData(_radioMainData);
   }
}


/**
 * Description     : Virtual function implemented to get update of DEMODULATION_SWITCH Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : error : The error message of 'DEMODULATION_SWITCH'
 * @return         : void
 */
void AMFMHDdata::onFID_TUN_G_DEMODULATION_SWITCH_STATUSError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_G_DEMODULATION_SWITCH_STATUSError >& /*error*/)
{
   ETG_TRACE_USR4(("onFID_TUN_G_DEMODULATION_SWITCH_STATUSError"));
   _isDemodSet = false;
}


/**
 * Description     : Virtual function implemented to get update of DEMODULATION_SWITCH Status
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : status : The status message of the property "DEMODULATION_SWITCH"
 * @return         : void
 */
void AMFMHDdata::onFID_TUN_G_DEMODULATION_SWITCH_STATUSStatus(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_G_DEMODULATION_SWITCH_STATUSStatus >& status)
{
   ETG_TRACE_USR4(("onFID_TUN_G_DEMODULATION_SWITCH_STATUSStatus property band= %d", _band));
   bool isRDSStation = false;
   ETG_TRACE_USR4(("Demodulation:_AMHDActive = %d _FMHDActive = %d", _AMHDActive, _FMHDActive));

   if (_isDemodSet)
   {
      switch (_band)
      {
         case T_e8_Tun_TunerBand__TUN_BAND_FM:
            ETG_TRACE_USR4(("Demodulation case FM"));
            _FMHDActive = status->getDemodulationSwitch().getFM_HD();
            if (!_FMHDActive)
            {
               isRDSStation = true;
            }
            break;
         case T_e8_Tun_TunerBand__TUN_BAND_MW:
            ETG_TRACE_USR4(("Demodulation case AM"));
            _AMHDActive = status->getDemodulationSwitch().getAM_HD();
            if (!_AMHDActive)
            {
               isRDSStation = true;
            }
            break;
         default:
            break;
      }
      ETG_TRACE_USR4(("Demodulation case:_AMHDActive = %d _FMHDActive = %d", _AMHDActive, _FMHDActive));
      _isDemodSet = false;
   }
   else
   {
      _FMHDActive = status->getDemodulationSwitch().getFM_HD();
      _AMHDActive = status->getDemodulationSwitch().getAM_HD();
      ETG_TRACE_USR4(("Demodulation Else case:_AMHDActive = %d _FMHDActive = %d", _AMHDActive, _FMHDActive));
   }

   if ((isRDSStation) && (NULL != _AmFmInfoDataBinder))
   {
      _switchIndex = NONHDSTATION;
      //  _AmFmInfoDataBinder->setSwitchIndex(_switchIndex);  //Need to check for data binding
   }
   ETG_TRACE_USR4(("onFID_TUN_G_DEMODULATION_SWITCH_STATUSStatus:_AMHDActive = %d _FMHDActive = %d", _AMHDActive, _FMHDActive));
   if (NULL != _AMFMStationIF)
   {
      _AMFMStationIF->setFMHDActiveStatus(_FMHDActive);
      _AMFMStationIF->setAMHDActiveStatus(_AMHDActive);
   }
   if (NULL != _AmFmInfoDataBinder)
   {
      _AmFmInfoDataBinder->updateFMHDIndicatorStatus(_FMHDActive);
      _AmFmInfoDataBinder->updateAMHDIndicatorStatus(_AMHDActive);
   }

   if (NULL !=  PresetListHandling::poGetInstance())
   {
      PresetListHandling::poGetInstance()->setFMHDActiveStatus(_FMHDActive);
      PresetListHandling::poGetInstance()->setAMHDActiveStatus(_AMHDActive);
   }

   if ((!_FMHDActive || !_AMHDActive) && (NULL != _AmFmInfoDataBinder))
   {
      // _radioMainData._advisoryText = EMPTY_STRING;
      enHDStatus = EN_HD_ALL_OK;
      _AmFmInfoDataBinder->updateAdvisoryData((tU8)enHDStatus);
   }
   if (NULL != _AmFmInfo)
   {
      _AmFmInfo->setFMHDIndicatorStatus(_FMHDActive);
   }
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1_R
   if (NULL != _RadioSettings)
   {
      _RadioSettings->FMSetHDRadiostatus(_FMHDActive);
      _RadioSettings->AMSetHDRadiostatus(_AMHDActive);
   }
   uint32 listID = LIST_ID_RADIO_SETTINGS_NUM;
   updateListInfo(listID);
#endif
}


/**
 * Description     : Function implemented to check demod switch based on currently tuned band.
 *
 * @return         : void
 */
bool AMFMHDdata::bIsHDActive()
{
   ETG_TRACE_USR4(("AMFMHDdata::bIsHDActive()_AMHDActive = %d _FMHDActive = %d", _AMHDActive, _FMHDActive));
   if (_band == T_e8_Tun_TunerBand__TUN_BAND_FM)
   {
      return _FMHDActive;
   }
   else if (_band == T_e8_Tun_TunerBand__TUN_BAND_MW)
   {
      return _AMHDActive;
   }

   return false;
}


/**
 * Description     : Function implements to clear the HD Info variables.
 *
 * @return         : void
 */
void AMFMHDdata::clearHDInfoData()
{
   _radioMainData._commerialPriceString = "";
   _radioMainData._commerialValidity = "";
   _radioMainData._commerialContactURL = "";
   _radioMainData._commerialRecievedAs = "";
   _radioMainData._commerialSellerName = "";
   _radioMainData._commerialDesc = "";
   _radioMainData._commentLanguage = "";
   _radioMainData._commentShortDec = "";
   _radioMainData._commentActualTxt = "";
}


/**
 * Description     : Function implements to clear the PSD Data.
 *
 * @return         : void
 */
void AMFMHDdata::clearPSDData()
{
   _radioMainData._artistName = "";
   _radioMainData._albumName = "";
   _radioMainData._trackName = "";
   if ((NULL != _AmFmInfoDataBinder) && (!_categoryName.empty()))
   {
      std::string category = "";
      _AmFmInfoDataBinder->setCategory(category);
   }
}


void AMFMHDdata::clearOtherHDData()
{
   _radioMainData._HDInfoGenre = "";
   _radioMainData._advisoryText = "";
   _radioMainData._radiotxt = "";
   _radioMainData._slogan = "";
   _radioMainData._artistName = "";
   _radioMainData._albumName = "";
   _radioMainData._trackName = "";
}


/**
 * Description     : Virtual function implemented to get update of DEMODULATION_SWITCH Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : error : The error message of 'DEMODULATION_SWITCH'
 * @return         : void
 */
void AMFMHDdata::onFID_TUN_S_SET_DEMODULATION_SWITCHError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_SET_DEMODULATION_SWITCHError >& /*error*/)
{
   ETG_TRACE_USR4(("onFID_TUN_S_SET_DEMODULATION_SWITCHError"));
}


/**
 * Description     : Virtual function implemented to set the result of method start Set DEMODULATION_SWITCH
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : result : The result message of the method "Set DEMODULATION_SWITCH"
 * @return         : void
 */
void AMFMHDdata::onFID_TUN_S_SET_DEMODULATION_SWITCHResult(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_SET_DEMODULATION_SWITCHResult >& result)
{
   ETG_TRACE_USR4(("onFID_TUN_S_SET_DEMODULATION_SWITCHResultstatus %d", result->getTunerServer_StatusResponse().getU8TunerStatus()));
   ETG_TRACE_USR4(("onFID_TUN_S_SET_DEMODULATION_SWITCHResult %d", result->getTunerServer_StatusResponse().getU8TunerState()));
}


/**
 * Description     : Function called to request Toggle HD
 *                   Implementation includes sending HD ON/OFF status to Tuner Middleware
 * @param[in]      : bandType : AM /FM band
 * @return         : void
 */
void AMFMHDdata::onToggleHD(const uint8& bandType)
{
   ETG_TRACE_USR3(("inside function onToggleHD: bandType= %d", bandType));
   T_b8_Tun_DemodSwitch demodSwitchValue;
   T_b8_Tun_DemodSwitch demodSwitchMask;
   T_b8_Tun_DemodSwitch demodSwitch;
   clearPSDData();
   _isDemodSet = true;

   if (NULL != _AmFmInfoDataBinder)
   {
      _AmFmInfoDataBinder->updatePSDData(_radioMainData);
   }

   if (bandType == 0)
   {
      ETG_TRACE_USR3(("inside bandtype FM"));
      demodSwitch.setFM_HD(false);
      demodSwitchMask.setFM_HD(true);
      demodSwitchValue.setFM_HD(!_FMHDActive);
   }
   else if (bandType == 1)
   {
      ETG_TRACE_USR3(("inside bandtype AM"));
      demodSwitch.setAM_HD(false);
      demodSwitchMask.setAM_HD(true);
      demodSwitchValue.setAM_HD(!_AMHDActive);
   }

   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_SET_DEMODULATION_SWITCHStart(*this, demodSwitch, demodSwitchMask, demodSwitchValue)));
}


/**
 * Description     : Virtual function implemented to get update of HD_IMAGEFILE_UPDATE Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : error : The error message of 'HD_IMAGEFILE_UPDATE'
 * @return         : void
 */
void AMFMHDdata::onFID_TUN_G_HD_IMAGEFILE_UPDATEError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_G_HD_IMAGEFILE_UPDATEError >& /*error*/)
{
   ETG_TRACE_USR4(("onFID_TUN_G_HD_IMAGEFILE_UPDATEError"));
}


/**
 * Description     : Virtual function implemented to get update of HD_IMAGEFILE_UPDATE Status
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : status : The status message of the property "HD_IMAGEFILE_UPDATE"
 * @return         : void
 */
void AMFMHDdata::onFID_TUN_G_HD_IMAGEFILE_UPDATEStatus(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_G_HD_IMAGEFILE_UPDATEStatus >& status)
{
   bool displayStatus = status->getBDisplayImage();
   ETG_TRACE_USR4(("onFID_TUN_G_HD_IMAGEFILE_UPDATEStatus,Display status = %d", displayStatus));
   _HDImageFileInfo = status->getImageFileInfo().getImageFileBytes();
   if (NULL != _AmFmInfoDataBinder)
   {
      if (_activeHDAudioPrg == status->getImageFileInfo().getE8HDAudioProgram())
      {
         _AmFmInfoDataBinder->updateHDAlbumArt(_HDImageFileInfo, status->getImageFileInfo().getU32NoOfImageFileBytes(), displayStatus);
      }
      else
      {
         _HDImageFileInfo.clear();
         _AmFmInfoDataBinder->updateHDAlbumArt(_HDImageFileInfo, _HDImageFileInfo.size(), false);
      }
   }
}


/**
 * Description     : Virtual function implemented to get update of TUN_G_HD_LONGNAME Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : error : The error message of 'TUN_G_HD_LONGNAME'
 * @return         : void
 */
void AMFMHDdata::onFID_TUN_G_HD_LONGNAMEError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_G_HD_LONGNAMEError >& /*error*/)
{
   ETG_TRACE_USR4(("onFID_TUN_G_HD_LONGNAMEError"));
}


/**
 * Description     : Virtual function implemented to get update of TUN_G_HD_LONGNAME Status
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : status : The status message of the property "TUN_G_HD_LONGNAME"
 * @return         : void
 */
void AMFMHDdata::onFID_TUN_G_HD_LONGNAMEStatus(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_G_HD_LONGNAMEStatus >& status)
{
   _radioMainData._slogan = "";
   if (_activeHDAudioPrg == status->getE8HDAudioProgram())
   {
      ETG_TRACE_USR4(("onFID_TUN_G_HD_LONGNAMEStatus,_slogan = %s", status->getSHDLongName().c_str()));
      _radioMainData._slogan = status->getSHDLongName();
   }
   if (NULL != _AmFmInfoDataBinder)
   {
      _AmFmInfoDataBinder->updateHdInfoData(_radioMainData); //Need to check for data binding
   }
}


/**
 * Description     : Virtual function implemented to get update of TUN_G_HD_AVISORY_INFO Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : error : The error message of 'TUN_G_HD_AVISORY_INFO'
 * @return         : void
 */
void AMFMHDdata::onFID_TUN_G_HD_AVISORY_INFOError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_G_HD_AVISORY_INFOError >& /*error*/)
{
   ETG_TRACE_USR4(("onFID_TUN_G_HD_AVISORY_INFOError"));
}


/**
 * Description     : Virtual function implemented to get update of FID_TUN_G_HD_AVISORY_INFO Status
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : status : The status message of the property "FID_TUN_G_HD_AVISORY_INFO"
 * @return         : void
 */
void AMFMHDdata::onFID_TUN_G_HD_AVISORY_INFOStatus(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_G_HD_AVISORY_INFOStatus >& status)
{
   ETG_TRACE_USR4(("status->getE8HDAdvisoryInfo = %d", status->getE8HDAdvisoryInfo()));
   ETG_TRACE_USR4(("AMFMHDdata::onFID_TUN_G_HD_AVISORY_INFOStatus()"));

   switch (status->getE8HDAdvisoryInfo())
   {
      case T_e8_Tun_HDAdvisoryInfo__TUN_HD_ADVISORY_SIGNAL_LOST:

         enHDStatus = EN_HD_SIGNAL_LOST;    // when status is 1
         //   _radioMainData._advisoryText = "Station is currently not available";
         _switchIndex = ADVISORIES;
         break;
      case T_e8_Tun_HDAdvisoryInfo__TUN_HD_ADVISORY_WAITING_FOR_SIGNAL:

         enHDStatus = EN_HD_SIGNAL_WAIT;    // when status is 2
         // _radioMainData._advisoryText = "Loading...";
         _switchIndex = ADVISORIES;
         break;
      case T_e8_Tun_HDAdvisoryInfo__TUN_HD_ADVISORY_CHANNEL_NOT_AVAILABLE:

         enHDStatus = EN_HD_NO_CHANNEL;    // when status is 3
         //  _radioMainData._advisoryText = "Station is currently not available";

         _switchIndex = ADVISORIES;
         break;
      case T_e8_Tun_HDAdvisoryInfo__TUN_HD_ADVISORY_HDAUDIO_AVAILABLE:
         _switchIndex = HDSTATION;      // when status is 4
         //_radioMainData._advisoryText = "";
         enHDStatus = EN_HD_AVAILABLE;
         break;
      default:
         break;
   }
   if ((NULL != _AmFmInfoDataBinder) && bIsHDActive())
   {
      _AmFmInfoDataBinder->updateAdvisoryData((tU8) enHDStatus);
   }
}


/**
* Description     : Function implemented to set Ball game mode status variable.
*
* @param[in]      : BallGameMode : Ball Game mode status Enabled/Disabled
* @return         : void
*/
void AMFMHDdata::setBallGameModeData(bool& BallGameMode)
{
   if (NULL != _AmFmInfoDataBinder)
   {
      if (bIsHDActive())
      {
         _AmFmInfoDataBinder->setBallGameMode(BallGameMode);
      }
      else
      {
         BallGameMode = false;
         _AmFmInfoDataBinder->setBallGameMode(BallGameMode);
      }
   }
}


/**
 * Description     : Virtual function implemented to set HD Info data.
 *
 * @param[in]      : HDcallSign : HD CallSign data
 * @param[in]      : activeHDAudioPrg : Active HD Audio program service data
 * @param[in]      : currentFreq : current frequency data
 * @param[in]      : currentBand : current Band data
 * @param[in]      : psName : PS name of currently tuned frequency
 * @return         : void
 */

void AMFMHDdata::setHDInfoData(std::string& HDcallSign, uint8& activeHDAudioPrg, uint32& currentFreq, uint8& currentBand, std::string& psName, uint8& HDStationGenre)
{
   ETG_TRACE_USR4(("inside function AMFMHDdata : setHDInfoData "));
   ETG_TRACE_USR4(("bIsHDActive %d", bIsHDActive()));
   _psName = psName;
   _HDCallSign = HDcallSign;
   if (((_activeHDAudioPrg != activeHDAudioPrg) || (_CurrentFreq != currentFreq) || (_band != currentBand)) && bIsHDActive())
   {
      clearHDInfoData();
      clearPSDData();
      if (NULL != _AmFmInfoDataBinder)
      {
         _radioMainData._slogan = "";
         _AmFmInfoDataBinder->updateHdInfoData(_radioMainData);
         uint32 listID = LIST_ID_HD_INFO_LIST;
         updateListInfo(listID);
         if (_HDImageFileInfo.size() != 0)
         {
            _HDImageFileInfo.clear();
            _AmFmInfoDataBinder->updateHDAlbumArt(_HDImageFileInfo, _HDImageFileInfo.size(), false);
         }
         enHDStatus = EN_HD_ALL_OK;
         _AmFmInfoDataBinder->updateAdvisoryData((tU8)enHDStatus);
      }
   }
   _activeHDAudioPrg = activeHDAudioPrg;
   _CurrentFreq = currentFreq;
   _band = currentBand;
   if ((_HDDataDecoding && _HDAudioDecoding) && (NULL != _AmFmInfoDataBinder))
   {
#ifdef HDMAINSCREENGENRE_MAP
      if (HDStationGenre <= HD_MAINSCREEN_GENRE_SIZE)
      {
         _categoryName = _HDMainScreenGenre[HDStationGenre].hdMainScrngenre;
      }
#endif
      if (_switchIndex == HDSTATION)
      {
         _AmFmInfoDataBinder->setCategory(_categoryName);
      }
   }
   updatePSName();
   if (NULL != _AmFmInfoDataBinder)
   {
      _AmFmInfoDataBinder->updatePSDData(_radioMainData);
   }
}


/**
 * Description     : Function implemented to set the switch status for HD logo and  audio program visibility
 *
 * @param[in]      :
 * @return         : void
 */
void AMFMHDdata::setHdLogoAndAudioProgramStatus()
{
   ETG_TRACE_USR3(("inside function AMFMHDdata::setHdLogoAndAudioProgramStatus()"));
   int8 HdLogoStatus = 0;
   if ((_HDDataDecoding) && (_HDAudioDecoding))
   {
      HdLogoStatus = 2;
   }
   else if ((_HDDataDecoding) || (_HDAudioDecoding))
   {
      HdLogoStatus = 1;
   }
   ETG_TRACE_USR3(("AMFMHDdata::setHdLogoAndAudioProgramStatus() HdLogoStatus=%d ", HdLogoStatus));
   bool AudioProgramVisibility = HdLogoStatus > 1 ? true : false;
   ETG_TRACE_USR3(("the value of Band is: %d", _band));

   if (NULL != _AmFmInfoDataBinder)
   {
      _AmFmInfoDataBinder->setHdAudioProgramVisibilityStatus(AudioProgramVisibility);

      //if (_band == T_e8_Tun_TunerBand__TUN_BAND_FM)
      //{
      //   _AmFmInfoDataBinder->updateFMHDLogo(HdLogoStatus);
      //}
      //else
      //{
      //   _AmFmInfoDataBinder->updateAMHDLogo(HdLogoStatus);
      //}

      uint8 HomeScreenFMLogo = 0;
      uint8 HomeScreenAMLogo = 0;

      if (_band == T_e8_Tun_TunerBand__TUN_BAND_FM)
      {
         HomeScreenAMLogo = 0;
         ((_FMHDActive == 1) && (TunerCommonHandler::instance()->isNARVariant())) ? HomeScreenFMLogo = 1 : HomeScreenFMLogo = 0;
         _AmFmInfoDataBinder->updateFMHDLogo(HdLogoStatus);
      }

      if (_band == T_e8_Tun_TunerBand__TUN_BAND_MW)
      {
         HomeScreenFMLogo = 0;
         ((_AMHDActive == 1) && (TunerCommonHandler::instance()->isNARVariant())) ? HomeScreenAMLogo = 1 : HomeScreenAMLogo = 0;
         _AmFmInfoDataBinder->updateAMHDLogo(HdLogoStatus);
      }
      _AmFmInfoDataBinder->HomeScreenFMHDLogo(HomeScreenFMLogo);
      _AmFmInfoDataBinder->HomeScreenAMHDLogo(HomeScreenAMLogo);
   }
}


/**
 * Description     : Function implemented to set HD logo timer status
 *
 * @return         : void
 */
void AMFMHDdata::setHdLogotoAcquiring()
{
   ETG_TRACE_USR3(("inside function AMFMHDdata::setHdLogotoAcquiring()"));

   if ((NULL != _AmFmInfoDataBinder) && (bIsHDActive()))
   {
      uint32 HDLogoStatus = 1;
      if (_band == T_e8_Tun_TunerBand__TUN_BAND_FM)
      {
         _AmFmInfoDataBinder->updateFMHDLogo(HDLogoStatus);
      }
      else
      {
         _AmFmInfoDataBinder->updateAMHDLogo(HDLogoStatus);
      }
   }
}


/**
 * Description     : Function implemented to update HD logo status
 *
 * @return         : void
 */
void AMFMHDdata::updateHDLogo()
{
   if (!_timerStatus)
   {
      setHdLogoAndAudioProgramStatus();
   }
   else if (_switchIndex != ADVISORIES)
   {
      setHdLogotoAcquiring();
   }
}


/**
 * Description     : Virtual function implemented to get update of FID_TUN_G_HD_SIGNALLOST_TIMER Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : error : The error message of 'FID_TUN_G_HD_SIGNALLOST_TIMER'
 * @return         : void
 */
void AMFMHDdata::onFID_TUN_G_HD_SIGNALLOST_TIMERError(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_HD_SIGNALLOST_TIMERError >& /*error*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_TUN_G_HD_SIGNALLOST_TIMER Status
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : status : The status message of the property "FID_TUN_G_HD_SIGNALLOST_TIMER"
 * @return         : void
 */
void AMFMHDdata::onFID_TUN_G_HD_SIGNALLOST_TIMERStatus(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_HD_SIGNALLOST_TIMERStatus >& status)
{
   _timerStatus = status->getBTimer();
   updateHDLogo();
   ETG_TRACE_USR4(("_timerStatus %d ", _timerStatus));
   if (NULL != _AMFMStationIF)
   {
      _AMFMStationIF->setHDTimerStatus(_timerStatus); //from station if need to check
   }
   if (!_timerStatus)
   {
      setSwitchIndex();
   }
}


/**
 * Description     : Function called to get Switch Index
 *
 * @return         : uint8
 */
uint8 AMFMHDdata::getHDSwitchIndex()
{
   return _switchIndex;
}


/**
 * Description     : Virtual function implemented to get update of RadioText Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : error : The error message of 'RadioText'
 * @return         : void
 */
void AMFMHDdata::onFID_TUN_G_RADIO_TEXTError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_G_RADIO_TEXTError >& /*error*/)
{
}


/**
 * Description     : Virtual function implemented to get update of Radio Text Status
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : status : The status message of the property "Radio Text"
 * @return         : void
 */
void AMFMHDdata::onFID_TUN_G_RADIO_TEXTStatus(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_G_RADIO_TEXTStatus >& status)
{
   if (NULL != _AmFmInfoDataBinder)
   {
      if (strcmp(_radiotxt.c_str(), status->getTunerRadioText().getSRadioTextData().c_str()) != 0)
      {
         _radiotxt = status->getTunerRadioText().getSRadioTextData();
         if ((TunerCommonHandler::instance()->isHDFeatureEnabled() == true) && (((!_HDDataDecoding) && (!_HDAudioDecoding)) && (!_timerStatus)))
         {
            _switchIndex  = NONHDSTATION;
            // _AmFmInfoDataBinder->setSwitchIndex(_switchIndex);
            // _AmFmInfoDataBinder->setRadioTextData(_radiotxt);
         }
      }
   }
}


void AMFMHDdata::updateListInfo(uint32& listID)
{
   ETG_TRACE_USR3(("AMFMHDdata : updateListInfo"));
   //ListDateProviderReqMsg listDataInfo;
   //listDataInfo.GetListId() = listID;
   ListRegistry::s_getInstance().updateList(listID);
}


/*void AMFMHDdata::updateListInfo(uint32& listID)
{
	ListDataInfo listDataInfo;
	listDataInfo.listId = listID;
	ListRegistry::s_getInstance().updateList(listDataInfo);
}*/

/**
* Description     : Function called to request if HD Info screen is entered or not
* @param[in]      : HDInfoScrStatus : Screen entry or screen exit of HDInfo screen.
* @return         : bool
*/
void AMFMHDdata::onHDInfoScreenAccessStatus(uint8& HDInfoScrStatus)
{
   ETG_TRACE_USR4(("AMFMHDdata::onHDInfoScreenAccessStatus HDInfoScrStatus %d", HDInfoScrStatus));

   if (HDInfoScrStatus == SCENE_ENTRY && (NULL != _AmFmInfoDataBinder))
   {
      _AmFmInfoDataBinder->updateHdInfoData(_radioMainData);
      _HdInfoScreenStatus = true;
   }
   else
   {
      _HdInfoScreenStatus = false;
   }
}


std::string AMFMHDdata::getPSDGenre(std::string HDInfoGenre)
{
   ETG_TRACE_USR4(("AMFMHDdata::getPSDGenre HDInfoGenre %s", HDInfoGenre.c_str()));

   std::string Genre = HDInfoGenre;
   std::string parsedText = "";
   std::string InfoGenre = "";
   int ParanthesisPos = 0;
   bool isParanthesisFound = false;
   while ((ParanthesisPos = Genre.find_first_of("(")) >= 0)
   {
      isParanthesisFound = true;
      std::string MultiGenre = "";
      // to get the first instance of ( .. will return pos or -1 if not found
      parsedText = Genre.substr(ParanthesisPos + 1, (Genre.find_first_of(")") - (ParanthesisPos + 1)));
      Genre.erase(ParanthesisPos, 1);
      Genre.erase(Genre.find_first_of(")"), 1);
      if (is_digits(parsedText))
      {
         int GenreID = atoi(parsedText.c_str()); // use num for map function
         if (GenreID <= HD_INFOSCREEN_GENRE_SIZE)
         {
            MultiGenre = _HDGenreListArray[GenreID].hdInfogenreName;
         }
         InfoGenre = InfoGenre + " " + MultiGenre;
      }
      else
      {
         InfoGenre = parsedText;
      }
   }
   if (isParanthesisFound == false)
   {
      if (is_digits(Genre))
      {
         int GenreID = std::atoi(Genre.c_str());
         if (GenreID <= HD_INFOSCREEN_GENRE_SIZE)
         {
            InfoGenre = _HDGenreListArray[GenreID].hdInfogenreName;
         }
      }
      else
      {
         InfoGenre = Genre;
      }
   }
   return InfoGenre;
}


bool AMFMHDdata::is_digits(const std::string& str)
{
   if (str.empty())
   {
      return false;
   }
   else if (str.find_first_not_of("0123456789") == std::string::npos)
   {
      return true;
   }
   else
   {
      return false;
   }
}


void AMFMHDdata::updatePSName()
{
   if (NULL != _AmFmInfoDataBinder)
   {
      if (_switchIndex == HDSTATION)
      {
         //std::string HDCallSign = getPSname(_HDCallSign, _CurrentFreq);
         if ((_band == T_e8_Tun_TunerBand__TUN_BAND_FM))
         {
            _AmFmInfoDataBinder->updateFmPsName(_HDCallSign);
            if (!_HDCallSign.empty())
            {
               VehicleSrvClient::poGetInstance()->pushTunerCANLongData(TUNER_MAINLINE, _HDCallSign);
            }
         }
         else if ((_band == T_e8_Tun_TunerBand__TUN_BAND_MW) || (_band == T_e8_Tun_TunerBand__TUN_BAND_LW))
         {
            _AmFmInfoDataBinder->updateAmPsName(_HDCallSign);
            if (!_HDCallSign.empty())
            {
               VehicleSrvClient::poGetInstance()->pushTunerCANLongData(TUNER_MAINLINE, _HDCallSign);
            }
         }
      }
      else if (_switchIndex == NONHDSTATION)
      {
         //std::string PSName = getPSname(_psName, _CurrentFreq);
         if (_band == T_e8_Tun_TunerBand__TUN_BAND_FM)
         {
            _AmFmInfoDataBinder->updateFmPsName(_psName);
            if (!_psName.empty())
            {
               VehicleSrvClient::poGetInstance()->pushTunerCANLongData(TUNER_MAINLINE, _psName);
            }
         }
         else if ((_band == T_e8_Tun_TunerBand__TUN_BAND_MW) || (_band == T_e8_Tun_TunerBand__TUN_BAND_LW))
         {
            _AmFmInfoDataBinder->updateAmPsName(_psName);
            if (!_psName.empty())
            {
               VehicleSrvClient::poGetInstance()->pushTunerCANLongData(TUNER_MAINLINE, _psName);
            }
         }
      }
   }
}


} // namespace Core
} // namespace App
