/**************************************************************************************
* @file         : SxmTestMonitor.cpp
* @addtogroup   : AppHmi_Sxm
* @brief        : Functional class to handle business logic for Sxm Diagnosis application.
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/
#include <hmibase/hall_std_if.h>   // Including HALL framework code
#include "SxmTestMonitor.h"
#include "PhoneTestMode.h"
#include "Common/DataModel/ListRegistry.h"
#include "Core/LanguageDefines.h"
#include "CanderaWidget/String/String.h"
#include "SxmAudioDataBindingUtils.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"

#define INITIALISE_SUBSRC_ID -1

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_TESTMODE_HALL
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_TESTMODE
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_Testmode_"
#define ETG_I_FILE_PREFIX             App::Core::SxmTestMonitor::
#include "trcGenProj/Header/SxmTestMonitor.cpp.trc.h"
#endif

#define MAXPRESETNOBANDONE          ((VS_UINT32) 6ul)
#define MAXPRESETNOBANDTHREE        ((VS_UINT32) 18ul)
#define MAXPRESETNOBANDTWO          ((VS_UINT32) 12ul)

#define MINPRESETNOBANDONE          ((VS_UINT32) 1ul)
#define MINPRESETNOBANDOTWO         ((VS_UINT32) 7ul)
#define MINPRESETNOBANDTHREE        ((VS_UINT32) 13ul)

#define CHANNELLIST_WINDOW_SIZE 20


namespace App {
namespace Core {
using namespace ::sxm_audio_main_fi;
using namespace ::sxm_main_fi_types;
using namespace ::sxm_audio_main_fi_types;
using namespace ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange;

std::map<uint8, std::string> SxmTestMonitor::_diagnosisMonitorList;

SxmTestMonitor* SxmTestMonitor::m_poSxmTestMonitor = NULL;
/**
* SxmTestMonitor Class constructor
* @param [in] : pSxmAudioFiProxy - SXM Audio FI proxy
*/
SxmTestMonitor::SxmTestMonitor()
   : _sxmAudioFiProxy(Sxm_audio_main_fiProxy::createProxy("SxmAudioFiPort", *this))
   , _audioSrcChgProxy(AudioSourceChangeProxy::createProxy("SxmAudioSourceChangePort", *this))
   , _isExternalDiagmodeOn(true)
   , _isDataServiceInfoSubscribeOn(false)
   , _isAudioSubInfoSubscribeOn(false)
   , _isClearAllDBandData(false)
   , _isPresetListAvailable(false)
   , _cStepSizeOne(1)
   , m_StartIndex_Channellist(0)
   , m_WindowSize_Channellist(CHANNELLIST_WINDOW_SIZE)
   , m_bBTSreamingAvailable(false)
{
   ETG_I_REGISTER_FILE();
   ListRegistry::s_getInstance().addListImplementation(LISTID_SXM, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_SXM_CHN_LIST, this);
   _diagnosisStmList.clear();
   _diagnosisMonitorList.clear();
   _diagnosisInfoList.clear();
   _channelInfoStatus.clear();
   _presetList.clear();
   aSXM_PresetChannelName.clear();
   ETG_TRACE_USR4(("SxmTestMonitor() Constructor"));

   if (_sxmAudioFiProxy)
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _sxmAudioFiProxy->getPortName());
      //ETG_TRACE_USR4(("SxmAudioFiPort proxy created.......%d", _sxmAudioFiProxy));
   }

   if (_audioSrcChgProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _audioSrcChgProxy->getPortName());
      //ETG_TRACE_USR4(("audioSourceChangePort proxy created.......%d", _audioSrcChgProxy));
   }
}


/**
* SxmTestMonitor Class destructor
*/
SxmTestMonitor::~SxmTestMonitor()
{
   ETG_TRACE_USR4(("~SxmTestMonitor() Destructor"));
   ETG_I_UNREGISTER_FILE();
   ListRegistry::s_getInstance().removeListImplementation(LISTID_SXM);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_SXM_CHN_LIST);
   _sxmAudioFiProxy.reset();
   _audioSrcChgProxy.reset();
   _diagnosisStmList.clear();
   _diagnosisMonitorList.clear();
   _diagnosisInfoList.clear();
   _isExternalDiagmodeOn = true;
   _isDataServiceInfoSubscribeOn = false;
   _isAudioSubInfoSubscribeOn = false;
   _isClearAllDBandData = false;
}


/**
* Utility function that registers for notifications updates from SXM Audio MIDW layer
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : stateChange - Current ASF state
*/
void SxmTestMonitor::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmTestMonitor::registerProperties"));
   if (_sxmAudioFiProxy && (proxy == _sxmAudioFiProxy))
   {
      //To Check : Register updates on entry to STM/DTM screens.
   }
}


/**
* Utility function that deregisters for notifications updates from SXM Audio MIDW layer
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : stateChange - Current ASF state
*/
void SxmTestMonitor::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmTestMonitor::deregisterProperties"));
   if ((proxy == _sxmAudioFiProxy) && (_sxmAudioFiProxy))
   {
      //To Check : De-register updates on exit to STM/DTM screens.
   }
}


void SxmTestMonitor::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3(("SxmTestMonitor::onAvailable()"));

   if ((proxy == _sxmAudioFiProxy) && (_sxmAudioFiProxy.get()))
   {
      ETG_TRACE_USR4(("SxmTestMonitor::onAvailable Sucess"));
      //To Check : Register updates on entry to STM/DTM screens.
      _sxmAudioFiProxy->sendSxmRadioIdUpReg(*this);
      _sxmAudioFiProxy->sendSxmAudioSubInfoUpReg(*this);
      _sxmAudioFiProxy->sendSxmDiagAntennaStateUpReg(*this);
      _sxmAudioFiProxy->sendDiagTestModeDataUpReg(*this);
      _sxmAudioFiProxy->sendSystemMessageUpReg(*this);
      _sxmAudioFiProxy->sendChannelInformationUpReg(*this);
      _sxmAudioFiProxy->sendPresetChannelUpdateUpReg(*this);
      _sxmAudioFiProxy->sendAutoCompareUpReg(*this);
   }
   if ((proxy == _audioSrcChgProxy) && (_audioSrcChgProxy.get()))
   {
      ETG_TRACE_USR4(("SxmTestMonitor::onAvailable Sucess audioSrcChgProxy"));
      _audioSrcChgProxy->sendActiveSourceGet(*this);
      _audioSrcChgProxy->sendActiveSourceListRegister(*this);
      _audioSrcChgProxy->sendActiveSourceRegister(*this);
   }
}


void SxmTestMonitor::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& stateChange)
{
   if ((proxy == _sxmAudioFiProxy) && (_sxmAudioFiProxy.get()))
   {
      _sxmAudioFiProxy->sendSxmRadioIdRelUpRegAll();
      _sxmAudioFiProxy->sendSxmAudioSubInfoRelUpRegAll();
      _sxmAudioFiProxy->sendSxmDiagAntennaStateRelUpRegAll();
      _sxmAudioFiProxy->sendDiagTestModeDataRelUpRegAll();
      _sxmAudioFiProxy->sendChannelInformationRelUpRegAll();
      _sxmAudioFiProxy->sendSystemMessageRelUpRegAll();
      _sxmAudioFiProxy->sendPresetChannelUpdateRelUpRegAll();
      _sxmAudioFiProxy->sendAutoCompareRelUpRegAll();
   }

   if ((proxy == _audioSrcChgProxy) && (_audioSrcChgProxy.get()))
   {
      _audioSrcChgProxy->sendActiveSourceDeregisterAll();
      _audioSrcChgProxy->sendActiveSourceListDeregisterAll();
   }
}


void SxmTestMonitor::onActivateSourceError(const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::ActivateSourceError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmTestMonitor::onActivateSourceError"));
}


void SxmTestMonitor::onActivateSourceResponse(const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::ActivateSourceResponse >& /*response*/)
{
   ETG_TRACE_USR4(("SxmTestMonitor::onActivateSourceResponse"));
}


void SxmTestMonitor::onActiveSourceError(const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::ActiveSourceError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmTestMonitor::onActiveSourceError"));
}


void SxmTestMonitor::onActiveSourceUpdate(const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::ActiveSourceUpdate >& update)
{
   ETG_TRACE_USR4(("SxmTestMonitor:: onActiveSourceUpdate():SrcId:%d  SubSrcId:%d  SinkSrcId:%d", update->getActiveSource().getSrcId(), update->getActiveSource().getSubSrcId(), update->getActiveSource().getSinkId()));

   if (update->getActiveSource().getSrcId() == SRC_PHONE_BTAUDIO)
   {
      ETG_TRACE_USR4(("SxmTestMonitor:: onActiveSourceUpdate() SRC_PHONE_BTAUDIO %d", update->getActiveSource().getSrcId()));
      m_bBTSreamingAvailable = true;
   }
   else if ((IsMediaSourceRange(update->getActiveSource().getSrcId())) || (IsTunerSourceRange(update->getActiveSource().getSrcId())))
   {
      m_bBTSreamingAvailable = false;
   }
   PhoneTestMode::poGetInstance()->updateBTList(m_bBTSreamingAvailable);
}


bool SxmTestMonitor::IsTunerSourceRange(int srcid)
{
   ETG_TRACE_USR4(("SxmTestMonitor:: IsTunerSourceRange() %d ", srcid));
   if (((srcid >= SRC_TUNER_FM) && (srcid <= SRC_DAB)) || (srcid == SRC_TUNER_XM) || (srcid == SRC_TUNER_DRM) || (srcid == SRC_TUNER_WX))
   {
      return true;
   }
   else
   {
      return false;
   }
}


bool SxmTestMonitor::IsMediaSourceRange(int srcid)
{
   ETG_TRACE_USR4(("SxmTestMonitor:: CheckMediaSourceRange() %d ", srcid));
   if (((srcid >= SRC_MEDIA_AUX) && (srcid <= SRC_SPI_ENTERTAIN)) || ((srcid >= SRC_CAMERAPORT1_CAM1) && (srcid <= SRC_CAMERAPORT4_HDMI)))
   {
      return true;
   }
   else
   {
      return false;
   }
}


bool SxmTestMonitor::getBTStreamingAvailability()
{
   return m_bBTSreamingAvailable;
}


void SxmTestMonitor::onActiveSourceListError(const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::ActiveSourceListError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmTestMonitor::onActiveSourceListError"));
}


void SxmTestMonitor::onActiveSourceListUpdate(const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::ActiveSourceListUpdate >& update)
{
   ETG_TRACE_USR4(("SxmTestMonitor::onActiveSourceListUpdate"));
}


bool SxmTestMonitor:: onCourierMessage(const SxmDTMTestModeMonitorSelReqMsg& oMsg)
{
   ETG_TRACE_USR4(("SxmTestMonitor::SxmDTMTestModeMonitorSelReqMsg"));
   if (_sxmAudioFiProxy)
   {
      ETG_TRACE_USR4(("SxmTestMonitor::Requesting Test Mode"));
      _sxmAudioFiProxy->sendDiagTestModeEnableStart(*this, oMsg.GetStartMonitor());
   }
   return true;
}


/**
* Callback to handle Courier Message SXMAudioCustomizeListReqMsg Request
* @param [in] : Reference of Courier::SXMAudioCustomizeListReqMsg
* @return     :  True-When Message is processed.
*/
bool SxmTestMonitor::onCourierMessage(const SXMAudioCustomizeListReqMsg& /*oMsg*/)
{
   ETG_TRACE_USR4(("SxmTestMonitor::SXMAudioCustomizeListReqMsg"));
   return true;
}


/**
* Send Midw Customize channel list request.
*/
void SxmTestMonitor::getCustomizeChannelList(uint16 m_StartIndex_Channellist, uint16 m_WindowSize_Channellist)
{
   if (_sxmAudioFiProxy)
   {
      //_sxmAudioFiProxy->sendGetChannelListStart(*this, m_StartIndex_Channellist, m_WindowSize_Channellist);  // Index:0,Range:0 for complete list.
      _sxmAudioFiProxy->sendGetChannelListStart(*this, 0, 0);       // Index:0,Range:0 for complete list.
   }
}


/**
* Error handling function for sxm Radio Id property
*/
void SxmTestMonitor::onSxmRadioIdError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
                                       const ::boost::shared_ptr< ::sxm_audio_main_fi::SxmRadioIdError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmTestMonitor::onSxmRadioIdError"));
}


/**
* Status handling function for sxm Radio Id property
*/
void SxmTestMonitor::onSxmRadioIdStatus(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
                                        const ::boost::shared_ptr< ::sxm_audio_main_fi::SxmRadioIdStatus >& status)
{
   ETG_TRACE_USR4(("SxmTestMonitor::onSxmRadioIdStatus"));

   if (status)
   {
      setRadioData(status->getRadioId());
   }
}


/**
* Error handling function for sxm audio sub info property
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : error Error type
*/
void SxmTestMonitor::onSxmAudioSubInfoError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::SxmAudioSubInfoError >& /* error*/)
{
   ETG_TRACE_ERR(("SxmTestMonitor::onSxmAudioSubInfoError"));
}


/**
* Status handling function for sxm audio sub info property
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : status - Property status info
*/
void SxmTestMonitor::onSxmAudioSubInfoStatus(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::SxmAudioSubInfoStatus >& status)
{
   ETG_TRACE_USR4(("SxmTestMonitor::onSxmAudioSubInfoStatus"));
   if (status)
   {
      ETG_TRACE_USR4(("SxmTestMonitor::onSxmAudioSubInfoStatus status:%d", status->getData().getSubStatus()));
      uint32 subStatus[Item_5] =
      {
         SXM__AUDIO_DIAG_NOTSUBSCRIBED,
         SXM__AUDIO_DIAG_SUBSCRIBED,
         SXM__AUDIO_DIAG_SUSPENDALERT,
         SXM__AUDIO_DIAG_SUSPENDED,
         SXM__AUDIO_DIAG_INVALID
      };

      std::string StrStatus = safeGetCString(subStatus[status->getData().getSubStatus()]);
      if (StrStatus == safeGetCString(SXM__AUDIO_DIAG_NOTSUBSCRIBED))
      {
         _isAudioSubInfoSubscribeOn = true;
      }
      else
      {
         _isAudioSubInfoSubscribeOn = false;
      }
   }
}


/**
* Error handling function for  sxm diag antenna state
*/
void SxmTestMonitor::onSxmDiagAntennaStateError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::SxmDiagAntennaStateError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmTestMonitor::onSxmDiagAntennaStateError"));
}


/**
* Status handling function for  sxm diag antenna state
*/
void SxmTestMonitor::onSxmDiagAntennaStateStatus(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::SxmDiagAntennaStateStatus >& status)
{
   ETG_TRACE_USR4(("SxmTestMonitor::onSxmDiagAntennaStateStatus"));
   if (status)
   {
      ETG_TRACE_USR4(("SxmTestMonitor::onSxmDiagAntennaStateStatus status:%d", status->getAntennaStatus()));
      uint32 antennaStatus[Item_5] =
      {
         SXM__AUDIO_DIAG_WAIT,
         SXM__AUDIO_DIAG_OK,
         SXM__AUDIO_DIAG_DISCONNECTED,
         SXM__AUDIO_DIAG_SHORTED,
         SXM__AUDIO_DIAG_UNKNOWN
      };

      setDiagAntennaStatus(antennaStatus[status->getAntennaStatus()]);
   }
   SxmAudioDataBindingUtils::poGetInstance()->  SxmUpdateAntennaStatus(status->getAntennaStatus());
}


/**
* Error handling function for diag test mode data property
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : error Error type
*/
void SxmTestMonitor::onDiagTestModeDataError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::DiagTestModeDataError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmTestMonitor::onDiagTestModeDataError"));
}


/**
* Status handling function for diag test mode data property
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : status - Property status info
*/
void SxmTestMonitor::onDiagTestModeDataStatus(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::DiagTestModeDataStatus >& status)
{
   ETG_TRACE_USR4(("SxmTestMonitor::onDiagTestModeDataStatus"));

   const T_SxmDataSubStatus& sxmSubscriptionState = status->getData().getSxmSubscriptionState();
   ETG_TRACE_USR4(("SxmAudioListHelper::setDiagTestModeData substatus:%d", sxmSubscriptionState.getSubStatus()));

   if (status)
   {
      setDiagTestModeData(status->getData());
      setDiagSxmVersion();
      refreshSxmAudioList(LISTID_SXM);
      //refreshSxmAudioList(LIST_ID_SXM_DTM_TESTMODE_MONITOR_SELECTION, Sxm_Scenes_SERVICE__SXM_DEV_TESTMODE_MONITOR);
      //refreshSxmAudioList(LIST_ID_SXM_MONITOR, Sxm_Scenes_SERVICE__SXM_STM_MAIN);*/
   }
}


/**
* Error handling function for diag test mode enable method
*/
void SxmTestMonitor::onDiagTestModeEnableError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::DiagTestModeEnableError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmTestMonitor::onDiagTestModeEnableError"));
}


/**
* Result handling function for diag test mode enable method
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : result - Method result data
*/
void SxmTestMonitor::onDiagTestModeEnableResult(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::DiagTestModeEnableResult >& /*result*/)
{
   ETG_TRACE_USR4(("SxmTestMonitor::onDiagTestModeEnableResult"));
}


void SxmTestMonitor::setRadioData(std::string radioId)
{
   _radioID = radioId;
}


void SxmTestMonitor::setDiagAntennaStatus(uint32 antennastatus)
{
   _diagnosisStmList[SXM_DIAG_ANTENNA_STATUS] = antennastatus;
}


void SxmTestMonitor::setDiagAudioSubInfo(uint32 audioSubInfo)
{
   _diagnosisStmList[SXM_DIAG_AUDIO_SUBSTATUS] = audioSubInfo;
   _diagnosisInfoList[SXMINFO_AUDIO] = audioSubInfo;
}


void SxmTestMonitor::refreshSxmAudioList(unsigned int listId)
{
   ETG_TRACE_USR4(("SxmAudioListHelper::refreshSxmAudioList"));
   tSharedPtrDataProvider dataProvider = tSharedPtrDataProvider();

   switch (listId)
   {
      case LISTID_SXM:
      {
         dataProvider = getSxmTestModeMonitorSelection();
         break;
      }

      case LISTID_SXM_CHN_LIST:
      {
         break;
      }
      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
   }
}


tSharedPtrDataProvider SxmTestMonitor::getSxmTestModeMonitorSelection()
{
   ETG_TRACE_USR4(("SxmTestMonitor::getSxmTestModeMonitorSelection"));

   //Look up table for Dtm Labels
   const char* SxmDtmDataLabel[] =
   {
      "Signal Strength:",
      "Tuner Status:",
      "Lock A:",
      "Lock B:",
      "BER:",
      "C/N S1:",
      "C/N S2:",
      "RSErrs:",
      "Tuner Freq:",
      "RSSI:",
      "Overlay Status:",
      "OBER S1:",
      "OBER S2:",
      "OBER T:",
      "OTP 0:",
      "OTP 1:",
      "OTP 2:",
      "Module Type:",
      "Module HW:",
      "Module SW:",
      "SXI:",
      "BB:",
      "HDEC:",
      "RF:",
      "SPL:",
      "Capability:",
      "Sub Status:",
      "Reason Code:",
      "Suspend Date:",
      "SMS Version:",
      "H/U SW Version:",
      "H/U HW Version:",
      "CGS DB:",
      "Tab Weather DB:",
      "WSA DB:",
      "Fuel (US) DB:",
      "Fuel (Canada) DB:",
      "Movies DB:",
      "Parking DB:",
      "Sports DB:",
      "Stocks DB:",
      "Phonetics DB:",
   };

   ListDataProviderBuilder listBuilder(LISTID_SXM, "Layer_SxmListEntry");
   ETG_TRACE_USR4(("Map size=%d", _diagnosisMonitorList.size()));

   for (uint8 idx = SXM_DIAG_SIGNAL_STRENGTH; idx <= SXM_DIAG_PHONETICS_DB; ++idx)
   {
      listBuilder.AddItem(idx, 0, "Layer_SxmListEntry").AddData(SxmDtmDataLabel[idx]).AddData(SxmTestMonitor::_diagnosisMonitorList[idx].c_str());
      ETG_TRACE_USR4(("SxmTestMonitor sxmdatalabel == %d diagnosis == %s", SxmDtmDataLabel[idx], SxmTestMonitor::_diagnosisMonitorList[idx]));
   }
   tSharedPtrDataProvider dataprovider = listBuilder.CreateDataProvider();
   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataprovider)));
   return dataprovider;
}


void SxmTestMonitor::setDiagTestModeData(const ::sxm_audio_main_fi_types::T_SxmDiagTestModeData& testModeData)
{
   const T_SxmDataSubStatus& sxmSubscriptionState = testModeData.getSxmSubscriptionState();

   ETG_TRACE_USR4(("SxmAudioListHelper::setDiagTestModeData substatus:%d", sxmSubscriptionState.getSubStatus()));
   std::string subStatus = "x";
   std::string signal_Quality;
   char iStr[25];
   snprintf(iStr, sizeof(iStr), "%02x", sxmSubscriptionState.getSubStatus());
   subStatus.append(iStr);
   _diagnosisMonitorList[SXM_DIAG_SUBSTATUS] = subStatus;

   ETG_TRACE_USR4(("SxmAudioListHelper::setDiagTestModeData ReasonCode:%d", sxmSubscriptionState.getReasonCode()));
   std::string reasonCode = "x";
   char iStr2[25];
   snprintf(iStr2, sizeof(iStr2), "%02x", sxmSubscriptionState.getReasonCode());
   reasonCode.append(iStr2);
   _diagnosisMonitorList[SXM_DIAG_REASON_CODE] = reasonCode;

   const T_SxmDetSignalQuality& sxmDetSignalQuality = testModeData.getDetSignalQuality();
   _diagnosisMonitorList[SXM_DIAG_SIGNAL_STRENGTH] = sxmDetSignalQuality.getSigQuality();
   signal_Quality = sxmDetSignalQuality.getSigQuality();
   //ETG_TRACE_USR4(("SxmAudioListHelper::signal_Quality == %x", signal_Quality));
   //ETG_TRACE_USR4(("SxmAudioListHelper::Signal Quality == %x", sxmDetSignalQuality.getSigQuality()));
   _diagnosisMonitorList[SXM_DIAG_TUNER_STATUS] = sxmDetSignalQuality.getTunerStatus();
   _diagnosisMonitorList[SXM_DIAG_LOCKA] = sxmDetSignalQuality.getENSALockStatus();
   _diagnosisMonitorList[SXM_DIAG_LOCKB] = sxmDetSignalQuality.getENSBLockStatus();
   _diagnosisMonitorList[SXM_DIAG_BER] = "S1: " + sxmDetSignalQuality.getBerS1() + " S2: " + sxmDetSignalQuality.getBerS2() +
                                         " T: " + sxmDetSignalQuality.getBerT();
   _diagnosisMonitorList[SXM_DIAG_CN_S1] = "CN1A: " + sxmDetSignalQuality.getCtoNS1A() + " B: " + sxmDetSignalQuality.getCtoNS1B();
   _diagnosisMonitorList[SXM_DIAG_CN_S2] = "CN2A: " + sxmDetSignalQuality.getCtoNS2A() + " B: " + sxmDetSignalQuality.getCtoNS2B();
   _diagnosisMonitorList[SXM_DIAG_RSERRS] = "W: " + sxmDetSignalQuality.getRSErrsWords() + " S: " + sxmDetSignalQuality.getRSErrSatSymb() +
         " T: " + sxmDetSignalQuality.getRSErrTerrSymb();
   _diagnosisMonitorList[SXM_DIAG_TUNER_FREQUENCY] = sxmDetSignalQuality.getTunerCarrFreqOfset();
   _diagnosisMonitorList[SXM_DIAG_RSSI] = sxmDetSignalQuality.getRSSI();

   const T_SxmDetOvrlSignalQuality& sxmDetOvrlSignalQuality = testModeData.getDetOvrlSignalQuality();
   _diagnosisMonitorList[SXM_DIAG_OVERLAY_STATUS] = sxmDetOvrlSignalQuality.getReceiverState();
   _diagnosisMonitorList[SXM_DIAG_OBER_S1] = "A: " + sxmDetOvrlSignalQuality.getOBerS1A() + " B: " + sxmDetOvrlSignalQuality.getOBerS1B();
   _diagnosisMonitorList[SXM_DIAG_OBER_S2] = "A: " + sxmDetOvrlSignalQuality.getOBerS2A() + " B: " + sxmDetOvrlSignalQuality.getOBerS2B();
   _diagnosisMonitorList[SXM_DIAG_OBER_T] = "A: " + sxmDetOvrlSignalQuality.getOBerTA() + " B: " + sxmDetOvrlSignalQuality.getOBerTB();
   _diagnosisMonitorList[SXM_DIAG_OTP0] = "A: " + sxmDetOvrlSignalQuality.getOBERTrbWrdErRt0A() + " B: " + sxmDetOvrlSignalQuality.getOBERTrbWrdErRt0B();
   _diagnosisMonitorList[SXM_DIAG_OTP1] = "A: " + sxmDetOvrlSignalQuality.getOBERTrbWrdErRt1A() + " B: " + sxmDetOvrlSignalQuality.getOBERTrbWrdErRt1B();
   _diagnosisMonitorList[SXM_DIAG_OTP2] = "A: " + sxmDetOvrlSignalQuality.getOBERTrbWrdErRt2A() + " B: " + sxmDetOvrlSignalQuality.getOBERTrbWrdErRt2B();

   const T_SxmVersionInfo& sxmVersionInfo = testModeData.getSxmVersionInfo();
   _diagnosisMonitorList[SXM_DIAG_MODULE_TYPE] = sxmVersionInfo.getModuleId();
   _diagnosisMonitorList[SXM_DIAG_MODULE_HW] = sxmVersionInfo.getHwRev();
   _diagnosisMonitorList[SXM_DIAG_MODULE_SW] = sxmVersionInfo.getSwRev();
   _diagnosisMonitorList[SXM_DIAG_SXI] = sxmVersionInfo.getSxiRev();
   _diagnosisMonitorList[SXM_DIAG_BB] = sxmVersionInfo.getBbRev();
   _diagnosisMonitorList[SXM_DIAG_HDEC] = sxmVersionInfo.getHDecRev();
   _diagnosisMonitorList[SXM_DIAG_RF] = sxmVersionInfo.getRFRev();
   _diagnosisMonitorList[SXM_DIAG_SPL] = sxmVersionInfo.getSPLRev();
   _diagnosisMonitorList[SXM_DIAG_CAPABILITY] = sxmVersionInfo.getCapability();

   ETG_TRACE_USR4(("Diagnosis Map size=%d", _diagnosisMonitorList.size()));
   for (std::map<uint8, std::string>::const_iterator it = _diagnosisMonitorList.begin(); it != _diagnosisMonitorList.end(); it++)
   {
      ETG_TRACE_USR4(("Map index == %d value == %s", it->first, it->second.c_str()));
   }
}


void SxmTestMonitor::setDiagSxmVersion()
{
   ETG_TRACE_USR4(("SxmTestMonitor::setDiagSxmVersion"));
}


/**
* Helper Function to have synchronized access to FeatStd::String via GetCString interface
*
* param[in] : textId - text id of the string
* @return    : string after conversion
*/
std::string SxmTestMonitor::safeGetCString(unsigned int textId)
{
   FeatStd::String safestring = Candera::String(textId);  //get a text via text id
   std::string formatedString;
   formatedString.clear();
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(safestring);
   formatedString = safestring.GetCString();
   SECURE_FEATSTD_STRING_ACCESS_END()
   return formatedString;
}


bool SxmTestMonitor::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   ETG_TRACE_USR3(("DABStationList : onCourierMessage ListDateProviderReqMsg"));
   return ListRegistry::s_getInstance().updateList(oMsg.GetListId(), oMsg.GetStartIndex(), oMsg.GetWindowElementSize());
}


void SxmTestMonitor::updateList(const uint32& listID)
{
   ETG_TRACE_USR3((" SxmAudioListHelper: updateList"));
   ListRegistry::s_getInstance().updateList(listID);
}


tSharedPtrDataProvider SxmTestMonitor::getListDataProvider(const ListDateProviderReqMsg& listDataInfo)
{
   m_StartIndex_Channellist = listDataInfo.GetStartIndex();
   m_WindowSize_Channellist = listDataInfo.GetWindowElementSize();

   ETG_TRACE_USR3(("SxmTestMonitor : getListDataProvider"));
   tSharedPtrDataProvider ptr = tSharedPtrDataProvider();
   ETG_TRACE_USR3(("SxmAudioList : list ID %d", listDataInfo.GetListId()));
   switch (listDataInfo.GetListId())
   {
      case LISTID_SXM:
      {
         ETG_TRACE_USR3(("SxmTestMonitor : LISTID_SXM"));
         getSxmTestModeMonitorSelection();
         break;
      }
      case LISTID_SXM_CHN_LIST:
      {
         ETG_TRACE_USR3(("SxmTestMonitor:: StartIndex_Channellist %d", m_StartIndex_Channellist));
         ETG_TRACE_USR3(("SxmTestMonitor:: WindowSize_Channellist %d", m_WindowSize_Channellist));
         ETG_TRACE_USR3(("SxmTestMonitor : LISTID_SXM_CHN_LIST"));
         getCustomizeChannelList(m_StartIndex_Channellist, m_WindowSize_Channellist);
         break;
      }
      default:
         break;
   }
   return ptr;
}


/**
* Error handling function for channel information property
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : error Error type
*/
void SxmTestMonitor::onChannelInformationError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< ChannelInformationError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmTestMonitor::onChannelInformationError"));
}


/**
* Status handling function for channel information property
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : status - Property status info
*/
void SxmTestMonitor::onChannelInformationStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< ChannelInformationStatus >& status)
{
   ETG_TRACE_USR4(("SxmTestMonitor::onChannelInformationStatus"));

   uint16 lchnid = status->getChannelInformation().getChannelNumber();

   if ((lchnid <= END_CHANNEL_ID))
   {
      _channelInfoStatus = status->getChannelInformation();
      SxmAudioDataBindingUtils* pAudioUtilsInstance = SxmAudioDataBindingUtils::poGetInstance();
      //Send Information to update channel information.
      pAudioUtilsInstance->setSxmChannelInfoData(_channelInfoStatus);
      //TChannelInformation channelInfo = {_channelInfoStatus.getChannelNumber(), _channelInfoStatus.getChannelName()};
   }
}


/**
* Error handling function for select channel method
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : error - Error type
*/
void SxmTestMonitor::onSelectChannelError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SelectChannelError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmTestMonitor::onSelectChannelError"));
}


/**
* Result handling function for select channel method
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : result - Method result data
*/
void SxmTestMonitor::onSelectChannelResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SelectChannelResult >& result)
{
   ETG_TRACE_USR4(("SxmTestMonitor::onSelectChannelResult"));
}


/**
* Error handling function for system messages property
* @param [in] : proxy SXM Audio FI proxy class instance
* @param [in] : error Error type
*/
void SxmTestMonitor::onSystemMessageError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SystemMessageError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmTestMonitor::onSystemMessageError"));
}


/**
* Status handling function for system messages property
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : status - Property status info
*/
void SxmTestMonitor::onSystemMessageStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SystemMessageStatus >& status)
{
   const T_e8_SystemMessage&  systemErrorType = status->getSystemErrorType();
   const uint16& cid = status->getCID();
   ETG_TRACE_USR4(("SxmTestMonitor::onSystemMessageStatus u8ErrorType:%d,""channelid:%d", systemErrorType , cid));

   //Post message to controller based on advisory message.
   // _noSystemMessageError = (T_e8_SystemMessage__XMTUN_SYSTEM_MSG_NO_ERROR == systemErrorType);
   // SxmAudioDataBindingUtils::instance()->setSystemMessageStatus(systemErrorType);
   // postPopupMessageForAudio(systemErrorType, cid);

   /* if ((T_e8_SystemMessage__XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE == systemErrorType ||
   T_e8_SystemMessage__XMTUN_SYSTEM_MSG_CHANNEL_UNAUTHORIZED == systemErrorType ||
   T_e8_SystemMessage__XMTUN_SYSTEM_MSG_CHANNEL_MATURE == systemErrorType ||
   T_e8_SystemMessage__XMTUN_SYSTEM_MSG_CHANNEL_LOCKED == systemErrorType)
   &&      _channelInfoStatus.getChannelNumber() == cid)
   {
   } */
}


/**
* Error handling function for get channel list method
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : error - Error type
*/
void SxmTestMonitor::onGetChannelListError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& proxy, const boost::shared_ptr< GetChannelListError >& error)
{
   ETG_TRACE_ERR(("SxmTestMonitor::onGetChannelListError"));
}


/**
* Result handling function for get channel list method
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : result - Method result data
*/
void SxmTestMonitor::onGetChannelListResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& proxy, const boost::shared_ptr< GetChannelListResult >& result)
{
   ETG_TRACE_USR4(("SxmTestMonitor::onGetChannelListResult"));
   ETG_TRACE_USR4(("SxmTestMonitor::onGetChannelListSize %d",  result->getChannelList().size()));
   if (T_e8_MethodStatus__XMTUN_MS_SUCCESS == result->getStatus())
   {
      const ::std::vector<T_ChannelListEntry >& lChannelList = result->getChannelList();

      ListDataProviderBuilder listBuilder(LISTID_SXM_CHN_LIST, "Layer_SxmChannelListEntry");

      uint16 lastItemIndex_List = m_StartIndex_Channellist + m_WindowSize_Channellist - 1;
      if ((lastItemIndex_List == 0) || (lastItemIndex_List > lChannelList.size()))
      {
         lastItemIndex_List = lChannelList.size() - 1;
      }

      uint8 dataindex = 0;
      if (result->getChannelList().size())
      {
         for (uint8 idx = 0, dataindex = m_StartIndex_Channellist; ((idx < lChannelList.size()) && (dataindex <= lastItemIndex_List)); ++idx, dataindex++)
         {
            ETG_TRACE_USR4(("SxmTestMonitor CHANEL ID == %d", lChannelList[dataindex].getChannelID()));
            ETG_TRACE_USR4(("SxmTestMonitor CHANEL NAME == %s", lChannelList[dataindex].getChannelName().c_str())) ;
            listBuilder.AddItem(dataindex, 0, "Layer_SxmChannelListEntry").AddData(lChannelList[dataindex].getChannelName().c_str()).AddData(lChannelList[dataindex].getChannelID());
         }
         tSharedPtrDataProvider dataprovider = listBuilder.CreateDataProvider(m_StartIndex_Channellist, lChannelList.size());
         POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataprovider)));
      }
   }
}


/**
* Error handling function for auto compare property;
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : error Error type
*/
void SxmTestMonitor::onAutoCompareError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
                                        const boost::shared_ptr< AutoCompareError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmTestMonitor::onAutoCompareError"));
}


/**
* Status handling function for auto compare property
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : status - Property status info
*/
void SxmTestMonitor::onAutoCompareStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< AutoCompareStatus >& status)
{
   ETG_TRACE_USR4(("SxmTestMonitor::onAutoCompareStatus CurrentTunerElemNr:%d,TunerElemNr:%d", status->getCurrentTunerElemNr(), status->getTunerElemNr()));
   //if (status)
   //{
   //   SxmAudioListHelper* pAudioUtilsInstance = SxmAudioListHelper::instance();
   //   uint8 presetIndex = status->getTunerElemNr();
   //   if (!SxmAudioDataBindingUtils::instance()->getSxmTuneScanStatus())
   //   {
   //      _activePresetIndex = presetIndex;
   //   }
   //   pAudioUtilsInstance->setActivePresetIndex(presetIndex);
   //   if (_sxmAudioFiProxy->hasSystemMessage() &&
   //         T_e8_SystemMessage__XMTUN_SYSTEM_MSG_SXM_INITIALIZING != _sxmAudioFiProxy->getSystemMessage().getSystemErrorType() &&
   //         SxmAudioDataBindingUtils::instance()->getSxmChannelInfoSyncStatus())
   //   {
   //      pAudioUtilsInstance->updatePresetList(true);
   //   }
   //   //To display preset info in TPop Audio.
   //   SxmAudioDataBindingUtils::instance()->setSxmChannelInfoPresetNumber(presetIndex);
   //}
}


/**
* Error handling function for get PresetChannelUpdate Property
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : error - Error type
*/
void SxmTestMonitor::onPresetChannelUpdateError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr<PresetChannelUpdateError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmTestMonitor::onPresetChannelUpdateError"));
}


/**
* Status handling function for get PresetChannelUpdate Property
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : status - preset ChannelUpdate status
*/
void SxmTestMonitor::onPresetChannelUpdateStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr<PresetChannelUpdateStatus >& status)
{
   //ETG_TRACE_USR4(("SxmTestMonitor::onPresetChannelUpdateStatus PresetBand:%d,PresetIndex:%d", status->getPresetBand(),  status->getPresetIndex()));

   if (status && _isPresetListAvailable && status->getPresetBand() && status->getPresetIndex())
   {
      uint8 presetIndex = status->getPresetIndex() + ((status->getPresetBand() - 1) * MAXPRESETNOBANDONE);
      ETG_TRACE_USR4(("SxmTestMonitor::onPresetChannelUpdateStatus PresetBand:%d,PresetIndex:%d", status->getPresetBand(), presetIndex));
      updatePresetItem(presetIndex, status->getChannelInfo());// max presets is 18.
      //if ((presetIndex < (MAXPRESETNOBANDTHREE + 1)) &&
      //      (SxmAudioListHelper::instance()->updatePresetItem(presetIndex, status->getChannelInfo())))// max presets is 18.
      //{
      //   getPresetChannlLogo(false, presetIndex , status->getChannelInfo().getServiceID());
      //}
   }
   else if (!_isPresetListAvailable)
   {
      getPresetList();
   }
}


/**
* Helper function to get preset list on startup.
*/
void SxmTestMonitor::getPresetList()
{
   //Set Autocompare parameters
   if (_sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendSetAutoCompareParametersStart(*this, MINPRESETNOBANDONE, MAXPRESETNOBANDTHREE);
      //get Preset list for band1;
      _sxmAudioFiProxy->sendGetPresetListStart(*this, 255);//255:Get all presets.
   }
}


/**
* Error handling function for get PresetList Method
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : error Error type
*/
void SxmTestMonitor::onGetPresetListError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< GetPresetListError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmTestMonitor::onGetPresetListError"));
}


/**
* Result handling function for get PresetList Method
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : result - preset list result
*/
void SxmTestMonitor::onGetPresetListResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< GetPresetListResult >& result)
{
   ETG_TRACE_USR4(("SxmTestMonitor::onGetPresetListResult size:%d", result->getPresetChannels().size()));

   if (T_e8_MethodStatus__XMTUN_MS_SUCCESS == result->getStatus())
   {
      //Send preset list & Check if their is change in content.
      setPresetList(result->getPresetChannels());
      _isPresetListAvailable = true;
   }
}


/**
* Helper function to set preset list.
* @param [In] : presetList - Preset list received from middleware.
*/
void SxmTestMonitor::setPresetList(const ::std::vector< T_PresetListEntry >& presetList)
{
   _presetList = presetList;
}


/**
* Error handling function for set auto compare parameters method
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : error - Error type
*/
void SxmTestMonitor::onSetAutoCompareParametersError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SetAutoCompareParametersError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioPreset::onSetAutoCompareParametersError"));
}


/**
* Result handling function for set auto compare parameters method
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : result - Method result data
*/
void SxmTestMonitor::onSetAutoCompareParametersResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SetAutoCompareParametersResult >& /*result*/)
{
   ETG_TRACE_USR4(("SxmAudioPreset::onSetAutoCompareParametersResult"));
}


/**
* Helper function to GetPresetListElement
* @param [In] : presetIdx - PresetIndx.
* @param [In] : presetElement - Channel Info for preset entry
* return value : bool - true - if data is updated
*/
bool SxmTestMonitor::updatePresetItem(uint8 presetIdx, const ::sxm_audio_main_fi_types::T_PresetListEntry& presetElement)
{
   uint8 updatedIdx =  presetIdx - 1;//Presetindex starts from 1.
   bool isUpdated = false;
   ETG_TRACE_USR4(("SxmTestMonitor::updatePresetItem sid:%d,presetIdx:%d", presetElement.getServiceID(), updatedIdx));

   if ((updatedIdx < _presetList.size()) && (_presetList[updatedIdx] != presetElement))
   {
      ETG_TRACE_USR4(("SxmTestMonitor::updatePresetItem data updated"));
      setPresetListElement(presetIdx, presetElement);
      isUpdated = true;
   }
   return isUpdated;
}


/**
* Helper function to SetPresetListElement
* @param [In] : presetIdx - PresetIndx Modified.
* @param [In] : presetElement - Channel Info for preset entry
*/
void SxmTestMonitor::setPresetListElement(uint8 presetIdx, const T_PresetListEntry& presetElement)
{
   uint8 updatedIdx =  presetIdx - 1;//Presetindex starts from 1.
   if (updatedIdx < _presetList.size())
   {
      ETG_TRACE_USR4(("SxmTestMonitor::setPresetListElement sid:%d,presetIdx:%d", presetElement.getServiceID(), updatedIdx));
      _presetList[updatedIdx].setServiceID(presetElement.getServiceID());
      _presetList[updatedIdx].setChannelID(presetElement.getChannelID());
      _presetList[updatedIdx].setChannelNameAvail(presetElement.getChannelNameAvail());
      _presetList[updatedIdx].setChannelName(presetElement.getChannelName());
      _presetList[updatedIdx].setChannelActive(presetElement.getChannelActive());
      _presetList[updatedIdx].setChnGraphicAvail(presetElement.getChnGraphicAvail());
      _presetList[updatedIdx].setIsSmartFavorite(presetElement.getIsSmartFavorite());
      _presetList[updatedIdx].setIsLocked(presetElement.getIsLocked());
      _presetList[updatedIdx].setIsAvailable(presetElement.getIsAvailable());
      _presetList[updatedIdx].setIsMature(presetElement.getIsMature());
      _presetList[updatedIdx].setIsSkipped(presetElement.getIsSkipped());

      aSXM_PresetChannelName.push_back(StructPrestList(updatedIdx, presetElement.getChannelName()));
   }
   ETG_TRACE_USR3(("No.of aSXM_PresetChannelName Presets %d", aSXM_PresetChannelName.size()));
   if (aSXM_PresetChannelName.size() == 18)
   {
      SxmAudioDataBindingUtils::poGetInstance()->updateSXMPresets(aSXM_PresetChannelName);
   }
}


/**
* Description     : Function called on any button press in the application
*
* @param[in]      : ButtonReactionMsg
* @return         : bool
*/
/*  bool SxmTestMonitor::onCourierMessage(const ButtonReactionMsg& oMsg)
  {
      bool isRetVal = false;
      ListProviderEventInfo info;

      if ((ListProviderEventInfo::GetItemIdentifierInfo(oMsg.GetSender(), info) == true))
      {
          unsigned int listId = info.getListId();     // the list id for generic access
          uint32 hdlRow = info.getHdlRow();     // normaly the itr
          unsigned int hdlCol = info.getHdlCol();     // if more than 1 active element in one list row, e.g. Button in a button line

          ETG_TRACE_USR4(("AMFMStationList::onCourierMessage::ButtonReactionMsg::%d", oMsg.GetEnReaction()));

          if (oMsg.GetEnReaction() == enRelease)
          {
              switch (info.getListId())
              {
              case LISTID_SXM_CHN_LIST:
                  {
                      ETG_TRACE_USR4(("AMFMStationList::onCourierMessage::ButtonReactionMsg LISTID_FM %d", hdlRow));
                      SxmListSelectedIndex(hdlRow);
                      isRetVal = true;
                  }
                  break;
              default:
                  break;
              }
              if (isRetVal)
              {
                  POST_MSG((COURIER_MESSAGE_NEW(ButtonListItemUpdMsg)(listId, hdlRow, hdlCol, enRelease)));
              }
          }
      }

      return isRetVal;
  }*/

void SxmTestMonitor::SxmListSelectedIndex(uint32& selectedIndex)
{
   if (selectedIndex < 150)
   {
      ETG_TRACE_USR4(("SxmListSelectedIndex::onCourierMessage::on touch of list "));
      ((_sxmAudioFiProxy.get()) && _sxmAudioFiProxy->sendSelectChannelStart(*this, T_e8_SelectChannelMode__XMTUN_SELECT_CHANNEL, selectedIndex, 0, _cStepSizeOne));
   }
}


void SxmTestMonitor::onRecallPresetError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< RecallPresetError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioPreset::onRecallPresetError"));
}


void SxmTestMonitor::onRecallPresetResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< RecallPresetResult >& result)
{
   ETG_TRACE_USR4(("SxmTestMonitor::onRecallPresetResult"));
   if (result->getStatus())
   {
      //
   }
}


/**
* Callback to handle Courier Message SXMAudioRecallorStoreReqMsg Request
* @param [in] : Reference of Courier::SXMAudioRecallorStoreReqMsg
* @return     : True-When Message is processed.
*/
bool SxmTestMonitor::onCourierMessage(const SXMAudioRecallorStoreReqMsg& oMsg)
{
   uint8 u8PresetElementNo = oMsg.GetPresetRowIndex();
   ETG_TRACE_USR4(("u8PresetElementNois %d ", u8PresetElementNo));
   // const SXMAudioRecallorStoreReqMsg *copoSXMAudioRecallorStoreReqMsg = <const SXMAudioRecallorStoreReqMsg *>(&corfoMsg);
   //uint8 PresetNo = copoSXMAudioRecallorStoreReqMsg->GetPresetRowIndex();
   if (_sxmAudioFiProxy)
   {
      //if (!oMsg.GetPresetAction())
      //{
      //    //Store Preset
      //    _sxmAudioFiProxy->sendStorePresetStart(*this, u8PresetElementNo);
      //}
      //else

      //Recall Preset
      _sxmAudioFiProxy->sendRecallPresetStart(*this, u8PresetElementNo);
   }
   //SxmAudioDataBindingUtils::instance()->setDefaultSxmAudioPlayPauseData(T_e8_IRPlaybackStatus__SXM_AUDIO_IR_PLAY_STATE);
   return true;
}


/**
* Callback to handle Courier Message SXMAudioAllocateReqMsg Request
* @param [in] : Reference of Courier::SXMAudioAllocateReqMsg
* @return     : True-When Message is processed.
*/
bool SxmTestMonitor::onCourierMessage(const SXMAudioAllocateReqMsg& oMsg)
{
   ETG_TRACE_USR4(("SxmTestMonitor::SXMAudioAllocateReqMsg"));
   int sourceID = SRC_TUNER_XM;
   sourceData srcData;
   srcData.setSrcId(sourceID);
   srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
   ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData , true)));
   return true;
}


}  // namespace Core
}  // namespace App
