/**************************************************************************************
* @file         : HmiStateHandler.cpp
* @author       : Sriganesh
* @addtogroup   : System HALL
* @brief        : Class contains necessary business logic required to handle Display Settings.
* @copyright    : (C) 2018 Robert Bosch GmbH
*                 (C) 2018 Robert Bosch Engineering and Business Solutions Limited
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "HmiStateHandler.h"
//#include "ClientServicesIFProxy.h"
#include "midw_fi_if.h"
#include "spm_core_fi_typesConst.h"
#include "CabinASettingListhandler.h"
#include "CabinAGlassSettingListhandler.h"
#include "spm_core_fi_typesConst.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_SYSTEM_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_SYSTEM
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_SYSTEM_"
#define ETG_I_FILE_PREFIX                  HmiStateHandler::
#include "trcGenProj/Header/HmiStateHandler.cpp.trc.h"
#endif


using namespace ::SPM_CORE_FI;
using namespace ::spm_core_fi_types;

namespace App {
namespace Core {

HmiStateHandler* HmiStateHandler::_HmiStateHandler = NULL;

/*****************************************************************************
* @brief   HmiStateHandler Constructor
*****************************************************************************/
HmiStateHandler::HmiStateHandler()
{
   ETG_TRACE_USR2(("HmiStateHandler Constructor"));
   _spmProxy = (SPM_CORE_FIProxy::createProxy("spmFiPort", *this));
   HMIState = 0;
}


/*****************************************************************************
* @brief   HmiStateHandler Destructor
*****************************************************************************/
HmiStateHandler::~HmiStateHandler()
{
   ETG_TRACE_USR2(("HmiStateHandler Destructor"));
}


/*****************************************************************************
* onAvailable: Register for properties once service is available
*****************************************************************************/
void HmiStateHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
                                  const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_COMP(("HmiStateHandler::onAvailable"));
   _spmProxy->sendHmiStateUpReg(*this);
}


/*****************************************************************************
* onUnavailable: De-register for properties once service is unavailable
*****************************************************************************/
void HmiStateHandler::onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
                                    const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_COMP(("HmiStateHandler::onUnavailable"));
   _spmProxy->sendHmiStateRelUpRegAll();
}


/*****************************************************************************
* @brief   HmiStateHandler onHmiStateError
*****************************************************************************/
void HmiStateHandler::onHmiStateError(const ::boost::shared_ptr< SPM_CORE_FIProxy >& proxy, const boost::shared_ptr< HmiStateError >& error)
{
   if ((NULL != _spmProxy.get()) && (_spmProxy == proxy))
   {
      ETG_TRACE_ERR(("HmiStateHandler: onHmiStateError"));
   }
}


/*****************************************************************************
* @brief   HmiStateHandler onHmiStateStatus
*****************************************************************************/
void HmiStateHandler::onHmiStateStatus(const ::boost::shared_ptr< SPM_CORE_FIProxy >& proxy, const boost::shared_ptr< HmiStateStatus >& status)
{
   if ((NULL != _spmProxy.get()) && (_spmProxy == proxy) && (status->hasHmiState()))
   {
      HMIState = status->getHmiState(); // lcm state , pnm state,
      ETG_TRACE_USR1(("HmiStateHandler: onHmiStateStatus: <- %d", HMIState));
#ifdef VARIANT_S_FTR_ENABLE_CABIN_A_SUPPORT
      if (HMIState == T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_ON_PARTIAL_USER) //T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_ON_PARTIAL_USER=46u
      {
         ETG_TRACE_USR1(("HmiStateHandler: onHmiStateStatus:UpdateCockpitModeStatus <- %d", HMIState));
         CabinAGlassSettingListhandler::poGetInstance()->UpdateCockpitModeStatus(HMIState);
      }
      CabinASettingListhandler::poGetInstance()->UpdateHMIState2HMI(HMIState);
#endif
   }
}


bool HmiStateHandler::isPartialUserModeOn()
{
   bool partialusermode = false;

   if (HMIState == T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_ON_PARTIAL_USER) //T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_ON_PARTIAL_USER=46u// pnm group 10
   {
      partialusermode = true; //PNM Enable
   }
   ETG_TRACE_USR1(("HmiStateHandler::isPartialUserModeOn() HMIState partialusermode %d %d", HMIState, partialusermode));
   return partialusermode;
}


bool HmiStateHandler::isUserStandbyOff()
{
   bool UserStandbyOff = false;

   if (HMIState ==  T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_STANDBY) //T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_STANDBY=44u
   {
      UserStandbyOff = true;// user off or tipper hmi off
   }
   ETG_TRACE_USR1(("HmiStateHandler::isUserStandbyOff() HMIState UserStandbyOff %d %d", HMIState, UserStandbyOff));
   return UserStandbyOff;
}


/*****************************************************************************
* @brief   HmiStateHandler GetHmiState  - To inform the value to block any
           hardkey press in CGIAppController
*****************************************************************************/
uint8 HmiStateHandler::GetHmiState()
{
   ETG_TRACE_USR4(("HmiStateHandler::GetHmiState"));
   return HMIState;
}


}
}
