/**************************************************************************************
* @file         : <SWUpdateClient.cpp>
* @author       : <Ramesh Kesavan> <ECH> <INF4CV>
* @addtogroup   : <AppHmi_System>
* @brief        :
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "App/Core/Utils/SWUpdateUtility.h"
#include "SWUpdateClient.h"
#include "fcswupdatesrv/FcSwUpdateSrvProxy.h"
#include "App/Core/ClientHandler/SPM/SpmSrvClient.h"
#include "App/Core/AppLogic/SystemSettingListhandler/SystemSettingsListHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_SYSTEM_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_SYSTEM
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_SYSTEM_"
#define ETG_I_FILE_PREFIX                 App::Core::SWUpdateClient::
#include "trcGenProj/Header/SWUpdateClient.cpp.trc.h"
#endif

using namespace ::App::Core;
using namespace fcswupdatesrv::FcSwUpdateSrv;

namespace App {
namespace Core {

SWUpdateClient* SWUpdateClient::_swUpdateClient = NULL;

/**
* @Constructor
*/
SWUpdateClient::SWUpdateClient()
{
   ETG_TRACE_USR4(("SWUpdateClient Constructor is called"));
   _currentSystemVersion = "";
   _overallsoftwareVersion = "";
   _updateState = 0; // 0 - tenUpdState::tenUpdState__Idle
   // To create FcSwUpdate proxy
   _fcSwUpdateProxy = FcSwUpdateSrvProxy::createProxy("fcSwUpdatePort", *this);

   //Register the SWUpdateClient instance in StartupSync for
   // register/deregister call back whenever gui is started.
   if (_fcSwUpdateProxy)
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _fcSwUpdateProxy->getPortName());
      ETG_TRACE_USR4(("SWUpdateClient Constructor : registerPropertyRegistrationIF"));
   }
}


/**
* @Destructor
*/
SWUpdateClient::~SWUpdateClient()
{
   destroy();
}


void SWUpdateClient::destroy()
{
   delete _swUpdateClient;
   _swUpdateClient = NULL;
}


/**
* onAvailable - To Handle Service Availability
* 				(Here its inherited to Create FcSwUpdate service Proxy)
* @param[in] proxy
* @param[in] stateChange
* @parm[out] None
* @return void
*/
void SWUpdateClient::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
                                 const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("SWUpdateClient::onAvailable"));
   StartupSync::getInstance().onAvailable(proxy, stateChange);

   // Registering for property changes
   //registerProperties(proxy,stateChange);
}


/**
* onUnavailable - To Handle Service Unavailability
* @param[in] proxy
* @param[in] stateChange
* @parm[out] None
* @return void
*/
void SWUpdateClient::onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
                                   const asf::core::ServiceStateChange& stateChange)
{
   // Registering for property changes
   // deregisterProperties(proxy,stateChange);

   ETG_TRACE_USR4(("SWUpdateClient::onUnavailable"));
   StartupSync::getInstance().onUnavailable(proxy, stateChange);
}


/**
* registerProperties - To Register Service Properties
* @param[in] proxy
* @param[in] stateChange
* @parm[out] None
* @return void
*/
void SWUpdateClient::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
                                        const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SWUpdateClient:registerProperties is Entry "));
   if (_fcSwUpdateProxy && (_fcSwUpdateProxy == proxy))
   {
      ETG_TRACE_USR4(("SWUpdateClient:registerProperties is called for fcswupdate"));

      // re-ordered function to register first and place get request next
      _fcSwUpdateProxy->sendUpdateStateRegister(*this);
      _fcSwUpdateProxy->sendUpdateStateGet(*this);

      _fcSwUpdateProxy->sendReleaseDocRegister(*this);
      _fcSwUpdateProxy->sendReleaseDocGet(*this);

      _fcSwUpdateProxy->sendUpdateErrorsRegister(*this);
      _fcSwUpdateProxy->sendUpdateErrorsGet(*this);

      // Setting up the update type as Full update, as some default/incorrent est time was obtained if not set
      _fcSwUpdateProxy->sendSetUpdateOptionsRequest(*this, tenUpdOptions__Any);

      // Request for UpdateProgress as this will be the trigger on AutoResume of SWDL (Mw --> HMI)
      _fcSwUpdateProxy->sendUpdateProgressRegister(*this);
      _fcSwUpdateProxy->sendUpdateProgressGet(*this);
   }
   ETG_TRACE_USR4(("SWUpdateClient:registerProperties is Exit "));
}


/**
* deregisterProperties - To DeRegister Service Properties
* @param[in] proxy
* @param[in] stateChange
* @parm[out] None
* @return void
*/
void SWUpdateClient::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SWUpdateClient:deregisterProperties is called"));
   if (_fcSwUpdateProxy && (_fcSwUpdateProxy == proxy))
   {
      _fcSwUpdateProxy->sendUpdateStateDeregisterAll();
      _fcSwUpdateProxy->sendReleaseDocDeregisterAll();
      _fcSwUpdateProxy->sendUpdateErrorsDeregisterAll();
      _fcSwUpdateProxy->sendUpdateProgressDeregisterAll();
   }
}


/**
* onUpdateGetSourcesError - Response handling to get version flashed in Target.
* @param[in] proxy
* @param[in] response
* @parm[out] none
* @return void
*/
void SWUpdateClient::onUpdateStateUpdate(const ::boost::shared_ptr< FcSwUpdateSrvProxy >& /*proxy*/,
      const ::boost::shared_ptr< UpdateStateUpdate >& update)
{
   trUpdState updateStateData = update->getUpdateState();
   _currentSystemVersion = updateStateData.getMuVersion();
   _overallsoftwareVersion = updateStateData.getOverallCISSWVer();
   uint8 updateSource = updateStateData.getEnSourceType();
   _updateState = static_cast<uint8>(updateStateData.getEnState());
   ETG_TRACE_USR4(("SWUpdateClient::onUpdateStateUpdate: muVersion = %s", updateStateData.getMuVersion().c_str()));
   ETG_TRACE_USR4(("SWUpdateClient::onUpdateStateUpdate: Update State = %d", _updateState));
   ETG_TRACE_USR4(("SWUpdateClient::onUpdateStateUpdate: Update Source Type = %d", updateSource));
   ETG_TRACE_USR4(("SWUpdateClient::onUpdateStateUpdate: _currentSystemVersion = %s", _currentSystemVersion.c_str()));
   ETG_TRACE_USR4(("SWUpdateClient::onUpdateStateUpdate: overall software version = %s", _overallsoftwareVersion.c_str()));

   SWUpdate::GetInstance()->updateState(_currentSystemVersion, _updateState, updateSource);
}


/**
* getCurrentSytemVersion - Retrives Current version
* @param[in] null
* @return const std::string
*/
const std::string SWUpdateClient::getCurrentSytemVersion()
{
   return _currentSystemVersion;
}


const std::string SWUpdateClient::getOverallSoftwareVersion()
{
   return _overallsoftwareVersion;
}


/**
* onUpdateGetSourcesError - Error handling for onUpdateState.
* @param[in] proxy
* @param[in] error
* @return void
*/
void SWUpdateClient::onUpdateStateError(const ::boost::shared_ptr< FcSwUpdateSrvProxy >& /*proxy*/,
                                        const ::boost::shared_ptr< UpdateStateError >& /*error*/)
{
   ETG_TRACE_USR4(("SWUpdateClient::onUpdateStateError"));
}


/**
* onReleaseDocError - Update handling for Release Version
* 					  found on USBstick.
* @param[in] proxy
* @param[in] update
* @parm[out] none
* @return void
*/
void SWUpdateClient::onReleaseDocUpdate(const ::boost::shared_ptr< FcSwUpdateSrvProxy >& /*proxy*/,
                                        const ::boost::shared_ptr< ReleaseDocUpdate >& update)
{
   ReleaseDocInfo stReleaseDocInfo;
   trReleaseDoc releaseDocData = update->getReleaseDoc();
   stReleaseDocInfo.releaseName = releaseDocData.getName();
   stReleaseDocInfo.needFlashing = releaseDocData.getBNeedsFlashing();
   stReleaseDocInfo.estimatedTime = releaseDocData.getU32EstimatedUpdateTimeSec();
   stReleaseDocInfo.filterOption = releaseDocData.getEnUpdOptions();
   stReleaseDocInfo.UpdateDirection = releaseDocData.getEnUpdateDirection();
   stReleaseDocInfo.sourceType = releaseDocData.getEnSourceType();
   ETG_TRACE_USR4(("SWUpdateClient::onReleaseDocUpdate: ReleaseDoc = %s", stReleaseDocInfo.releaseName.c_str()));
   ETG_TRACE_USR4(("SWUpdateClient::onReleaseDocUpdate: needFlashing = %d", stReleaseDocInfo.needFlashing));
   ETG_TRACE_USR4(("SWUpdateClient::onReleaseDocUpdate: estimatedTime = %d", stReleaseDocInfo.estimatedTime));
   ETG_TRACE_USR4(("SWUpdateClient::onReleaseDocUpdate: filter Option = %d", stReleaseDocInfo.filterOption));
   ETG_TRACE_USR4(("SWUpdateClient::onReleaseDocUpdate: UpdateDirection = %d", stReleaseDocInfo.UpdateDirection));

   SWUpdate::GetInstance()->updateReleaseDoc(stReleaseDocInfo);
}


/**
* onReleaseDocError - Error handling for ReleaseDoc.
* @param[in] proxy
* @param[in] error
* @parm[out] none
* @return void
*/
void SWUpdateClient::onReleaseDocError(const ::boost::shared_ptr< FcSwUpdateSrvProxy >& /*proxy*/,
                                       const ::boost::shared_ptr< ReleaseDocError >& /*error*/)
{
   ETG_TRACE_USR4(("SWUpdateClient::onReleaseDocError"));
}


/**
* onUpdateErrorsUpdate - Update handling for error in data on stick.
* @param[in] proxy
* @param[in] update
* @parm[out] none
* @return void
*/
void SWUpdateClient::onUpdateErrorsUpdate(const ::boost::shared_ptr< FcSwUpdateSrvProxy >& /*proxy*/,
      const ::boost::shared_ptr< UpdateErrorsUpdate >& update)
{
   uint8 errorType = 0;
   uint8 errorStatus = 0;
   ETG_TRACE_USR4(("SWUpdateClient::onUpdateErrorsUpdate: ErrorValue"));
   trErrorIds updateErrorsStatus = update->getUpdateErrors();
   ::std::vector< tenErrorId > errorId = updateErrorsStatus.getErrorIds();
   ETG_TRACE_USR4(("SWUpdateClient::onUpdateErrorsUpdate Error list Size=%d", updateErrorsStatus.getErrorIds().size()));
   for (std::vector < tenErrorId>::const_iterator iter = errorId.begin(); iter != errorId.end(); ++iter)
   {
      errorType = *iter;
      ETG_TRACE_USR4(("SWUpdateClient::onUpdateErrorsUpdate: ErrorValue = %d", errorType));
   }
   switch (errorType)
   {
      case tenErrorId__SWL_ERROR_MEDIA_UNAVAILABLE:
      {
         errorStatus = error_unavailable_stick; // NO Stickk
      }
      break;
      case tenErrorId__SWL_ERROR_MEDIUM_REMOVED:
      {
         errorStatus = error_removed_stick;
      }
      break;
      case tenErrorId__SWL_ERROR_INCOMPATIBLE_USB_FORMAT:
      {
         errorStatus = error_incompatible_usb_format; //incompatiable
      }
      break;
      case tenErrorId__SWL_ERROR_IMAGE_INCOMPATIBLE_DOWNGRADE:
      {
         errorStatus = error_downgrade_stick;
      }
      break;
      case tenErrorId__SWL_ERROR_IMAGE_INCOMPATIBLE_SAME_VERSION:
      {
         errorStatus = error_samegrade_stick;
      }
      break;
      case tenErrorId__SWL_OK:
      {
         errorStatus = error_default_SWL_OK;
      }
      break;
      default:
      {
         if (errorType != tenErrorId__SWL_OK)
         {
            errorStatus = error_incompatible_stick;
         }
      }
      break;
   }

   SWUpdate::GetInstance()->errorTypeUpdate(errorStatus);
}


/**
* onUpdateErrorsError - update handling for UpdateErrors.
* @param[in] proxy
* @param[in] update
* @parm[out] none
* @return void
*/
void SWUpdateClient::onUpdateErrorsError(const ::boost::shared_ptr< FcSwUpdateSrvProxy >& /*proxy*/,
      const ::boost::shared_ptr< UpdateErrorsError >& /*error*/)
{
}


/**
* onUpdateErrorsError - update handling for SetAllowRecoveryModeErrors.
* @param[in] proxy
* @param[in] SetAllowRecoveryModeError
* @parm[out] none
* @return void
*/
void SWUpdateClient::onSetAllowRecoveryModeError(const ::boost::shared_ptr<  fcswupdatesrv::FcSwUpdateSrv::FcSwUpdateSrvProxy >& /*proxy*/,
      const ::boost::shared_ptr<  fcswupdatesrv::FcSwUpdateSrv::SetAllowRecoveryModeError >& /*error*/)
{
   ETG_TRACE_USR4(("SWUpdateClient::onSetAllowRecoveryModeError is called"));
}


/**
* onUpdateErrorsError - update handling for Allow Recovery Mode Response.
* @param[in] proxy
* @param[in] SetAllowRecoveryModeResponse
* @parm[out] none
* @return void
*/
void SWUpdateClient::onSetAllowRecoveryModeResponse(const ::boost::shared_ptr<  fcswupdatesrv::FcSwUpdateSrv::FcSwUpdateSrvProxy >& /*proxy*/,
      const ::boost::shared_ptr<  fcswupdatesrv::FcSwUpdateSrv::SetAllowRecoveryModeResponse >& /*error*/)
{
   ETG_TRACE_USR4(("SWUpdateClient::onSetAllowRecoveryModeResponse is called"));
}


/**
* onApplyDocResponse - update handling for ApplyDoc.
* @param[in] proxy
* @param[in] update
* @parm[out] none
* @return void
*/
void SWUpdateClient::onApplyDocResponse(const ::boost::shared_ptr< FcSwUpdateSrvProxy >& /*proxy*/,
                                        const ::boost::shared_ptr< ApplyDocResponse >& response)
{
   bool applyDocState = response->getBOk();
   ETG_TRACE_USR4(("SWUpdateClient::onApplyDocResponse bOK Value = %d", applyDocState));
}


/**
* onApplyDocError - Error handling for ApplyDoc.
* @param[in] proxy
* @param[in] error
* @parm[out] none
* @return void
*/
void SWUpdateClient::onApplyDocError(const ::boost::shared_ptr< FcSwUpdateSrvProxy >& /*proxy*/,
                                     const ::boost::shared_ptr< ApplyDocError >& /*error*/)
{
   ETG_TRACE_USR4(("SWUpdateClient::onApplyDocError"));
}


/**
* SetSoftwareUpdateType - Error handling for nature of update user request for
* @param[in] Update type (quick , Full)
* @parm[out] none
* @return void
*/
void SWUpdateClient::SetSoftwareUpdateType(const uint8 updateType)
{
   ETG_TRACE_USR4(("SWUpdateClient::SetSoftwareUpdateType with UpdateType = %d", updateType));

   if (_fcSwUpdateProxy)
   {
      // updateOption HMI values : Full Update - 0 (Any), Quick Update - 1 (DifferentOnly)
      if (UpdateType_Quick == updateType)
      {
         _fcSwUpdateProxy->sendSetUpdateOptionsRequest(*this, tenUpdOptions__DifferentOnly);
      }
      else
      {
         _fcSwUpdateProxy->sendSetUpdateOptionsRequest(*this, tenUpdOptions__Any);
      }

      // During MIS update, the SetAllowRecoveryMode is always set to false.
      // This is will set to true only after MIS update completion
#ifndef VARIANT_S_FTR_ENABLE_NAVIGATION
      unsigned int vehicleType = SWUpdate::GetInstance()->getKDSVehicleType();
      if (vehicleType == 2)
      {
         setAllowRecoveryModeRequest(true);
      }
      else
      {
         setAllowRecoveryModeRequest(false);
      }
#endif
      setAllowRecoveryModeRequest(false);
   }
}


/**
* SetAllowRecoveryModeRequest -  To Send the 'AllowRecoveryModeRequest', this is to restart the system in recovery mode
* @param[in] bool flag - whether to set it or not
* @parm[out] none
* @return
*/
void SWUpdateClient::setAllowRecoveryModeRequest(bool flag)
{
   ETG_TRACE_USR4(("SWUpdateClient::SetAllowRecoveryModeRequest. "));
   if (_fcSwUpdateProxy)
   {
      _fcSwUpdateProxy->sendSetAllowRecoveryModeRequest(*this, flag);
   }
}


/**
* handleStartDownloadRequest -  To handle 'ApplyDoc' property request
* @param[in] none
* @parm[out] none
* @return
*/
void SWUpdateClient::handleStartDownloadRequest()
{
   ETG_TRACE_USR4(("SWUpdateClient::handleStartDownloadRequest. "));
   if (_fcSwUpdateProxy)
   {
      ETG_TRACE_USR4(("SWUpdateClient::sendApplyDocRequest to middleware"));
      _fcSwUpdateProxy->sendApplyDocRequest(*this);
   }
}


/**
* onSetUpdateOptionsResponse - update handling for SetUpdateOptions.
* @param[in] proxy
* @param[in] response
* @parm[out] none
* @return void
*/
void SWUpdateClient::onSetUpdateOptionsResponse(const ::boost::shared_ptr< FcSwUpdateSrvProxy >& /*proxy*/,
      const ::boost::shared_ptr< SetUpdateOptionsResponse >& /*response*/)
{
   ETG_TRACE_USR4(("SWUpdateClient::onSetUpdateOptionsResponse"));
}


/**
* onSetUpdateOptionsError - Error handling for SetUpdateOptions.
* @param[in] proxy
* @param[in] error
* @parm[out] none
* @return void
*/
void SWUpdateClient::onSetUpdateOptionsError(const ::boost::shared_ptr< FcSwUpdateSrvProxy >& /*proxy*/,
      const ::boost::shared_ptr< SetUpdateOptionsError >& /*error*/)
{
   ETG_TRACE_USR4(("SWUpdateClient::onSetUpdateOptionsError"));
}


/**
* getDeviceDataReleaseDoc - To read the current value of the property 'ReleaseDoc'
* @param[in] none
* @parm[out] none
* @return void
*/
void SWUpdateClient::getDeviceDataReleaseDoc()
{
   _swUpdateDetailsVector.clear();
   if (_fcSwUpdateProxy)
   {
      if ((_fcSwUpdateProxy->hasReleaseDoc()))
      {
         trReleaseDoc releaseDocData = _fcSwUpdateProxy->getReleaseDoc();
         uint32 deviceDataListSize = releaseDocData.getDevices().size();
         ETG_TRACE_USR4(("getDeviceDataReleaseDoc : deviceDataListSize = %d", deviceDataListSize));
         std::vector < trDeviceData > deviceData = releaseDocData.getDevices();
         for (std::vector < trDeviceData >::const_iterator iter_device_data = deviceData.begin();
               iter_device_data != deviceData.end(); ++iter_device_data)
         {
            trDeviceData deviceListEntry = *iter_device_data;
            std::string deviceName = deviceListEntry.getName();
            uint32 moduleDataListSize = deviceListEntry.getModules().size();
            ETG_TRACE_USR4(("getDeviceDataReleaseDoc : moduleDataListSize = %d", moduleDataListSize));
            std::vector <trModuleData> moduleData = deviceListEntry.getModules();
            ETG_TRACE_USR4(("getDeviceDataReleaseDoc: DeviceName =%s", deviceListEntry.getName().c_str()));

            for (std::vector < trModuleData >::const_iterator iter_module_data = moduleData.begin();
                  iter_module_data != moduleData.end(); ++iter_module_data)
            {
               trModuleData moduleListEntry = *iter_module_data;
               std::string moduleName = moduleListEntry.getName();
               uint32 subModuleDataListSize = moduleListEntry.getSubModules().size();
               ETG_TRACE_USR4(("getDeviceDataReleaseDoc : subModuleDataListSize = %d", subModuleDataListSize));
               std::vector <trSubModuleData> subModuleData = moduleListEntry.getSubModules();
               ETG_TRACE_USR4(("getDeviceDataReleaseDoc: ModuleName =%s", moduleListEntry.getName().c_str()));
               ETG_TRACE_USR4(("getDeviceDataReleaseDoc: HwIndex =%s", moduleListEntry.getHwIndex().c_str()));

               for (std::vector < trSubModuleData >::const_iterator iter_submodule_data = subModuleData.begin();
                     iter_submodule_data != subModuleData.end(); ++iter_submodule_data)
               {
                  trSubModuleData subModuleListEntry = *iter_submodule_data;
                  std::string subModuleName = subModuleListEntry.getName();
                  trSubModuleData submodule = subModuleListEntry.getDefaultInstance();
                  std::string fromVersion = subModuleListEntry.getFromVersion();
                  std::string toVersion = subModuleListEntry.getToVersion();
                  _stswDetailsInfo.subModuleName  = subModuleName;
                  _stswDetailsInfo.currentVersion = fromVersion;
                  _stswDetailsInfo.mediaVersion   = toVersion;
                  _swUpdateDetailsVector.push_back(_stswDetailsInfo);
                  ETG_TRACE_USR4(("getDeviceDataReleaseDoc: SubModuleName =%s", subModuleListEntry.getName().c_str()));
                  ETG_TRACE_USR4(("SubModule From Version=%s", subModuleListEntry.getFromVersion().c_str()));
                  ETG_TRACE_USR4(("SubModule To Version =%s", subModuleListEntry.getToVersion().c_str()));
               }
            }
         }
         SWUpdate::GetInstance()->readSWUpdateDetails();
      }
   }
}


/**
* getSWDetailsVector - To Handle vector data.
* @param[in] none
* @parm[out] none
* @return std::vector <std::string>
*/
std::vector <swDetailsInfo> SWUpdateClient::getSWDetailsVector()
{
   return _swUpdateDetailsVector;
}


/**
* onUpdateSetReleaseFilterResponse - Context Switch to System To handle Software Update.
* @param[in] proxy
* @param[in] response
* @parm[out] none
* @return void
*/
void SWUpdateClient::onUpdateSetReleaseFilterResponse(const ::boost::shared_ptr<FcSwUpdateSrvProxy >& /*proxy*/,
      const ::boost::shared_ptr< UpdateSetReleaseFilterResponse >& response)
{
   ETG_TRACE_USR4(("SWUpdateClient::onUpdateSetReleaseFilterResponse"));
   uint8 releaseFilter = static_cast<uint8>(response->getResFilter().getEnUpgradeDirectionFilter());
   ETG_TRACE_USR4(("SWUpdateClient::onUpdateSetReleaseFilterResponse = %d", releaseFilter));
   SWUpdate::GetInstance()->updateReleaseFilter(releaseFilter);
}


/**
* onUpdateSetReleaseFilterError - Error handling for UpdateSetReleaseFilter.
* @param[in] proxy
* @param[in] error
* @parm[out] none
* @return void
*/
void SWUpdateClient::onUpdateSetReleaseFilterError(const ::boost::shared_ptr<FcSwUpdateSrvProxy >& /*proxy*/,
      const ::boost::shared_ptr<UpdateSetReleaseFilterError >& /*error*/)
{
   ETG_TRACE_USR4(("SWUpdateClient::onUpdateSetReleaseFilterError"));
}


/**
* SetReleaseFilter -  To handle 'ReleaseFilter'  request
* @param[in] ReleaseFilterInfo& stReleaseFilterInfo
* @parm[out] none
* @return
*/
void SWUpdateClient::SetReleaseFilter(const uint8 downloadMode)
{
   ETG_TRACE_USR4(("SWUpdateClient::SetReleaseFilter received is %d", downloadMode));
   if (_fcSwUpdateProxy)
   {
      _fcSwUpdateProxy->sendUpdateSetReleaseFilterRequest(*this, static_cast<tenUpdOptions>(downloadMode));
   }
}


void SWUpdateClient::getUpdateHistory()
{
   ETG_TRACE_USR4(("SWUpdateClient::getUpdateHistory"));
   if (_fcSwUpdateProxy)
   {
      _fcSwUpdateProxy->sendGetUpdateHistoryRequest(*this);
   }
}


/**
* getSWUpdateHistoryVector - To Handle vector data.
* @param[in] none
* @parm[out] none
* @return std::vector <std::string>
*/
std::vector < UpdateHistoryInfo > SWUpdateClient::getSWUpdateHistoryVector()
{
   return _swUpdateHistoryVector;
}


/**
* getSWUpdateHistoryDetailsVector - To Handle vector data.
* @param[in] none
* @parm[out] none
* @return std::vector <std::string>
*/
std::vector <std::string> SWUpdateClient::getSWUpdateHistoryDetailsVector()
{
   return _swUpdateHistoryDetailsVector;
}


/**
* onGetUpdateHistoryResponse - Response handling for UpdateHistory.
* @param[in] proxy
* @param[in] error
* @parm[out] none
* @return void
*/
void SWUpdateClient::onGetUpdateHistoryResponse(const ::boost::shared_ptr<FcSwUpdateSrvProxy >& /*proxy*/,
      const ::boost::shared_ptr<GetUpdateHistoryResponse >& response)
{
   _swUpdateHistoryVector.clear();
   _swUpdateHistoryDetailsVector.clear();
   ETG_TRACE_USR4(("SWUpdateClient::onGetUpdateHistoryResponse is received"));
   trUpdLogs updLogs = response->getLogs();
   std::vector< trUpdLog > updLog = updLogs.getUpdateLogs();
   uint32 updateLogSize = updLogs.getUpdateLogs().size();
   ETG_TRACE_USR4(("onGetUpdateHistoryResponse : UpdateLog size is = %d", updateLogSize));
   bool Variant_Type = SystemSettingsListHandler::poGetInstance()->VariantType();
   ETG_TRACE_USR4(("onGetUpdateHistoryResponse : Variant_Type = %d", Variant_Type));
   for (std::vector < trUpdLog >::const_iterator iter_upLog_data = updLog.begin();
         iter_upLog_data != updLog.end(); ++iter_upLog_data)
   {
      SWUpdateUtility swUpdateUtility;
      trUpdLog upLogEntry = *iter_upLog_data;

      UpdateHistoryInfo stUpdateHistoryInfo;
      stUpdateHistoryInfo.versionName = upLogEntry.getName();

      stUpdateHistoryInfo.status = (upLogEntry.getSuccess());
      trTimeDate timeDate = upLogEntry.getTimeDate();
      std::string day = swUpdateUtility.itoa(timeDate.getDay());
      if (timeDate.getDay() < 10)
      {
         day = "0" + day;
      }
      std::string month = swUpdateUtility.itoa(timeDate.getMonth());
      if (timeDate.getMonth() < 10)
      {
         month = "0" + month;
      }
      std::string year = swUpdateUtility.itoa(timeDate.getYear());
      stUpdateHistoryInfo.date = day + "." + month + "." + year;
      //std::string historyLog = "     " + versionNameWithSpaces + status.GetCString() + "              " + day + "." + month + "." + year;
      _swUpdateHistoryVector.push_back(stUpdateHistoryInfo);

      uint32 deviceSize = upLogEntry.getDevices().size();
      std::vector< trDeviceData > deviceData = upLogEntry.getDevices();
      ETG_TRACE_USR4(("onGetUpdateHistoryResponse : Device size is = %d", deviceSize));
      std::string historyDetails = "";
      for (std::vector < trDeviceData >::const_iterator iter_device_data = deviceData.begin();
            iter_device_data != deviceData.end(); ++iter_device_data)
      {
         trDeviceData DeviceData = *iter_device_data;
         std::string deviceName = "Device Name: " + DeviceData.getName();
         historyDetails += deviceName + "\n";
         ETG_TRACE_USR4(("onGetUpdateHistoryResponse: DeviceName =%s", DeviceData.getName().c_str()));
         uint32 moduleDataSize = DeviceData.getModules().size();
         ETG_TRACE_USR4(("onGetUpdateHistoryResponse : Module Data size is = %d", moduleDataSize));
         std::vector< trModuleData > moduleData = DeviceData.getModules();
         for (std::vector < trModuleData >::const_iterator iter_module_data = moduleData.begin();
               iter_module_data != moduleData.end(); ++iter_module_data)
         {
            trModuleData moduleListEntry = *iter_module_data;
            std::string moduleName = "Module Name:  " + moduleListEntry.getName();
            historyDetails +=  moduleName + "\n";
            uint32 subModuleDataListSize = moduleListEntry.getSubModules().size();
            ETG_TRACE_USR4(("onGetUpdateHistoryResponse : subModuleDataListSize = %d", subModuleDataListSize));
            std::vector <trSubModuleData> subModuleData = moduleListEntry.getSubModules();
            ETG_TRACE_USR4(("onGetUpdateHistoryResponse: ModuleName =%s", moduleListEntry.getName().c_str()));
            for (std::vector < trSubModuleData >::const_iterator iter_submodule_data = subModuleData.begin();
                  iter_submodule_data != subModuleData.end(); ++iter_submodule_data)
            {
               trSubModuleData subModuleListEntry = *iter_submodule_data;
               ETG_TRACE_USR4(("onGetUpdateHistoryResponse: SubModuleName =%s", subModuleListEntry.getName().c_str()));
               if ((Variant_Type == NON_NAVI) && (subModuleListEntry.getName() == "NAVAPP"))
               {
                  ETG_TRACE_USR4(("SWUpdateClient::onGetUpdateHistoryResponse Submodule name is NAVAPP so ignoring the data for NON_NAVI variant"));
               }
               else
               {
                  ETG_TRACE_USR4(("SWUpdateClient::onGetUpdateHistoryResponse Submodule name "));
                  std::string subModuleName = "SubModule Name:  " + subModuleListEntry.getName();
                  historyDetails += subModuleName + "\n";
                  std::string Info = "Version:  " + subModuleListEntry.getInfo();
                  historyDetails +=  Info + "\n";
                  ETG_TRACE_USR4(("onGetUpdateHistoryResponse: SubModuleName =%s", subModuleListEntry.getName().c_str()));
                  ETG_TRACE_USR4(("SubModuleVersion =%s", subModuleListEntry.getInfo().c_str()));
               }
            }
         }
      }
      historyDetails += "\n ";
      _swUpdateHistoryDetailsVector.push_back(historyDetails);
   }
   SWUpdate::GetInstance()->readSWUpdateHistory();
}


/**
* onGetUpdateHistoryError - Error handling for UpdateHistory.
* @param[in] proxy
* @param[in] error
* @parm[out] none
* @return void
*/
void SWUpdateClient::onGetUpdateHistoryError(const ::boost::shared_ptr<FcSwUpdateSrvProxy >& /*proxy*/,
      const ::boost::shared_ptr< GetUpdateHistoryError >& /*error*/)
{
   ETG_TRACE_USR4(("SWUpdateClient::onGetUpdateHistoryError is received"));
}


void SWUpdateClient::onUpdateProgressError(const ::boost::shared_ptr< fcswupdatesrv::FcSwUpdateSrv::FcSwUpdateSrvProxy >& /*proxy*/, const ::boost::shared_ptr< fcswupdatesrv::FcSwUpdateSrv::UpdateProgressError >& /*error*/)
{
   ETG_TRACE_USR4(("SWUpdateClient::onUpdateProgressError is received"));
}


void SWUpdateClient::onUpdateProgressUpdate(const ::boost::shared_ptr< fcswupdatesrv::FcSwUpdateSrv::FcSwUpdateSrvProxy >& /*proxy*/, const ::boost::shared_ptr< fcswupdatesrv::FcSwUpdateSrv::UpdateProgressUpdate >& update)
{
   ETG_TRACE_USR4(("SWUpdateClient::onUpdateProgressUpdate is received"));
   if (update->hasUpdateProgress())
   {
      // if we receive, this property in the boot-up. (Headunit restarted during cis update,with usb plugged in)
      // its evident that USBstick with valid software is pulgged in. we will not receive ReleaseDoc or UpdateErrorState here

      const trUpdProgress currUpdProgress = update->getUpdateProgress();
      std::string subModuleName = currUpdProgress.getSubModuleName().c_str();
      ETG_TRACE_USR4(("SWUpdateClient::onUpdateProgressUpdate subModuleName= %s", subModuleName.c_str()));
      ETG_TRACE_USR4(("SWUpdateClient::onUpdateProgressUpdate Update State = %d", _updateState));

      // progress.SubModuleName = MIS as in <-- SWUPDATE_UTIL_US1: "subModuleName": "MIS",-->::FCSWUPDATE_MAIN
      if ((strncmp(subModuleName.c_str(), "MIS", 4) == 0) && (_updateState == UpdState_Running))
      {
         ETG_TRACE_USR4(("SWUpdateClient::onUpdateProgressUpdate - MIS update is running in Background. Trigger HMI Transition if needed"));
         SWUpdate::GetInstance()->ValdiateSWUpdateHMIStatus();
      }
   }
   else
   {
      ETG_TRACE_USR4(("SWUpdateClient::onUpdateProgressUpdate is received - No Updates"));
   }
}


void SWUpdateClient::sendAbortRequest()
{
   ETG_TRACE_USR4(("SWUpdateClient::sendAbortRequest"));
   if (_fcSwUpdateProxy)
   {
      ETG_TRACE_USR4(("SWUpdateClient::sendAbortRequest : Sending Abort request to SW Update Component"));
      _fcSwUpdateProxy->sendUpdateAbortRequest(*this, true);
   }
}


void SWUpdateClient::onUpdateAbortError(const ::boost::shared_ptr< fcswupdatesrv::FcSwUpdateSrv::FcSwUpdateSrvProxy >& proxy, const ::boost::shared_ptr< fcswupdatesrv::FcSwUpdateSrv::UpdateAbortError >& error)
{
   ETG_TRACE_USR4(("SWUpdateClient::onUpdateAbortError is received"));
}


void SWUpdateClient::onUpdateAbortResponse(const ::boost::shared_ptr< fcswupdatesrv::FcSwUpdateSrv::FcSwUpdateSrvProxy >& proxy, const ::boost::shared_ptr< fcswupdatesrv::FcSwUpdateSrv::UpdateAbortResponse >& response)
{
   ETG_TRACE_USR4(("SWUpdateClient::onUpdateAbortResponse is received"));
   if (_fcSwUpdateProxy)
   {
      SpmSrvClient::GetInstance()->TriggerCMCRestartdownloadaborted();
   }
}


}
} // Core
