/**************************************************************************************
* @file         : <ClockSrvClient.cpp>
* @author       : <INF4CV>
* @addtogroup   : <AppHmi_System>
* @brief        :
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "ClockSrvClient.h"
#include <sstream> // for string streams 
#include <string> // for string 

#include "App/Core/AppLogic/ClockSettings/ClockSettings.h"
#include "App/Core/AppLogic/ClockSettings/GpsClockListHandler.h"
#include "App/Core/AppLogic/ClockSettings/ClockDateFormat/ClockDateFormatRbtn.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_SYSTEM_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_SYSTEM
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_SYSTEM_"
#define ETG_I_FILE_PREFIX                 App::Core::ClockSrvClient::
#include "trcGenProj/Header/ClockSrvClient.cpp.trc.h"
#endif

using namespace clock_main_fi;
using namespace clock_main_fi_types;

static const int TIME_STR_MAX_LENGTH = 12;   // For example, 12:00pm
static const int TIME_STR_MAX_LENGTH_AM_PM = 3;

namespace App {
namespace Core {
/**
 * @Constructor
 */
ClockSrvClient* ClockSrvClient::_ClockSrvClient = NULL;
ClockSrvClient::ClockSrvClient(): _clockproxy(Clock_main_fiProxy::createProxy("clockFiPort", *this))
{
   ETG_TRACE_USR4(("ClockSrvClient Constructor is called"));
   _timeFormat = 1;
   _dateFormat = 1;
   _day = 0;
   _month = 0;
   _Year = 0;
   mymap.clear();

   //DayLightSavingTimeStatus = 0; // Removed the DaylightSavingTime based on Bug 963829

   if (_clockproxy)
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _clockproxy->getPortName());
      ETG_TRACE_USR4(("ClockSrvClient Constructor after creating proxy"));
   }
}


/**
* @Destructor
*/

ClockSrvClient::~ClockSrvClient()
{
   ETG_TRACE_USR4(("ClockSrvClient destructor is called"));
   destroy();
}


void ClockSrvClient::destroy()
{
   delete _ClockSrvClient;
   _ClockSrvClient = NULL;
}


/**
 * @onAvailable
 */
void ClockSrvClient::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   StartupSync::getInstance().onAvailable(proxy, stateChange);
   ETG_TRACE_USR4(("ClockSrvClient onAvailable is called"));
}


/**
 * @onUnavailable
 */
void ClockSrvClient::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   StartupSync::getInstance().onUnavailable(proxy, stateChange);
   ETG_TRACE_USR4(("ClockSrvClient onAvailable is called"));
}


void ClockSrvClient::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /* stateChange */)
{
   ETG_TRACE_USR4(("ClockSrvClient register property  is called"));

   if (_clockproxy && (_clockproxy == proxy))
   {
      ETG_TRACE_USR4(("Clock_fi_proxy is  received register Properties........"));

      getAvailableTimzoneList();
      _clockproxy->sendTimeFormatUpReg(*this);
      _clockproxy->sendLocalTimeDateUpReg(*this, 2);
      _clockproxy->sendNewDateFormatUpReg(*this);
      _clockproxy->sendGPS_AutoSyncUpReg(*this);

      //Removed the DaylightSavingTime based on Bug 963829
      //_clockproxy->sendDaylightSavingTimeUpReg(*this);

      _clockproxy->sendRegion_TimeZoneUpReg(*this);
      _clockproxy->sendAuto_TimeZoneOffsetUpReg(*this);
      //Removed the DaylightSavingTime based on Bug 963829
      //_clockproxy->sendTimeZone_DSTAvailabilityUpReg(*this);
   }
}


void ClockSrvClient::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /* stateChange */)
{
   ETG_TRACE_USR4(("ClockSrvClient de register property  is called"));

   if (_clockproxy && (_clockproxy == proxy))
   {
      ETG_TRACE_USR4(("Clock_fi_proxy is  received deregister Properties........"));
      ETG_TRACE_USR4(("Clock_fi_proxy is  released........"));
      _clockproxy->sendTimeFormatRelUpRegAll();
      _clockproxy->sendLocalTimeDateRelUpRegAll();
      _clockproxy->sendNewDateFormatRelUpRegAll();
      _clockproxy->sendGPS_AutoSyncRelUpRegAll();
      //Removed the DaylightSavingTime based on Bug 963829
      //_clockproxy->sendDaylightSavingTimeRelUpRegAll();

      _clockproxy->sendRegion_TimeZoneRelUpRegAll();
      _clockproxy->sendAuto_TimeZoneOffsetRelUpRegAll();
      //Removed the DaylightSavingTime based on Bug 963829
      // _clockproxy->sendTimeZone_DSTAvailabilityRelUpRegAll();
   }
}


void ClockSrvClient::onNewDateFormatError(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< clock_main_fi::NewDateFormatError >& /*error*/)
{
   ETG_TRACE_USR4(("ClockSrvClient::onNewDateFormatError"));
}


void ClockSrvClient::onNewDateFormatStatus(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& /* proxy */,
      const ::boost::shared_ptr< clock_main_fi::NewDateFormatStatus >& status)
{
   ETG_TRACE_USR4(("void ClockSrvClient::onNewDateFormatStatus"));
   _dateFormat = static_cast<uint8>(status->getEnDateFormat());

   ETG_TRACE_USR4(("ClockSrvClient:onDateFormatStatus is called %d", _dateFormat));

   ClockDateFormatRadioBtn::poGetInstance()->UpdateClkDateFmtTglStatus(_dateFormat);
   GpsClockListHandler::poGetInstance()->GpsDateUpdateStatus(_dateFormat);
   ClockSettings::poGetInstance()->ManualDateUpdateStatus(_dateFormat);
}


//Removed the DaylightSavingTime based on Bug 963829
/**
 * onDaylightSavingTimeError - Error handling for DSTProperty
 * @param[in] proxy
 * @param[in] error
 * @param[out] none
 * @return void
 */
//void ClockSrvClient::onDaylightSavingTimeError(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& /*proxy*/,
//    const ::boost::shared_ptr< clock_main_fi::DaylightSavingTimeError >& /*error*/)
//{
//  ETG_TRACE_ERR(("ClockSrvClient:onDaylightSavingTimeError is called"));
//}


//Removed the DaylightSavingTime based on Bug 963829
/**
 * onDaylightSavingTimeStatus - Status handling for DSTProperty
 * @param[in] proxy
 * @param[in] Status
 * @param[out] none
 * @return void
 */
/*void ClockSrvClient::onDaylightSavingTimeStatus(const ::boost::shared_ptr< Clock_main_fiProxy >& proxy,
      const ::boost::shared_ptr< DaylightSavingTimeStatus >& status)
{
   if (proxy && (proxy == _clockproxy))
   {
      DayLightSavingTimeStatus = static_cast<uint8>(status->getEnDST_Status());
      ETG_TRACE_USR4(("ClockSrvClient::onDaylightSavingTimeStatus-->DayLightSavingTimeStatus = %d", DayLightSavingTimeStatus));

      GpsClockListHandler::poGetInstance()->DisplayDayLightSavingTimeStatus(static_cast<uint8>(status->getEnDST_Status()));
   }
}*/

// Bug 963829 - Removing the DaylightSavingTime
/*int ClockSrvClient::getDayLightSavingTimeStatus()
{
   ETG_TRACE_USR4(("ClockSrvClient::getDayLightSavingTimeStatus = %d", DayLightSavingTimeStatus));
   return DayLightSavingTimeStatus;
} */


// Removed the DaylightSavingTime based on Bug 963829
/**
 * setDSTPropertyToService - Handle to send set DSTProperty Request to Clock Service
 * @param[in] dstMode
 * @param[out] none
 * @return
 */
/*void ClockSrvClient::setDSTPropertyToService(uint8 dstMode)
{
   ETG_TRACE_USR4(("ClockSrvClient:setDSTPropertyToService is called"));

   if (_clockproxy)
   {
      _clockproxy->sendDaylightSavingTimeSet(*this, static_cast<VDCLK_TEN_DST_Setting>(dstMode));
      ETG_TRACE_ERR(("ClockSrvClient:setDSTPropertyToService:%d", dstMode));
   }
   else
   {
      ETG_TRACE_ERR(("ClockSrvClient:ERROR:Invalid DST Mode"));
   }
}*/


/**
 * onTimeFormatError - Error handling for TimeFormatProperty
 * @param[in] proxy
 * @param[in] error
 * @param[out] none
 * @return void
 */
void ClockSrvClient::onTimeFormatError(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& /* proxy */,
                                       const ::boost::shared_ptr< clock_main_fi::TimeFormatError >& /* error */)
{
   ETG_TRACE_ERR(("ClockSrvClient:onTimeFormatError is called"));
}


/**
 * onTimeFormatStatus - Status handling for TimeFormatProperty
 * @param[in] proxy
 * @param[in] Status
 * @param[out] none
 * @return void
 */
void ClockSrvClient::onTimeFormatStatus(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
                                        const ::boost::shared_ptr< clock_main_fi::TimeFormatStatus >& status)
{
   ETG_TRACE_USR4(("ClockSrvClient:onTimeFormatStatus is called"));
   if (proxy && (proxy == _clockproxy))
   {
      _timeFormat = static_cast<uint8>(status->getEnTimeFormat());
      ETG_TRACE_USR4(("ClockSrvClient:onTimeFormatStatus is called _timeFormat =%d", _timeFormat));

      ClockSettings::poGetInstance()->bonsetformatpropertyUpdateHMI(_timeFormat);
      // check time format and return correct value
      GpsClockListHandler::poGetInstance()->SetGpsClkformatpropertyUpdateHMI(_timeFormat);
   }
}


void ClockSrvClient::covsetTimeFormatProperty(const int& timeFormat)
{
   ETG_TRACE_USR2(("setTimeFormatProperty: TimeFormat:=%d ", timeFormat));
   if (timeFormat != INVALID_FORMAT)
   {
      if (_clockproxy)
      {
         //Set the Value to Clock service
         ETG_TRACE_USR2(("covSetTimeFormatPropertyToService: TimeFormattoService: = %d ", timeFormat));
         _clockproxy->sendTimeFormatSet(*this, static_cast<VDCLK_TEN_TimeFormat>(timeFormat));
      }
   }
   else
   {
      ETG_TRACE_ERR(("_clockproxy:ERROR:Invalid Time Format"));
   }
}


void ClockSrvClient::covLocalTimesetMode(const uint8& mhour, const uint8& mminute)
{
   ETG_TRACE_USR2(("covLocalDateTimesetMode: hour: %d minute: %d", mhour, mminute));

   if (_clockproxy)
   {
      //Send the Value to Clock service for set
      ETG_TRACE_USR2(("ClockSrvClient covLocalTimesetMode: _day, _month,_Year mhour, mminute %d %d %d %d %d ", _day, _month, _Year, mhour, mminute));

      _clockproxy->sendLocalTimeDateSet(*this, _Year, _month, _day, mhour, mminute, INVALID_FORMAT, INVALID_FORMAT, MASK);
   }

   else
   {
      ETG_TRACE_USR2(("_clockproxy:ERROR:Invalid LocalDateTimesetMode"));
   }
}


void ClockSrvClient::onLocalTimeDateStatus(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy, const ::boost::shared_ptr< clock_main_fi::LocalTimeDateStatus >& status)
{
   ETG_TRACE_ERR(("ClockSrvClient:onLocalTimeDateStatus is called"));

   tU8  u8Hours = 0;
   tU8  u8Minutes = 0;
   char buffer[TIME_STR_MAX_LENGTH] = {'\0'};
   char buffer1[TIME_STR_MAX_LENGTH_AM_PM] = {'\0'};
   //ostringstream strhrs;
   // ostringstream strmin;

   if (proxy && (proxy == _clockproxy))
   {
      ETG_TRACE_USR2(("ClockSrvClient::onLocalTimeDateStatus"));
      if (status != NULL)
      {
         u8Hours = status->getU8Hours();
         u8Minutes = status->getU8Minutes();
         _day = status->getU8Day();
         _month = status->getU8Month();
         _Year = status->getS16Year();
      }

      ETG_TRACE_USR4(("ClockSrvClient::onLocalTimeDate_MinuteUpdateStatus(%d)", status->getU8Hours()));
      //ETG_TRACE_USR4(("ClockSrvClient::_isTwelveHrFormat value(%d)", _isTwelveHrFormat));

      uint8 hrs = status->getU8Hours();
      if (_timeFormat == 1)
      {
         if (hrs == 0)
         {
            hrs = 12;
            snprintf(buffer, TIME_STR_MAX_LENGTH, " %02d:%02d", hrs, status->getU8Minutes());
            snprintf(buffer1, TIME_STR_MAX_LENGTH_AM_PM, "%2s", "AM");
         }
         else if (hrs == 12)
         {
            snprintf(buffer, TIME_STR_MAX_LENGTH, " %02d:%02d", hrs, status->getU8Minutes());
            snprintf(buffer1, TIME_STR_MAX_LENGTH_AM_PM, "%2s", "PM");
         }
         else if (hrs > 12)
         {
            hrs = static_cast<uint8>(hrs - 12);
            snprintf(buffer, TIME_STR_MAX_LENGTH, " %02d:%02d", hrs, status->getU8Minutes());
            snprintf(buffer1, TIME_STR_MAX_LENGTH_AM_PM, "%2s", "PM");
         }
         else
         {
            snprintf(buffer, TIME_STR_MAX_LENGTH, " %02d:%02d", hrs, status->getU8Minutes());
            snprintf(buffer1, TIME_STR_MAX_LENGTH_AM_PM, "%2s", "AM");
         }
      }
      else if (_timeFormat == 2)
      {
         snprintf(buffer, TIME_STR_MAX_LENGTH, "    %02d:%02d", hrs, status->getU8Minutes());
      }
      else
      {
         //Nothing
      }

      (*oMasterGadgetSystem).mTimevalue = buffer; //mbrightstringval.c_str();
      (*oMasterGadgetSystem).mTimeAmPm = buffer1;

      oMasterGadgetSystem.MarkAllItemsModified();
      oMasterGadgetSystem.SendUpdate(true);

      // strhrs << u8Hours;
      // strmin << u8Minutes;
      //string strmerghrs = strhrs.str();
      //string strmergmin = strmin.str();

      //strmerghrs.append(":");
      //strmerghrs.append(strmergmin);

      // (*oMasterGadgetSystem).mTimevalue = strmerghrs.c_str();
      //(*oMasterGadgetSystem).mTimeAmPm = "AM";

      ETG_TRACE_USR2(("ClockSrvClient u8Hours: %d u8Minutes: %d  _day: %d, _month: %d,_Year: %d", u8Hours, u8Minutes, _day, _month, _Year)); //_day, _month, _Year));

      ClockSettings::poGetInstance()->vUpdateMin_Hr_Value(u8Hours, u8Minutes);
   }
}


void ClockSrvClient::onLocalTimeDateError(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& /* proxy */, const ::boost::shared_ptr< clock_main_fi::LocalTimeDateError >& /* error */)
{
   ETG_TRACE_USR2(("ClockSrvClient:onLocalTimeDateError is called"));
}


void ClockSrvClient::covSetAutoSyncPropertyToService(const int& gpsEnableStatus, const int& positionstatus)
{
   ETG_TRACE_USR2(("covSetAutoSyncPropertyToService: gpsEnableStatus:=%d,positionstatus =%d", gpsEnableStatus, positionstatus));

   if (_clockproxy)
   {
      _clockproxy->sendGPS_AutoSyncSet(*this, gpsEnableStatus, positionstatus);
   }
   else
   {
      ETG_TRACE_ERR(("_clocksetproxy:ERROR:Invalid Time Format"));
   }
}


/**
 * onGPS_AutoSyncError - Error handling for GPS_AutoSyncProperty
 * @param[in] proxy
 * @param[in] error
 * @param[out] none
 * @return void
 */
void ClockSrvClient::onGPS_AutoSyncError(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< clock_main_fi::GPS_AutoSyncError >& /*error*/)
{
   ETG_TRACE_ERR(("ClockSrvClient:onGPS_AutoSyncError is called"));
}


/* void ClockSrvClient::onLocalTimeDate_MinuteUpdateStatus(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
                                                            const ::boost::shared_ptr< clock_main_fi::LocalTimeDate_MinuteUpdateStatus >& status)
{
   ETG_TRACE_USR4(("ClockSrvClient::onLocalTimeDate_MinuteUpdateStatus(%d)", status->getU8Hours()));
   ETG_TRACE_USR4(("ClockSrvClient::_isTwelveHrFormat value(%d)", _isTwelveHrFormat));
   char buffer[TIME_STR_MAX_LENGTH] = {'\0'};
   uint8 hrs = status->getU8Hours();
   if (_timeFormat==1)
   {
      if (hrs > 12)
      {
         hrs = hrs - 12;
         snprintf(buffer, TIME_STR_MAX_LENGTH, "%02d:%02d%02s", hrs, status->getU8Minutes(), "pm");
      }
      else
      {
         snprintf(buffer, TIME_STR_MAX_LENGTH, "%02d:%02d%02s", hrs, status->getU8Minutes(), "am");
      }
   }
   else if (_timeFormat==2)
   {
      snprintf(buffer, TIME_STR_MAX_LENGTH, "%02d:%02d", hrs, status->getU8Minutes());
   }
   else
   {
	   //Nothing
   }


		(*oMasterGadgetSystem).mTimevalue = buffer; //mbrightstringval.c_str();

		oMasterGadgetSystem.MarkAllItemsModified();
		oMasterGadgetSystem.SendUpdate(true);
} */


/**
 * onGPS_AutoSyncStatus - Status handling for GPS_AutoSyncProperty
 * @param[in] proxy
 * @param[in] Status
 * @param[out]
 * @return void
 */
void ClockSrvClient::onGPS_AutoSyncStatus(const ::boost::shared_ptr< Clock_main_fiProxy >& proxy,
      const ::boost::shared_ptr< GPS_AutoSyncStatus >& status)
{
   ETG_TRACE_USR4(("ClockSrvClient:onGPS_AutoSyncStatus is called"));
   if (proxy && (proxy == _clockproxy))
   {
      bool gpsenablestatus = status->getBEnabled();
      bool positionstatus =  status->getBPositionBased();
      ETG_TRACE_USR4(("ClockSrvClient:onGPS_AutoSyncStatus is called, gpsenablestatus=%d,positionstatus=%d", gpsenablestatus, positionstatus));
      if (gpsenablestatus == 1)
      {
         ClockSettings::poGetInstance()->callGps();
      }
      else
      {
         ClockSettings::poGetInstance()->callManual();
      }
      ClockSettings::poGetInstance()->bonsetsynmodetoPL(gpsenablestatus, positionstatus);
      GpsClockListHandler::poGetInstance()->bonsetsynmodetoPL(gpsenablestatus);
   }
}


/**
 * setDateFormattPropertyToService - Handle to send set DateformatFormatProperty Request to Clock Service
 * @param[in] DateFormat
 * @param[out] none
 * @return voi
 **/
bool ClockSrvClient::setDateFormatPropertyToService(uint8 dateFormat)
{
   if (_clockproxy)
   {
      ETG_TRACE_USR4(("ClockSrvClient::setDateFormatPropertyToService =%d", dateFormat));
      _clockproxy->sendNewDateFormatSet(*this, static_cast<VDCLK_TEN_NewDateFormat>(dateFormat));
      return true;
   }
   else
   {
      return false;
   }
}


/**
 * setRegion_TimeZoneToService -  Handler for Setting the timezoneIndex Based on User Input
 * @param[in] index
 * @param[out] none
 * @return
 */
void ClockSrvClient::setRegion_TimeZoneToService(uint16 timeZoneIndex)
{
   timeZoneIndex = static_cast<uint16>(timeZoneIndex + 1); //to align with MIDW index
   ETG_TRACE_USR4(("ClockSrvClient:setTimezoneToService is called index = %d", timeZoneIndex));
   if (_clockproxy)
   {
      _clockproxy->sendRegion_TimeZoneSet(*this, static_cast<VDCLK_TEN_TIMEZONE_Setting>(timeZoneIndex));
   }
}


/**
 * onRegion_TimeZoneError -
 * @param[in] proxy
 * @param[in] error
 * @param[out] none
 * @return void
 */
void ClockSrvClient::onRegion_TimeZoneError(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< clock_main_fi::Region_TimeZoneError >& /*error*/)
{
   ETG_TRACE_ERR(("ClockSrvClient:onTimeZoneError is called"));
}


/**
 * onRegion_TimeZoneStatus -
 * @param[in] proxy
 * @param[in] Status
 * @param[out] none
 * @return void
 */
void ClockSrvClient::onRegion_TimeZoneStatus(const ::boost::shared_ptr< Clock_main_fiProxy >& proxy,
      const ::boost::shared_ptr< Region_TimeZoneStatus >& status)
{
   if (proxy && (proxy == _clockproxy))
   {
      ETG_TRACE_USR4(("ClockSrvClient:onRegion_TimeZoneStatus is called %d", status->getEnTIMEZONE_Status()));
      ClockSettings::poGetInstance()->updateRegionTimeZoneResponse(static_cast<uint8>(status->getEnTIMEZONE_Status() - 1));
   }
}


/**
 * getAvailableTimzoneList - send request to Get Available TimeZoneProperty
 * @param[in] none
 * @param[out] none
 * @return void
 */
void ClockSrvClient::getAvailableTimzoneList() //TODO: call from Applogic is required
{
   ETG_TRACE_USR4(("ClockSrvClient:getAvailableTimzoneList is called"));
   if (_clockproxy)
   {
      _clockproxy->sendGetAvailableTimeZonesStart(*this);
   }
   else
   {
      ETG_TRACE_ERR(("ClockSrvClient:getAvailableTimzoneList cannot"));
   }
}


/**
 * onGetAvailableTimeZonesError - Error handling for Get Available TimeZoneProperty
 * @param[in] proxy
 * @param[in] error
 * @param[out] none
 * @return void
 */
void ClockSrvClient::onGetAvailableTimeZonesError(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< clock_main_fi::GetAvailableTimeZonesError >& /*error*/)
{
   ETG_TRACE_ERR(("ClockSrvClient:onGetAvailableTimeZonesError is called"));
}


/**
 * onGetAvailableTimeZonesResult - Status handling for TimeZoneStatus
 * @param[in] proxy
 * @param[in] result
 * @param[out] none
 * @return void
 */
void ClockSrvClient::onGetAvailableTimeZonesResult(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
      const ::boost::shared_ptr< clock_main_fi::GetAvailableTimeZonesResult >& result)
{
   ETG_TRACE_USR4(("ClockSrvClient:onGetAvailableTimeZonesResult is called"));
   if (proxy && (proxy == _clockproxy))
   {
      TimeZoneList stTimeZoneList;
      std::vector< TimeZoneList > availableTimeZoneList;
      availableTimeZoneList.clear();
      stTimeZoneList.Index            = result->getU16Indexes();
      stTimeZoneList.TimeZoneTextList = result->getAszDescriptions();

      availableTimeZoneList.push_back(stTimeZoneList);

      if (!availableTimeZoneList.empty())
      {
         ClockSettings::poGetInstance()->handleAvailableTimezonePropertyUpdate(availableTimeZoneList); //TODO: Impl in Applogic is required
      }
      else
      {
         ETG_TRACE_ERR(("ClockSrvClient:onGetAvailableTimeZonesResult TZ list is NULL. Cannot return to HMI"));
      }
   }
}


/**
 * setAuto_TimeZoneOffsetProperty -  Handle to set Auto Time Zone in Clock Service
 * @param[in] TimeDatePropertyInfo& stTimeDatePropertyInfo
 * @param[out] none
 * @return
 */
/* //TODO: if OffSet hours and minutes is appcable then this code will be uncommented
void ClockSrvClient::setAuto_TimeZoneOffsetPropertyToService(Auto_TimeZonePropertyInfo& _OffsetPropertyInfo) //TODO: call from Applogic is required
{
  performTimeZoneOffsetChange(_OffsetPropertyInfo);
  //check any parameter is invalid.
  bool isValid = true;
  if ((_OffsetPropertyInfo.hour < MIN_OFFSET_TIMEZONE_INVALID_HOUR) || (_OffsetPropertyInfo.minutes < MIN_OFFSET_TIMEZONE_INVALID_MINUTE) \
        || (_OffsetPropertyInfo.hour > MAX_OFFSET_TIMEZONE_INVALID_HOUR) || (_OffsetPropertyInfo.minutes >= INVALID_MINUTE))
  {
     isValid = false;
  }
  //If any one of parameter is invalid, value will not be set to service.
  if (isValid)
  {
     if (_clockproxy)
     {
        _clockproxy->sendAuto_TimeZoneOffsetSet(*this, _OffsetPropertyInfo.hour, _OffsetPropertyInfo.minutes);
     }
  }
  else
  {
     ETG_TRACE_ERR(("ClockSrvClient:ERROR:Invalid hour or minute (out of range)"));
  }
}*/


/**
 * onAuto_TimeZoneOffsetError - Error handling for Auto_TimeZoneOffsetProperty
 * @param[in] proxy
 * @param[in] error
 * @param[out] none
 * @return void
 */
void ClockSrvClient::onAuto_TimeZoneOffsetError(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< clock_main_fi::Auto_TimeZoneOffsetError >& /*error*/)
{
   ETG_TRACE_ERR(("ClockSrvClient::onAuto_TimeZoneOffsetError is called"));
}


/**
 * onAuto_TimeZoneOffsetStatus - Status handling for Auto_TimeZoneOffsetProperty
 * @param[in] proxy
 * @param[in] Status
 * @param[out] none
 * @return void
 */
void ClockSrvClient::onAuto_TimeZoneOffsetStatus(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
      const ::boost::shared_ptr< clock_main_fi::Auto_TimeZoneOffsetStatus >& status)
{
   ETG_TRACE_USR4(("ClockSrvClient::onAuto_TimeZoneOffsetStatus is called"));
   if (proxy && (proxy == _clockproxy))
   {
      Auto_TimeZonePropertyInfo stAuto_TimeZonePropertyInfo;
      stAuto_TimeZonePropertyInfo.hour = status->getS8Offset_Hour();
      stAuto_TimeZonePropertyInfo.minutes = status->getS8Offset_Minute();
      GpsClockListHandler::poGetInstance()->handleAuto_TimeZoneOffsetPropertyUpdate(stAuto_TimeZonePropertyInfo); //TODO: Impl in Applogic is required
   }
}


/**
 * performTimeZoneOffsetChange - Handle to perform Time Zone Offset change
 * @param[in] Auto_TimeZonePropertyInfo& _OffsetPropertyInfo
 * @param[out] hour
 * @return void
 */
/*
void ClockSrvClient::performTimeZoneOffsetChange(Auto_TimeZonePropertyInfo& _OffsetPropertyInfo)
{
  ClockUtility _clockUtility;
  switch (_OffsetPropertyInfo.propertytype)// List Index
  {
#ifndef VARIANT_S_FTR_ENABLE_A2S1
     case ITEM_4://OFFSET_HOUR_TYPE:
#endif
#ifdef VARIANT_S_FTR_ENABLE_A2S1
     case ITEM_2://OFFSET_HOUR_TYPE:
#endif
     {
        _OffsetPropertyInfo.hour = _clockUtility.performHourIncDec(_OffsetPropertyInfo.mode, \
                                   _OffsetPropertyInfo.hour);
     }
     break;
#ifndef VARIANT_S_FTR_ENABLE_A2S1
     case ITEM_5://OFFSET_MINUTE_TYPE:
#endif
#ifdef VARIANT_S_FTR_ENABLE_A2S1
     case ITEM_3://OFFSET_MINUTE_TYPE:
#endif
     {
        _OffsetPropertyInfo.minutes = _clockUtility.performMinuteIncDec(_OffsetPropertyInfo.mode, \
                                      _OffsetPropertyInfo.minutes);
     }
     break;
     default:
        break;
  }
}*/


/**
 * onTimeZone_DSTAvailabilityError - Error handling for imeZone_DSTAvailabilityProperty
 * @param[in] proxy
 * @param[in] error
 * @param[out] none
 * @return void
 */
void ClockSrvClient::onTimeZone_DSTAvailabilityError(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< clock_main_fi::TimeZone_DSTAvailabilityError >& /*error*/)
{
   ETG_TRACE_USR4(("ClockSrvClient:onTimeZone_DSTAvailabilityError is called"));
}


/**
 * onTimeZone_DSTAvailabilityStatus - Status handling for imeZone_DSTAvailabilityProperty
 * @param[in] proxy
 * @param[in] status
 * @param[out] none
 * @return void
 */
void ClockSrvClient::onTimeZone_DSTAvailabilityStatus(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
      const ::boost::shared_ptr< clock_main_fi::TimeZone_DSTAvailabilityStatus >& /* status */)
{
   ETG_TRACE_USR4(("ClockSrvClient:onTimeZone_DSTAvailabilityStatus is called"));
   if (proxy && (proxy == _clockproxy))
   {
      //bool statusTimeZone_DST = status->getBTimeZone_DSTAvailabilityForRegion();
      // ClockSettings::poGetInstance()->handleTimeZone_DSTAvailabilityUpdate(statusTimeZone_DST);
   }
}


}
}
